import { select, local } from 'd3-selection'

export default function () {
  let nodeTitle = (d) => d.title !== undefined ? d.title : d.id
  let nodeVisible = (d) => !!nodeTitle(d)

  function sankeyNode (context) {
    const selection = context.selection ? context.selection() : context

    if (selection.select('text').empty()) {
      selection.append('title')
      selection.append('text')
        .attr('dy', '.35em')
      selection.append('line')
        .attr('x1', 0)
        .attr('x2', 0)
      selection.append('rect')
        .attr('x', -5)
        .attr('y', -5)
        .attr('width', 10)
        .style('fill', 'none')
        .style('visibility', 'hidden')
        .style('pointer-events', 'all')

      selection
        .attr('transform', nodeTransform)
    }

    let title = selection.select('title')
    let text = selection.select('text')
    let line = selection.select('line')
    let clickTarget = selection.select('rect')

    // Local var for title position of each node
    const nodeLayout = local()
    selection.each(function (d) {
      const layoutData = titlePosition(d)
      layoutData.dy = (d.y0 === d.y1) ? 0 : Math.max(1, d.y1 - d.y0)
      nodeLayout.set(this, layoutData)
    })

    // Update un-transitioned
    title
      .text(nodeTitle)

    text
      .attr('text-anchor', function (d) { return nodeLayout.get(this).right ? 'end' : 'start' })
      .text(nodeTitle)
      .each(wrap, 100)

    // Are we in a transition?
    if (context !== selection) {
      text = text.transition(context)
      line = line.transition(context)
      clickTarget = clickTarget.transition(context)
    }

    // Update
    context
      .attr('transform', nodeTransform)

    line
      .attr('y1', function (d) { return nodeLayout.get(this).titleAbove ? -5 : 0 })
      .attr('y2', function (d) { return nodeLayout.get(this).dy })
      .style('display', function (d) {
        return (d.y0 === d.y1 || !nodeVisible(d)) ? 'none' : 'inline'
      })

    clickTarget
      .attr('height', function (d) { return nodeLayout.get(this).dy + 5 })

    text
      .attr('transform', textTransform)
      .style('display', function (d) {
        return (d.y0 === d.y1 || !nodeVisible(d)) ? 'none' : 'inline'
      })

    function textTransform (d) {
      const layout = nodeLayout.get(this)
      const y = layout.titleAbove ? -10 : (d.y1 - d.y0) / 2
      const x = (layout.right ? 1 : -1) * (layout.titleAbove ? 4 : -4)
      return 'translate(' + x + ',' + y + ')'
    }
  }

  sankeyNode.nodeVisible = function (x) {
    if (arguments.length) {
      nodeVisible = required(x)
      return sankeyNode
    }
    return nodeVisible
  }

  sankeyNode.nodeTitle = function (x) {
    if (arguments.length) {
      nodeTitle = required(x)
      return sankeyNode
    }
    return nodeTitle
  }

  return sankeyNode
}

function nodeTransform (d) {
  return 'translate(' + d.x0 + ',' + d.y0 + ')'
}

function titlePosition (d) {
  let titleAbove = false
  let right = false

  // If thin, and there's enough space, put above
  if (d.spaceAbove > 20 && d.style !== 'type') {
    titleAbove = true
  } else {
    titleAbove = false
  }

  if (d.incoming.length === 0) {
    right = true
    titleAbove = false
  } else if (d.outgoing.length === 0) {
    right = false
    titleAbove = false
  }

  return {titleAbove, right}
}

function wrap (d, width) {
  var text = select(this)
  var lines = text.text().split(/\n/)
  var lineHeight = 1.1 // ems
  if (lines.length === 1) return
  text.text(null)
  lines.forEach(function (line, i) {
    text.append('tspan')
      .attr('x', 0)
      .attr('dy', (i === 0 ? 0.7 - lines.length / 2 : 1) * lineHeight + 'em')
      .text(line)
  })
}

function required (f) {
  if (typeof f !== 'function') throw new Error()
  return f
}
