(function (global, factory) {
  typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports, require('d3-scale'), require('d3-selection'), require('d3-path'), require('d3-shape'), require('d3-array')) :
  typeof define === 'function' && define.amd ? define(['exports', 'd3-scale', 'd3-selection', 'd3-path', 'd3-shape', 'd3-array'], factory) :
  (factory((global.fc = global.fc || {}),global.d3,global.d3,global.d3,global.d3,global.d3));
}(this, (function (exports,d3Scale,d3Selection,d3Path,d3Shape,d3Array) { 'use strict';

// "Caution: avoid interpolating to or from the number zero when the interpolator is used to generate
// a string (such as with attr).
// Very small values, when stringified, may be converted to scientific notation and
// cause a temporarily invalid attribute or style property value.
// For example, the number 0.0000001 is converted to the string "1e-7".
// This is particularly noticeable when interpolating opacity values.
// To avoid scientific notation, start or end the transition at 1e-6,
// which is the smallest value that is not stringified in exponential notation."
// - https://github.com/mbostock/d3/wiki/Transitions#d3_interpolateNumber
var effectivelyZero = 1e-6;

// Wrapper around d3's selectAll/data data-join, which allows decoration of the result.
// This is achieved by appending the element to the enter selection before exposing it.
// A default transition of fade in/out is also implicitly added but can be modified.
var dataJoin = (function (element, className) {
    element = element || 'g';

    var key = function key(_, i) {
        return i;
    };
    var explicitTransition = null;

    var dataJoin = function dataJoin(container, data) {
        data = data || function (d) {
            return d;
        };

        var implicitTransition = container.selection ? container : null;
        if (implicitTransition) {
            container = container.selection();
        }

        var selected = container.selectAll(function (d, i, nodes) {
            return Array.from(nodes[i].childNodes).filter(function (node) {
                return node.nodeType === 1;
            });
        }).filter(className == null ? element : element + '.' + className);
        var update = selected.data(data, key);

        var enter = update.enter().append(element).attr('class', className);

        var exit = update.exit();

        // automatically merge in the enter selection
        update = update.merge(enter);

        // if transitions are enabled apply a default fade in/out transition
        var transition = implicitTransition || explicitTransition;
        if (transition) {
            update = update.transition(transition).style('opacity', 1);
            enter.style('opacity', effectivelyZero);
            exit = exit.transition(transition).style('opacity', effectivelyZero);
        }

        exit.remove();

        update.enter = function () {
            return enter;
        };
        update.exit = function () {
            return exit;
        };

        return update;
    };

    dataJoin.element = function () {
        if (!arguments.length) {
            return element;
        }
        element = arguments.length <= 0 ? undefined : arguments[0];
        return dataJoin;
    };
    dataJoin.className = function () {
        if (!arguments.length) {
            return className;
        }
        className = arguments.length <= 0 ? undefined : arguments[0];
        return dataJoin;
    };
    dataJoin.key = function () {
        if (!arguments.length) {
            return key;
        }
        key = arguments.length <= 0 ? undefined : arguments[0];
        return dataJoin;
    };
    dataJoin.transition = function () {
        if (!arguments.length) {
            return explicitTransition;
        }
        explicitTransition = arguments.length <= 0 ? undefined : arguments[0];
        return dataJoin;
    };

    return dataJoin;
});

var functor = (function (v) {
  return typeof v === 'function' ? v : function () {
    return v;
  };
});

// Renders an OHLC as an SVG path based on the given array of datapoints. Each
// OHLC has a fixed width, whilst the x, open, high, low and close positions are
// obtained from each point via the supplied accessor functions.

// Renders a bar series as an SVG path based on the given array of datapoints. Each
// bar has a fixed width, whilst the x, y and height are obtained from each data
// point via the supplied accessor functions.
var shapeBar = (function () {

    var context = null;
    var x = function x(d) {
        return d.x;
    };
    var y = function y(d) {
        return d.y;
    };
    var horizontalAlign = 'center';
    var verticalAlign = 'center';
    var height = function height(d) {
        return d.height;
    };
    var width = functor(3);

    var bar = function bar(data, index) {

        var drawingContext = context || d3Path.path();

        data.forEach(function (d, i) {
            var xValue = x.call(this, d, index || i);
            var yValue = y.call(this, d, index || i);
            var barHeight = height.call(this, d, index || i);
            var barWidth = width.call(this, d, index || i);

            var horizontalOffset = void 0;
            switch (horizontalAlign) {
                case 'left':
                    horizontalOffset = barWidth;
                    break;
                case 'right':
                    horizontalOffset = 0;
                    break;
                case 'center':
                    horizontalOffset = barWidth / 2;
                    break;
                default:
                    throw new Error('Invalid horizontal alignment ' + horizontalAlign);
            }

            var verticalOffset = void 0;
            switch (verticalAlign) {
                case 'bottom':
                    verticalOffset = -barHeight;
                    break;
                case 'top':
                    verticalOffset = 0;
                    break;
                case 'center':
                    verticalOffset = barHeight / 2;
                    break;
                default:
                    throw new Error('Invalid vertical alignment ' + verticalAlign);
            }

            drawingContext.rect(xValue - horizontalOffset, yValue - verticalOffset, barWidth, barHeight);
        }, this);

        return context ? null : drawingContext.toString();
    };

    bar.context = function () {
        if (!arguments.length) {
            return context;
        }
        context = arguments.length <= 0 ? undefined : arguments[0];
        return bar;
    };
    bar.x = function () {
        if (!arguments.length) {
            return x;
        }
        x = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return bar;
    };
    bar.y = function () {
        if (!arguments.length) {
            return y;
        }
        y = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return bar;
    };
    bar.width = function () {
        if (!arguments.length) {
            return width;
        }
        width = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return bar;
    };
    bar.horizontalAlign = function () {
        if (!arguments.length) {
            return horizontalAlign;
        }
        horizontalAlign = arguments.length <= 0 ? undefined : arguments[0];
        return bar;
    };
    bar.height = function () {
        if (!arguments.length) {
            return height;
        }
        height = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return bar;
    };
    bar.verticalAlign = function () {
        if (!arguments.length) {
            return verticalAlign;
        }
        verticalAlign = arguments.length <= 0 ? undefined : arguments[0];
        return bar;
    };

    return bar;
});

// Renders a candlestick as an SVG path based on the given array of datapoints. Each
// candlestick has a fixed width, whilst the x, open, high, low and close positions are
// obtained from each point via the supplied accessor functions.

// Renders a box plot series as an SVG path based on the given array of datapoints.

// Renders an error bar series as an SVG path based on the given array of datapoints.

var constant = (function (value) {
  return typeof value === 'function' ? value : function () {
    return value;
  };
});

var band = (function () {

    var xScale = d3Scale.scaleIdentity();
    var yScale = d3Scale.scaleIdentity();
    var orient = 'horizontal';
    var fromValue = function fromValue(d) {
        return d.from;
    };
    var toValue = function toValue(d) {
        return d.to;
    };
    var decorate = function decorate() {};

    var join = dataJoin('g', 'annotation-band');

    var pathGenerator = shapeBar().horizontalAlign('center').verticalAlign('center').x(0).y(0);

    var instance = function instance(selection$$1) {

        if (selection$$1.selection) {
            join.transition(selection$$1);
        }

        if (orient !== 'horizontal' && orient !== 'vertical') {
            throw new Error('Invalid orientation');
        }

        var horizontal = orient === 'horizontal';
        var translation = horizontal ? function (a, b) {
            return 'translate(' + a + ', ' + b + ')';
        } : function (a, b) {
            return 'translate(' + b + ', ' + a + ')';
        };
        // the value scale which the annotation 'value' relates to, the crossScale
        // is the other. Which is which depends on the orienation!
        var crossScale = horizontal ? xScale : yScale;
        var valueScale = horizontal ? yScale : xScale;
        var crossScaleRange = crossScale.range();
        var crossScaleSize = crossScaleRange[1] - crossScaleRange[0];
        var valueAxisDimension = horizontal ? 'height' : 'width';
        var crossAxisDimension = horizontal ? 'width' : 'height';
        var containerTransform = function containerTransform() {
            return translation((crossScaleRange[1] + crossScaleRange[0]) / 2, (valueScale(toValue.apply(undefined, arguments)) + valueScale(fromValue.apply(undefined, arguments))) / 2);
        };

        pathGenerator[crossAxisDimension](crossScaleSize);
        pathGenerator[valueAxisDimension](function () {
            return valueScale(toValue.apply(undefined, arguments)) - valueScale(fromValue.apply(undefined, arguments));
        });

        selection$$1.each(function (data, index, nodes) {

            var g = join(d3Selection.select(nodes[index]), data);

            g.enter().attr('transform', containerTransform).append('path').classed('band', true);

            g.attr('class', 'annotation-band ' + orient).attr('transform', containerTransform).select('path')
            // the path generator is being used to render a single path, hence
            // an explicit index is provided
            .attr('d', function (d, i) {
                return pathGenerator([d], i);
            });

            decorate(g, data, index);
        });
    };

    instance.xScale = function () {
        if (!arguments.length) {
            return xScale;
        }
        xScale = arguments.length <= 0 ? undefined : arguments[0];
        return instance;
    };
    instance.yScale = function () {
        if (!arguments.length) {
            return yScale;
        }
        yScale = arguments.length <= 0 ? undefined : arguments[0];
        return instance;
    };
    instance.orient = function () {
        if (!arguments.length) {
            return orient;
        }
        orient = arguments.length <= 0 ? undefined : arguments[0];
        return instance;
    };
    instance.decorate = function () {
        if (!arguments.length) {
            return decorate;
        }
        decorate = arguments.length <= 0 ? undefined : arguments[0];
        return instance;
    };
    instance.fromValue = function () {
        if (!arguments.length) {
            return fromValue;
        }
        fromValue = constant(arguments.length <= 0 ? undefined : arguments[0]);
        return instance;
    };
    instance.toValue = function () {
        if (!arguments.length) {
            return toValue;
        }
        toValue = constant(arguments.length <= 0 ? undefined : arguments[0]);
        return instance;
    };

    return instance;
});

var createReboundMethod = (function (target, source, name) {
    var method = source[name];
    if (typeof method !== 'function') {
        throw new Error('Attempt to rebind ' + name + ' which isn\'t a function on the source object');
    }
    return function () {
        for (var _len = arguments.length, args = Array(_len), _key = 0; _key < _len; _key++) {
            args[_key] = arguments[_key];
        }

        var value = method.apply(source, args);
        return value === source ? target : value;
    };
});

var rebind = (function (target, source) {
    for (var _len = arguments.length, names = Array(_len > 2 ? _len - 2 : 0), _key = 2; _key < _len; _key++) {
        names[_key - 2] = arguments[_key];
    }

    var _iteratorNormalCompletion = true;
    var _didIteratorError = false;
    var _iteratorError = undefined;

    try {
        for (var _iterator = names[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
            var name = _step.value;

            target[name] = createReboundMethod(target, source, name);
        }
    } catch (err) {
        _didIteratorError = true;
        _iteratorError = err;
    } finally {
        try {
            if (!_iteratorNormalCompletion && _iterator.return) {
                _iterator.return();
            }
        } finally {
            if (_didIteratorError) {
                throw _iteratorError;
            }
        }
    }

    return target;
});

var createTransform = function createTransform(transforms) {
    return function (name) {
        return transforms.reduce(function (name, fn) {
            return name && fn(name);
        }, name);
    };
};

var rebindAll = (function (target, source) {
    for (var _len = arguments.length, transforms = Array(_len > 2 ? _len - 2 : 0), _key = 2; _key < _len; _key++) {
        transforms[_key - 2] = arguments[_key];
    }

    var transform = createTransform(transforms);
    var _iteratorNormalCompletion = true;
    var _didIteratorError = false;
    var _iteratorError = undefined;

    try {
        for (var _iterator = Object.keys(source)[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
            var name = _step.value;

            var result = transform(name);
            if (result) {
                target[result] = createReboundMethod(target, source, name);
            }
        }
    } catch (err) {
        _didIteratorError = true;
        _iteratorError = err;
    } finally {
        try {
            if (!_iteratorNormalCompletion && _iterator.return) {
                _iterator.return();
            }
        } finally {
            if (_didIteratorError) {
                throw _iteratorError;
            }
        }
    }

    return target;
});

var regexify = (function (strsOrRegexes) {
    return strsOrRegexes.map(function (strOrRegex) {
        return typeof strOrRegex === 'string' ? new RegExp('^' + strOrRegex + '$') : strOrRegex;
    });
});

var exclude = (function () {
    for (var _len = arguments.length, exclusions = Array(_len), _key = 0; _key < _len; _key++) {
        exclusions[_key] = arguments[_key];
    }

    exclusions = regexify(exclusions);
    return function (name) {
        return exclusions.every(function (exclusion) {
            return !exclusion.test(name);
        }) && name;
    };
});

var include = (function () {
    for (var _len = arguments.length, inclusions = Array(_len), _key = 0; _key < _len; _key++) {
        inclusions[_key] = arguments[_key];
    }

    inclusions = regexify(inclusions);
    return function (name) {
        return inclusions.some(function (inclusion) {
            return inclusion.test(name);
        }) && name;
    };
});

var includeMap = (function (mappings) {
  return function (name) {
    return mappings[name];
  };
});

var capitalizeFirstLetter = function capitalizeFirstLetter(str) {
  return str[0].toUpperCase() + str.slice(1);
};

var prefix = (function (prefix) {
  return function (name) {
    return prefix + capitalizeFirstLetter(name);
  };
});

var annotationLine = (function () {

    var xScale = d3Scale.scaleIdentity();
    var yScale = d3Scale.scaleIdentity();
    var value = function value(d) {
        return d;
    };
    var label = value;
    var decorate = function decorate() {};
    var orient = 'horizontal';

    var join = dataJoin('g', 'annotation-line');

    var instance = function instance(selection$$1) {

        if (selection$$1.selection) {
            join.transition(selection$$1);
        }

        if (orient !== 'horizontal' && orient !== 'vertical') {
            throw new Error('Invalid orientation');
        }
        var horizontal = orient === 'horizontal';
        var translation = horizontal ? function (a, b) {
            return 'translate(' + a + ', ' + b + ')';
        } : function (a, b) {
            return 'translate(' + b + ', ' + a + ')';
        };
        var lineProperty = horizontal ? 'x2' : 'y2';
        // the value scale which the annotation 'value' relates to, the crossScale
        // is the other. Which is which depends on the orienation!
        var crossScale = horizontal ? xScale : yScale;
        var valueScale = horizontal ? yScale : xScale;
        var handleOne = horizontal ? 'left-handle' : 'bottom-handle';
        var handleTwo = horizontal ? 'right-handle' : 'top-handle';
        var textOffsetX = horizontal ? '9' : '0';
        var textOffsetY = horizontal ? '0' : '9';
        var textOffsetDeltaY = horizontal ? '0.32em' : '0.71em';
        var textAnchor = horizontal ? 'start' : 'middle';

        var scaleRange = crossScale.range();
        // the transform that sets the 'origin' of the annotation
        var containerTransform = function containerTransform() {
            return translation(scaleRange[0], valueScale(value.apply(undefined, arguments)));
        };
        var scaleWidth = scaleRange[1] - scaleRange[0];

        selection$$1.each(function (data, selectionIndex, nodes) {

            var g = join(d3Selection.select(nodes[selectionIndex]), data);

            // create the outer container and line
            var enter = g.enter().attr('transform', containerTransform).style('stroke', '#bbb');
            enter.append('line').attr(lineProperty, scaleWidth);

            // create containers at each end of the annotation
            enter.append('g').classed(handleOne, true).style('stroke', 'none');

            enter.append('g').classed(handleTwo, true).style('stroke', 'none').attr('transform', translation(scaleWidth, 0)).append('text').attr('text-anchor', textAnchor).attr('x', textOffsetX).attr('y', textOffsetY).attr('dy', textOffsetDeltaY);

            // Update
            g.attr('class', 'annotation-line ' + orient);

            // translate the parent container to the left hand edge of the annotation
            g.attr('transform', containerTransform);

            // update the elements that depend on scale width
            g.select('line').attr(lineProperty, scaleWidth);
            g.select('g.' + handleTwo).attr('transform', translation(scaleWidth, 0));

            // Update the text label
            g.select('text').text(label);

            decorate(g, data, selectionIndex);
        });
    };

    instance.xScale = function () {
        if (!arguments.length) {
            return xScale;
        }
        xScale = arguments.length <= 0 ? undefined : arguments[0];
        return instance;
    };
    instance.yScale = function () {
        if (!arguments.length) {
            return yScale;
        }
        yScale = arguments.length <= 0 ? undefined : arguments[0];
        return instance;
    };
    instance.value = function () {
        if (!arguments.length) {
            return value;
        }
        value = constant(arguments.length <= 0 ? undefined : arguments[0]);
        return instance;
    };
    instance.label = function () {
        if (!arguments.length) {
            return label;
        }
        label = constant(arguments.length <= 0 ? undefined : arguments[0]);
        return instance;
    };
    instance.decorate = function () {
        if (!arguments.length) {
            return decorate;
        }
        decorate = arguments.length <= 0 ? undefined : arguments[0];
        return instance;
    };
    instance.orient = function () {
        if (!arguments.length) {
            return orient;
        }
        orient = arguments.length <= 0 ? undefined : arguments[0];
        return instance;
    };

    return instance;
});

var functor$1 = (function (d) {
  return typeof d === 'function' ? d : function () {
    return d;
  };
});

// Checks that passes properties are 'defined', meaning that calling them with (d, i) returns non null values
function defined$1() {
    var outerArguments = arguments;
    return function (d, i) {
        for (var c = 0, j = outerArguments.length; c < j; c++) {
            if (outerArguments[c](d, i) == null) {
                return false;
            }
        }
        return true;
    };
}

// determines the offset required along the cross scale based
// on the series alignment
var alignOffset = (function (align, width) {
    switch (align) {
        case 'left':
            return width / 2;
        case 'right':
            return -width / 2;
        default:
            return 0;
    }
});

var createBase = (function (initialValues) {

    var env = Object.assign({}, initialValues);
    var base = function base() {};

    Object.keys(env).forEach(function (key) {
        base[key] = function () {
            if (!arguments.length) {
                return env[key];
            }
            env[key] = arguments.length <= 0 ? undefined : arguments[0];
            return base;
        };
    });

    return base;
});

var xyBase = (function () {

    var baseValue = function baseValue() {
        return 0;
    };
    var crossValue = function crossValue(d) {
        return d.x;
    };
    var mainValue = function mainValue(d) {
        return d.y;
    };
    var align = 'center';
    var bandwidth = function bandwidth() {
        return 5;
    };
    var orient = 'vertical';

    var base = createBase({
        decorate: function decorate() {},
        defined: function defined(d, i) {
            return defined$1(baseValue, crossValue, mainValue)(d, i);
        },
        xScale: d3Scale.scaleIdentity(),
        yScale: d3Scale.scaleIdentity()
    });

    base.values = function (d, i) {
        var width = bandwidth(d, i);
        var offset = alignOffset(align, width);
        var xScale = base.xScale();
        var yScale = base.yScale();

        if (orient === 'vertical') {
            var y = yScale(mainValue(d, i), i);
            var y0 = yScale(baseValue(d, i), i);
            var x = xScale(crossValue(d, i), i) + offset;
            return {
                d: d,
                x: x,
                y: y,
                y0: y0,
                width: width,
                height: y - y0,
                origin: [x, y],
                baseOrigin: [x, y0],
                transposedX: x,
                transposedY: y
            };
        } else {
            var _y = xScale(mainValue(d, i), i);
            var _y2 = xScale(baseValue(d, i), i);
            var _x = yScale(crossValue(d, i), i) + offset;
            return {
                d: d,
                x: _x,
                y: _y,
                y0: _y2,
                width: width,
                height: _y - _y2,
                origin: [_y, _x],
                baseOrigin: [_y2, _x],
                transposedX: _y,
                transposedY: _x
            };
        }
    };

    base.baseValue = function () {
        if (!arguments.length) {
            return baseValue;
        }
        baseValue = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return base;
    };
    base.crossValue = function () {
        if (!arguments.length) {
            return crossValue;
        }
        crossValue = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return base;
    };
    base.mainValue = function () {
        if (!arguments.length) {
            return mainValue;
        }
        mainValue = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return base;
    };
    base.bandwidth = function () {
        if (!arguments.length) {
            return bandwidth;
        }
        bandwidth = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return base;
    };
    base.align = function () {
        if (!arguments.length) {
            return align;
        }
        align = arguments.length <= 0 ? undefined : arguments[0];
        return base;
    };
    base.orient = function () {
        if (!arguments.length) {
            return orient;
        }
        orient = arguments.length <= 0 ? undefined : arguments[0];
        return base;
    };

    return base;
});

var red = '#c60';
var green = '#6c0';
var black = '#000';
var gray = '#ddd';
var darkGray = '#999';

var colors = {
    red: red,
    green: green,
    black: black,
    gray: gray,
    darkGray: darkGray
};

var seriesSvgPoint = (function () {
    var symbol$$1 = d3Shape.symbol();

    var base = xyBase();

    var join = dataJoin('g', 'point');

    var containerTransform = function containerTransform(origin) {
        return 'translate(' + origin[0] + ', ' + origin[1] + ')';
    };

    var point = function point(selection$$1) {

        if (selection$$1.selection) {
            join.transition(selection$$1);
        }

        selection$$1.each(function (data, index, group) {

            var filteredData = data.filter(base.defined());

            var g = join(d3Selection.select(group[index]), filteredData);
            g.enter().attr('transform', function (d, i) {
                return containerTransform(base.values(d, i).origin);
            }).attr('fill', colors.gray).attr('stroke', colors.black).append('path');

            g.attr('transform', function (d, i) {
                return containerTransform(base.values(d, i).origin);
            }).select('path').attr('d', symbol$$1);

            base.decorate()(g, data, index);
        });
    };

    rebindAll(point, base, exclude('baseValue', 'bandwidth', 'align'));
    rebind(point, join, 'key');
    rebind(point, symbol$$1, 'type', 'size');

    return point;
});

var multiBase = (function () {

    var series = [];
    var mapping = function mapping(d) {
        return d;
    };
    var key = function key(_, i) {
        return i;
    };

    var multi = createBase({
        decorate: function decorate() {},
        xScale: d3Scale.scaleIdentity(),
        yScale: d3Scale.scaleIdentity()
    });

    multi.mapping = function () {
        if (!arguments.length) {
            return mapping;
        }
        mapping = arguments.length <= 0 ? undefined : arguments[0];
        return multi;
    };
    multi.key = function () {
        if (!arguments.length) {
            return key;
        }
        key = arguments.length <= 0 ? undefined : arguments[0];
        return multi;
    };
    multi.series = function () {
        if (!arguments.length) {
            return series;
        }
        series = arguments.length <= 0 ? undefined : arguments[0];
        return multi;
    };

    return multi;
});

var multiSeries = (function () {

    var base = multiBase();

    var innerJoin = dataJoin('g');

    var join = dataJoin('g', 'multi');

    var multi = function multi(selection$$1) {

        if (selection$$1.selection) {
            join.transition(selection$$1);
            innerJoin.transition(selection$$1);
        }

        var mapping = base.mapping();
        var series = base.series();
        var xScale = base.xScale();
        var yScale = base.yScale();

        selection$$1.each(function (data, index, group) {

            var container = join(d3Selection.select(group[index]), series);

            // iterate over the containers, 'call'-ing the series for each
            container.each(function (dataSeries, seriesIndex, seriesGroup) {
                dataSeries.xScale(xScale).yScale(yScale);

                var seriesData = mapping(data, seriesIndex, series);
                var innerContainer = innerJoin(d3Selection.select(seriesGroup[seriesIndex]), [seriesData]);

                innerContainer.call(dataSeries);
            });

            var unwrappedSelection = container.selection ? container.selection() : container;
            unwrappedSelection.order();

            base.decorate()(container, data, index);
        });
    };

    rebindAll(multi, base);
    rebind(multi, join, 'key');

    return multi;
});

var toConsumableArray = function (arr) {
  if (Array.isArray(arr)) {
    for (var i = 0, arr2 = Array(arr.length); i < arr.length; i++) arr2[i] = arr[i];

    return arr2;
  } else {
    return Array.from(arr);
  }
};

var crosshair = function () {

    var x = function x(d) {
        return d.x;
    };
    var y = function y(d) {
        return d.y;
    };
    var xScale = d3Scale.scaleIdentity();
    var yScale = d3Scale.scaleIdentity();
    var decorate = function decorate() {};

    var join = dataJoin('g', 'annotation-crosshair');

    var point = seriesSvgPoint();

    var horizontalLine = annotationLine();

    var verticalLine = annotationLine().orient('vertical');

    // The line annotations and point series used to render the crosshair are positioned using
    // screen coordinates. This function constructs an identity scale for these components.
    var xIdentity = d3Scale.scaleIdentity();
    var yIdentity = d3Scale.scaleIdentity();

    var multi = multiSeries().series([horizontalLine, verticalLine, point]).xScale(xIdentity).yScale(yIdentity).mapping(function (data) {
        return [data];
    });

    var instance = function instance(selection$$1) {

        if (selection$$1.selection) {
            join.transition(selection$$1);
        }

        selection$$1.each(function (data, index, nodes) {

            var g = join(d3Selection.select(nodes[index]), data);

            // Prevent the crosshair triggering pointer events on itself
            g.enter().style('pointer-events', 'none');

            // Assign the identity scales an accurate range to allow the line annotations to cover
            // the full width/height of the chart.
            xIdentity.range(xScale.range());
            yIdentity.range(yScale.range());

            point.crossValue(x).mainValue(y);

            horizontalLine.value(y);

            verticalLine.value(x);

            g.call(multi);

            decorate(g, data, index);
        });
    };

    // Don't use the xValue/yValue convention to indicate that these values are in screen
    // not domain co-ordinates and are therefore not scaled.
    instance.x = function () {
        if (!arguments.length) {
            return x;
        }
        x = arguments.length <= 0 ? undefined : arguments[0];
        return instance;
    };
    instance.y = function () {
        if (!arguments.length) {
            return y;
        }
        y = arguments.length <= 0 ? undefined : arguments[0];
        return instance;
    };
    instance.xScale = function () {
        if (!arguments.length) {
            return xScale;
        }
        xScale = arguments.length <= 0 ? undefined : arguments[0];
        return instance;
    };
    instance.yScale = function () {
        if (!arguments.length) {
            return yScale;
        }
        yScale = arguments.length <= 0 ? undefined : arguments[0];
        return instance;
    };
    instance.decorate = function () {
        if (!arguments.length) {
            return decorate;
        }
        decorate = arguments.length <= 0 ? undefined : arguments[0];
        return instance;
    };

    var lineIncludes = include('label');
    rebindAll(instance, horizontalLine, lineIncludes, prefix('y'));
    rebindAll(instance, verticalLine, lineIncludes, prefix('x'));

    return instance;
};

var ticks = (function () {

    var scale = d3Scale.scaleIdentity();
    var tickArguments = [10];
    var tickValues = null;

    var ticks = function ticks() {
        var _scale;

        return tickValues != null ? tickValues : scale.ticks ? (_scale = scale).ticks.apply(_scale, toConsumableArray(tickArguments)) : scale.domain();
    };

    ticks.scale = function () {
        if (!arguments.length) {
            return scale;
        }
        scale = arguments.length <= 0 ? undefined : arguments[0];
        return ticks;
    };

    ticks.ticks = function () {
        for (var _len = arguments.length, args = Array(_len), _key = 0; _key < _len; _key++) {
            args[_key] = arguments[_key];
        }

        tickArguments = args;
        return ticks;
    };

    ticks.tickArguments = function () {
        if (!arguments.length) {
            return tickArguments;
        }
        tickArguments = arguments.length <= 0 ? undefined : arguments[0];
        return ticks;
    };

    ticks.tickValues = function () {
        if (!arguments.length) {
            return tickValues;
        }
        tickValues = arguments.length <= 0 ? undefined : arguments[0];
        return ticks;
    };

    return ticks;
});

var identity = function identity(d) {
    return d;
};

var gridline = (function () {

    var xDecorate = function xDecorate() {};
    var yDecorate = function yDecorate() {};

    var xTicks = ticks();
    var yTicks = ticks();
    var xJoin = dataJoin('line', 'gridline-y').key(identity);
    var yJoin = dataJoin('line', 'gridline-x').key(identity);

    var instance = function instance(selection$$1) {

        if (selection$$1.selection) {
            xJoin.transition(selection$$1);
            yJoin.transition(selection$$1);
        }

        selection$$1.each(function (data, index, nodes) {

            var element = nodes[index];
            var container = d3Selection.select(nodes[index]);

            var xScale = xTicks.scale();
            var yScale = yTicks.scale();

            // Stash a snapshot of the scale, and retrieve the old snapshot.
            var xScaleOld = element.__x_scale__ || xScale;
            element.__x_scale__ = xScale.copy();

            var xData = xTicks();
            var xLines = xJoin(container, xData);

            xLines.enter().attr('x1', xScaleOld).attr('x2', xScaleOld).attr('y1', yScale.range()[0]).attr('y2', yScale.range()[1]);

            xLines.attr('x1', xScale).attr('x2', xScale).attr('y1', yScale.range()[0]).attr('y2', yScale.range()[1]).attr('stroke', '#bbb');

            xLines.exit().attr('x1', xScale).attr('x2', xScale);

            xDecorate(xLines, xData, index);

            // Stash a snapshot of the scale, and retrieve the old snapshot.
            var yScaleOld = element.__y_scale__ || yScale;
            element.__y_scale__ = yScale.copy();

            var yData = yTicks();
            var yLines = yJoin(container, yData);

            yLines.enter().attr('y1', yScaleOld).attr('y2', yScaleOld).attr('x1', xScale.range()[0]).attr('x2', xScale.range()[1]);

            yLines.attr('y1', yScale).attr('y2', yScale).attr('x1', xScale.range()[0]).attr('x2', xScale.range()[1]).attr('stroke', '#bbb');

            yLines.exit().attr('y1', yScale).attr('y2', yScale);

            yDecorate(yLines, yData, index);
        });
    };

    instance.yDecorate = function () {
        if (!arguments.length) {
            return yDecorate;
        }
        yDecorate = arguments.length <= 0 ? undefined : arguments[0];
        return instance;
    };

    instance.xDecorate = function () {
        if (!arguments.length) {
            return xDecorate;
        }
        xDecorate = arguments.length <= 0 ? undefined : arguments[0];
        return instance;
    };

    rebindAll(instance, xJoin, includeMap({ 'key': 'xKey' }));
    rebindAll(instance, yJoin, includeMap({ 'key': 'yKey' }));

    rebindAll(instance, xTicks, prefix('x'));
    rebindAll(instance, yTicks, prefix('y'));

    return instance;
});

exports.annotationSvgBand = band;
exports.annotationSvgCrosshair = crosshair;
exports.annotationSvgLine = annotationLine;
exports.annotationSvgGridline = gridline;

Object.defineProperty(exports, '__esModule', { value: true });

})));
