(function (global, factory) {
  typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports, require('d3-selection'), require('d3-shape'), require('d3-scale'), require('d3-path'), require('d3-array')) :
  typeof define === 'function' && define.amd ? define(['exports', 'd3-selection', 'd3-shape', 'd3-scale', 'd3-path', 'd3-array'], factory) :
  (factory((global.fc = global.fc || {}),global.d3,global.d3,global.d3,global.d3,global.d3));
}(this, (function (exports,d3Selection,d3Shape,d3Scale,d3Path,d3Array) { 'use strict';

// "Caution: avoid interpolating to or from the number zero when the interpolator is used to generate
// a string (such as with attr).
// Very small values, when stringified, may be converted to scientific notation and
// cause a temporarily invalid attribute or style property value.
// For example, the number 0.0000001 is converted to the string "1e-7".
// This is particularly noticeable when interpolating opacity values.
// To avoid scientific notation, start or end the transition at 1e-6,
// which is the smallest value that is not stringified in exponential notation."
// - https://github.com/mbostock/d3/wiki/Transitions#d3_interpolateNumber
var effectivelyZero = 1e-6;

// Wrapper around d3's selectAll/data data-join, which allows decoration of the result.
// This is achieved by appending the element to the enter selection before exposing it.
// A default transition of fade in/out is also implicitly added but can be modified.
var dataJoin = (function (element, className) {
    element = element || 'g';

    var key = function key(_, i) {
        return i;
    };
    var explicitTransition = null;

    var dataJoin = function dataJoin(container, data) {
        data = data || function (d) {
            return d;
        };

        var implicitTransition = container.selection ? container : null;
        if (implicitTransition) {
            container = container.selection();
        }

        var selected = container.selectAll(function (d, i, nodes) {
            return Array.from(nodes[i].childNodes).filter(function (node) {
                return node.nodeType === 1;
            });
        }).filter(className == null ? element : element + '.' + className);
        var update = selected.data(data, key);

        var enter = update.enter().append(element).attr('class', className);

        var exit = update.exit();

        // automatically merge in the enter selection
        update = update.merge(enter);

        // if transitions are enabled apply a default fade in/out transition
        var transition = implicitTransition || explicitTransition;
        if (transition) {
            update = update.transition(transition).style('opacity', 1);
            enter.style('opacity', effectivelyZero);
            exit = exit.transition(transition).style('opacity', effectivelyZero);
        }

        exit.remove();

        update.enter = function () {
            return enter;
        };
        update.exit = function () {
            return exit;
        };

        return update;
    };

    dataJoin.element = function () {
        if (!arguments.length) {
            return element;
        }
        element = arguments.length <= 0 ? undefined : arguments[0];
        return dataJoin;
    };
    dataJoin.className = function () {
        if (!arguments.length) {
            return className;
        }
        className = arguments.length <= 0 ? undefined : arguments[0];
        return dataJoin;
    };
    dataJoin.key = function () {
        if (!arguments.length) {
            return key;
        }
        key = arguments.length <= 0 ? undefined : arguments[0];
        return dataJoin;
    };
    dataJoin.transition = function () {
        if (!arguments.length) {
            return explicitTransition;
        }
        explicitTransition = arguments.length <= 0 ? undefined : arguments[0];
        return dataJoin;
    };

    return dataJoin;
});

var createReboundMethod = (function (target, source, name) {
    var method = source[name];
    if (typeof method !== 'function') {
        throw new Error('Attempt to rebind ' + name + ' which isn\'t a function on the source object');
    }
    return function () {
        for (var _len = arguments.length, args = Array(_len), _key = 0; _key < _len; _key++) {
            args[_key] = arguments[_key];
        }

        var value = method.apply(source, args);
        return value === source ? target : value;
    };
});

var rebind = (function (target, source) {
    for (var _len = arguments.length, names = Array(_len > 2 ? _len - 2 : 0), _key = 2; _key < _len; _key++) {
        names[_key - 2] = arguments[_key];
    }

    var _iteratorNormalCompletion = true;
    var _didIteratorError = false;
    var _iteratorError = undefined;

    try {
        for (var _iterator = names[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
            var name = _step.value;

            target[name] = createReboundMethod(target, source, name);
        }
    } catch (err) {
        _didIteratorError = true;
        _iteratorError = err;
    } finally {
        try {
            if (!_iteratorNormalCompletion && _iterator.return) {
                _iterator.return();
            }
        } finally {
            if (_didIteratorError) {
                throw _iteratorError;
            }
        }
    }

    return target;
});

var createTransform = function createTransform(transforms) {
    return function (name) {
        return transforms.reduce(function (name, fn) {
            return name && fn(name);
        }, name);
    };
};

var rebindAll = (function (target, source) {
    for (var _len = arguments.length, transforms = Array(_len > 2 ? _len - 2 : 0), _key = 2; _key < _len; _key++) {
        transforms[_key - 2] = arguments[_key];
    }

    var transform = createTransform(transforms);
    var _iteratorNormalCompletion = true;
    var _didIteratorError = false;
    var _iteratorError = undefined;

    try {
        for (var _iterator = Object.keys(source)[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
            var name = _step.value;

            var result = transform(name);
            if (result) {
                target[result] = createReboundMethod(target, source, name);
            }
        }
    } catch (err) {
        _didIteratorError = true;
        _iteratorError = err;
    } finally {
        try {
            if (!_iteratorNormalCompletion && _iterator.return) {
                _iterator.return();
            }
        } finally {
            if (_didIteratorError) {
                throw _iteratorError;
            }
        }
    }

    return target;
});

var regexify = (function (strsOrRegexes) {
    return strsOrRegexes.map(function (strOrRegex) {
        return typeof strOrRegex === 'string' ? new RegExp('^' + strOrRegex + '$') : strOrRegex;
    });
});

var exclude = (function () {
    for (var _len = arguments.length, exclusions = Array(_len), _key = 0; _key < _len; _key++) {
        exclusions[_key] = arguments[_key];
    }

    exclusions = regexify(exclusions);
    return function (name) {
        return exclusions.every(function (exclusion) {
            return !exclusion.test(name);
        }) && name;
    };
});

var includeMap = (function (mappings) {
  return function (name) {
    return mappings[name];
  };
});

var functor = (function (d) {
  return typeof d === 'function' ? d : function () {
    return d;
  };
});

// Checks that passes properties are 'defined', meaning that calling them with (d, i) returns non null values
function defined$1() {
    var outerArguments = arguments;
    return function (d, i) {
        for (var c = 0, j = outerArguments.length; c < j; c++) {
            if (outerArguments[c](d, i) == null) {
                return false;
            }
        }
        return true;
    };
}

// determines the offset required along the cross scale based
// on the series alignment
var alignOffset = (function (align, width) {
    switch (align) {
        case 'left':
            return width / 2;
        case 'right':
            return -width / 2;
        default:
            return 0;
    }
});

var createBase = (function (initialValues) {

    var env = Object.assign({}, initialValues);
    var base = function base() {};

    Object.keys(env).forEach(function (key) {
        base[key] = function () {
            if (!arguments.length) {
                return env[key];
            }
            env[key] = arguments.length <= 0 ? undefined : arguments[0];
            return base;
        };
    });

    return base;
});

var xyBase = (function () {

    var baseValue = function baseValue() {
        return 0;
    };
    var crossValue = function crossValue(d) {
        return d.x;
    };
    var mainValue = function mainValue(d) {
        return d.y;
    };
    var align = 'center';
    var bandwidth = function bandwidth() {
        return 5;
    };
    var orient = 'vertical';

    var base = createBase({
        decorate: function decorate() {},
        defined: function defined(d, i) {
            return defined$1(baseValue, crossValue, mainValue)(d, i);
        },
        xScale: d3Scale.scaleIdentity(),
        yScale: d3Scale.scaleIdentity()
    });

    base.values = function (d, i) {
        var width = bandwidth(d, i);
        var offset = alignOffset(align, width);
        var xScale = base.xScale();
        var yScale = base.yScale();

        if (orient === 'vertical') {
            var y = yScale(mainValue(d, i), i);
            var y0 = yScale(baseValue(d, i), i);
            var x = xScale(crossValue(d, i), i) + offset;
            return {
                d: d,
                x: x,
                y: y,
                y0: y0,
                width: width,
                height: y - y0,
                origin: [x, y],
                baseOrigin: [x, y0],
                transposedX: x,
                transposedY: y
            };
        } else {
            var _y = xScale(mainValue(d, i), i);
            var _y2 = xScale(baseValue(d, i), i);
            var _x = yScale(crossValue(d, i), i) + offset;
            return {
                d: d,
                x: _x,
                y: _y,
                y0: _y2,
                width: width,
                height: _y - _y2,
                origin: [_y, _x],
                baseOrigin: [_y2, _x],
                transposedX: _y,
                transposedY: _x
            };
        }
    };

    base.baseValue = function () {
        if (!arguments.length) {
            return baseValue;
        }
        baseValue = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return base;
    };
    base.crossValue = function () {
        if (!arguments.length) {
            return crossValue;
        }
        crossValue = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return base;
    };
    base.mainValue = function () {
        if (!arguments.length) {
            return mainValue;
        }
        mainValue = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return base;
    };
    base.bandwidth = function () {
        if (!arguments.length) {
            return bandwidth;
        }
        bandwidth = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return base;
    };
    base.align = function () {
        if (!arguments.length) {
            return align;
        }
        align = arguments.length <= 0 ? undefined : arguments[0];
        return base;
    };
    base.orient = function () {
        if (!arguments.length) {
            return orient;
        }
        orient = arguments.length <= 0 ? undefined : arguments[0];
        return base;
    };

    return base;
});

var red = '#c60';
var green = '#6c0';
var black = '#000';
var gray = '#ddd';
var darkGray = '#999';

var colors = {
    red: red,
    green: green,
    black: black,
    gray: gray,
    darkGray: darkGray
};

var line$1 = (function () {
    var base = xyBase();

    var lineData = d3Shape.line().x(function (d, i) {
        return base.values(d, i).transposedX;
    }).y(function (d, i) {
        return base.values(d, i).transposedY;
    });

    var join = dataJoin('path', 'line');

    var line$$1 = function line$$1(selection$$1) {

        if (selection$$1.selection) {
            join.transition(selection$$1);
        }

        lineData.defined(base.defined());

        selection$$1.each(function (data, index, group) {
            var path$$1 = join(d3Selection.select(group[index]), [data]);

            path$$1.enter().attr('fill', 'none').attr('stroke', colors.black);

            path$$1.attr('d', lineData);

            base.decorate()(path$$1, data, index);
        });
    };

    rebindAll(line$$1, base, exclude('baseValue', 'bandwidth', 'align'));
    rebind(line$$1, join, 'key');
    rebind(line$$1, lineData, 'curve');

    return line$$1;
});

var line$2 = (function () {
    var base = xyBase();

    var lineData = d3Shape.line().x(function (d, i) {
        return base.values(d, i).transposedX;
    }).y(function (d, i) {
        return base.values(d, i).transposedY;
    });

    var line$$1 = function line$$1(data) {
        var context = lineData.context();

        context.beginPath();
        lineData.defined(base.defined())(data);
        context.strokeStyle = colors.black;
        context.fillStyle = 'transparent';

        base.decorate()(context, data);

        context.fill();
        context.stroke();
        context.closePath();
    };

    rebindAll(line$$1, base, exclude('baseValue', 'bandwidth', 'align'));
    rebind(line$$1, lineData, 'curve', 'context');

    return line$$1;
});

var point = (function () {
    var symbol$$1 = d3Shape.symbol();

    var base = xyBase();

    var join = dataJoin('g', 'point');

    var containerTransform = function containerTransform(origin) {
        return 'translate(' + origin[0] + ', ' + origin[1] + ')';
    };

    var point = function point(selection$$1) {

        if (selection$$1.selection) {
            join.transition(selection$$1);
        }

        selection$$1.each(function (data, index, group) {

            var filteredData = data.filter(base.defined());

            var g = join(d3Selection.select(group[index]), filteredData);
            g.enter().attr('transform', function (d, i) {
                return containerTransform(base.values(d, i).origin);
            }).attr('fill', colors.gray).attr('stroke', colors.black).append('path');

            g.attr('transform', function (d, i) {
                return containerTransform(base.values(d, i).origin);
            }).select('path').attr('d', symbol$$1);

            base.decorate()(g, data, index);
        });
    };

    rebindAll(point, base, exclude('baseValue', 'bandwidth', 'align'));
    rebind(point, join, 'key');
    rebind(point, symbol$$1, 'type', 'size');

    return point;
});

var point$1 = (function () {

    var symbol$$1 = d3Shape.symbol();

    var base = xyBase();

    var point = function point(data) {
        var filteredData = data.filter(base.defined());
        var context = symbol$$1.context();

        filteredData.forEach(function (d, i) {
            context.save();

            var values = base.values(d, i);
            context.translate(values.origin[0], values.origin[1]);
            context.beginPath();

            symbol$$1(data);

            context.strokeStyle = colors.black;
            context.fillStyle = colors.gray;

            base.decorate()(context, d, i);

            context.fill();
            context.stroke();
            context.closePath();

            context.restore();
        });
    };

    rebindAll(point, base, exclude('baseValue', 'bandwidth', 'align'));
    rebind(point, symbol$$1, 'size', 'type', 'context');

    return point;
});

var functor$1 = (function (v) {
  return typeof v === 'function' ? v : function () {
    return v;
  };
});

// Renders an OHLC as an SVG path based on the given array of datapoints. Each
// OHLC has a fixed width, whilst the x, open, high, low and close positions are
// obtained from each point via the supplied accessor functions.
var shapeOhlc = (function () {

    var context = null;
    var x = function x(d) {
        return d.date;
    };
    var open = function open(d) {
        return d.open;
    };
    var high = function high(d) {
        return d.high;
    };
    var low = function low(d) {
        return d.low;
    };
    var close = function close(d) {
        return d.close;
    };
    var orient = 'vertical';
    var width = functor$1(3);

    var ohlc = function ohlc(data) {

        var drawingContext = context || d3Path.path();

        data.forEach(function (d, i) {
            var xValue = x(d, i);
            var yOpen = open(d, i);
            var yHigh = high(d, i);
            var yLow = low(d, i);
            var yClose = close(d, i);
            var halfWidth = width(d, i) / 2;

            if (orient === 'vertical') {
                drawingContext.moveTo(xValue, yLow);
                drawingContext.lineTo(xValue, yHigh);

                drawingContext.moveTo(xValue, yOpen);
                drawingContext.lineTo(xValue - halfWidth, yOpen);
                drawingContext.moveTo(xValue, yClose);
                drawingContext.lineTo(xValue + halfWidth, yClose);
            } else {
                drawingContext.moveTo(yLow, xValue);
                drawingContext.lineTo(yHigh, xValue);

                drawingContext.moveTo(yOpen, xValue);
                drawingContext.lineTo(yOpen, xValue + halfWidth);
                drawingContext.moveTo(yClose, xValue);
                drawingContext.lineTo(yClose, xValue - halfWidth);
            }
        });

        return context ? null : drawingContext.toString();
    };

    ohlc.context = function () {
        if (!arguments.length) {
            return context;
        }
        context = arguments.length <= 0 ? undefined : arguments[0];
        return ohlc;
    };
    ohlc.x = function () {
        if (!arguments.length) {
            return x;
        }
        x = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return ohlc;
    };
    ohlc.open = function () {
        if (!arguments.length) {
            return open;
        }
        open = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return ohlc;
    };
    ohlc.high = function () {
        if (!arguments.length) {
            return high;
        }
        high = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return ohlc;
    };
    ohlc.low = function () {
        if (!arguments.length) {
            return low;
        }
        low = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return ohlc;
    };
    ohlc.close = function () {
        if (!arguments.length) {
            return close;
        }
        close = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return ohlc;
    };
    ohlc.width = function () {
        if (!arguments.length) {
            return width;
        }
        width = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return ohlc;
    };
    ohlc.orient = function () {
        if (!arguments.length) {
            return orient;
        }
        orient = arguments.length <= 0 ? undefined : arguments[0];
        return ohlc;
    };

    return ohlc;
});

// Renders a bar series as an SVG path based on the given array of datapoints. Each
// bar has a fixed width, whilst the x, y and height are obtained from each data
// point via the supplied accessor functions.
var shapeBar = (function () {

    var context = null;
    var x = function x(d) {
        return d.x;
    };
    var y = function y(d) {
        return d.y;
    };
    var horizontalAlign = 'center';
    var verticalAlign = 'center';
    var height = function height(d) {
        return d.height;
    };
    var width = functor$1(3);

    var bar = function bar(data, index) {

        var drawingContext = context || d3Path.path();

        data.forEach(function (d, i) {
            var xValue = x.call(this, d, index || i);
            var yValue = y.call(this, d, index || i);
            var barHeight = height.call(this, d, index || i);
            var barWidth = width.call(this, d, index || i);

            var horizontalOffset = void 0;
            switch (horizontalAlign) {
                case 'left':
                    horizontalOffset = barWidth;
                    break;
                case 'right':
                    horizontalOffset = 0;
                    break;
                case 'center':
                    horizontalOffset = barWidth / 2;
                    break;
                default:
                    throw new Error('Invalid horizontal alignment ' + horizontalAlign);
            }

            var verticalOffset = void 0;
            switch (verticalAlign) {
                case 'bottom':
                    verticalOffset = -barHeight;
                    break;
                case 'top':
                    verticalOffset = 0;
                    break;
                case 'center':
                    verticalOffset = barHeight / 2;
                    break;
                default:
                    throw new Error('Invalid vertical alignment ' + verticalAlign);
            }

            drawingContext.rect(xValue - horizontalOffset, yValue - verticalOffset, barWidth, barHeight);
        }, this);

        return context ? null : drawingContext.toString();
    };

    bar.context = function () {
        if (!arguments.length) {
            return context;
        }
        context = arguments.length <= 0 ? undefined : arguments[0];
        return bar;
    };
    bar.x = function () {
        if (!arguments.length) {
            return x;
        }
        x = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return bar;
    };
    bar.y = function () {
        if (!arguments.length) {
            return y;
        }
        y = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return bar;
    };
    bar.width = function () {
        if (!arguments.length) {
            return width;
        }
        width = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return bar;
    };
    bar.horizontalAlign = function () {
        if (!arguments.length) {
            return horizontalAlign;
        }
        horizontalAlign = arguments.length <= 0 ? undefined : arguments[0];
        return bar;
    };
    bar.height = function () {
        if (!arguments.length) {
            return height;
        }
        height = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return bar;
    };
    bar.verticalAlign = function () {
        if (!arguments.length) {
            return verticalAlign;
        }
        verticalAlign = arguments.length <= 0 ? undefined : arguments[0];
        return bar;
    };

    return bar;
});

// Renders a candlestick as an SVG path based on the given array of datapoints. Each
// candlestick has a fixed width, whilst the x, open, high, low and close positions are
// obtained from each point via the supplied accessor functions.
var shapeCandlestick = (function () {

    var context = null;
    var x = function x(d) {
        return d.date;
    };
    var open = function open(d) {
        return d.open;
    };
    var high = function high(d) {
        return d.high;
    };
    var low = function low(d) {
        return d.low;
    };
    var close = function close(d) {
        return d.close;
    };
    var width = functor$1(3);

    var candlestick = function candlestick(data) {

        var drawingContext = context || d3Path.path();

        data.forEach(function (d, i) {
            var xValue = x(d, i);
            var yOpen = open(d, i);
            var yHigh = high(d, i);
            var yLow = low(d, i);
            var yClose = close(d, i);
            var barWidth = width(d, i);
            var halfBarWidth = barWidth / 2;

            // Body
            drawingContext.rect(xValue - halfBarWidth, yOpen, barWidth, yClose - yOpen);
            // High wick
            // // Move to the max price of close or open; draw the high wick
            // N.B. Math.min() is used as we're dealing with pixel values,
            // the lower the pixel value, the higher the price!
            drawingContext.moveTo(xValue, Math.min(yClose, yOpen));
            drawingContext.lineTo(xValue, yHigh);
            // Low wick
            // // Move to the min price of close or open; draw the low wick
            // N.B. Math.max() is used as we're dealing with pixel values,
            // the higher the pixel value, the lower the price!
            drawingContext.moveTo(xValue, Math.max(yClose, yOpen));
            drawingContext.lineTo(xValue, yLow);
        });

        return context ? null : drawingContext.toString();
    };

    candlestick.context = function () {
        if (!arguments.length) {
            return context;
        }
        context = arguments.length <= 0 ? undefined : arguments[0];
        return candlestick;
    };
    candlestick.x = function () {
        if (!arguments.length) {
            return x;
        }
        x = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return candlestick;
    };
    candlestick.open = function () {
        if (!arguments.length) {
            return open;
        }
        open = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return candlestick;
    };
    candlestick.high = function () {
        if (!arguments.length) {
            return high;
        }
        high = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return candlestick;
    };
    candlestick.low = function () {
        if (!arguments.length) {
            return low;
        }
        low = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return candlestick;
    };
    candlestick.close = function () {
        if (!arguments.length) {
            return close;
        }
        close = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return candlestick;
    };
    candlestick.width = function () {
        if (!arguments.length) {
            return width;
        }
        width = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return candlestick;
    };

    return candlestick;
});

// Renders a box plot series as an SVG path based on the given array of datapoints.
var shapeBoxPlot = (function () {

    var context = null;
    var value = function value(d) {
        return d.value;
    };
    var median = function median(d) {
        return d.median;
    };
    var upperQuartile = function upperQuartile(d) {
        return d.upperQuartile;
    };
    var lowerQuartile = function lowerQuartile(d) {
        return d.lowerQuartile;
    };
    var high = function high(d) {
        return d.high;
    };
    var low = function low(d) {
        return d.low;
    };
    var orient = 'vertical';
    var width = functor$1(5);
    var cap = functor$1(0.5);

    var boxPlot = function boxPlot(data) {

        var drawingContext = context || d3Path.path();

        data.forEach(function (d, i) {
            // naming convention is for vertical orientation
            var _value = value(d, i);
            var _width = width(d, i);
            var halfWidth = _width / 2;
            var capWidth = _width * cap(d, i);
            var halfCapWidth = capWidth / 2;
            var _high = high(d, i);
            var _upperQuartile = upperQuartile(d, i);
            var _median = median(d, i);
            var _lowerQuartile = lowerQuartile(d, i);
            var _low = low(d, i);
            var upperQuartileToLowerQuartile = _lowerQuartile - _upperQuartile;

            if (orient === 'vertical') {
                // Upper whisker
                drawingContext.moveTo(_value - halfCapWidth, _high);
                drawingContext.lineTo(_value + halfCapWidth, _high);
                drawingContext.moveTo(_value, _high);
                drawingContext.lineTo(_value, _upperQuartile);

                // Box
                drawingContext.rect(_value - halfWidth, _upperQuartile, _width, upperQuartileToLowerQuartile);
                drawingContext.moveTo(_value - halfWidth, _median);
                // Median line
                drawingContext.lineTo(_value + halfWidth, _median);

                // Lower whisker
                drawingContext.moveTo(_value, _lowerQuartile);
                drawingContext.lineTo(_value, _low);
                drawingContext.moveTo(_value - halfCapWidth, _low);
                drawingContext.lineTo(_value + halfCapWidth, _low);
            } else {
                // Lower whisker
                drawingContext.moveTo(_low, _value - halfCapWidth);
                drawingContext.lineTo(_low, _value + halfCapWidth);
                drawingContext.moveTo(_low, _value);
                drawingContext.lineTo(_lowerQuartile, _value);

                // Box
                drawingContext.rect(_lowerQuartile, _value - halfWidth, -upperQuartileToLowerQuartile, _width);
                drawingContext.moveTo(_median, _value - halfWidth);
                drawingContext.lineTo(_median, _value + halfWidth);

                // Upper whisker
                drawingContext.moveTo(_upperQuartile, _value);
                drawingContext.lineTo(_high, _value);
                drawingContext.moveTo(_high, _value - halfCapWidth);
                drawingContext.lineTo(_high, _value + halfCapWidth);
            }
        });

        return context ? null : drawingContext.toString();
    };

    boxPlot.context = function () {
        if (!arguments.length) {
            return context;
        }
        context = arguments.length <= 0 ? undefined : arguments[0];
        return boxPlot;
    };
    boxPlot.value = function () {
        if (!arguments.length) {
            return value;
        }
        value = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return boxPlot;
    };
    boxPlot.median = function () {
        if (!arguments.length) {
            return median;
        }
        median = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return boxPlot;
    };
    boxPlot.upperQuartile = function () {
        if (!arguments.length) {
            return upperQuartile;
        }
        upperQuartile = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return boxPlot;
    };
    boxPlot.lowerQuartile = function () {
        if (!arguments.length) {
            return lowerQuartile;
        }
        lowerQuartile = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return boxPlot;
    };
    boxPlot.high = function () {
        if (!arguments.length) {
            return high;
        }
        high = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return boxPlot;
    };
    boxPlot.low = function () {
        if (!arguments.length) {
            return low;
        }
        low = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return boxPlot;
    };
    boxPlot.width = function () {
        if (!arguments.length) {
            return width;
        }
        width = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return boxPlot;
    };
    boxPlot.orient = function () {
        if (!arguments.length) {
            return orient;
        }
        orient = arguments.length <= 0 ? undefined : arguments[0];
        return boxPlot;
    };
    boxPlot.cap = function () {
        if (!arguments.length) {
            return cap;
        }
        cap = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return boxPlot;
    };

    return boxPlot;
});

// Renders an error bar series as an SVG path based on the given array of datapoints.
var shapeErrorBar = (function () {

    var context = null;
    var value = function value(d) {
        return d.x;
    };
    var high = function high(d) {
        return d.high;
    };
    var low = function low(d) {
        return d.low;
    };
    var orient = 'vertical';
    var width = functor$1(5);

    var errorBar = function errorBar(data) {

        var drawingContext = context || d3Path.path();

        data.forEach(function (d, i) {
            // naming convention is for vertical orientation
            var _value = value(d, i);
            var _width = width(d, i);
            var halfWidth = _width / 2;
            var _high = high(d, i);
            var _low = low(d, i);

            if (orient === 'vertical') {
                drawingContext.moveTo(_value - halfWidth, _high);
                drawingContext.lineTo(_value + halfWidth, _high);
                drawingContext.moveTo(_value, _high);
                drawingContext.lineTo(_value, _low);
                drawingContext.moveTo(_value - halfWidth, _low);
                drawingContext.lineTo(_value + halfWidth, _low);
            } else {
                drawingContext.moveTo(_low, _value - halfWidth);
                drawingContext.lineTo(_low, _value + halfWidth);
                drawingContext.moveTo(_low, _value);
                drawingContext.lineTo(_high, _value);
                drawingContext.moveTo(_high, _value - halfWidth);
                drawingContext.lineTo(_high, _value + halfWidth);
            }
        });

        return context ? null : drawingContext.toString();
    };

    errorBar.context = function () {
        if (!arguments.length) {
            return context;
        }
        context = arguments.length <= 0 ? undefined : arguments[0];
        return errorBar;
    };
    errorBar.value = function () {
        if (!arguments.length) {
            return value;
        }
        value = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return errorBar;
    };
    errorBar.high = function () {
        if (!arguments.length) {
            return high;
        }
        high = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return errorBar;
    };
    errorBar.low = function () {
        if (!arguments.length) {
            return low;
        }
        low = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return errorBar;
    };
    errorBar.width = function () {
        if (!arguments.length) {
            return width;
        }
        width = functor$1(arguments.length <= 0 ? undefined : arguments[0]);
        return errorBar;
    };
    errorBar.orient = function () {
        if (!arguments.length) {
            return orient;
        }
        orient = arguments.length <= 0 ? undefined : arguments[0];
        return errorBar;
    };

    return errorBar;
});

var bar = (function () {

    var pathGenerator = shapeBar().x(0).y(0);

    var base = xyBase();

    var join = dataJoin('g', 'bar');

    var valueAxisDimension = function valueAxisDimension(generator) {
        return base.orient() === 'vertical' ? generator.height : generator.width;
    };

    var crossAxisDimension = function crossAxisDimension(generator) {
        return base.orient() === 'vertical' ? generator.width : generator.height;
    };

    var translation = function translation(origin) {
        return 'translate(' + origin[0] + ', ' + origin[1] + ')';
    };

    var bar = function bar(selection$$1) {

        if (selection$$1.selection) {
            join.transition(selection$$1);
        }

        selection$$1.each(function (data, index, group) {

            var orient = base.orient();
            if (orient !== 'vertical' && orient !== 'horizontal') {
                throw new Error('The bar series does not support an orientation of ' + orient);
            }

            var filteredData = data.filter(base.defined());
            var projectedData = filteredData.map(base.values);

            pathGenerator.width(0).height(0);

            if (base.orient() === 'vertical') {
                pathGenerator.verticalAlign('top');
                pathGenerator.horizontalAlign('center');
            } else {
                pathGenerator.horizontalAlign('right');
                pathGenerator.verticalAlign('center');
            }

            var g = join(d3Selection.select(group[index]), filteredData);

            // within the enter selection the pathGenerator creates a zero
            // height bar on the baseline. As a result, when used with a transition the bar grows
            // from y0 to y1 (y)
            g.enter().attr('transform', function (_, i) {
                return translation(projectedData[i].baseOrigin);
            }).attr('class', 'bar ' + base.orient()).attr('fill', colors.darkGray).append('path').attr('d', function (d, i) {
                crossAxisDimension(pathGenerator)(projectedData[i].width);
                return pathGenerator([d]);
            });

            // the container translation sets the origin to the 'tip'
            // of each bar as per the decorate pattern
            g.attr('transform', function (_, i) {
                return translation(projectedData[i].origin);
            }).select('path').attr('d', function (d, i) {
                crossAxisDimension(pathGenerator)(projectedData[i].width);
                valueAxisDimension(pathGenerator)(-projectedData[i].height);
                return pathGenerator([d]);
            });

            base.decorate()(g, filteredData, index);
        });
    };

    rebindAll(bar, base);
    rebind(bar, join, 'key');

    return bar;
});

var bar$1 = (function () {
    var base = xyBase();

    var pathGenerator = shapeBar().x(0).y(0);

    var valueAxisDimension = function valueAxisDimension(generator) {
        return base.orient() === 'vertical' ? generator.height : generator.width;
    };

    var crossAxisDimension = function crossAxisDimension(generator) {
        return base.orient() === 'vertical' ? generator.width : generator.height;
    };

    var bar = function bar(data) {
        var context = pathGenerator.context();

        var filteredData = data.filter(base.defined());
        var projectedData = filteredData.map(base.values);

        if (base.orient() === 'vertical') {
            pathGenerator.verticalAlign('top');
            pathGenerator.horizontalAlign('center');
        } else {
            pathGenerator.horizontalAlign('right');
            pathGenerator.verticalAlign('center');
        }

        projectedData.forEach(function (datum, i) {
            context.save();
            context.beginPath();
            context.translate(datum.origin[0], datum.origin[1]);

            valueAxisDimension(pathGenerator)(-datum.height);
            crossAxisDimension(pathGenerator)(datum.width);
            pathGenerator([datum]);

            context.fillStyle = colors.darkGray;
            context.strokeStyle = 'transparent';
            base.decorate()(context, datum.d, i);
            context.fill();
            context.stroke();

            context.closePath();
            context.restore();
        });
    };

    rebindAll(bar, base);
    rebind(bar, pathGenerator, 'context');

    return bar;
});

var errorBarBase = (function () {

    var highValue = function highValue(d) {
        return d.high;
    };
    var lowValue = function lowValue(d) {
        return d.low;
    };
    var crossValue = function crossValue(d) {
        return d.cross;
    };
    var orient = 'vertical';
    var align = 'center';
    var bandwidth = function bandwidth() {
        return 5;
    };

    var base = createBase({
        decorate: function decorate() {},
        defined: function defined(d, i) {
            return defined$1(lowValue, highValue, crossValue)(d, i);
        },
        xScale: d3Scale.scaleIdentity(),
        yScale: d3Scale.scaleIdentity()
    });

    base.values = function (d, i) {
        var width = bandwidth(d, i);
        var offset = alignOffset(align, width);
        var xScale = base.xScale();
        var yScale = base.yScale();

        if (orient === 'vertical') {
            var y = yScale(highValue(d, i));
            return {
                origin: [xScale(crossValue(d, i)) + offset, y],
                high: 0,
                low: yScale(lowValue(d, i)) - y,
                width: width
            };
        } else {
            var x = xScale(lowValue(d, i));
            return {
                origin: [x, yScale(crossValue(d, i)) + offset],
                high: xScale(highValue(d, i)) - x,
                low: 0,
                width: width
            };
        }
    };

    base.orient = function () {
        if (!arguments.length) {
            return orient;
        }
        orient = arguments.length <= 0 ? undefined : arguments[0];
        return base;
    };
    base.lowValue = function () {
        if (!arguments.length) {
            return lowValue;
        }
        lowValue = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return base;
    };
    base.highValue = function () {
        if (!arguments.length) {
            return highValue;
        }
        highValue = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return base;
    };
    base.crossValue = function () {
        if (!arguments.length) {
            return crossValue;
        }
        crossValue = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return base;
    };
    base.bandwidth = function () {
        if (!arguments.length) {
            return bandwidth;
        }
        bandwidth = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return base;
    };
    base.align = function () {
        if (!arguments.length) {
            return align;
        }
        align = arguments.length <= 0 ? undefined : arguments[0];
        return base;
    };

    return base;
});

var errorBar = (function () {

    var base = errorBarBase();

    var join = dataJoin('g', 'error-bar');

    var pathGenerator = shapeErrorBar().value(0);

    var propagateTransition = function propagateTransition(maybeTransition) {
        return function (selection$$1) {
            return maybeTransition.selection ? selection$$1.transition(maybeTransition) : selection$$1;
        };
    };

    var containerTranslation = function containerTranslation(values) {
        return 'translate(' + values.origin[0] + ', ' + values.origin[1] + ')';
    };

    var errorBar = function errorBar(selection$$1) {

        if (selection$$1.selection) {
            join.transition(selection$$1);
        }

        var transitionPropagator = propagateTransition(selection$$1);

        selection$$1.each(function (data, index, group) {

            var filteredData = data.filter(base.defined());
            var projectedData = filteredData.map(base.values);
            var g = join(d3Selection.select(group[index]), filteredData);

            g.enter().attr('stroke', colors.black).attr('fill', colors.gray).attr('transform', function (d, i) {
                return containerTranslation(base.values(d, i)) + ' scale(1e-6, 1)';
            }).append('path');

            pathGenerator.orient(base.orient());

            g.each(function (d, i, g) {
                var values = projectedData[i];
                pathGenerator.high(values.high).low(values.low).width(values.width);

                transitionPropagator(d3Selection.select(g[i])).attr('transform', containerTranslation(values) + ' scale(1)').select('path').attr('d', pathGenerator([d]));
            });

            base.decorate()(g, data, index);
        });
    };

    rebindAll(errorBar, base);
    rebind(errorBar, join, 'key');

    return errorBar;
});

var errorBar$1 = (function () {

    var base = errorBarBase();

    var pathGenerator = shapeErrorBar().value(0);

    var errorBar = function errorBar(data) {
        var filteredData = data.filter(base.defined());
        var context = pathGenerator.context();

        pathGenerator.orient(base.orient());

        filteredData.forEach(function (d, i) {
            context.save();

            var values = base.values(d, i);
            context.translate(values.origin[0], values.origin[1]);
            context.beginPath();

            pathGenerator.high(values.high).width(values.width).low(values.low)([d]);

            context.strokeStyle = colors.black;
            context.fillStyle = colors.gray;

            base.decorate()(context, d, i);

            context.fill();
            context.stroke();
            context.closePath();

            context.restore();
        });
    };

    rebindAll(errorBar, base);
    rebind(errorBar, pathGenerator, 'context');

    return errorBar;
});

var area$1 = (function () {
    var base = xyBase();

    var areaData = d3Shape.area();

    var join = dataJoin('path', 'area');

    var area$$1 = function area$$1(selection$$1) {

        if (selection$$1.selection) {
            join.transition(selection$$1);
        }

        areaData.defined(base.defined());

        selection$$1.each(function (data, index, group) {

            var projectedData = data.map(base.values);
            areaData.x(function (_, i) {
                return projectedData[i].transposedX;
            }).y(function (_, i) {
                return projectedData[i].transposedY;
            });

            var valueComponent = base.orient() === 'vertical' ? 'y' : 'x';
            areaData[valueComponent + '0'](function (_, i) {
                return projectedData[i].y0;
            });
            areaData[valueComponent + '1'](function (_, i) {
                return projectedData[i].y;
            });

            var path$$1 = join(d3Selection.select(group[index]), [data]);

            path$$1.enter().attr('fill', colors.gray);

            path$$1.attr('d', areaData);

            base.decorate()(path$$1, data, index);
        });
    };

    rebindAll(area$$1, base, exclude('bandwidth', 'align'));
    rebind(area$$1, join, 'key');
    rebind(area$$1, areaData, 'curve');

    return area$$1;
});

var area$2 = (function () {
    var base = xyBase();

    var areaData = d3Shape.area();

    var area$$1 = function area$$1(data) {
        var context = areaData.context();

        areaData.defined(base.defined());

        var projectedData = data.map(base.values);
        areaData.x(function (_, i) {
            return projectedData[i].transposedX;
        }).y(function (_, i) {
            return projectedData[i].transposedY;
        });

        var valueComponent = base.orient() === 'vertical' ? 'y' : 'x';
        areaData[valueComponent + '0'](function (_, i) {
            return projectedData[i].y0;
        });
        areaData[valueComponent + '1'](function (_, i) {
            return projectedData[i].y;
        });

        context.beginPath();
        areaData(data);
        context.fillStyle = colors.gray;
        context.strokeStyle = 'transparent';

        base.decorate()(context, data);

        context.fill();
        context.stroke();
        context.closePath();
    };

    rebindAll(area$$1, base, exclude('bandwidth', 'align'));
    rebind(area$$1, areaData, 'curve', 'context');

    return area$$1;
});

var ohlcBase$1 = (function () {

    var base = void 0;
    var crossValue = function crossValue(d) {
        return d.date;
    };
    var openValue = function openValue(d) {
        return d.open;
    };
    var highValue = function highValue(d) {
        return d.high;
    };
    var lowValue = function lowValue(d) {
        return d.low;
    };
    var closeValue = function closeValue(d) {
        return d.close;
    };
    var bandwidth = function bandwidth() {
        return 5;
    };
    var align = 'center';
    var crossValueScaled = function crossValueScaled(d, i) {
        return base.xScale()(crossValue(d, i));
    };

    base = createBase({
        decorate: function decorate() {},
        defined: function defined(d, i) {
            return defined$1(crossValue, openValue, lowValue, highValue, closeValue)(d, i);
        },
        xScale: d3Scale.scaleIdentity(),
        yScale: d3Scale.scaleIdentity()
    });

    base.values = function (d, i) {
        var closeRaw = closeValue(d, i);
        var openRaw = openValue(d, i);
        var width = bandwidth(d, i);
        var offset = alignOffset(align, width);

        var direction = '';
        if (closeRaw > openRaw) {
            direction = 'up';
        } else if (closeRaw < openRaw) {
            direction = 'down';
        }

        return {
            cross: crossValueScaled(d, i) + offset,
            open: base.yScale()(openRaw),
            high: base.yScale()(highValue(d, i)),
            low: base.yScale()(lowValue(d, i)),
            close: base.yScale()(closeRaw),
            width: width,
            direction: direction
        };
    };

    base.crossValue = function () {
        if (!arguments.length) {
            return crossValue;
        }
        crossValue = arguments.length <= 0 ? undefined : arguments[0];
        return base;
    };
    base.openValue = function () {
        if (!arguments.length) {
            return openValue;
        }
        openValue = arguments.length <= 0 ? undefined : arguments[0];
        return base;
    };
    base.highValue = function () {
        if (!arguments.length) {
            return highValue;
        }
        highValue = arguments.length <= 0 ? undefined : arguments[0];
        return base;
    };
    base.lowValue = function () {
        if (!arguments.length) {
            return lowValue;
        }
        lowValue = arguments.length <= 0 ? undefined : arguments[0];
        return base;
    };
    base.yValue = base.closeValue = function () {
        if (!arguments.length) {
            return closeValue;
        }
        closeValue = arguments.length <= 0 ? undefined : arguments[0];
        return base;
    };
    base.bandwidth = function () {
        if (!arguments.length) {
            return bandwidth;
        }
        bandwidth = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return base;
    };
    base.align = function () {
        if (!arguments.length) {
            return align;
        }
        align = arguments.length <= 0 ? undefined : arguments[0];
        return base;
    };

    return base;
});

var ohlcBase = (function (pathGenerator, seriesName) {
    var base = ohlcBase$1();
    var join = dataJoin('g', seriesName);
    var containerTranslation = function containerTranslation(values) {
        return 'translate(' + values.cross + ', ' + values.high + ')';
    };

    var propagateTransition = function propagateTransition(maybeTransition) {
        return function (selection$$1) {
            return maybeTransition.selection ? selection$$1.transition(maybeTransition) : selection$$1;
        };
    };

    var candlestick = function candlestick(selection$$1) {

        if (selection$$1.selection) {
            join.transition(selection$$1);
        }

        var transitionPropagator = propagateTransition(selection$$1);

        selection$$1.each(function (data, index, group) {

            var filteredData = data.filter(base.defined());

            var g = join(d3Selection.select(group[index]), filteredData);

            g.enter().attr('transform', function (d, i) {
                return containerTranslation(base.values(d, i)) + ' scale(1e-6, 1)';
            }).append('path');

            g.each(function (d, i, g) {

                var values = base.values(d, i);
                var color = values.direction === 'up' ? colors.green : colors.red;

                var singleCandlestick = transitionPropagator(d3Selection.select(g[i])).attr('class', seriesName + ' ' + values.direction).attr('stroke', color).attr('fill', color).attr('transform', function () {
                    return containerTranslation(values) + ' scale(1)';
                });

                pathGenerator.x(0).width(values.width).open(function () {
                    return values.open - values.high;
                }).high(0).low(function () {
                    return values.low - values.high;
                }).close(function () {
                    return values.close - values.high;
                });

                singleCandlestick.select('path').attr('d', pathGenerator([d]));
            });

            base.decorate()(g, data, index);
        });
    };

    rebind(candlestick, join, 'key');
    rebindAll(candlestick, base);

    return candlestick;
});

var candlestick = (function () {
  return ohlcBase(shapeCandlestick(), 'candlestick');
});

var ohlcBase$2 = (function (pathGenerator) {

    var base = ohlcBase$1();

    var candlestick = function candlestick(data) {
        var filteredData = data.filter(base.defined());
        var context = pathGenerator.context();

        filteredData.forEach(function (d, i) {
            context.save();

            var values = base.values(d, i);
            context.translate(values.cross, values.high);
            context.beginPath();

            pathGenerator.x(0).open(function () {
                return values.open - values.high;
            }).width(values.width).high(0).low(function () {
                return values.low - values.high;
            }).close(function () {
                return values.close - values.high;
            })([d]);

            var color = values.direction === 'up' ? colors.green : colors.red;
            context.strokeStyle = color;
            context.fillStyle = color;

            base.decorate()(context, d, i);

            context.fill();
            context.stroke();
            context.closePath();

            context.restore();
        });
    };

    rebind(candlestick, pathGenerator, 'context');
    rebindAll(candlestick, base);

    return candlestick;
});

var candlestick$1 = (function () {
  return ohlcBase$2(shapeCandlestick());
});

var boxPlotBase = (function () {

    var upperQuartileValue = function upperQuartileValue(d) {
        return d.upperQuartile;
    };
    var lowerQuartileValue = function lowerQuartileValue(d) {
        return d.lowerQuartile;
    };
    var highValue = function highValue(d) {
        return d.high;
    };
    var lowValue = function lowValue(d) {
        return d.low;
    };
    var crossValue = function crossValue(d) {
        return d.value;
    };
    var medianValue = function medianValue(d) {
        return d.median;
    };
    var orient = 'vertical';
    var align = 'center';
    var bandwidth = function bandwidth() {
        return 5;
    };

    var base = createBase({
        decorate: function decorate() {},
        defined: function defined(d, i) {
            return defined$1(lowValue, highValue, lowerQuartileValue, upperQuartileValue, crossValue, medianValue)(d, i);
        },
        xScale: d3Scale.scaleIdentity(),
        yScale: d3Scale.scaleIdentity()
    });

    base.values = function (d, i) {
        var width = bandwidth(d, i);
        var offset = alignOffset(align, width);
        var xScale = base.xScale();
        var yScale = base.yScale();

        if (orient === 'vertical') {
            var y = yScale(highValue(d, i));
            return {
                origin: [xScale(crossValue(d, i)) + offset, y],
                high: 0,
                upperQuartile: yScale(upperQuartileValue(d, i)) - y,
                median: yScale(medianValue(d, i)) - y,
                lowerQuartile: yScale(lowerQuartileValue(d, i)) - y,
                low: yScale(lowValue(d, i)) - y,
                width: width
            };
        } else {
            var x = xScale(lowValue(d, i));
            return {
                origin: [x, yScale(crossValue(d, i)) + offset],
                high: xScale(highValue(d, i)) - x,
                upperQuartile: xScale(upperQuartileValue(d, i)) - x,
                median: xScale(medianValue(d, i)) - x,
                lowerQuartile: xScale(lowerQuartileValue(d, i)) - x,
                low: 0,
                width: width
            };
        }
    };

    base.orient = function () {
        if (!arguments.length) {
            return orient;
        }
        orient = arguments.length <= 0 ? undefined : arguments[0];
        return base;
    };
    base.lowerQuartileValue = function () {
        if (!arguments.length) {
            return lowerQuartileValue;
        }
        lowerQuartileValue = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return base;
    };
    base.upperQuartileValue = function () {
        if (!arguments.length) {
            return upperQuartileValue;
        }
        upperQuartileValue = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return base;
    };
    base.lowValue = function () {
        if (!arguments.length) {
            return lowValue;
        }
        lowValue = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return base;
    };
    base.highValue = function () {
        if (!arguments.length) {
            return highValue;
        }
        highValue = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return base;
    };
    base.crossValue = function () {
        if (!arguments.length) {
            return crossValue;
        }
        crossValue = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return base;
    };
    base.medianValue = function () {
        if (!arguments.length) {
            return medianValue;
        }
        medianValue = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return base;
    };
    base.bandwidth = function () {
        if (!arguments.length) {
            return bandwidth;
        }
        bandwidth = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return base;
    };
    base.align = function () {
        if (!arguments.length) {
            return align;
        }
        align = arguments.length <= 0 ? undefined : arguments[0];
        return base;
    };

    return base;
});

var boxPlot = (function () {

    var base = boxPlotBase();

    var join = dataJoin('g', 'box-plot');

    var pathGenerator = shapeBoxPlot().value(0);

    var propagateTransition = function propagateTransition(maybeTransition) {
        return function (selection$$1) {
            return maybeTransition.selection ? selection$$1.transition(maybeTransition) : selection$$1;
        };
    };

    var containerTranslation = function containerTranslation(values) {
        return 'translate(' + values.origin[0] + ', ' + values.origin[1] + ')';
    };

    var boxPlot = function boxPlot(selection$$1) {

        if (selection$$1.selection) {
            join.transition(selection$$1);
        }

        var transitionPropagator = propagateTransition(selection$$1);

        selection$$1.each(function (data, index, group) {

            var filteredData = data.filter(base.defined());
            var g = join(d3Selection.select(group[index]), filteredData);

            g.enter().attr('stroke', colors.black).attr('fill', colors.gray).attr('transform', function (d, i) {
                return containerTranslation(base.values(d, i)) + ' scale(1e-6, 1)';
            }).append('path');

            pathGenerator.orient(base.orient());

            g.each(function (d, i, g) {
                var values = base.values(d, i);
                pathGenerator.median(values.median).upperQuartile(values.upperQuartile).lowerQuartile(values.lowerQuartile).width(values.width).high(values.high).low(values.low);

                transitionPropagator(d3Selection.select(g[i])).attr('transform', containerTranslation(values)).select('path').attr('d', pathGenerator([d]));
            });

            base.decorate()(g, data, index);
        });
    };

    rebindAll(boxPlot, base);
    rebind(boxPlot, join, 'key');
    rebind(boxPlot, pathGenerator, 'cap');

    return boxPlot;
});

var boxPlot$1 = (function () {

    var base = boxPlotBase();

    var pathGenerator = shapeBoxPlot().value(0);

    var boxPlot = function boxPlot(data) {
        var filteredData = data.filter(base.defined());
        var context = pathGenerator.context();

        pathGenerator.orient(base.orient());

        filteredData.forEach(function (d, i) {
            context.save();

            var values = base.values(d, i);
            context.translate(values.origin[0], values.origin[1]);
            context.beginPath();

            pathGenerator.median(values.median).upperQuartile(values.upperQuartile).lowerQuartile(values.lowerQuartile).high(values.high).width(values.width).low(values.low)([d]);

            context.fillStyle = colors.gray;
            context.strokeStyle = colors.black;

            base.decorate()(context, d, i);

            context.fill();
            context.stroke();
            context.closePath();

            context.restore();
        });
    };

    rebindAll(boxPlot, base);
    rebind(boxPlot, pathGenerator, 'cap', 'context');

    return boxPlot;
});

var ohlc = (function () {
  return ohlcBase(shapeOhlc(), 'ohlc');
});

var ohlc$1 = (function () {
  return ohlcBase$2(shapeOhlc());
});

var multiBase = (function () {

    var series = [];
    var mapping = function mapping(d) {
        return d;
    };
    var key = function key(_, i) {
        return i;
    };

    var multi = createBase({
        decorate: function decorate() {},
        xScale: d3Scale.scaleIdentity(),
        yScale: d3Scale.scaleIdentity()
    });

    multi.mapping = function () {
        if (!arguments.length) {
            return mapping;
        }
        mapping = arguments.length <= 0 ? undefined : arguments[0];
        return multi;
    };
    multi.key = function () {
        if (!arguments.length) {
            return key;
        }
        key = arguments.length <= 0 ? undefined : arguments[0];
        return multi;
    };
    multi.series = function () {
        if (!arguments.length) {
            return series;
        }
        series = arguments.length <= 0 ? undefined : arguments[0];
        return multi;
    };

    return multi;
});

var multiSeries = (function () {

    var base = multiBase();

    var innerJoin = dataJoin('g');

    var join = dataJoin('g', 'multi');

    var multi = function multi(selection$$1) {

        if (selection$$1.selection) {
            join.transition(selection$$1);
            innerJoin.transition(selection$$1);
        }

        var mapping = base.mapping();
        var series = base.series();
        var xScale = base.xScale();
        var yScale = base.yScale();

        selection$$1.each(function (data, index, group) {

            var container = join(d3Selection.select(group[index]), series);

            // iterate over the containers, 'call'-ing the series for each
            container.each(function (dataSeries, seriesIndex, seriesGroup) {
                dataSeries.xScale(xScale).yScale(yScale);

                var seriesData = mapping(data, seriesIndex, series);
                var innerContainer = innerJoin(d3Selection.select(seriesGroup[seriesIndex]), [seriesData]);

                innerContainer.call(dataSeries);
            });

            var unwrappedSelection = container.selection ? container.selection() : container;
            unwrappedSelection.order();

            base.decorate()(container, data, index);
        });
    };

    rebindAll(multi, base);
    rebind(multi, join, 'key');

    return multi;
});

var multiSeries$1 = (function () {

    var context = null;
    var base = multiBase();

    var multi = function multi(data) {
        var mapping = base.mapping();
        var series = base.series();
        var xScale = base.xScale();
        var yScale = base.yScale();

        series.forEach(function (dataSeries, index) {
            var seriesData = mapping(data, index, series);
            dataSeries.context(context).xScale(xScale).yScale(yScale);

            var adaptedDecorate = dataSeries.decorate();
            dataSeries.decorate(function (c, d, i) {
                base.decorate()(c, data, index);
                adaptedDecorate(c, d, i);
            });

            dataSeries(seriesData);

            dataSeries.decorate(adaptedDecorate);
        });
    };

    multi.context = function () {
        if (!arguments.length) {
            return context;
        }
        context = arguments.length <= 0 ? undefined : arguments[0];
        return multi;
    };

    rebindAll(multi, base);

    return multi;
});

var groupedBase = (function (series) {

    var bandwidth = function bandwidth() {
        return 50;
    };
    var align = 'center';

    // the offset scale is used to offset each of the series within a group
    var offsetScale = d3Scale.scaleBand();

    var grouped = createBase({
        decorate: function decorate() {},
        xScale: d3Scale.scaleLinear()
    });

    // the bandwidth for the grouped series can be a function of datum / index. As a result
    // the offset scale required to cluster the 'sub' series is also dependent on datum / index.
    // This function computes the offset scale for a specific datum / index of the grouped series
    grouped.offsetScaleForDatum = function (data, d, i) {
        var width = bandwidth(d, i);
        var offset = alignOffset(align, width);

        var halfWidth = width / 2;
        return offsetScale.domain(d3Array.range(0, data.length)).range([-halfWidth + offset, halfWidth + offset]);
    };

    grouped.bandwidth = function () {
        if (!arguments.length) {
            return bandwidth;
        }
        bandwidth = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return grouped;
    };
    grouped.align = function () {
        if (!arguments.length) {
            return align;
        }
        align = arguments.length <= 0 ? undefined : arguments[0];
        return grouped;
    };

    rebindAll(grouped, offsetScale, includeMap({ 'paddingInner': 'paddingOuter' }));

    return grouped;
});

var grouped = (function (series) {

    var base = groupedBase(series);

    var join = dataJoin('g', 'grouped');

    var grouped = function grouped(selection$$1) {

        if (selection$$1.selection) {
            join.transition(selection$$1);
        }

        selection$$1.each(function (data, index, group) {

            var g = join(d3Selection.select(group[index]), data);

            g.enter().append('g');

            g.select('g').each(function (_, index, group) {
                var container = d3Selection.select(group[index]);

                // create a composite scale that applies the required offset
                var compositeScale = function compositeScale(d, i) {
                    var offset = base.offsetScaleForDatum(data, d, i);
                    return base.xScale()(d) + offset(index) + offset.bandwidth() / 2;
                };
                series.xScale(compositeScale);

                // if the sub-series has a bandwidth, set this from the offset scale
                if (series.bandwidth) {
                    series.bandwidth(function (d, i) {
                        return base.offsetScaleForDatum(data, d, i).bandwidth();
                    });
                }

                // adapt the decorate function to give each series the correct index
                series.decorate(function (s, d) {
                    return base.decorate()(s, d, index);
                });

                container.call(series);
            });
        });
    };

    rebindAll(grouped, series, exclude('decorate', 'xScale'));
    rebindAll(grouped, base, exclude('offsetScaleForDatum'));

    return grouped;
});

var grouped$1 = function (series) {

    var base = groupedBase(series);

    var grouped = function grouped(data) {
        data.forEach(function (seriesData, index) {

            // create a composite scale that applies the required offset
            var compositeScale = function compositeScale(d, i) {
                var offset = base.offsetScaleForDatum(data, d, i);
                return base.xScale()(d) + offset(index) + offset.bandwidth() / 2;
            };
            series.xScale(compositeScale);

            // if the sub-series has a bandwidth, set this from the offset scale
            if (series.bandwidth) {
                series.bandwidth(function (d, i) {
                    return base.offsetScaleForDatum(data, d, i).bandwidth();
                });
            }

            // adapt the decorate function to give each series the correct index
            series.decorate(function (c, d) {
                return base.decorate()(c, d, index);
            });
            series(seriesData);
        });
    };

    rebindAll(grouped, series, exclude('decorate', 'xScale'));
    rebindAll(grouped, base, exclude('configureOffsetScale', 'configureOffset'));

    return grouped;
};

var repeat = (function () {

    var orient = 'vertical';
    var series = line$1();
    var multi = multiSeries();

    var repeat = function repeat(selection$$1) {
        return selection$$1.each(function (data, index, group) {
            if (orient === 'vertical') {
                multi.series(data[0].map(function (_) {
                    return series;
                })).mapping(function (data, index) {
                    return data.map(function (d) {
                        return d[index];
                    });
                });
            } else {
                multi.series(data.map(function (_) {
                    return series;
                })).mapping(function (data, index) {
                    return data[index];
                });
            }
            d3Selection.select(group[index]).call(multi);
        });
    };

    repeat.series = function () {
        if (!arguments.length) {
            return series;
        }
        series = arguments.length <= 0 ? undefined : arguments[0];
        return repeat;
    };

    repeat.orient = function () {
        if (!arguments.length) {
            return orient;
        }
        orient = arguments.length <= 0 ? undefined : arguments[0];
        return repeat;
    };

    rebindAll(repeat, multi, exclude('series', 'mapping'));

    return repeat;
});

var repeat$1 = (function () {

    var orient = 'vertical';
    var series = line$2();
    var multi = multiSeries$1();

    var repeat = function repeat(data) {
        if (orient === 'vertical') {
            multi.series(data[0].map(function (_) {
                return series;
            })).mapping(function (data, index) {
                return data.map(function (d) {
                    return d[index];
                });
            });
        } else {
            multi.series(data.map(function (_) {
                return series;
            })).mapping(function (data, index) {
                return data[index];
            });
        }
        multi(data);
    };

    repeat.series = function () {
        if (!arguments.length) {
            return series;
        }
        series = arguments.length <= 0 ? undefined : arguments[0];
        return repeat;
    };

    repeat.orient = function () {
        if (!arguments.length) {
            return orient;
        }
        orient = arguments.length <= 0 ? undefined : arguments[0];
        return repeat;
    };

    rebindAll(repeat, multi, exclude('series', 'mapping'));

    return repeat;
});

var toConsumableArray = function (arr) {
  if (Array.isArray(arr)) {
    for (var i = 0, arr2 = Array(arr.length); i < arr.length; i++) arr2[i] = arr[i];

    return arr2;
  } else {
    return Array.from(arr);
  }
};

var sortUnique = function sortUnique(arr) {
    return arr.sort(d3Array.ascending).filter(function (value, index, self) {
        return self.indexOf(value, index + 1) === -1;
    });
};

var autoBandwidth = (function (adaptee) {

    var widthFraction = 0.75;

    // computes the bandwidth as a fraction of the smallest distance between the datapoints
    var computeBandwidth = function computeBandwidth(screenValues) {
        // return some default value if there are not enough datapoints to compute the width
        if (screenValues.length <= 1) {
            return 10;
        }

        screenValues = sortUnique(screenValues);

        // compute the distance between neighbouring items
        var neighbourDistances = d3Array.pairs(screenValues).map(function (tuple) {
            return Math.abs(tuple[0] - tuple[1]);
        });

        var minDistance = d3Array.min(neighbourDistances);
        return widthFraction * minDistance;
    };

    var determineBandwith = function determineBandwith(crossScale, data, accessor) {
        // if the cross-scale has a bandwidth function, i.e. it is a scaleBand, use
        // this to determine the width
        if (crossScale.bandwidth) {
            return crossScale.bandwidth();
        } else {
            var _ref;

            // grouped series expect a nested array, which is flattened out
            var flattenedData = Array.isArray(data) ? (_ref = []).concat.apply(_ref, toConsumableArray(data)) : data;

            // obtain an array of points along the crossValue axis, mapped to screen coordinates.
            var crossValuePoints = flattenedData.filter(adaptee.defined()).map(accessor()).map(crossScale);

            var width = computeBandwidth(crossValuePoints);

            return width;
        }
    };

    var autoBandwidth = function autoBandwidth(arg) {

        var computeWidth = function computeWidth(data) {

            if (adaptee.xBandwidth && adaptee.yBandwidth) {
                adaptee.xBandwidth(determineBandwith(adaptee.xScale(), data, adaptee.xValue));
                adaptee.yBandwidth(determineBandwith(adaptee.yScale(), data, adaptee.yValue));
            } else {
                // if the series has an orient property, use this to determine the cross-scale, otherwise
                // assume it is the x-scale
                var crossScale = adaptee.orient && adaptee.orient() === 'horizontal' ? adaptee.yScale() : adaptee.xScale();

                adaptee.bandwidth(determineBandwith(crossScale, data, adaptee.crossValue));
            }
        };

        if (arg instanceof d3Selection.selection) {
            arg.each(function (data, index, group) {
                computeWidth(data);
                adaptee(d3Selection.select(group[index]));
            });
        } else {
            computeWidth(arg);
            adaptee(arg);
        }
    };

    rebindAll(autoBandwidth, adaptee);

    autoBandwidth.widthFraction = function () {
        if (!arguments.length) {
            return widthFraction;
        }
        widthFraction = arguments.length <= 0 ? undefined : arguments[0];
        return autoBandwidth;
    };

    return autoBandwidth;
});

var heatmapBase = (function () {

    var xValue = function xValue(d) {
        return d.x;
    };
    var yValue = function yValue(d) {
        return d.y;
    };
    var colorValue = function colorValue(d) {
        return d.color;
    };
    var yBandwidth = function yBandwidth() {
        return 5;
    };
    var xBandwidth = function xBandwidth() {
        return 5;
    };
    var colorInterpolate = d3Scale.interpolateViridis;

    var heatmap = createBase({
        decorate: function decorate() {},
        defined: function defined(d, i) {
            return defined$1(xValue, yValue, colorValue)(d, i);
        },
        xScale: d3Scale.scaleIdentity(),
        yScale: d3Scale.scaleIdentity()
    });

    heatmap.pathGenerator = shapeBar().x(0).y(0);

    heatmap.colorScale = function (data) {
        var colorValues = data.map(colorValue);
        // a scale that maps the color values onto a unit range, [0, 1]
        return d3Scale.scaleLinear().domain([d3Array.min(colorValues), d3Array.max(colorValues)]);
    };

    heatmap.values = function (d, i) {
        return {
            x: heatmap.xScale()(xValue(d, i)),
            y: heatmap.yScale()(yValue(d, i)),
            colorValue: colorValue(d, i),
            width: xBandwidth(d, i),
            height: yBandwidth(d, i)
        };
    };

    heatmap.xValue = function () {
        if (!arguments.length) {
            return xValue;
        }
        xValue = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return heatmap;
    };
    heatmap.yValue = function () {
        if (!arguments.length) {
            return yValue;
        }
        yValue = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return heatmap;
    };
    heatmap.colorValue = function () {
        if (!arguments.length) {
            return colorValue;
        }
        colorValue = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return heatmap;
    };
    heatmap.colorInterpolate = function () {
        if (!arguments.length) {
            return colorInterpolate;
        }
        colorInterpolate = arguments.length <= 0 ? undefined : arguments[0];
        return heatmap;
    };
    heatmap.xBandwidth = function () {
        if (!arguments.length) {
            return xBandwidth;
        }
        xBandwidth = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return heatmap;
    };
    heatmap.yBandwidth = function () {
        if (!arguments.length) {
            return yBandwidth;
        }
        yBandwidth = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return heatmap;
    };

    rebindAll(heatmap, heatmap.pathGenerator, includeMap({
        'horizontalAlign': 'xAlign',
        'verticalAlign': 'yAlign'
    }));

    return heatmap;
});

var heatmap = (function () {

    var base = heatmapBase();

    var join = dataJoin('g', 'box');

    var containerTransform = function containerTransform(values) {
        return 'translate(' + values.x + ', ' + values.y + ')';
    };

    var heatmap = function heatmap(selection$$1) {

        selection$$1.each(function (data, index, group) {

            var filteredData = data.filter(base.defined());
            var colorValue = base.colorValue();
            var colorInterpolate = base.colorInterpolate();
            var colorScale = base.colorScale(filteredData);

            var g = join(d3Selection.select(group[index]), filteredData);

            g.enter().append('path').attr('stroke', 'transparent');

            g.attr('transform', function (d, i) {
                return containerTransform(base.values(d, i));
            }).select('path').attr('d', function (d, i) {
                return base.pathGenerator.width(base.values(d, i).width).height(base.values(d, i).height)([d]);
            }).attr('fill', function (d, i) {
                return colorInterpolate(colorScale(colorValue(d, i)));
            });

            base.decorate()(g, data, index);
        });
    };

    rebindAll(heatmap, base);

    return heatmap;
});

var heatmap$1 = (function () {

    var context = null;
    var base = heatmapBase();

    var heatmap = function heatmap(data) {
        var filteredData = data.filter(base.defined());
        var colorValue = base.colorValue();
        var colorInterpolate = base.colorInterpolate();
        var colorScale = base.colorScale(filteredData);
        var context = base.pathGenerator.context();

        filteredData.forEach(function (d, i) {
            context.save();
            context.beginPath();

            var values = base.values(d, i);
            context.translate(values.x, values.y);

            context.fillStyle = colorInterpolate(colorScale(values.colorValue));
            context.strokeStyle = 'transparent';

            base.pathGenerator.height(values.height).width(values.width)([d]);

            base.decorate()(context, d, i);

            context.fill();
            context.stroke();
            context.closePath();
            context.restore();
        });
    };

    rebind(heatmap, base.pathGenerator, 'context');
    rebindAll(heatmap, base);

    return heatmap;
});

exports.seriesSvgLine = line$1;
exports.seriesCanvasLine = line$2;
exports.seriesSvgPoint = point;
exports.seriesCanvasPoint = point$1;
exports.seriesSvgBar = bar;
exports.seriesCanvasBar = bar$1;
exports.seriesSvgErrorBar = errorBar;
exports.seriesCanvasErrorBar = errorBar$1;
exports.seriesSvgArea = area$1;
exports.seriesCanvasArea = area$2;
exports.seriesSvgCandlestick = candlestick;
exports.seriesCanvasCandlestick = candlestick$1;
exports.seriesSvgBoxPlot = boxPlot;
exports.seriesCanvasBoxPlot = boxPlot$1;
exports.seriesSvgOhlc = ohlc;
exports.seriesCanvasOhlc = ohlc$1;
exports.seriesSvgMulti = multiSeries;
exports.seriesCanvasMulti = multiSeries$1;
exports.seriesSvgGrouped = grouped;
exports.seriesCanvasGrouped = grouped$1;
exports.seriesSvgRepeat = repeat;
exports.seriesCanvasRepeat = repeat$1;
exports.autoBandwidth = autoBandwidth;
exports.seriesSvgHeatmap = heatmap;
exports.seriesCanvasHeatmap = heatmap$1;

Object.defineProperty(exports, '__esModule', { value: true });

})));
