(function (global, factory) {
  typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports, require('d3-path')) :
  typeof define === 'function' && define.amd ? define(['exports', 'd3-path'], factory) :
  (factory((global.fc = global.fc || {}),global.d3));
}(this, (function (exports,d3Path) { 'use strict';

var functor = (function (v) {
  return typeof v === 'function' ? v : function () {
    return v;
  };
});

// Renders an OHLC as an SVG path based on the given array of datapoints. Each
// OHLC has a fixed width, whilst the x, open, high, low and close positions are
// obtained from each point via the supplied accessor functions.
var ohlc = (function () {

    var context = null;
    var x = function x(d) {
        return d.date;
    };
    var open = function open(d) {
        return d.open;
    };
    var high = function high(d) {
        return d.high;
    };
    var low = function low(d) {
        return d.low;
    };
    var close = function close(d) {
        return d.close;
    };
    var orient = 'vertical';
    var width = functor(3);

    var ohlc = function ohlc(data) {

        var drawingContext = context || d3Path.path();

        data.forEach(function (d, i) {
            var xValue = x(d, i);
            var yOpen = open(d, i);
            var yHigh = high(d, i);
            var yLow = low(d, i);
            var yClose = close(d, i);
            var halfWidth = width(d, i) / 2;

            if (orient === 'vertical') {
                drawingContext.moveTo(xValue, yLow);
                drawingContext.lineTo(xValue, yHigh);

                drawingContext.moveTo(xValue, yOpen);
                drawingContext.lineTo(xValue - halfWidth, yOpen);
                drawingContext.moveTo(xValue, yClose);
                drawingContext.lineTo(xValue + halfWidth, yClose);
            } else {
                drawingContext.moveTo(yLow, xValue);
                drawingContext.lineTo(yHigh, xValue);

                drawingContext.moveTo(yOpen, xValue);
                drawingContext.lineTo(yOpen, xValue + halfWidth);
                drawingContext.moveTo(yClose, xValue);
                drawingContext.lineTo(yClose, xValue - halfWidth);
            }
        });

        return context ? null : drawingContext.toString();
    };

    ohlc.context = function () {
        if (!arguments.length) {
            return context;
        }
        context = arguments.length <= 0 ? undefined : arguments[0];
        return ohlc;
    };
    ohlc.x = function () {
        if (!arguments.length) {
            return x;
        }
        x = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return ohlc;
    };
    ohlc.open = function () {
        if (!arguments.length) {
            return open;
        }
        open = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return ohlc;
    };
    ohlc.high = function () {
        if (!arguments.length) {
            return high;
        }
        high = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return ohlc;
    };
    ohlc.low = function () {
        if (!arguments.length) {
            return low;
        }
        low = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return ohlc;
    };
    ohlc.close = function () {
        if (!arguments.length) {
            return close;
        }
        close = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return ohlc;
    };
    ohlc.width = function () {
        if (!arguments.length) {
            return width;
        }
        width = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return ohlc;
    };
    ohlc.orient = function () {
        if (!arguments.length) {
            return orient;
        }
        orient = arguments.length <= 0 ? undefined : arguments[0];
        return ohlc;
    };

    return ohlc;
});

// Renders a bar series as an SVG path based on the given array of datapoints. Each
// bar has a fixed width, whilst the x, y and height are obtained from each data
// point via the supplied accessor functions.
var bar = (function () {

    var context = null;
    var x = function x(d) {
        return d.x;
    };
    var y = function y(d) {
        return d.y;
    };
    var horizontalAlign = 'center';
    var verticalAlign = 'center';
    var height = function height(d) {
        return d.height;
    };
    var width = functor(3);

    var bar = function bar(data, index) {

        var drawingContext = context || d3Path.path();

        data.forEach(function (d, i) {
            var xValue = x.call(this, d, index || i);
            var yValue = y.call(this, d, index || i);
            var barHeight = height.call(this, d, index || i);
            var barWidth = width.call(this, d, index || i);

            var horizontalOffset = void 0;
            switch (horizontalAlign) {
                case 'left':
                    horizontalOffset = barWidth;
                    break;
                case 'right':
                    horizontalOffset = 0;
                    break;
                case 'center':
                    horizontalOffset = barWidth / 2;
                    break;
                default:
                    throw new Error('Invalid horizontal alignment ' + horizontalAlign);
            }

            var verticalOffset = void 0;
            switch (verticalAlign) {
                case 'bottom':
                    verticalOffset = -barHeight;
                    break;
                case 'top':
                    verticalOffset = 0;
                    break;
                case 'center':
                    verticalOffset = barHeight / 2;
                    break;
                default:
                    throw new Error('Invalid vertical alignment ' + verticalAlign);
            }

            drawingContext.rect(xValue - horizontalOffset, yValue - verticalOffset, barWidth, barHeight);
        }, this);

        return context ? null : drawingContext.toString();
    };

    bar.context = function () {
        if (!arguments.length) {
            return context;
        }
        context = arguments.length <= 0 ? undefined : arguments[0];
        return bar;
    };
    bar.x = function () {
        if (!arguments.length) {
            return x;
        }
        x = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return bar;
    };
    bar.y = function () {
        if (!arguments.length) {
            return y;
        }
        y = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return bar;
    };
    bar.width = function () {
        if (!arguments.length) {
            return width;
        }
        width = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return bar;
    };
    bar.horizontalAlign = function () {
        if (!arguments.length) {
            return horizontalAlign;
        }
        horizontalAlign = arguments.length <= 0 ? undefined : arguments[0];
        return bar;
    };
    bar.height = function () {
        if (!arguments.length) {
            return height;
        }
        height = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return bar;
    };
    bar.verticalAlign = function () {
        if (!arguments.length) {
            return verticalAlign;
        }
        verticalAlign = arguments.length <= 0 ? undefined : arguments[0];
        return bar;
    };

    return bar;
});

// Renders a candlestick as an SVG path based on the given array of datapoints. Each
// candlestick has a fixed width, whilst the x, open, high, low and close positions are
// obtained from each point via the supplied accessor functions.
var candlestick = (function () {

    var context = null;
    var x = function x(d) {
        return d.date;
    };
    var open = function open(d) {
        return d.open;
    };
    var high = function high(d) {
        return d.high;
    };
    var low = function low(d) {
        return d.low;
    };
    var close = function close(d) {
        return d.close;
    };
    var width = functor(3);

    var candlestick = function candlestick(data) {

        var drawingContext = context || d3Path.path();

        data.forEach(function (d, i) {
            var xValue = x(d, i);
            var yOpen = open(d, i);
            var yHigh = high(d, i);
            var yLow = low(d, i);
            var yClose = close(d, i);
            var barWidth = width(d, i);
            var halfBarWidth = barWidth / 2;

            // Body
            drawingContext.rect(xValue - halfBarWidth, yOpen, barWidth, yClose - yOpen);
            // High wick
            // // Move to the max price of close or open; draw the high wick
            // N.B. Math.min() is used as we're dealing with pixel values,
            // the lower the pixel value, the higher the price!
            drawingContext.moveTo(xValue, Math.min(yClose, yOpen));
            drawingContext.lineTo(xValue, yHigh);
            // Low wick
            // // Move to the min price of close or open; draw the low wick
            // N.B. Math.max() is used as we're dealing with pixel values,
            // the higher the pixel value, the lower the price!
            drawingContext.moveTo(xValue, Math.max(yClose, yOpen));
            drawingContext.lineTo(xValue, yLow);
        });

        return context ? null : drawingContext.toString();
    };

    candlestick.context = function () {
        if (!arguments.length) {
            return context;
        }
        context = arguments.length <= 0 ? undefined : arguments[0];
        return candlestick;
    };
    candlestick.x = function () {
        if (!arguments.length) {
            return x;
        }
        x = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return candlestick;
    };
    candlestick.open = function () {
        if (!arguments.length) {
            return open;
        }
        open = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return candlestick;
    };
    candlestick.high = function () {
        if (!arguments.length) {
            return high;
        }
        high = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return candlestick;
    };
    candlestick.low = function () {
        if (!arguments.length) {
            return low;
        }
        low = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return candlestick;
    };
    candlestick.close = function () {
        if (!arguments.length) {
            return close;
        }
        close = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return candlestick;
    };
    candlestick.width = function () {
        if (!arguments.length) {
            return width;
        }
        width = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return candlestick;
    };

    return candlestick;
});

// Renders a box plot series as an SVG path based on the given array of datapoints.
var boxPlot = (function () {

    var context = null;
    var value = function value(d) {
        return d.value;
    };
    var median = function median(d) {
        return d.median;
    };
    var upperQuartile = function upperQuartile(d) {
        return d.upperQuartile;
    };
    var lowerQuartile = function lowerQuartile(d) {
        return d.lowerQuartile;
    };
    var high = function high(d) {
        return d.high;
    };
    var low = function low(d) {
        return d.low;
    };
    var orient = 'vertical';
    var width = functor(5);
    var cap = functor(0.5);

    var boxPlot = function boxPlot(data) {

        var drawingContext = context || d3Path.path();

        data.forEach(function (d, i) {
            // naming convention is for vertical orientation
            var _value = value(d, i);
            var _width = width(d, i);
            var halfWidth = _width / 2;
            var capWidth = _width * cap(d, i);
            var halfCapWidth = capWidth / 2;
            var _high = high(d, i);
            var _upperQuartile = upperQuartile(d, i);
            var _median = median(d, i);
            var _lowerQuartile = lowerQuartile(d, i);
            var _low = low(d, i);
            var upperQuartileToLowerQuartile = _lowerQuartile - _upperQuartile;

            if (orient === 'vertical') {
                // Upper whisker
                drawingContext.moveTo(_value - halfCapWidth, _high);
                drawingContext.lineTo(_value + halfCapWidth, _high);
                drawingContext.moveTo(_value, _high);
                drawingContext.lineTo(_value, _upperQuartile);

                // Box
                drawingContext.rect(_value - halfWidth, _upperQuartile, _width, upperQuartileToLowerQuartile);
                drawingContext.moveTo(_value - halfWidth, _median);
                // Median line
                drawingContext.lineTo(_value + halfWidth, _median);

                // Lower whisker
                drawingContext.moveTo(_value, _lowerQuartile);
                drawingContext.lineTo(_value, _low);
                drawingContext.moveTo(_value - halfCapWidth, _low);
                drawingContext.lineTo(_value + halfCapWidth, _low);
            } else {
                // Lower whisker
                drawingContext.moveTo(_low, _value - halfCapWidth);
                drawingContext.lineTo(_low, _value + halfCapWidth);
                drawingContext.moveTo(_low, _value);
                drawingContext.lineTo(_lowerQuartile, _value);

                // Box
                drawingContext.rect(_lowerQuartile, _value - halfWidth, -upperQuartileToLowerQuartile, _width);
                drawingContext.moveTo(_median, _value - halfWidth);
                drawingContext.lineTo(_median, _value + halfWidth);

                // Upper whisker
                drawingContext.moveTo(_upperQuartile, _value);
                drawingContext.lineTo(_high, _value);
                drawingContext.moveTo(_high, _value - halfCapWidth);
                drawingContext.lineTo(_high, _value + halfCapWidth);
            }
        });

        return context ? null : drawingContext.toString();
    };

    boxPlot.context = function () {
        if (!arguments.length) {
            return context;
        }
        context = arguments.length <= 0 ? undefined : arguments[0];
        return boxPlot;
    };
    boxPlot.value = function () {
        if (!arguments.length) {
            return value;
        }
        value = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return boxPlot;
    };
    boxPlot.median = function () {
        if (!arguments.length) {
            return median;
        }
        median = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return boxPlot;
    };
    boxPlot.upperQuartile = function () {
        if (!arguments.length) {
            return upperQuartile;
        }
        upperQuartile = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return boxPlot;
    };
    boxPlot.lowerQuartile = function () {
        if (!arguments.length) {
            return lowerQuartile;
        }
        lowerQuartile = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return boxPlot;
    };
    boxPlot.high = function () {
        if (!arguments.length) {
            return high;
        }
        high = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return boxPlot;
    };
    boxPlot.low = function () {
        if (!arguments.length) {
            return low;
        }
        low = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return boxPlot;
    };
    boxPlot.width = function () {
        if (!arguments.length) {
            return width;
        }
        width = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return boxPlot;
    };
    boxPlot.orient = function () {
        if (!arguments.length) {
            return orient;
        }
        orient = arguments.length <= 0 ? undefined : arguments[0];
        return boxPlot;
    };
    boxPlot.cap = function () {
        if (!arguments.length) {
            return cap;
        }
        cap = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return boxPlot;
    };

    return boxPlot;
});

// Renders an error bar series as an SVG path based on the given array of datapoints.
var errorBar = (function () {

    var context = null;
    var value = function value(d) {
        return d.x;
    };
    var high = function high(d) {
        return d.high;
    };
    var low = function low(d) {
        return d.low;
    };
    var orient = 'vertical';
    var width = functor(5);

    var errorBar = function errorBar(data) {

        var drawingContext = context || d3Path.path();

        data.forEach(function (d, i) {
            // naming convention is for vertical orientation
            var _value = value(d, i);
            var _width = width(d, i);
            var halfWidth = _width / 2;
            var _high = high(d, i);
            var _low = low(d, i);

            if (orient === 'vertical') {
                drawingContext.moveTo(_value - halfWidth, _high);
                drawingContext.lineTo(_value + halfWidth, _high);
                drawingContext.moveTo(_value, _high);
                drawingContext.lineTo(_value, _low);
                drawingContext.moveTo(_value - halfWidth, _low);
                drawingContext.lineTo(_value + halfWidth, _low);
            } else {
                drawingContext.moveTo(_low, _value - halfWidth);
                drawingContext.lineTo(_low, _value + halfWidth);
                drawingContext.moveTo(_low, _value);
                drawingContext.lineTo(_high, _value);
                drawingContext.moveTo(_high, _value - halfWidth);
                drawingContext.lineTo(_high, _value + halfWidth);
            }
        });

        return context ? null : drawingContext.toString();
    };

    errorBar.context = function () {
        if (!arguments.length) {
            return context;
        }
        context = arguments.length <= 0 ? undefined : arguments[0];
        return errorBar;
    };
    errorBar.value = function () {
        if (!arguments.length) {
            return value;
        }
        value = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return errorBar;
    };
    errorBar.high = function () {
        if (!arguments.length) {
            return high;
        }
        high = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return errorBar;
    };
    errorBar.low = function () {
        if (!arguments.length) {
            return low;
        }
        low = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return errorBar;
    };
    errorBar.width = function () {
        if (!arguments.length) {
            return width;
        }
        width = functor(arguments.length <= 0 ? undefined : arguments[0]);
        return errorBar;
    };
    errorBar.orient = function () {
        if (!arguments.length) {
            return orient;
        }
        orient = arguments.length <= 0 ? undefined : arguments[0];
        return errorBar;
    };

    return errorBar;
});

exports.shapeOhlc = ohlc;
exports.shapeBar = bar;
exports.shapeCandlestick = candlestick;
exports.shapeBoxPlot = boxPlot;
exports.shapeErrorBar = errorBar;

Object.defineProperty(exports, '__esModule', { value: true });

})));
