define('echarts/chart/bar', [
    'require',
    'zrender/core/util',
    '../coord/cartesian/Grid',
    './bar/BarSeries',
    './bar/BarView',
    '../layout/barGrid',
    '../echarts'
], function (require) {
    var zrUtil = require('zrender/core/util');
    require('../coord/cartesian/Grid');
    require('./bar/BarSeries');
    require('./bar/BarView');
    var barLayoutGrid = require('../layout/barGrid');
    var echarts = require('../echarts');
    echarts.registerLayout(zrUtil.curry(barLayoutGrid, 'bar'));
    echarts.registerVisualCoding('chart', function (ecModel) {
        ecModel.eachSeriesByType('bar', function (seriesModel) {
            var data = seriesModel.getData();
            data.setVisual('legendSymbol', 'roundRect');
        });
    });
});define('echarts/coord/cartesian/Grid', [
    'require',
    'exports',
    '../../util/layout',
    '../../coord/axisHelper',
    'zrender/core/util',
    './Cartesian2D',
    './Axis2D',
    './GridModel',
    '../../CoordinateSystem'
], function (require, factory) {
    var layout = require('../../util/layout');
    var axisHelper = require('../../coord/axisHelper');
    var zrUtil = require('zrender/core/util');
    var Cartesian2D = require('./Cartesian2D');
    var Axis2D = require('./Axis2D');
    var each = zrUtil.each;
    var ifAxisCrossZero = axisHelper.ifAxisCrossZero;
    var ifAxisNeedsCrossZero = axisHelper.ifAxisNeedsCrossZero;
    var niceScaleExtent = axisHelper.niceScaleExtent;
    require('./GridModel');
    function isAxisUsedInTheGrid(axisModel, gridModel, ecModel) {
        return ecModel.getComponent('grid', axisModel.get('gridIndex')) === gridModel;
    }
    function getLabelUnionRect(axis) {
        var axisModel = axis.model;
        var labels = axisModel.getFormattedLabels();
        var rect;
        for (var i = 0; i < labels.length; i++) {
            if (!axis.isLabelIgnored(i)) {
                var singleRect = axisModel.getTextRect(labels[i]);
                rect ? rect.union(singleRect) : rect = singleRect;
            }
        }
        return rect;
    }
    function Grid(gridModel, ecModel, api) {
        this._coordsMap = {};
        this._coordsList = [];
        this._axesMap = {};
        this._axesList = [];
        this._initCartesian(gridModel, ecModel, api);
    }
    var gridProto = Grid.prototype;
    gridProto.type = 'grid';
    gridProto.getRect = function () {
        return this._rect;
    };
    gridProto.resize = function (gridModel, api) {
        var gridRect = layout.getLayoutRect(gridModel.getBoxLayoutParams(), {
            width: api.getWidth(),
            height: api.getHeight()
        });
        this._rect = gridRect;
        var axesList = this._axesList;
        adjustAxes();
        if (gridModel.get('containLabel')) {
            each(axesList, function (axis) {
                if (!axis.model.get('axisLabel.inside')) {
                    var labelUnionRect = getLabelUnionRect(axis);
                    if (labelUnionRect) {
                        var dim = axis.isHorizontal() ? 'height' : 'width';
                        var margin = axis.model.get('axisLabel.margin');
                        gridRect[dim] -= labelUnionRect[dim] + margin;
                        if (axis.position === 'top') {
                            gridRect.y += labelUnionRect.height + margin;
                        } else if (axis.position === 'left') {
                            gridRect.x += labelUnionRect.width + margin;
                        }
                    }
                }
            });
            adjustAxes();
        }
        function adjustAxes() {
            each(axesList, function (axis) {
                var isHorizontal = axis.isHorizontal();
                var extent = isHorizontal ? [
                    0,
                    gridRect.width
                ] : [
                    0,
                    gridRect.height
                ];
                var idx = axis.inverse ? 1 : 0;
                axis.setExtent(extent[idx], extent[1 - idx]);
                updateAxisTransfrom(axis, isHorizontal ? gridRect.x : gridRect.y);
            });
        }
    };
    gridProto.getAxis = function (axisType, axisIndex) {
        if (axisIndex != null) {
            var key = axisType + axisIndex;
            return this._axesMap[key];
        } else {
            var axesList = this._axesList;
            for (var i = 0; i < axesList.length; i++) {
                if (axesList[i].dim === axisType) {
                    return axesList[i];
                }
            }
        }
    };
    gridProto.getCartesian = function (xAxisIndex, yAxisIndex) {
        var key = 'x' + xAxisIndex + 'y' + yAxisIndex;
        return this._coordsMap[key];
    };
    gridProto._initCartesian = function (gridModel, ecModel, api) {
        var axisPositionUsed = {
            left: false,
            right: false,
            top: false,
            bottom: false
        };
        var axesMap = {
            x: {},
            y: {}
        };
        var axesCount = {
            x: 0,
            y: 0
        };
        ecModel.eachComponent('xAxis', createAxisCreator('x'), this);
        ecModel.eachComponent('yAxis', createAxisCreator('y'), this);
        if (!axesCount.x || !axesCount.y) {
            this._axesMap = {};
            this._axesList = [];
            return;
        }
        each(axesMap.x, function (xAxis, xAxisIndex) {
            each(axesMap.y, function (yAxis, yAxisIndex) {
                var key = 'x' + xAxisIndex + 'y' + yAxisIndex;
                var cartesian = new Cartesian2D(key);
                cartesian.grid = this;
                this._coordsMap[key] = cartesian;
                this._coordsList.push(cartesian);
                cartesian.addAxis(xAxis);
                cartesian.addAxis(yAxis);
            }, this);
        }, this);
        this._updateCartesianFromSeries(ecModel, gridModel);
        function ifAxisCanNotOnZero(otherAxisDim) {
            var axes = axesMap[otherAxisDim];
            return axes[0] && (axes[0].type === 'category' || !ifAxisCrossZero(axes[0])) || axes[1] && (axes[1].type === 'category' || !ifAxisCrossZero(axes[1]));
        }
        each(axesMap.x, function (xAxis) {
            if (ifAxisCanNotOnZero('y')) {
                xAxis.onZero = false;
            }
            if (ifAxisNeedsCrossZero(xAxis)) {
                xAxis.scale.unionExtent([
                    0,
                    0
                ]);
            }
            niceScaleExtent(xAxis, xAxis.model);
        }, this);
        each(axesMap.y, function (yAxis) {
            if (ifAxisCanNotOnZero('x')) {
                yAxis.onZero = false;
            }
            if (ifAxisNeedsCrossZero(yAxis)) {
                yAxis.scale.unionExtent([
                    0,
                    0
                ]);
            }
            niceScaleExtent(yAxis, yAxis.model);
        }, this);
        function createAxisCreator(axisType) {
            return function (axisModel, idx) {
                if (!isAxisUsedInTheGrid(axisModel, gridModel, ecModel)) {
                    return;
                }
                var axisPosition = axisModel.get('position');
                if (axisType === 'x') {
                    if (axisPosition !== 'top' && axisPosition !== 'bottom') {
                        axisPosition = 'bottom';
                    }
                    if (axisPositionUsed[axisPosition]) {
                        axisPosition = axisPosition === 'top' ? 'bottom' : 'top';
                    }
                } else {
                    if (axisPosition !== 'left' && axisPosition !== 'right') {
                        axisPosition = 'left';
                    }
                    if (axisPositionUsed[axisPosition]) {
                        axisPosition = axisPosition === 'left' ? 'right' : 'left';
                    }
                }
                axisPositionUsed[axisPosition] = true;
                var axis = new Axis2D(axisType, axisHelper.createScaleByModel(axisModel), [
                    0,
                    0
                ], axisModel.get('type'), axisPosition);
                var isCategory = axis.type === 'category';
                axis.onBand = isCategory && axisModel.get('boundaryGap');
                axis.inverse = axisModel.get('inverse');
                axis.onZero = axisModel.get('axisLine.onZero');
                axisModel.axis = axis;
                axis.model = axisModel;
                axis.index = idx;
                this._axesList.push(axis);
                this._axesMap[axisType + idx] = axis;
                axesMap[axisType][idx] = axis;
                axesCount[axisType]++;
            };
        }
    };
    gridProto._updateCartesianFromSeries = function (ecModel, gridModel) {
        ecModel.eachSeries(function (seriesModel) {
            if (seriesModel.get('coordinateSystem') === 'cartesian2d') {
                var xAxisIndex = seriesModel.get('xAxisIndex');
                var yAxisIndex = seriesModel.get('yAxisIndex');
                var xAxisModel = ecModel.getComponent('xAxis', xAxisIndex);
                var yAxisModel = ecModel.getComponent('yAxis', yAxisIndex);
                if (!isAxisUsedInTheGrid(xAxisModel, gridModel, ecModel) || !isAxisUsedInTheGrid(yAxisModel, gridModel, ecModel)) {
                    return;
                }
                var cartesian = this.getCartesian(xAxisIndex, yAxisIndex);
                var data = seriesModel.getData();
                if (data.type === 'list') {
                    unionExtent(data, cartesian.getAxis('x'), 'x', seriesModel);
                    unionExtent(data, cartesian.getAxis('y'), 'y', seriesModel);
                }
            }
        }, this);
        function unionExtent(data, axis, axisDim, seriesModel) {
            each(seriesModel.getDimensionsOnAxis(axisDim), function (dim) {
                axis.scale.unionExtent(data.getDataExtent(dim, axis.scale.type !== 'ordinal'));
            });
        }
    };
    function updateAxisTransfrom(axis, coordBase) {
        var axisExtent = axis.getExtent();
        var axisExtentSum = axisExtent[0] + axisExtent[1];
        axis.toGlobalCoord = axis.dim === 'x' ? function (coord) {
            return coord + coordBase;
        } : function (coord) {
            return axisExtentSum - coord + coordBase;
        };
        axis.toLocalCoord = axis.dim === 'x' ? function (coord) {
            return coord - coordBase;
        } : function (coord) {
            return axisExtentSum - coord + coordBase;
        };
    }
    Grid.create = function (ecModel, api) {
        var grids = [];
        ecModel.eachComponent('grid', function (gridModel, idx) {
            var grid = new Grid(gridModel, ecModel, api);
            grid.name = 'grid_' + idx;
            grid.resize(gridModel, api);
            gridModel.coordinateSystem = grid;
            grids.push(grid);
        });
        ecModel.eachSeries(function (seriesModel) {
            if (seriesModel.get('coordinateSystem') !== 'cartesian2d') {
                return;
            }
            var xAxisIndex = seriesModel.get('xAxisIndex');
            var xAxisModel = ecModel.getComponent('xAxis', xAxisIndex);
            var grid = grids[xAxisModel.get('gridIndex')];
            seriesModel.coordinateSystem = grid.getCartesian(xAxisIndex, seriesModel.get('yAxisIndex'));
        });
        return grids;
    };
    require('../../CoordinateSystem').register('grid', Grid);
    return Grid;
});define('echarts/chart/bar/BarSeries', [
    'require',
    '../../model/Series',
    '../helper/createListFromArray'
], function (require) {
    'use strict';
    var SeriesModel = require('../../model/Series');
    var createListFromArray = require('../helper/createListFromArray');
    return SeriesModel.extend({
        type: 'series.bar',
        dependencies: [
            'grid',
            'polar'
        ],
        getInitialData: function (option, ecModel) {
            return createListFromArray(option.data, this, ecModel);
        },
        defaultOption: {
            zlevel: 0,
            z: 2,
            coordinateSystem: 'cartesian2d',
            legendHoverLink: true,
            xAxisIndex: 0,
            yAxisIndex: 0,
            barMinHeight: 0,
            barGap: '30%',
            barCategoryGap: '20%',
            itemStyle: {
                normal: {
                    barBorderColor: '#fff',
                    barBorderWidth: 0
                },
                emphasis: {
                    barBorderColor: '#fff',
                    barBorderWidth: 0
                }
            }
        }
    });
});define('echarts/chart/bar/BarView', [
    'require',
    'zrender/core/util',
    '../../util/graphic',
    '../../model/Model',
    './barItemStyle',
    '../../echarts'
], function (require) {
    'use strict';
    var zrUtil = require('zrender/core/util');
    var graphic = require('../../util/graphic');
    zrUtil.extend(require('../../model/Model').prototype, require('./barItemStyle'));
    function fixLayoutWithLineWidth(layout, lineWidth) {
        var signX = layout.width > 0 ? 1 : -1;
        var signY = layout.height > 0 ? 1 : -1;
        layout.x += signX * lineWidth / 2;
        layout.y += signY * lineWidth / 2;
        layout.width -= signX * lineWidth;
        layout.height -= signY * lineWidth;
    }
    return require('../../echarts').extendChartView({
        type: 'bar',
        render: function (seriesModel, ecModel, api) {
            var coordinateSystemType = seriesModel.get('coordinateSystem');
            if (coordinateSystemType === 'cartesian2d') {
                this._renderOnCartesian(seriesModel, ecModel, api);
            }
            return this.group;
        },
        _renderOnCartesian: function (seriesModel, ecModel, api) {
            var group = this.group;
            var data = seriesModel.getData();
            var oldData = this._data;
            var cartesian = seriesModel.coordinateSystem;
            var baseAxis = cartesian.getBaseAxis();
            var isHorizontal = baseAxis.isHorizontal();
            var enableAnimation = seriesModel.get('animation');
            var barBorderWidthQuery = [
                'itemStyle',
                'normal',
                'barBorderWidth'
            ];
            function createRect(dataIndex, isUpdate) {
                var layout = data.getItemLayout(dataIndex);
                var lineWidth = data.getItemModel(dataIndex).get(barBorderWidthQuery) || 0;
                fixLayoutWithLineWidth(layout, lineWidth);
                var rect = new graphic.Rect({ shape: zrUtil.extend({}, layout) });
                if (enableAnimation) {
                    var rectShape = rect.shape;
                    var animateProperty = isHorizontal ? 'height' : 'width';
                    var animateTarget = {};
                    rectShape[animateProperty] = 0;
                    animateTarget[animateProperty] = layout[animateProperty];
                    graphic[isUpdate ? 'updateProps' : 'initProps'](rect, { shape: animateTarget }, seriesModel);
                }
                return rect;
            }
            data.diff(oldData).add(function (dataIndex) {
                if (!data.hasValue(dataIndex)) {
                    return;
                }
                var rect = createRect(dataIndex);
                data.setItemGraphicEl(dataIndex, rect);
                group.add(rect);
            }).update(function (newIndex, oldIndex) {
                var rect = oldData.getItemGraphicEl(oldIndex);
                if (!data.hasValue(newIndex)) {
                    group.remove(rect);
                    return;
                }
                if (!rect) {
                    rect = createRect(newIndex, true);
                }
                var layout = data.getItemLayout(newIndex);
                var lineWidth = data.getItemModel(newIndex).get(barBorderWidthQuery) || 0;
                fixLayoutWithLineWidth(layout, lineWidth);
                graphic.updateProps(rect, { shape: layout }, seriesModel);
                data.setItemGraphicEl(newIndex, rect);
                group.add(rect);
            }).remove(function (idx) {
                var rect = oldData.getItemGraphicEl(idx);
                if (rect) {
                    rect.style.text = '';
                    graphic.updateProps(rect, { shape: { width: 0 } }, seriesModel, function () {
                        group.remove(rect);
                    });
                }
            }).execute();
            this._updateStyle(seriesModel, data, isHorizontal);
            this._data = data;
        },
        _updateStyle: function (seriesModel, data, isHorizontal) {
            function setLabel(style, model, color, labelText, labelPositionOutside) {
                graphic.setText(style, model, color);
                style.text = labelText;
                if (style.textPosition === 'outside') {
                    style.textPosition = labelPositionOutside;
                }
            }
            data.eachItemGraphicEl(function (rect, idx) {
                var itemModel = data.getItemModel(idx);
                var labelModel = itemModel.getModel('label.normal');
                var color = data.getItemVisual(idx, 'color');
                var layout = data.getItemLayout(idx);
                var hoverStyle = itemModel.getModel('itemStyle.emphasis').getItemStyle();
                rect.setStyle(zrUtil.defaults({ fill: color }, itemModel.getModel('itemStyle.normal').getBarItemStyle()));
                var labelPositionOutside = isHorizontal ? layout.height > 0 ? 'bottom' : 'top' : layout.width > 0 ? 'left' : 'right';
                var labelModel = itemModel.getModel('label.normal');
                var hoverLabelModel = itemModel.getModel('label.emphasis');
                var rectStyle = rect.style;
                if (labelModel.get('show')) {
                    setLabel(rectStyle, labelModel, color, seriesModel.getFormattedLabel(idx, 'normal') || seriesModel.getRawValue(idx), labelPositionOutside);
                } else {
                    rectStyle.text = '';
                }
                if (hoverLabelModel.get('show')) {
                    setLabel(hoverStyle, hoverLabelModel, color, seriesModel.getFormattedLabel(idx, 'emphasis') || seriesModel.getRawValue(idx), labelPositionOutside);
                } else {
                    hoverStyle.text = '';
                }
                graphic.setHoverStyle(rect, hoverStyle);
            });
        },
        remove: function (ecModel, api) {
            var group = this.group;
            if (ecModel.get('animation')) {
                if (this._data) {
                    this._data.eachItemGraphicEl(function (el) {
                        el.style.text = '';
                        graphic.updateProps(el, { shape: { width: 0 } }, ecModel, function () {
                            group.remove(el);
                        });
                    });
                }
            } else {
                group.removeAll();
            }
        }
    });
});define('echarts/layout/barGrid', [
    'require',
    'zrender/core/util',
    '../util/number'
], function (require) {
    'use strict';
    var zrUtil = require('zrender/core/util');
    var numberUtil = require('../util/number');
    var parsePercent = numberUtil.parsePercent;
    function getSeriesStackId(seriesModel) {
        return seriesModel.get('stack') || '__ec_stack_' + seriesModel.seriesIndex;
    }
    function calBarWidthAndOffset(barSeries, api) {
        var columnsMap = {};
        zrUtil.each(barSeries, function (seriesModel, idx) {
            var cartesian = seriesModel.coordinateSystem;
            var baseAxis = cartesian.getBaseAxis();
            var columnsOnAxis = columnsMap[baseAxis.index] || {
                remainedWidth: baseAxis.getBandWidth(),
                autoWidthCount: 0,
                categoryGap: '20%',
                gap: '30%',
                axis: baseAxis,
                stacks: {}
            };
            var stacks = columnsOnAxis.stacks;
            columnsMap[baseAxis.index] = columnsOnAxis;
            var stackId = getSeriesStackId(seriesModel);
            if (!stacks[stackId]) {
                columnsOnAxis.autoWidthCount++;
            }
            stacks[stackId] = stacks[stackId] || {
                width: 0,
                maxWidth: 0
            };
            var barWidth = seriesModel.get('barWidth');
            var barMaxWidth = seriesModel.get('barMaxWidth');
            var barGap = seriesModel.get('barGap');
            var barCategoryGap = seriesModel.get('barCategoryGap');
            if (barWidth && !stacks[stackId].width) {
                barWidth = Math.min(columnsOnAxis.remainedWidth, barWidth);
                stacks[stackId].width = barWidth;
                columnsOnAxis.remainedWidth -= barWidth;
            }
            barMaxWidth && (stacks[stackId].maxWidth = barMaxWidth);
            barGap != null && (columnsOnAxis.gap = barGap);
            barCategoryGap != null && (columnsOnAxis.categoryGap = barCategoryGap);
        });
        var result = {};
        zrUtil.each(columnsMap, function (columnsOnAxis, coordSysName) {
            result[coordSysName] = {};
            var stacks = columnsOnAxis.stacks;
            var baseAxis = columnsOnAxis.axis;
            var bandWidth = baseAxis.getBandWidth();
            var categoryGap = parsePercent(columnsOnAxis.categoryGap, bandWidth);
            var barGapPercent = parsePercent(columnsOnAxis.gap, 1);
            var remainedWidth = columnsOnAxis.remainedWidth;
            var autoWidthCount = columnsOnAxis.autoWidthCount;
            var autoWidth = (remainedWidth - categoryGap) / (autoWidthCount + (autoWidthCount - 1) * barGapPercent);
            autoWidth = Math.max(autoWidth, 0);
            zrUtil.each(stacks, function (column, stack) {
                var maxWidth = column.maxWidth;
                if (!column.width && maxWidth && maxWidth < autoWidth) {
                    maxWidth = Math.min(maxWidth, remainedWidth);
                    remainedWidth -= maxWidth;
                    column.width = maxWidth;
                    autoWidthCount--;
                }
            });
            autoWidth = (remainedWidth - categoryGap) / (autoWidthCount + (autoWidthCount - 1) * barGapPercent);
            autoWidth = Math.max(autoWidth, 0);
            var widthSum = 0;
            var lastColumn;
            zrUtil.each(stacks, function (column, idx) {
                if (!column.width) {
                    column.width = autoWidth;
                }
                lastColumn = column;
                widthSum += column.width * (1 + barGapPercent);
            });
            if (lastColumn) {
                widthSum -= lastColumn.width * barGapPercent;
            }
            var offset = -widthSum / 2;
            zrUtil.each(stacks, function (column, stackId) {
                result[coordSysName][stackId] = result[coordSysName][stackId] || {
                    offset: offset,
                    width: column.width
                };
                offset += column.width * (1 + barGapPercent);
            });
        });
        return result;
    }
    function barLayoutGrid(seriesType, ecModel, api) {
        var barWidthAndOffset = calBarWidthAndOffset(zrUtil.filter(ecModel.getSeriesByType(seriesType), function (seriesModel) {
            return !ecModel.isSeriesFiltered(seriesModel) && seriesModel.coordinateSystem && seriesModel.coordinateSystem.type === 'cartesian2d';
        }));
        var lastStackCoords = {};
        ecModel.eachSeriesByType(seriesType, function (seriesModel) {
            var data = seriesModel.getData();
            var cartesian = seriesModel.coordinateSystem;
            var baseAxis = cartesian.getBaseAxis();
            var stackId = getSeriesStackId(seriesModel);
            var columnLayoutInfo = barWidthAndOffset[baseAxis.index][stackId];
            var columnOffset = columnLayoutInfo.offset;
            var columnWidth = columnLayoutInfo.width;
            var valueAxis = cartesian.getOtherAxis(baseAxis);
            var barMinHeight = seriesModel.get('barMinHeight') || 0;
            var valueAxisStart = baseAxis.onZero ? valueAxis.toGlobalCoord(valueAxis.dataToCoord(0)) : valueAxis.getGlobalExtent()[0];
            var coords = cartesian.dataToPoints(data, true);
            lastStackCoords[stackId] = lastStackCoords[stackId] || [];
            data.each(valueAxis.dim, function (value, idx) {
                if (isNaN(value)) {
                    return;
                }
                if (!lastStackCoords[stackId][idx]) {
                    lastStackCoords[stackId][idx] = {
                        p: valueAxisStart,
                        n: valueAxisStart
                    };
                }
                var sign = value >= 0 ? 'p' : 'n';
                var coord = coords[idx];
                var lastCoord = lastStackCoords[stackId][idx][sign];
                var x, y, width, height;
                if (valueAxis.isHorizontal()) {
                    x = lastCoord;
                    y = coord[1] + columnOffset;
                    width = coord[0] - lastCoord;
                    height = columnWidth;
                    if (Math.abs(width) < barMinHeight) {
                        width = (width < 0 ? -1 : 1) * barMinHeight;
                    }
                    lastStackCoords[stackId][idx][sign] += width;
                } else {
                    x = coord[0] + columnOffset;
                    y = lastCoord;
                    width = columnWidth;
                    height = coord[1] - lastCoord;
                    if (Math.abs(height) < barMinHeight) {
                        height = (height < 0 ? -1 : 1) * barMinHeight;
                    }
                    lastStackCoords[stackId][idx][sign] += height;
                }
                data.setItemLayout(idx, {
                    x: x,
                    y: y,
                    width: width,
                    height: height
                });
            }, true);
        }, this);
    }
    return barLayoutGrid;
});define('echarts/util/layout', [
    'require',
    'zrender/core/util',
    'zrender/core/BoundingRect',
    './number',
    './format'
], function (require) {
    'use strict';
    var zrUtil = require('zrender/core/util');
    var BoundingRect = require('zrender/core/BoundingRect');
    var numberUtil = require('./number');
    var formatUtil = require('./format');
    var parsePercent = numberUtil.parsePercent;
    var layout = {};
    function boxLayout(orient, group, gap, maxWidth, maxHeight) {
        var x = 0;
        var y = 0;
        if (maxWidth == null) {
            maxWidth = Infinity;
        }
        if (maxHeight == null) {
            maxHeight = Infinity;
        }
        var currentLineMaxSize = 0;
        group.eachChild(function (child, idx) {
            var position = child.position;
            var rect = child.getBoundingRect();
            var nextChild = group.childAt(idx + 1);
            var nextChildRect = nextChild && nextChild.getBoundingRect();
            var nextX;
            var nextY;
            if (orient === 'horizontal') {
                nextX = x + rect.width + (nextChildRect ? -nextChildRect.x + rect.x : 0);
                if (nextX > maxWidth || child.newline) {
                    x = 0;
                    y += currentLineMaxSize + gap;
                    currentLineMaxSize = 0;
                } else {
                    currentLineMaxSize = Math.max(currentLineMaxSize, rect.height);
                }
            } else {
                nextY = y + rect.height + (nextChildRect ? -nextChildRect.y + rect.y : 0);
                if (nextY > maxHeight || child.newline) {
                    x += currentLineMaxSize + gap;
                    y = 0;
                    currentLineMaxSize = 0;
                } else {
                    currentLineMaxSize = Math.max(currentLineMaxSize, rect.width);
                }
            }
            if (child.newline) {
                return;
            }
            position[0] = x;
            position[1] = y;
            orient === 'horizontal' ? x = nextX + gap : y = nextY + gap;
        });
    }
    layout.box = boxLayout;
    layout.vbox = zrUtil.curry(boxLayout, 'vertical');
    layout.hbox = zrUtil.curry(boxLayout, 'horizontal');
    layout.getAvailableSize = function (positionInfo, containerRect, margin) {
        var containerWidth = containerRect.width;
        var containerHeight = containerRect.height;
        var x = parsePercent(positionInfo.x, containerWidth);
        var y = parsePercent(positionInfo.y, containerHeight);
        var x2 = parsePercent(positionInfo.x2, containerWidth);
        var y2 = parsePercent(positionInfo.y2, containerHeight);
        (isNaN(x) || isNaN(parseFloat(positionInfo.x))) && (x = 0);
        (isNaN(x2) || isNaN(parseFloat(positionInfo.x2))) && (x2 = containerWidth);
        (isNaN(y) || isNaN(parseFloat(positionInfo.y))) && (y = 0);
        (isNaN(y2) || isNaN(parseFloat(positionInfo.y2))) && (y2 = containerHeight);
        margin = formatUtil.normalizeCssArray(margin || 0);
        return {
            width: Math.max(x2 - x - margin[1] - margin[3], 0),
            height: Math.max(y2 - y - margin[0] - margin[2], 0)
        };
    };
    layout.getLayoutRect = function (positionInfo, containerRect, margin) {
        margin = formatUtil.normalizeCssArray(margin || 0);
        var containerWidth = containerRect.width;
        var containerHeight = containerRect.height;
        var left = parsePercent(positionInfo.left, containerWidth);
        var top = parsePercent(positionInfo.top, containerHeight);
        var right = parsePercent(positionInfo.right, containerWidth);
        var bottom = parsePercent(positionInfo.bottom, containerHeight);
        var width = parsePercent(positionInfo.width, containerWidth);
        var height = parsePercent(positionInfo.height, containerHeight);
        var verticalMargin = margin[2] + margin[0];
        var horizontalMargin = margin[1] + margin[3];
        var aspect = positionInfo.aspect;
        if (isNaN(width)) {
            width = containerWidth - right - horizontalMargin - left;
        }
        if (isNaN(height)) {
            height = containerHeight - bottom - verticalMargin - top;
        }
        if (isNaN(width) && isNaN(height)) {
            if (aspect > containerWidth / containerHeight) {
                width = containerWidth * 0.8;
            } else {
                height = containerHeight * 0.8;
            }
        }
        if (aspect != null) {
            if (isNaN(width)) {
                width = aspect * height;
            }
            if (isNaN(height)) {
                height = width / aspect;
            }
        }
        if (isNaN(left)) {
            left = containerWidth - right - width - horizontalMargin;
        }
        if (isNaN(top)) {
            top = containerHeight - bottom - height - verticalMargin;
        }
        switch (positionInfo.left || positionInfo.right) {
        case 'center':
            left = containerWidth / 2 - width / 2 - margin[3];
            break;
        case 'right':
            left = containerWidth - width - horizontalMargin;
            break;
        }
        switch (positionInfo.top || positionInfo.bottom) {
        case 'middle':
        case 'center':
            top = containerHeight / 2 - height / 2 - margin[0];
            break;
        case 'bottom':
            top = containerHeight - height - verticalMargin;
            break;
        }
        var rect = new BoundingRect(left + margin[3], top + margin[0], width, height);
        rect.margin = margin;
        return rect;
    };
    layout.positionGroup = function (group, positionInfo, containerRect, margin) {
        var groupRect = group.getBoundingRect();
        positionInfo = zrUtil.extend(zrUtil.clone(positionInfo), {
            width: groupRect.width,
            height: groupRect.height
        });
        positionInfo = layout.getLayoutRect(positionInfo, containerRect, margin);
        group.position = [
            positionInfo.x - groupRect.x,
            positionInfo.y - groupRect.y
        ];
    };
    return layout;
});define('echarts/coord/axisHelper', [
    'require',
    '../scale/Ordinal',
    '../scale/Interval',
    '../scale/Time',
    '../scale/Log',
    '../scale/Scale',
    '../util/number',
    'zrender/core/util',
    'zrender/contain/text'
], function (require) {
    var OrdinalScale = require('../scale/Ordinal');
    var IntervalScale = require('../scale/Interval');
    require('../scale/Time');
    require('../scale/Log');
    var Scale = require('../scale/Scale');
    var numberUtil = require('../util/number');
    var zrUtil = require('zrender/core/util');
    var textContain = require('zrender/contain/text');
    var axisHelper = {};
    axisHelper.niceScaleExtent = function (axis, model) {
        var scale = axis.scale;
        if (scale.type === 'ordinal') {
            return;
        }
        var min = model.get('min');
        var max = model.get('max');
        var boundaryGap = model.get('boundaryGap');
        if (!zrUtil.isArray(boundaryGap)) {
            boundaryGap = [
                boundaryGap || 0,
                boundaryGap || 0
            ];
        }
        boundaryGap[0] = numberUtil.parsePercent(boundaryGap[0], 1);
        boundaryGap[1] = numberUtil.parsePercent(boundaryGap[1], 1);
        var originalExtent = scale.getExtent();
        var span = originalExtent[1] - originalExtent[0];
        var fixMin = true;
        var fixMax = true;
        if (min == null) {
            min = originalExtent[0] - boundaryGap[0] * span;
            fixMin = false;
        }
        if (max == null) {
            max = originalExtent[1] + boundaryGap[1] * span;
            fixMax = false;
        }
        if (min === 'dataMin') {
            min = originalExtent[0];
        }
        if (max === 'dataMax') {
            max = originalExtent[1];
        }
        scale.setExtent(min, max);
        scale.niceExtent(model.get('splitNumber'), fixMin, fixMax);
        var interval = model.get('interval');
        if (interval != null) {
            scale.setInterval && scale.setInterval(interval);
        }
    };
    axisHelper.createScaleByModel = function (model, axisType) {
        axisType = axisType || model.get('type');
        if (axisType) {
            switch (axisType) {
            case 'category':
                return new OrdinalScale(model.getCategories(), [
                    Infinity,
                    -Infinity
                ]);
            case 'value':
                return new IntervalScale();
            default:
                return (Scale.getClass(axisType) || IntervalScale).create(model);
            }
        }
    };
    axisHelper.ifAxisCrossZero = function (axis) {
        var dataExtent = axis.scale.getExtent();
        var min = dataExtent[0];
        var max = dataExtent[1];
        var optMin = axis.model.get('min');
        var optMax = axis.model.get('max');
        if (!isNaN(optMin)) {
            min = Math.min(optMin, min);
        }
        if (!isNaN(optMax)) {
            max = Math.max(optMax, max);
        }
        return !(min > 0 && max > 0 || min < 0 && max < 0) || axisHelper.ifAxisNeedsCrossZero(axis);
    };
    axisHelper.ifAxisNeedsCrossZero = function (axis) {
        return !axis.model.get('scale');
    };
    axisHelper.getAxisLabelInterval = function (tickCoords, labels, font, isAxisHorizontal) {
        var textSpaceTakenRect;
        var autoLabelInterval = 0;
        var accumulatedLabelInterval = 0;
        for (var i = 0; i < tickCoords.length; i++) {
            var tickCoord = tickCoords[i];
            var rect = textContain.getBoundingRect(labels[i], font, 'center', 'top');
            rect[isAxisHorizontal ? 'x' : 'y'] += tickCoord;
            rect[isAxisHorizontal ? 'width' : 'height'] *= 1.5;
            if (!textSpaceTakenRect) {
                textSpaceTakenRect = rect.clone();
            } else if (textSpaceTakenRect.intersect(rect)) {
                accumulatedLabelInterval++;
                autoLabelInterval = Math.max(autoLabelInterval, accumulatedLabelInterval);
            } else {
                textSpaceTakenRect.union(rect);
                accumulatedLabelInterval = 0;
            }
        }
        return autoLabelInterval;
    };
    axisHelper.getFormattedLabels = function (axis, labelFormatter) {
        var scale = axis.scale;
        var labels = scale.getTicksLabels();
        var ticks = scale.getTicks();
        if (typeof labelFormatter === 'string') {
            labelFormatter = function (tpl) {
                return function (val) {
                    return tpl.replace('{value}', val);
                };
            }(labelFormatter);
            return zrUtil.map(labels, labelFormatter);
        } else if (typeof labelFormatter === 'function') {
            return zrUtil.map(ticks, function (tick, idx) {
                return labelFormatter(axis.type === 'category' ? scale.getLabel(tick) : tick, idx);
            }, this);
        } else {
            return labels;
        }
    };
    return axisHelper;
});define('echarts/coord/cartesian/Cartesian2D', [
    'require',
    'zrender/core/util',
    './Cartesian'
], function (require) {
    'use strict';
    var zrUtil = require('zrender/core/util');
    var Cartesian = require('./Cartesian');
    function Cartesian2D(name) {
        Cartesian.call(this, name);
        this.dimensions = [
            'x',
            'y'
        ];
    }
    Cartesian2D.prototype = {
        constructor: Cartesian2D,
        type: 'cartesian2d',
        getBaseAxis: function () {
            return this.getAxesByScale('ordinal')[0] || this.getAxesByScale('time')[0] || this.getAxis('x');
        },
        containPoint: function (point) {
            var axisX = this.getAxis('x');
            var axisY = this.getAxis('y');
            return axisX.contain(axisX.toLocalCoord(point[0])) && axisY.contain(axisY.toLocalCoord(point[1]));
        },
        containData: function (data) {
            return this.getAxis('x').containData(data[0]) && this.getAxis('y').containData(data[1]);
        },
        dataToPoints: function (data, stack) {
            return data.mapArray([
                'x',
                'y'
            ], function (x, y) {
                return this.dataToPoint([
                    x,
                    y
                ]);
            }, stack, this);
        },
        dataToPoint: function (data, clamp) {
            var xAxis = this.getAxis('x');
            var yAxis = this.getAxis('y');
            return [
                xAxis.toGlobalCoord(xAxis.dataToCoord(data[0], clamp)),
                yAxis.toGlobalCoord(yAxis.dataToCoord(data[1], clamp))
            ];
        },
        pointToData: function (point, clamp) {
            var xAxis = this.getAxis('x');
            var yAxis = this.getAxis('y');
            return [
                xAxis.coordToData(xAxis.toLocalCoord(point[0]), clamp),
                yAxis.coordToData(yAxis.toLocalCoord(point[1]), clamp)
            ];
        },
        getOtherAxis: function (axis) {
            return this.getAxis(axis.dim === 'x' ? 'y' : 'x');
        }
    };
    zrUtil.inherits(Cartesian2D, Cartesian);
    return Cartesian2D;
});define('echarts/coord/cartesian/Axis2D', [
    'require',
    'zrender/core/util',
    '../Axis',
    './axisLabelInterval'
], function (require) {
    var zrUtil = require('zrender/core/util');
    var Axis = require('../Axis');
    var axisLabelInterval = require('./axisLabelInterval');
    var Axis2D = function (dim, scale, coordExtent, axisType, position) {
        Axis.call(this, dim, scale, coordExtent);
        this.type = axisType || 'value';
        this.position = position || 'bottom';
    };
    Axis2D.prototype = {
        constructor: Axis2D,
        index: 0,
        onZero: false,
        model: null,
        isHorizontal: function () {
            var position = this.position;
            return position === 'top' || position === 'bottom';
        },
        getGlobalExtent: function () {
            var ret = this.getExtent();
            ret[0] = this.toGlobalCoord(ret[0]);
            ret[1] = this.toGlobalCoord(ret[1]);
            return ret;
        },
        getLabelInterval: function () {
            var labelInterval = this._labelInterval;
            if (!labelInterval) {
                labelInterval = this._labelInterval = axisLabelInterval(this);
            }
            return labelInterval;
        },
        isLabelIgnored: function (idx) {
            if (this.type === 'category') {
                var labelInterval = this.getLabelInterval();
                return typeof labelInterval === 'function' && !labelInterval(idx, this.scale.getLabel(idx)) || idx % (labelInterval + 1);
            }
        },
        toLocalCoord: null,
        toGlobalCoord: null
    };
    zrUtil.inherits(Axis2D, Axis);
    return Axis2D;
});define('echarts/coord/cartesian/GridModel', [
    'require',
    './AxisModel',
    '../../model/Component'
], function (require) {
    'use strict';
    require('./AxisModel');
    var ComponentModel = require('../../model/Component');
    return ComponentModel.extend({
        type: 'grid',
        dependencies: [
            'xAxis',
            'yAxis'
        ],
        mergeDefaultAndTheme: function (option, ecModel) {
            if (option.right == null || option.width == null || option.left != null) {
                option.left = option.left != null ? option.left : '10%';
                option.right = option.right != null ? option.right : '10%';
            }
            if (option.bottom == null || option.height == null || option.top != null) {
                option.top = option.top != null ? option.top : 60;
                option.bottom = option.bottom != null ? option.bottom : 60;
            }
            ComponentModel.prototype.mergeDefaultAndTheme.call(this, option, ecModel);
        },
        coordinateSystem: null,
        defaultOption: {
            show: false,
            zlevel: 0,
            z: 0,
            containLabel: false,
            backgroundColor: 'rgba(0,0,0,0)',
            borderWidth: 1,
            borderColor: '#ccc'
        }
    });
});define('echarts/scale/Ordinal', [
    'require',
    'zrender/core/util',
    './Scale'
], function (require) {
    var zrUtil = require('zrender/core/util');
    var Scale = require('./Scale');
    var scaleProto = Scale.prototype;
    var OrdinalScale = Scale.extend({
        type: 'ordinal',
        init: function (data, extent) {
            this._data = data;
            this._extent = extent || [
                0,
                data.length - 1
            ];
        },
        contain: function (rank) {
            return scaleProto.contain.call(this, rank) && this._data[rank] != null;
        },
        normalize: function (val) {
            if (typeof val === 'string') {
                val = zrUtil.indexOf(this._data, val);
            }
            return scaleProto.normalize.call(this, val);
        },
        scale: function (val) {
            return Math.round(scaleProto.scale.call(this, val));
        },
        getTicks: function () {
            var ticks = [];
            var extent = this._extent;
            var rank = extent[0];
            while (rank <= extent[1]) {
                ticks.push(rank);
                rank++;
            }
            return ticks;
        },
        getLabel: function (n) {
            return this._data[n];
        },
        count: function () {
            return this._extent[1] - this._extent[0] + 1;
        },
        niceTicks: zrUtil.noop,
        niceExtent: zrUtil.noop
    });
    OrdinalScale.create = function () {
        return new OrdinalScale();
    };
    return OrdinalScale;
});define('echarts/scale/Interval', [
    'require',
    '../util/number',
    '../util/format',
    './Scale'
], function (require) {
    var numberUtil = require('../util/number');
    var formatUtil = require('../util/format');
    var Scale = require('./Scale');
    var mathFloor = Math.floor;
    var mathCeil = Math.ceil;
    var IntervalScale = Scale.extend({
        type: 'interval',
        _interval: 0,
        setExtent: function (start, end) {
            var thisExtent = this._extent;
            if (!isNaN(start)) {
                thisExtent[0] = start;
            }
            if (!isNaN(end)) {
                thisExtent[1] = end;
            }
            if (thisExtent[0] === thisExtent[1]) {
                var expandSize = thisExtent[0] / 2 || 1;
                thisExtent[0] -= expandSize;
                thisExtent[1] += expandSize;
            }
        },
        unionExtent: function (other) {
            var extent = this._extent;
            other[0] < extent[0] && (extent[0] = other[0]);
            other[1] > extent[1] && (extent[1] = other[1]);
            IntervalScale.prototype.setExtent.call(this, extent[0], extent[1]);
        },
        getInterval: function () {
            if (!this._interval) {
                this.niceTicks();
            }
            return this._interval;
        },
        setInterval: function (interval) {
            this._interval = interval;
            this._niceExtent = this._extent.slice();
        },
        getTicks: function () {
            if (!this._interval) {
                this.niceTicks();
            }
            var interval = this._interval;
            var extent = this._extent;
            var ticks = [];
            if (interval) {
                var niceExtent = this._niceExtent;
                if (extent[0] < niceExtent[0]) {
                    ticks.push(extent[0]);
                }
                var tick = niceExtent[0];
                while (tick <= niceExtent[1]) {
                    ticks.push(tick);
                    tick = numberUtil.round(tick + interval);
                }
                if (extent[1] > niceExtent[1]) {
                    ticks.push(extent[1]);
                }
            }
            return ticks;
        },
        getTicksLabels: function () {
            var labels = [];
            var ticks = this.getTicks();
            for (var i = 0; i < ticks.length; i++) {
                labels.push(this.getLabel(ticks[i]));
            }
            return labels;
        },
        getLabel: function (data) {
            return formatUtil.addCommas(data);
        },
        niceTicks: function (approxTickNum) {
            approxTickNum = approxTickNum || 10;
            var extent = this._extent;
            var span = extent[1] - extent[0];
            if (span === Infinity || span <= 0) {
                return;
            }
            var interval = Math.pow(10, Math.floor(Math.log(span / approxTickNum) / Math.LN10));
            var err = approxTickNum / span * interval;
            if (err <= 0.15) {
                interval *= 10;
            } else if (err <= 0.3) {
                interval *= 5;
            } else if (err <= 0.5) {
                interval *= 3;
            } else if (err <= 0.75) {
                interval *= 2;
            }
            var niceExtent = [
                numberUtil.round(mathCeil(extent[0] / interval) * interval),
                numberUtil.round(mathFloor(extent[1] / interval) * interval)
            ];
            this._interval = interval;
            this._niceExtent = niceExtent;
        },
        niceExtent: function (approxTickNum, fixMin, fixMax) {
            this.niceTicks(approxTickNum, fixMin, fixMax);
            var extent = this._extent;
            var interval = this._interval;
            if (!fixMin) {
                extent[0] = numberUtil.round(mathFloor(extent[0] / interval) * interval);
            }
            if (!fixMax) {
                extent[1] = numberUtil.round(mathCeil(extent[1] / interval) * interval);
            }
        }
    });
    IntervalScale.create = function () {
        return new IntervalScale();
    };
    return IntervalScale;
});define('echarts/scale/Time', [
    'require',
    'zrender/core/util',
    '../util/number',
    './Interval'
], function (require) {
    var zrUtil = require('zrender/core/util');
    var numberUtil = require('../util/number');
    var IntervalScale = require('./Interval');
    var intervalScaleProto = IntervalScale.prototype;
    var mathCeil = Math.ceil;
    var mathFloor = Math.floor;
    var bisect = function (a, x, lo, hi) {
        while (lo < hi) {
            var mid = lo + hi >>> 1;
            if (a[mid][2] < x) {
                lo = mid + 1;
            } else {
                hi = mid;
            }
        }
        return lo;
    };
    var s2d = function (str) {
        return str < 10 ? '0' + str : str;
    };
    var format = function (tpl, value) {
        if (tpl === 'week' || tpl === 'month' || tpl === 'quarter' || tpl === 'half-year' || tpl === 'year') {
            tpl = 'MM-dd\nyyyy';
        }
        var date = numberUtil.parseDate(value);
        var y = date.getFullYear();
        var M = date.getMonth() + 1;
        var d = date.getDate();
        var h = date.getHours();
        var m = date.getMinutes();
        var s = date.getSeconds();
        tpl = tpl.replace('MM', s2d(M)).toLowerCase().replace('yyyy', y).replace('yy', y % 100).replace('dd', s2d(d)).replace('d', d).replace('hh', s2d(h)).replace('h', h).replace('mm', s2d(m)).replace('m', m).replace('ss', s2d(s)).replace('s', s);
        return tpl;
    };
    var TimeScale = IntervalScale.extend({
        type: 'time',
        getLabel: function (val) {
            var stepLvl = this._stepLvl;
            var date = new Date(val);
            return format(stepLvl[0], date);
        },
        niceTicks: function (approxTickNum) {
            approxTickNum = approxTickNum || 10;
            var extent = this._extent;
            var span = extent[1] - extent[0];
            var approxInterval = span / approxTickNum;
            var scaleLevelsLen = scaleLevels.length;
            var idx = bisect(scaleLevels, approxInterval, 0, scaleLevelsLen);
            var level = scaleLevels[Math.min(idx, scaleLevelsLen - 1)];
            var interval = level[2];
            var niceExtent = [
                mathCeil(extent[0] / interval) * interval,
                mathFloor(extent[1] / interval) * interval
            ];
            this._stepLvl = level;
            this._interval = interval;
            this._niceExtent = niceExtent;
        }
    });
    zrUtil.each([
        'contain',
        'normalize'
    ], function (methodName) {
        TimeScale.prototype[methodName] = function (val) {
            val = +numberUtil.parseDate(val);
            return intervalScaleProto[methodName].call(this, val);
        };
    });
    var scaleLevels = [
        [
            'hh:mm:ss',
            1,
            1000
        ],
        [
            'hh:mm:ss',
            5,
            1000 * 5
        ],
        [
            'hh:mm:ss',
            10,
            1000 * 10
        ],
        [
            'hh:mm:ss',
            15,
            1000 * 15
        ],
        [
            'hh:mm:ss',
            30,
            1000 * 30
        ],
        [
            'hh:mm\nMM-dd',
            1,
            60000
        ],
        [
            'hh:mm\nMM-dd',
            5,
            60000 * 5
        ],
        [
            'hh:mm\nMM-dd',
            10,
            60000 * 10
        ],
        [
            'hh:mm\nMM-dd',
            15,
            60000 * 15
        ],
        [
            'hh:mm\nMM-dd',
            30,
            60000 * 30
        ],
        [
            'hh:mm\nMM-dd',
            1,
            3600000
        ],
        [
            'hh:mm\nMM-dd',
            2,
            3600000 * 2
        ],
        [
            'hh:mm\nMM-dd',
            6,
            3600000 * 6
        ],
        [
            'hh:mm\nMM-dd',
            12,
            3600000 * 12
        ],
        [
            'MM-dd\nyyyy',
            1,
            3600000 * 24
        ],
        [
            'week',
            7,
            3600000 * 24 * 7
        ],
        [
            'month',
            1,
            3600000 * 24 * 31
        ],
        [
            'quarter',
            3,
            3600000 * 24 * 380 / 4
        ],
        [
            'half-year',
            6,
            3600000 * 24 * 380 / 2
        ],
        [
            'year',
            1,
            3600000 * 24 * 380
        ]
    ];
    TimeScale.create = function () {
        return new TimeScale();
    };
    return TimeScale;
});define('echarts/scale/Log', [
    'require',
    'zrender/core/util',
    './Scale',
    '../util/number',
    './Interval'
], function (require) {
    var zrUtil = require('zrender/core/util');
    var Scale = require('./Scale');
    var numberUtil = require('../util/number');
    var IntervalScale = require('./Interval');
    var scaleProto = Scale.prototype;
    var intervalScaleProto = IntervalScale.prototype;
    var mathFloor = Math.floor;
    var mathCeil = Math.ceil;
    var mathPow = Math.pow;
    var LOG_BASE = 10;
    var mathLog = Math.log;
    var LogScale = Scale.extend({
        type: 'log',
        getTicks: function () {
            return zrUtil.map(intervalScaleProto.getTicks.call(this), function (val) {
                return numberUtil.round(mathPow(LOG_BASE, val));
            });
        },
        getLabel: intervalScaleProto.getLabel,
        scale: function (val) {
            val = scaleProto.scale.call(this, val);
            return mathPow(LOG_BASE, val);
        },
        setExtent: function (start, end) {
            start = mathLog(start) / mathLog(LOG_BASE);
            end = mathLog(end) / mathLog(LOG_BASE);
            intervalScaleProto.setExtent.call(this, start, end);
        },
        getExtent: function () {
            var extent = scaleProto.getExtent.call(this);
            extent[0] = mathPow(LOG_BASE, extent[0]);
            extent[1] = mathPow(LOG_BASE, extent[1]);
            return extent;
        },
        unionExtent: function (extent) {
            extent[0] = mathLog(extent[0]) / mathLog(LOG_BASE);
            extent[1] = mathLog(extent[1]) / mathLog(LOG_BASE);
            scaleProto.unionExtent.call(this, extent);
        },
        niceTicks: function (approxTickNum) {
            approxTickNum = approxTickNum || 10;
            var extent = this._extent;
            var span = extent[1] - extent[0];
            if (span === Infinity || span <= 0) {
                return;
            }
            var interval = mathPow(10, mathFloor(mathLog(span / approxTickNum) / Math.LN10));
            var err = approxTickNum / span * interval;
            if (err <= 0.5) {
                interval *= 10;
            }
            var niceExtent = [
                numberUtil.round(mathCeil(extent[0] / interval) * interval),
                numberUtil.round(mathFloor(extent[1] / interval) * interval)
            ];
            this._interval = interval;
            this._niceExtent = niceExtent;
        },
        niceExtent: intervalScaleProto.niceExtent
    });
    zrUtil.each([
        'contain',
        'normalize'
    ], function (methodName) {
        LogScale.prototype[methodName] = function (val) {
            val = mathLog(val) / mathLog(LOG_BASE);
            return scaleProto[methodName].call(this, val);
        };
    });
    LogScale.create = function () {
        return new LogScale();
    };
    return LogScale;
});define('echarts/scale/Scale', [
    'require',
    '../util/clazz'
], function (require) {
    var clazzUtil = require('../util/clazz');
    function Scale() {
        this._extent = [
            Infinity,
            -Infinity
        ];
        this._interval = 0;
        this.init && this.init.apply(this, arguments);
    }
    var scaleProto = Scale.prototype;
    scaleProto.contain = function (val) {
        var extent = this._extent;
        return val >= extent[0] && val <= extent[1];
    };
    scaleProto.normalize = function (val) {
        var extent = this._extent;
        if (extent[1] === extent[0]) {
            return 0.5;
        }
        return (val - extent[0]) / (extent[1] - extent[0]);
    };
    scaleProto.scale = function (val) {
        var extent = this._extent;
        return val * (extent[1] - extent[0]) + extent[0];
    };
    scaleProto.unionExtent = function (other) {
        var extent = this._extent;
        other[0] < extent[0] && (extent[0] = other[0]);
        other[1] > extent[1] && (extent[1] = other[1]);
    };
    scaleProto.getExtent = function () {
        return this._extent.slice();
    };
    scaleProto.setExtent = function (start, end) {
        var thisExtent = this._extent;
        if (!isNaN(start)) {
            thisExtent[0] = start;
        }
        if (!isNaN(end)) {
            thisExtent[1] = end;
        }
    };
    scaleProto.getTicksLabels = function () {
        var labels = [];
        var ticks = this.getTicks();
        for (var i = 0; i < ticks.length; i++) {
            labels.push(this.getLabel(ticks[i]));
        }
        return labels;
    };
    clazzUtil.enableClassExtend(Scale);
    clazzUtil.enableClassManagement(Scale, { registerWhenExtend: true });
    return Scale;
});define('echarts/coord/cartesian/Cartesian', [
    'require',
    'zrender/core/util'
], function (require) {
    'use strict';
    var zrUtil = require('zrender/core/util');
    function dimAxisMapper(dim) {
        return this._axes[dim];
    }
    var Cartesian = function (name) {
        this._axes = {};
        this._dimList = [];
        this.name = name || '';
    };
    Cartesian.prototype = {
        constructor: Cartesian,
        type: 'cartesian',
        getAxis: function (dim) {
            return this._axes[dim];
        },
        getAxes: function () {
            return zrUtil.map(this._dimList, dimAxisMapper, this);
        },
        getAxesByScale: function (scaleType) {
            scaleType = scaleType.toLowerCase();
            return zrUtil.filter(this.getAxes(), function (axis) {
                return axis.scale.type === scaleType;
            });
        },
        addAxis: function (axis) {
            var dim = axis.dim;
            this._axes[dim] = axis;
            this._dimList.push(dim);
        },
        dataToCoord: function (val) {
            return this._dataCoordConvert(val, 'dataToCoord');
        },
        coordToData: function (val) {
            return this._dataCoordConvert(val, 'coordToData');
        },
        _dataCoordConvert: function (input, method) {
            var dimList = this._dimList;
            var output = input instanceof Array ? [] : {};
            for (var i = 0; i < dimList.length; i++) {
                var dim = dimList[i];
                var axis = this._axes[dim];
                output[dim] = axis[method](input[dim]);
            }
            return output;
        }
    };
    return Cartesian;
});define('echarts/coord/Axis', [
    'require',
    '../util/number',
    'zrender/core/util'
], function (require) {
    var numberUtil = require('../util/number');
    var linearMap = numberUtil.linearMap;
    var zrUtil = require('zrender/core/util');
    function fixExtentWithBands(extent, nTick) {
        var size = extent[1] - extent[0];
        var len = nTick;
        var margin = size / len / 2;
        extent[0] += margin;
        extent[1] -= margin;
    }
    var Axis = function (dim, scale, extent) {
        this.dim = dim;
        this.scale = scale;
        this._extent = extent || [
            0,
            0
        ];
        this.inverse = false;
        this.onBand = false;
    };
    Axis.prototype = {
        constructor: Axis,
        contain: function (coord) {
            var extent = this._extent;
            var min = Math.min(extent[0], extent[1]);
            var max = Math.max(extent[0], extent[1]);
            return coord >= min && coord <= max;
        },
        containData: function (data) {
            return this.contain(this.dataToCoord(data));
        },
        getExtent: function () {
            var ret = this._extent.slice();
            return ret;
        },
        getFormatPrecision: function (dataExtent) {
            dataExtent = dataExtent || this.scale.getExtent();
            var log = Math.log;
            var LN10 = Math.LN10;
            var extent = this._extent;
            var dataQuantity = Math.floor(log(dataExtent[1] - dataExtent[0]) / LN10);
            var sizeQuantity = Math.round(log(Math.abs(extent[1] - extent[0])) / LN10);
            return Math.max(-dataQuantity + sizeQuantity, 0);
        },
        setExtent: function (start, end) {
            var extent = this._extent;
            extent[0] = start;
            extent[1] = end;
        },
        dataToCoord: function (data, clamp) {
            data = this.scale.normalize(data);
            var extent = this.getExtent();
            var scale = this.scale;
            if (this.onBand && scale.type === 'ordinal') {
                fixExtentWithBands(extent, scale.count());
            }
            return linearMap(data, [
                0,
                1
            ], extent, clamp);
        },
        coordToData: function (coord, clamp) {
            var extent = this.getExtent();
            if (this.onBand) {
                fixExtentWithBands(extent, this.scale.count());
            }
            var t = linearMap(coord, extent, [
                0,
                1
            ], clamp);
            return this.scale.scale(t);
        },
        getTicksCoords: function () {
            if (this.onBand) {
                var bands = this.getBands();
                var coords = [];
                for (var i = 0; i < bands.length; i++) {
                    coords.push(bands[i][0]);
                }
                if (bands[i - 1]) {
                    coords.push(bands[i - 1][1]);
                }
                return coords;
            } else {
                return zrUtil.map(this.scale.getTicks(), this.dataToCoord, this);
            }
        },
        getLabelsCoords: function () {
            if (this.onBand) {
                var bands = this.getBands();
                var coords = [];
                var band;
                for (var i = 0; i < bands.length; i++) {
                    band = bands[i];
                    coords.push((band[0] + band[1]) / 2);
                }
                return coords;
            } else {
                return zrUtil.map(this.scale.getTicks(), this.dataToCoord, this);
            }
        },
        getBands: function () {
            var extent = this.getExtent();
            var bands = [];
            var len = this.scale.count();
            var start = extent[0];
            var end = extent[1];
            var span = end - start;
            for (var i = 0; i < len; i++) {
                bands.push([
                    span * i / len + start,
                    span * (i + 1) / len + start
                ]);
            }
            return bands;
        },
        getBandWidth: function () {
            var axisExtent = this._extent;
            var dataExtent = this.scale.getExtent();
            var len = dataExtent[1] - dataExtent[0] + (this.onBand ? 1 : 0);
            var size = Math.abs(axisExtent[1] - axisExtent[0]);
            return Math.abs(size) / len;
        }
    };
    return Axis;
});define('echarts/coord/cartesian/axisLabelInterval', [
    'require',
    'zrender/core/util',
    '../axisHelper'
], function (require) {
    'use strict';
    var zrUtil = require('zrender/core/util');
    var axisHelper = require('../axisHelper');
    return function (axis) {
        var axisModel = axis.model;
        var labelModel = axisModel.getModel('axisLabel');
        var labelInterval = labelModel.get('interval');
        if (!(axis.type === 'category' && labelInterval === 'auto')) {
            return labelInterval === 'auto' ? 0 : labelInterval;
        }
        return axisHelper.getAxisLabelInterval(zrUtil.map(axis.scale.getTicks(), axis.dataToCoord, axis), axisModel.getFormattedLabels(), labelModel.getModel('textStyle').getFont(), axis.isHorizontal());
    };
});define('echarts/coord/cartesian/AxisModel', [
    'require',
    '../../model/Component',
    'zrender/core/util',
    '../axisModelCreator',
    '../axisModelCommonMixin'
], function (require) {
    'use strict';
    var ComponentModel = require('../../model/Component');
    var zrUtil = require('zrender/core/util');
    var axisModelCreator = require('../axisModelCreator');
    var AxisModel = ComponentModel.extend({
        type: 'cartesian2dAxis',
        axis: null,
        setNeedsCrossZero: function (needs) {
            this.option.scale = !needs;
        }
    });
    function getAxisType(axisDim, option) {
        return option.type || (option.data ? 'category' : 'value');
    }
    zrUtil.merge(AxisModel.prototype, require('../axisModelCommonMixin'));
    var extraOption = { gridIndex: 0 };
    axisModelCreator('x', AxisModel, getAxisType, extraOption);
    axisModelCreator('y', AxisModel, getAxisType, extraOption);
    return AxisModel;
});define('echarts/coord/axisModelCreator', [
    'require',
    './axisDefault',
    'zrender/core/util',
    '../model/Component'
], function (require) {
    var axisDefault = require('./axisDefault');
    var zrUtil = require('zrender/core/util');
    var ComponentModel = require('../model/Component');
    var AXIS_TYPES = [
        'value',
        'category',
        'time',
        'log'
    ];
    return function (axisName, BaseAxisModelClass, axisTypeDefaulter, extraDefaultOption) {
        zrUtil.each(AXIS_TYPES, function (axisType) {
            BaseAxisModelClass.extend({
                type: axisName + 'Axis.' + axisType,
                mergeDefaultAndTheme: function (option, ecModel) {
                    var themeModel = ecModel.getTheme();
                    zrUtil.merge(option, themeModel.get(axisType + 'Axis'));
                    zrUtil.merge(option, this.getDefaultOption());
                    option.type = axisTypeDefaulter(axisName, option);
                },
                defaultOption: zrUtil.mergeAll([
                    {},
                    axisDefault[axisType + 'Axis'],
                    extraDefaultOption
                ], true)
            });
        });
        ComponentModel.registerSubTypeDefaulter(axisName + 'Axis', zrUtil.curry(axisTypeDefaulter, axisName));
    };
});define('echarts/coord/axisModelCommonMixin', [
    'require',
    'zrender/core/util',
    './axisHelper'
], function (require) {
    var zrUtil = require('zrender/core/util');
    var axisHelper = require('./axisHelper');
    function getName(obj) {
        if (zrUtil.isObject(obj) && obj.value != null) {
            return obj.value;
        } else {
            return obj;
        }
    }
    function getCategories() {
        return this.get('type') === 'category' && zrUtil.map(this.get('data'), getName);
    }
    function getFormattedLabels() {
        return axisHelper.getFormattedLabels(this.axis, this.get('axisLabel.formatter'));
    }
    return {
        getFormattedLabels: getFormattedLabels,
        getCategories: getCategories
    };
});define('echarts/coord/axisDefault', [
    'require',
    'zrender/core/util'
], function (require) {
    var zrUtil = require('zrender/core/util');
    var defaultOption = {
        show: true,
        zlevel: 0,
        z: 0,
        inverse: false,
        name: '',
        nameLocation: 'end',
        nameTextStyle: {},
        nameGap: 15,
        axisLine: {
            show: true,
            onZero: true,
            lineStyle: {
                color: '#333',
                width: 1,
                type: 'solid'
            }
        },
        axisTick: {
            show: true,
            inside: false,
            length: 5,
            lineStyle: {
                color: '#333',
                width: 1
            }
        },
        axisLabel: {
            show: true,
            inside: false,
            rotate: 0,
            margin: 8,
            textStyle: {
                color: '#333',
                fontSize: 12
            }
        },
        splitLine: {
            show: true,
            lineStyle: {
                color: ['#ccc'],
                width: 1,
                type: 'solid'
            }
        },
        splitArea: {
            show: false,
            areaStyle: {
                color: [
                    'rgba(250,250,250,0.3)',
                    'rgba(200,200,200,0.3)'
                ]
            }
        }
    };
    var categoryAxis = zrUtil.merge({
        boundaryGap: true,
        axisTick: { interval: 'auto' },
        axisLabel: { interval: 'auto' }
    }, defaultOption);
    var valueAxis = zrUtil.defaults({
        boundaryGap: [
            0,
            0
        ],
        splitNumber: 6
    }, defaultOption);
    var timeAxis = zrUtil.defaults({
        scale: true,
        min: 'dataMin',
        max: 'dataMax'
    }, valueAxis);
    var logAxis = zrUtil.defaults({}, valueAxis);
    logAxis.scale = true;
    return {
        categoryAxis: categoryAxis,
        valueAxis: valueAxis,
        timeAxis: timeAxis,
        logAxis: logAxis
    };
});define('echarts/chart/bar/barItemStyle', [
    'require',
    '../../model/mixin/makeStyleMapper'
], function (require) {
    return {
        getBarItemStyle: require('../../model/mixin/makeStyleMapper')([
            [
                'fill',
                'color'
            ],
            [
                'stroke',
                'barBorderColor'
            ],
            [
                'lineWidth',
                'barBorderWidth'
            ],
            ['opacity'],
            ['shadowBlur'],
            ['shadowOffsetX'],
            ['shadowOffsetY'],
            ['shadowColor']
        ])
    };
});