define('echarts/chart/line', [
    'require',
    'zrender/core/util',
    '../echarts',
    './line/LineSeries',
    './line/LineView',
    '../visual/symbol',
    '../layout/points'
], function (require) {
    var zrUtil = require('zrender/core/util');
    var echarts = require('../echarts');
    require('./line/LineSeries');
    require('./line/LineView');
    echarts.registerVisualCoding('chart', zrUtil.curry(require('../visual/symbol'), 'line', 'circle', 'line'));
    echarts.registerLayout(zrUtil.curry(require('../layout/points'), 'line'));
});define('echarts/chart/line/LineSeries', [
    'require',
    '../helper/createListFromArray',
    '../../model/Series'
], function (require) {
    'use strict';
    var createListFromArray = require('../helper/createListFromArray');
    var SeriesModel = require('../../model/Series');
    return SeriesModel.extend({
        type: 'series.line',
        dependencies: [
            'grid',
            'polar'
        ],
        getInitialData: function (option, ecModel) {
            return createListFromArray(option.data, this, ecModel);
        },
        defaultOption: {
            zlevel: 0,
            z: 2,
            coordinateSystem: 'cartesian2d',
            legendHoverLink: true,
            hoverAnimation: true,
            xAxisIndex: 0,
            yAxisIndex: 0,
            polarIndex: 0,
            label: {
                normal: { position: 'top' },
                emphasis: { position: 'top' }
            },
            lineStyle: {
                normal: {
                    width: 2,
                    type: 'solid'
                }
            },
            symbol: 'emptyCircle',
            symbolSize: 4,
            showSymbol: true,
            animationEasing: 'linear'
        }
    });
});define('echarts/chart/line/LineView', [
    'require',
    'zrender/core/util',
    '../helper/SymbolDraw',
    '../helper/Symbol',
    './lineAnimationDiff',
    '../../util/graphic',
    './poly',
    '../../view/Chart'
], function (require) {
    'use strict';
    var zrUtil = require('zrender/core/util');
    var SymbolDraw = require('../helper/SymbolDraw');
    var Symbol = require('../helper/Symbol');
    var lineAnimationDiff = require('./lineAnimationDiff');
    var graphic = require('../../util/graphic');
    var polyHelper = require('./poly');
    var ChartView = require('../../view/Chart');
    function isPointsSame(points1, points2) {
        if (points1.length !== points2.length) {
            return;
        }
        for (var i = 0; i < points1.length; i++) {
            var p1 = points1[i];
            var p2 = points2[i];
            if (p1[0] !== p2[0] || p1[1] !== p2[1]) {
                return;
            }
        }
        return true;
    }
    function getSmooth(smooth) {
        return typeof smooth === 'number' ? smooth : smooth ? 0.3 : 0;
    }
    function getAxisExtentWithGap(axis) {
        var extent = axis.getGlobalExtent();
        if (axis.onBand) {
            var halfBandWidth = axis.getBandWidth() / 2 - 1;
            var dir = extent[1] > extent[0] ? 1 : -1;
            extent[0] += dir * halfBandWidth;
            extent[1] -= dir * halfBandWidth;
        }
        return extent;
    }
    function sign(val) {
        return val >= 0 ? 1 : -1;
    }
    function getStackedOnPoints(coordSys, data) {
        var baseAxis = coordSys.getBaseAxis();
        var valueAxis = coordSys.getOtherAxis(baseAxis);
        var valueStart = baseAxis.onZero ? 0 : valueAxis.scale.getExtent()[0];
        var valueDim = valueAxis.dim;
        var baseDataOffset = valueDim === 'x' || valueDim === 'radius' ? 1 : 0;
        return data.mapArray([valueDim], function (val, idx) {
            var stackedOnSameSign;
            var stackedOn = data.stackedOn;
            while (stackedOn && sign(stackedOn.get(valueDim, idx)) === sign(val)) {
                stackedOnSameSign = stackedOn;
                break;
            }
            var stackedData = [];
            stackedData[baseDataOffset] = data.get(baseAxis.dim, idx);
            stackedData[1 - baseDataOffset] = stackedOnSameSign ? stackedOnSameSign.get(valueDim, idx, true) : valueStart;
            return coordSys.dataToPoint(stackedData);
        }, true);
    }
    function queryDataIndex(data, payload) {
        if (payload.dataIndex != null) {
            return payload.dataIndex;
        } else if (payload.name != null) {
            return data.indexOfName(payload.name);
        }
    }
    function createGridClipShape(cartesian, hasAnimation, seriesModel) {
        var xExtent = getAxisExtentWithGap(cartesian.getAxis('x'));
        var yExtent = getAxisExtentWithGap(cartesian.getAxis('y'));
        var clipPath = new graphic.Rect({
            shape: {
                x: xExtent[0],
                y: yExtent[0],
                width: xExtent[1] - xExtent[0],
                height: yExtent[1] - yExtent[0]
            }
        });
        if (hasAnimation) {
            clipPath.shape[cartesian.getBaseAxis().isHorizontal() ? 'width' : 'height'] = 0;
            graphic.initProps(clipPath, {
                shape: {
                    width: xExtent[1] - xExtent[0],
                    height: yExtent[1] - yExtent[0]
                }
            }, seriesModel);
        }
        return clipPath;
    }
    function createPolarClipShape(polar, hasAnimation, seriesModel) {
        var angleAxis = polar.getAngleAxis();
        var radiusAxis = polar.getRadiusAxis();
        var radiusExtent = radiusAxis.getExtent();
        var angleExtent = angleAxis.getExtent();
        var RADIAN = Math.PI / 180;
        var clipPath = new graphic.Sector({
            shape: {
                cx: polar.cx,
                cy: polar.cy,
                r0: radiusExtent[0],
                r: radiusExtent[1],
                startAngle: -angleExtent[0] * RADIAN,
                endAngle: -angleExtent[1] * RADIAN,
                clockwise: angleAxis.inverse
            }
        });
        if (hasAnimation) {
            clipPath.shape.endAngle = -angleExtent[0] * RADIAN;
            graphic.initProps(clipPath, { shape: { endAngle: -angleExtent[1] * RADIAN } }, seriesModel);
        }
        return clipPath;
    }
    function createClipShape(coordSys, hasAnimation, seriesModel) {
        return coordSys.type === 'polar' ? createPolarClipShape(coordSys, hasAnimation, seriesModel) : createGridClipShape(coordSys, hasAnimation, seriesModel);
    }
    return ChartView.extend({
        type: 'line',
        init: function () {
            var lineGroup = new graphic.Group();
            var symbolDraw = new SymbolDraw();
            this.group.add(symbolDraw.group);
            this.group.add(lineGroup);
            this._symbolDraw = symbolDraw;
            this._lineGroup = lineGroup;
        },
        render: function (seriesModel, ecModel, api) {
            var coordSys = seriesModel.coordinateSystem;
            var group = this.group;
            var data = seriesModel.getData();
            var lineStyleModel = seriesModel.getModel('lineStyle.normal');
            var areaStyleModel = seriesModel.getModel('areaStyle.normal');
            var points = data.mapArray(data.getItemLayout, true);
            var isCoordSysPolar = coordSys.type === 'polar';
            var prevCoordSys = this._coordSys;
            var symbolDraw = this._symbolDraw;
            var polyline = this._polyline;
            var polygon = this._polygon;
            var lineGroup = this._lineGroup;
            var hasAnimation = seriesModel.get('animation');
            var isAreaChart = !areaStyleModel.isEmpty();
            var stackedOnPoints = getStackedOnPoints(coordSys, data);
            var showSymbol = seriesModel.get('showSymbol');
            var isSymbolIgnore = showSymbol && !isCoordSysPolar && !seriesModel.get('showAllSymbol') && this._getSymbolIgnoreFunc(data, coordSys);
            var oldData = this._data;
            oldData && oldData.eachItemGraphicEl(function (el, idx) {
                if (el.__temp) {
                    group.remove(el);
                    oldData.setItemGraphicEl(idx, null);
                }
            });
            if (!showSymbol) {
                symbolDraw.remove();
            }
            if (!(polyline && prevCoordSys.type === coordSys.type)) {
                showSymbol && symbolDraw.updateData(data, isSymbolIgnore);
                polyline = this._newPolyline(group, points, coordSys, hasAnimation);
                if (isAreaChart) {
                    polygon = this._newPolygon(group, points, stackedOnPoints, coordSys, hasAnimation);
                }
                lineGroup.setClipPath(createClipShape(coordSys, true, seriesModel));
            } else {
                if (hasAnimation) {
                    lineGroup.setClipPath(createClipShape(coordSys, false, seriesModel));
                }
                showSymbol && symbolDraw.updateData(data, isSymbolIgnore);
                data.eachItemGraphicEl(function (el) {
                    el.stopSymbolAnimation(true);
                });
                if (!isPointsSame(this._stackedOnPoints, stackedOnPoints) || !isPointsSame(this._points, points)) {
                    if (hasAnimation) {
                        this._updateAnimation(data, stackedOnPoints, coordSys, api);
                    } else {
                        polyline.setShape({ points: points });
                        polygon && polygon.setShape({
                            points: points,
                            stackedOnPoints: stackedOnPoints
                        });
                    }
                }
                group.add(lineGroup);
            }
            polyline.setStyle(zrUtil.defaults(lineStyleModel.getLineStyle(), {
                stroke: data.getVisual('color'),
                lineJoin: 'bevel'
            }));
            var smooth = seriesModel.get('smooth');
            smooth = getSmooth(seriesModel.get('smooth'));
            polyline.shape.smooth = smooth;
            if (polygon) {
                var polygonShape = polygon.shape;
                var stackedOn = data.stackedOn;
                var stackedOnSmooth = 0;
                polygon.style.opacity = 0.7;
                polygon.setStyle(zrUtil.defaults(areaStyleModel.getAreaStyle(), {
                    fill: data.getVisual('color'),
                    lineJoin: 'bevel'
                }));
                polygonShape.smooth = smooth;
                if (stackedOn) {
                    var stackedOnSeries = stackedOn.hostModel;
                    stackedOnSmooth = getSmooth(stackedOnSeries.get('smooth'));
                }
                polygonShape.stackedOnSmooth = stackedOnSmooth;
            }
            this._data = data;
            this._coordSys = coordSys;
            this._stackedOnPoints = stackedOnPoints;
            this._points = points;
        },
        highlight: function (seriesModel, ecModel, api, payload) {
            var data = seriesModel.getData();
            var dataIndex = queryDataIndex(data, payload);
            if (dataIndex != null && dataIndex >= 0) {
                var symbol = data.getItemGraphicEl(dataIndex);
                if (!symbol) {
                    var pt = data.getItemLayout(dataIndex);
                    symbol = new Symbol(data, dataIndex, api);
                    symbol.position = pt;
                    symbol.setZ(seriesModel.get('zlevel'), seriesModel.get('z'));
                    symbol.ignore = isNaN(pt[0]) || isNaN(pt[1]);
                    symbol.__temp = true;
                    data.setItemGraphicEl(dataIndex, symbol);
                    symbol.stopSymbolAnimation(true);
                    this.group.add(symbol);
                }
                symbol.highlight();
            } else {
                ChartView.prototype.highlight.call(this, seriesModel, ecModel, api, payload);
            }
        },
        downplay: function (seriesModel, ecModel, api, payload) {
            var data = seriesModel.getData();
            var dataIndex = queryDataIndex(data, payload);
            if (dataIndex != null && dataIndex >= 0) {
                var symbol = data.getItemGraphicEl(dataIndex);
                if (symbol) {
                    if (symbol.__temp) {
                        data.setItemGraphicEl(dataIndex, null);
                        this.group.remove(symbol);
                    } else {
                        symbol.downplay();
                    }
                }
            } else {
                ChartView.prototype.downplay.call(this, seriesModel, ecModel, api, payload);
            }
        },
        _newPolyline: function (group, points) {
            var polyline = this._polyline;
            if (polyline) {
                group.remove(polyline);
            }
            polyline = new polyHelper.Polyline({
                shape: { points: points },
                silent: true,
                z2: 10
            });
            this._lineGroup.add(polyline);
            this._polyline = polyline;
            return polyline;
        },
        _newPolygon: function (group, points, stackedOnPoints) {
            var polygon = this._polygon;
            if (polygon) {
                group.remove(polygon);
            }
            polygon = new polyHelper.Polygon({
                shape: {
                    points: points,
                    stackedOnPoints: stackedOnPoints
                },
                silent: true
            });
            this._lineGroup.add(polygon);
            this._polygon = polygon;
            return polygon;
        },
        _getSymbolIgnoreFunc: function (data, coordSys) {
            var categoryAxis = coordSys.getAxesByScale('ordinal')[0];
            if (categoryAxis && categoryAxis.isLabelIgnored) {
                return zrUtil.bind(categoryAxis.isLabelIgnored, categoryAxis);
            }
        },
        _updateAnimation: function (data, stackedOnPoints, coordSys, api) {
            var polyline = this._polyline;
            var polygon = this._polygon;
            var seriesModel = data.hostModel;
            var diff = lineAnimationDiff(this._data, data, this._stackedOnPoints, stackedOnPoints, this._coordSys, coordSys);
            polyline.shape.points = diff.current;
            graphic.updateProps(polyline, { shape: { points: diff.next } }, seriesModel);
            if (polygon) {
                polygon.setShape({
                    points: diff.current,
                    stackedOnPoints: diff.stackedOnCurrent
                });
                graphic.updateProps(polygon, {
                    shape: {
                        points: diff.next,
                        stackedOnPoints: diff.stackedOnNext
                    }
                }, seriesModel);
            }
            var updatedDataInfo = [];
            var diffStatus = diff.status;
            for (var i = 0; i < diffStatus.length; i++) {
                var cmd = diffStatus[i].cmd;
                if (cmd === '=') {
                    var el = data.getItemGraphicEl(diffStatus[i].idx1);
                    if (el) {
                        updatedDataInfo.push({
                            el: el,
                            ptIdx: i
                        });
                    }
                }
            }
            if (polyline.animators && polyline.animators.length) {
                polyline.animators[0].during(function () {
                    for (var i = 0; i < updatedDataInfo.length; i++) {
                        var el = updatedDataInfo[i].el;
                        el.attr('position', polyline.shape.points[updatedDataInfo[i].ptIdx]);
                    }
                });
            }
        },
        remove: function (ecModel) {
            var group = this.group;
            group.remove(this._lineGroup);
            this._symbolDraw.remove(true);
        }
    });
});define('echarts/visual/symbol', ['require'], function (require) {
    function isSymbolNone(symbolType) {
        return symbolType === 'none';
    }
    return function (seriesType, defaultSymbolType, legendSymbol, ecModel, api) {
        ecModel.eachRawSeriesByType(seriesType, function (seriesModel) {
            var data = seriesModel.getData();
            var symbolType = seriesModel.get('symbol') || defaultSymbolType;
            var symbolSize = seriesModel.get('symbolSize');
            data.setVisual({
                legendSymbol: legendSymbol || symbolType,
                symbol: symbolType,
                symbolSize: symbolSize
            });
            if (!ecModel.isSeriesFiltered(seriesModel)) {
                if (typeof symbolSize === 'function') {
                    data.each(function (idx) {
                        var rawValue = seriesModel.getRawValue(idx);
                        var params = seriesModel.getDataParams(idx);
                        data.setItemVisual(idx, 'symbolSize', symbolSize(rawValue, params));
                    });
                }
                data.each(function (idx) {
                    var itemModel = data.getItemModel(idx);
                    var itemSymbolType = itemModel.get('symbol', true);
                    var itemSymbolSize = itemModel.get('symbolSize', true);
                    if (itemSymbolType != null) {
                        data.setItemVisual(idx, 'symbol', itemSymbolType);
                    }
                    if (itemSymbolSize != null) {
                        data.setItemVisual(idx, 'symbolSize', itemSymbolSize);
                    }
                });
            }
        });
    };
});define('echarts/layout/points', ['require'], function (require) {
    return function (seriesType, ecModel, api) {
        ecModel.eachSeriesByType(seriesType, function (seriesModel) {
            var data = seriesModel.getData();
            var coordSys = seriesModel.coordinateSystem;
            var dims = coordSys.dimensions;
            data.each(dims, function (x, y, idx) {
                var point;
                if (!isNaN(x) && !isNaN(y)) {
                    point = coordSys.dataToPoint([
                        x,
                        y
                    ]);
                } else {
                    point = [
                        NaN,
                        NaN
                    ];
                }
                data.setItemLayout(idx, point);
            }, true);
        });
    };
});define('echarts/chart/helper/SymbolDraw', [
    'require',
    '../../util/graphic',
    './Symbol'
], function (require) {
    var graphic = require('../../util/graphic');
    var Symbol = require('./Symbol');
    function SymbolDraw(symbolCtor) {
        this.group = new graphic.Group();
        this._symbolCtor = symbolCtor || Symbol;
    }
    var symbolDrawProto = SymbolDraw.prototype;
    function symbolNeedsDraw(data, idx, isIgnore) {
        var point = data.getItemLayout(idx);
        return point && !isNaN(point[0]) && !isNaN(point[1]) && !(isIgnore && isIgnore(idx)) && data.getItemVisual(idx, 'symbol') !== 'none';
    }
    symbolDrawProto.updateData = function (data, isIgnore) {
        var group = this.group;
        var seriesModel = data.hostModel;
        var oldData = this._data;
        var SymbolCtor = this._symbolCtor;
        data.diff(oldData).add(function (newIdx) {
            var point = data.getItemLayout(newIdx);
            if (symbolNeedsDraw(data, newIdx, isIgnore)) {
                var symbolEl = new SymbolCtor(data, newIdx);
                symbolEl.attr('position', point);
                data.setItemGraphicEl(newIdx, symbolEl);
                group.add(symbolEl);
            }
        }).update(function (newIdx, oldIdx) {
            var symbolEl = oldData.getItemGraphicEl(oldIdx);
            var point = data.getItemLayout(newIdx);
            if (!symbolNeedsDraw(data, newIdx, isIgnore)) {
                group.remove(symbolEl);
                return;
            }
            if (!symbolEl) {
                symbolEl = new SymbolCtor(data, newIdx);
                symbolEl.attr('position', point);
            } else {
                symbolEl.updateData(data, newIdx);
                graphic.updateProps(symbolEl, { position: point }, seriesModel);
            }
            group.add(symbolEl);
            data.setItemGraphicEl(newIdx, symbolEl);
        }).remove(function (oldIdx) {
            var el = oldData.getItemGraphicEl(oldIdx);
            el && el.fadeOut(function () {
                group.remove(el);
            });
        }).execute();
        this._data = data;
    };
    symbolDrawProto.updateLayout = function () {
        var data = this._data;
        if (data) {
            data.eachItemGraphicEl(function (el, idx) {
                el.attr('position', data.getItemLayout(idx));
            });
        }
    };
    symbolDrawProto.remove = function (enableAnimation) {
        var group = this.group;
        var data = this._data;
        if (data) {
            if (enableAnimation) {
                data.eachItemGraphicEl(function (el) {
                    el.fadeOut(function () {
                        group.remove(el);
                    });
                });
            } else {
                group.removeAll();
            }
        }
    };
    return SymbolDraw;
});define('echarts/chart/helper/Symbol', [
    'require',
    'zrender/core/util',
    '../../util/symbol',
    '../../util/graphic',
    '../../util/number'
], function (require) {
    var zrUtil = require('zrender/core/util');
    var symbolUtil = require('../../util/symbol');
    var graphic = require('../../util/graphic');
    var numberUtil = require('../../util/number');
    function normalizeSymbolSize(symbolSize) {
        if (!zrUtil.isArray(symbolSize)) {
            symbolSize = [
                +symbolSize,
                +symbolSize
            ];
        }
        return symbolSize;
    }
    function Symbol(data, idx) {
        graphic.Group.call(this);
        this.updateData(data, idx);
    }
    var symbolProto = Symbol.prototype;
    function driftSymbol(dx, dy) {
        this.parent.drift(dx, dy);
    }
    symbolProto._createSymbol = function (symbolType, data, idx) {
        this.removeAll();
        var seriesModel = data.hostModel;
        var color = data.getItemVisual(idx, 'color');
        var symbolPath = symbolUtil.createSymbol(symbolType, -0.5, -0.5, 1, 1, color);
        symbolPath.attr({
            style: { strokeNoScale: true },
            z2: 100,
            scale: [
                0,
                0
            ]
        });
        symbolPath.drift = driftSymbol;
        var size = normalizeSymbolSize(data.getItemVisual(idx, 'symbolSize'));
        graphic.initProps(symbolPath, { scale: size }, seriesModel);
        this._symbolType = symbolType;
        this.add(symbolPath);
    };
    symbolProto.stopSymbolAnimation = function (toLastFrame) {
        this.childAt(0).stopAnimation(toLastFrame);
    };
    symbolProto.getScale = function () {
        return this.childAt(0).scale;
    };
    symbolProto.highlight = function () {
        this.childAt(0).trigger('emphasis');
    };
    symbolProto.downplay = function () {
        this.childAt(0).trigger('normal');
    };
    symbolProto.setZ = function (zlevel, z) {
        var symbolPath = this.childAt(0);
        symbolPath.zlevel = zlevel;
        symbolPath.z = z;
    };
    symbolProto.setDraggable = function (draggable) {
        var symbolPath = this.childAt(0);
        symbolPath.draggable = draggable;
        symbolPath.cursor = draggable ? 'move' : 'pointer';
    };
    symbolProto.updateData = function (data, idx) {
        var symbolType = data.getItemVisual(idx, 'symbol') || 'circle';
        var seriesModel = data.hostModel;
        var symbolSize = normalizeSymbolSize(data.getItemVisual(idx, 'symbolSize'));
        if (symbolType !== this._symbolType) {
            this._createSymbol(symbolType, data, idx);
        } else {
            var symbolPath = this.childAt(0);
            graphic.updateProps(symbolPath, { scale: symbolSize }, seriesModel);
        }
        this._updateCommon(data, idx, symbolSize);
        this._seriesModel = seriesModel;
    };
    var normalStyleAccessPath = [
        'itemStyle',
        'normal'
    ];
    var emphasisStyleAccessPath = [
        'itemStyle',
        'emphasis'
    ];
    var normalLabelAccessPath = [
        'label',
        'normal'
    ];
    var emphasisLabelAccessPath = [
        'label',
        'emphasis'
    ];
    symbolProto._updateCommon = function (data, idx, symbolSize) {
        var symbolPath = this.childAt(0);
        var seriesModel = data.hostModel;
        var itemModel = data.getItemModel(idx);
        var normalItemStyleModel = itemModel.getModel(normalStyleAccessPath);
        var color = data.getItemVisual(idx, 'color');
        var hoverStyle = itemModel.getModel(emphasisStyleAccessPath).getItemStyle();
        symbolPath.rotation = itemModel.getShallow('symbolRotate') * Math.PI / 180 || 0;
        var symbolOffset = itemModel.getShallow('symbolOffset');
        if (symbolOffset) {
            var pos = symbolPath.position;
            pos[0] = numberUtil.parsePercent(symbolOffset[0], symbolSize[0]);
            pos[1] = numberUtil.parsePercent(symbolOffset[1], symbolSize[1]);
        }
        symbolPath.setColor(color);
        zrUtil.extend(symbolPath.style, normalItemStyleModel.getItemStyle(['color']));
        var labelModel = itemModel.getModel(normalLabelAccessPath);
        var hoverLabelModel = itemModel.getModel(emphasisLabelAccessPath);
        var lastDim = data.dimensions[data.dimensions.length - 1];
        var labelText = seriesModel.getFormattedLabel(idx, 'normal') || data.get(lastDim, idx);
        var elStyle = symbolPath.style;
        if (labelModel.get('show')) {
            graphic.setText(elStyle, labelModel, color);
            elStyle.text = labelText;
        } else {
            elStyle.text = '';
        }
        if (hoverLabelModel.getShallow('show')) {
            graphic.setText(hoverStyle, hoverLabelModel, color);
            hoverStyle.text = labelText;
        } else {
            hoverStyle.text = '';
        }
        graphic.setHoverStyle(symbolPath, hoverStyle);
        var size = normalizeSymbolSize(data.getItemVisual(idx, 'symbolSize'));
        symbolPath.off('mouseover').off('mouseout').off('emphasis').off('normal');
        if (itemModel.getShallow('hoverAnimation')) {
            var onEmphasis = function () {
                var ratio = size[1] / size[0];
                this.animateTo({
                    scale: [
                        Math.max(size[0] * 1.1, size[0] + 6),
                        Math.max(size[1] * 1.1, size[1] + 6 * ratio)
                    ]
                }, 400, 'elasticOut');
            };
            var onNormal = function () {
                this.animateTo({ scale: size }, 400, 'elasticOut');
            };
            symbolPath.on('mouseover', onEmphasis).on('mouseout', onNormal).on('emphasis', onEmphasis).on('normal', onNormal);
        }
    };
    symbolProto.fadeOut = function (cb) {
        var symbolPath = this.childAt(0);
        symbolPath.style.text = '';
        graphic.updateProps(symbolPath, {
            scale: [
                0,
                0
            ]
        }, this._seriesModel, cb);
    };
    zrUtil.inherits(Symbol, graphic.Group);
    return Symbol;
});define('echarts/chart/line/lineAnimationDiff', [
    'require',
    'zrender/core/arrayDiff'
], function (require) {
    var arrayDiff = require('zrender/core/arrayDiff');
    function sign(val) {
        return val >= 0 ? 1 : -1;
    }
    function getStackedOnPoint(coordSys, data, idx) {
        var baseAxis = coordSys.getBaseAxis();
        var valueAxis = coordSys.getOtherAxis(baseAxis);
        var valueStart = baseAxis.onZero ? 0 : valueAxis.scale.getExtent()[0];
        var valueDim = valueAxis.dim;
        var baseDataOffset = valueDim === 'x' || valueDim === 'radius' ? 1 : 0;
        var stackedOnSameSign;
        var stackedOn = data.stackedOn;
        var val = data.get(valueDim, idx);
        while (stackedOn && sign(stackedOn.get(valueDim, idx)) === sign(val)) {
            stackedOnSameSign = stackedOn;
            break;
        }
        var stackedData = [];
        stackedData[baseDataOffset] = data.get(baseAxis.dim, idx);
        stackedData[1 - baseDataOffset] = stackedOnSameSign ? stackedOnSameSign.get(valueDim, idx, true) : valueStart;
        return coordSys.dataToPoint(stackedData);
    }
    return function (oldData, newData, oldStackedOnPoints, newStackedOnPoints, oldCoordSys, newCoordSys) {
        var newIdList = newData.mapArray(newData.getId);
        var oldIdList = oldData.mapArray(oldData.getId);
        var currPoints = [];
        var nextPoints = [];
        var currStackedPoints = [];
        var nextStackedPoints = [];
        var status = [];
        var sortedIndices = [];
        var rawIndices = [];
        var diff = arrayDiff(oldIdList, newIdList);
        var dims = newCoordSys.dimensions;
        for (var i = 0; i < diff.length; i++) {
            var diffItem = diff[i];
            var pointAdded = true;
            switch (diffItem.cmd) {
            case '=':
                currPoints.push(oldData.getItemLayout(diffItem.idx));
                nextPoints.push(newData.getItemLayout(diffItem.idx1));
                currStackedPoints.push(oldStackedOnPoints[diffItem.idx]);
                nextStackedPoints.push(newStackedOnPoints[diffItem.idx1]);
                rawIndices.push(newData.getRawIndex(diffItem.idx1));
                break;
            case '+':
                var idx = diffItem.idx;
                currPoints.push(oldCoordSys.dataToPoint([
                    newData.get(dims[0], idx, true),
                    newData.get(dims[1], idx, true)
                ]));
                nextPoints.push(newData.getItemLayout(idx).slice());
                currStackedPoints.push(getStackedOnPoint(oldCoordSys, newData, idx));
                nextStackedPoints.push(newStackedOnPoints[idx]);
                rawIndices.push(newData.getRawIndex(idx));
                break;
            case '-':
                var idx = diffItem.idx;
                var rawIndex = oldData.getRawIndex(idx);
                if (rawIndex !== idx) {
                    currPoints.push(oldData.getItemLayout(idx));
                    nextPoints.push(newCoordSys.dataToPoint([
                        oldData.get(dims[0], idx, true),
                        oldData.get(dims[1], idx, true)
                    ]));
                    currStackedPoints.push(oldStackedOnPoints[idx]);
                    nextStackedPoints.push(getStackedOnPoint(newCoordSys, oldData, idx));
                    rawIndices.push(rawIndex);
                } else {
                    pointAdded = false;
                }
            }
            if (pointAdded) {
                status.push(diffItem);
                sortedIndices.push(sortedIndices.length);
            }
        }
        sortedIndices.sort(function (a, b) {
            return rawIndices[a] - rawIndices[b];
        });
        var sortedCurrPoints = [];
        var sortedNextPoints = [];
        var sortedCurrStackedPoints = [];
        var sortedNextStackedPoints = [];
        var sortedStatus = [];
        for (var i = 0; i < sortedIndices.length; i++) {
            var idx = sortedIndices[i];
            sortedCurrPoints[i] = currPoints[idx];
            sortedNextPoints[i] = nextPoints[idx];
            sortedCurrStackedPoints[i] = currStackedPoints[idx];
            sortedNextStackedPoints[i] = nextStackedPoints[idx];
            sortedStatus[i] = status[idx];
        }
        return {
            current: sortedCurrPoints,
            next: sortedNextPoints,
            stackedOnCurrent: sortedCurrStackedPoints,
            stackedOnNext: sortedNextStackedPoints,
            status: sortedStatus
        };
    };
});define('echarts/chart/line/poly', [
    'require',
    'zrender/graphic/Path',
    'zrender/core/vector',
    'zrender/core/bbox'
], function (require) {
    var Path = require('zrender/graphic/Path');
    var vec2 = require('zrender/core/vector');
    var bbox = require('zrender/core/bbox');
    var mathMin = Math.min;
    var mathMax = Math.max;
    var vec2Min = vec2.min;
    var vec2Max = vec2.max;
    var scaleAndAdd = vec2.scaleAndAdd;
    var v2Copy = vec2.copy;
    var v = [];
    var cp0 = [];
    var cp1 = [];
    function drawSegment(ctx, points, start, allLen, segLen, dir, smoothMin, smoothMax, smooth) {
        var idx = start;
        for (var k = 0; k < segLen; k++) {
            var p = points[idx];
            if (idx >= allLen || idx < 0 || isNaN(p[0]) || isNaN(p[1])) {
                break;
            }
            if (idx === start) {
                ctx[dir > 0 ? 'moveTo' : 'lineTo'](p[0], p[1]);
                v2Copy(cp0, p);
            } else {
                if (smooth > 0) {
                    var prevIdx = idx - dir;
                    var nextIdx = idx + dir;
                    if (dir > 0) {
                        prevIdx = mathMax(prevIdx, start);
                        nextIdx = mathMin(nextIdx, allLen - 1);
                    } else {
                        nextIdx = mathMax(nextIdx, 0);
                        prevIdx = mathMin(prevIdx, start);
                    }
                    var prevP = points[prevIdx];
                    var nextP = points[nextIdx];
                    if (isNaN(nextP[0]) || isNaN(nextP[1])) {
                        nextP = p;
                    }
                    vec2.sub(v, nextP, prevP);
                    scaleAndAdd(cp1, p, v, -smooth / 2);
                    ctx.bezierCurveTo(cp0[0], cp0[1], cp1[0], cp1[1], p[0], p[1]);
                    vec2Min(cp0, cp0, smoothMin);
                    vec2Max(cp0, cp0, smoothMax);
                    vec2Min(cp1, cp1, smoothMin);
                    vec2Max(cp1, cp1, smoothMax);
                    scaleAndAdd(cp0, p, v, smooth / 2);
                } else {
                    ctx.lineTo(p[0], p[1]);
                }
            }
            idx += dir;
        }
        return k;
    }
    return {
        Polyline: Path.extend({
            type: 'ec-polyline',
            shape: {
                points: [],
                smooth: 0
            },
            style: {
                fill: null,
                stroke: '#000',
                smooth: 0
            },
            buildPath: function (ctx, shape) {
                var points = shape.points;
                var i = 0;
                var len = points.length;
                var ptMin = [];
                var ptMax = [];
                bbox.fromPoints(points, ptMin, ptMax);
                while (i < len) {
                    i += drawSegment(ctx, points, i, len, len, 1, ptMin, ptMax, shape.smooth) + 1;
                }
            }
        }),
        Polygon: Path.extend({
            type: 'ec-polygon',
            shape: {
                points: [],
                stackedOnPoints: [],
                smooth: 0,
                stackedOnSmooth: 0
            },
            buildPath: function (ctx, shape) {
                var points = shape.points;
                var stackedOnPoints = shape.stackedOnPoints;
                var i = 0;
                var len = points.length;
                var ptMin = [];
                var ptMax = [];
                bbox.fromPoints(points, ptMin, ptMax);
                while (i < len) {
                    var k = drawSegment(ctx, points, i, len, len, 1, ptMin, ptMax, shape.smooth);
                    drawSegment(ctx, stackedOnPoints, i + k - 1, len, k, -1, ptMin, ptMax, shape.stackedOnSmooth);
                    i += k + 1;
                    ctx.closePath();
                }
            }
        })
    };
});define('echarts/util/symbol', [
    'require',
    './graphic',
    'zrender/core/BoundingRect'
], function (require) {
    'use strict';
    var graphic = require('./graphic');
    var BoundingRect = require('zrender/core/BoundingRect');
    var Triangle = graphic.extendShape({
        type: 'triangle',
        shape: {
            cx: 0,
            cy: 0,
            width: 0,
            height: 0
        },
        buildPath: function (path, shape) {
            var cx = shape.cx;
            var cy = shape.cy;
            var width = shape.width / 2;
            var height = shape.height / 2;
            path.moveTo(cx, cy - height);
            path.lineTo(cx + width, cy + height);
            path.lineTo(cx - width, cy + height);
            path.closePath();
        }
    });
    var Diamond = graphic.extendShape({
        type: 'diamond',
        shape: {
            cx: 0,
            cy: 0,
            width: 0,
            height: 0
        },
        buildPath: function (path, shape) {
            var cx = shape.cx;
            var cy = shape.cy;
            var width = shape.width / 2;
            var height = shape.height / 2;
            path.moveTo(cx, cy - height);
            path.lineTo(cx + width, cy);
            path.lineTo(cx, cy + height);
            path.lineTo(cx - width, cy);
            path.closePath();
        }
    });
    var Pin = graphic.extendShape({
        type: 'pin',
        shape: {
            x: 0,
            y: 0,
            width: 0,
            height: 0
        },
        buildPath: function (path, shape) {
            var x = shape.x;
            var y = shape.y;
            var w = shape.width / 5 * 3;
            var h = Math.max(w, shape.height);
            var r = w / 2;
            var dy = r * r / (h - r);
            var cy = y - h + r + dy;
            var angle = Math.asin(dy / r);
            var dx = Math.cos(angle) * r;
            var tanX = Math.sin(angle);
            var tanY = Math.cos(angle);
            path.arc(x, cy, r, Math.PI - angle, Math.PI * 2 + angle);
            var cpLen = r * 0.6;
            var cpLen2 = r * 0.7;
            path.bezierCurveTo(x + dx - tanX * cpLen, cy + dy + tanY * cpLen, x, y - cpLen2, x, y);
            path.bezierCurveTo(x, y - cpLen2, x - dx + tanX * cpLen, cy + dy + tanY * cpLen, x - dx, cy + dy);
            path.closePath();
        }
    });
    var Arrow = graphic.extendShape({
        type: 'arrow',
        shape: {
            x: 0,
            y: 0,
            width: 0,
            height: 0
        },
        buildPath: function (ctx, shape) {
            var height = shape.height;
            var width = shape.width;
            var x = shape.x;
            var y = shape.y;
            var dx = width / 3 * 2;
            ctx.moveTo(x, y);
            ctx.lineTo(x + dx, y + height);
            ctx.lineTo(x, y + height / 4 * 3);
            ctx.lineTo(x - dx, y + height);
            ctx.lineTo(x, y);
            ctx.closePath();
        }
    });
    var symbolCtors = {
        line: graphic.Line,
        rect: graphic.Rect,
        roundRect: graphic.Rect,
        square: graphic.Rect,
        circle: graphic.Circle,
        diamond: Diamond,
        pin: Pin,
        arrow: Arrow,
        triangle: Triangle
    };
    var symbolShapeMakers = {
        line: function (x, y, w, h, shape) {
            shape.x1 = x;
            shape.y1 = y + h / 2;
            shape.x2 = x + w;
            shape.y2 = y + h / 2;
        },
        rect: function (x, y, w, h, shape) {
            shape.x = x;
            shape.y = y;
            shape.width = w;
            shape.height = h;
        },
        roundRect: function (x, y, w, h, shape) {
            shape.x = x;
            shape.y = y;
            shape.width = w;
            shape.height = h;
            shape.r = Math.min(w, h) / 4;
        },
        square: function (x, y, w, h, shape) {
            var size = Math.min(w, h);
            shape.x = x;
            shape.y = y;
            shape.width = size;
            shape.height = size;
        },
        circle: function (x, y, w, h, shape) {
            shape.cx = x + w / 2;
            shape.cy = y + h / 2;
            shape.r = Math.min(w, h) / 2;
        },
        diamond: function (x, y, w, h, shape) {
            shape.cx = x + w / 2;
            shape.cy = y + h / 2;
            shape.width = w;
            shape.height = h;
        },
        pin: function (x, y, w, h, shape) {
            shape.x = x + w / 2;
            shape.y = y + h / 2;
            shape.width = w;
            shape.height = h;
        },
        arrow: function (x, y, w, h, shape) {
            shape.x = x + w / 2;
            shape.y = y + h / 2;
            shape.width = w;
            shape.height = h;
        },
        triangle: function (x, y, w, h, shape) {
            shape.cx = x + w / 2;
            shape.cy = y + h / 2;
            shape.width = w;
            shape.height = h;
        }
    };
    var symbolBuildProxies = {};
    for (var name in symbolCtors) {
        symbolBuildProxies[name] = new symbolCtors[name]();
    }
    var Symbol = graphic.extendShape({
        type: 'symbol',
        shape: {
            symbolType: '',
            x: 0,
            y: 0,
            width: 0,
            height: 0
        },
        beforeBrush: function () {
            var style = this.style;
            var shape = this.shape;
            if (shape.symbolType === 'pin' && style.textPosition === 'inside') {
                style.textPosition = [
                    '50%',
                    '40%'
                ];
                style.textAlign = 'center';
                style.textBaseline = 'middle';
            }
        },
        buildPath: function (ctx, shape) {
            var symbolType = shape.symbolType;
            var proxySymbol = symbolBuildProxies[symbolType];
            if (shape.symbolType !== 'none') {
                if (!proxySymbol) {
                    symbolType = 'rect';
                    proxySymbol = symbolBuildProxies[symbolType];
                }
                symbolShapeMakers[symbolType](shape.x, shape.y, shape.width, shape.height, proxySymbol.shape);
                proxySymbol.buildPath(ctx, proxySymbol.shape);
            }
        }
    });
    var symbolPathSetColor = function (color) {
        if (this.type !== 'image') {
            var symbolStyle = this.style;
            var symbolShape = this.shape;
            if (symbolShape && symbolShape.symbolType === 'line') {
                symbolStyle.stroke = color;
            } else if (this.__isEmptyBrush) {
                symbolStyle.stroke = color;
                symbolStyle.fill = '#fff';
            } else {
                symbolStyle.fill && (symbolStyle.fill = color);
                symbolStyle.stroke && (symbolStyle.stroke = color);
            }
            this.dirty();
        }
    };
    var symbolUtil = {
        createSymbol: function (symbolType, x, y, w, h, color) {
            var isEmpty = symbolType.indexOf('empty') === 0;
            if (isEmpty) {
                symbolType = symbolType.substr(5, 1).toLowerCase() + symbolType.substr(6);
            }
            var symbolPath;
            if (symbolType.indexOf('image://') === 0) {
                symbolPath = new graphic.Image({
                    style: {
                        image: symbolType.slice(8),
                        x: x,
                        y: y,
                        width: w,
                        height: h
                    }
                });
            } else if (symbolType.indexOf('path://') === 0) {
                symbolPath = graphic.makePath(symbolType.slice(7), {}, new BoundingRect(x, y, w, h));
            } else {
                symbolPath = new Symbol({
                    shape: {
                        symbolType: symbolType,
                        x: x,
                        y: y,
                        width: w,
                        height: h
                    }
                });
            }
            symbolPath.__isEmptyBrush = isEmpty;
            symbolPath.setColor = symbolPathSetColor;
            symbolPath.setColor(color);
            return symbolPath;
        }
    };
    return symbolUtil;
});define('zrender/core/arrayDiff', ['require'], function (require) {
    function defaultCompareFunc(a, b) {
        return a === b;
    }
    function createItem(cmd, idx, idx1) {
        var res = {
            cmd: cmd,
            idx: idx
        };
        if (cmd === '=') {
            res.idx1 = idx1;
        }
        return res;
    }
    ;
    function append(out, cmd, idx, idx1) {
        out.push(createItem(cmd, idx, idx1));
    }
    var abs = Math.abs;
    function score(arr0, arr1, i0, i1, j0, j1, equal, memo) {
        var last;
        var invM = i0 > i1;
        var invN = j0 > j1;
        var m = abs(i1 - i0);
        var n = abs(j1 - j0);
        var i;
        var j;
        for (i = 0; i <= m; i++) {
            for (j = 0; j <= n; j++) {
                if (i === 0) {
                    memo[j] = j;
                } else if (j === 0) {
                    last = memo[j];
                    memo[j] = i;
                } else {
                    var val0 = arr0[invM ? i0 - i : i - 1 + i0];
                    var val1 = arr1[invN ? j0 - j : j - 1 + j0];
                    var score0 = last + (equal(val0, val1) ? 0 : 2);
                    var score1 = memo[j] + 1;
                    var score2 = memo[j - 1] + 1;
                    last = memo[j];
                    memo[j] = score0 < score1 ? score0 : score1;
                    score2 < memo[j] && (memo[j] = score2);
                }
            }
        }
        return memo;
    }
    function hirschberg(arr0, arr1, i0, i1, j0, j1, equal, score0, score1) {
        var out = [];
        var len0 = i1 - i0;
        var len1 = j1 - j0;
        var i;
        var j;
        if (!len0) {
            for (j = 0; j < len1; j++) {
                append(out, '+', j + j0);
            }
        } else if (!len1) {
            for (i = 0; i < len0; i++) {
                append(out, '-', i + i0);
            }
        } else if (len0 === 1) {
            var a = arr0[i0];
            var matched = false;
            for (j = 0; j < len1; j++) {
                if (equal(a, arr1[j + j0]) && !matched) {
                    matched = true;
                    append(out, '=', i0, j + j0);
                } else {
                    append(out, '+', j + j0);
                }
            }
            if (!matched) {
                append(out, '-', i0);
            }
        } else if (len1 === 1) {
            var b = arr1[j0];
            var matched = false;
            for (i = 0; i < len0; i++) {
                if (equal(b, arr0[i + i0]) && !matched) {
                    matched = true;
                    append(out, '=', i + i0, j0);
                } else {
                    append(out, '-', i + i0);
                }
            }
            if (!matched) {
                append(out, '+', j0);
            }
        } else {
            var imid = (len0 / 2 | 0) + i0;
            score(arr0, arr1, i0, imid, j0, j1, equal, score0);
            score(arr0, arr1, i1, imid + 1, j1, j0, equal, score1);
            var min = Infinity;
            var jmid = 0;
            var sum;
            for (j = 0; j <= len1; j++) {
                sum = score0[j] + score1[len1 - j];
                if (sum < min) {
                    min = sum;
                    jmid = j;
                }
            }
            jmid += j0;
            out = hirschberg(arr0, arr1, i0, imid, j0, jmid, equal, score0, score1);
            var out1 = hirschberg(arr0, arr1, imid, i1, jmid, j1, equal, score0, score1);
            for (i = 0; i < out1.length; i++) {
                out.push(out1[i]);
            }
        }
        return out;
    }
    function arrayDiff(arr0, arr1, equal) {
        equal = equal || defaultCompareFunc;
        var i;
        var j;
        var len0 = arr0.length;
        var len1 = arr1.length;
        var lenMin = Math.min(len0, len1);
        var head = [];
        for (i = 0; i < lenMin; i++) {
            if (!equal(arr0[i], arr1[i])) {
                break;
            }
            append(head, '=', i, i);
        }
        for (j = 0; j < lenMin; j++) {
            if (!equal(arr0[len0 - j - 1], arr1[len1 - j - 1])) {
                break;
            }
        }
        if (len0 - j >= i || len1 - j >= i) {
            var middle = hirschberg(arr0, arr1, i, len0 - j, i, len1 - j, equal, [], []);
            for (i = 0; i < middle.length; i++) {
                head.push(middle[i]);
            }
            for (i = 0; i < j; i++) {
                append(head, '=', len0 - j + i, len1 - j + i);
            }
        }
        return head;
    }
    return arrayDiff;
});