define('echarts/chart/pie', [
    'require',
    'zrender/core/util',
    '../echarts',
    './pie/PieSeries',
    './pie/PieView',
    '../action/createDataSelectAction',
    '../visual/dataColor',
    './pie/pieLayout',
    '../processor/dataFilter'
], function (require) {
    var zrUtil = require('zrender/core/util');
    var echarts = require('../echarts');
    require('./pie/PieSeries');
    require('./pie/PieView');
    require('../action/createDataSelectAction')('pie', {
        type: 'pieToggleSelect',
        event: 'pieSelected',
        update: 'updateView'
    });
    echarts.registerVisualCoding('chart', zrUtil.curry(require('../visual/dataColor'), 'pie'));
    echarts.registerLayout(zrUtil.curry(require('./pie/pieLayout'), 'pie'));
    echarts.registerProcessor('filter', zrUtil.curry(require('../processor/dataFilter'), 'pie'));
});define('echarts/chart/pie/PieSeries', [
    'require',
    '../../data/List',
    'zrender/core/util',
    '../../util/model',
    '../helper/dataSelectableMixin',
    '../../echarts'
], function (require) {
    'use strict';
    var List = require('../../data/List');
    var zrUtil = require('zrender/core/util');
    var modelUtil = require('../../util/model');
    var dataSelectableMixin = require('../helper/dataSelectableMixin');
    var PieSeries = require('../../echarts').extendSeriesModel({
        type: 'series.pie',
        init: function (option) {
            this.$superApply('init', arguments);
            this.legendDataProvider = function () {
                return this._dataBeforeProcessed;
            };
            this.updateSelectedMap();
            this._defaultLabelLine(option);
        },
        mergeOption: function (newOption) {
            this.$superCall('mergeOption', newOption);
            this.updateSelectedMap();
        },
        getInitialData: function (option, ecModel) {
            var list = new List(['value'], this);
            list.initData(option.data);
            return list;
        },
        getDataParams: function (dataIndex) {
            var data = this._data;
            var params = this.$superCall('getDataParams', dataIndex);
            params.percent = +(data.get('value', dataIndex) / data.getSum('value') * 100).toFixed(2);
            params.$vars.push('percent');
            return params;
        },
        _defaultLabelLine: function (option) {
            modelUtil.defaultEmphasis(option.labelLine, ['show']);
            var labelLineNormalOpt = option.labelLine.normal;
            var labelLineEmphasisOpt = option.labelLine.emphasis;
            labelLineNormalOpt.show = labelLineNormalOpt.show && option.label.normal.show;
            labelLineEmphasisOpt.show = labelLineEmphasisOpt.show && option.label.emphasis.show;
        },
        defaultOption: {
            zlevel: 0,
            z: 2,
            legendHoverLink: true,
            hoverAnimation: true,
            center: [
                '50%',
                '50%'
            ],
            radius: [
                0,
                '75%'
            ],
            clockwise: true,
            startAngle: 90,
            minAngle: 0,
            selectedOffset: 10,
            avoidLabelOverlap: true,
            label: {
                normal: {
                    rotate: false,
                    show: true,
                    position: 'outer'
                },
                emphasis: {}
            },
            labelLine: {
                normal: {
                    show: true,
                    length: 20,
                    length2: 5,
                    lineStyle: {
                        width: 1,
                        type: 'solid'
                    }
                }
            },
            itemStyle: {
                normal: {
                    borderColor: 'rgba(0,0,0,0)',
                    borderWidth: 1
                },
                emphasis: {
                    borderColor: 'rgba(0,0,0,0)',
                    borderWidth: 1
                }
            },
            animationEasing: 'cubicOut'
        }
    });
    zrUtil.mixin(PieSeries, dataSelectableMixin);
    return PieSeries;
});define('echarts/chart/pie/PieView', [
    'require',
    '../../util/graphic',
    'zrender/core/util',
    '../../view/Chart'
], function (require) {
    var graphic = require('../../util/graphic');
    var zrUtil = require('zrender/core/util');
    function updateDataSelected(uid, seriesModel, hasAnimation, api) {
        var data = seriesModel.getData();
        var dataIndex = this.dataIndex;
        var name = data.getName(dataIndex);
        var selectedOffset = seriesModel.get('selectedOffset');
        api.dispatchAction({
            type: 'pieToggleSelect',
            from: uid,
            name: name,
            seriesId: seriesModel.id
        });
        data.each(function (idx) {
            toggleItemSelected(data.getItemGraphicEl(idx), data.getItemLayout(idx), seriesModel.isSelected(data.getName(idx)), selectedOffset, hasAnimation);
        });
    }
    function toggleItemSelected(el, layout, isSelected, selectedOffset, hasAnimation) {
        var midAngle = (layout.startAngle + layout.endAngle) / 2;
        var dx = Math.cos(midAngle);
        var dy = Math.sin(midAngle);
        var offset = isSelected ? selectedOffset : 0;
        var position = [
            dx * offset,
            dy * offset
        ];
        hasAnimation ? el.animate().when(200, { position: position }).start('bounceOut') : el.attr('position', position);
    }
    function PiePiece(data, idx) {
        graphic.Group.call(this);
        var sector = new graphic.Sector();
        var polyline = new graphic.Polyline();
        var text = new graphic.Text();
        this.add(sector);
        this.add(polyline);
        this.add(text);
        this.updateData(data, idx, true);
        function onEmphasis() {
            polyline.ignore = polyline.hoverIgnore;
            text.ignore = text.hoverIgnore;
        }
        function onNormal() {
            polyline.ignore = polyline.normalIgnore;
            text.ignore = text.normalIgnore;
        }
        this.on('emphasis', onEmphasis).on('normal', onNormal).on('mouseover', onEmphasis).on('mouseout', onNormal);
    }
    var piePieceProto = PiePiece.prototype;
    function getLabelStyle(data, idx, state, labelModel) {
        var textStyleModel = labelModel.getModel('textStyle');
        var position = labelModel.get('position');
        var isLabelInside = position === 'inside' || position === 'inner';
        return {
            fill: textStyleModel.getTextColor() || (isLabelInside ? '#fff' : data.getItemVisual(idx, 'color')),
            textFont: textStyleModel.getFont(),
            text: data.hostModel.getFormattedLabel(idx, state) || data.getName(idx)
        };
    }
    piePieceProto.updateData = function (data, idx, firstCreate) {
        var sector = this.childAt(0);
        var labelLine = this.childAt(1);
        var labelText = this.childAt(2);
        var seriesModel = data.hostModel;
        var itemModel = data.getItemModel(idx);
        var layout = data.getItemLayout(idx);
        var sectorShape = zrUtil.extend({}, layout);
        sectorShape.label = null;
        if (firstCreate) {
            sector.setShape(sectorShape);
            sector.shape.endAngle = layout.startAngle;
            graphic.updateProps(sector, { shape: { endAngle: layout.endAngle } }, seriesModel);
        } else {
            graphic.updateProps(sector, { shape: sectorShape }, seriesModel);
        }
        var labelLayout = layout.label;
        graphic.updateProps(labelLine, {
            shape: {
                points: labelLayout.linePoints || [
                    [
                        labelLayout.x,
                        labelLayout.y
                    ],
                    [
                        labelLayout.x,
                        labelLayout.y
                    ],
                    [
                        labelLayout.x,
                        labelLayout.y
                    ]
                ]
            }
        }, seriesModel);
        graphic.updateProps(labelText, {
            style: {
                x: labelLayout.x,
                y: labelLayout.y
            }
        }, seriesModel);
        labelText.attr({
            style: {
                textAlign: labelLayout.textAlign,
                textBaseline: labelLayout.textBaseline,
                textFont: labelLayout.font
            },
            rotation: labelLayout.rotation,
            origin: [
                labelLayout.x,
                labelLayout.y
            ],
            z2: 10
        });
        var itemStyleModel = itemModel.getModel('itemStyle');
        var visualColor = data.getItemVisual(idx, 'color');
        sector.setStyle(zrUtil.extend({ fill: visualColor }, itemStyleModel.getModel('normal').getItemStyle()));
        graphic.setHoverStyle(sector, itemStyleModel.getModel('emphasis').getItemStyle());
        var labelModel = itemModel.getModel('label.normal');
        var labelHoverModel = itemModel.getModel('label.emphasis');
        var labelLineModel = itemModel.getModel('labelLine.normal');
        var labelLineHoverModel = itemModel.getModel('labelLine.emphasis');
        labelText.setStyle(getLabelStyle(data, idx, 'normal', labelModel));
        labelText.ignore = labelText.normalIgnore = !labelModel.get('show');
        labelText.hoverIgnore = !labelHoverModel.get('show');
        labelLine.ignore = labelLine.normalIgnore = !labelLineModel.get('show');
        labelLine.hoverIgnore = !labelLineHoverModel.get('show');
        labelLine.setStyle({ stroke: visualColor });
        labelLine.setStyle(labelLineModel.getModel('lineStyle').getLineStyle());
        sector.setStyle(zrUtil.extend({ fill: visualColor }, itemStyleModel.getModel('normal').getItemStyle()));
        sector.hoverStyle = itemStyleModel.getModel('emphasis').getItemStyle();
        labelText.hoverStyle = getLabelStyle(data, idx, 'emphasis', labelHoverModel);
        labelLine.hoverStyle = labelLineHoverModel.getModel('lineStyle').getLineStyle();
        graphic.setHoverStyle(this);
        toggleItemSelected(this, data.getItemLayout(idx), itemModel.get('selected'), seriesModel.get('selectedOffset'), seriesModel.get('animation'));
        function onEmphasis() {
            sector.stopAnimation(true);
            sector.animateTo({ shape: { r: layout.r + 10 } }, 300, 'elasticOut');
        }
        function onNormal() {
            sector.stopAnimation(true);
            sector.animateTo({ shape: { r: layout.r } }, 300, 'elasticOut');
        }
        sector.off('mouseover').off('mouseout').off('emphasis').off('normal');
        if (itemModel.get('hoverAnimation')) {
            sector.on('mouseover', onEmphasis).on('mouseout', onNormal).on('emphasis', onEmphasis).on('normal', onNormal);
        }
    };
    zrUtil.inherits(PiePiece, graphic.Group);
    var Pie = require('../../view/Chart').extend({
        type: 'pie',
        init: function () {
            var sectorGroup = new graphic.Group();
            this._sectorGroup = sectorGroup;
        },
        render: function (seriesModel, ecModel, api, payload) {
            if (payload && (payload.from === this.uid || payload.type === 'pieToggleSelect' && payload.seriesName !== seriesModel.name)) {
                return;
            }
            var data = seriesModel.getData();
            var oldData = this._data;
            var group = this.group;
            var hasAnimation = ecModel.get('animation');
            var isFirstRender = !oldData;
            var onSectorClick = zrUtil.curry(updateDataSelected, this.uid, seriesModel, hasAnimation, api);
            var selectedMode = seriesModel.get('selectedMode');
            data.diff(oldData).add(function (idx) {
                var piePiece = new PiePiece(data, idx);
                if (isFirstRender) {
                    piePiece.eachChild(function (child) {
                        child.stopAnimation(true);
                    });
                }
                selectedMode && piePiece.on('click', onSectorClick);
                data.setItemGraphicEl(idx, piePiece);
                group.add(piePiece);
            }).update(function (newIdx, oldIdx) {
                var piePiece = oldData.getItemGraphicEl(oldIdx);
                piePiece.updateData(data, newIdx);
                piePiece.off('click');
                selectedMode && piePiece.on('click', onSectorClick);
                group.add(piePiece);
                data.setItemGraphicEl(newIdx, piePiece);
            }).remove(function (idx) {
                var piePiece = oldData.getItemGraphicEl(idx);
                group.remove(piePiece);
            }).execute();
            if (hasAnimation && isFirstRender && data.count() > 0) {
                var shape = data.getItemLayout(0);
                var r = Math.max(api.getWidth(), api.getHeight()) / 2;
                var removeClipPath = zrUtil.bind(group.removeClipPath, group);
                group.setClipPath(this._createClipPath(shape.cx, shape.cy, r, shape.startAngle, shape.clockwise, removeClipPath, seriesModel));
            }
            this._data = data;
        },
        _createClipPath: function (cx, cy, r, startAngle, clockwise, cb, seriesModel) {
            var clipPath = new graphic.Sector({
                shape: {
                    cx: cx,
                    cy: cy,
                    r0: 0,
                    r: r,
                    startAngle: startAngle,
                    endAngle: startAngle,
                    clockwise: clockwise
                }
            });
            graphic.initProps(clipPath, { shape: { endAngle: startAngle + (clockwise ? 1 : -1) * Math.PI * 2 } }, seriesModel, cb);
            return clipPath;
        }
    });
    return Pie;
});define('echarts/action/createDataSelectAction', [
    'require',
    '../echarts'
], function (require) {
    var echarts = require('../echarts');
    return function (seriesType, actionInfo) {
        echarts.registerAction(actionInfo, function (payload, ecModel) {
            ecModel.eachComponent({
                mainType: 'series',
                subType: seriesType,
                query: payload
            }, function (seriesModel) {
                if (seriesModel.toggleSelected) {
                    seriesModel.toggleSelected(payload.name);
                }
            });
        });
    };
});define('echarts/visual/dataColor', ['require'], function (require) {
    return function (seriesType, ecModel) {
        var colorList = ecModel.get('color');
        ecModel.eachSeriesByType(seriesType, function (seriesModel) {
            var dataAll = seriesModel.getRawData();
            if (!ecModel.isSeriesFiltered(seriesModel)) {
                var data = seriesModel.getData();
                data.each(function (idx) {
                    var itemModel = data.getItemModel(idx);
                    var rawIdx = data.getRawIndex(idx);
                    var color = itemModel.get('itemStyle.normal.color') || colorList[rawIdx % colorList.length];
                    dataAll.setItemVisual(rawIdx, 'color', color);
                    data.setItemVisual(idx, 'color', color);
                });
            }
        });
    };
});define('echarts/chart/pie/pieLayout', [
    'require',
    '../../util/number',
    './labelLayout',
    'zrender/core/util'
], function (require) {
    var numberUtil = require('../../util/number');
    var parsePercent = numberUtil.parsePercent;
    var labelLayout = require('./labelLayout');
    var zrUtil = require('zrender/core/util');
    var PI2 = Math.PI * 2;
    var RADIAN = Math.PI / 180;
    return function (seriesType, ecModel, api) {
        ecModel.eachSeriesByType(seriesType, function (seriesModel) {
            var center = seriesModel.get('center');
            var radius = seriesModel.get('radius');
            if (!zrUtil.isArray(radius)) {
                radius = [
                    0,
                    radius
                ];
            }
            if (!zrUtil.isArray(center)) {
                center = [
                    center,
                    center
                ];
            }
            var width = api.getWidth();
            var height = api.getHeight();
            var size = Math.min(width, height);
            var cx = parsePercent(center[0], width);
            var cy = parsePercent(center[1], height);
            var r0 = parsePercent(radius[0], size / 2);
            var r = parsePercent(radius[1], size / 2);
            var data = seriesModel.getData();
            var startAngle = -seriesModel.get('startAngle') * RADIAN;
            var minAngle = seriesModel.get('minAngle') * RADIAN;
            var sum = data.getSum('value');
            if (sum === 0) {
                sum = data.count();
            }
            var unitRadian = Math.PI / sum * 2;
            var clockwise = seriesModel.get('clockwise');
            var roseType = seriesModel.get('roseType');
            var extent = data.getDataExtent('value');
            extent[0] = 0;
            var restAngle = PI2;
            var valueSumLargerThanMinAngle = 0;
            var currentAngle = startAngle;
            var dir = clockwise ? 1 : -1;
            data.each('value', function (value, idx) {
                var angle;
                if (roseType !== 'area') {
                    angle = sum === 0 ? unitRadian : value * unitRadian;
                } else {
                    angle = PI2 / (data.count() || 1);
                }
                if (angle < minAngle) {
                    angle = minAngle;
                    restAngle -= minAngle;
                } else {
                    valueSumLargerThanMinAngle += value;
                }
                var endAngle = currentAngle + dir * angle;
                data.setItemLayout(idx, {
                    angle: angle,
                    startAngle: currentAngle,
                    endAngle: endAngle,
                    clockwise: clockwise,
                    cx: cx,
                    cy: cy,
                    r0: r0,
                    r: roseType ? numberUtil.linearMap(value, extent, [
                        r0,
                        r
                    ]) : r
                });
                currentAngle = endAngle;
            }, true);
            if (restAngle < PI2) {
                if (restAngle <= 0.001) {
                    var angle = PI2 / data.count();
                    data.each(function (idx) {
                        var layout = data.getItemLayout(idx);
                        layout.startAngle = startAngle + dir * idx * angle;
                        layout.endAngle = startAngle + dir * (idx + 1) * angle;
                    });
                } else {
                    unitRadian = restAngle / valueSumLargerThanMinAngle;
                    currentAngle = startAngle;
                    data.each('value', function (value, idx) {
                        var layout = data.getItemLayout(idx);
                        var angle = layout.angle === minAngle ? minAngle : value * unitRadian;
                        layout.startAngle = currentAngle;
                        layout.endAngle = currentAngle + dir * angle;
                        currentAngle += angle;
                    });
                }
            }
            labelLayout(seriesModel, r, width, height);
        });
    };
});define('echarts/processor/dataFilter', [], function () {
    return function (seriesType, ecModel) {
        var legendModel = ecModel.getComponent('legend');
        if (!legendModel) {
            return;
        }
        ecModel.eachSeriesByType(seriesType, function (series) {
            var data = series.getData();
            data.filterSelf(function (idx) {
                return legendModel.isSelected(data.getName(idx));
            }, this);
        }, this);
    };
});define('echarts/chart/helper/dataSelectableMixin', [
    'require',
    'zrender/core/util'
], function (require) {
    var zrUtil = require('zrender/core/util');
    return {
        updateSelectedMap: function () {
            var option = this.option;
            this._dataOptMap = zrUtil.reduce(option.data, function (dataOptMap, dataOpt) {
                dataOptMap[dataOpt.name] = dataOpt;
                return dataOptMap;
            }, {});
        },
        select: function (name) {
            var dataOptMap = this._dataOptMap;
            var dataOpt = dataOptMap[name];
            var selectedMode = this.get('selectedMode');
            if (selectedMode === 'single') {
                zrUtil.each(dataOptMap, function (dataOpt) {
                    dataOpt.selected = false;
                });
            }
            dataOpt && (dataOpt.selected = true);
        },
        unSelect: function (name) {
            var dataOpt = this._dataOptMap[name];
            dataOpt && (dataOpt.selected = false);
        },
        toggleSelected: function (name) {
            var dataOpt = this._dataOptMap[name];
            if (dataOpt != null) {
                this[dataOpt.selected ? 'unSelect' : 'select'](name);
                return dataOpt.selected;
            }
        },
        isSelected: function (name) {
            var dataOpt = this._dataOptMap[name];
            return dataOpt && dataOpt.selected;
        }
    };
});define('echarts/chart/pie/labelLayout', [
    'require',
    'zrender/contain/text'
], function (require) {
    'use strict';
    var textContain = require('zrender/contain/text');
    function adjustSingleSide(list, cx, cy, r, dir, viewWidth, viewHeight) {
        list.sort(function (a, b) {
            return a.y - b.y;
        });
        function shiftDown(start, end, delta, dir) {
            for (var j = start; j < end; j++) {
                list[j].y += delta;
                if (j > start && j + 1 < end && list[j + 1].y > list[j].y + list[j].height) {
                    shiftUp(j, delta / 2);
                    return;
                }
            }
            shiftUp(end - 1, delta / 2);
        }
        function shiftUp(end, delta) {
            for (var j = end; j >= 0; j--) {
                list[j].y -= delta;
                if (j > 0 && list[j].y > list[j - 1].y + list[j - 1].height) {
                    break;
                }
            }
        }
        var lastY = 0;
        var delta;
        var len = list.length;
        var upList = [];
        var downList = [];
        for (var i = 0; i < len; i++) {
            delta = list[i].y - lastY;
            if (delta < 0) {
                shiftDown(i, len, -delta, dir);
            }
            lastY = list[i].y + list[i].height;
        }
        if (viewHeight - lastY < 0) {
            shiftUp(len - 1, lastY - viewHeight);
        }
        for (var i = 0; i < len; i++) {
            if (list[i].y >= cy) {
                downList.push(list[i]);
            } else {
                upList.push(list[i]);
            }
        }
    }
    function avoidOverlap(labelLayoutList, cx, cy, r, viewWidth, viewHeight) {
        var leftList = [];
        var rightList = [];
        for (var i = 0; i < labelLayoutList.length; i++) {
            if (labelLayoutList[i].x < cx) {
                leftList.push(labelLayoutList[i]);
            } else {
                rightList.push(labelLayoutList[i]);
            }
        }
        adjustSingleSide(leftList, cx, cy, r, -1, viewWidth, viewHeight);
        adjustSingleSide(rightList, cx, cy, r, 1, viewWidth, viewHeight);
        for (var i = 0; i < labelLayoutList.length; i++) {
            var linePoints = labelLayoutList[i].linePoints;
            if (linePoints) {
                if (labelLayoutList[i].x < cx) {
                    linePoints[2][0] = labelLayoutList[i].x + 3;
                } else {
                    linePoints[2][0] = labelLayoutList[i].x - 3;
                }
                linePoints[1][1] = linePoints[2][1] = labelLayoutList[i].y;
            }
        }
    }
    return function (seriesModel, r, viewWidth, viewHeight) {
        var data = seriesModel.getData();
        var labelLayoutList = [];
        var cx;
        var cy;
        var hasLabelRotate = false;
        data.each(function (idx) {
            var layout = data.getItemLayout(idx);
            var itemModel = data.getItemModel(idx);
            var labelModel = itemModel.getModel('label.normal');
            var labelPosition = labelModel.get('position');
            var labelLineModel = itemModel.getModel('labelLine.normal');
            var labelLineLen = labelLineModel.get('length');
            var labelLineLen2 = labelLineModel.get('length2');
            var midAngle = (layout.startAngle + layout.endAngle) / 2;
            var dx = Math.cos(midAngle);
            var dy = Math.sin(midAngle);
            var textX;
            var textY;
            var linePoints;
            var textAlign;
            cx = layout.cx;
            cy = layout.cy;
            if (labelPosition === 'center') {
                textX = layout.cx;
                textY = layout.cy;
                textAlign = 'center';
            } else {
                var isLabelInside = labelPosition === 'inside' || labelPosition === 'inner';
                var x1 = (isLabelInside ? layout.r / 2 * dx : layout.r * dx) + cx;
                var y1 = (isLabelInside ? layout.r / 2 * dy : layout.r * dy) + cy;
                labelLineLen += r - layout.r;
                textX = x1 + dx * 3;
                textY = y1 + dy * 3;
                if (!isLabelInside) {
                    var x2 = x1 + dx * labelLineLen;
                    var y2 = y1 + dy * labelLineLen;
                    var x3 = x2 + (dx < 0 ? -1 : 1) * labelLineLen2;
                    var y3 = y2;
                    textX = x3 + (dx < 0 ? -5 : 5);
                    textY = y3;
                    linePoints = [
                        [
                            x1,
                            y1
                        ],
                        [
                            x2,
                            y2
                        ],
                        [
                            x3,
                            y3
                        ]
                    ];
                }
                textAlign = isLabelInside ? 'center' : dx > 0 ? 'left' : 'right';
            }
            var textBaseline = 'middle';
            var font = labelModel.getModel('textStyle').getFont();
            var labelRotate = labelModel.get('rotate') ? dx < 0 ? -midAngle + Math.PI : -midAngle : 0;
            var text = seriesModel.getFormattedLabel(idx, 'normal') || data.getName(idx);
            var textRect = textContain.getBoundingRect(text, font, textAlign, textBaseline);
            hasLabelRotate = !!labelRotate;
            layout.label = {
                x: textX,
                y: textY,
                height: textRect.height,
                length: labelLineLen,
                length2: labelLineLen2,
                linePoints: linePoints,
                textAlign: textAlign,
                textBaseline: textBaseline,
                font: font,
                rotation: labelRotate
            };
            labelLayoutList.push(layout.label);
        });
        if (!hasLabelRotate && seriesModel.get('avoidLabelOverlap')) {
            avoidOverlap(labelLayoutList, cx, cy, r, viewWidth, viewHeight);
        }
    };
});