"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.shouldIgnoreElement = exports.getTextContent = exports.censorText = exports.shouldMaskNode = exports.getNodeSelfPrivacyLevel = exports.reducePrivacyLevel = exports.getNodePrivacyLevel = exports.MAX_ATTRIBUTE_VALUE_CHAR_LENGTH = void 0;
var constants_1 = require("../../constants");
exports.MAX_ATTRIBUTE_VALUE_CHAR_LENGTH = 100000;
var TEXT_MASKING_CHAR = 'x';
/**
 * Get node privacy level by iterating over its ancestors. When the direct parent privacy level is
 * know, it is best to use something like:
 *
 * derivePrivacyLevelGivenParent(getNodeSelfPrivacyLevel(node), parentNodePrivacyLevel)
 */
function getNodePrivacyLevel(node, defaultPrivacyLevel) {
    var parentNodePrivacyLevel = node.parentNode
        ? getNodePrivacyLevel(node.parentNode, defaultPrivacyLevel)
        : defaultPrivacyLevel;
    var selfNodePrivacyLevel = getNodeSelfPrivacyLevel(node);
    return reducePrivacyLevel(selfNodePrivacyLevel, parentNodePrivacyLevel);
}
exports.getNodePrivacyLevel = getNodePrivacyLevel;
/**
 * Reduces the next privacy level based on self + parent privacy levels
 */
function reducePrivacyLevel(childPrivacyLevel, parentNodePrivacyLevel) {
    switch (parentNodePrivacyLevel) {
        // These values cannot be overridden
        case constants_1.NodePrivacyLevel.HIDDEN:
        case constants_1.NodePrivacyLevel.IGNORE:
            return parentNodePrivacyLevel;
    }
    switch (childPrivacyLevel) {
        case constants_1.NodePrivacyLevel.ALLOW:
        case constants_1.NodePrivacyLevel.MASK:
        case constants_1.NodePrivacyLevel.MASK_USER_INPUT:
        case constants_1.NodePrivacyLevel.HIDDEN:
        case constants_1.NodePrivacyLevel.IGNORE:
            return childPrivacyLevel;
        default:
            return parentNodePrivacyLevel;
    }
}
exports.reducePrivacyLevel = reducePrivacyLevel;
/**
 * Determines the node's own privacy level without checking for ancestors.
 */
function getNodeSelfPrivacyLevel(node) {
    // Only Element types can have a privacy level set
    if (!isElement(node)) {
        return;
    }
    var privAttr = node.getAttribute(constants_1.PRIVACY_ATTR_NAME);
    // Overrules for replay purpose
    if (node.tagName === 'BASE') {
        return constants_1.NodePrivacyLevel.ALLOW;
    }
    // Overrules to enforce end-user protection
    if (node.tagName === 'INPUT') {
        var inputElement = node;
        if (inputElement.type === 'password' || inputElement.type === 'email' || inputElement.type === 'tel') {
            return constants_1.NodePrivacyLevel.MASK;
        }
        if (inputElement.type === 'hidden') {
            return constants_1.NodePrivacyLevel.MASK;
        }
        var autocomplete = inputElement.getAttribute('autocomplete');
        // Handle input[autocomplete=cc-number/cc-csc/cc-exp/cc-exp-month/cc-exp-year]
        if (autocomplete && autocomplete.indexOf('cc-') === 0) {
            return constants_1.NodePrivacyLevel.MASK;
        }
    }
    // Check HTML privacy attributes and classes
    if (privAttr === constants_1.PRIVACY_ATTR_VALUE_HIDDEN || node.classList.contains(constants_1.PRIVACY_CLASS_HIDDEN)) {
        return constants_1.NodePrivacyLevel.HIDDEN;
    }
    if (privAttr === constants_1.PRIVACY_ATTR_VALUE_MASK || node.classList.contains(constants_1.PRIVACY_CLASS_MASK)) {
        return constants_1.NodePrivacyLevel.MASK;
    }
    if (privAttr === constants_1.PRIVACY_ATTR_VALUE_MASK_USER_INPUT || node.classList.contains(constants_1.PRIVACY_CLASS_MASK_USER_INPUT)) {
        return constants_1.NodePrivacyLevel.MASK_USER_INPUT;
    }
    if (privAttr === constants_1.PRIVACY_ATTR_VALUE_ALLOW || node.classList.contains(constants_1.PRIVACY_CLASS_ALLOW)) {
        return constants_1.NodePrivacyLevel.ALLOW;
    }
    if (shouldIgnoreElement(node)) {
        return constants_1.NodePrivacyLevel.IGNORE;
    }
}
exports.getNodeSelfPrivacyLevel = getNodeSelfPrivacyLevel;
/**
 * Helper aiming to unify `mask` and `mask-user-input` privacy levels:
 *
 * In the `mask` case, it is trivial: we should mask the element.
 *
 * In the `mask-user-input` case, we should mask the element only if it is a "form" element or the
 * direct parent is a form element for text nodes).
 *
 * Other `shouldMaskNode` cases are edge cases that should not matter too much (ex: should we mask a
 * node if it is ignored or hidden? it doesn't matter since it won't be serialized).
 */
function shouldMaskNode(node, privacyLevel) {
    switch (privacyLevel) {
        case constants_1.NodePrivacyLevel.MASK:
        case constants_1.NodePrivacyLevel.HIDDEN:
        case constants_1.NodePrivacyLevel.IGNORE:
            return true;
        case constants_1.NodePrivacyLevel.MASK_USER_INPUT:
            return isTextNode(node) ? isFormElement(node.parentNode) : isFormElement(node);
        default:
            return false;
    }
}
exports.shouldMaskNode = shouldMaskNode;
function isElement(node) {
    return node.nodeType === node.ELEMENT_NODE;
}
function isTextNode(node) {
    return node.nodeType === node.TEXT_NODE;
}
function isFormElement(node) {
    if (!node || node.nodeType !== node.ELEMENT_NODE) {
        return false;
    }
    var element = node;
    if (element.tagName === 'INPUT') {
        switch (element.type) {
            case 'button':
            case 'color':
            case 'reset':
            case 'submit':
                return false;
        }
    }
    return !!constants_1.FORM_PRIVATE_TAG_NAMES[element.tagName];
}
/**
 * Text censoring non-destructively maintains whitespace characters in order to preserve text shape
 * during replay.
 */
var censorText = function (text) { return text.replace(/\S/g, TEXT_MASKING_CHAR); };
exports.censorText = censorText;
function getTextContent(textNode, ignoreWhiteSpace, parentNodePrivacyLevel) {
    var _a;
    // The parent node may not be a html element which has a tagName attribute.
    // So just let it be undefined which is ok in this use case.
    var parentTagName = (_a = textNode.parentElement) === null || _a === void 0 ? void 0 : _a.tagName;
    var textContent = textNode.textContent || '';
    if (ignoreWhiteSpace && !textContent.trim()) {
        return;
    }
    var nodePrivacyLevel = parentNodePrivacyLevel;
    var isStyle = parentTagName === 'STYLE' ? true : undefined;
    var isScript = parentTagName === 'SCRIPT';
    if (isScript) {
        // For perf reasons, we don't record script (heuristic)
        textContent = constants_1.CENSORED_STRING_MARK;
    }
    else if (nodePrivacyLevel === constants_1.NodePrivacyLevel.HIDDEN) {
        // Should never occur, but just in case, we set to CENSORED_MARK.
        textContent = constants_1.CENSORED_STRING_MARK;
    }
    else if (shouldMaskNode(textNode, nodePrivacyLevel) &&
        // Style tags are `overruled` (Use `hide` to enforce privacy)
        !isStyle) {
        if (
        // Scrambling the child list breaks text nodes for DATALIST/SELECT/OPTGROUP
        parentTagName === 'DATALIST' ||
            parentTagName === 'SELECT' ||
            parentTagName === 'OPTGROUP') {
            if (!textContent.trim()) {
                return;
            }
        }
        else if (parentTagName === 'OPTION') {
            // <Option> has low entropy in charset + text length, so use `CENSORED_STRING_MARK` when masked
            textContent = constants_1.CENSORED_STRING_MARK;
        }
        else {
            textContent = (0, exports.censorText)(textContent);
        }
    }
    return textContent;
}
exports.getTextContent = getTextContent;
/**
 * TODO: Preserve CSS element order, and record the presence of the tag, just don't render
 * We don't need this logic on the recorder side.
 * For security related meta's, customer can mask themmanually given they
 * are easy to identify in the HEAD tag.
 */
function shouldIgnoreElement(element) {
    if (element.nodeName === 'SCRIPT') {
        return true;
    }
    if (element.nodeName === 'LINK') {
        var relAttribute = getLowerCaseAttribute('rel');
        return (
        // Scripts
        (relAttribute === 'preload' && getLowerCaseAttribute('as') === 'script') ||
            // Favicons
            relAttribute === 'shortcut icon' ||
            relAttribute === 'icon');
    }
    if (element.nodeName === 'META') {
        var nameAttribute = getLowerCaseAttribute('name');
        var relAttribute = getLowerCaseAttribute('rel');
        var propertyAttribute = getLowerCaseAttribute('property');
        return (
        // Favicons
        /^msapplication-tile(image|color)$/.test(nameAttribute) ||
            nameAttribute === 'application-name' ||
            relAttribute === 'icon' ||
            relAttribute === 'apple-touch-icon' ||
            relAttribute === 'shortcut icon' ||
            // Description
            nameAttribute === 'keywords' ||
            nameAttribute === 'description' ||
            // Social
            /^(og|twitter|fb):/.test(propertyAttribute) ||
            /^(og|twitter):/.test(nameAttribute) ||
            nameAttribute === 'pinterest' ||
            // Robots
            nameAttribute === 'robots' ||
            nameAttribute === 'googlebot' ||
            nameAttribute === 'bingbot' ||
            // Http headers. Ex: X-UA-Compatible, Content-Type, Content-Language, cache-control,
            // X-Translated-By
            element.hasAttribute('http-equiv') ||
            // Authorship
            nameAttribute === 'author' ||
            nameAttribute === 'generator' ||
            nameAttribute === 'framework' ||
            nameAttribute === 'publisher' ||
            nameAttribute === 'progid' ||
            /^article:/.test(propertyAttribute) ||
            /^product:/.test(propertyAttribute) ||
            // Verification
            nameAttribute === 'google-site-verification' ||
            nameAttribute === 'yandex-verification' ||
            nameAttribute === 'csrf-token' ||
            nameAttribute === 'p:domain_verify' ||
            nameAttribute === 'verify-v1' ||
            nameAttribute === 'verification' ||
            nameAttribute === 'shopify-checkout-api-token');
    }
    function getLowerCaseAttribute(name) {
        return (element.getAttribute(name) || '').toLowerCase();
    }
    return false;
}
exports.shouldIgnoreElement = shouldIgnoreElement;
//# sourceMappingURL=privacy.js.map