/*! © SpryMedia Ltd - datatables.net/license */

import $ from 'jquery';
import DataTable from 'datatables.net';

/**
 * When searching a table with accented characters, it can be frustrating to have
 * an input such as _Zurich_ not match _Zürich_ in the table (`u !== ü`). This
 * type based search plug-in replaces the built-in string formatter in
 * DataTables with a function that will replace the accented characters
 * with their unaccented counterparts for fast and easy filtering.
 *
 * Note that this plug-in uses the Javascript I18n API that was introduced in
 * ES6. For older browser's this plug-in will have no effect.
 *
 *  @summary Replace accented characters with unaccented counterparts
 *  @name Accent neutralise
 *  @author Allan Jardine
 *
 *  @example
 *    $(document).ready(function() {
 *        $('#example').dataTable();
 *    } );
 */
function removeAccents(data) {
    if (data.normalize) {
        // Use I18n API if avaiable to split characters and accents, then remove
        // the accents wholesale. Note that we use the original data as well as
        // the new to allow for searching of either form.
        return data + ' ' + data.normalize('NFD').replace(/[\u0300-\u036f]/g, '');
    }
    return data;
}
var searchType = DataTable.ext.type.search;
searchType.string = function (data) {
    return !data ? '' : typeof data === 'string' ? removeAccents(data) : data;
};
searchType.html = function (data) {
    return !data
        ? ''
        : typeof data === 'string'
            ? removeAccents(data.replace(/<.*?>/g, ''))
            : data;
};


export default DataTable;
