# Internationalization

## Table of Contents

- [Usage](#usage)

- [Supported Languages](#supported-languages)

- [Adding New Language](#adding-new-language)

## Usage

There are just a few functions that support I18n:

- [`format`](https://date-fns.org/docs/format)
- [`formatDistance`](https://date-fns.org/docs/formatDistance)
- [`formatDistanceStrict`](https://date-fns.org/docs/formatDistanceStrict)
- [`formatRelative`](https://date-fns.org/docs/formatRelative)

To use a locale, you need to require it and then pass
as an option to a function:

```js
import {formatDistance} from 'date-fns/esm'
// Require Esperanto locale
import {eo} from 'date-fns/esm/locale'

const result = formatDistance(
  new Date(2016, 7, 1),
  new Date(2015, 0, 1),
  {locale: eo} // Pass the locale as an option
)
//=> 'pli ol 1 jaro'
```

It might seem complicated to require and pass locales as options,
but unlike Moment.js which bloats your build with all the locales
by default date-fns forces developer to manually require locales when needed.
To make API simple, we encourage you to write tiny wrappers and use those
instead of original functions:

```js
// app/_lib/format.js

import {format} from 'date-fns/esm'
import {en, eo, ru} from 'date-fns/esm/locale'

const locales = {en, eo, ru}

export default function (date, formatStr) {
  return format(date, formatStr, {
    locale: locales[window.__localeId__] // or global.__localeId__
  })
}

// Later:

import format from 'app/_lib/format'

window.__localeId__ = 'en'
format(friday13, 'dddd D')
//=> 'Friday 13'

window.__localeId__ = 'eo'
format(friday13, 'dddd D')
//=> 'vendredo 13'
```

## Supported Languages

date-fns currently supports:

* [Arabic](https://github.com/date-fns/date-fns/tree/master/src/locale/ar)
  (`ar`; kudos to Abdallah Hassan [@AbdallahAHO](https://github.com/AbdallahAHO))
* [Bulgarian](https://github.com/date-fns/date-fns/tree/master/src/locale/bg)
  (`bg`; kudos to Nikolay Stoynov [@arvigeus](https://github.com/arvigeus))
* [Catalan](https://github.com/date-fns/date-fns/tree/master/src/locale/ca)
  (`ca`; kudos to Guillermo Grau [@guigrpa](https://github.com/guigrpa))
* [Chinese Simplified](https://github.com/date-fns/date-fns/tree/master/src/locale/zh-CN)
  (`zh-CN`; kudos to Changyu Geng [@KingMario](https://github.com/KingMario)
  and Song Shuoyun [@fnlctrl](https://github.com/fnlctrl))
* [Chinese Traditional](https://github.com/date-fns/date-fns/tree/master/src/locale/zh-TW)
  (`zh-TW`; kudos to tonypai [@tpai](https://github.com/tpai))
* [Croatian](https://github.com/date-fns/date-fns/tree/master/src/locale/hr)
  (`hr`; kudos to Matija Marohnić [@silvenon](https://github.com/silvenon))
* [Czech](https://github.com/date-fns/date-fns/tree/master/src/locale/cs)
  (`cs`; kudos to David Rus [@davidrus](https://github.com/davidrus))
* [Danish](https://github.com/date-fns/date-fns/tree/master/src/locale/da)
  (`da`; kudos to Anders B. Hansen [@Andersbiha](https://github.com/Andersbiha)
  and [@kgram](https://github.com/kgram))
* [Dutch](https://github.com/date-fns/date-fns/tree/master/src/locale/nl)
  (`nl`; kudos to Jorik Tangelder [@jtangelder](https://github.com/jtangelder)
  and Ruben Stolk [@rubenstolk](https://github.com/rubenstolk))
* [English](https://github.com/date-fns/date-fns/tree/master/src/locale/en-US)
  (`en-US`; it's the built-in language and doesn't require any setup)
* [Esperanto](https://github.com/date-fns/date-fns/tree/master/src/locale/eo) (`eo`)
* [Filipino](https://github.com/date-fns/date-fns/tree/master/src/locale/fil)
  (`fil`; kudos to Ian De La Cruz [@RIanDeLaCruz](https://github.com/RIanDeLaCruz))
* [Finnish](https://github.com/date-fns/date-fns/tree/master/src/locale/fi)
  (`fi`; kudos to Pyry-Samuli Lahti [@Pyppe](https://github.com/Pyppe))
* [French](https://github.com/date-fns/date-fns/tree/master/src/locale/fr)
  (`fr`; kudos to Jean Dupouy [@izeau](https://github.com/izeau)
  and François B [@fbonzon](https://github.com/fbonzon))
* [German](https://github.com/date-fns/date-fns/tree/master/src/locale/de)
  (`de`; kudos to Thomas Eilmsteiner [@DeMuu](https://github.com/DeMuu)
  and Asia [@asia-t](https://github.com/asia-t))
* [Greek](https://github.com/date-fns/date-fns/tree/master/src/locale/el)
  (`el`; kudos to Theodoros Orfanidis [@teoulas](https://github.com/teoulas))
* [Icelandic](https://github.com/date-fns/date-fns/tree/master/src/locale/is)
  (`is`; kudos to Derek Blank [@derekblank](https://github.com/derekblank))
* [Indonesian](https://github.com/date-fns/date-fns/tree/master/src/locale/id)
  (`id`; kudos to Rahmat Budiharso [@rbudiharso](https://github.com/rbudiharso)
  and Benget Nata [@bentinata](https://github.com/bentinata))
* [Italian](https://github.com/date-fns/date-fns/tree/master/src/locale/it)
  (`it`; kudos to Alberto Restifo [@albertorestifo](https://github.com/albertorestifo))
* [Japanese](https://github.com/date-fns/date-fns/tree/master/src/locale/ja)
  (`ja`; kudos to Thomas Eilmsteiner [@DeMuu](https://github.com/DeMuu)
  and Yamagishi Kazutoshi [@ykzts](https://github.com/ykzts))
* [Korean](https://github.com/date-fns/date-fns/tree/master/src/locale/ko)
  (`ko`; kudos to Hong Chulju [@angdev](https://github.com/angdev))
* [Macedonian](https://github.com/date-fns/date-fns/tree/master/src/locale/mk)
  (`mk`; kudos to Petar Vlahu [@vlahupetar](https://github.com/vlahupetar))
* [Norwegian Bokmål](https://github.com/date-fns/date-fns/tree/master/src/locale/nb)
  (`nb`; kudos to Hans-Kristian Koren [@Hanse](https://github.com/Hanse))
* [Polish](https://github.com/date-fns/date-fns/tree/master/src/locale/pl)
  (`pl`; kudos to Mateusz Derks [@ertrzyiks](https://github.com/ertrzyiks)
  and Just RAG [@justrag](https://github.com/justrag))
* [Portuguese](https://github.com/date-fns/date-fns/tree/master/src/locale/pt)
  (`pt`; kudos to Dário Freire [@dfreire](https://github.com/dfreire))
* [Romanian](https://github.com/date-fns/date-fns/tree/master/src/locale/ro)
  (`ro`; kudos to Sergiu Munteanu [@jsergiu](https://github.com/jsergiu))
* [Russian](https://github.com/date-fns/date-fns/tree/master/src/locale/ru) (`ru`)
* [Slovak](https://github.com/date-fns/date-fns/tree/master/src/locale/sk)
  (`sk`; kudos to Marek Suscak [@mareksuscak](https://github.com/mareksuscak))
* [Spanish](https://github.com/date-fns/date-fns/tree/master/src/locale/es)
  (`es`; kudos to Juan Angosto [@juanangosto](https://github.com/juanangosto),
  Guillermo Grau [@guigrpa](https://github.com/guigrpa)
  and Fernando Agüero [@fjaguero](https://github.com/fjaguero))
* [Swedish](https://github.com/date-fns/date-fns/tree/master/src/locale/sv)
  (`sv`; kudos to Johannes Ulén [@ejulen](https://github.com/ejulen))
* [Thai](https://github.com/date-fns/date-fns/tree/master/src/locale/th)
  (`th`; kudos to Athiwat Hirunworawongkun [@athivvat](https://github.com/athivvat))
* [Turkish](https://github.com/date-fns/date-fns/tree/master/src/locale/tr)
  (`tr`; kudos to Alpcan Aydın [@alpcanaydin](https://github.com/alpcanaydin))

More is coming, help is welcome!

## Adding New Language

At the moment there is no definitive guide, so if you feel brave enough,
use this quick guide:

- First of all, [create an issue](https://github.com/date-fns/date-fns/issues/new?title=XXX%20language%20support&labels[]=I18n)
  so you won't overlap with others.
- Use [English locale](https://github.com/date-fns/date-fns/tree/master/src/locale/en)
  as the basis and then incrementally adjust the tests and the code.
- If you have questions or need guidance, leave a comment in the issue.

Thank you for your support!
