import { createIterable } from '@deck.gl/core';
import { getGridOffset } from '../utils/grid-aggregation-utils';
export function pointToDensityGridDataCPU(props, aggregationParams) {
  const hashInfo = pointsToGridHashing(props, aggregationParams);
  const result = getGridLayerDataFromGridHash(hashInfo);
  return {
    gridHash: hashInfo.gridHash,
    gridOffset: hashInfo.gridOffset,
    data: result
  };
}

function pointsToGridHashing(props, aggregationParams) {
  const {
    data = [],
    cellSize
  } = props;
  const {
    attributes,
    viewport,
    projectPoints,
    numInstances
  } = aggregationParams;
  const positions = attributes.positions.value;
  const {
    size
  } = attributes.positions.getAccessor();
  const boundingBox = aggregationParams.boundingBox || getPositionBoundingBox(attributes.positions, numInstances);
  const offsets = aggregationParams.posOffset || [180, 90];
  const gridOffset = aggregationParams.gridOffset || getGridOffset(boundingBox, cellSize);

  if (gridOffset.xOffset <= 0 || gridOffset.yOffset <= 0) {
    return {
      gridHash: {},
      gridOffset
    };
  }

  const {
    width,
    height
  } = viewport;
  const numCol = Math.ceil(width / gridOffset.xOffset);
  const numRow = Math.ceil(height / gridOffset.yOffset);
  const gridHash = {};
  const {
    iterable,
    objectInfo
  } = createIterable(data);
  const position = new Array(3);

  for (const pt of iterable) {
    objectInfo.index++;
    position[0] = positions[objectInfo.index * size];
    position[1] = positions[objectInfo.index * size + 1];
    position[2] = size >= 3 ? positions[objectInfo.index * size + 2] : 0;
    const [x, y] = projectPoints ? viewport.project(position) : position;

    if (Number.isFinite(x) && Number.isFinite(y)) {
      const yIndex = Math.floor((y + offsets[1]) / gridOffset.yOffset);
      const xIndex = Math.floor((x + offsets[0]) / gridOffset.xOffset);

      if (!projectPoints || xIndex >= 0 && xIndex < numCol && yIndex >= 0 && yIndex < numRow) {
        const key = "".concat(yIndex, "-").concat(xIndex);
        gridHash[key] = gridHash[key] || {
          count: 0,
          points: [],
          lonIdx: xIndex,
          latIdx: yIndex
        };
        gridHash[key].count += 1;
        gridHash[key].points.push({
          source: pt,
          index: objectInfo.index
        });
      }
    }
  }

  return {
    gridHash,
    gridOffset,
    offsets: [offsets[0] * -1, offsets[1] * -1]
  };
}

function getGridLayerDataFromGridHash({
  gridHash,
  gridOffset,
  offsets
}) {
  const data = new Array(Object.keys(gridHash).length);
  let i = 0;

  for (const key in gridHash) {
    const idxs = key.split('-');
    const latIdx = parseInt(idxs[0], 10);
    const lonIdx = parseInt(idxs[1], 10);
    const index = i++;
    data[index] = {
      index,
      position: [offsets[0] + gridOffset.xOffset * lonIdx, offsets[1] + gridOffset.yOffset * latIdx],
      ...gridHash[key]
    };
  }

  return data;
}

function getPositionBoundingBox(positionAttribute, numInstance) {
  const positions = positionAttribute.value;
  const {
    size
  } = positionAttribute.getAccessor();
  let yMin = Infinity;
  let yMax = -Infinity;
  let xMin = Infinity;
  let xMax = -Infinity;
  let y;
  let x;

  for (let i = 0; i < numInstance; i++) {
    x = positions[i * size];
    y = positions[i * size + 1];

    if (Number.isFinite(x) && Number.isFinite(y)) {
      yMin = y < yMin ? y : yMin;
      yMax = y > yMax ? y : yMax;
      xMin = x < xMin ? x : xMin;
      xMax = x > xMax ? x : xMax;
    }
  }

  return {
    xMin,
    xMax,
    yMin,
    yMax
  };
}
//# sourceMappingURL=grid-aggregator.js.map