import _defineProperty from "@babel/runtime/helpers/esm/defineProperty";
import AggregationLayer from './aggregation-layer';
import GPUGridAggregator from './utils/gpu-grid-aggregation/gpu-grid-aggregator';
import { Buffer } from '@luma.gl/core';
import { log } from '@deck.gl/core';
import BinSorter from './utils/bin-sorter';
import { pointToDensityGridDataCPU } from './cpu-grid-layer/grid-aggregator';
export default class GridAggregationLayer extends AggregationLayer {
  constructor(...args) {
    super(...args);

    _defineProperty(this, "state", void 0);
  }

  initializeAggregationLayer({
    dimensions
  }) {
    const {
      gl
    } = this.context;
    super.initializeAggregationLayer(dimensions);
    this.setState({
      layerData: {},
      gpuGridAggregator: new GPUGridAggregator(gl, {
        id: "".concat(this.id, "-gpu-aggregator")
      }),
      cpuGridAggregator: pointToDensityGridDataCPU
    });
  }

  updateState(opts) {
    super.updateState(opts);
    this.updateAggregationState(opts);
    const {
      aggregationDataDirty,
      aggregationWeightsDirty,
      gpuAggregation
    } = this.state;

    if (this.getNumInstances() <= 0) {
      return;
    }

    let aggregationDirty = false;

    if (aggregationDataDirty || gpuAggregation && aggregationWeightsDirty) {
      this._updateAggregation(opts);

      aggregationDirty = true;
    }

    if (!gpuAggregation && (aggregationDataDirty || aggregationWeightsDirty)) {
      this._updateWeightBins();

      this._uploadAggregationResults();

      aggregationDirty = true;
    }

    this.setState({
      aggregationDirty
    });
  }

  finalizeState(context) {
    var _this$state$gpuGridAg;

    const {
      count
    } = this.state.weights;

    if (count && count.aggregationBuffer) {
      count.aggregationBuffer.delete();
    }

    (_this$state$gpuGridAg = this.state.gpuGridAggregator) === null || _this$state$gpuGridAg === void 0 ? void 0 : _this$state$gpuGridAg.delete();
    super.finalizeState(context);
  }

  updateShaders(shaders) {
    if (this.state.gpuAggregation) {
      this.state.gpuGridAggregator.updateShaders(shaders);
    }
  }

  updateAggregationState(opts) {
    log.assert(false);
  }

  allocateResources(numRow, numCol) {
    if (this.state.numRow !== numRow || this.state.numCol !== numCol) {
      const dataBytes = numCol * numRow * 4 * 4;
      const gl = this.context.gl;
      const {
        weights
      } = this.state;

      for (const name in weights) {
        const weight = weights[name];

        if (weight.aggregationBuffer) {
          weight.aggregationBuffer.delete();
        }

        weight.aggregationBuffer = new Buffer(gl, {
          byteLength: dataBytes,
          accessor: {
            size: 4,
            type: 5126,
            divisor: 1
          }
        });
      }
    }
  }

  updateResults({
    aggregationData,
    maxMinData,
    maxData,
    minData
  }) {
    const {
      count
    } = this.state.weights;

    if (count) {
      count.aggregationData = aggregationData;
      count.maxMinData = maxMinData;
      count.maxData = maxData;
      count.minData = minData;
    }
  }

  _updateAggregation(opts) {
    const {
      cpuGridAggregator,
      gpuGridAggregator,
      gridOffset,
      posOffset,
      translation = [0, 0],
      scaling = [0, 0, 0],
      boundingBox,
      projectPoints,
      gpuAggregation,
      numCol,
      numRow
    } = this.state;
    const {
      props
    } = opts;
    const {
      viewport
    } = this.context;
    const attributes = this.getAttributes();
    const vertexCount = this.getNumInstances();

    if (!gpuAggregation) {
      const result = cpuGridAggregator(props, {
        gridOffset,
        projectPoints,
        attributes,
        viewport,
        posOffset,
        boundingBox
      });
      this.setState({
        layerData: result
      });
    } else {
      const {
        weights
      } = this.state;
      gpuGridAggregator.run({
        weights,
        cellSize: [gridOffset.xOffset, gridOffset.yOffset],
        numCol,
        numRow,
        translation,
        scaling,
        vertexCount,
        projectPoints,
        attributes,
        moduleSettings: this.getModuleSettings()
      });
    }
  }

  _updateWeightBins() {
    const {
      getValue
    } = this.state;
    const sortedBins = new BinSorter(this.state.layerData.data || [], {
      getValue
    });
    this.setState({
      sortedBins
    });
  }

  _uploadAggregationResults() {
    const {
      numCol,
      numRow
    } = this.state;
    const {
      data
    } = this.state.layerData;
    const {
      aggregatedBins,
      minValue,
      maxValue,
      totalCount
    } = this.state.sortedBins;
    const ELEMENTCOUNT = 4;
    const aggregationSize = numCol * numRow * ELEMENTCOUNT;
    const aggregationData = new Float32Array(aggregationSize).fill(0);

    for (const bin of aggregatedBins) {
      const {
        lonIdx,
        latIdx
      } = data[bin.i];
      const {
        value,
        counts
      } = bin;
      const cellIndex = (lonIdx + latIdx * numCol) * ELEMENTCOUNT;
      aggregationData[cellIndex] = value;
      aggregationData[cellIndex + ELEMENTCOUNT - 1] = counts;
    }

    const maxMinData = new Float32Array([maxValue, 0, 0, minValue]);
    const maxData = new Float32Array([maxValue, 0, 0, totalCount]);
    const minData = new Float32Array([minValue, 0, 0, totalCount]);
    this.updateResults({
      aggregationData,
      maxMinData,
      maxData,
      minData
    });
  }

}

_defineProperty(GridAggregationLayer, "layerName", 'GridAggregationLayer');
//# sourceMappingURL=grid-aggregation-layer.js.map