import _defineProperty from "@babel/runtime/helpers/esm/defineProperty";
import { getBounds, boundsContain, packVertices, scaleToAspectRatio, getTextureCoordinates, getTextureParams } from './heatmap-layer-utils';
import { Buffer, Texture2D, Transform, getParameters, withParameters, FEATURES, hasFeatures } from '@luma.gl/core';
import { AttributeManager, COORDINATE_SYSTEM, log } from '@deck.gl/core';
import TriangleLayer from './triangle-layer';
import AggregationLayer from '../aggregation-layer';
import { defaultColorRange, colorRangeToFlatArray } from '../utils/color-utils';
import weightsVs from './weights-vs.glsl';
import weightsFs from './weights-fs.glsl';
import vsMax from './max-vs.glsl';
import fsMax from './max-fs.glsl';
const RESOLUTION = 2;
const TEXTURE_OPTIONS = {
  mipmaps: false,
  parameters: {
    [10240]: 9729,
    [10241]: 9729,
    [10242]: 33071,
    [10243]: 33071
  },
  dataFormat: 6408
};
const DEFAULT_COLOR_DOMAIN = [0, 0];
const AGGREGATION_MODE = {
  SUM: 0,
  MEAN: 1
};
const defaultProps = {
  getPosition: {
    type: 'accessor',
    value: x => x.position
  },
  getWeight: {
    type: 'accessor',
    value: 1
  },
  intensity: {
    type: 'number',
    min: 0,
    value: 1
  },
  radiusPixels: {
    type: 'number',
    min: 1,
    max: 100,
    value: 50
  },
  colorRange: defaultColorRange,
  threshold: {
    type: 'number',
    min: 0,
    max: 1,
    value: 0.05
  },
  colorDomain: {
    type: 'array',
    value: null,
    optional: true
  },
  aggregation: 'SUM',
  weightsTextureSize: {
    type: 'number',
    min: 128,
    max: 2048,
    value: 2048
  },
  debounceTimeout: {
    type: 'number',
    min: 0,
    max: 1000,
    value: 500
  }
};
const REQUIRED_FEATURES = [FEATURES.BLEND_EQUATION_MINMAX, FEATURES.TEXTURE_FLOAT];
const FLOAT_TARGET_FEATURES = [FEATURES.COLOR_ATTACHMENT_RGBA32F, FEATURES.FLOAT_BLEND];
const DIMENSIONS = {
  data: {
    props: ['radiusPixels']
  }
};
export default class HeatmapLayer extends AggregationLayer {
  constructor(...args) {
    super(...args);

    _defineProperty(this, "state", void 0);
  }

  initializeState() {
    const {
      gl
    } = this.context;

    if (!hasFeatures(gl, REQUIRED_FEATURES)) {
      this.setState({
        supported: false
      });
      log.error("HeatmapLayer: ".concat(this.id, " is not supported on this browser"))();
      return;
    }

    super.initializeAggregationLayer(DIMENSIONS);
    this.setState({
      supported: true,
      colorDomain: DEFAULT_COLOR_DOMAIN
    });

    this._setupTextureParams();

    this._setupAttributes();

    this._setupResources();
  }

  shouldUpdateState({
    changeFlags
  }) {
    return changeFlags.somethingChanged;
  }

  updateState(opts) {
    if (!this.state.supported) {
      return;
    }

    super.updateState(opts);

    this._updateHeatmapState(opts);
  }

  _updateHeatmapState(opts) {
    const {
      props,
      oldProps
    } = opts;

    const changeFlags = this._getChangeFlags(opts);

    if (changeFlags.dataChanged || changeFlags.viewportChanged) {
      changeFlags.boundsChanged = this._updateBounds(changeFlags.dataChanged);

      this._updateTextureRenderingBounds();
    }

    if (changeFlags.dataChanged || changeFlags.boundsChanged) {
      clearTimeout(this.state.updateTimer);
      this.setState({
        isWeightMapDirty: true
      });
    } else if (changeFlags.viewportZoomChanged) {
      this._debouncedUpdateWeightmap();
    }

    if (props.colorRange !== oldProps.colorRange) {
      this._updateColorTexture(opts);
    }

    if (this.state.isWeightMapDirty) {
      this._updateWeightmap();
    }

    this.setState({
      zoom: opts.context.viewport.zoom
    });
  }

  renderLayers() {
    if (!this.state.supported) {
      return [];
    }

    const {
      weightsTexture,
      triPositionBuffer,
      triTexCoordBuffer,
      maxWeightsTexture,
      colorTexture,
      colorDomain
    } = this.state;
    const {
      updateTriggers,
      intensity,
      threshold,
      aggregation
    } = this.props;
    const TriangleLayerClass = this.getSubLayerClass('triangle', TriangleLayer);
    return new TriangleLayerClass(this.getSubLayerProps({
      id: 'triangle-layer',
      updateTriggers
    }), {
      coordinateSystem: COORDINATE_SYSTEM.DEFAULT,
      data: {
        attributes: {
          positions: triPositionBuffer,
          texCoords: triTexCoordBuffer
        }
      },
      vertexCount: 4,
      maxTexture: maxWeightsTexture,
      colorTexture,
      aggregationMode: AGGREGATION_MODE[aggregation] || 0,
      texture: weightsTexture,
      intensity,
      threshold,
      colorDomain
    });
  }

  finalizeState(context) {
    super.finalizeState(context);
    const {
      weightsTransform,
      weightsTexture,
      maxWeightTransform,
      maxWeightsTexture,
      triPositionBuffer,
      triTexCoordBuffer,
      colorTexture,
      updateTimer
    } = this.state;
    weightsTransform === null || weightsTransform === void 0 ? void 0 : weightsTransform.delete();
    weightsTexture === null || weightsTexture === void 0 ? void 0 : weightsTexture.delete();
    maxWeightTransform === null || maxWeightTransform === void 0 ? void 0 : maxWeightTransform.delete();
    maxWeightsTexture === null || maxWeightsTexture === void 0 ? void 0 : maxWeightsTexture.delete();
    triPositionBuffer === null || triPositionBuffer === void 0 ? void 0 : triPositionBuffer.delete();
    triTexCoordBuffer === null || triTexCoordBuffer === void 0 ? void 0 : triTexCoordBuffer.delete();
    colorTexture === null || colorTexture === void 0 ? void 0 : colorTexture.delete();

    if (updateTimer) {
      clearTimeout(updateTimer);
    }
  }

  _getAttributeManager() {
    return new AttributeManager(this.context.gl, {
      id: this.props.id,
      stats: this.context.stats
    });
  }

  _getChangeFlags(opts) {
    const changeFlags = {};
    const {
      dimensions
    } = this.state;
    changeFlags.dataChanged = this.isAttributeChanged() || this.isAggregationDirty(opts, {
      compareAll: true,
      dimension: dimensions.data
    });
    changeFlags.viewportChanged = opts.changeFlags.viewportChanged;
    const {
      zoom
    } = this.state;

    if (!opts.context.viewport || opts.context.viewport.zoom !== zoom) {
      changeFlags.viewportZoomChanged = true;
    }

    return changeFlags;
  }

  _createTextures() {
    const {
      gl
    } = this.context;
    const {
      textureSize,
      format,
      type
    } = this.state;
    this.setState({
      weightsTexture: new Texture2D(gl, {
        width: textureSize,
        height: textureSize,
        format,
        type,
        ...TEXTURE_OPTIONS
      }),
      maxWeightsTexture: new Texture2D(gl, {
        format,
        type,
        ...TEXTURE_OPTIONS
      })
    });
  }

  _setupAttributes() {
    const attributeManager = this.getAttributeManager();
    attributeManager.add({
      positions: {
        size: 3,
        type: 5130,
        accessor: 'getPosition'
      },
      weights: {
        size: 1,
        accessor: 'getWeight'
      }
    });
    this.setState({
      positionAttributeName: 'positions'
    });
  }

  _setupTextureParams() {
    const {
      gl
    } = this.context;
    const {
      weightsTextureSize
    } = this.props;
    const textureSize = Math.min(weightsTextureSize, getParameters(gl, 3379));
    const floatTargetSupport = hasFeatures(gl, FLOAT_TARGET_FEATURES);
    const {
      format,
      type
    } = getTextureParams({
      gl,
      floatTargetSupport
    });
    const weightsScale = floatTargetSupport ? 1 : 1 / 255;
    this.setState({
      textureSize,
      format,
      type,
      weightsScale
    });

    if (!floatTargetSupport) {
      log.warn("HeatmapLayer: ".concat(this.id, " rendering to float texture not supported, fallingback to low precession format"))();
    }
  }

  getShaders(type) {
    return super.getShaders(type === 'max-weights-transform' ? {
      vs: vsMax,
      _fs: fsMax
    } : {
      vs: weightsVs,
      _fs: weightsFs
    });
  }

  _createWeightsTransform(shaders = {}) {
    var _weightsTransform;

    const {
      gl
    } = this.context;
    let {
      weightsTransform
    } = this.state;
    const {
      weightsTexture
    } = this.state;
    (_weightsTransform = weightsTransform) === null || _weightsTransform === void 0 ? void 0 : _weightsTransform.delete();
    weightsTransform = new Transform(gl, {
      id: "".concat(this.id, "-weights-transform"),
      elementCount: 1,
      _targetTexture: weightsTexture,
      _targetTextureVarying: 'weightsTexture',
      ...shaders
    });
    this.setState({
      weightsTransform
    });
  }

  _setupResources() {
    const {
      gl
    } = this.context;

    this._createTextures();

    const {
      textureSize,
      weightsTexture,
      maxWeightsTexture
    } = this.state;
    const weightsTransformShaders = this.getShaders('weights-transform');

    this._createWeightsTransform(weightsTransformShaders);

    const maxWeightsTransformShaders = this.getShaders('max-weights-transform');
    const maxWeightTransform = new Transform(gl, {
      id: "".concat(this.id, "-max-weights-transform"),
      _sourceTextures: {
        inTexture: weightsTexture
      },
      _targetTexture: maxWeightsTexture,
      _targetTextureVarying: 'outTexture',
      ...maxWeightsTransformShaders,
      elementCount: textureSize * textureSize
    });
    this.setState({
      weightsTexture,
      maxWeightsTexture,
      maxWeightTransform,
      zoom: null,
      triPositionBuffer: new Buffer(gl, {
        byteLength: 48,
        accessor: {
          size: 3
        }
      }),
      triTexCoordBuffer: new Buffer(gl, {
        byteLength: 48,
        accessor: {
          size: 2
        }
      })
    });
  }

  updateShaders(shaderOptions) {
    this._createWeightsTransform(shaderOptions);
  }

  _updateMaxWeightValue() {
    const {
      maxWeightTransform
    } = this.state;
    maxWeightTransform.run({
      parameters: {
        blend: true,
        depthTest: false,
        blendFunc: [1, 1],
        blendEquation: 32776
      }
    });
  }

  _updateBounds(forceUpdate = false) {
    const {
      viewport
    } = this.context;
    const viewportCorners = [viewport.unproject([0, 0]), viewport.unproject([viewport.width, 0]), viewport.unproject([viewport.width, viewport.height]), viewport.unproject([0, viewport.height])].map(p => p.map(Math.fround));
    const visibleWorldBounds = getBounds(viewportCorners);
    const newState = {
      visibleWorldBounds,
      viewportCorners
    };
    let boundsChanged = false;

    if (forceUpdate || !this.state.worldBounds || !boundsContain(this.state.worldBounds, visibleWorldBounds)) {
      const scaledCommonBounds = this._worldToCommonBounds(visibleWorldBounds);

      const worldBounds = this._commonToWorldBounds(scaledCommonBounds);

      if (this.props.coordinateSystem === COORDINATE_SYSTEM.LNGLAT) {
        worldBounds[1] = Math.max(worldBounds[1], -85.051129);
        worldBounds[3] = Math.min(worldBounds[3], 85.051129);
        worldBounds[0] = Math.max(worldBounds[0], -360);
        worldBounds[2] = Math.min(worldBounds[2], 360);
      }

      const normalizedCommonBounds = this._worldToCommonBounds(worldBounds);

      newState.worldBounds = worldBounds;
      newState.normalizedCommonBounds = normalizedCommonBounds;
      boundsChanged = true;
    }

    this.setState(newState);
    return boundsChanged;
  }

  _updateTextureRenderingBounds() {
    const {
      triPositionBuffer,
      triTexCoordBuffer,
      normalizedCommonBounds,
      viewportCorners
    } = this.state;
    const {
      viewport
    } = this.context;
    triPositionBuffer.subData(packVertices(viewportCorners, 3));
    const textureBounds = viewportCorners.map(p => getTextureCoordinates(viewport.projectPosition(p), normalizedCommonBounds));
    triTexCoordBuffer.subData(packVertices(textureBounds, 2));
  }

  _updateColorTexture(opts) {
    const {
      colorRange
    } = opts.props;
    let {
      colorTexture
    } = this.state;
    const colors = colorRangeToFlatArray(colorRange, false, Uint8Array);

    if (colorTexture) {
      colorTexture.setImageData({
        data: colors,
        width: colorRange.length
      });
    } else {
      colorTexture = new Texture2D(this.context.gl, {
        data: colors,
        width: colorRange.length,
        height: 1,
        ...TEXTURE_OPTIONS
      });
    }

    this.setState({
      colorTexture
    });
  }

  _updateWeightmap() {
    const {
      radiusPixels,
      colorDomain,
      aggregation
    } = this.props;
    const {
      weightsTransform,
      worldBounds,
      textureSize,
      weightsTexture,
      weightsScale
    } = this.state;
    this.state.isWeightMapDirty = false;

    const commonBounds = this._worldToCommonBounds(worldBounds, {
      useLayerCoordinateSystem: true
    });

    if (colorDomain && aggregation === 'SUM') {
      const {
        viewport
      } = this.context;
      const metersPerPixel = viewport.distanceScales.metersPerUnit[2] * (commonBounds[2] - commonBounds[0]) / textureSize;
      this.state.colorDomain = colorDomain.map(x => x * metersPerPixel * weightsScale);
    } else {
      this.state.colorDomain = colorDomain || DEFAULT_COLOR_DOMAIN;
    }

    const uniforms = {
      radiusPixels,
      commonBounds,
      textureWidth: textureSize,
      weightsScale
    };
    weightsTransform.update({
      elementCount: this.getNumInstances()
    });
    withParameters(this.context.gl, {
      clearColor: [0, 0, 0, 0]
    }, () => {
      weightsTransform.run({
        uniforms,
        parameters: {
          blend: true,
          depthTest: false,
          blendFunc: [1, 1],
          blendEquation: 32774
        },
        clearRenderTarget: true,
        attributes: this.getAttributes(),
        moduleSettings: this.getModuleSettings()
      });
    });

    this._updateMaxWeightValue();

    weightsTexture.setParameters({
      [10240]: 9729,
      [10241]: 9729
    });
  }

  _debouncedUpdateWeightmap(fromTimer = false) {
    let {
      updateTimer
    } = this.state;
    const {
      debounceTimeout
    } = this.props;

    if (fromTimer) {
      updateTimer = null;

      this._updateBounds(true);

      this._updateTextureRenderingBounds();

      this.setState({
        isWeightMapDirty: true
      });
    } else {
      this.setState({
        isWeightMapDirty: false
      });
      clearTimeout(updateTimer);
      updateTimer = setTimeout(this._debouncedUpdateWeightmap.bind(this, true), debounceTimeout);
    }

    this.setState({
      updateTimer
    });
  }

  _worldToCommonBounds(worldBounds, opts = {}) {
    const {
      useLayerCoordinateSystem = false
    } = opts;
    const [minLong, minLat, maxLong, maxLat] = worldBounds;
    const {
      viewport
    } = this.context;
    const {
      textureSize
    } = this.state;
    const {
      coordinateSystem
    } = this.props;
    const offsetMode = useLayerCoordinateSystem && (coordinateSystem === COORDINATE_SYSTEM.LNGLAT_OFFSETS || coordinateSystem === COORDINATE_SYSTEM.METER_OFFSETS);
    const offsetOriginCommon = offsetMode ? viewport.projectPosition(this.props.coordinateOrigin) : [0, 0];
    const size = textureSize * RESOLUTION / viewport.scale;
    let bottomLeftCommon;
    let topRightCommon;

    if (useLayerCoordinateSystem && !offsetMode) {
      bottomLeftCommon = this.projectPosition([minLong, minLat, 0]);
      topRightCommon = this.projectPosition([maxLong, maxLat, 0]);
    } else {
      bottomLeftCommon = viewport.projectPosition([minLong, minLat, 0]);
      topRightCommon = viewport.projectPosition([maxLong, maxLat, 0]);
    }

    return scaleToAspectRatio([bottomLeftCommon[0] - offsetOriginCommon[0], bottomLeftCommon[1] - offsetOriginCommon[1], topRightCommon[0] - offsetOriginCommon[0], topRightCommon[1] - offsetOriginCommon[1]], size, size);
  }

  _commonToWorldBounds(commonBounds) {
    const [xMin, yMin, xMax, yMax] = commonBounds;
    const {
      viewport
    } = this.context;
    const bottomLeftWorld = viewport.unprojectPosition([xMin, yMin]);
    const topRightWorld = viewport.unprojectPosition([xMax, yMax]);
    return bottomLeftWorld.slice(0, 2).concat(topRightWorld.slice(0, 2));
  }

}

_defineProperty(HeatmapLayer, "layerName", 'HeatmapLayer');

_defineProperty(HeatmapLayer, "defaultProps", defaultProps);
//# sourceMappingURL=heatmap-layer.js.map