import _defineProperty from "@babel/runtime/helpers/esm/defineProperty";
import { log } from '@deck.gl/core';
import { ColumnLayer } from '@deck.gl/layers';
import { defaultColorRange } from '../utils/color-utils';
import { pointToHexbin } from './hexagon-aggregator';
import CPUAggregator from '../utils/cpu-aggregator';
import AggregationLayer from '../aggregation-layer';

function nop() {}

const defaultProps = {
  colorDomain: null,
  colorRange: defaultColorRange,
  getColorValue: {
    type: 'accessor',
    value: null
  },
  getColorWeight: {
    type: 'accessor',
    value: 1
  },
  colorAggregation: 'SUM',
  lowerPercentile: {
    type: 'number',
    value: 0,
    min: 0,
    max: 100
  },
  upperPercentile: {
    type: 'number',
    value: 100,
    min: 0,
    max: 100
  },
  colorScaleType: 'quantize',
  onSetColorDomain: nop,
  elevationDomain: null,
  elevationRange: [0, 1000],
  getElevationValue: {
    type: 'accessor',
    value: null
  },
  getElevationWeight: {
    type: 'accessor',
    value: 1
  },
  elevationAggregation: 'SUM',
  elevationLowerPercentile: {
    type: 'number',
    value: 0,
    min: 0,
    max: 100
  },
  elevationUpperPercentile: {
    type: 'number',
    value: 100,
    min: 0,
    max: 100
  },
  elevationScale: {
    type: 'number',
    min: 0,
    value: 1
  },
  elevationScaleType: 'linear',
  onSetElevationDomain: nop,
  radius: {
    type: 'number',
    value: 1000,
    min: 1
  },
  coverage: {
    type: 'number',
    min: 0,
    max: 1,
    value: 1
  },
  extruded: false,
  hexagonAggregator: pointToHexbin,
  getPosition: {
    type: 'accessor',
    value: x => x.position
  },
  material: true,
  _filterData: {
    type: 'function',
    value: null,
    optional: true
  }
};
export default class HexagonLayer extends AggregationLayer {
  constructor(...args) {
    super(...args);

    _defineProperty(this, "state", void 0);
  }

  initializeState() {
    const cpuAggregator = new CPUAggregator({
      getAggregator: props => props.hexagonAggregator,
      getCellSize: props => props.radius
    });
    this.state = {
      cpuAggregator,
      aggregatorState: cpuAggregator.state,
      vertices: null
    };
    const attributeManager = this.getAttributeManager();
    attributeManager.add({
      positions: {
        size: 3,
        type: 5130,
        accessor: 'getPosition'
      }
    });
  }

  updateState(opts) {
    super.updateState(opts);

    if (opts.changeFlags.propsOrDataChanged) {
      const aggregatorState = this.state.cpuAggregator.updateState(opts, {
        viewport: this.context.viewport,
        attributes: this.getAttributes()
      });

      if (this.state.aggregatorState.layerData !== aggregatorState.layerData) {
        const {
          hexagonVertices
        } = aggregatorState.layerData || {};
        this.setState({
          vertices: hexagonVertices && this.convertLatLngToMeterOffset(hexagonVertices)
        });
      }

      this.setState({
        aggregatorState
      });
    }
  }

  convertLatLngToMeterOffset(hexagonVertices) {
    const {
      viewport
    } = this.context;

    if (Array.isArray(hexagonVertices) && hexagonVertices.length === 6) {
      const vertex0 = hexagonVertices[0];
      const vertex3 = hexagonVertices[3];
      const centroid = [(vertex0[0] + vertex3[0]) / 2, (vertex0[1] + vertex3[1]) / 2];
      const centroidFlat = viewport.projectFlat(centroid);
      const {
        metersPerUnit
      } = viewport.getDistanceScales(centroid);
      const vertices = hexagonVertices.map(vt => {
        const vtFlat = viewport.projectFlat(vt);
        return [(vtFlat[0] - centroidFlat[0]) * metersPerUnit[0], (vtFlat[1] - centroidFlat[1]) * metersPerUnit[1]];
      });
      return vertices;
    }

    log.error('HexagonLayer: hexagonVertices needs to be an array of 6 points')();
    return null;
  }

  getPickingInfo({
    info
  }) {
    return this.state.cpuAggregator.getPickingInfo({
      info
    });
  }

  _onGetSublayerColor(cell) {
    return this.state.cpuAggregator.getAccessor('fillColor')(cell);
  }

  _onGetSublayerElevation(cell) {
    return this.state.cpuAggregator.getAccessor('elevation')(cell);
  }

  _getSublayerUpdateTriggers() {
    return this.state.cpuAggregator.getUpdateTriggers(this.props);
  }

  renderLayers() {
    const {
      elevationScale,
      extruded,
      coverage,
      material,
      transitions
    } = this.props;
    const {
      aggregatorState,
      vertices
    } = this.state;
    const SubLayerClass = this.getSubLayerClass('hexagon-cell', ColumnLayer);

    const updateTriggers = this._getSublayerUpdateTriggers();

    const geometry = vertices ? {
      vertices,
      radius: 1
    } : {
      radius: aggregatorState.layerData.radiusCommon || 1,
      radiusUnits: 'common',
      angle: 90
    };
    return new SubLayerClass({ ...geometry,
      diskResolution: 6,
      elevationScale,
      extruded,
      coverage,
      material,
      getFillColor: this._onGetSublayerColor.bind(this),
      getElevation: this._onGetSublayerElevation.bind(this),
      transitions: transitions && {
        getFillColor: transitions.getColorValue || transitions.getColorWeight,
        getElevation: transitions.getElevationValue || transitions.getElevationWeight
      }
    }, this.getSubLayerProps({
      id: 'hexagon-cell',
      updateTriggers
    }), {
      data: aggregatorState.layerData.data
    });
  }

}

_defineProperty(HexagonLayer, "layerName", 'HexagonLayer');

_defineProperty(HexagonLayer, "defaultProps", defaultProps);
//# sourceMappingURL=hexagon-layer.js.map