import _defineProperty from "@babel/runtime/helpers/esm/defineProperty";
import { Model, Geometry, FEATURES, hasFeatures } from '@luma.gl/core';
import { Layer, log, picking } from '@deck.gl/core';
import { defaultColorRange, colorRangeToFlatArray } from '../utils/color-utils';
import vs from './screen-grid-layer-vertex.glsl';
import fs from './screen-grid-layer-fragment.glsl';
const DEFAULT_MINCOLOR = [0, 0, 0, 0];
const DEFAULT_MAXCOLOR = [0, 255, 0, 255];
const COLOR_PROPS = ['minColor', 'maxColor', 'colorRange', 'colorDomain'];
const defaultProps = {
  cellSizePixels: {
    value: 100,
    min: 1
  },
  cellMarginPixels: {
    value: 2,
    min: 0,
    max: 5
  },
  colorDomain: null,
  colorRange: defaultColorRange
};
export default class ScreenGridCellLayer extends Layer {
  constructor(...args) {
    super(...args);

    _defineProperty(this, "state", void 0);
  }

  static isSupported(gl) {
    return hasFeatures(gl, [FEATURES.TEXTURE_FLOAT]);
  }

  getShaders() {
    return {
      vs,
      fs,
      modules: [picking]
    };
  }

  initializeState() {
    const {
      gl
    } = this.context;
    const attributeManager = this.getAttributeManager();
    attributeManager.addInstanced({
      instancePositions: {
        size: 3,
        update: this.calculateInstancePositions
      },
      instanceCounts: {
        size: 4,
        noAlloc: true
      }
    });
    this.setState({
      model: this._getModel(gl)
    });
  }

  shouldUpdateState({
    changeFlags
  }) {
    return changeFlags.somethingChanged;
  }

  updateState(params) {
    super.updateState(params);
    const {
      oldProps,
      props,
      changeFlags
    } = params;
    const attributeManager = this.getAttributeManager();

    if (props.numInstances !== oldProps.numInstances) {
      attributeManager.invalidateAll();
    } else if (oldProps.cellSizePixels !== props.cellSizePixels) {
      attributeManager.invalidate('instancePositions');
    }

    this._updateUniforms(oldProps, props, changeFlags);
  }

  draw({
    uniforms
  }) {
    const {
      parameters,
      maxTexture
    } = this.props;
    const minColor = this.props.minColor || DEFAULT_MINCOLOR;
    const maxColor = this.props.maxColor || DEFAULT_MAXCOLOR;
    const colorDomain = this.props.colorDomain || [1, 0];
    const {
      model
    } = this.state;
    model.setUniforms(uniforms).setUniforms({
      minColor,
      maxColor,
      maxTexture,
      colorDomain
    }).draw({
      parameters: {
        depthTest: false,
        depthMask: false,
        ...parameters
      }
    });
  }

  calculateInstancePositions(attribute, {
    numInstances
  }) {
    const {
      width,
      height
    } = this.context.viewport;
    const {
      cellSizePixels
    } = this.props;
    const numCol = Math.ceil(width / cellSizePixels);
    const {
      value,
      size
    } = attribute;

    for (let i = 0; i < numInstances; i++) {
      const x = i % numCol;
      const y = Math.floor(i / numCol);
      value[i * size + 0] = x * cellSizePixels / width * 2 - 1;
      value[i * size + 1] = 1 - y * cellSizePixels / height * 2;
      value[i * size + 2] = 0;
    }
  }

  _getModel(gl) {
    return new Model(gl, { ...this.getShaders(),
      id: this.props.id,
      geometry: new Geometry({
        drawMode: 6,
        attributes: {
          positions: new Float32Array([0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 1, 0])
        }
      }),
      isInstanced: true
    });
  }

  _shouldUseMinMax() {
    const {
      minColor,
      maxColor,
      colorDomain,
      colorRange
    } = this.props;

    if (minColor || maxColor) {
      log.deprecated('ScreenGridLayer props: minColor and maxColor', 'colorRange, colorDomain')();
      return true;
    }

    if (colorDomain || colorRange) {
      return false;
    }

    return true;
  }

  _updateUniforms(oldProps, props, changeFlags) {
    const {
      model
    } = this.state;

    if (COLOR_PROPS.some(key => oldProps[key] !== props[key])) {
      model.setUniforms({
        shouldUseMinMax: this._shouldUseMinMax()
      });
    }

    if (oldProps.colorRange !== props.colorRange) {
      model.setUniforms({
        colorRange: colorRangeToFlatArray(props.colorRange)
      });
    }

    if (oldProps.cellMarginPixels !== props.cellMarginPixels || oldProps.cellSizePixels !== props.cellSizePixels || changeFlags.viewportChanged) {
      const {
        width,
        height
      } = this.context.viewport;
      const {
        cellSizePixels,
        cellMarginPixels
      } = this.props;
      const margin = cellSizePixels > cellMarginPixels ? cellMarginPixels : 0;
      const cellScale = new Float32Array([(cellSizePixels - margin) / width * 2, -(cellSizePixels - margin) / height * 2, 1]);
      model.setUniforms({
        cellScale
      });
    }
  }

}

_defineProperty(ScreenGridCellLayer, "layerName", 'ScreenGridCellLayer');

_defineProperty(ScreenGridCellLayer, "defaultProps", defaultProps);
//# sourceMappingURL=screen-grid-cell-layer.js.map