import { Model, Transform, FEATURES, hasFeatures, isWebGL2, readPixelsToBuffer, withParameters } from '@luma.gl/core';
import { fp64arithmetic } from '@luma.gl/shadertools';
import { log, project32, _mergeShaders as mergeShaders } from '@deck.gl/core';
import { DEFAULT_RUN_PARAMS, MAX_32_BIT_FLOAT, MIN_BLEND_EQUATION, MAX_BLEND_EQUATION, MAX_MIN_BLEND_EQUATION, EQUATION_MAP, DEFAULT_WEIGHT_PARAMS, PIXEL_SIZE } from './gpu-grid-aggregator-constants';
import { AGGREGATION_OPERATION } from '../aggregation-operation-utils';
import AGGREGATE_TO_GRID_VS from './aggregate-to-grid-vs.glsl';
import AGGREGATE_TO_GRID_FS from './aggregate-to-grid-fs.glsl';
import AGGREGATE_ALL_VS from './aggregate-all-vs.glsl';
import AGGREGATE_ALL_FS from './aggregate-all-fs.glsl';
import TRANSFORM_MEAN_VS from './transform-mean-vs.glsl';
import { getFloatTexture, getFramebuffer } from './../resource-utils.js';
const BUFFER_NAMES = ['aggregationBuffer', 'maxMinBuffer', 'minBuffer', 'maxBuffer'];
const ARRAY_BUFFER_MAP = {
  maxData: 'maxBuffer',
  minData: 'minBuffer',
  maxMinData: 'maxMinBuffer'
};
const REQUIRED_FEATURES = [FEATURES.WEBGL2, FEATURES.COLOR_ATTACHMENT_RGBA32F, FEATURES.BLEND_EQUATION_MINMAX, FEATURES.FLOAT_BLEND, FEATURES.TEXTURE_FLOAT];
export default class GPUGridAggregator {
  static getAggregationData({
    aggregationData,
    maxData,
    minData,
    maxMinData,
    pixelIndex
  }) {
    const index = pixelIndex * PIXEL_SIZE;
    const results = {};

    if (aggregationData) {
      results.cellCount = aggregationData[index + 3];
      results.cellWeight = aggregationData[index];
    }

    if (maxMinData) {
      results.maxCellWieght = maxMinData[0];
      results.minCellWeight = maxMinData[3];
    } else {
      if (maxData) {
        results.maxCellWieght = maxData[0];
        results.totalCount = maxData[3];
      }

      if (minData) {
        results.minCellWeight = minData[0];
        results.totalCount = maxData[3];
      }
    }

    return results;
  }

  static getCellData({
    countsData,
    size = 1
  }) {
    const numCells = countsData.length / 4;
    const cellWeights = new Float32Array(numCells * size);
    const cellCounts = new Uint32Array(numCells);

    for (let i = 0; i < numCells; i++) {
      for (let sizeIndex = 0; sizeIndex < size; sizeIndex++) {
        cellWeights[i * size + sizeIndex] = countsData[i * 4 + sizeIndex];
      }

      cellCounts[i] = countsData[i * 4 + 3];
    }

    return {
      cellCounts,
      cellWeights
    };
  }

  static isSupported(gl) {
    return hasFeatures(gl, REQUIRED_FEATURES);
  }

  constructor(gl, opts = {}) {
    this.id = opts.id || 'gpu-grid-aggregator';
    this.gl = gl;
    this.state = {
      weightAttributes: {},
      textures: {},
      meanTextures: {},
      buffers: {},
      framebuffers: {},
      maxMinFramebuffers: {},
      minFramebuffers: {},
      maxFramebuffers: {},
      equations: {},
      resources: {},
      results: {}
    };
    this._hasGPUSupport = isWebGL2(gl) && hasFeatures(this.gl, FEATURES.BLEND_EQUATION_MINMAX, FEATURES.COLOR_ATTACHMENT_RGBA32F, FEATURES.TEXTURE_FLOAT);

    if (this._hasGPUSupport) {
      this._setupModels();
    }
  }

  delete() {
    const {
      gridAggregationModel,
      allAggregationModel,
      meanTransform
    } = this;
    const {
      textures,
      framebuffers,
      maxMinFramebuffers,
      minFramebuffers,
      maxFramebuffers,
      meanTextures,
      resources
    } = this.state;
    gridAggregationModel === null || gridAggregationModel === void 0 ? void 0 : gridAggregationModel.delete();
    allAggregationModel === null || allAggregationModel === void 0 ? void 0 : allAggregationModel.delete();
    meanTransform === null || meanTransform === void 0 ? void 0 : meanTransform.delete();
    deleteResources([framebuffers, textures, maxMinFramebuffers, minFramebuffers, maxFramebuffers, meanTextures, resources]);
  }

  run(opts = {}) {
    this.setState({
      results: {}
    });

    const aggregationParams = this._normalizeAggregationParams(opts);

    if (!this._hasGPUSupport) {
      log.log(1, 'GPUGridAggregator: not supported')();
    }

    return this._runAggregation(aggregationParams);
  }

  getData(weightId) {
    const data = {};
    const results = this.state.results;

    if (!results[weightId].aggregationData) {
      results[weightId].aggregationData = results[weightId].aggregationBuffer.getData();
    }

    data.aggregationData = results[weightId].aggregationData;

    for (const arrayName in ARRAY_BUFFER_MAP) {
      const bufferName = ARRAY_BUFFER_MAP[arrayName];

      if (results[weightId][arrayName] || results[weightId][bufferName]) {
        results[weightId][arrayName] = results[weightId][arrayName] || results[weightId][bufferName].getData();
        data[arrayName] = results[weightId][arrayName];
      }
    }

    return data;
  }

  updateShaders(shaderOptions = {}) {
    this.setState({
      shaderOptions,
      modelDirty: true
    });
  }

  _normalizeAggregationParams(opts) {
    const aggregationParams = { ...DEFAULT_RUN_PARAMS,
      ...opts
    };
    const {
      weights
    } = aggregationParams;

    if (weights) {
      aggregationParams.weights = normalizeWeightParams(weights);
    }

    return aggregationParams;
  }

  setState(updateObject) {
    Object.assign(this.state, updateObject);
  }

  _getAggregateData(opts) {
    const results = {};
    const {
      textures,
      framebuffers,
      maxMinFramebuffers,
      minFramebuffers,
      maxFramebuffers,
      resources
    } = this.state;
    const {
      weights
    } = opts;

    for (const id in weights) {
      results[id] = {};
      const {
        needMin,
        needMax,
        combineMaxMin
      } = weights[id];
      results[id].aggregationTexture = textures[id];
      results[id].aggregationBuffer = readPixelsToBuffer(framebuffers[id], {
        target: weights[id].aggregationBuffer,
        sourceType: 5126
      });

      if (needMin && needMax && combineMaxMin) {
        results[id].maxMinBuffer = readPixelsToBuffer(maxMinFramebuffers[id], {
          target: weights[id].maxMinBuffer,
          sourceType: 5126
        });
        results[id].maxMinTexture = resources["".concat(id, "-maxMinTexture")];
      } else {
        if (needMin) {
          results[id].minBuffer = readPixelsToBuffer(minFramebuffers[id], {
            target: weights[id].minBuffer,
            sourceType: 5126
          });
          results[id].minTexture = resources["".concat(id, "-minTexture")];
        }

        if (needMax) {
          results[id].maxBuffer = readPixelsToBuffer(maxFramebuffers[id], {
            target: weights[id].maxBuffer,
            sourceType: 5126
          });
          results[id].maxTexture = resources["".concat(id, "-maxTexture")];
        }
      }
    }

    this._trackGPUResultBuffers(results, weights);

    return results;
  }

  _renderAggregateData(opts) {
    const {
      cellSize,
      projectPoints,
      attributes,
      moduleSettings,
      numCol,
      numRow,
      weights,
      translation,
      scaling
    } = opts;
    const {
      maxMinFramebuffers,
      minFramebuffers,
      maxFramebuffers
    } = this.state;
    const gridSize = [numCol, numRow];
    const parameters = {
      blend: true,
      depthTest: false,
      blendFunc: [1, 1]
    };
    const uniforms = {
      cellSize,
      gridSize,
      projectPoints,
      translation,
      scaling
    };

    for (const id in weights) {
      const {
        needMin,
        needMax
      } = weights[id];
      const combineMaxMin = needMin && needMax && weights[id].combineMaxMin;

      this._renderToWeightsTexture({
        id,
        parameters,
        moduleSettings,
        uniforms,
        gridSize,
        attributes,
        weights
      });

      if (combineMaxMin) {
        this._renderToMaxMinTexture({
          id,
          parameters: { ...parameters,
            blendEquation: MAX_MIN_BLEND_EQUATION
          },
          gridSize,
          minOrMaxFb: maxMinFramebuffers[id],
          clearParams: {
            clearColor: [0, 0, 0, MAX_32_BIT_FLOAT]
          },
          combineMaxMin
        });
      } else {
        if (needMin) {
          this._renderToMaxMinTexture({
            id,
            parameters: { ...parameters,
              blendEquation: MIN_BLEND_EQUATION
            },
            gridSize,
            minOrMaxFb: minFramebuffers[id],
            clearParams: {
              clearColor: [MAX_32_BIT_FLOAT, MAX_32_BIT_FLOAT, MAX_32_BIT_FLOAT, 0]
            },
            combineMaxMin
          });
        }

        if (needMax) {
          this._renderToMaxMinTexture({
            id,
            parameters: { ...parameters,
              blendEquation: MAX_BLEND_EQUATION
            },
            gridSize,
            minOrMaxFb: maxFramebuffers[id],
            clearParams: {
              clearColor: [0, 0, 0, 0]
            },
            combineMaxMin
          });
        }
      }
    }
  }

  _renderToMaxMinTexture(opts) {
    const {
      id,
      parameters,
      gridSize,
      minOrMaxFb,
      combineMaxMin,
      clearParams = {}
    } = opts;
    const {
      framebuffers
    } = this.state;
    const {
      gl,
      allAggregationModel
    } = this;
    withParameters(gl, { ...clearParams,
      framebuffer: minOrMaxFb,
      viewport: [0, 0, gridSize[0], gridSize[1]]
    }, () => {
      gl.clear(16384);
      allAggregationModel.draw({
        parameters,
        uniforms: {
          uSampler: framebuffers[id].texture,
          gridSize,
          combineMaxMin
        }
      });
    });
  }

  _renderToWeightsTexture(opts) {
    const {
      id,
      parameters,
      moduleSettings,
      uniforms,
      gridSize,
      weights
    } = opts;
    const {
      framebuffers,
      equations,
      weightAttributes
    } = this.state;
    const {
      gl,
      gridAggregationModel
    } = this;
    const {
      operation
    } = weights[id];
    const clearColor = operation === AGGREGATION_OPERATION.MIN ? [MAX_32_BIT_FLOAT, MAX_32_BIT_FLOAT, MAX_32_BIT_FLOAT, 0] : [0, 0, 0, 0];
    withParameters(gl, {
      framebuffer: framebuffers[id],
      viewport: [0, 0, gridSize[0], gridSize[1]],
      clearColor
    }, () => {
      gl.clear(16384);
      const attributes = {
        weights: weightAttributes[id]
      };
      gridAggregationModel.draw({
        parameters: { ...parameters,
          blendEquation: equations[id]
        },
        moduleSettings,
        uniforms,
        attributes
      });
    });

    if (operation === AGGREGATION_OPERATION.MEAN) {
      const {
        meanTextures,
        textures
      } = this.state;
      const transformOptions = {
        _sourceTextures: {
          aggregationValues: meanTextures[id]
        },
        _targetTexture: textures[id],
        elementCount: textures[id].width * textures[id].height
      };

      if (this.meanTransform) {
        this.meanTransform.update(transformOptions);
      } else {
        this.meanTransform = getMeanTransform(gl, transformOptions);
      }

      this.meanTransform.run({
        parameters: {
          blend: false,
          depthTest: false
        }
      });
      framebuffers[id].attach({
        [36064]: textures[id]
      });
    }
  }

  _runAggregation(opts) {
    this._updateModels(opts);

    this._setupFramebuffers(opts);

    this._renderAggregateData(opts);

    const results = this._getAggregateData(opts);

    this.setState({
      results
    });
    return results;
  }

  _setupFramebuffers(opts) {
    const {
      textures,
      framebuffers,
      maxMinFramebuffers,
      minFramebuffers,
      maxFramebuffers,
      meanTextures,
      equations
    } = this.state;
    const {
      weights
    } = opts;
    const {
      numCol,
      numRow
    } = opts;
    const framebufferSize = {
      width: numCol,
      height: numRow
    };

    for (const id in weights) {
      const {
        needMin,
        needMax,
        combineMaxMin,
        operation
      } = weights[id];
      textures[id] = weights[id].aggregationTexture || textures[id] || getFloatTexture(this.gl, {
        id: "".concat(id, "-texture"),
        width: numCol,
        height: numRow
      });
      textures[id].resize(framebufferSize);
      let texture = textures[id];

      if (operation === AGGREGATION_OPERATION.MEAN) {
        meanTextures[id] = meanTextures[id] || getFloatTexture(this.gl, {
          id: "".concat(id, "-mean-texture"),
          width: numCol,
          height: numRow
        });
        meanTextures[id].resize(framebufferSize);
        texture = meanTextures[id];
      }

      if (framebuffers[id]) {
        framebuffers[id].attach({
          [36064]: texture
        });
      } else {
        framebuffers[id] = getFramebuffer(this.gl, {
          id: "".concat(id, "-fb"),
          width: numCol,
          height: numRow,
          texture
        });
      }

      framebuffers[id].resize(framebufferSize);
      equations[id] = EQUATION_MAP[operation] || EQUATION_MAP.SUM;

      if (needMin || needMax) {
        if (needMin && needMax && combineMaxMin) {
          if (!maxMinFramebuffers[id]) {
            texture = weights[id].maxMinTexture || this._getMinMaxTexture("".concat(id, "-maxMinTexture"));
            maxMinFramebuffers[id] = getFramebuffer(this.gl, {
              id: "".concat(id, "-maxMinFb"),
              texture
            });
          }
        } else {
          if (needMin) {
            if (!minFramebuffers[id]) {
              texture = weights[id].minTexture || this._getMinMaxTexture("".concat(id, "-minTexture"));
              minFramebuffers[id] = getFramebuffer(this.gl, {
                id: "".concat(id, "-minFb"),
                texture
              });
            }
          }

          if (needMax) {
            if (!maxFramebuffers[id]) {
              texture = weights[id].maxTexture || this._getMinMaxTexture("".concat(id, "-maxTexture"));
              maxFramebuffers[id] = getFramebuffer(this.gl, {
                id: "".concat(id, "-maxFb"),
                texture
              });
            }
          }
        }
      }
    }
  }

  _getMinMaxTexture(name) {
    const {
      resources
    } = this.state;

    if (!resources[name]) {
      resources[name] = getFloatTexture(this.gl, {
        id: "resourceName"
      });
    }

    return resources[name];
  }

  _setupModels({
    numCol = 0,
    numRow = 0
  } = {}) {
    var _this$gridAggregation;

    const {
      gl
    } = this;
    const {
      shaderOptions
    } = this.state;
    (_this$gridAggregation = this.gridAggregationModel) === null || _this$gridAggregation === void 0 ? void 0 : _this$gridAggregation.delete();
    this.gridAggregationModel = getAggregationModel(gl, shaderOptions);

    if (!this.allAggregationModel) {
      const instanceCount = numCol * numRow;
      this.allAggregationModel = getAllAggregationModel(gl, instanceCount);
    }
  }

  _setupWeightAttributes(opts) {
    const {
      weightAttributes
    } = this.state;
    const {
      weights
    } = opts;

    for (const id in weights) {
      weightAttributes[id] = opts.attributes[id];
    }
  }

  _trackGPUResultBuffers(results, weights) {
    const {
      resources
    } = this.state;

    for (const id in results) {
      if (results[id]) {
        for (const bufferName of BUFFER_NAMES) {
          if (results[id][bufferName] && weights[id][bufferName] !== results[id][bufferName]) {
            const name = "gpu-result-".concat(id, "-").concat(bufferName);

            if (resources[name]) {
              resources[name].delete();
            }

            resources[name] = results[id][bufferName];
          }
        }
      }
    }
  }

  _updateModels(opts) {
    const {
      vertexCount,
      attributes,
      numCol,
      numRow
    } = opts;
    const {
      modelDirty
    } = this.state;

    if (modelDirty) {
      this._setupModels(opts);

      this.setState({
        modelDirty: false
      });
    }

    this._setupWeightAttributes(opts);

    this.gridAggregationModel.setVertexCount(vertexCount);
    this.gridAggregationModel.setAttributes(attributes);
    this.allAggregationModel.setInstanceCount(numCol * numRow);
  }

}

function normalizeWeightParams(weights) {
  const result = {};

  for (const id in weights) {
    result[id] = { ...DEFAULT_WEIGHT_PARAMS,
      ...weights[id]
    };
  }

  return result;
}

function deleteResources(resources) {
  resources = Array.isArray(resources) ? resources : [resources];
  resources.forEach(obj => {
    for (const name in obj) {
      obj[name].delete();
    }
  });
}

function getAggregationModel(gl, shaderOptions) {
  const shaders = mergeShaders({
    vs: AGGREGATE_TO_GRID_VS,
    fs: AGGREGATE_TO_GRID_FS,
    modules: [fp64arithmetic, project32]
  }, shaderOptions);
  return new Model(gl, {
    id: 'Gird-Aggregation-Model',
    vertexCount: 1,
    drawMode: 0,
    ...shaders
  });
}

function getAllAggregationModel(gl, instanceCount) {
  return new Model(gl, {
    id: 'All-Aggregation-Model',
    vs: AGGREGATE_ALL_VS,
    fs: AGGREGATE_ALL_FS,
    modules: [fp64arithmetic],
    vertexCount: 1,
    drawMode: 0,
    isInstanced: true,
    instanceCount,
    attributes: {
      position: [0, 0]
    }
  });
}

function getMeanTransform(gl, opts) {
  return new Transform(gl, {
    vs: TRANSFORM_MEAN_VS,
    _targetTextureVarying: 'meanValues',
    ...opts
  });
}
//# sourceMappingURL=gpu-grid-aggregator.js.map