import { defaultClassicCredentials, defaultCloudNativeCredentials, getDefaultCredentials } from '../config';
import { API_VERSIONS, DEFAULT_MAPS_URL_FORMAT, DEFAULT_REGION_COMPONENT_IN_URL, DEFAULT_USER_COMPONENT_IN_URL, encodeParameter, MAP_TYPES } from './maps-api-common';
export const CONNECTIONS = {
  BIGQUERY: 'bigquery',
  CARTO: 'carto'
};
const BUFFER_SIZE = 16;
const TILE_EXTENT = 4096;
export async function getDataV2({
  type,
  source,
  credentials
}) {
  const defaultCredentials = getDefaultCredentials();
  const apiVersion = (credentials === null || credentials === void 0 ? void 0 : credentials.apiVersion) || defaultCredentials.apiVersion;
  const localCreds = { ...(apiVersion === API_VERSIONS.V3 ? defaultCloudNativeCredentials : defaultClassicCredentials),
    mapsUrl: DEFAULT_MAPS_URL_FORMAT[apiVersion],
    ...(defaultCredentials.apiVersion === apiVersion && defaultCredentials),
    ...credentials
  };
  let url;
  const connection = type === MAP_TYPES.TILESET ? CONNECTIONS.BIGQUERY : CONNECTIONS.CARTO;

  switch (apiVersion) {
    case API_VERSIONS.V1:
      const mapConfig = createMapConfig(source);
      url = buildURLMapsAPIv1({
        mapConfig,
        credentials: localCreds
      });
      const layergroup = await request({
        url,
        credentials: localCreds
      });
      return layergroup.metadata.tilejson.vector;

    case API_VERSIONS.V2:
      url = buildURLMapsAPIv2({
        connection,
        type,
        source,
        credentials: localCreds
      });
      return await request({
        url,
        credentials: localCreds
      });

    default:
      throw new Error("Invalid maps API version. It should be ".concat(API_VERSIONS.V1, " or ").concat(API_VERSIONS.V2));
  }
}

async function request({
  url,
  credentials
}) {
  let response;

  try {
    response = await fetch(url, {
      headers: {
        Accept: 'application/json'
      }
    });
  } catch (error) {
    throw new Error("Failed to connect to Maps API: ".concat(error));
  }

  const json = await response.json();

  if (!response.ok) {
    dealWithError({
      response,
      json,
      credentials
    });
  }

  return json;
}

function dealWithError({
  response,
  json,
  credentials
}) {
  switch (response.status) {
    case 401:
      throw new Error("Unauthorized access to Maps API: invalid combination of user ('".concat(credentials.username, "') and apiKey ('").concat(credentials.apiKey, "')"));

    case 403:
      throw new Error("Unauthorized access to dataset: the provided apiKey('".concat(credentials.apiKey, "') doesn't provide access to the requested data"));

    default:
      const e = credentials.apiVersion === API_VERSIONS.V1 ? JSON.stringify(json.errors) : json.error;
      throw new Error(e);
  }
}

function initURLParameters(credentials) {
  const encodedApiKey = encodeParameter('api_key', credentials.apiKey);
  const encodedClient = encodeParameter('client', 'deck-gl-carto');
  return [encodedApiKey, encodedClient];
}

function buildURLMapsAPIv1({
  mapConfig,
  credentials
}) {
  const parameters = initURLParameters(credentials);
  const cfg = JSON.stringify(mapConfig);
  return "".concat(mapsUrl(credentials), "?").concat(parameters.join('&'), "&").concat(encodeParameter('config', cfg));
}

function buildURLMapsAPIv2({
  connection,
  type,
  source,
  credentials
}) {
  const parameters = initURLParameters(credentials);
  const mapsApiType = type === MAP_TYPES.QUERY ? 'sql' : type;
  let url = "".concat(mapsUrl(credentials), "/").concat(connection, "/").concat(mapsApiType, "?");
  url += "".concat(encodeParameter('source', source), "&format=tilejson&").concat(parameters.join('&'));
  return url;
}

function mapsUrl(credentials) {
  return credentials.mapsUrl.replace(DEFAULT_USER_COMPONENT_IN_URL, credentials.username).replace(DEFAULT_REGION_COMPONENT_IN_URL, credentials.region);
}

function createMapConfig(sql) {
  return {
    version: '1.3.1',
    buffersize: {
      mvt: BUFFER_SIZE
    },
    layers: [{
      type: 'mapnik',
      options: {
        sql,
        vector_extent: TILE_EXTENT
      }
    }]
  };
}
//# sourceMappingURL=maps-client.js.map