import { getDefaultCredentials, buildMapsUrlFromBase, buildStatsUrlFromBase } from '../config';
import { API_VERSIONS, COLUMNS_SUPPORT, encodeParameter, FORMATS, GEO_COLUMN_SUPPORT, MAP_TYPES, REQUEST_TYPES, TILE_FORMATS } from './maps-api-common';
import { CartoAPIError } from './carto-api-error';
import { parseMap } from './parseMap';
import { log } from '@deck.gl/core';
import { assert } from '../utils';
const MAX_GET_LENGTH = 8192;
const DEFAULT_CLIENT = 'deck-gl-carto';
const V3_MINOR_VERSION = '3.2';

async function request({
  method,
  url,
  headers: customHeaders,
  accessToken,
  body,
  errorContext
}) {
  const headers = { ...customHeaders,
    Accept: 'application/json'
  };

  if (accessToken) {
    headers.Authorization = "Bearer ".concat(accessToken);
  }

  if (method === 'POST') {
    headers['Content-Type'] = 'application/json';
  }

  try {
    return await fetch(url, {
      method,
      headers,
      body
    });
  } catch (error) {
    throw new CartoAPIError(error, errorContext);
  }
}

async function requestJson({
  method,
  url,
  headers,
  accessToken,
  body,
  errorContext
}) {
  const response = await request({
    method,
    url,
    headers,
    accessToken,
    body,
    errorContext
  });
  let json;

  try {
    json = await response.json();
  } catch (error) {
    json = {
      error: ''
    };
  }

  if (!response.ok) {
    throw new CartoAPIError(json.error, errorContext, response);
  }

  return json;
}

async function requestData({
  method,
  url,
  accessToken,
  format,
  body,
  errorContext
}) {
  if (format === FORMATS.NDJSON) {
    return request({
      method,
      url,
      accessToken,
      body,
      errorContext
    });
  }

  const data = await requestJson({
    method,
    url,
    accessToken,
    body,
    errorContext
  });
  return data.rows ? data.rows : data;
}

function getParameters({
  type,
  source,
  geoColumn,
  columns,
  clientId,
  aggregationExp,
  aggregationResLevel,
  queryParameters,
  filters
}) {
  const parameters = {
    client: clientId || DEFAULT_CLIENT,
    v: V3_MINOR_VERSION
  };
  const sourceName = type === MAP_TYPES.QUERY ? 'q' : 'name';
  parameters[sourceName] = source;

  if (queryParameters) {
    parameters.queryParameters = queryParameters;
  }

  if (geoColumn) {
    parameters.geo_column = geoColumn;
  }

  if (columns) {
    parameters.columns = columns.join(',');
  }

  if (aggregationExp) {
    parameters.aggregationExp = aggregationExp;
  } else if (isSpatialIndexGeoColumn(geoColumn)) {
    parameters.aggregationExp = '1 AS value';
  }

  if (aggregationResLevel) {
    parameters.aggregationResLevel = aggregationResLevel;
  }

  if (filters) {
    parameters.filters = filters;
  }

  return parameters;
}

function isSpatialIndexGeoColumn(geoColumn) {
  const spatialIndex = geoColumn === null || geoColumn === void 0 ? void 0 : geoColumn.split(':')[0];
  return spatialIndex === 'h3' || spatialIndex === 'quadbin';
}

export async function mapInstantiation({
  type,
  source,
  connection,
  credentials,
  geoColumn,
  columns,
  clientId,
  headers,
  aggregationExp,
  aggregationResLevel,
  queryParameters,
  filters
}) {
  const baseUrl = "".concat(credentials.mapsUrl, "/").concat(connection, "/").concat(type);
  const parameters = getParameters({
    type,
    source,
    geoColumn,
    columns,
    clientId,
    aggregationResLevel,
    aggregationExp,
    queryParameters,
    filters
  });
  const encodedParameters = Object.entries(parameters).map(([key, value]) => {
    if (typeof value !== 'string') {
      value = JSON.stringify(value);
    }

    return encodeParameter(key, value);
  });
  const url = "".concat(baseUrl, "?").concat(encodedParameters.join('&'));
  const {
    accessToken
  } = credentials;
  const errorContext = {
    requestType: REQUEST_TYPES.INSTANTIATION,
    connection,
    type,
    source
  };

  if (url.length > MAX_GET_LENGTH && type === MAP_TYPES.QUERY) {
    return await requestJson({
      method: 'POST',
      url: baseUrl,
      headers,
      accessToken,
      body: JSON.stringify(parameters),
      errorContext
    });
  }

  return await requestJson({
    url,
    headers,
    accessToken,
    errorContext
  });
}

function getUrlFromMetadata(metadata, format) {
  const m = metadata[format];

  if (m && !m.error && m.url) {
    return m.url[0];
  }

  return null;
}

function checkFetchLayerDataParameters({
  type,
  source,
  connection,
  credentials,
  geoColumn,
  columns,
  aggregationExp,
  aggregationResLevel,
  filters
}) {
  assert(connection, 'Must define connection');
  assert(type, 'Must define a type');
  assert(source, 'Must define a source');
  assert(credentials.apiVersion === API_VERSIONS.V3, 'Method only available for v3');
  assert(credentials.apiBaseUrl, 'Must define apiBaseUrl');
  assert(credentials.accessToken, 'Must define an accessToken');

  if (columns) {
    assert(COLUMNS_SUPPORT.includes(type), "The columns parameter is not supported by type ".concat(type));
  }

  if (geoColumn) {
    assert(GEO_COLUMN_SUPPORT.includes(type), "The geoColumn parameter is not supported by type ".concat(type));
  } else {
    assert(!aggregationExp, 'Have aggregationExp, but geoColumn parameter is missing');
    assert(!aggregationResLevel, 'Have aggregationResLevel, but geoColumn parameter is missing');
  }

  if (filters) {
    assert(type === MAP_TYPES.TABLE || type === MAP_TYPES.QUERY, 'The filters parameter is only supported by type table and query');
  }
}

export async function fetchLayerData({
  type,
  source,
  connection,
  credentials,
  geoColumn,
  columns,
  format,
  formatTiles,
  clientId,
  headers,
  aggregationExp,
  aggregationResLevel,
  queryParameters,
  filters
}) {
  const {
    url,
    accessToken,
    mapFormat,
    metadata
  } = await _fetchDataUrl({
    type,
    source,
    connection,
    credentials,
    geoColumn,
    columns,
    format,
    formatTiles,
    clientId,
    headers,
    aggregationExp,
    aggregationResLevel,
    queryParameters,
    filters
  });
  const errorContext = {
    requestType: REQUEST_TYPES.DATA,
    connection,
    type,
    source
  };
  const data = await requestData({
    url,
    format: mapFormat,
    accessToken,
    errorContext
  });
  const result = {
    data,
    format: mapFormat,
    schema: metadata.schema
  };
  return result;
}

async function _fetchDataUrl({
  type,
  source,
  connection,
  credentials,
  geoColumn,
  columns,
  format,
  formatTiles,
  clientId,
  headers,
  aggregationExp,
  aggregationResLevel,
  queryParameters,
  filters
}) {
  const defaultCredentials = getDefaultCredentials();
  const localCreds = { ...(defaultCredentials.apiVersion === API_VERSIONS.V3 && defaultCredentials),
    ...credentials
  };
  checkFetchLayerDataParameters({
    type,
    source,
    connection,
    credentials: localCreds,
    geoColumn,
    columns,
    aggregationExp,
    aggregationResLevel,
    filters
  });

  if (!localCreds.mapsUrl) {
    localCreds.mapsUrl = buildMapsUrlFromBase(localCreds.apiBaseUrl);
  }

  const metadata = await mapInstantiation({
    type,
    source,
    connection,
    credentials: localCreds,
    geoColumn,
    columns,
    clientId,
    headers,
    aggregationExp,
    aggregationResLevel,
    queryParameters,
    filters
  });
  let url = null;
  let mapFormat;

  if (format) {
    mapFormat = format;
    url = getUrlFromMetadata(metadata, format);
    assert(url, "Format ".concat(format, " not available"));
  } else {
    const prioritizedFormats = [FORMATS.GEOJSON, FORMATS.JSON, FORMATS.NDJSON, FORMATS.TILEJSON];

    for (const f of prioritizedFormats) {
      url = getUrlFromMetadata(metadata, f);

      if (url) {
        mapFormat = f;
        break;
      }
    }

    assert(url && mapFormat, 'Unsupported data formats received from backend.');
  }

  if (format === FORMATS.TILEJSON && formatTiles) {
    log.assert(Object.values(TILE_FORMATS).includes(formatTiles), "Invalid value for formatTiles: ".concat(formatTiles, ". Use value from TILE_FORMATS"));
    url += "&".concat(encodeParameter('formatTiles', formatTiles));
  }

  const {
    accessToken
  } = localCreds;
  return {
    url,
    accessToken,
    mapFormat,
    metadata
  };
}

async function _fetchMapDataset(dataset, accessToken, credentials, clientId, headers) {
  const {
    aggregationExp,
    aggregationResLevel,
    connectionName: connection,
    columns,
    format,
    geoColumn,
    source,
    type,
    queryParameters
  } = dataset;
  const {
    url,
    mapFormat
  } = await _fetchDataUrl({
    aggregationExp,
    aggregationResLevel,
    clientId,
    credentials: { ...credentials,
      accessToken
    },
    connection,
    columns,
    format,
    geoColumn,
    headers,
    source,
    type,
    queryParameters
  });
  const cache = parseInt(new URL(url).searchParams.get('cache') || '', 10);

  if (cache && dataset.cache === cache) {
    return false;
  }

  dataset.cache = cache;
  const errorContext = {
    requestType: REQUEST_TYPES.DATA,
    connection,
    type,
    source
  };
  dataset.data = await requestData({
    url,
    format: mapFormat,
    accessToken,
    errorContext
  });
  return true;
}

async function _fetchTilestats(attribute, dataset, accessToken, credentials) {
  const {
    connectionName: connection,
    source,
    type
  } = dataset;
  const statsUrl = buildStatsUrlFromBase(credentials.apiBaseUrl);
  let baseUrl = "".concat(statsUrl, "/").concat(connection, "/");

  if (type === MAP_TYPES.QUERY) {
    baseUrl += attribute;
  } else {
    baseUrl += "".concat(source, "/").concat(attribute);
  }

  const errorContext = {
    requestType: REQUEST_TYPES.TILE_STATS,
    connection,
    type,
    source
  };
  let url = baseUrl;

  if (type === MAP_TYPES.QUERY) {
    url += "?".concat(encodeParameter('q', source));
  }

  let stats;

  if (url.length > MAX_GET_LENGTH && type === MAP_TYPES.QUERY) {
    stats = await requestJson({
      method: 'POST',
      url: baseUrl,
      accessToken,
      body: JSON.stringify({
        q: source
      }),
      errorContext
    });
  } else {
    stats = await requestJson({
      url,
      accessToken,
      errorContext
    });
  }

  const {
    attributes
  } = dataset.data.tilestats.layers[0];
  const index = attributes.findIndex(d => d.attribute === attribute);
  attributes[index] = stats;
  return true;
}

async function fillInMapDatasets({
  datasets,
  token
}, clientId, credentials, headers) {
  const promises = datasets.map(dataset => _fetchMapDataset(dataset, token, credentials, clientId, headers));
  return await Promise.all(promises);
}

async function fillInTileStats({
  datasets,
  keplerMapConfig,
  token
}, credentials) {
  const attributes = [];
  const {
    layers
  } = keplerMapConfig.config.visState;

  for (const layer of layers) {
    for (const channel of Object.keys(layer.visualChannels)) {
      var _layer$visualChannels;

      const attribute = (_layer$visualChannels = layer.visualChannels[channel]) === null || _layer$visualChannels === void 0 ? void 0 : _layer$visualChannels.name;

      if (attribute) {
        const dataset = datasets.find(d => d.id === layer.config.dataId);

        if (dataset.data.tilestats && dataset.type !== MAP_TYPES.TILESET) {
          attributes.push({
            attribute,
            dataset
          });
        }
      }
    }
  }

  const filteredAttributes = [];

  for (const a of attributes) {
    if (!filteredAttributes.find(({
      attribute,
      dataset
    }) => attribute === a.attribute && dataset === a.dataset)) {
      filteredAttributes.push(a);
    }
  }

  const promises = filteredAttributes.map(({
    attribute,
    dataset
  }) => _fetchTilestats(attribute, dataset, token, credentials));
  return await Promise.all(promises);
}

export async function fetchMap({
  cartoMapId,
  clientId,
  credentials,
  headers,
  autoRefresh,
  onNewData
}) {
  const defaultCredentials = getDefaultCredentials();
  const localCreds = { ...(defaultCredentials.apiVersion === API_VERSIONS.V3 && defaultCredentials),
    ...credentials
  };
  const {
    accessToken
  } = localCreds;
  assert(cartoMapId, 'Must define CARTO map id: fetchMap({cartoMapId: "XXXX-XXXX-XXXX"})');
  assert(localCreds.apiVersion === API_VERSIONS.V3, 'Method only available for v3');
  assert(localCreds.apiBaseUrl, 'Must define apiBaseUrl');

  if (!localCreds.mapsUrl) {
    localCreds.mapsUrl = buildMapsUrlFromBase(localCreds.apiBaseUrl);
  }

  if (autoRefresh || onNewData) {
    assert(onNewData, 'Must define `onNewData` when using autoRefresh');
    assert(typeof onNewData === 'function', '`onNewData` must be a function');
    assert(typeof autoRefresh === 'number' && autoRefresh > 0, '`autoRefresh` must be a positive number');
  }

  const url = "".concat(localCreds.mapsUrl, "/public/").concat(cartoMapId);
  const errorContext = {
    requestType: REQUEST_TYPES.PUBLIC_MAP,
    mapId: cartoMapId
  };
  const map = await requestJson({
    url,
    headers,
    accessToken,
    errorContext
  });
  let stopAutoRefresh;

  if (autoRefresh) {
    const intervalId = setInterval(async () => {
      const changed = await fillInMapDatasets(map, clientId, localCreds, headers);

      if (onNewData && changed.some(v => v === true)) {
        onNewData(parseMap(map));
      }
    }, autoRefresh * 1000);

    stopAutoRefresh = () => {
      clearInterval(intervalId);
    };
  }

  const geojsonLayers = map.keplerMapConfig.config.visState.layers.filter(({
    type
  }) => type === 'geojson' || type === 'point');
  const geojsonDatasetIds = geojsonLayers.map(({
    config
  }) => config.dataId);
  map.datasets.forEach(dataset => {
    if (geojsonDatasetIds.includes(dataset.id)) {
      const {
        config
      } = geojsonLayers.find(({
        config
      }) => config.dataId === dataset.id);
      dataset.format = 'geojson';

      if (!dataset.geoColumn && config.columns.geojson) {
        dataset.geoColumn = config.columns.geojson;
      }
    }
  });
  await fillInMapDatasets(map, clientId, localCreds, headers);
  await fillInTileStats(map, localCreds);
  const out = { ...parseMap(map),
    ...{
      stopAutoRefresh
    }
  };
  const textLayers = out.layers.filter(layer => {
    const pointType = layer.props.pointType || '';
    return pointType.includes('text');
  });

  if (textLayers.length && window.FontFace && !document.fonts.check('12px Inter')) {
    const font = new FontFace('Inter', 'url(https://fonts.gstatic.com/s/inter/v12/UcC73FwrK3iLTeHuS_fvQtMwCp50KnMa1ZL7W0Q5nw.woff2)');
    await font.load().then(f => document.fonts.add(f));
  }

  return out;
}
//# sourceMappingURL=maps-v3-client.js.map