import _defineProperty from "@babel/runtime/helpers/esm/defineProperty";
import { CompositeLayer, log } from '@deck.gl/core';
import { MVTLayer } from '@deck.gl/geo-layers';
import { fetchLayerData, getDataV2, API_VERSIONS } from '../api';
import { layerFromTileDataset } from '../api/layer-map';
import { COLUMNS_SUPPORT, FORMATS, GEO_COLUMN_SUPPORT, MAP_TYPES } from '../api/maps-api-common';
import { getDefaultCredentials } from '../config';
import { assert } from '../utils';
const defaultProps = {
  data: null,
  type: null,
  onDataLoad: {
    type: 'function',
    value: data => {}
  },
  onDataError: {
    type: 'function',
    value: null,
    optional: true
  },
  uniqueIdProperty: 'cartodb_id',
  credentials: null,
  connection: null,
  format: null,
  formatTiles: null,
  clientId: null,
  geoColumn: null,
  columns: {
    type: 'array',
    value: null
  },
  headers: {
    type: 'object',
    value: {},
    optional: true
  },
  aggregationExp: null,
  aggregationResLevel: null,
  queryParameters: null
};
export default class CartoLayer extends CompositeLayer {
  initializeState() {
    this.state = {
      data: null,
      apiVersion: null
    };
  }

  get isLoaded() {
    return this.getSubLayers().length > 0 && super.isLoaded;
  }

  _checkProps(props) {
    const {
      type,
      credentials,
      connection,
      geoColumn,
      columns
    } = props;
    const localCreds = { ...getDefaultCredentials(),
      ...credentials
    };
    const {
      apiVersion
    } = localCreds;
    log.assert(Object.values(API_VERSIONS).includes(apiVersion), "Invalid apiVersion ".concat(apiVersion, ". Use API_VERSIONS enum."));

    if (apiVersion === API_VERSIONS.V1 || apiVersion === API_VERSIONS.V2) {
      log.assert(type === MAP_TYPES.QUERY || type === MAP_TYPES.TILESET, "Invalid type ".concat(type, ". Use type MAP_TYPES.QUERY or MAP_TYPES.TILESET for apiVersion ").concat(apiVersion));
      log.assert(!connection, "Connection prop is not supported for apiVersion ".concat(apiVersion));
      log.assert(!geoColumn, "geoColumn prop is not supported for apiVersion ".concat(apiVersion));
      log.assert(!columns, "columns prop is not supported for apiVersion ".concat(apiVersion));
    } else if (apiVersion === API_VERSIONS.V3) {
      log.assert(connection, 'Missing mandatory connection parameter');
      log.assert(Object.values(MAP_TYPES).includes(type), "Invalid type ".concat(type, ". Use MAP_TYPES enum."));

      if (!COLUMNS_SUPPORT.includes(type)) {
        log.assert(!columns, "columns prop is only supported for types: ".concat(COLUMNS_SUPPORT.join()));
      }

      if (!GEO_COLUMN_SUPPORT.includes(type)) {
        log.assert(!geoColumn, "geoColumn prop is only supported for types: ".concat(GEO_COLUMN_SUPPORT.join()));
      }

      if (columns) {
        log.assert(Array.isArray(columns), 'columns prop must be an Array');
      }
    }
  }

  updateState({
    props,
    oldProps,
    changeFlags
  }) {
    this._checkProps(props);

    const shouldUpdateData = changeFlags.dataChanged || props.aggregationExp !== oldProps.aggregationExp || props.aggregationResLevel !== oldProps.aggregationResLevel || props.connection !== oldProps.connection || props.geoColumn !== oldProps.geoColumn || props.formatTiles !== oldProps.formatTiles || props.type !== oldProps.type || JSON.stringify(props.columns) !== JSON.stringify(oldProps.columns) || JSON.stringify(props.credentials) !== JSON.stringify(oldProps.credentials) || JSON.stringify(props.queryParameters) !== JSON.stringify(oldProps.queryParameters);

    if (shouldUpdateData) {
      this.setState({
        data: null,
        apiVersion: null
      });

      this._updateData();
    }
  }

  async _updateData() {
    try {
      var _this$props$onDataLoa, _this$props;

      const {
        type,
        data: source,
        credentials,
        connection,
        ...rest
      } = this.props;
      const localConfig = { ...getDefaultCredentials(),
        ...credentials
      };
      const {
        apiVersion
      } = localConfig;
      let result;

      if (apiVersion === API_VERSIONS.V1 || apiVersion === API_VERSIONS.V2) {
        result = {
          data: await getDataV2({
            type,
            source,
            credentials: credentials
          })
        };
      } else {
        result = await fetchLayerData({
          type,
          source,
          credentials: credentials,
          connection,
          ...rest,
          format: FORMATS.TILEJSON
        });
      }

      this.setState({ ...result,
        apiVersion
      });
      (_this$props$onDataLoa = (_this$props = this.props).onDataLoad) === null || _this$props$onDataLoa === void 0 ? void 0 : _this$props$onDataLoa.call(_this$props, result.data, {
        propName: 'data',
        layer: this
      });
    } catch (err) {
      if (this.props.onDataError) {
        this.props.onDataError(err);
      } else {
        throw err;
      }
    }
  }

  _getSubLayerAndProps() {
    assert(this.state);
    const {
      data,
      apiVersion
    } = this.state;
    const {
      uniqueIdProperty
    } = defaultProps;
    const {
      data: _notUsed,
      ...propsNoData
    } = this.props;
    const props = {
      uniqueIdProperty,
      ...propsNoData
    };

    if (apiVersion === API_VERSIONS.V1 || apiVersion === API_VERSIONS.V2) {
      return [MVTLayer, props];
    }

    const tileUrl = new URL(data.tiles[0]);
    props.formatTiles = props.formatTiles || tileUrl.searchParams.get('formatTiles');
    return [layerFromTileDataset(props.formatTiles, data.scheme, props.type), props];
  }

  renderLayers() {
    assert(this.state);
    const {
      apiVersion,
      data
    } = this.state;
    if (!data) return null;
    const {
      credentials,
      updateTriggers
    } = this.props;
    const loadOptions = this.getLoadOptions() || {};

    if (apiVersion === API_VERSIONS.V3) {
      var _loadOptions$fetch;

      const localConfig = { ...getDefaultCredentials(),
        ...credentials
      };
      const {
        accessToken
      } = localConfig;
      loadOptions.fetch = { ...loadOptions.fetch,
        headers: { ...((_loadOptions$fetch = loadOptions.fetch) === null || _loadOptions$fetch === void 0 ? void 0 : _loadOptions$fetch.headers),
          Authorization: "Bearer ".concat(accessToken)
        }
      };
    }

    const [layer, props] = this._getSubLayerAndProps();

    return new layer(props, this.getSubLayerProps({
      id: "carto-".concat(layer.layerName),
      data,
      loadOptions,
      updateTriggers
    }));
  }

}

_defineProperty(CartoLayer, "layerName", 'CartoLayer');

_defineProperty(CartoLayer, "defaultProps", defaultProps);
//# sourceMappingURL=carto-layer.js.map