import { _Tileset2D as Tileset2D } from '@deck.gl/geo-layers';
import { polyfill, geoToH3, h3GetResolution, h3ToGeoBoundary, h3ToParent, kRing } from 'h3-js';

function getHexagonsInBoundingBox({
  west,
  north,
  east,
  south
}, resolution) {
  const longitudeSpan = Math.abs(east - west);

  if (longitudeSpan > 180) {
    const nSegments = Math.ceil(longitudeSpan / 180);
    let h3Indices = [];

    for (let s = 0; s < nSegments; s++) {
      const segmentEast = east + s * 180;
      const segmentWest = Math.min(segmentEast + 179.9999999, west);
      h3Indices = h3Indices.concat(getHexagonsInBoundingBox({
        west: segmentWest,
        north,
        east: segmentEast,
        south
      }, resolution));
    }

    return [...new Set(h3Indices)];
  }

  const oversample = 2;
  const h3Indices = polyfill([[[west, north], [west, south], [east, south], [east, north], [west, north]]], resolution + oversample, true);
  return oversample ? [...new Set(h3Indices.map(i => h3ToParent(i, resolution)))] : h3Indices;
}

function tileToBoundingBox(index) {
  const coordinates = h3ToGeoBoundary(index);
  const latitudes = coordinates.map(c => c[0]);
  const longitudes = coordinates.map(c => c[1]);
  const west = Math.min(...longitudes);
  const south = Math.min(...latitudes);
  const east = Math.max(...longitudes);
  const north = Math.max(...latitudes);
  return {
    west,
    south,
    east,
    north
  };
}

const BIAS = 2;
export function getHexagonResolution(viewport) {
  const hexagonScaleFactor = 2 / 3 * viewport.zoom;
  const latitudeScaleFactor = Math.log(1 / Math.cos(Math.PI * viewport.latitude / 180));
  return Math.max(0, Math.floor(hexagonScaleFactor + latitudeScaleFactor - BIAS));
}
export default class H3Tileset2D extends Tileset2D {
  getTileIndices({
    viewport,
    minZoom,
    maxZoom
  }) {
    if (viewport.latitude === undefined) return [];
    const [east, south, west, north] = viewport.getBounds();
    let z = getHexagonResolution(viewport);
    let indices;

    if (typeof minZoom === 'number' && Number.isFinite(minZoom) && z < minZoom) {
      return [];
    }

    if (typeof maxZoom === 'number' && Number.isFinite(maxZoom) && z > maxZoom) {
      z = maxZoom;
      const center = geoToH3(viewport.latitude, viewport.longitude, maxZoom);
      indices = kRing(center, 1);
    } else {
      indices = getHexagonsInBoundingBox({
        west,
        north,
        east,
        south
      }, z);
    }

    return indices.map(i => ({
      i
    }));
  }

  getTileId({
    i
  }) {
    return i;
  }

  getTileMetadata({
    i
  }) {
    return {
      bbox: tileToBoundingBox(i)
    };
  }

  getTileZoom({
    i
  }) {
    return h3GetResolution(i);
  }

  getParentIndex(index) {
    const resolution = h3GetResolution(index.i);
    const i = h3ToParent(index.i, resolution - 1);
    return {
      i
    };
  }

}
//# sourceMappingURL=h3-tileset-2d.js.map