import _defineProperty from "@babel/runtime/helpers/esm/defineProperty";
import { CompositeLayer } from '@deck.gl/core';
import { ColumnLayer } from '@deck.gl/layers';
import { quadbinToOffset } from './quadbin-utils';
import vs from './raster-layer-vertex.glsl';
import { assert, createBinaryProxy } from '../utils';
const defaultProps = { ...ColumnLayer.defaultProps,
  extruded: false,
  diskResolution: 4,
  vertices: [[-0.5, -0.5], [0.5, -0.5], [0.5, 0.5], [-0.5, 0.5]]
};

class RasterColumnLayer extends ColumnLayer {
  getShaders() {
    const shaders = super.getShaders();
    return { ...shaders,
      vs
    };
  }

  initializeState() {
    const attributeManager = this.getAttributeManager();
    attributeManager.addInstanced({
      instanceElevations: {
        size: 1,
        transition: true,
        accessor: 'getElevation'
      },
      instanceFillColors: {
        size: this.props.colorFormat.length,
        type: 5121,
        normalized: true,
        transition: true,
        accessor: 'getFillColor',
        defaultValue: [0, 0, 0, 255]
      },
      instanceLineColors: {
        size: this.props.colorFormat.length,
        type: 5121,
        normalized: true,
        transition: true,
        accessor: 'getLineColor',
        defaultValue: [255, 255, 255, 255]
      }
    });
  }

}

_defineProperty(RasterColumnLayer, "layerName", 'RasterColumnLayer');

export default class RasterLayer extends CompositeLayer {
  renderLayers() {
    const {
      data,
      getElevation,
      getFillColor,
      getLineColor,
      getLineWidth,
      tileIndex,
      updateTriggers
    } = this.props;
    if (!data || !tileIndex) return null;
    const {
      blockWidth,
      blockHeight
    } = data;
    assert(blockWidth === blockHeight, "blockWidth (".concat(blockWidth, ") must equal blockHeight (").concat(blockHeight, ")"));
    const [xOffset, yOffset, scale] = quadbinToOffset(tileIndex);
    const offset = [xOffset, yOffset, scale / blockWidth];
    const CellLayer = this.getSubLayerClass('column', RasterColumnLayer);
    return new CellLayer(this.props, this.getSubLayerProps({
      id: 'cell',
      updateTriggers,
      getElevation: this.getSubLayerAccessor(getElevation),
      getFillColor: this.getSubLayerAccessor(getFillColor),
      getLineColor: this.getSubLayerAccessor(getLineColor),
      getLineWidth: this.getSubLayerAccessor(getLineWidth)
    }), {
      data: {
        data,
        length: blockWidth * blockHeight
      },
      offset
    });
  }

  getSubLayerAccessor(accessor) {
    if (typeof accessor !== 'function') {
      return super.getSubLayerAccessor(accessor);
    }

    return (object, info) => {
      const {
        data,
        index
      } = info;
      const binaryData = data.data;
      const proxy = createBinaryProxy(binaryData.cells, index);
      return accessor({
        properties: proxy
      }, info);
    };
  }

}

_defineProperty(RasterLayer, "layerName", 'RasterLayer');

_defineProperty(RasterLayer, "defaultProps", defaultProps);
//# sourceMappingURL=raster-layer.js.map