import earcut from 'earcut';
import { TileReader } from './carto-tile';
import { parsePbf } from './tile-loader-utils';
import { getWorkerUrl } from '../../utils';
const VERSION = typeof "8.9.36" !== 'undefined' ? "8.9.36" : 'latest';
const id = 'cartoVectorTile';
const DEFAULT_OPTIONS = {
  cartoVectorTile: {
    workerUrl: getWorkerUrl(id, VERSION)
  }
};
const CartoVectorTileLoader = {
  name: 'CARTO Vector Tile',
  version: VERSION,
  id,
  module: 'carto',
  extensions: ['pbf'],
  mimeTypes: ['application/vnd.carto-vector-tile'],
  category: 'geometry',
  parse: async (arrayBuffer, options) => parseCartoVectorTile(arrayBuffer, options),
  parseSync: parseCartoVectorTile,
  worker: true,
  options: DEFAULT_OPTIONS
};

function triangulatePolygon(polygons, target, {
  startPosition,
  endPosition,
  indices
}) {
  const coordLength = polygons.positions.size;
  const start = startPosition * coordLength;
  const end = endPosition * coordLength;
  const polygonPositions = polygons.positions.value.subarray(start, end);
  const holes = indices.slice(1).map(n => n - startPosition);
  const triangles = earcut(polygonPositions, holes, coordLength);

  for (let t = 0, tl = triangles.length; t < tl; ++t) {
    target.push(startPosition + triangles[t]);
  }
}

function triangulate(polygons) {
  const {
    polygonIndices,
    positions,
    primitivePolygonIndices
  } = polygons;
  const triangles = [];
  let rangeStart = 0;

  for (let i = 0; i < polygonIndices.value.length - 1; i++) {
    const startPosition = polygonIndices.value[i];
    const endPosition = polygonIndices.value[i + 1];
    const rangeEnd = primitivePolygonIndices.value.indexOf(endPosition);
    const indices = primitivePolygonIndices.value.subarray(rangeStart, rangeEnd);
    rangeStart = rangeEnd;
    triangulatePolygon(polygons, triangles, {
      startPosition,
      endPosition,
      indices
    });
  }

  polygons.triangles = {
    value: new Uint32Array(triangles),
    size: 1
  };
}

function parseCartoVectorTile(arrayBuffer, options) {
  if (!arrayBuffer) return null;
  const tile = parsePbf(arrayBuffer, TileReader);

  if (tile.polygons && !tile.polygons.triangles) {
    triangulate(tile.polygons);
  }

  return tile;
}

export default CartoVectorTileLoader;
//# sourceMappingURL=carto-vector-tile-loader.js.map