import _defineProperty from "@babel/runtime/helpers/esm/defineProperty";
import { Framebuffer, Renderbuffer, Texture2D, cssToDeviceRatio } from '@luma.gl/core';
import { equals } from '@math.gl/core';
import { _deepEqual as deepEqual } from '@deck.gl/core';
import CollisionFilterPass from './collision-filter-pass';
import MaskEffect from '../mask/mask-effect';
const DOWNSCALE = 2;
export default class CollisionFilterEffect {
  constructor() {
    _defineProperty(this, "id", 'collision-filter-effect');

    _defineProperty(this, "props", null);

    _defineProperty(this, "useInPicking", true);

    _defineProperty(this, "order", 1);

    _defineProperty(this, "channels", {});

    _defineProperty(this, "collisionFilterPass", void 0);

    _defineProperty(this, "collisionFBOs", {});

    _defineProperty(this, "dummyCollisionMap", void 0);

    _defineProperty(this, "lastViewport", void 0);
  }

  preRender(gl, {
    effects: allEffects,
    layers,
    layerFilter,
    viewports,
    onViewportActive,
    views,
    isPicking,
    preRenderStats = {}
  }) {
    var _preRenderStats$mask;

    if (!this.dummyCollisionMap) {
      this.dummyCollisionMap = new Texture2D(gl, {
        width: 1,
        height: 1
      });
    }

    if (isPicking) {
      return;
    }

    const collisionLayers = layers.filter(({
      props: {
        visible,
        collisionEnabled
      }
    }) => visible && collisionEnabled);

    if (collisionLayers.length === 0) {
      this.channels = {};
      return;
    }

    if (!this.collisionFilterPass) {
      this.collisionFilterPass = new CollisionFilterPass(gl, {
        id: 'default-collision-filter'
      });
    }

    const effects = allEffects === null || allEffects === void 0 ? void 0 : allEffects.filter(e => e.constructor === MaskEffect);
    const maskEffectRendered = (_preRenderStats$mask = preRenderStats['mask-effect']) === null || _preRenderStats$mask === void 0 ? void 0 : _preRenderStats$mask.didRender;

    const channels = this._groupByCollisionGroup(gl, collisionLayers);

    const viewport = viewports[0];
    const viewportChanged = !this.lastViewport || !this.lastViewport.equals(viewport) || maskEffectRendered;

    for (const collisionGroup in channels) {
      const collisionFBO = this.collisionFBOs[collisionGroup];
      const renderInfo = channels[collisionGroup];
      collisionFBO.resize({
        width: gl.canvas.width / DOWNSCALE,
        height: gl.canvas.height / DOWNSCALE
      });

      this._render(renderInfo, {
        effects,
        layerFilter,
        onViewportActive,
        views,
        viewport,
        viewportChanged
      });
    }
  }

  _render(renderInfo, {
    effects,
    layerFilter,
    onViewportActive,
    views,
    viewport,
    viewportChanged
  }) {
    const {
      collisionGroup
    } = renderInfo;
    const oldRenderInfo = this.channels[collisionGroup];

    if (!oldRenderInfo) {
      return;
    }

    const needsRender = viewportChanged || renderInfo === oldRenderInfo || !deepEqual(oldRenderInfo.layers, renderInfo.layers, 1) || renderInfo.layerBounds.some((b, i) => !equals(b, oldRenderInfo.layerBounds[i])) || renderInfo.allLayersLoaded !== oldRenderInfo.allLayersLoaded || renderInfo.layers.some(layer => layer.props.transitions);
    this.channels[collisionGroup] = renderInfo;

    if (needsRender) {
      this.lastViewport = viewport;
      const collisionFBO = this.collisionFBOs[collisionGroup];
      this.collisionFilterPass.renderCollisionMap(collisionFBO, {
        pass: 'collision-filter',
        isPicking: true,
        layers: renderInfo.layers,
        effects,
        layerFilter,
        viewports: viewport ? [viewport] : [],
        onViewportActive,
        views,
        moduleParameters: {
          dummyCollisionMap: this.dummyCollisionMap,
          devicePixelRatio: cssToDeviceRatio(collisionFBO.gl) / DOWNSCALE
        }
      });
    }
  }

  _groupByCollisionGroup(gl, collisionLayers) {
    const channelMap = {};

    for (const layer of collisionLayers) {
      const {
        collisionGroup
      } = layer.props;
      let channelInfo = channelMap[collisionGroup];

      if (!channelInfo) {
        channelInfo = {
          collisionGroup,
          layers: [],
          layerBounds: [],
          allLayersLoaded: true
        };
        channelMap[collisionGroup] = channelInfo;
      }

      channelInfo.layers.push(layer);
      channelInfo.layerBounds.push(layer.getBounds());

      if (!layer.isLoaded) {
        channelInfo.allLayersLoaded = false;
      }
    }

    for (const collisionGroup of Object.keys(channelMap)) {
      if (!this.collisionFBOs[collisionGroup]) {
        this.createFBO(gl, collisionGroup);
      }

      if (!this.channels[collisionGroup]) {
        this.channels[collisionGroup] = channelMap[collisionGroup];
      }
    }

    for (const collisionGroup of Object.keys(this.collisionFBOs)) {
      if (!channelMap[collisionGroup]) {
        this.destroyFBO(collisionGroup);
      }
    }

    return channelMap;
  }

  getModuleParameters(layer) {
    const {
      collisionGroup
    } = layer.props;
    const {
      collisionFBOs,
      dummyCollisionMap
    } = this;
    return {
      collisionFBO: collisionFBOs[collisionGroup],
      dummyCollisionMap
    };
  }

  cleanup() {
    if (this.dummyCollisionMap) {
      this.dummyCollisionMap.delete();
      this.dummyCollisionMap = undefined;
    }

    this.channels = {};

    for (const collisionGroup of Object.keys(this.collisionFBOs)) {
      this.destroyFBO(collisionGroup);
    }

    this.collisionFBOs = {};
    this.lastViewport = undefined;
  }

  createFBO(gl, collisionGroup) {
    const {
      width,
      height
    } = gl.canvas;
    const collisionMap = new Texture2D(gl, {
      width,
      height,
      parameters: {
        [10241]: 9728,
        [10240]: 9728,
        [10242]: 33071,
        [10243]: 33071
      }
    });
    const depthBuffer = new Renderbuffer(gl, {
      format: 33189,
      width,
      height
    });
    this.collisionFBOs[collisionGroup] = new Framebuffer(gl, {
      id: "Collision-".concat(collisionGroup),
      width,
      height,
      attachments: {
        [36064]: collisionMap,
        [36096]: depthBuffer
      }
    });
  }

  destroyFBO(collisionGroup) {
    const fbo = this.collisionFBOs[collisionGroup];

    for (const attachment of Object.values(fbo.attachments)) {
      attachment.delete();
    }

    fbo.delete();
    delete this.collisionFBOs[collisionGroup];
  }

}
//# sourceMappingURL=collision-filter-effect.js.map