import _defineProperty from "@babel/runtime/helpers/esm/defineProperty";
import { createRenderTarget } from './utils';
import { joinLayerBounds, makeViewport, getRenderBounds } from '../utils/projection-utils';
export class TerrainCover {
  constructor(targetLayer) {
    _defineProperty(this, "isDirty", true);

    _defineProperty(this, "targetLayer", void 0);

    _defineProperty(this, "renderViewport", null);

    _defineProperty(this, "bounds", null);

    _defineProperty(this, "fbo", void 0);

    _defineProperty(this, "pickingFbo", void 0);

    _defineProperty(this, "layers", []);

    _defineProperty(this, "tile", void 0);

    _defineProperty(this, "targetBounds", null);

    _defineProperty(this, "targetBoundsCommon", null);

    this.targetLayer = targetLayer;
    this.tile = getTile(targetLayer);
  }

  get id() {
    return this.targetLayer.id;
  }

  get isActive() {
    return Boolean(this.targetLayer.getCurrentLayer());
  }

  shouldUpdate({
    targetLayer,
    viewport,
    layers,
    layerNeedsRedraw
  }) {
    if (targetLayer) {
      this.targetLayer = targetLayer;
    }

    const sizeChanged = viewport ? this._updateViewport(viewport) : false;
    let layersChanged = layers ? this._updateLayers(layers) : false;

    if (layerNeedsRedraw) {
      for (const id of this.layers) {
        if (layerNeedsRedraw[id]) {
          layersChanged = true;
          break;
        }
      }
    }

    return layersChanged || sizeChanged;
  }

  _updateLayers(layers) {
    let needsRedraw = false;
    layers = this.tile ? getIntersectingLayers(this.tile, layers) : layers;

    if (layers.length !== this.layers.length) {
      needsRedraw = true;
    } else {
      for (let i = 0; i < layers.length; i++) {
        const id = layers[i].id;

        if (id !== this.layers[i]) {
          needsRedraw = true;
          break;
        }
      }
    }

    if (needsRedraw) {
      this.layers = layers.map(layer => layer.id);
    }

    return needsRedraw;
  }

  _updateViewport(viewport) {
    const targetLayer = this.targetLayer;
    let shouldRedraw = false;

    if (this.tile && 'boundingBox' in this.tile) {
      if (!this.targetBounds) {
        shouldRedraw = true;
        this.targetBounds = this.tile.boundingBox;
        const bottomLeftCommon = viewport.projectPosition(this.targetBounds[0]);
        const topRightCommon = viewport.projectPosition(this.targetBounds[1]);
        this.targetBoundsCommon = [bottomLeftCommon[0], bottomLeftCommon[1], topRightCommon[0], topRightCommon[1]];
      }
    } else if (this.targetBounds !== targetLayer.getBounds()) {
      shouldRedraw = true;
      this.targetBounds = targetLayer.getBounds();
      this.targetBoundsCommon = joinLayerBounds([targetLayer], viewport);
    }

    if (!this.targetBoundsCommon) {
      return false;
    }

    const newZoom = Math.ceil(viewport.zoom + 0.5);

    if (this.tile) {
      this.bounds = this.targetBoundsCommon;
    } else {
      var _this$renderViewport;

      const oldZoom = (_this$renderViewport = this.renderViewport) === null || _this$renderViewport === void 0 ? void 0 : _this$renderViewport.zoom;
      shouldRedraw = shouldRedraw || newZoom !== oldZoom;
      const newBounds = getRenderBounds(this.targetBoundsCommon, viewport);
      const oldBounds = this.bounds;
      shouldRedraw = shouldRedraw || !oldBounds || newBounds.some((x, i) => x !== oldBounds[i]);
      this.bounds = newBounds;
    }

    if (shouldRedraw) {
      this.renderViewport = makeViewport({
        bounds: this.bounds,
        zoom: newZoom,
        viewport
      });
    }

    return shouldRedraw;
  }

  getRenderFramebuffer() {
    if (!this.renderViewport || this.layers.length === 0) {
      return null;
    }

    if (!this.fbo) {
      this.fbo = createRenderTarget(this.targetLayer.context.gl, {
        id: this.id
      });
    }

    return this.fbo;
  }

  getPickingFramebuffer() {
    if (!this.renderViewport || this.layers.length === 0 && !this.targetLayer.props.pickable) {
      return null;
    }

    if (!this.pickingFbo) {
      this.pickingFbo = createRenderTarget(this.targetLayer.context.gl, {
        id: "".concat(this.id, "-picking")
      });
    }

    return this.pickingFbo;
  }

  filterLayers(layers) {
    return layers.filter(({
      id
    }) => this.layers.includes(id));
  }

  delete() {
    const {
      fbo,
      pickingFbo
    } = this;

    if (fbo) {
      fbo.texture.delete();
      fbo.delete();
    }

    if (pickingFbo) {
      pickingFbo.texture.delete();
      pickingFbo.delete();
    }
  }

}

function getIntersectingLayers(sourceTile, layers) {
  return layers.filter(layer => {
    const tile = getTile(layer);

    if (tile) {
      return intersect(sourceTile.boundingBox, tile.boundingBox);
    }

    return true;
  });
}

function getTile(layer) {
  while (layer) {
    const {
      tile
    } = layer.props;

    if (tile) {
      return tile;
    }

    layer = layer.parent;
  }

  return null;
}

function intersect(b1, b2) {
  if (b1 && b2) {
    return b1[0][0] < b2[1][0] && b2[0][0] < b1[1][0] && b1[0][1] < b2[1][1] && b2[0][1] < b1[1][1];
  }

  return false;
}
//# sourceMappingURL=terrain-cover.js.map