import _defineProperty from "@babel/runtime/helpers/esm/defineProperty";
import { Texture2D, ProgramManager } from '@luma.gl/core';
import { log } from '@deck.gl/core';
import { terrainModule } from './shader-module';
import { TerrainCover } from './terrain-cover';
import { TerrainPass } from './terrain-pass';
import { TerrainPickingPass } from './terrain-picking-pass';
import { HeightMapBuilder } from './height-map-builder';
export class TerrainEffect {
  constructor() {
    _defineProperty(this, "id", 'terrain-effect');

    _defineProperty(this, "props", null);

    _defineProperty(this, "useInPicking", true);

    _defineProperty(this, "isPicking", false);

    _defineProperty(this, "isDrapingEnabled", false);

    _defineProperty(this, "dummyHeightMap", void 0);

    _defineProperty(this, "heightMap", void 0);

    _defineProperty(this, "terrainPass", void 0);

    _defineProperty(this, "terrainPickingPass", void 0);

    _defineProperty(this, "terrainCovers", new Map());
  }

  initialize(gl) {
    this.dummyHeightMap = new Texture2D(gl, {
      width: 1,
      height: 1,
      data: new Uint8Array([0, 0, 0, 0])
    });
    this.terrainPass = new TerrainPass(gl, {
      id: 'terrain'
    });
    this.terrainPickingPass = new TerrainPickingPass(gl, {
      id: 'terrain-picking'
    });

    if (HeightMapBuilder.isSupported(gl)) {
      this.heightMap = new HeightMapBuilder(gl);
    } else {
      log.warn('Terrain offset mode is not supported by this browser')();
    }

    ProgramManager.getDefaultProgramManager(gl).addDefaultModule(terrainModule);
  }

  preRender(gl, opts) {
    if (!this.dummyHeightMap) {
      this.initialize(gl);

      for (const layer of opts.layers) {
        if (layer.props.operation.includes('terrain')) {
          layer.setChangeFlags({
            extensionsChanged: true
          });
        }
      }
    }

    if (opts.pickZ) {
      this.isDrapingEnabled = false;
      return;
    }

    const {
      viewports
    } = opts;
    const isPicking = opts.pass.startsWith('picking');
    this.isPicking = isPicking;
    this.isDrapingEnabled = true;
    const viewport = viewports[0];
    const layers = (isPicking ? this.terrainPickingPass : this.terrainPass).getRenderableLayers(viewport, opts);
    const terrainLayers = layers.filter(l => l.props.operation.includes('terrain'));

    if (terrainLayers.length === 0) {
      return;
    }

    if (!isPicking) {
      const offsetLayers = layers.filter(l => l.state.terrainDrawMode === 'offset');

      if (offsetLayers.length > 0) {
        this._updateHeightMap(terrainLayers, viewport, opts);
      }
    }

    const drapeLayers = layers.filter(l => l.state.terrainDrawMode === 'drape');

    this._updateTerrainCovers(terrainLayers, drapeLayers, viewport, opts);
  }

  getModuleParameters(layer) {
    var _this$heightMap, _this$heightMap2;

    const {
      terrainDrawMode
    } = layer.state;
    return {
      heightMap: (_this$heightMap = this.heightMap) === null || _this$heightMap === void 0 ? void 0 : _this$heightMap.getRenderFramebuffer(),
      heightMapBounds: (_this$heightMap2 = this.heightMap) === null || _this$heightMap2 === void 0 ? void 0 : _this$heightMap2.bounds,
      dummyHeightMap: this.dummyHeightMap,
      terrainCover: this.isDrapingEnabled ? this.terrainCovers.get(layer.id) : null,
      useTerrainHeightMap: terrainDrawMode === 'offset',
      terrainSkipRender: terrainDrawMode === 'drape' || !layer.props.operation.includes('draw')
    };
  }

  cleanup() {
    if (this.dummyHeightMap) {
      this.dummyHeightMap.delete();
      this.dummyHeightMap = undefined;
    }

    if (this.heightMap) {
      this.heightMap.delete();
      this.heightMap = undefined;
    }

    for (const terrainCover of this.terrainCovers.values()) {
      terrainCover.delete();
    }

    this.terrainCovers.clear();
  }

  _updateHeightMap(terrainLayers, viewport, opts) {
    if (!this.heightMap) {
      return;
    }

    const shouldUpdate = this.heightMap.shouldUpdate({
      layers: terrainLayers,
      viewport
    });

    if (!shouldUpdate) {
      return;
    }

    this.terrainPass.renderHeightMap(this.heightMap, { ...opts,
      layers: terrainLayers,
      moduleParameters: {
        heightMapBounds: this.heightMap.bounds,
        dummyHeightMap: this.dummyHeightMap,
        devicePixelRatio: 1,
        drawToTerrainHeightMap: true
      }
    });
  }

  _updateTerrainCovers(terrainLayers, drapeLayers, viewport, opts) {
    const layerNeedsRedraw = {};

    for (const layer of drapeLayers) {
      if (layer.state.terrainCoverNeedsRedraw) {
        layerNeedsRedraw[layer.id] = true;
        layer.state.terrainCoverNeedsRedraw = false;
      }
    }

    for (const terrainCover of this.terrainCovers.values()) {
      terrainCover.isDirty = terrainCover.isDirty || terrainCover.shouldUpdate({
        layerNeedsRedraw
      });
    }

    for (const layer of terrainLayers) {
      this._updateTerrainCover(layer, drapeLayers, viewport, opts);
    }

    if (!this.isPicking) {
      this._pruneTerrainCovers();
    }
  }

  _updateTerrainCover(terrainLayer, drapeLayers, viewport, opts) {
    const renderPass = this.isPicking ? this.terrainPickingPass : this.terrainPass;
    let terrainCover = this.terrainCovers.get(terrainLayer.id);

    if (!terrainCover) {
      terrainCover = new TerrainCover(terrainLayer);
      this.terrainCovers.set(terrainLayer.id, terrainCover);
    }

    try {
      const isDirty = terrainCover.shouldUpdate({
        targetLayer: terrainLayer,
        viewport,
        layers: drapeLayers
      });

      if (this.isPicking || terrainCover.isDirty || isDirty) {
        renderPass.renderTerrainCover(terrainCover, { ...opts,
          layers: drapeLayers,
          moduleParameters: {
            dummyHeightMap: this.dummyHeightMap,
            terrainSkipRender: false,
            devicePixelRatio: 1
          }
        });

        if (!this.isPicking) {
          terrainCover.isDirty = false;
        }
      }
    } catch (err) {
      terrainLayer.raiseError(err, "Error rendering terrain cover ".concat(terrainCover.id));
    }
  }

  _pruneTerrainCovers() {
    const idsToRemove = [];

    for (const [id, terrainCover] of this.terrainCovers) {
      if (!terrainCover.isActive) {
        idsToRemove.push(id);
      }
    }

    for (const id of idsToRemove) {
      this.terrainCovers.delete(id);
    }
  }

}
//# sourceMappingURL=terrain-effect.js.map