import _defineProperty from "@babel/runtime/helpers/esm/defineProperty";
import { CompositeLayer, COORDINATE_SYSTEM, _deepEqual as deepEqual } from '@deck.gl/core';
import { BitmapLayer } from '@deck.gl/layers';
import { ImageSource, createImageSource } from '@loaders.gl/wms';
import { WGS84ToPseudoMercator } from './utils';
const defaultProps = {
  id: 'imagery-layer',
  data: '',
  serviceType: 'auto',
  srs: 'auto',
  layers: {
    type: 'array',
    compare: true,
    value: []
  },
  onMetadataLoad: {
    type: 'function',
    value: () => {}
  },
  onMetadataLoadError: {
    type: 'function',
    value: console.error
  },
  onImageLoadStart: {
    type: 'function',
    value: () => {}
  },
  onImageLoad: {
    type: 'function',
    value: () => {}
  },
  onImageLoadError: {
    type: 'function',
    compare: false,
    value: (requestId, error) => console.error(error, requestId)
  }
};
export class WMSLayer extends CompositeLayer {
  constructor(...args) {
    super(...args);

    _defineProperty(this, "state", void 0);
  }

  get isLoaded() {
    var _this$state;

    return ((_this$state = this.state) === null || _this$state === void 0 ? void 0 : _this$state.loadCounter) === 0 && super.isLoaded;
  }

  shouldUpdateState() {
    return true;
  }

  initializeState() {
    this.state._nextRequestId = 0;
    this.state.lastRequestId = -1;
    this.state.loadCounter = 0;
  }

  updateState({
    changeFlags,
    props,
    oldProps
  }) {
    const {
      viewport
    } = this.context;

    if (changeFlags.dataChanged || props.serviceType !== oldProps.serviceType) {
      this.state.imageSource = this._createImageSource(props);

      this._loadMetadata();

      this.debounce(() => this.loadImage(viewport, 'image source changed'), 0);
    } else if (!deepEqual(props.layers, oldProps.layers, 1)) {
      this.debounce(() => this.loadImage(viewport, 'layers changed'), 0);
    } else if (changeFlags.viewportChanged) {
      this.debounce(() => this.loadImage(viewport, 'viewport changed'));
    }
  }

  finalizeState() {}

  renderLayers() {
    const {
      bounds,
      image,
      lastRequestParameters
    } = this.state;
    return image && new BitmapLayer({ ...this.getSubLayerProps({
        id: 'bitmap'
      }),
      _imageCoordinateSystem: lastRequestParameters.srs === 'EPSG:4326' ? COORDINATE_SYSTEM.LNGLAT : COORDINATE_SYSTEM.CARTESIAN,
      bounds,
      image
    });
  }

  async getFeatureInfoText(x, y) {
    const {
      lastRequestParameters
    } = this.state;

    if (lastRequestParameters) {
      var _this$state$imageSour, _this$state$imageSour2;

      const featureInfo = await ((_this$state$imageSour = (_this$state$imageSour2 = this.state.imageSource).getFeatureInfoText) === null || _this$state$imageSour === void 0 ? void 0 : _this$state$imageSour.call(_this$state$imageSour2, { ...lastRequestParameters,
        query_layers: lastRequestParameters.layers,
        x,
        y,
        info_format: 'application/vnd.ogc.gml'
      }));
      return featureInfo;
    }

    return '';
  }

  _createImageSource(props) {
    if (props.data instanceof ImageSource) {
      return props.data;
    }

    if (typeof props.data === 'string') {
      return createImageSource({
        url: props.data,
        loadOptions: props.loadOptions,
        type: props.serviceType
      });
    }

    throw new Error('invalid image source in props.data');
  }

  async _loadMetadata() {
    const {
      imageSource
    } = this.state;

    try {
      this.state.loadCounter++;
      const metadata = await imageSource.getMetadata();

      if (this.state.imageSource === imageSource) {
        var _this$getCurrentLayer;

        (_this$getCurrentLayer = this.getCurrentLayer()) === null || _this$getCurrentLayer === void 0 ? void 0 : _this$getCurrentLayer.props.onMetadataLoad(metadata);
      }
    } catch (error) {
      var _this$getCurrentLayer2;

      (_this$getCurrentLayer2 = this.getCurrentLayer()) === null || _this$getCurrentLayer2 === void 0 ? void 0 : _this$getCurrentLayer2.props.onMetadataLoadError(error);
    } finally {
      this.state.loadCounter--;
    }
  }

  async loadImage(viewport, reason) {
    const {
      layers,
      serviceType
    } = this.props;

    if (serviceType === 'wms' && layers.length === 0) {
      return;
    }

    const bounds = viewport.getBounds();
    const {
      width,
      height
    } = viewport;
    const requestId = this.getRequestId();
    let {
      srs
    } = this.props;

    if (srs === 'auto') {
      srs = viewport.resolution ? 'EPSG:4326' : 'EPSG:3857';
    }

    const requestParams = {
      width,
      height,
      bbox: bounds,
      layers,
      srs
    };

    if (srs === 'EPSG:3857') {
      const [minX, minY] = WGS84ToPseudoMercator([bounds[0], bounds[1]]);
      const [maxX, maxY] = WGS84ToPseudoMercator([bounds[2], bounds[3]]);
      requestParams.bbox = [minX, minY, maxX, maxY];
    }

    try {
      this.state.loadCounter++;
      this.props.onImageLoadStart(requestId);
      const image = await this.state.imageSource.getImage(requestParams);

      if (this.state.lastRequestId < requestId) {
        var _this$getCurrentLayer3;

        (_this$getCurrentLayer3 = this.getCurrentLayer()) === null || _this$getCurrentLayer3 === void 0 ? void 0 : _this$getCurrentLayer3.props.onImageLoad(requestId);
        this.setState({
          image,
          bounds,
          lastRequestParameters: requestParams,
          lastRequestId: requestId
        });
      }
    } catch (error) {
      var _this$getCurrentLayer4;

      this.raiseError(error, 'Load image');
      (_this$getCurrentLayer4 = this.getCurrentLayer()) === null || _this$getCurrentLayer4 === void 0 ? void 0 : _this$getCurrentLayer4.props.onImageLoadError(requestId, error);
    } finally {
      this.state.loadCounter--;
    }
  }

  getRequestId() {
    return this.state._nextRequestId++;
  }

  debounce(fn, ms = 500) {
    clearTimeout(this.state._timeoutId);
    this.state._timeoutId = setTimeout(() => fn(), ms);
  }

}

_defineProperty(WMSLayer, "layerName", 'WMSLayer');

_defineProperty(WMSLayer, "defaultProps", defaultProps);
//# sourceMappingURL=wms-layer.js.map