import _defineProperty from "@babel/runtime/helpers/esm/defineProperty";
import { getParameters, setParameters, withParameters } from '@luma.gl/core';
import { createDeckInstance, destroyDeckInstance, getViewPropsFromOverlay, getViewPropsFromCoordinateTransformer } from './utils';

const HIDE_ALL_LAYERS = () => false;

const GL_STATE = {
  depthMask: true,
  depthTest: true,
  blend: true,
  blendFunc: [770, 771, 1, 771],
  blendEquation: 32774
};

function noop() {}

const defaultProps = {
  interleaved: true
};
export default class GoogleMapsOverlay {
  constructor(props) {
    _defineProperty(this, "props", {});

    _defineProperty(this, "_map", null);

    _defineProperty(this, "_deck", null);

    _defineProperty(this, "_overlay", null);

    this.setProps({ ...defaultProps,
      ...props
    });
  }

  setMap(map) {
    if (map === this._map) {
      return;
    }

    const {
      VECTOR,
      UNINITIALIZED
    } = google.maps.RenderingType;

    if (this._map) {
      var _this$_overlay;

      if (!map && this._map.getRenderingType() === VECTOR && this.props.interleaved) {
        this._overlay.requestRedraw();
      }

      (_this$_overlay = this._overlay) === null || _this$_overlay === void 0 ? void 0 : _this$_overlay.setMap(null);
      this._map = null;
    }

    if (map) {
      this._map = map;
      const renderingType = map.getRenderingType();

      if (renderingType !== UNINITIALIZED) {
        this._createOverlay(map);
      } else {
        map.addListener('renderingtype_changed', () => {
          this._createOverlay(map);
        });
      }
    }
  }

  setProps(props) {
    Object.assign(this.props, props);

    if (this._deck) {
      const canvas = this._deck.getCanvas();

      if (props.style && canvas !== null && canvas !== void 0 && canvas.parentElement) {
        const parentStyle = canvas.parentElement.style;
        Object.assign(parentStyle, props.style);
        props.style = null;
      }

      this._deck.setProps(props);
    }
  }

  pickObject(params) {
    return this._deck && this._deck.pickObject(params);
  }

  pickMultipleObjects(params) {
    return this._deck && this._deck.pickMultipleObjects(params);
  }

  pickObjects(params) {
    return this._deck && this._deck.pickObjects(params);
  }

  finalize() {
    this.setMap(null);

    if (this._deck) {
      destroyDeckInstance(this._deck);
      this._deck = null;
    }
  }

  _createOverlay(map) {
    const {
      interleaved
    } = this.props;
    const {
      VECTOR,
      UNINITIALIZED
    } = google.maps.RenderingType;
    const renderingType = map.getRenderingType();

    if (renderingType === UNINITIALIZED) {
      return;
    }

    const isVectorMap = renderingType === VECTOR && google.maps.WebGLOverlayView;
    const OverlayView = isVectorMap ? google.maps.WebGLOverlayView : google.maps.OverlayView;
    const overlay = new OverlayView();

    if (overlay instanceof google.maps.WebGLOverlayView) {
      if (interleaved) {
        overlay.onAdd = noop;
        overlay.onContextRestored = this._onContextRestored.bind(this);
        overlay.onDraw = this._onDrawVectorInterleaved.bind(this);
      } else {
        overlay.onAdd = this._onAdd.bind(this);
        overlay.onContextRestored = noop;
        overlay.onDraw = this._onDrawVectorOverlay.bind(this);
      }

      overlay.onContextLost = this._onContextLost.bind(this);
    } else {
      overlay.onAdd = this._onAdd.bind(this);
      overlay.draw = this._onDrawRaster.bind(this);
    }

    overlay.onRemove = this._onRemove.bind(this);
    this._overlay = overlay;

    this._overlay.setMap(map);
  }

  _onAdd() {
    this._deck = createDeckInstance(this._map, this._overlay, this._deck, this.props);
  }

  _onContextRestored({
    gl
  }) {
    if (!this._map || !this._overlay) {
      return;
    }

    const _customRender = () => {
      if (this._overlay) {
        this._overlay.requestRedraw();
      }
    };

    const deck = createDeckInstance(this._map, this._overlay, this._deck, {
      gl,
      _customRender,
      ...this.props
    });
    this._deck = deck;
    const {
      animationLoop
    } = deck;

    animationLoop._renderFrame = () => {
      const ab = gl.getParameter(34964);
      withParameters(gl, {}, () => {
        animationLoop.onRender();
      });
      gl.bindBuffer(34962, ab);
    };
  }

  _onContextLost() {
    if (this._deck) {
      destroyDeckInstance(this._deck);
      this._deck = null;
    }
  }

  _onRemove() {
    var _this$_deck;

    (_this$_deck = this._deck) === null || _this$_deck === void 0 ? void 0 : _this$_deck.setProps({
      layerFilter: HIDE_ALL_LAYERS
    });
  }

  _onDrawRaster() {
    if (!this._deck || !this._map) {
      return;
    }

    const deck = this._deck;
    const {
      width,
      height,
      left,
      top,
      ...rest
    } = getViewPropsFromOverlay(this._map, this._overlay);
    const canvas = deck.getCanvas();

    if (canvas !== null && canvas !== void 0 && canvas.parentElement) {
      const parentStyle = canvas.parentElement.style;
      parentStyle.left = "".concat(left, "px");
      parentStyle.top = "".concat(top, "px");
    }

    const altitude = 10000;
    deck.setProps({
      width,
      height,
      viewState: {
        altitude,
        repeat: true,
        ...rest
      }
    });
    deck.redraw();
  }

  _onDrawVectorInterleaved({
    gl,
    transformer
  }) {
    if (!this._deck || !this._map) {
      return;
    }

    const deck = this._deck;
    deck.setProps({ ...getViewPropsFromCoordinateTransformer(this._map, transformer),
      width: null,
      height: null
    });

    if (deck.isInitialized) {
      const _framebuffer = getParameters(gl, 36006);

      deck.setProps({
        _framebuffer
      });
      deck.needsRedraw({
        clearRedrawFlags: true
      });
      setParameters(gl, {
        viewport: [0, 0, gl.canvas.width, gl.canvas.height],
        scissor: [0, 0, gl.canvas.width, gl.canvas.height],
        stencilFunc: [519, 0, 255, 519, 0, 255]
      });
      withParameters(gl, GL_STATE, () => {
        deck._drawLayers('google-vector', {
          clearCanvas: false
        });
      });
    }
  }

  _onDrawVectorOverlay({
    transformer
  }) {
    if (!this._deck || !this._map) {
      return;
    }

    const deck = this._deck;
    deck.setProps({ ...getViewPropsFromCoordinateTransformer(this._map, transformer)
    });
    deck.redraw();
  }

}
//# sourceMappingURL=google-maps-overlay.js.map