import { Deck } from '@deck.gl/core';
import { Matrix4, Vector2 } from '@math.gl/core';
const MAX_LATITUDE = 85.05113;
export function createDeckInstance(map, overlay, deck, props) {
  if (deck) {
    if (deck.userData._googleMap === map) {
      return deck;
    }

    destroyDeckInstance(deck);
  }

  const eventListeners = {
    click: null,
    rightclick: null,
    dblclick: null,
    mousemove: null,
    mouseout: null
  };
  const newDeck = new Deck({ ...props,
    useDevicePixels: props.interleaved ? true : props.useDevicePixels,
    style: props.interleaved ? null : {
      pointerEvents: 'none'
    },
    parent: getContainer(overlay, props.style),
    initialViewState: {
      longitude: 0,
      latitude: 0,
      zoom: 1
    },
    controller: false
  });

  for (const eventType in eventListeners) {
    eventListeners[eventType] = map.addListener(eventType, evt => handleMouseEvent(newDeck, eventType, evt));
  }

  newDeck.userData._googleMap = map;
  newDeck.userData._eventListeners = eventListeners;
  return newDeck;
}

function getContainer(overlay, style) {
  const container = document.createElement('div');
  container.style.position = 'absolute';
  Object.assign(container.style, style);

  if ('getPanes' in overlay) {
    var _overlay$getPanes;

    (_overlay$getPanes = overlay.getPanes()) === null || _overlay$getPanes === void 0 ? void 0 : _overlay$getPanes.overlayLayer.appendChild(container);
  } else {
    var _overlay$getMap;

    (_overlay$getMap = overlay.getMap()) === null || _overlay$getMap === void 0 ? void 0 : _overlay$getMap.getDiv().appendChild(container);
  }

  return container;
}

export function destroyDeckInstance(deck) {
  const {
    _eventListeners: eventListeners
  } = deck.userData;

  for (const eventType in eventListeners) {
    if (eventListeners[eventType]) {
      eventListeners[eventType].remove();
    }
  }

  deck.finalize();
}
export function getViewPropsFromOverlay(map, overlay) {
  const {
    width,
    height
  } = getMapSize(map);
  const projection = overlay.getProjection();
  const bounds = map.getBounds();

  if (!bounds) {
    return {
      width,
      height,
      left: 0,
      top: 0
    };
  }

  const ne = bounds.getNorthEast();
  const sw = bounds.getSouthWest();
  const topRight = projection.fromLatLngToDivPixel(ne);
  const bottomLeft = projection.fromLatLngToDivPixel(sw);
  const centerLngLat = pixelToLngLat(projection, width / 2, height / 2);
  const centerH = new google.maps.LatLng(0, centerLngLat[0]);
  const centerContainerPx = projection.fromLatLngToContainerPixel(centerH);
  const centerDivPx = projection.fromLatLngToDivPixel(centerH);

  if (!topRight || !bottomLeft || !centerDivPx || !centerContainerPx) {
    return {
      width,
      height,
      left: 0,
      top: 0
    };
  }

  const leftOffset = Math.round(centerDivPx.x - centerContainerPx.x);
  let topOffset = centerDivPx.y - centerContainerPx.y;
  const topLngLat = pixelToLngLat(projection, width / 2, 0);
  const bottomLngLat = pixelToLngLat(projection, width / 2, height);
  let latitude = centerLngLat[1];
  const longitude = centerLngLat[0];

  if (Math.abs(latitude) > MAX_LATITUDE) {
    latitude = latitude > 0 ? MAX_LATITUDE : -MAX_LATITUDE;
    const center = new google.maps.LatLng(latitude, longitude);
    const centerPx = projection.fromLatLngToContainerPixel(center);
    topOffset += centerPx.y - height / 2;
  }

  topOffset = Math.round(topOffset);
  const delta = new Vector2(topLngLat).sub(bottomLngLat);
  let bearing = 180 * delta.verticalAngle() / Math.PI;
  if (bearing < 0) bearing += 360;
  const heading = map.getHeading() || 0;
  let zoom = map.getZoom() - 1;
  let scale;

  if (bearing === 0) {
    scale = height ? (bottomLeft.y - topRight.y) / height : 1;
  } else if (bearing === heading) {
    const viewDiagonal = new Vector2([topRight.x, topRight.y]).sub([bottomLeft.x, bottomLeft.y]).len();
    const mapDiagonal = new Vector2([width, -height]).len();
    scale = mapDiagonal ? viewDiagonal / mapDiagonal : 1;
  }

  zoom += Math.log2(scale || 1);
  return {
    width,
    height,
    left: leftOffset,
    top: topOffset,
    zoom,
    bearing,
    pitch: map.getTilt(),
    latitude,
    longitude
  };
}
export function getViewPropsFromCoordinateTransformer(map, transformer) {
  const {
    width,
    height
  } = getMapSize(map);
  const {
    center,
    heading: bearing,
    tilt: pitch,
    zoom
  } = transformer.getCameraParams();
  const fovy = 25;
  const aspect = height ? width / height : 1;
  const near = 0.75;
  const far = 300000000000000;
  const projectionMatrix = new Matrix4().perspective({
    fovy: fovy * Math.PI / 180,
    aspect,
    near,
    far
  });
  const focalDistance = 0.5 * projectionMatrix[5];
  return {
    width,
    height,
    viewState: {
      altitude: focalDistance,
      bearing,
      latitude: center.lat(),
      longitude: center.lng(),
      pitch,
      projectionMatrix,
      repeat: true,
      zoom: zoom - 1
    }
  };
}

function getMapSize(map) {
  const container = map.getDiv().firstChild;
  return {
    width: container.offsetWidth,
    height: container.offsetHeight
  };
}

function pixelToLngLat(projection, x, y) {
  const point = new google.maps.Point(x, y);
  const latLng = projection.fromContainerPixelToLatLng(point);
  return [latLng.lng(), latLng.lat()];
}

function getEventPixel(event, deck) {
  if (event.pixel) {
    return event.pixel;
  }

  const point = deck.getViewports()[0].project([event.latLng.lng(), event.latLng.lat()]);
  return {
    x: point[0],
    y: point[1]
  };
}

function handleMouseEvent(deck, type, event) {
  if (!deck.isInitialized) {
    return;
  }

  const mockEvent = {
    type,
    offsetCenter: getEventPixel(event, deck),
    srcEvent: event
  };

  switch (type) {
    case 'click':
    case 'rightclick':
      mockEvent.type = 'click';
      mockEvent.tapCount = 1;

      deck._onPointerDown(mockEvent);

      deck._onEvent(mockEvent);

      break;

    case 'dblclick':
      mockEvent.type = 'click';
      mockEvent.tapCount = 2;

      deck._onEvent(mockEvent);

      break;

    case 'mousemove':
      mockEvent.type = 'pointermove';

      deck._onPointerMove(mockEvent);

      break;

    case 'mouseout':
      mockEvent.type = 'pointerleave';

      deck._onPointerMove(mockEvent);

      break;

    default:
      return;
  }
}
//# sourceMappingURL=utils.js.map