"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDeckInstance = getDeckInstance;
exports.removeDeckInstance = removeDeckInstance;
exports.getInterleavedProps = getInterleavedProps;
exports.addLayer = addLayer;
exports.removeLayer = removeLayer;
exports.updateLayer = updateLayer;
exports.drawLayer = drawLayer;
exports.getViewState = getViewState;

var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));

var _core = require("@deck.gl/core");

var _webMercator = require("@math.gl/web-mercator");

function ownKeys(object, enumerableOnly) { var keys = Object.keys(object); if (Object.getOwnPropertySymbols) { var symbols = Object.getOwnPropertySymbols(object); if (enumerableOnly) { symbols = symbols.filter(function (sym) { return Object.getOwnPropertyDescriptor(object, sym).enumerable; }); } keys.push.apply(keys, symbols); } return keys; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; if (i % 2) { ownKeys(Object(source), true).forEach(function (key) { (0, _defineProperty2.default)(target, key, source[key]); }); } else if (Object.getOwnPropertyDescriptors) { Object.defineProperties(target, Object.getOwnPropertyDescriptors(source)); } else { ownKeys(Object(source)).forEach(function (key) { Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key)); }); } } return target; }

var TILE_SIZE = 512;
var DEGREES_TO_RADIANS = Math.PI / 180;

function getDeckInstance(_ref) {
  var map = _ref.map,
      gl = _ref.gl,
      deck = _ref.deck;

  if (map.__deck) {
    return map.__deck;
  }

  var customRender = deck === null || deck === void 0 ? void 0 : deck.props._customRender;
  var onLoad = deck === null || deck === void 0 ? void 0 : deck.props.onLoad;
  var deckProps = getInterleavedProps(_objectSpread(_objectSpread({}, deck === null || deck === void 0 ? void 0 : deck.props), {}, {
    _customRender: function _customRender() {
      map.triggerRepaint();
      customRender === null || customRender === void 0 ? void 0 : customRender('');
    }
  }));
  var deckInstance;

  if (!deck || deck.props.gl === gl) {
    Object.assign(deckProps, {
      gl: gl,
      width: null,
      height: null,
      touchAction: 'unset',
      viewState: getViewState(map)
    });

    if (deck !== null && deck !== void 0 && deck.isInitialized) {
      watchMapMove(deck, map);
    } else {
      deckProps.onLoad = function () {
        onLoad === null || onLoad === void 0 ? void 0 : onLoad();
        watchMapMove(deckInstance, map);
      };
    }
  }

  if (deck) {
    deckInstance = deck;
    deck.setProps(deckProps);
    deck.userData.isExternal = true;
  } else {
    deckInstance = new _core.Deck(deckProps);
    map.on('remove', function () {
      removeDeckInstance(map);
    });
  }

  deckInstance.userData.mapboxLayers = new Set();
  map.__deck = deckInstance;
  map.on('render', function () {
    if (deckInstance.isInitialized) afterRender(deckInstance, map);
  });
  return deckInstance;
}

function watchMapMove(deck, map) {
  var _handleMapMove = function _handleMapMove() {
    if (deck.isInitialized) {
      onMapMove(deck, map);
    } else {
      map.off('move', _handleMapMove);
    }
  };

  map.on('move', _handleMapMove);
}

function removeDeckInstance(map) {
  var _map$__deck;

  (_map$__deck = map.__deck) === null || _map$__deck === void 0 ? void 0 : _map$__deck.finalize();
  map.__deck = null;
}

function getInterleavedProps(currProps) {
  var nextProps = _objectSpread(_objectSpread({}, currProps), {}, {
    parameters: _objectSpread({
      depthMask: true,
      depthTest: true,
      blend: true,
      blendFunc: [770, 771, 1, 771],
      polygonOffsetFill: true,
      depthFunc: 515,
      blendEquation: 32774
    }, currProps.parameters),
    views: currProps.views || [new _core.MapView({
      id: 'mapbox'
    })]
  });

  return nextProps;
}

function addLayer(deck, layer) {
  deck.userData.mapboxLayers.add(layer);
  updateLayers(deck);
}

function removeLayer(deck, layer) {
  deck.userData.mapboxLayers.delete(layer);
  updateLayers(deck);
}

function updateLayer(deck, layer) {
  updateLayers(deck);
}

function drawLayer(deck, map, layer) {
  var _ref2 = deck.userData,
      currentViewport = _ref2.currentViewport;
  var clearStack = false;

  if (!currentViewport) {
    currentViewport = getViewport(deck, map, true);
    deck.userData.currentViewport = currentViewport;
    clearStack = true;
  }

  if (!deck.isInitialized) {
    return;
  }

  deck._drawLayers('mapbox-repaint', {
    viewports: [currentViewport],
    layerFilter: function layerFilter(_ref3) {
      var deckLayer = _ref3.layer;
      return layer.id === deckLayer.id || deckLayer.props.operation.includes('terrain');
    },
    clearStack: clearStack,
    clearCanvas: false
  });
}

function getViewState(map) {
  var _map$getTerrain;

  var _map$getCenter = map.getCenter(),
      lng = _map$getCenter.lng,
      lat = _map$getCenter.lat;

  var viewState = {
    longitude: (lng + 540) % 360 - 180,
    latitude: lat,
    zoom: map.getZoom(),
    bearing: map.getBearing(),
    pitch: map.getPitch(),
    padding: map.getPadding(),
    repeat: map.getRenderWorldCopies()
  };

  if ((_map$getTerrain = map.getTerrain) !== null && _map$getTerrain !== void 0 && _map$getTerrain.call(map)) {
    centerCameraOnTerrain(map, viewState);
  }

  return viewState;
}

function centerCameraOnTerrain(map, viewState) {
  if (map.getFreeCameraOptions) {
    var _map$getFreeCameraOpt = map.getFreeCameraOptions(),
        position = _map$getFreeCameraOpt.position;

    if (!position || position.z === undefined) {
      return;
    }

    var height = map.transform.height;
    var longitude = viewState.longitude,
        latitude = viewState.latitude,
        pitch = viewState.pitch;
    var cameraX = position.x * TILE_SIZE;
    var cameraY = (1 - position.y) * TILE_SIZE;
    var cameraZ = position.z * TILE_SIZE;
    var center = (0, _webMercator.lngLatToWorld)([longitude, latitude]);
    var dx = cameraX - center[0];
    var dy = cameraY - center[1];
    var cameraToCenterDistanceGround = Math.sqrt(dx * dx + dy * dy);
    var pitchRadians = pitch * DEGREES_TO_RADIANS;
    var altitudePixels = 1.5 * height;
    var scale = pitchRadians < 0.001 ? altitudePixels * Math.cos(pitchRadians) / cameraZ : altitudePixels * Math.sin(pitchRadians) / cameraToCenterDistanceGround;
    viewState.zoom = Math.log2(scale);
    var cameraZFromSurface = altitudePixels * Math.cos(pitchRadians) / scale;
    var surfaceElevation = cameraZ - cameraZFromSurface;
    viewState.position = [0, 0, surfaceElevation / (0, _webMercator.unitsPerMeter)(latitude)];
  } else if (typeof map.transform.elevation === 'number') {
    viewState.position = [0, 0, map.transform.elevation];
  }
}

function getViewport(deck, map) {
  var useMapboxProjection = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : true;
  return new _core.WebMercatorViewport(_objectSpread(_objectSpread({
    id: 'mapbox',
    x: 0,
    y: 0,
    width: deck.width,
    height: deck.height
  }, getViewState(map)), {}, {
    nearZMultiplier: useMapboxProjection ? 0.02 : 0.1,
    nearZ: map.transform._nearZ / map.transform.height,
    farZ: map.transform._farZ / map.transform.height
  }));
}

function afterRender(deck, map) {
  var _ref4 = deck.userData,
      mapboxLayers = _ref4.mapboxLayers,
      isExternal = _ref4.isExternal;

  if (isExternal) {
    var mapboxLayerIds = Array.from(mapboxLayers, function (layer) {
      return layer.id;
    });
    var deckLayers = (0, _core._flatten)(deck.props.layers, Boolean);
    var hasNonMapboxLayers = deckLayers.some(function (layer) {
      return layer && !mapboxLayerIds.includes(layer.id);
    });
    var viewports = deck.getViewports();
    var mapboxViewportIdx = viewports.findIndex(function (vp) {
      return vp.id === 'mapbox';
    });
    var hasNonMapboxViews = viewports.length > 1 || mapboxViewportIdx < 0;

    if (hasNonMapboxLayers || hasNonMapboxViews) {
      if (mapboxViewportIdx >= 0) {
        viewports = viewports.slice();
        viewports[mapboxViewportIdx] = getViewport(deck, map, false);
      }

      deck._drawLayers('mapbox-repaint', {
        viewports: viewports,
        layerFilter: function layerFilter(params) {
          return (!deck.props.layerFilter || deck.props.layerFilter(params)) && (params.viewport.id !== 'mapbox' || !mapboxLayerIds.includes(params.layer.id));
        },
        clearCanvas: false
      });
    }
  }

  deck.userData.currentViewport = null;
}

function onMapMove(deck, map) {
  deck.setProps({
    viewState: getViewState(map)
  });
  deck.needsRedraw({
    clearRedrawFlags: true
  });
}

function updateLayers(deck) {
  if (deck.userData.isExternal) {
    return;
  }

  var layers = [];
  deck.userData.mapboxLayers.forEach(function (deckLayer) {
    var LayerType = deckLayer.props.type;
    var layer = new LayerType(deckLayer.props);
    layers.push(layer);
  });
  deck.setProps({
    layers: layers
  });
}
//# sourceMappingURL=deck-utils.js.map