import { Deck, WebMercatorViewport, MapView, _flatten as flatten } from '@deck.gl/core';
import { lngLatToWorld, unitsPerMeter } from '@math.gl/web-mercator';
const TILE_SIZE = 512;
const DEGREES_TO_RADIANS = Math.PI / 180;
export function getDeckInstance({
  map,
  gl,
  deck
}) {
  if (map.__deck) {
    return map.__deck;
  }

  const customRender = deck === null || deck === void 0 ? void 0 : deck.props._customRender;
  const onLoad = deck === null || deck === void 0 ? void 0 : deck.props.onLoad;
  const deckProps = getInterleavedProps({ ...(deck === null || deck === void 0 ? void 0 : deck.props),
    _customRender: () => {
      map.triggerRepaint();
      customRender === null || customRender === void 0 ? void 0 : customRender('');
    }
  });
  let deckInstance;

  if (!deck || deck.props.gl === gl) {
    Object.assign(deckProps, {
      gl,
      width: null,
      height: null,
      touchAction: 'unset',
      viewState: getViewState(map)
    });

    if (deck !== null && deck !== void 0 && deck.isInitialized) {
      watchMapMove(deck, map);
    } else {
      deckProps.onLoad = () => {
        onLoad === null || onLoad === void 0 ? void 0 : onLoad();
        watchMapMove(deckInstance, map);
      };
    }
  }

  if (deck) {
    deckInstance = deck;
    deck.setProps(deckProps);
    deck.userData.isExternal = true;
  } else {
    deckInstance = new Deck(deckProps);
    map.on('remove', () => {
      removeDeckInstance(map);
    });
  }

  deckInstance.userData.mapboxLayers = new Set();
  map.__deck = deckInstance;
  map.on('render', () => {
    if (deckInstance.isInitialized) afterRender(deckInstance, map);
  });
  return deckInstance;
}

function watchMapMove(deck, map) {
  const _handleMapMove = () => {
    if (deck.isInitialized) {
      onMapMove(deck, map);
    } else {
      map.off('move', _handleMapMove);
    }
  };

  map.on('move', _handleMapMove);
}

export function removeDeckInstance(map) {
  var _map$__deck;

  (_map$__deck = map.__deck) === null || _map$__deck === void 0 ? void 0 : _map$__deck.finalize();
  map.__deck = null;
}
export function getInterleavedProps(currProps) {
  const nextProps = { ...currProps,
    parameters: {
      depthMask: true,
      depthTest: true,
      blend: true,
      blendFunc: [770, 771, 1, 771],
      polygonOffsetFill: true,
      depthFunc: 515,
      blendEquation: 32774,
      ...currProps.parameters
    },
    views: currProps.views || [new MapView({
      id: 'mapbox'
    })]
  };
  return nextProps;
}
export function addLayer(deck, layer) {
  deck.userData.mapboxLayers.add(layer);
  updateLayers(deck);
}
export function removeLayer(deck, layer) {
  deck.userData.mapboxLayers.delete(layer);
  updateLayers(deck);
}
export function updateLayer(deck, layer) {
  updateLayers(deck);
}
export function drawLayer(deck, map, layer) {
  let {
    currentViewport
  } = deck.userData;
  let clearStack = false;

  if (!currentViewport) {
    currentViewport = getViewport(deck, map, true);
    deck.userData.currentViewport = currentViewport;
    clearStack = true;
  }

  if (!deck.isInitialized) {
    return;
  }

  deck._drawLayers('mapbox-repaint', {
    viewports: [currentViewport],
    layerFilter: ({
      layer: deckLayer
    }) => layer.id === deckLayer.id || deckLayer.props.operation.includes('terrain'),
    clearStack,
    clearCanvas: false
  });
}
export function getViewState(map) {
  var _map$getTerrain;

  const {
    lng,
    lat
  } = map.getCenter();
  const viewState = {
    longitude: (lng + 540) % 360 - 180,
    latitude: lat,
    zoom: map.getZoom(),
    bearing: map.getBearing(),
    pitch: map.getPitch(),
    padding: map.getPadding(),
    repeat: map.getRenderWorldCopies()
  };

  if ((_map$getTerrain = map.getTerrain) !== null && _map$getTerrain !== void 0 && _map$getTerrain.call(map)) {
    centerCameraOnTerrain(map, viewState);
  }

  return viewState;
}

function centerCameraOnTerrain(map, viewState) {
  if (map.getFreeCameraOptions) {
    const {
      position
    } = map.getFreeCameraOptions();

    if (!position || position.z === undefined) {
      return;
    }

    const height = map.transform.height;
    const {
      longitude,
      latitude,
      pitch
    } = viewState;
    const cameraX = position.x * TILE_SIZE;
    const cameraY = (1 - position.y) * TILE_SIZE;
    const cameraZ = position.z * TILE_SIZE;
    const center = lngLatToWorld([longitude, latitude]);
    const dx = cameraX - center[0];
    const dy = cameraY - center[1];
    const cameraToCenterDistanceGround = Math.sqrt(dx * dx + dy * dy);
    const pitchRadians = pitch * DEGREES_TO_RADIANS;
    const altitudePixels = 1.5 * height;
    const scale = pitchRadians < 0.001 ? altitudePixels * Math.cos(pitchRadians) / cameraZ : altitudePixels * Math.sin(pitchRadians) / cameraToCenterDistanceGround;
    viewState.zoom = Math.log2(scale);
    const cameraZFromSurface = altitudePixels * Math.cos(pitchRadians) / scale;
    const surfaceElevation = cameraZ - cameraZFromSurface;
    viewState.position = [0, 0, surfaceElevation / unitsPerMeter(latitude)];
  } else if (typeof map.transform.elevation === 'number') {
    viewState.position = [0, 0, map.transform.elevation];
  }
}

function getViewport(deck, map, useMapboxProjection = true) {
  return new WebMercatorViewport({
    id: 'mapbox',
    x: 0,
    y: 0,
    width: deck.width,
    height: deck.height,
    ...getViewState(map),
    nearZMultiplier: useMapboxProjection ? 0.02 : 0.1,
    nearZ: map.transform._nearZ / map.transform.height,
    farZ: map.transform._farZ / map.transform.height
  });
}

function afterRender(deck, map) {
  const {
    mapboxLayers,
    isExternal
  } = deck.userData;

  if (isExternal) {
    const mapboxLayerIds = Array.from(mapboxLayers, layer => layer.id);
    const deckLayers = flatten(deck.props.layers, Boolean);
    const hasNonMapboxLayers = deckLayers.some(layer => layer && !mapboxLayerIds.includes(layer.id));
    let viewports = deck.getViewports();
    const mapboxViewportIdx = viewports.findIndex(vp => vp.id === 'mapbox');
    const hasNonMapboxViews = viewports.length > 1 || mapboxViewportIdx < 0;

    if (hasNonMapboxLayers || hasNonMapboxViews) {
      if (mapboxViewportIdx >= 0) {
        viewports = viewports.slice();
        viewports[mapboxViewportIdx] = getViewport(deck, map, false);
      }

      deck._drawLayers('mapbox-repaint', {
        viewports,
        layerFilter: params => (!deck.props.layerFilter || deck.props.layerFilter(params)) && (params.viewport.id !== 'mapbox' || !mapboxLayerIds.includes(params.layer.id)),
        clearCanvas: false
      });
    }
  }

  deck.userData.currentViewport = null;
}

function onMapMove(deck, map) {
  deck.setProps({
    viewState: getViewState(map)
  });
  deck.needsRedraw({
    clearRedrawFlags: true
  });
}

function updateLayers(deck) {
  if (deck.userData.isExternal) {
    return;
  }

  const layers = [];
  deck.userData.mapboxLayers.forEach(deckLayer => {
    const LayerType = deckLayer.props.type;
    const layer = new LayerType(deckLayer.props);
    layers.push(layer);
  });
  deck.setProps({
    layers
  });
}
//# sourceMappingURL=deck-utils.js.map