import _defineProperty from "@babel/runtime/helpers/esm/defineProperty";
import { Layer, project32, picking, log } from '@deck.gl/core';
import { isWebGL2 } from '@luma.gl/core';
import { pbr } from '@luma.gl/shadertools';
import { ScenegraphNode, createGLTFObjects } from '@luma.gl/experimental';
import { GLTFLoader } from '@loaders.gl/gltf';
import { waitForGLTFAssets } from './gltf-utils';
import { MATRIX_ATTRIBUTES, shouldComposeModelMatrix } from '../utils/matrix';
import vs from './scenegraph-layer-vertex.glsl';
import fs from './scenegraph-layer-fragment.glsl';
const DEFAULT_COLOR = [255, 255, 255, 255];
const defaultProps = {
  scenegraph: {
    type: 'object',
    value: null,
    async: true
  },
  getScene: gltf => {
    if (gltf && gltf.scenes) {
      return typeof gltf.scene === 'object' ? gltf.scene : gltf.scenes[gltf.scene || 0];
    }

    return gltf;
  },
  getAnimator: scenegraph => scenegraph && scenegraph.animator,
  _animations: null,
  sizeScale: {
    type: 'number',
    value: 1,
    min: 0
  },
  sizeMinPixels: {
    type: 'number',
    min: 0,
    value: 0
  },
  sizeMaxPixels: {
    type: 'number',
    min: 0,
    value: Number.MAX_SAFE_INTEGER
  },
  getPosition: {
    type: 'accessor',
    value: x => x.position
  },
  getColor: {
    type: 'accessor',
    value: DEFAULT_COLOR
  },
  _lighting: 'flat',
  _imageBasedLightingEnvironment: null,
  getOrientation: {
    type: 'accessor',
    value: [0, 0, 0]
  },
  getScale: {
    type: 'accessor',
    value: [1, 1, 1]
  },
  getTranslation: {
    type: 'accessor',
    value: [0, 0, 0]
  },
  getTransformMatrix: {
    type: 'accessor',
    value: []
  },
  loaders: [GLTFLoader]
};
export default class ScenegraphLayer extends Layer {
  constructor(...args) {
    super(...args);

    _defineProperty(this, "state", void 0);
  }

  getShaders() {
    const modules = [project32, picking];

    if (this.props._lighting === 'pbr') {
      modules.push(pbr);
    }

    return super.getShaders({
      vs,
      fs,
      modules
    });
  }

  initializeState() {
    const attributeManager = this.getAttributeManager();
    attributeManager.addInstanced({
      instancePositions: {
        size: 3,
        type: 5130,
        fp64: this.use64bitPositions(),
        accessor: 'getPosition',
        transition: true
      },
      instanceColors: {
        type: 5121,
        size: this.props.colorFormat.length,
        accessor: 'getColor',
        normalized: true,
        defaultValue: DEFAULT_COLOR,
        transition: true
      },
      instanceModelMatrix: MATRIX_ATTRIBUTES
    });
  }

  updateState(params) {
    super.updateState(params);
    const {
      props,
      oldProps
    } = params;

    if (props.scenegraph !== oldProps.scenegraph) {
      this._updateScenegraph();
    } else if (props._animations !== oldProps._animations) {
      this._applyAnimationsProp(this.state.scenegraph, this.state.animator, props._animations);
    }
  }

  finalizeState(context) {
    super.finalizeState(context);

    this._deleteScenegraph();
  }

  get isLoaded() {
    var _this$state;

    return ((_this$state = this.state) === null || _this$state === void 0 ? void 0 : _this$state.scenegraph) && super.isLoaded;
  }

  _updateScenegraph() {
    const props = this.props;
    const {
      gl
    } = this.context;
    let scenegraphData = null;

    if (props.scenegraph instanceof ScenegraphNode) {
      scenegraphData = {
        scenes: [props.scenegraph]
      };
    } else if (props.scenegraph && !props.scenegraph.gltf) {
      const gltf = props.scenegraph;
      const gltfObjects = createGLTFObjects(gl, gltf, this._getModelOptions());
      scenegraphData = {
        gltf,
        ...gltfObjects
      };
      waitForGLTFAssets(gltfObjects).then(() => this.setNeedsRedraw());
    } else if (props.scenegraph) {
      log.deprecated('ScenegraphLayer.props.scenegraph', 'Use GLTFLoader instead of GLTFScenegraphLoader')();
      scenegraphData = props.scenegraph;
    }

    const options = {
      layer: this,
      gl
    };
    const scenegraph = props.getScene(scenegraphData, options);
    const animator = props.getAnimator(scenegraphData, options);

    if (scenegraph instanceof ScenegraphNode) {
      this._deleteScenegraph();

      this._applyAllAttributes(scenegraph);

      this._applyAnimationsProp(scenegraph, animator, props._animations);

      this.setState({
        scenegraph,
        animator
      });
    } else if (scenegraph !== null) {
      log.warn('invalid scenegraph:', scenegraph)();
    }
  }

  _applyAllAttributes(scenegraph) {
    if (this.state.attributesAvailable) {
      const allAttributes = this.getAttributeManager().getAttributes();
      scenegraph.traverse(model => {
        this._setModelAttributes(model.model, allAttributes);
      });
    }
  }

  _applyAnimationsProp(scenegraph, animator, animationsProp) {
    if (!scenegraph || !animator || !animationsProp) {
      return;
    }

    const animations = animator.getAnimations();
    Object.keys(animationsProp).sort().forEach(key => {
      const value = animationsProp[key];

      if (key === '*') {
        animations.forEach(animation => {
          Object.assign(animation, value);
        });
      } else if (Number.isFinite(Number(key))) {
        const number = Number(key);

        if (number >= 0 && number < animations.length) {
          Object.assign(animations[number], value);
        } else {
          log.warn("animation ".concat(key, " not found"))();
        }
      } else {
        const findResult = animations.find(({
          name
        }) => name === key);

        if (findResult) {
          Object.assign(findResult, value);
        } else {
          log.warn("animation ".concat(key, " not found"))();
        }
      }
    });
  }

  _deleteScenegraph() {
    const {
      scenegraph
    } = this.state;

    if (scenegraph instanceof ScenegraphNode) {
      scenegraph.delete();
    }
  }

  _getModelOptions() {
    const {
      _imageBasedLightingEnvironment
    } = this.props;
    let env = null;

    if (_imageBasedLightingEnvironment) {
      if (typeof _imageBasedLightingEnvironment === 'function') {
        env = _imageBasedLightingEnvironment({
          gl: this.context.gl,
          layer: this
        });
      } else {
        env = _imageBasedLightingEnvironment;
      }
    }

    return {
      gl: this.context.gl,
      waitForFullLoad: true,
      imageBasedLightingEnvironment: env,
      modelOptions: {
        isInstanced: true,
        transpileToGLSL100: !isWebGL2(this.context.gl),
        ...this.getShaders()
      },
      useTangents: false
    };
  }

  updateAttributes(changedAttributes) {
    this.setState({
      attributesAvailable: true
    });
    if (!this.state.scenegraph) return;
    this.state.scenegraph.traverse(model => {
      this._setModelAttributes(model.model, changedAttributes);
    });
  }

  draw({
    moduleParameters = null,
    parameters = {},
    context
  }) {
    if (!this.state.scenegraph) return;

    if (this.props._animations && this.state.animator) {
      this.state.animator.animate(context.timeline.getTime());
      this.setNeedsRedraw();
    }

    const {
      viewport
    } = this.context;
    const {
      sizeScale,
      sizeMinPixels,
      sizeMaxPixels,
      opacity,
      coordinateSystem
    } = this.props;
    const numInstances = this.getNumInstances();
    this.state.scenegraph.traverse((model, {
      worldMatrix
    }) => {
      model.model.setInstanceCount(numInstances);
      model.updateModuleSettings(moduleParameters);
      model.draw({
        parameters,
        uniforms: {
          sizeScale,
          opacity,
          sizeMinPixels,
          sizeMaxPixels,
          composeModelMatrix: shouldComposeModelMatrix(viewport, coordinateSystem),
          sceneModelMatrix: worldMatrix,
          u_Camera: model.model.getUniforms().project_uCameraPosition
        }
      });
    });
  }

}

_defineProperty(ScenegraphLayer, "defaultProps", defaultProps);

_defineProperty(ScenegraphLayer, "layerName", 'ScenegraphLayer');
//# sourceMappingURL=scenegraph-layer.js.map