import _defineProperty from "@babel/runtime/helpers/esm/defineProperty";
import { Layer, project32, phongLighting, picking, log } from '@deck.gl/core';
import { Model, Geometry, Texture2D, isWebGL2 } from '@luma.gl/core';
import { hasFeature, FEATURES } from '@luma.gl/webgl';
import { MATRIX_ATTRIBUTES, shouldComposeModelMatrix } from '../utils/matrix';
import vs from './simple-mesh-layer-vertex.glsl';
import fs from './simple-mesh-layer-fragment.glsl';
import { getMeshBoundingBox } from '@loaders.gl/schema';

function validateGeometryAttributes(attributes, useMeshColors) {
  const hasColorAttribute = attributes.COLOR_0 || attributes.colors;
  const useColorAttribute = hasColorAttribute && useMeshColors;

  if (!useColorAttribute) {
    attributes.colors = {
      constant: true,
      value: new Float32Array([1, 1, 1])
    };
  }

  log.assert(attributes.positions || attributes.POSITION, 'no "postions" or "POSITION" attribute in mesh');
}

function getGeometry(data, useMeshColors) {
  if (data.attributes) {
    validateGeometryAttributes(data.attributes, useMeshColors);

    if (data instanceof Geometry) {
      return data;
    } else {
      return new Geometry(data);
    }
  } else if (data.positions || data.POSITION) {
    validateGeometryAttributes(data, useMeshColors);
    return new Geometry({
      attributes: data
    });
  }

  throw Error('Invalid mesh');
}

const DEFAULT_COLOR = [0, 0, 0, 255];
const defaultProps = {
  mesh: {
    type: 'object',
    value: null,
    async: true
  },
  texture: {
    type: 'image',
    value: null,
    async: true
  },
  sizeScale: {
    type: 'number',
    value: 1,
    min: 0
  },
  _useMeshColors: {
    type: 'boolean',
    value: false
  },
  _instanced: true,
  wireframe: false,
  material: true,
  getPosition: {
    type: 'accessor',
    value: x => x.position
  },
  getColor: {
    type: 'accessor',
    value: DEFAULT_COLOR
  },
  getOrientation: {
    type: 'accessor',
    value: [0, 0, 0]
  },
  getScale: {
    type: 'accessor',
    value: [1, 1, 1]
  },
  getTranslation: {
    type: 'accessor',
    value: [0, 0, 0]
  },
  getTransformMatrix: {
    type: 'accessor',
    value: []
  },
  textureParameters: {
    type: 'object',
    ignore: true
  }
};
export default class SimpleMeshLayer extends Layer {
  constructor(...args) {
    super(...args);

    _defineProperty(this, "state", void 0);
  }

  getShaders() {
    const transpileToGLSL100 = !isWebGL2(this.context.gl);
    const defines = {};

    if (hasFeature(this.context.gl, FEATURES.GLSL_DERIVATIVES)) {
      defines.DERIVATIVES_AVAILABLE = 1;
    }

    return super.getShaders({
      vs,
      fs,
      modules: [project32, phongLighting, picking],
      transpileToGLSL100,
      defines
    });
  }

  getBounds() {
    var _mesh$header;

    if (this.props._instanced) {
      return super.getBounds();
    }

    let result = this.state.positionBounds;

    if (result) {
      return result;
    }

    const {
      mesh
    } = this.props;

    if (!mesh) {
      return null;
    }

    result = (_mesh$header = mesh.header) === null || _mesh$header === void 0 ? void 0 : _mesh$header.boundingBox;

    if (!result) {
      const {
        attributes
      } = getGeometry(mesh, this.props._useMeshColors);
      attributes.POSITION = attributes.POSITION || attributes.positions;
      result = getMeshBoundingBox(attributes);
    }

    this.state.positionBounds = result;
    return result;
  }

  initializeState() {
    const attributeManager = this.getAttributeManager();
    attributeManager.addInstanced({
      instancePositions: {
        transition: true,
        type: 5130,
        fp64: this.use64bitPositions(),
        size: 3,
        accessor: 'getPosition'
      },
      instanceColors: {
        type: 5121,
        transition: true,
        size: this.props.colorFormat.length,
        normalized: true,
        accessor: 'getColor',
        defaultValue: [0, 0, 0, 255]
      },
      instanceModelMatrix: MATRIX_ATTRIBUTES
    });
    this.setState({
      emptyTexture: new Texture2D(this.context.gl, {
        data: new Uint8Array(4),
        width: 1,
        height: 1
      })
    });
  }

  updateState(params) {
    super.updateState(params);
    const {
      props,
      oldProps,
      changeFlags
    } = params;

    if (props.mesh !== oldProps.mesh || changeFlags.extensionsChanged) {
      var _this$state$model;

      this.state.positionBounds = null;
      (_this$state$model = this.state.model) === null || _this$state$model === void 0 ? void 0 : _this$state$model.delete();

      if (props.mesh) {
        this.state.model = this.getModel(props.mesh);
        const attributes = props.mesh.attributes || props.mesh;
        this.setState({
          hasNormals: Boolean(attributes.NORMAL || attributes.normals)
        });
      }

      this.getAttributeManager().invalidateAll();
    }

    if (props.texture !== oldProps.texture) {
      this.setTexture(props.texture);
    }

    if (this.state.model) {
      this.state.model.setDrawMode(this.props.wireframe ? 3 : 4);
    }
  }

  finalizeState(context) {
    super.finalizeState(context);
    this.state.emptyTexture.delete();
  }

  draw({
    uniforms
  }) {
    if (!this.state.model) {
      return;
    }

    const {
      viewport
    } = this.context;
    const {
      sizeScale,
      coordinateSystem,
      _instanced
    } = this.props;
    this.state.model.setUniforms(uniforms).setUniforms({
      sizeScale,
      composeModelMatrix: !_instanced || shouldComposeModelMatrix(viewport, coordinateSystem),
      flatShading: !this.state.hasNormals
    }).draw();
  }

  get isLoaded() {
    var _this$state;

    return ((_this$state = this.state) === null || _this$state === void 0 ? void 0 : _this$state.model) && super.isLoaded;
  }

  getModel(mesh) {
    const model = new Model(this.context.gl, { ...this.getShaders(),
      id: this.props.id,
      geometry: getGeometry(mesh, this.props._useMeshColors),
      isInstanced: true
    });
    const {
      texture
    } = this.props;
    const {
      emptyTexture
    } = this.state;
    model.setUniforms({
      sampler: texture || emptyTexture,
      hasTexture: Boolean(texture)
    });
    return model;
  }

  setTexture(texture) {
    const {
      emptyTexture,
      model
    } = this.state;

    if (model) {
      model.setUniforms({
        sampler: texture || emptyTexture,
        hasTexture: Boolean(texture)
      });
    }
  }

}

_defineProperty(SimpleMeshLayer, "defaultProps", defaultProps);

_defineProperty(SimpleMeshLayer, "layerName", 'SimpleMeshLayer');
//# sourceMappingURL=simple-mesh-layer.js.map