import { createElement, useRef, useState, useMemo, useEffect, useImperativeHandle, forwardRef } from 'react';
import { Deck } from '@deck.gl/core';
import useIsomorphicLayoutEffect from './utils/use-isomorphic-layout-effect';
import extractJSXLayers from './utils/extract-jsx-layers';
import positionChildrenUnderViews from './utils/position-children-under-views';
import extractStyles from './utils/extract-styles';

function getRefHandles(thisRef) {
  return {
    get deck() {
      return thisRef.deck;
    },

    pickObject: opts => thisRef.deck.pickObject(opts),
    pickMultipleObjects: opts => thisRef.deck.pickMultipleObjects(opts),
    pickObjects: opts => thisRef.deck.pickObjects(opts)
  };
}

function redrawDeck(thisRef) {
  if (thisRef.redrawReason) {
    thisRef.deck._drawLayers(thisRef.redrawReason);

    thisRef.redrawReason = null;
  }
}

function createDeckInstance(thisRef, DeckClass, props) {
  const deck = new DeckClass({ ...props,
    _customRender: redrawReason => {
      thisRef.redrawReason = redrawReason;
      const viewports = deck.getViewports();

      if (thisRef.lastRenderedViewports !== viewports) {
        thisRef.forceUpdate();
      } else {
        redrawDeck(thisRef);
      }
    }
  });
  return deck;
}

const DeckGL = forwardRef((props, ref) => {
  const [version, setVersion] = useState(0);

  const _thisRef = useRef({
    control: null,
    version,
    forceUpdate: () => setVersion(v => v + 1)
  });

  const thisRef = _thisRef.current;
  const containerRef = useRef(null);
  const canvasRef = useRef(null);
  const jsxProps = useMemo(() => extractJSXLayers(props), [props.layers, props.views, props.children]);
  let inRender = true;

  const handleViewStateChange = params => {
    var _props$onViewStateCha;

    if (inRender && props.viewState) {
      thisRef.viewStateUpdateRequested = params;
      return null;
    }

    thisRef.viewStateUpdateRequested = null;
    return (_props$onViewStateCha = props.onViewStateChange) === null || _props$onViewStateCha === void 0 ? void 0 : _props$onViewStateCha.call(props, params);
  };

  const handleInteractionStateChange = params => {
    if (inRender) {
      thisRef.interactionStateUpdateRequested = params;
    } else {
      var _props$onInteractionS;

      thisRef.interactionStateUpdateRequested = null;
      (_props$onInteractionS = props.onInteractionStateChange) === null || _props$onInteractionS === void 0 ? void 0 : _props$onInteractionS.call(props, params);
    }
  };

  const deckProps = useMemo(() => {
    const forwardProps = { ...props,
      style: null,
      width: '100%',
      height: '100%',
      parent: containerRef.current,
      canvas: canvasRef.current,
      layers: jsxProps.layers,
      views: jsxProps.views,
      onViewStateChange: handleViewStateChange,
      onInteractionStateChange: handleInteractionStateChange
    };
    delete forwardProps._customRender;

    if (thisRef.deck) {
      thisRef.deck.setProps(forwardProps);
    }

    return forwardProps;
  }, [props]);
  useEffect(() => {
    const DeckClass = props.Deck || Deck;
    thisRef.deck = createDeckInstance(thisRef, DeckClass, { ...deckProps,
      parent: containerRef.current,
      canvas: canvasRef.current
    });
    return () => {
      var _thisRef$deck;

      return (_thisRef$deck = thisRef.deck) === null || _thisRef$deck === void 0 ? void 0 : _thisRef$deck.finalize();
    };
  }, []);
  useIsomorphicLayoutEffect(() => {
    redrawDeck(thisRef);
    const {
      viewStateUpdateRequested,
      interactionStateUpdateRequested
    } = thisRef;

    if (viewStateUpdateRequested) {
      handleViewStateChange(viewStateUpdateRequested);
    }

    if (interactionStateUpdateRequested) {
      handleInteractionStateChange(interactionStateUpdateRequested);
    }
  });
  useImperativeHandle(ref, () => getRefHandles(thisRef), []);
  const currentViewports = thisRef.deck && thisRef.deck.isInitialized ? thisRef.deck.getViewports() : undefined;
  const {
    ContextProvider,
    width,
    height,
    id,
    style
  } = props;
  const {
    containerStyle,
    canvasStyle
  } = useMemo(() => extractStyles({
    width,
    height,
    style
  }), [width, height, style]);

  if (!thisRef.viewStateUpdateRequested && thisRef.lastRenderedViewports === currentViewports || thisRef.version !== version) {
    thisRef.lastRenderedViewports = currentViewports;
    thisRef.version = version;
    const childrenUnderViews = positionChildrenUnderViews({
      children: jsxProps.children,
      deck: thisRef.deck,
      ContextProvider
    });
    const canvas = createElement('canvas', {
      key: 'canvas',
      id: id || 'deckgl-overlay',
      ref: canvasRef,
      style: canvasStyle
    });
    thisRef.control = createElement('div', {
      id: "".concat(id || 'deckgl', "-wrapper"),
      ref: containerRef,
      style: containerStyle
    }, [canvas, childrenUnderViews]);
  }

  inRender = false;
  return thisRef.control;
});
DeckGL.defaultProps = Deck.defaultProps;
export default DeckGL;
//# sourceMappingURL=deckgl.js.map