"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var util = require("../../util");
function makeShader(inputsInfo, outputShape, userCode, broadcast) {
    var inputPrefixSnippet = inputsInfo.map(function (x) { return "uniform sampler2D " + x.name + ";"; }).join('\n');
    var inputSamplingSnippet = inputsInfo.map(function (x) { return getInputSamplingSnippet(x, outputShape, broadcast); })
        .join('\n');
    var outTexShape = outputShape.texShape;
    var outputSamplingSnippet = getOutputSamplingSnippet(outputShape.logicalShape, outTexShape);
    var source = [
        SHADER_PREFIX, inputPrefixSnippet, inputSamplingSnippet,
        outputSamplingSnippet, userCode
    ].join('\n');
    return source;
}
exports.makeShader = makeShader;
function getInputSamplingSnippet(inInfo, outShapeInfo, broadcast) {
    var shape = inInfo.shapeInfo.logicalShape;
    var texShape = inInfo.shapeInfo.texShape;
    var outTexShape = outShapeInfo.texShape;
    var res = '';
    switch (shape.length) {
        case 0:
            res += getSamplerScalar(inInfo.name);
            break;
        case 1:
            res += getSampler1D(inInfo.name, texShape);
            break;
        case 2:
            res += getSampler2D(inInfo.name, shape, texShape);
            break;
        case 3:
            res += getSampler3D(inInfo.name, shape, texShape);
            break;
        case 4:
            res += getSampler4D(inInfo.name, shape, texShape);
            break;
        default:
            throw new Error(shape.length + "-D input sampling" +
                " is not yet supported");
    }
    if (broadcast ||
        util.arraysEqual(inInfo.shapeInfo.logicalShape, outShapeInfo.logicalShape)) {
        res +=
            getSamplerAtOutputCoords(inInfo.name, texShape, outTexShape, broadcast);
    }
    res += getSamplerFlat(inInfo.name, texShape);
    return res;
}
function getOutputSamplingSnippet(outShape, outTexShape) {
    switch (outShape.length) {
        case 0:
            return '';
        case 1:
            return getOutput1DCoords(outShape, outTexShape);
        case 2:
            return getOutput2DCoords(outShape, outTexShape);
        case 3:
            return getOutput3DCoords(outShape, outTexShape);
        case 4:
            return getOutput4DCoords(outShape, outTexShape);
        default:
            throw new Error(outShape.length + "-D output sampling is not yet supported");
    }
}
var SAMPLE_1D_SNIPPET = "\nvec2 UVfrom1D(float texNumR, float texNumC, float index) {\n  float texR = floor(index / texNumC);\n  float texC = mod(index, texNumC);\n  return (vec2(texC, texR) + halfCR) / vec2(texNumC, texNumR);\n}\n";
var SAMPLE_2D_SNIPPET = "\nvec2 UVfrom2D(float texNumR, float texNumC, float numC, float row,\n    float col) {\n  float index = dot(vec2(row, col), vec2(numC, 1.0));\n  float texR = floor(index / texNumC);\n  float texC = mod(index, texNumC);\n  return (vec2(texC, texR) + halfCR) / vec2(texNumC, texNumR);\n}\n";
var SAMPLE_3D_SNIPPET = "\nvec2 UVfrom3D(float texNumR, float texNumC, float stride0,\n    float stride1, float row, float col, float depth) {\n  float index = dot(vec3(row, col, depth), vec3(stride0, stride1, 1.0));\n  float texR = floor(index / texNumC);\n  float texC = mod(index, texNumC);\n  return (vec2(texC, texR) + halfCR) / vec2(texNumC, texNumR);\n}\n";
var SAMPLE_4D_SNIPPET = "\nvec2 UVfrom4D(float texNumR, float texNumC, float stride0,\n    float stride1, float stride2, float row, float col, float depth,\n    float depth2) {\n  float index = dot(vec4(row, col, depth, depth2),\n                    vec4(stride0, stride1, stride2, 1.0));\n  float texR = floor(index / texNumC);\n  float texC = mod(index, texNumC);\n  return (vec2(texC, texR) + halfCR) / vec2(texNumC, texNumR);\n}\n";
var SHADER_PREFIX = "\n  precision highp float;\n  varying vec2 resultUV;\n  const vec2 halfCR = vec2(0.5, 0.5);\n\n  float sample(sampler2D texture, vec2 uv) {\n    return texture2D(texture, uv).r;\n  }\n\n  void setOutput(float val) {\n    gl_FragColor = vec4(val, 0, 0, 0);\n  }\n\n  bool isNaN(float val) {\n    return val == val ? false : true;\n  }\n  " + SAMPLE_1D_SNIPPET + "\n  " + SAMPLE_2D_SNIPPET + "\n  " + SAMPLE_3D_SNIPPET + "\n  " + SAMPLE_4D_SNIPPET + "\n";
function getOutput1DCoords(shape, texShape) {
    if (texShape[0] === 1) {
        return "\n      float getOutputCoords() {\n        return floor(gl_FragCoord.x);\n      }\n    ";
    }
    if (texShape[1] === 1) {
        return "\n      float getOutputCoords() {\n        return floor(gl_FragCoord.y);\n      }\n    ";
    }
    return "\n    float getOutputCoords() {\n      vec2 resTexRC = floor(gl_FragCoord.yx);\n      return dot(resTexRC, vec2(" + texShape[1] + ".0, 1.0));\n    }\n  ";
}
function getOutput3DCoords(shape, texShape) {
    var stride0 = shape[1] * shape[2];
    var stride1 = shape[2];
    return "\n    vec3 getOutputCoords() {\n      vec2 resTexRC = floor(gl_FragCoord.yx);\n      float index = dot(resTexRC, vec2(" + texShape[1] + ".0, 1.0));\n      float r = floor(index / " + stride0 + ".0);\n      index -= r * " + stride0 + ".0;\n      float c = floor(index / " + stride1 + ".0);\n      float d = mod(index, " + stride1 + ".0);\n      return vec3(r, c, d);\n    }\n  ";
}
function getOutput4DCoords(shape, texShape) {
    var stride2 = shape[3];
    var stride1 = shape[2] * stride2;
    var stride0 = shape[1] * stride1;
    return "\n    vec4 getOutputCoords() {\n      vec2 resTexRC = floor(gl_FragCoord.yx);\n      float index = dot(resTexRC, vec2(" + texShape[1] + ".0, 1.0));\n\n      float r = floor(index / " + stride0 + ".0);\n      index -= r * " + stride0 + ".0;\n\n      float c = floor(index / " + stride1 + ".0);\n      index -= c * " + stride1 + ".0;\n\n      float d = floor(index / " + stride2 + ".0);\n      float d2 = mod(index, " + stride2 + ".0);\n\n      return vec4(r, c, d, d2);\n    }\n  ";
}
function getOutput2DCoords(shape, texShape) {
    if (util.arraysEqual(shape, texShape)) {
        return "\n      vec2 getOutputCoords() {\n        return floor(gl_FragCoord.yx);\n      }\n    ";
    }
    return "\n    vec2 getOutputCoords() {\n      vec2 resTexRC = floor(gl_FragCoord.yx);\n      float index = dot(resTexRC, vec2(" + texShape[1] + ".0, 1.0));\n      float r = floor(index / " + shape[1] + ".0);\n      float c = mod(index, " + shape[1] + ".0);\n      return vec2(r, c);\n    }\n  ";
}
function getSamplerScalar(texName) {
    var funcName = 'get' + texName.charAt(0).toUpperCase() + texName.slice(1);
    return "\n    float " + funcName + "() {\n      return sample(" + texName + ", halfCR);\n    }\n  ";
}
function getSampler1D(texName, texShape) {
    var funcName = 'get' + texName.charAt(0).toUpperCase() + texName.slice(1);
    var tR = texShape[0];
    var tC = texShape[1];
    if (texShape[0] === 1 && texShape[1] === 1) {
        return "\n      float " + funcName + "(float index) {\n        return sample(" + texName + ", halfCR);\n      }\n    ";
    }
    if (texShape[1] === 1) {
        return "\n      float " + funcName + "(float index) {\n        vec2 uv = vec2(0.5, (index + 0.5) / " + tR + ".0);\n        return sample(" + texName + ", uv);\n      }\n    ";
    }
    if (texShape[0] === 1) {
        return "\n      float " + funcName + "(float index) {\n        vec2 uv = vec2((index + 0.5) / " + tC + ".0, 0.5);\n        return sample(" + texName + ", uv);\n      }\n    ";
    }
    return "\n    float " + funcName + "(float index) {\n      vec2 uv = UVfrom1D(" + tR + ".0, " + tC + ".0, index);\n      return sample(" + texName + ", uv);\n    }\n  ";
}
function getSampler3D(texName, shape, texShape) {
    var funcName = 'get' + texName.charAt(0).toUpperCase() + texName.slice(1);
    var tR = texShape[0];
    var tC = texShape[1];
    var stride0 = shape[1] * shape[2];
    var stride1 = shape[2];
    if (tC === stride0) {
        return "\n      float " + funcName + "(float row, float col, float depth) {\n        float texR = row;\n        float texC = dot(vec2(col, depth), vec2(" + stride1 + ", 1.0));\n        vec2 uv = (vec2(texC, texR) + halfCR) / vec2(" + tC + ".0, " + tR + ".0);\n        return sample(" + texName + ", uv);\n      }\n    ";
    }
    return "\n    float " + funcName + "(float row, float col, float depth) {\n      vec2 uv = UVfrom3D(" + tR + ".0, " + tC + ".0, " + stride0 + ".0, " + stride1 + ".0, row,\n        col, depth);\n      return sample(" + texName + ", uv);\n    }\n  ";
}
function getSampler4D(texName, shape, texShape) {
    var funcName = 'get' + texName.charAt(0).toUpperCase() + texName.slice(1);
    var tR = texShape[0];
    var tC = texShape[1];
    var stride2 = shape[3];
    var stride1 = shape[2] * stride2;
    var stride0 = shape[1] * stride1;
    if (tC === stride0) {
        return "\n      float " + funcName + "(float row, float col, float depth, float depth2) {\n        float texR = row;\n        float texC = dot(vec3(col, depth, depth2),\n                         vec3(" + stride1 + ".0, " + stride2 + ".0, 1.0));\n        vec2 uv = (vec2(texC, texR) + halfCR) / vec2(" + tC + ".0, " + tR + ".0);\n        return sample(" + texName + ", uv);\n      }\n    ";
    }
    return "\n    float " + funcName + "(float row, float col, float depth, float depth2) {\n      vec2 uv = UVfrom4D(" + tR + ".0, " + tC + ".0, " + stride0 + ".0, " + stride1 + ".0,\n          " + stride2 + ".0, row, col, depth, depth2);\n      return sample(" + texName + ", uv);\n    }\n  ";
}
function getSampler2D(texName, shape, texShape) {
    var funcName = 'get' + texName.charAt(0).toUpperCase() + texName.slice(1);
    var tR = texShape[0];
    var tC = texShape[1];
    if (util.arraysEqual(shape, texShape)) {
        return "\n      float " + funcName + "(float row, float col) {\n        vec2 uv = (vec2(col, row) + halfCR) / vec2(" + tC + ".0, " + tR + ".0);\n        return sample(" + texName + ", uv);\n      }\n    ";
    }
    if (tC === 1) {
        return "\n      float " + funcName + "(float row, float col) {\n        float index = dot(vec2(row, col), vec2(" + shape[1] + ".0, 1.0));\n        vec2 uv = vec2(0.5, (index + 0.5) / " + tR + ".0);\n        return sample(" + texName + ", uv);\n      }\n    ";
    }
    if (tR === 1) {
        return "\n      float " + funcName + "(float row, float col) {\n        float index = dot(vec2(row, col), vec2(" + shape[1] + ".0, 1.0));\n        vec2 uv = vec2((index + 0.5) / " + tC + ".0, 0.5);\n        return sample(" + texName + ", uv);\n      }\n    ";
    }
    return "\n    float " + funcName + "(float row, float col) {\n      vec2 uv = UVfrom2D(" + tR + ".0, " + tC + ".0, " + shape[1] + ".0, row, col);\n      return sample(" + texName + ", uv);\n    }\n  ";
}
function getSamplerFlat(texName, texShape) {
    var funcName = 'get' + texName.charAt(0).toUpperCase() + texName.slice(1) + 'Flat';
    var tNumR = texShape[0];
    var tNumC = texShape[1];
    if (tNumC === 1 && tNumR === 1) {
        return "\n      float " + funcName + "(float index) {\n        return sample(" + texName + ", halfCR);\n      }\n    ";
    }
    if (tNumC === 1) {
        return "\n      float " + funcName + "(float index) {\n        vec2 uv = vec2(0.5, (index + 0.5) / " + tNumR + ".0);\n        return sample(" + texName + ", uv);\n      }\n    ";
    }
    if (tNumR === 1) {
        return "\n      float " + funcName + "(float index) {\n        vec2 uv = vec2((index + 0.5) / " + tNumC + ".0, 0.5);\n        return sample(" + texName + ", uv);\n      }\n    ";
    }
    return "\n    float " + funcName + "(float index) {\n      float texR = floor(index / " + tNumC + ".0);\n      float texC = mod(index, " + tNumC + ".0);\n      vec2 uv = (vec2(texC, texR) + halfCR) / vec2(" + tNumC + ".0, " + tNumR + ".0);\n      return sample(" + texName + ", uv);\n    }\n  ";
}
function getSamplerAtOutputCoords(texName, inTexShape, outTexShape, broadcast) {
    var funcName = 'get' + texName.charAt(0).toUpperCase() + texName.slice(1) +
        'AtOutCoords';
    if (util.arraysEqual(inTexShape, outTexShape)) {
        return "\n      float " + funcName + "() {\n        return sample(" + texName + ", resultUV);\n      }\n    ";
    }
    var inSize = util.sizeFromShape(inTexShape);
    var broadcastSnippet = broadcast ? "index = mod(index, " + inSize + ".0);" : '';
    return "\n    float " + funcName + "() {\n      vec2 resTexRC = floor(gl_FragCoord.yx);\n      float index = dot(resTexRC, vec2(" + outTexShape[1] + ".0, 1.0));\n      " + broadcastSnippet + "\n      float texR = floor(index / " + inTexShape[1] + ".0);\n      float texC = mod(index, " + inTexShape[1] + ".0);\n      vec2 uv = (vec2(texC, texR) + halfCR) /\n                 vec2(" + inTexShape[1] + ".0, " + inTexShape[0] + ".0);\n      return sample(" + texName + ", uv);\n    }\n  ";
}
//# sourceMappingURL=shader_compiler.js.map