'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.renderString = renderString;

var _element = require('../element');

/**
 * Turn an object of key/value pairs into a HTML attribute string. This
 * function is responsible for what attributes are allowed to be rendered and
 * should handle any other special cases specific to deku.
 */

function attributesToString(attributes) {
  var str = '';
  for (var name in attributes) {
    var value = attributes[name];
    if (name === 'innerHTML') continue;
    if ((0, _element.isValidAttribute)(value)) str += ' ' + name + '="' + attributes[name] + '"';
  }
  return str;
}

/**
 * Render a virtual element to a string. You can pass in an option state context
 * object that will be given to all components.
 */

function renderString(element, context) {
  var path = arguments.length <= 2 || arguments[2] === undefined ? '0' : arguments[2];

  if ((0, _element.isText)(element)) {
    return element.nodeValue;
  }

  if ((0, _element.isEmpty)(element)) {
    return '<noscript></noscript>';
  }

  if ((0, _element.isThunk)(element)) {
    var props = element.props;
    var component = element.component;
    var _children = element.children;
    var render = component.render;

    var output = render({
      children: _children,
      props: props,
      path: path,
      context: context
    });
    return renderString(output, context, path);
  }

  var attributes = element.attributes;
  var type = element.type;
  var children = element.children;

  var innerHTML = attributes.innerHTML;
  var str = '<' + type + attributesToString(attributes) + '>';

  if (innerHTML) {
    str += innerHTML;
  } else {
    str += children.map(function (child, i) {
      return renderString(child, context, path + '.' + (child.key == null ? i : child.key));
    }).join('');
  }

  str += '</' + type + '>';
  return str;
}