module.exports = DocumentLayout;

/**
 * Translate page layouts, as generated by page-layouts, into an
 * object which computes layout information for the document as
 * a whole.
 */
function DocumentLayout(config, zoomLevel)
{
    var computedLayout = getComputedLayout(config, zoomLevel);

    this.dimensions = computedLayout.dimensions;
    this.pageGroups = computedLayout.pageGroups;
    this._pageLookup = getPageLookup(computedLayout.pageGroups);
}

/**
 * @typedef {Object} PageInfo
 * @property {number} index
 * @property {{index, dimensions, pages, region, padding}} group
 * @property {{height: number, width: number}} dimensions
 * @property {{top: number, left: number}} groupOffset
 */

/**
 * @param pageIndex
 * @returns {PageInfo|null}
 */
DocumentLayout.prototype.getPageInfo = function (pageIndex)
{
    return this._pageLookup[pageIndex] || null;
};

/**
 * Get the dimensions of a page
 *
 * @param pageIndex
 * @returns {{height: number, width: number}}
 */
DocumentLayout.prototype.getPageDimensions = function (pageIndex)
{
    if (!this._pageLookup || !this._pageLookup[pageIndex])
        return null;

    var region = getPageRegionFromPageInfo(this._pageLookup[pageIndex]);

    return {
        height: region.bottom - region.top,
        width: region.right - region.left
    };
};

// TODO(wabain): Get rid of this; it's a subset of the page region, so
// give that instead
/**
 * Get the top-left coordinates of a page, including*** padding
 *
 * @param pageIndex
 * @param options
 * @returns {{top: number, left: number} | null}
 */
DocumentLayout.prototype.getPageOffset = function (pageIndex, options)
{
    var region = this.getPageRegion(pageIndex, options);

    if (!region)
        return null;

    return {
        top: region.top,
        left: region.left
    };
};

DocumentLayout.prototype.getPageRegion = function (pageIndex, options)
{
    var pageInfo = this._pageLookup[pageIndex];

    if (!pageInfo)
        return null;

    var region = getPageRegionFromPageInfo(pageInfo);

    if (options && options.excludePadding)
    {
        // FIXME?
        var padding = pageInfo.group.padding;

        return {
            top: region.top + padding.top,
            left: region.left + padding.left,
            bottom: region.bottom,
            right: region.right
        };
    }

    return region;
};

/**
 * Get the distance from the top-right of the page to the center of the
 * specified viewport region
 *
 * @param pageIndex
 * @param viewport {{top: number, left: number, bottom: number, right: number}}
 * @returns {{x: number, y: number}}
 */
DocumentLayout.prototype.getPageToViewportCenterOffset = function (pageIndex, viewport)
{
    var scrollLeft = viewport.left;
    var elementWidth = viewport.right - viewport.left;

    var offset = this.getPageOffset(pageIndex);

    var x = scrollLeft - offset.left + parseInt(elementWidth / 2, 10);

    var scrollTop = viewport.top;
    var elementHeight = viewport.bottom - viewport.top;

    var y = scrollTop - offset.top + parseInt(elementHeight / 2, 10);

    return {
        x: x,
        y: y
    };
};

function getPageRegionFromPageInfo(page)
{
    var top    = page.groupOffset.top  + page.group.region.top;
    var bottom = top + page.dimensions.height;
    var left   = page.groupOffset.left + page.group.region.left;
    var right  = left + page.dimensions.width;

    return {
        top: top,
        bottom: bottom,
        left: left,
        right: right
    };
}

function getPageLookup(pageGroups)
{
    var pageLookup = {};

    pageGroups.forEach(function (group)
    {
        group.pages.forEach(function (page)
        {
            pageLookup[page.index] = {
                index: page.index,
                group: group,
                dimensions: page.dimensions,
                groupOffset: page.groupOffset
            };
        });
    });

    return pageLookup;
}

function getComputedLayout(config, zoomLevel)
{
    var scaledLayouts = zoomLevel === null ? config.pageLayouts : getScaledPageLayouts(config, zoomLevel);

    var documentSecondaryExtent = getExtentAlongSecondaryAxis(config, scaledLayouts);

    // The current position in the document along the primary axis
    var primaryDocPosition = config.verticallyOriented ?
        config.padding.document.top :
        config.padding.document.left;

    var pageGroups = [];

    // TODO: Use bottom, right as well
    var pagePadding = {
        top: config.padding.page.top,
        left: config.padding.page.left
    };

    scaledLayouts.forEach(function (layout, index)
    {
        var top, left;

        if (config.verticallyOriented)
        {
            top = primaryDocPosition;
            left = (documentSecondaryExtent - layout.dimensions.width) / 2;
        }
        else
        {
            top = (documentSecondaryExtent - layout.dimensions.height) / 2;
            left = primaryDocPosition;
        }

        var region = {
            top: top,
            bottom: top + pagePadding.top + layout.dimensions.height,
            left: left,
            right: left + pagePadding.left + layout.dimensions.width
        };

        pageGroups.push({
            index: index,
            dimensions: layout.dimensions,
            pages: layout.pages,
            region: region,
            padding: pagePadding
        });

        primaryDocPosition = config.verticallyOriented ? region.bottom : region.right;
    });

    var height, width;

    if (config.verticallyOriented)
    {
        height = primaryDocPosition + pagePadding.top;
        width = documentSecondaryExtent;
    }
    else
    {
        height = documentSecondaryExtent;
        width = primaryDocPosition + pagePadding.left;
    }

    return {
        dimensions: {
            height: height,
            width: width
        },
        pageGroups: pageGroups
    };
}

function getScaledPageLayouts(config, zoomLevel)
{
    var scaleRatio = Math.pow(2, zoomLevel - config.maxZoomLevel);

    return config.pageLayouts.map(function (group)
    {
        return {
            dimensions: scaleDimensions(group.dimensions, scaleRatio),
            pages: group.pages.map(function (page)
            {
                return {
                    index: page.index,
                    groupOffset: {
                        top: Math.floor(page.groupOffset.top * scaleRatio),
                        left: Math.floor(page.groupOffset.left * scaleRatio)
                    },
                    dimensions: scaleDimensions(page.dimensions, scaleRatio)
                };
            })
        };
    });
}

function scaleDimensions(dimensions, scaleRatio)
{
    return {
        height: Math.floor(dimensions.height * scaleRatio),
        width: Math.floor(dimensions.width * scaleRatio)
    };
}

function getExtentAlongSecondaryAxis(config, scaledLayouts)
{
    // Get the extent of the document along the secondary axis
    var secondaryDim, secondaryPadding;
    var docPadding = config.padding.document;

    if (config.verticallyOriented)
    {
        secondaryDim = 'width';
        secondaryPadding = docPadding.left + docPadding.right;
    }
    else
    {
        secondaryDim = 'height';
        secondaryPadding = docPadding.top + docPadding.bottom;
    }

    return secondaryPadding + scaledLayouts.reduce(function (maxDim, layout)
    {
        return Math.max(layout.dimensions[secondaryDim], maxDim);
    }, 0);
}
