import './utils/vanilla.kinetic';
import './utils/dragscroll';
import { elt } from "./utils/elt";
import {
    DivaParentElementNotFoundException,
    NotAnIIIFManifestException,
    ObjectDataNotSuppliedException
} from "./exceptions";
import diva from "./diva-global";
import ViewerCore from "./viewer-core";
import ImageManifest from "./image-manifest";
import Toolbar from "./toolbar";
import HashParams from "./utils/hash-params";


/**
 * The top-level class for Diva objects. This is instantiated by passing in an HTML element
 * ID or HTML Element node and an object containing a list of options, of which the 'objectData'
 * option is required and which must point to a IIIF Presentation API Manifest:
 *
 * var diva = new Diva('element-id', {
 *     objectData: "http://example.com/iiif-manifest.json"
 * });
 *
 * This class also serves as the entry point for the Events system, in which applications can subscribe
 * to notifications sent from Diva instances:
 *
 * Diva.Events.subscribe('VisiblePageDidChange', function () { console.log("Visible Page Changed"); });
 *
 *
 *
 **/
class Diva
{
    constructor (element, options)
    {
        /*
         * If a string is passed in, convert that to an element.
         * */
        if (!(element instanceof HTMLElement))
        {
            this.element = document.getElementById(element);

            if (this.element === null)
            {
                throw new DivaParentElementNotFoundException();
            }
        }

        if (!options.objectData)
        {
            throw new ObjectDataNotSuppliedException('You must supply either a URL or a literal object to the `objectData` key.');
        }

        this.options = Object.assign({
            acceptHeader: "application/json", // The header to send off to the server in content negotiation
            adaptivePadding: 0.05,      // The ratio of padding to the page dimension
            arrowScrollAmount: 40,      // The amount (in pixels) to scroll by when using arrow keys
            blockMobileMove: false,     // Prevent moving or scrolling the page on mobile devices
            objectData: '',             // A IIIF Manifest or a JSON file generated by process.py that provides the object dimension data, or a URL pointing to such data - *REQUIRED*
            enableAutoTitle: true,      // Shows the title within a div of id diva-title
            enableFilename: true,       // Uses filenames and not page numbers for links (i=bm_001.tif, not p=1)
            enableFullscreen: true,     // Enable or disable fullscreen icon (mode still available)
            enableGotoPage: true,       // A "go to page" jump box
            enableGotoSuggestions: true, // Controls whether suggestions are shown under the input field when the user is typing in the 'go to page' form
            enableGridIcon: true,       // A grid view of all the pages
            enableGridControls: 'buttons',  // Specify control of pages per grid row in Grid view. Possible values: 'buttons' (+/-), 'slider'. Any other value disables the controls.
            enableImageTitles: true,    // Adds "Page {n}" title to page images if true
            enableIndexAsLabel: false,	// Use index numbers instead of page labels in the page n-m display.
            enableKeyScroll: true,      // Captures scrolling using the arrow and page up/down keys regardless of page focus. When off, defers to default browser scrolling behavior.
            enableLinkIcon: true,       // Controls the visibility of the link icon
            enableNonPagedVisibilityIcon: true, // Controls the visibility of the icon to toggle the visibility of non-paged pages. (Automatically hidden if no 'non-paged' pages).
            enableSpaceScroll: false,   // Scrolling down by pressing the space key
            enableToolbar: true,        // Enables the toolbar. Note that disabling this means you have to handle all controls yourself.
            enableZoomControls: 'buttons', // Specify controls for zooming in and out. Possible values: 'buttons' (+/-), 'slider'. Any other value disables the controls.
            fillParentHeight: true,     // Use a flexbox layout to allow Diva to fill its parent's height
            fixedPadding: 10,           // Fallback if adaptive padding is set to 0
            fixedHeightGrid: true,      // So each page in grid view has the same height (only widths differ)
            goDirectlyTo: 0,            // Default initial page to show (0-indexed)
            hashParamSuffix: null,      // Used when there are multiple document viewers on a page
            inFullscreen: false,        // Set to true to load fullscreen mode initially
            inBookLayout: false,       // Set to true to view the document with facing pages in document mode
            inGrid: false,              // Set to true to load grid view initially
            maxPagesPerRow: 8,          // Maximum number of pages per row in grid view
            maxZoomLevel: -1,           // Optional; defaults to the max zoom returned in the JSON response
            minPagesPerRow: 2,          // Minimum pages per row in grid view. Recommended default.
            minZoomLevel: 0,            // Defaults to 0 (the minimum zoom)
            onGotoSubmit: null,         // When set to a function that takes a string and returns a page index, this will override the default behaviour of the 'go to page' form submission
            pageAliases: {},            // An object mapping specific page indices to aliases (has priority over 'pageAliasFunction'
            pageAliasFunction: function(){return false;},  // A function mapping page indices to an alias. If false is returned, default page number is displayed
            pageLoadTimeout: 200,       // Number of milliseconds to wait before loading pages
            pagesPerRow: 5,             // The default number of pages per row in grid view
            showNonPagedPages: false,   // Whether pages tagged as 'non-paged' (in IIIF manifests only) should be visible after initial load
            throbberTimeout: 100,       // Number of milliseconds to wait before showing throbber
            tileHeight: 256,            // The height of each tile, in pixels; usually 256
            tileWidth: 256,             // The width of each tile, in pixels; usually 256
            toolbarParentObject: null,  // The toolbar parent object.
            verticallyOriented: true,   // Determines vertical vs. horizontal orientation
            viewportMargin: 200,        // Pretend tiles +/- 200px away from viewport are in
            zoomLevel: 2                // The initial zoom level (used to store the current zoom level)
        }, options);

        // In order to fill the height, use a wrapper div displayed using a flexbox layout
        const wrapperElement = elt('div', {
            class: `diva-wrapper${this.options.fillParentHeight ? " diva-wrapper-flexbox" : ""}`
        });

        this.element.appendChild(wrapperElement);

        this.options.toolbarParentObject = this.options.toolbarParentObject || wrapperElement;

        const viewerCore = new ViewerCore(wrapperElement, this.options, this);

        this.viewerState = viewerCore.getInternalState();
        this.settings = viewerCore.getSettings();
        this.toolbar = this.settings.enableToolbar ? new Toolbar(this) : null;

        wrapperElement.id = this.settings.ID + 'wrapper';

        this.divaState = {
            viewerCore: viewerCore,
            toolbar: this.toolbar
        };

        // only render the toolbar after the object has been loaded
        let handle = diva.Events.subscribe('ObjectDidLoad', () =>
        {
            if (this.toolbar !== null)
            {
                this.toolbar.render();
            }

            diva.Events.unsubscribe(handle);
        });

        this.hashState = this._getHashParamState();

        this._loadOrFetchObjectData();
    }

    /**
     * @private
     **/
    _loadOrFetchObjectData ()
    {
        if (typeof this.settings.objectData === 'object')
        {
            // Defer execution until initialization has completed
            setTimeout(() =>
            {
                this._loadObjectData(this.settings.objectData, this.hashState);
            }, 0);
        }
        else
        {
            const pendingManifestRequest = fetch(this.settings.objectData, {
                headers: {
                    "Accept": this.settings.acceptHeader
                }
            }).then( (response) =>
            {
                if (!response.ok)
                {
                    this._ajaxError(response);

                    let error = new Error(response.statusText);
                    error.response = response;
                    throw error;
                }
                return response.json();

            }).then( (data) =>
            {
                this._loadObjectData(data, this.hashState);
            });

            // Store the pending request so that it can be cancelled in the event that Diva needs to be destroyed
            this.divaState.viewerCore.setPendingManifestRequest(pendingManifestRequest);
        }
    }

    /**
     * @private
     **/
    _showError (message)
    {
        this.divaState.viewerCore.showError(message);
    }

    /**
     * @private
     * */
    _ajaxError (response)
    {
        // Show a basic error message within the document viewer pane
        const errorMessage = ['Invalid objectData setting. Error code: ' + response.status + ' ' + response.statusText];

        // Detect and handle CORS errors
        const dataHasAbsolutePath = this.settings.objectData.lastIndexOf('http', 0) === 0;

        if (dataHasAbsolutePath)
        {
            const jsonHost = this.settings.objectData.replace(/https?:\/\//i, "").split(/[/?#]/)[0];

            if (window.location.hostname !== jsonHost)
            {
                errorMessage.push(
                    elt('p', 'Attempted to access cross-origin data without CORS.'),
                    elt('p',
                        'You may need to update your server configuration to support CORS. For help, see the ',
                        elt('a', {
                            href: 'https://github.com/DDMAL/diva.js/wiki/Installation#a-note-about-cross-site-requests',
                            target: '_blank'
                        }, 'cross-site request documentation.')
                    )
                );
            }
        }

        this._showError(errorMessage);
    }

    /**
     * @private
     **/
    _loadObjectData (responseData, hashState)
    {
        let manifest;

        // TODO improve IIIF detection method
        if (!responseData.hasOwnProperty('@context') && (responseData['@context'].indexOf('iiif') === -1 || responseData['@context'].indexOf('shared-canvas') === -1))
        {
            throw new NotAnIIIFManifestException('This does not appear to be a IIIF Manifest.');
        }

        // trigger ManifestDidLoad event
        diva.Events.publish('ManifestDidLoad', [responseData], this);
        manifest = ImageManifest.fromIIIF(responseData);
        const loadOptions = hashState ? this._getLoadOptionsForState(hashState, manifest) : {};

        this.divaState.viewerCore.setManifest(manifest, loadOptions);
    }

    /**
     * Parse the hash parameters into the format used by getState and setState
     *
     * @private
     **/
    _getHashParamState ()
    {
        const state = {};

        ['f', 'v', 'z', 'n', 'i', 'p', 'y', 'x'].forEach( (param) =>
        {
            const value = HashParams.get(param + this.settings.hashParamSuffix);

            // `false` is returned if the value is missing
            if (value !== false)
                state[param] = value;
        });

        // Do some awkward special-casing, since this format is kind of weird.

        // For inFullscreen (f), true and false strings should be interpreted
        // as booleans.
        if (state.f === 'true')
            state.f = true;
        else if (state.f === 'false')
            state.f = false;

        // Convert numerical values to integers, if provided
        ['z', 'n', 'p', 'x', 'y'].forEach( (param) =>
        {
            if (param in state)
                state[param] = parseInt(state[param], 10);
        });

        return state;
    }

    /**
     * @private
     **/
    _getLoadOptionsForState (state, manifest)
    {
        manifest = manifest || this.settings.manifest;

        const options = ('v' in state) ? this._getViewState(state.v) : {};

        if ('f' in state)
            options.inFullscreen = state.f;

        if ('z' in state)
            options.zoomLevel = state.z;

        if ('n' in state)
            options.pagesPerRow = state.n;

        // Only change specify the page if state.i or state.p is valid
        let pageIndex = this._getPageIndexForManifest(manifest, state.i);

        if (!(pageIndex >= 0 && pageIndex < manifest.pages.length))
        {
            pageIndex = state.p - 1;

            // Possibly NaN
            if (!(pageIndex >= 0 && pageIndex < manifest.pages.length))
                pageIndex = null;
        }

        if (pageIndex !== null)
        {
            const horizontalOffset = parseInt(state.x, 10);
            const verticalOffset = parseInt(state.y, 10);

            options.goDirectlyTo = pageIndex;
            options.horizontalOffset = horizontalOffset;
            options.verticalOffset = verticalOffset;
        }

        return options;
    }

    /**
     * @private
     * */
    _getViewState (view)
    {
        switch (view)
        {
            case 'd':
                return {
                    inGrid: false,
                    inBookLayout: false
                };

            case 'b':
                return {
                    inGrid: false,
                    inBookLayout: true
                };

            case 'g':
                return {
                    inGrid: true,
                    inBookLayout: false
                };

            default:
                return {};
        }
    }

    /**
     * @private
     * */
    _getPageIndexForManifest (manifest, filename)
    {
        let i;
        const np = manifest.pages.length;

        for (i = 0; i < np; i++)
        {
            if (manifest.pages[i].f === filename)
            {
                return i;
            }
        }

        return -1;
    }

    /**
     * @private
     * */
    _getState ()
    {
        let view;

        if (this.settings.inGrid)
        {
            view = 'g';
        }
        else if (this.settings.inBookLayout)
        {
            view = 'b';
        }
        else
        {
            view = 'd';
        }

        const layout = this.divaState.viewerCore.getCurrentLayout();
        const pageOffset = layout.getPageToViewportCenterOffset(this.settings.activePageIndex, this.viewerState.viewport);

        return {
            'f': this.settings.inFullscreen,
            'v': view,
            'z': this.settings.zoomLevel,
            'n': this.settings.pagesPerRow,
            'i': this.settings.enableFilename ? this.settings.manifest.pages[this.settings.activePageIndex].f : false,
            'p': this.settings.enableFilename ? false : this.settings.activePageIndex + 1,
            'y': pageOffset ? pageOffset.y : false,
            'x': pageOffset ? pageOffset.x : false
        };
    }

    /**
     * @private
     **/
    _getURLHash ()
    {
        const hashParams = this._getState();
        const hashStringBuilder = [];
        let param;

        for (param in hashParams)
        {
            if (hashParams[param] !== false)
                hashStringBuilder.push(param + this.settings.hashParamSuffix + '=' + encodeURIComponent(hashParams[param]));
        }

        return hashStringBuilder.join('&');
    }

    /**
     * Returns the page index associated with the given filename; must called after setting settings.manifest
     *
     * @private
     **/
    _getPageIndex (filename)
    {
        return this._getPageIndexForManifest(this.settings.manifest, filename);
    }

    /**
     * @private
     * */
    _checkLoaded ()
    {
        if (!this.viewerState.loaded)
        {
            console.warn("The viewer is not completely initialized. This is likely because it is still downloading data. To fix this, only call this function if the isReady() method returns true.");
            return false;
        }
        return true;
    }

    /**
     * Called when the fullscreen icon is clicked
     *
     * @private
     **/
    _toggleFullscreen ()
    {
        this._reloadViewer({
            inFullscreen: !this.settings.inFullscreen
        });

        // handle toolbar opacity in fullscreen
        let t;
        let hover = false;
        let tools = document.getElementById(this.settings.selector + 'tools');
        const TIMEOUT = 2000;

        if (this.settings.inFullscreen) 
        {
            tools.classList.add("diva-fullscreen-tools");

            document.addEventListener('mousemove', toggleOpacity.bind(this));
            document.getElementsByClassName('diva-viewport')[0].addEventListener('scroll', toggleOpacity.bind(this));
            tools.addEventListener('mouseenter', function () {
                hover = true;
            });
            tools.addEventListener('mouseleave', function () {
                hover = false;
            });
        }
        else
        {
            tools.classList.remove("diva-fullscreen-tools");
        }

        function toggleOpacity () 
        {
            tools.style.opacity = 1;
            clearTimeout(t);
            if (!hover && this.settings.inFullscreen) {
                t = setTimeout(function () 
                {
                    tools.style.opacity = 0;
                }, TIMEOUT);
            }
        }
    }

    /**
     * Toggles between orientations
     *
     * @private
     * */
    _togglePageLayoutOrientation ()
    {
        const verticallyOriented = !this.settings.verticallyOriented;

        //if in grid, switch out of grid
        this._reloadViewer({
            inGrid: false,
            verticallyOriented: verticallyOriented,
            goDirectlyTo: this.settings.activePageIndex,
            verticalOffset: this.divaState.viewerCore.getYOffset(),
            horizontalOffset: this.divaState.viewerCore.getXOffset()
        });

        return verticallyOriented;
    }

    /**
     * Called when the change view icon is clicked
     *
     * @private
     **/
    _changeView (destinationView)
    {
        switch (destinationView)
        {
            case 'document':
                return this._reloadViewer({
                    inGrid: false,
                    inBookLayout: false
                });

            case 'book':
                return this._reloadViewer({
                    inGrid: false,
                    inBookLayout: true
                });

            case 'grid':
                return this._reloadViewer({
                    inGrid: true
                });

            default:
                return false;
        }
    }

    /**
     * @private
     *
     * @param {Number} pageIndex - 0-based page index.
     * @param {Number} xAnchor - x coordinate to jump to on resulting page.
     * @param {Number} yAnchor - y coordinate to jump to on resulting page.
     * @returns {Boolean} - Whether the jump was successful.
     **/
    _gotoPageByIndex (pageIndex, xAnchor, yAnchor)
    {
        let pidx = parseInt(pageIndex, 10);

        if (this._isPageIndexValid(pidx))
        {
            const xOffset = this.divaState.viewerCore.getXOffset(pidx, xAnchor);
            const yOffset = this.divaState.viewerCore.getYOffset(pidx, yAnchor);

            this.viewerState.renderer.goto(pidx, yOffset, xOffset);
            return true;
        }

        return false;
    }

    /**
     * Check if a page index is valid
     *
     * @private
     * @param {Number} pageIndex - Numeric (0-based) page index
     * @return {Boolean} whether the page index is valid or not.
     */
    _isPageIndexValid (pageIndex)
    {
        return this.settings.manifest.isPageValid(pageIndex, this.settings.showNonPagedPages);
    }

    /**
     * Given a pageX and pageY value, returns either the page visible at that (x,y)
     * position or -1 if no page is.
     *
     * @private
     */
    _getPageIndexForPageXYValues (pageX, pageY)
    {
        //get the four edges of the outer element
        const outerOffset = this.viewerState.outerElement.getBoundingClientRect();
        const outerTop = outerOffset.top;
        const outerLeft = outerOffset.left;
        const outerBottom = outerOffset.bottom;
        const outerRight = outerOffset.right;

        //if the clicked position was outside the diva-outer object, it was not on a visible portion of a page
        if (pageX < outerLeft || pageX > outerRight)
            return -1;

        if (pageY < outerTop || pageY > outerBottom)
            return -1;

        //navigate through all diva page objects
        const pages = document.getElementsByClassName('diva-page');
        let curPageIdx = pages.length;
        while (curPageIdx--)
        {
            //get the offset for each page
            const curPage = pages[curPageIdx];
            const curOffset = curPage.getBoundingClientRect();

            //if this point is outside the horizontal boundaries of the page, continue
            if (pageX < curOffset.left || pageX > curOffset.right)
                continue;

            //same with vertical boundaries
            if (pageY < curOffset.top || pageY > curOffset.bottom)
                continue;

            //if we made it through the above two, we found the page we're looking for
            return curPage.getAttribute('data-index');
        }

        //if we made it through that entire while loop, we didn't click on a page
        return -1;
    }

    /**
     * @private
     **/
    _reloadViewer (newOptions)
    {
        return this.divaState.viewerCore.reload(newOptions);
    }

    /**
     * @private
     */
    _getCurrentURL ()
    {
        return location.protocol + '//' + location.host + location.pathname + location.search + '#' + this._getURLHash();
    }

    /**
     * ===============================================
     *                PUBLIC FUNCTIONS
     * ===============================================
     **/

    /**
     *  Activate this instance of diva via the active Diva controller.
     *
     *  @public
     */
    activate ()
    {
        this.viewerState.isActiveDiva = true;
    }

    /**
     * Change the object (objectData) parameter currently being rendered by Diva.
     *
     * @public
     * @params {object} objectData - An IIIF Manifest object OR a URL to a IIIF manifest.
     */
    changeObject (objectData)
    {
        this.viewerState.loaded = false;
        this.divaState.viewerCore.clear();

        if (this.viewerState.renderer)
            this.viewerState.renderer.destroy();

        this.viewerState.options.objectData = objectData;

        this._loadOrFetchObjectData();
    }

    /**
     * Change views. Takes 'document', 'book', or 'grid' to specify which view to switch into
     *
     * @public
     * @params {string} destinationView - the destination view to change to.
     */
    changeView (destinationView)
    {
        this._changeView(destinationView);
    }

    /**
     *  Deactivate this diva instance through the active Diva controller.
     *
     *  @public
     **/
    deactivate ()
    {
        this.viewerState.isActiveDiva = false;
    }

    /**
     * Destroys this instance, tells plugins to do the same
     *
     * @public
     **/
    destroy ()
    {
        this.divaState.viewerCore.destroy();
    }

    /**
     * Disables document dragging, scrolling (by keyboard if set), and zooming by double-clicking
     *
     * @public
     **/
    disableScrollable ()
    {
        this.divaState.viewerCore.disableScrollable();
    }

    /**
     * Re-enables document dragging, scrolling (by keyboard if set), and zooming by double-clicking
     *
     * @public
     **/
    enableScrollable ()
    {
        this.divaState.viewerCore.enableScrollable();
    }

    /**
     * Disables document drag scrolling
     *
     * @public
     */
    disableDragScrollable ()
    {
        this.divaState.viewerCore.disableDragScrollable();
    }
    
    /**
     * Enables document drag scrolling
     *
     * @public
     */
    enableDragScrollable ()
    {
        this.divaState.viewerCore.enableDragScrollable();
    }

    /**
     * Enter fullscreen mode if currently not in fullscreen mode. If currently in fullscreen
     * mode this will have no effect.
     *
     * This function will work even if enableFullscreen is set to false in the options.
     *
     * @public
     * @returns {boolean} - Whether the switch to fullscreen was successful or not.
     **/
    enterFullscreenMode ()
    {
        if (!this.settings.inFullscreen)
        {
            this._toggleFullscreen();
            return true;
        }

        return false;
    }

    /**
     * Enter grid view if currently not in grid view. If currently in grid view mode
     * this will have no effect.
     *
     * @public
     * @returns {boolean} - Whether the switch to grid view was successful or not.
     **/
    enterGridView ()
    {
        if (!this.settings.inGrid)
        {
            this._changeView('grid');
            return true;
        }

        return false;
    }

    /**
     * Returns an array of all page image URIs in the document.
     *
     * @public
     * @returns {Array} - An array of all the URIs in the document.
     * */
    getAllPageURIs ()
    {
        return this.settings.manifest.pages.map( (pg) =>
        {
            return pg.f;
        });
    }

    /**
     * Get the canvas identifier for the currently visible page.
     *
     * @public
     * @returns {string} - The URI of the currently visible canvas.
     **/
    getCurrentCanvas ()
    {
        return this.settings.manifest.pages[this.settings.activePageIndex].canvas;
    }

    /**
     * Returns the dimensions of the current page at the current zoom level. Also works in
     * grid view.
     *
     * @public
     * @returns {object} - An object containing the current page dimensions at the current zoom level.
     **/
    getCurrentPageDimensionsAtCurrentZoomLevel ()
    {
        return this.getPageDimensionsAtCurrentZoomLevel(this.settings.activePageIndex);
    }

    /**
     * Returns the current filename (deprecated). Returns the URI for current page.
     *
     * @public
     * @deprecated
     * @returns {string} - The URI for the current page image.
     **/
    getCurrentPageFilename ()
    {
        console.warn('This method will be deprecated in the next version of Diva. Please use getCurrentPageURI instead.');
        return this.settings.manifest.pages[this.settings.activePageIndex].f;
    }

    /**
     * Returns an array of page indices that are visible in the viewport.
     *
     * @public
     * @returns {array} - The 0-based indices array for the currently visible pages.
     **/
    getCurrentPageIndices ()
    {
        return this.settings.currentPageIndices;
    }

    /**
     * Returns the 0-based index for the current page.
     *
     * @public
     * @returns {number} - The 0-based index for the currently visible page.
     **/
     getActivePageIndex ()
     {
        return this.settings.activePageIndex;
     }

    /**
     * Shortcut to getPageOffset for current page.
     *
     * @public
     * @returns {} -
     * */
    getCurrentPageOffset ()
    {
        return this.getPageOffset(this.settings.activePageIndex);
    }

    /**
     * Returns the current URI for the visible page.
     *
     * @public
     * @returns {string} - The URI for the current page image.
     **/
    getCurrentPageURI ()
    {
        return this.settings.manifest.pages[this.settings.activePageIndex].f;
    }

    /**
     * Return the current URL for the viewer, including the hash parameters reflecting
     * the current state of the viewer.
     *
     * @public
     * @returns {string} - The URL for the current view state.
     * */
    getCurrentURL ()
    {
        return this._getCurrentURL();
    }

    /**
     * Returns an array of all filenames in the document. Deprecated.
     *
     * @public
     * @deprecated
     * @returns {Array} - An array of all the URIs in the document.
     * */
    getFilenames ()
    {
        console.warn('This will be removed in the next version of Diva. Use getAllPageURIs instead.');

        return this.settings.manifest.pages.map( (pg) =>
        {
            return pg.f;
        });
    }

    /**
     * Get the number of grid pages per row.
     *
     * @public
     * @returns {number} - The number of grid pages per row.
     **/
    getGridPagesPerRow ()
    {
        // TODO(wabain): Add test case
        return this.settings.pagesPerRow;
    }

    /**
     * Get the instance ID number.
     *
     * @public
     * @returns {number} - The instance ID.
     * */
    //
    getInstanceId ()
    {
        return this.settings.ID;
    }

    /**
     * Get the instance selector for this instance. This is the selector for the parent
     * div.
     *
     * @public
     * @returns {string} - The viewport selector.
     * */
    getInstanceSelector ()
    {
        return this.divaState.viewerCore.selector;
    }

    /**
     * Returns the title of the document, based on the label in the IIIF manifest.
     *
     * @public
     * @returns {string} - The current title of the object from the label key in the IIIF Manifest.
     **/
    getItemTitle ()
    {
        return this.settings.manifest.itemTitle;
    }

    /**
     * Gets the maximum zoom level for the entire document.
     *
     * @public
     * @returns {number} - The maximum zoom level for the document
     * */
    getMaxZoomLevel ()
    {
        return this.settings.maxZoomLevel;
    }

    /**
     * Gets the max zoom level for a given page.
     *
     * @public
     * @param {number} pageIdx - The 0-based index number for the page.
     * @returns {number} - The maximum zoom level for that page.
     * */
    getMaxZoomLevelForPage (pageIdx)
    {
        if (!this._checkLoaded())
            return false;

        return this.settings.manifest.pages[pageIdx].m;
    }

    /**
     * Gets the minimum zoom level for the entire document.
     *
     * @public
     * @returns {number} - The minimum zoom level for the document
     * */
    getMinZoomLevel ()
    {
        return this.settings.minZoomLevel;
    }

    /**
     * Gets the number of pages in the document.
     *
     * @public
     * @returns {number} - The number of pages in the document.
     * */
    getNumberOfPages ()
    {
        if (!this._checkLoaded())
            return false;

        return this.settings.numPages;
    }

    /**
     * If a canvas has multiple images defined, returns the non-primary image.
     *
     * @public
     * @params {number} pageIndex - The page index for which to return the other images.
     * @returns {object} An object containing the other images.
     **/
    getOtherImages (pageIndex)
    {
        return this.settings.manifest.pages[pageIndex].otherImages;
    }

    /**
     * Get page dimensions in the current view and zoom level
     *
     * @public
     * @params {number} pageIndex - A valid 0-based page index
     * @returns {object} - An object containing the dimensions of the page
     * */
    getPageDimensions (pageIndex)
    {
        if (!this._checkLoaded())
            return null;

        return this.divaState.viewerCore.getCurrentLayout().getPageDimensions(pageIndex);
    }

    /**
     * Returns the dimensions of a given page at the current zoom level.
     * Also works in Grid view
     *
     * @public
     * @param {number} pageIndex - The 0-based page index
     * @returns {object} - An object containing the page dimensions at the current zoom level.
     * */
    getPageDimensionsAtCurrentZoomLevel (pageIndex)
    {
        let pidx = parseInt(pageIndex, 10);

        if (!this._isPageIndexValid(pidx))
            throw new Error('Invalid Page Index');

        return this.divaState.viewerCore.getCurrentLayout().getPageDimensions(pidx);
    }

    /**
     * Get page dimensions at a given zoom level
     *
     * @public
     * @params {number} pageIdx - A valid 0-based page index
     * @params {number} zoomLevel - A candidate zoom level.
     * @returns {object} - An object containing the dimensions of the page at the given zoom level.
     **/
    getPageDimensionsAtZoomLevel (pageIdx, zoomLevel)
    {
        if (!this._checkLoaded())
            return false;

        if (zoomLevel > this.settings.maxZoomLevel)
            zoomLevel = this.settings.maxZoomLevel;

        const pg = this.settings.manifest.pages[parseInt(pageIdx, 10)];
        const pgAtZoom = pg.d[parseInt(zoomLevel, 10)];

        return {
            width: pgAtZoom.w,
            height: pgAtZoom.h
        };
    }

    /**
     * Returns a URL for the image of the page at the given index. The
     * optional size parameter supports setting the image width or height
     * (default is full-sized).
     *
     * @public
     * @params {number} pageIndex - 0-based page index
     * @params {?object} size - an object containing width and height information
     * @returns {string} - The IIIF URL for a given page at an optional size
     */
    getPageImageURL (pageIndex, size)
    {
        return this.settings.manifest.getPageImageURL(pageIndex, size);
    }

    /**
     * Given a set of co-ordinates (e.g., from a mouse click), return the 0-based page index
     * for which it matches.
     *
     * @public
     * @params {number} pageX - The x co-ordinate
     * @params {number} pageY - The y co-ordinate
     * @returns {number} - The page index matching the co-ordinates.
     * */
    getPageIndexForPageXYValues (pageX, pageY)
    {
        return this._getPageIndexForPageXYValues(pageX, pageY);
    }

    /**
     * Returns distance between the northwest corners of diva-inner and page index.
     *
     * @public
     * @params {number} pageIndex - The 0-based page index
     * @params {?options} options - A set of options to pass in.
     * @returns {object} - The offset between the upper left corner and the page.
     *
     * */
    getPageOffset (pageIndex, options)
    {
        const region = this.divaState.viewerCore.getPageRegion(pageIndex, options);

        return {
            top: region.top,
            left: region.left
        };
    }

    /**
     * Get the instance settings.
     *
     * @public
     * @returns {object} - The current instance settings.
     * */
    getSettings ()
    {
        return this.settings;
    }

    /**
     * Get an object representing the complete state of the viewer.
     *
     * @public
     * @returns {object} - The current instance state.
     * */
    getState ()
    {
        return this._getState();
    }

    /**
     * Get the current zoom level.
     *
     * @public
     * @returns {number} - The current zoom level.
     * */
    getZoomLevel ()
    {
        return this.settings.zoomLevel;
    }

    /**
     *  Go to a particular page (with indexing starting at 0).
     *  The (xAnchor) side of the page will be anchored to the (xAnchor) side of the diva-outer element
     *
     *  @public
     *  @params {number} pageIndex - 0-based page index.
     *  @params {?string} xAnchor - may either be "left", "right", or default "center"
     *  @params {?string} yAnchor - may either be "top", "bottom", or default "center"; same process as xAnchor.
     *  @returns {boolean} - True if the page index is valid; false if it is not.
     * */
    gotoPageByIndex (pageIndex, xAnchor, yAnchor)
    {
        return this._gotoPageByIndex(pageIndex, xAnchor, yAnchor);
    }

    /**
     * Given a canvas label, attempt to go to that page. If no label was found.
     * the label will be attempted to match against the page index.
     *
     * @public
     * @params {string} label - The label to search on.
     * @params {?string} xAnchor - may either be "left", "right", or default "center"
     * @params {?string} yAnchor - may either be "top", "bottom", or default "center"
     * @returns {boolean} - True if the page index is valid; false if it is not.
     * */
    gotoPageByLabel (label, xAnchor, yAnchor)
    {
        const pages = this.settings.manifest.pages;
        let llc = label.toLowerCase();

        for (let i = 0, len = pages.length; i < len; i++)
        {
            if (pages[i].l.toLowerCase().indexOf(llc) > -1)
                return this._gotoPageByIndex(i, xAnchor, yAnchor);
        }

        const pageIndex = parseInt(label, 10) - 1;
        return this._gotoPageByIndex(pageIndex, xAnchor, yAnchor);
    }

    /**
     * Jump to a page based on its filename. Deprecated. Use gotoPageByURI instead.
     *
     * @public
     * @params {string} filename - The filename of the image to jump to.
     * @params {?string} xAnchor - may either be "left", "right", or default "center"
     * @params {?string} yAnchor - may either be "top", "bottom", or default "center"
     * @returns {boolean} true if successful and false if the filename is not found.
    */
    gotoPageByName (filename, xAnchor, yAnchor)
    {
        console.warn('This method will be removed in the next version of Diva.js. Use gotoPageByURI instead.');
        const pageIndex = this._getPageIndex(filename);
        return this._gotoPageByIndex(pageIndex, xAnchor, yAnchor);
    }

    /**
     * Jump to a page based on its URI.
     *
     * @public
     * @params {string} uri - The URI of the image to jump to.
     * @params {?string} xAnchor - may either be "left", "right", or default "center"
     * @params {?string} yAnchor - may either be "top", "bottom", or default "center"
     * @returns {boolean} true if successful and false if the URI is not found.
     */
    gotoPageByURI (uri, xAnchor, yAnchor)
    {
        const pageIndex = this._getPageIndex(uri);
        return this._gotoPageByIndex(pageIndex, xAnchor, yAnchor);
    }

    /**
     * Whether the page has other images to display.
     *
     * @public
     * @params {number} pageIndex - The 0-based page index
     * @returns {boolean} Whether the page has other images to display.
     **/
    hasOtherImages (pageIndex)
    {
        return this.settings.manifest.pages[pageIndex].otherImages === true;
    }

    /**
     * Hides the pages that are marked "non-paged" in the IIIF manifest.
     *
     * @public
     **/
    hideNonPagedPages ()
    {
        this._reloadViewer({ showNonPagedPages: false });
    }

    /**
     * Is the viewer currently in full-screen mode?
     *
     * @public
     * @returns {boolean} - Whether the viewer is in fullscreen mode.
     **/
    isInFullscreen ()
    {
        return this.settings.inFullscreen;
    }

    /**
     * Check if a page index is within the range of the document
     *
     * @public
     * @returns {boolean} - Whether the page index is valid.
     **/
    isPageIndexValid (pageIndex)
    {
        return this._isPageIndexValid(pageIndex);
    }

    /**
     * Determines if a page is currently in the viewport
     *
     * @public
     * @params {number} pageIndex - The 0-based page index
     * @returns {boolean} - Whether the page is currently in the viewport.
     **/
    isPageInViewport (pageIndex)
    {
        return this.viewerState.renderer.isPageVisible(pageIndex);
    }

    /**
     * Whether the Diva viewer has been fully initialized.
     *
     * @public
     * @returns {boolean} - True if the viewer is initialized; false otherwise.
     **/
    isReady ()
    {
        return this.viewerState.loaded;
    }

    /**
     * Check if something (e.g. a highlight box on a particular page) is visible
     *
     * @public
     * @params {number} pageIndex - The 0-based page index
     * @params {number} leftOffset - The distance of the region from the left of the viewport
     * @params {number} topOffset - The distance of the region from the top of the viewport
     * @params {number} width - The width of the region
     * @params {number} height - The height of the region
     * @returns {boolean} - Whether the region is in the viewport.
     **/
    isRegionInViewport (pageIndex, leftOffset, topOffset, width, height)
    {
        const layout = this.divaState.viewerCore.getCurrentLayout();

        if (!layout)
            return false;

        const offset = layout.getPageOffset(pageIndex);

        const top = offset.top + topOffset;
        const left = offset.left + leftOffset;

        return this.viewerState.viewport.intersectsRegion({
            top: top,
            bottom: top + height,
            left: left,
            right: left + width
        });
    }

    /**
     * Whether the page layout is vertically or horizontally oriented.
     *
     * @public
     * @returns {boolean} - True if vertical; false if horizontal.
     **/
    isVerticallyOriented ()
    {
        return this.settings.verticallyOriented;
    }

    /**
     * Leave fullscreen mode if currently in fullscreen mode.
     *
     * @public
     * @returns {boolean} - true if in fullscreen mode intitially, false otherwise
     **/
    leaveFullscreenMode ()
    {
        if (this.settings.inFullscreen)
        {
            this._toggleFullscreen();
            return true;
        }

        return false;
    }

    /**
     * Leave grid view if currently in grid view.
     *
     * @public
     * @returns {boolean} - true if in grid view initially, false otherwise
     **/
    leaveGridView ()
    {
        if (this.settings.inGrid)
        {
            this._reloadViewer({ inGrid: false });
            return true;
        }

        return false;
    }

    /**
     * Set the number of grid pages per row.
     *
     * @public
     * @params {number} pagesPerRow - The number of pages per row
     * @returns {boolean} - True if the operation was successful.
     **/
    setGridPagesPerRow (pagesPerRow)
    {
        // TODO(wabain): Add test case
        if (!this.divaState.viewerCore.isValidOption('pagesPerRow', pagesPerRow))
            return false;

        return this._reloadViewer({
            inGrid: true,
            pagesPerRow: pagesPerRow
        });
    }

    /**
     * Align this diva instance with a state object (as returned by getState)
     *
     * @public
     * @params {object} state - A Diva state object.
     * @returns {boolean} - True if the operation was successful.
     **/
    setState (state)
    {
        this._reloadViewer(this._getLoadOptionsForState(state));
    }

    /**
     * Sets the zoom level.
     *
     * @public
     * @returns {boolean} - True if the operation was successful.
     **/
    setZoomLevel (zoomLevel)
    {
        if (this.settings.inGrid)
        {
            this._reloadViewer({
                inGrid: false
            });
        }

        return this.divaState.viewerCore.zoom(zoomLevel);
    }

    /**
     * Show non-paged pages.
     *
     * @public
     * @returns {boolean} - True if the operation was successful.
     **/
    showNonPagedPages ()
    {
        this._reloadViewer({ showNonPagedPages: true });
    }

    /**
     * Toggle fullscreen mode.
     *
     * @public
     * @returns {boolean} - True if the operation was successful.
     **/
    toggleFullscreenMode ()
    {
        this._toggleFullscreen();
    }

    /**
     * Show/Hide non-paged pages
     *
     * @public
     * @returns {boolean} - True if the operation was successful.
     **/
    toggleNonPagedPagesVisibility ()
    {
        this._reloadViewer({
            showNonPagedPages: !this.settings.showNonPagedPages
        });
    }

    //Changes between horizontal layout and vertical layout. Returns true if document is now vertically oriented, false otherwise.
    toggleOrientation ()
    {
        return this._togglePageLayoutOrientation();
    }

    /**
     * Translates a measurement from the zoom level on the largest size
     * to one on the current zoom level.
     *
     * For example, a point 1000 on an image that is on zoom level 2 of 5
     * translates to a position of 111.111... (1000 / (5 - 2)^2).
     *
     * Works for a single pixel co-ordinate or a dimension (e.g., translates a box
     * that is 1000 pixels wide on the original to one that is 111.111 pixels wide
     * on the current zoom level).
     *
     * @public
     * @params {number} position - A point on the max zoom level
     * @returns {number} - The same point on the current zoom level.
    */
    translateFromMaxZoomLevel (position)
    {
        const zoomDifference = this.settings.maxZoomLevel - this.settings.zoomLevel;
        return position / Math.pow(2, zoomDifference);
    }

    /**
     * Translates a measurement from the current zoom level to the position on the
     * largest zoom level.
     *
     * Works for a single pixel co-ordinate or a dimension (e.g., translates a box
     * that is 111.111 pixels wide on the current image to one that is 1000 pixels wide
     * on the current zoom level).
     *
     * @public
     * @params {number} position - A point on the current zoom level
     * @returns {number} - The same point on the max zoom level.
    */
    translateToMaxZoomLevel (position)
    {
        const zoomDifference = this.settings.maxZoomLevel - this.settings.zoomLevel;

        // if there is no difference, it's a box on the max zoom level and
        // we can just return the position.
        if (zoomDifference === 0)
            return position;

        return position * Math.pow(2, zoomDifference);
    }

    /**
     * Zoom in.
     *
     * @public
     * @returns {boolean} - false if it's at the maximum zoom
     **/
    zoomIn ()
    {
        return this.setZoomLevel(this.settings.zoomLevel + 1);
    }

    /**
     * Zoom out.
     * @returns {boolean} - false if it's at the minimum zoom
     **/
    zoomOut ()
    {
        return this.setZoomLevel(this.settings.zoomLevel - 1);
    }
}

export default Diva;

/**
 * Make `Diva` available in the global context.
 * */
(function (global)
{
    global.Diva = global.Diva || Diva;
    global.Diva.Events = diva.Events;
})(window);
