/**
 * Translate page layouts, as generated by page-layouts, into an
 * object which computes layout information for the document as
 * a whole.
 */
export default class DocumentLayout
{
    constructor (config, zoomLevel)
    {
        const computedLayout = getComputedLayout(config, zoomLevel);

        this.dimensions = computedLayout.dimensions;
        this.pageGroups = computedLayout.pageGroups;
        this._pageLookup = getPageLookup(computedLayout.pageGroups);
    }

    /**
     * @typedef {Object} PageInfo
     * @property {number} index
     * @property {{index, dimensions, pages, region, padding}} group
     * @property {{height: number, width: number}} dimensions
     * @property {{top: number, left: number}} groupOffset
     */

    /**
     * @param pageIndex
     * @returns {PageInfo|null}
     */
    getPageInfo (pageIndex)
    {
        return this._pageLookup[pageIndex] || null;
    }

    /**
     * Get the dimensions of a page
     *
     * @param pageIndex
     * @returns {{height: number, width: number}}
     */
    getPageDimensions (pageIndex)
    {
        if (!this._pageLookup || !this._pageLookup[pageIndex])
            return null;

        const region = getPageRegionFromPageInfo(this._pageLookup[pageIndex]);

        return {
            height: region.bottom - region.top,
            width: region.right - region.left
        };
    }

    // TODO(wabain): Get rid of this; it's a subset of the page region, so
    // give that instead
    /**
     * Get the top-left coordinates of a page, including*** padding
     *
     * @param pageIndex
     * @param options
     * @returns {{top: number, left: number} | null}
     */
    getPageOffset (pageIndex, options)
    {
        const region = this.getPageRegion(pageIndex, options);

        if (!region)
            return null;

        return {
            top: region.top,
            left: region.left
        };
    }

    getPageRegion (pageIndex, options)
    {
        const pageInfo = this._pageLookup[pageIndex];

        if (!pageInfo)
            return null;

        const region = getPageRegionFromPageInfo(pageInfo);
        const padding = pageInfo.group.padding;

        if (options && options.includePadding)
        {
            return {
                top: region.top + padding.top,
                left: region.left + padding.left,
                bottom: region.bottom,
                right: region.right
            };
        }

        return {
            top: region.top,
            left: region.left,
            // need to account for plugin icons below the page, see 
            // https://github.com/DDMAL/diva.js/issues/436
            bottom: region.bottom + padding.top, 
            right: region.right
        };
    }

    /**
     * Get the distance from the top-right of the page to the center of the
     * specified viewport region
     *
     * @param pageIndex
     * @param viewport {{top: number, left: number, bottom: number, right: number}}
     * @returns {{x: number, y: number}}
     */
    getPageToViewportCenterOffset (pageIndex, viewport)
    {
        const scrollLeft = viewport.left;
        const elementWidth = viewport.right - viewport.left;

        const offset = this.getPageOffset(pageIndex);

        const x = scrollLeft - offset.left + parseInt(elementWidth / 2, 10);

        const scrollTop = viewport.top;
        const elementHeight = viewport.bottom - viewport.top;

        const y = scrollTop - offset.top + parseInt(elementHeight / 2, 10);

        return {
            x: x,
            y: y
        };
    }
}

function getPageRegionFromPageInfo (page)
{
    const top    = page.groupOffset.top  + page.group.region.top;
    const bottom = top + page.dimensions.height;
    const left   = page.groupOffset.left + page.group.region.left;
    const right  = left + page.dimensions.width;

    return {
        top: top,
        bottom: bottom,
        left: left,
        right: right
    };
}

function getPageLookup (pageGroups)
{
    const pageLookup = {};

    pageGroups.forEach(group => {
        group.pages.forEach(page => {
            pageLookup[page.index] = {
                index: page.index,
                group: group,
                dimensions: page.dimensions,
                groupOffset: page.groupOffset
            };
        });
    });

    return pageLookup;
}

function getComputedLayout (config, zoomLevel)
{
    const scaledLayouts = zoomLevel === null ? config.pageLayouts : getScaledPageLayouts(config, zoomLevel);

    const documentSecondaryExtent = getExtentAlongSecondaryAxis(config, scaledLayouts);

    // The current position in the document along the primary axis
    let primaryDocPosition = config.verticallyOriented ?
        config.padding.document.top :
        config.padding.document.left;

    const pageGroups = [];

    // TODO: Use bottom, right as well
    const pagePadding = {
        top: config.padding.page.top,
        left: config.padding.page.left
    };

    scaledLayouts.forEach((layout, index) => {
        let top, left;

        if (config.verticallyOriented)
        {
            top = primaryDocPosition;
            left = (documentSecondaryExtent - layout.dimensions.width) / 2;
        }
        else
        {
            top = (documentSecondaryExtent - layout.dimensions.height) / 2;
            left = primaryDocPosition;
        }

        const region = {
            top: top,
            bottom: top + pagePadding.top + layout.dimensions.height,
            left: left,
            right: left + pagePadding.left + layout.dimensions.width
        };

        pageGroups.push({
            index: index,
            dimensions: layout.dimensions,
            pages: layout.pages,
            region: region,
            padding: pagePadding
        });

        primaryDocPosition = config.verticallyOriented ? region.bottom : region.right;
    });

    let height, width;

    if (config.verticallyOriented)
    {
        height = primaryDocPosition + pagePadding.top;
        width = documentSecondaryExtent;
    }
    else
    {
        height = documentSecondaryExtent;
        width = primaryDocPosition + pagePadding.left;
    }

    return {
        dimensions: {
            height: height,
            width: width
        },
        pageGroups: pageGroups
    };
}

function getScaledPageLayouts (config, zoomLevel)
{
    const scaleRatio = Math.pow(2, zoomLevel - config.maxZoomLevel);

    return config.pageLayouts.map(group => ({
        dimensions: scaleDimensions(group.dimensions, scaleRatio),

        pages: group.pages.map(page => ({
            index: page.index,

            groupOffset: {
                top: Math.floor(page.groupOffset.top * scaleRatio),
                left: Math.floor(page.groupOffset.left * scaleRatio)
            },

            dimensions: scaleDimensions(page.dimensions, scaleRatio)
        }))
    }));
}

function scaleDimensions (dimensions, scaleRatio)
{
    return {
        height: Math.floor(dimensions.height * scaleRatio),
        width: Math.floor(dimensions.width * scaleRatio)
    };
}

function getExtentAlongSecondaryAxis (config, scaledLayouts)
{
    // Get the extent of the document along the secondary axis
    let secondaryDim, secondaryPadding;
    const docPadding = config.padding.document;

    if (config.verticallyOriented)
    {
        secondaryDim = 'width';
        secondaryPadding = docPadding.left + docPadding.right;
    }
    else
    {
        secondaryDim = 'height';
        secondaryPadding = docPadding.top + docPadding.bottom;
    }

    return secondaryPadding + scaledLayouts.reduce((maxDim, layout) => Math.max(layout.dimensions[secondaryDim], maxDim), 0);
}
