import { ensureFocus, findSelectableAncestor, getElementCoords, getElementId, getNodeByCoords, getNodeById, isUnselectableNode } from './CellSelectionUtil';
var LOW_PRIORITY = 500;
var VALID_DIRECTIONS = {
  above: true,
  below: true,
  right: true,
  left: true
};
/**
 * A cell selection utlity; allows selection of elements, independent from
 * whether they are backed by a business object or not.
 *
 * Works together with the {@link SelectionAware} trait.
 *
 * @param {RenderConfig} config
 * @param {EventBus} eventBus
 * @param {Sheet} sheet
 * @param {Selection} selection
 * @param {ElementRegistry} elementRegistry
 */

export default function CellSelection(config, eventBus, sheet, selection, elementRegistry) {
  var container = config.container;
  var lastSelection = null;

  function emit(elementId, newSelection) {
    eventBus.fire('selection.' + elementId + '.changed', newSelection);
    eventBus.fire('cellSelection.changed', {
      elementId: elementId,
      selection: newSelection
    });
  }

  function click(event) {
    var target = event.target;

    if (isUnselectableNode(target)) {
      return;
    }

    var selectableNode = findSelectableAncestor(target);
    var elementId = selectableNode && getElementId(selectableNode);
    var focussed = !event.defaultPrevented;
    realSelect(elementId, focussed);
  }

  function focus(event) {
    var elementId = getElementId(event.target);
    var focussed = !event.defaultPrevented;
    event.stopPropagation();
    return realSelect(elementId, focussed);
  }

  function unfocus(event) {
    var elementId = getElementId(event.target);
    emit(elementId, {
      focussed: false
    });
  }

  function realSelect(elementId) {
    var focussed = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : true;

    if (lastSelection !== elementId) {
      emit(lastSelection, {
        selected: false,
        focussed: false
      });
    }

    lastSelection = elementId;

    if (elementId) {
      emit(elementId, {
        selected: true,
        focussed: focussed
      });
    }

    if (elementId) {
      selection.select(elementId);
    } else {
      selection.deselect();
    }
  }

  eventBus.on('cell.click', LOW_PRIORITY, click);
  eventBus.on('cell.focusin', LOW_PRIORITY, focus);
  eventBus.on('cell.focusout', LOW_PRIORITY, unfocus);
  eventBus.on('cellSelection.changed', function (event) {
    var elementId = event.elementId,
        selection = event.selection;
    var actualElement = getNodeById(elementId, container);

    if (selection.focussed && actualElement) {
      ensureFocus(actualElement);
    }
  });
  eventBus.on('selection.changed', function (event) {
    var selection = event.selection,
        oldSelection = event.oldSelection;
    var elementId = selection && selection.id;
    var oldElementId = oldSelection && oldSelection.id; // select new element

    if (elementId && elementId !== lastSelection) {
      realSelect(selection.id);
    } else // deselect old element
      if (oldElementId && oldElementId === lastSelection) {
        realSelect();
      }
  }); // API //////////////////////

  /**
   * Return true if a cell is currently selected.
   *
   * @return {boolean}
   */

  this.isCellSelected = function () {
    return !!lastSelection;
  };
  /**
   * Get the currently active cellSelection.
   *
   * @return {string} selection
   */


  this.getCellSelection = function () {
    return lastSelection;
  };
  /**
   * Select next cell in given direction.
   *
   * Returns true on success; false on fail (i.e. if no next selection
   * in direction could be found).
   *
   * @param {string} direction
   *
   * @return {boolean}
   */


  this.selectCell = function (direction) {
    if (!lastSelection) {
      return;
    }

    if (!(direction in VALID_DIRECTIONS)) {
      throw new Error('direction must be any of { above, below, left, right }');
    }

    var selectionEl = getNodeById(lastSelection, container);
    var coords = getElementCoords(selectionEl);

    if (!coords) {
      return false;
    }

    var nextCoords = getNextCoords(coords, direction);
    var nextNode = getNodeByCoords(nextCoords, container);

    if (!nextNode) {
      return false;
    }

    var nextElId = getElementId(nextNode);

    if (nextElId) {
      realSelect(nextElId, {
        focussed: true,
        selected: true
      });
    }

    return true;
  };

  eventBus.on('contextMenu.close', function () {
    if (lastSelection) {
      return realSelect(lastSelection);
    }
  });
}
CellSelection.$inject = ['config.renderer', 'eventBus', 'sheet', 'selection', 'elementRegistry']; // helpers ////////////////

function getNextCoords(coords, direction) {
  var row = coords.row,
      col = coords.col;

  if (direction === 'above' || direction === 'below') {
    var rowIndex = parseInt(row, 10);

    if (isNaN(rowIndex)) {
      return coords;
    }

    var nextRowIndex = direction === 'above' ? rowIndex - 1 : rowIndex + 1;
    return {
      col: col,
      row: nextRowIndex
    };
  }

  if (direction === 'left' || direction === 'right') {
    var colIndex = parseInt(col, 10);

    if (isNaN(colIndex)) {
      return coords;
    }

    var nextColIndex = direction === 'left' ? colIndex - 1 : colIndex + 1;
    return {
      row: row,
      col: nextColIndex
    };
  }

  throw new Error('invalid direction <' + direction + '>');
}
//# sourceMappingURL=CellSelection.js.map