import { createVNode, createTextVNode } from "inferno";

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

function _defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } }

function _createClass(Constructor, protoProps, staticProps) { if (protoProps) _defineProperties(Constructor.prototype, protoProps); if (staticProps) _defineProperties(Constructor, staticProps); return Constructor; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

import { classes as domClasses, domify, queryAll as domQueryAll, remove as domRemove } from 'min-dom';
import { isInput, isOutput } from 'dmn-js-shared/lib/util/ModelUtil';
import { Row, Col } from 'table-js/lib/model';
import { forEach } from 'min-dash';
var TOP = 'top',
    RIGHT = 'right',
    BOTTOM = 'bottom',
    LEFT = 'left';

var DragAndDrop =
/*#__PURE__*/
function () {
  function DragAndDrop(components, elementRegistry, eventBus, dragAndDrop, renderer, rules, sheet) {
    var _this = this;

    _classCallCheck(this, DragAndDrop);

    _defineProperty(this, "_cleanup", function () {
      var container = _this._renderer.getContainer();

      removeHighlight(container);
      removeFadeOut(container);

      if (_this._dragImage) {
        domRemove(_this._dragImage);
        _this._dragImage = null;
      }
    });

    this._elementRegistry = elementRegistry;
    this._dragAndDrop = dragAndDrop;
    this._renderer = renderer;
    this._rules = rules;
    this._sheet = sheet; // provide drag handle for drag and drop

    components.onGetComponent('cell-inner', function (_ref) {
      var cellType = _ref.cellType,
          col = _ref.col,
          row = _ref.row;

      if (cellType === 'rule-index') {
        return function () {
          return createVNode(1, "span", "dmn-icon-drag vertical", createTextVNode("\xA0"), 2, {
            "draggable": "true",
            "onDragStart": function onDragStart(e) {
              return _this.startDrag(row, e);
            },
            "title": "Move rule"
          });
        };
      } else if (cellType === 'input-cell' || cellType === 'output-cell') {
        var title = "Move ".concat(isInput(col) ? 'Input' : 'Output');
        return function () {
          return createVNode(1, "span", "dmn-icon-drag horizontal", null, 1, {
            "draggable": "true",
            "onDragStart": function onDragStart(e) {
              return _this.startDrag(col, e);
            },
            "title": title
          });
        };
      }
    }); // validate allowed rules

    eventBus.on('dragAndDrop.dragEnter', function (event) {
      var dragContext = event.dragContext;
      var draggedElement = dragContext.draggedElement,
          hoverEl = dragContext.hoverEl; // can always drag rows

      if (draggedElement instanceof Row) {
        return true;
      }

      if (draggedElement instanceof Col) {
        var dropIndex = getTargetColIndex(hoverEl, _this._elementRegistry, _this._sheet); // cannot drop as we cannot compute the drop index

        if (dropIndex === -1) {
          return false;
        }

        var allowed = _this._rules.allowed('col.move', {
          col: draggedElement,
          index: dropIndex
        });

        return allowed;
      }

      return false;
    }); // clear previous UI

    eventBus.on('dragAndDrop.dragLeave', function (event) {
      var dragContext = event.dragContext;
      var targetEl = dragContext.targetEl;

      if (!targetEl) {
        return;
      }

      var container = _this._renderer.getContainer();

      removeHighlight(container);
    }); // update UI

    eventBus.on('dragAndDrop.dragOver', function (event) {
      var dragContext = event.dragContext,
          originalEvent = event.originalEvent;
      var draggedElement = dragContext.draggedElement,
          lastPosition = dragContext.lastPosition,
          targetEl = dragContext.targetEl;

      var container = _this._renderer.getContainer();

      if (!targetEl) {
        return false;
      }

      var newPosition;

      if (draggedElement instanceof Row) {
        newPosition = getVerticalPosition(originalEvent, targetEl);
      }

      if (draggedElement instanceof Col) {
        newPosition = getHorizontalPosition(originalEvent, targetEl);
      } // nothing to do


      if (lastPosition === newPosition) {
        return true;
      } // remove old highlight


      removeHighlight(container);

      if (draggedElement instanceof Row) {
        if (newPosition === TOP) {
          // drop above
          highlightRow(targetEl, container, 'top');
        } else {
          // drop below
          highlightRow(targetEl, container, 'bottom');
        }
      }

      if (draggedElement instanceof Col) {
        if (newPosition === LEFT) {
          // drop left
          highlightCol(targetEl, container, 'left');
        } else {
          // drop right
          highlightCol(targetEl, container, 'right');
        }
      } // remember position


      dragContext.lastPosition = newPosition; // allowed

      return true;
    }); // perform drop operation

    eventBus.on('dragAndDrop.drop', function (event) {
      var dragContext = event.dragContext,
          originalEvent = event.originalEvent;
      var draggedElement = dragContext.draggedElement,
          targetEl = dragContext.targetEl;

      if (!targetEl) {
        return false;
      }

      if (draggedElement instanceof Row) {
        var verticalPosition = getVerticalPosition(originalEvent, targetEl);

        var rowId = targetEl.dataset.rowId,
            row = _this._elementRegistry.get(rowId);

        if (!row || row === draggedElement) {
          return;
        }

        var targetRow = getTargetRow(draggedElement, row, verticalPosition, _this._sheet.getRoot().rows);

        if (targetRow === draggedElement) {
          return;
        }

        return targetRow;
      }

      if (draggedElement instanceof Col) {
        var horizontalPosition = getHorizontalPosition(originalEvent, targetEl); // no need to check rules; we verified on
        // dragEnter that dropping is O.K.

        var colId = targetEl.dataset.colId,
            col = _this._elementRegistry.get(colId);

        if (!col || col === draggedElement) {
          return;
        }

        var targetCol = getTargetCol(draggedElement, col, horizontalPosition, _this._sheet.getRoot().cols);

        if (targetCol === draggedElement) {
          return;
        }

        return targetCol;
      }
    });
    eventBus.on('dragAndDrop.dragEnd', this._cleanup);
  }

  _createClass(DragAndDrop, [{
    key: "startDrag",
    value: function startDrag(element, event) {
      var container = this._renderer.getContainer();

      this._dragImage = domify("<span style=\"\n          visibility: hidden;\n          position: fixed;\n          top: -10000px\n      \"></span>"); // needs to be present in DOM

      document.body.appendChild(this._dragImage); // QUIRK: not supported by Edge and Internet Explorer

      if (event.dataTransfer.setDragImage) {
        event.dataTransfer.setDragImage(this._dragImage, 0, 0);
      }

      if (element instanceof Row) {
        fadeOutRow(element, container);
      } else if (element instanceof Col) {
        fadeOutCol(element, container);
      }

      this._dragAndDrop.startDrag(element, event);
    }
  }]);

  return DragAndDrop;
}();

export { DragAndDrop as default };
DragAndDrop.$inject = ['components', 'elementRegistry', 'eventBus', 'dragAndDrop', 'renderer', 'rules', 'sheet']; // helpers //////////

function getTargetColIndex(cellEl, elementRegistry, sheet) {
  var targetCol = elementRegistry.get(cellEl.dataset.colId);

  if (!targetCol) {
    return -1;
  }

  var _sheet$getRoot = sheet.getRoot(),
      cols = _sheet$getRoot.cols;

  return cols.indexOf(targetCol);
}

function highlightRow(dragOverCell, container, position) {
  var rowId = dragOverCell.dataset.rowId;

  if (!rowId) {
    return;
  }

  var cells = domQueryAll("[data-row-id=".concat(rowId, "]"), container);
  forEach(cells, function (cell) {
    // QUIRK: PhantomJS might return object instead of NodeList
    if (isNode(cell)) {
      domClasses(cell).add('dragover');
      domClasses(cell).add(position);
    }
  });
}

function highlightCol(dragOverCell, container, position) {
  var colId = dragOverCell.dataset.colId;

  if (!colId) {
    return;
  }

  var cells = domQueryAll("[data-col-id=".concat(colId, "]"), container);
  forEach(cells, function (cell) {
    // QUIRK: PhantomJS might return object instead of NodeList
    if (isNode(cell)) {
      domClasses(cell).add('dragover');
      domClasses(cell).add(position);
    }
  });
}

function removeHighlight(container) {
  var cells = domQueryAll('.dragover', container);
  forEach(cells, function (cell) {
    // QUIRK: PhantomJS might return object instead of NodeList
    if (isNode(cell)) {
      domClasses(cell).remove('dragover');
      domClasses(cell).remove('top');
      domClasses(cell).remove('right');
      domClasses(cell).remove('bottom');
      domClasses(cell).remove('left');
    }
  });
}

function fadeOutRow(row, container) {
  var cells = domQueryAll("[data-row-id=".concat(row.id, "]"), container);
  forEach(cells, function (cell) {
    // QUIRK: PhantomJS might return object instead of NodeList
    if (isNode(cell)) {
      domClasses(cell).add('dragged');
    }
  });
}

function fadeOutCol(col, container) {
  var cells = domQueryAll("[data-col-id=".concat(col.id, "]"), container);
  forEach(cells, function (cell) {
    // QUIRK: PhantomJS might return object instead of NodeList
    if (isNode(cell)) {
      domClasses(cell).add('dragged');
    }
  });
}

function removeFadeOut(container) {
  var cells = domQueryAll('.dragged', container);
  forEach(cells, function (cell) {
    // QUIRK: PhantomJS might return object instead of NodeList
    if (isNode(cell)) {
      domClasses(cell).remove('dragged');
    }
  });
}

function getHorizontalPosition(event, dragOverElement) {
  var bounds = dragOverElement.getBoundingClientRect();
  return event.clientX < bounds.left + bounds.width / 2 ? LEFT : RIGHT;
}

function getVerticalPosition(event, dragOverElement) {
  var bounds = dragOverElement.getBoundingClientRect();
  return event.clientY < bounds.top + bounds.height / 2 ? TOP : BOTTOM;
}

function getTargetRow(draggedRow, targetRow, verticalPosition, rows) {
  if (rows.indexOf(draggedRow) > rows.indexOf(targetRow)) {
    targetRow = getRowBelow(targetRow, rows);
  }

  if (verticalPosition === TOP) {
    // return row above or row
    return getRowAbove(targetRow, rows);
  } else {
    // return row
    return targetRow;
  }
}

function getTargetCol(draggedCol, targetCol, horizontalPosition, cols) {
  if (cols.indexOf(draggedCol) > cols.indexOf(targetCol)) {
    targetCol = getColRight(targetCol, cols);
  }

  if (horizontalPosition === LEFT) {
    // return col left or col
    return getColLeft(targetCol, cols);
  } else {
    // return col
    return targetCol;
  }
}

function getRowAbove(row, rows) {
  var index = rows.indexOf(row);
  return rows[Math.max(0, index - 1)];
}

function getRowBelow(row, rows) {
  var index = rows.indexOf(row);
  return rows[Math.min(rows.length - 1, index + 1)];
}

function getColLeft(col, cols) {
  var index = cols.indexOf(col);

  if (isOutput(col)) {
    var firstOutput = cols.filter(function (col) {
      return isOutput(col);
    })[0];
    var firstOutputIndex = cols.indexOf(firstOutput);
    return cols[Math.max(firstOutputIndex, index - 1)];
  }

  return cols[Math.max(0, index - 1)];
}

function getColRight(col, cols) {
  var index = cols.indexOf(col);

  if (isInput(col)) {
    var inputs = cols.filter(function (col) {
      return isInput(col);
    });
    var lastInput = inputs[inputs.length - 1];
    var lastInputIndex = cols.indexOf(lastInput);
    return cols[Math.min(lastInputIndex, index + 1)];
  }

  return cols[Math.min(cols.length - 1, index + 1)];
} // QUIRK: PhantomJS requires check if actual DOM node


function isNode(node) {
  return node && (node.nodeType === 1 || node.nodeType == 11);
}
//# sourceMappingURL=DragAndDrop.js.map