function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

function _defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } }

function _createClass(Constructor, protoProps, staticProps) { if (protoProps) _defineProperties(Constructor.prototype, protoProps); if (staticProps) _defineProperties(Constructor, staticProps); return Constructor; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

import { event as domEvent } from 'min-dom';
import { findSelectableAncestor } from '../cell-selection/CellSelectionUtil';
import { isCmd, isShift } from './KeyboardUtil';
/**
 * A keyboard abstraction that may be activated and
 * deactivated by users at will, consuming key events
 * and triggering diagram actions.
 *
 * The implementation fires the following key events that allow
 * other components to hook into key handling:
 *
 *  - keyboard.bind
 *  - keyboard.unbind
 *  - keyboard.init
 *  - keyboard.destroy
 *
 * All events contain the fields (node, listeners).
 *
 * A default binding for the keyboard may be specified via the
 * `keyboard.bindTo` configuration option.
 *
 * @param {Config} config
 * @param {EventBus} eventBus
 * @param {EditorActions} editorActions
 * @param {CellSelection} cellSelection
 */

var Keyboard =
/*#__PURE__*/
function () {
  function Keyboard(config, eventBus, editorActions, cellSelection) {
    var _this = this;

    _classCallCheck(this, Keyboard);

    _defineProperty(this, "_init", function () {
      _this._registerDefaultBindings();

      _this._fire('init');
    });

    _defineProperty(this, "_destroy", function () {
      _this._fire('destroy');

      _this.unbind();

      _this._listeners = null;
    });

    _defineProperty(this, "_keyHandler", function (event) {
      var i,
          l,
          listeners = _this._listeners,
          code = event.keyCode || event.charCode || -1;

      for (i = 0; l = listeners[i]; i++) {
        if (l(code, event)) {
          event.preventDefault();
          event.stopPropagation();
          return;
        }
      }
    });

    _defineProperty(this, "unbind", function () {
      var node = _this._node;

      if (node) {
        _this._fire('unbind'); // unbind key events


        domEvent.unbind(node, 'keydown', _this._keyHandler, true);
      }

      _this._node = null;
    });

    this._config = config || {};
    this._editorActions = editorActions;
    this._eventBus = eventBus;
    this._cellSelection = cellSelection;
    this._listeners = [];
    eventBus.on('table.destroy', this._destroy);
    eventBus.on('table.init', this._init);
    eventBus.on('attach', function () {
      if (_this._config.bindTo) {
        _this.bind(config.bindTo);
      }
    });
    eventBus.on('detach', this.unbind);
  }

  _createClass(Keyboard, [{
    key: "bind",
    value: function bind(node) {
      // make sure that the keyboard is only bound once to the DOM
      this.unbind();
      this._node = node; // bind key events

      domEvent.bind(node, 'keydown', this._keyHandler, true);

      this._fire('bind');
    }
  }, {
    key: "getBinding",
    value: function getBinding() {
      return this._node;
    }
  }, {
    key: "_fire",
    value: function _fire(event) {
      this._eventBus.fire('keyboard.' + event, {
        node: this._node,
        listeners: this._listeners
      });
    }
  }, {
    key: "_registerDefaultBindings",
    value: function _registerDefaultBindings() {
      var listeners = this._listeners;
      var editorActions = this._editorActions;
      var cellSelection = this._cellSelection; // init default listeners
      // undo
      // (CTRL|CMD) + Z

      function undo(key, modifiers) {
        if (isCmd(modifiers) && !isShift(modifiers) && key === 90) {
          editorActions.trigger('undo');
          return true;
        }
      } // redo
      // CTRL + Y
      // CMD + SHIFT + Z


      function redo(key, modifiers) {
        if (isCmd(modifiers) && (key === 89 || key === 90 && isShift(modifiers))) {
          editorActions.trigger('redo');
          return true;
        }
      }

      listeners.push(undo);
      listeners.push(redo);

      function selectCellAbove(key, event) {
        if (key !== 13 || isCmd(event) || !isShift(event)) {
          return;
        }

        if (!findSelectableAncestor(event.target)) {
          return;
        }

        editorActions.trigger('selectCellAbove');
        return true;
      }

      listeners.push(selectCellAbove);

      function selectCellBelow(key, event) {
        if (key !== 13 || isCmd(event) || isShift(event)) {
          return;
        }

        if (!findSelectableAncestor(event.target)) {
          return;
        }

        var changed = editorActions.trigger('selectCellBelow');
        var selectedCell = cellSelection.getCellSelection(); // add new rule if no next rule

        if (!changed && selectedCell && !isDecisionNameCell(selectedCell)) {
          var rule = editorActions.trigger('addRule');
          editorActions.trigger('selectCellBelow');
          return rule;
        }

        return true;
      }

      listeners.push(selectCellBelow);
    }
    /**
     * Add a listener function that is notified with (key, modifiers) whenever
     * the keyboard is bound and the user presses a key.
     *
     * @param {Function} listenerFn
     */

  }, {
    key: "addListener",
    value: function addListener(listenerFn) {
      this._listeners.unshift(listenerFn);
    }
  }, {
    key: "removeListener",
    value: function removeListener(listenerFn) {
      this._listeners = this._listeners.filter(function (l) {
        return l !== listenerFn;
      });
    }
  }]);

  return Keyboard;
}();

export { Keyboard as default };
Keyboard.$inject = ['config.keyboard', 'eventBus', 'editorActions', 'cellSelection']; // helper /////

function isDecisionNameCell(cell) {
  return cell === '__decisionProperties_name';
}
//# sourceMappingURL=Keyboard.js.map