import TableTreeWalker from './TableTreeWalker';
/**
 * Import the decision table into a table.
 *
 * Errors and warnings are reported through the specified callback.
 *
 * @param  {decisionTable} decisionTable instance of DecisionTable
 * @param  {ModdleElement} decision moddle element
 * @param  {Function} done
 *         the callback, invoked with (err, [ warning ]) once the import is done
 */

export function importDecision(decisionTable, decision, done) {
  var importer = decisionTable.get('tableImporter'),
      eventBus = decisionTable.get('eventBus'),
      sheet = decisionTable.get('sheet');
  var hasModeling = decisionTable.get('modeling', false);
  var error,
      warnings = [];

  function render(decision) {
    var visitor = {
      create: function create(type, parent, clause, rule) {
        return importer.create(type, parent, clause, rule);
      },
      table: function table(element) {
        return importer.add(element);
      },
      element: function element(_element, parentShape, definitions) {
        return importer.add(_element, parentShape, definitions);
      },
      error: function error(message, context) {
        warnings.push({
          message: message,
          context: context
        });
      }
    };
    var walker = new TableTreeWalker(visitor, {
      canAddMissingEntries: hasModeling
    }); // import

    walker.handleDecision(decision);
  }

  eventBus.fire('import.render.start', {
    decision: decision
  });

  try {
    render(decision);
  } catch (e) {
    error = e;
  }

  eventBus.fire('import.render.complete', {
    error: error,
    warnings: warnings
  });
  eventBus.fire('elements.changed', {
    elements: [sheet.getRoot()]
  });
  done(error, warnings);
}
//# sourceMappingURL=Importer.js.map