/*!
 * dmn-js - dmn-navigated-viewer v17.2.0
 *
 * Copyright (c) 2014-present, camunda Services GmbH
 *
 * Released under the bpmn.io license
 * http://bpmn.io/license
 *
 * Source Code: https://github.com/bpmn-io/dmn-js
 *
 * Date: 2025-03-12
 */
(function (global, factory) {
  typeof exports === 'object' && typeof module !== 'undefined' ? module.exports = factory() :
  typeof define === 'function' && define.amd ? define(factory) :
  (global = typeof globalThis !== 'undefined' ? globalThis : global || self, global.DmnJS = factory());
})(this, (function () { 'use strict';

  /**
   * Flatten array, one level deep.
   *
   * @template T
   *
   * @param {T[][] | T[] | null} [arr]
   *
   * @return {T[]}
   */
  const nativeToString$4 = Object.prototype.toString;
  const nativeHasOwnProperty$3 = Object.prototype.hasOwnProperty;
  function isUndefined$5(obj) {
    return obj === undefined;
  }
  function isDefined(obj) {
    return obj !== undefined;
  }
  function isNil(obj) {
    return obj == null;
  }
  function isArray$5(obj) {
    return nativeToString$4.call(obj) === '[object Array]';
  }
  function isObject$2(obj) {
    return nativeToString$4.call(obj) === '[object Object]';
  }
  function isNumber$1(obj) {
    return nativeToString$4.call(obj) === '[object Number]';
  }

  /**
   * @param {any} obj
   *
   * @return {boolean}
   */
  function isFunction$2(obj) {
    const tag = nativeToString$4.call(obj);
    return tag === '[object Function]' || tag === '[object AsyncFunction]' || tag === '[object GeneratorFunction]' || tag === '[object AsyncGeneratorFunction]' || tag === '[object Proxy]';
  }
  function isString$4(obj) {
    return nativeToString$4.call(obj) === '[object String]';
  }

  /**
   * Return true, if target owns a property with the given key.
   *
   * @param {Object} target
   * @param {String} key
   *
   * @return {Boolean}
   */
  function has$3(target, key) {
    return !isNil(target) && nativeHasOwnProperty$3.call(target, key);
  }

  /**
   * @template T
   * @typedef { (
   *   ((e: T) => boolean) |
   *   ((e: T, idx: number) => boolean) |
   *   ((e: T, key: string) => boolean) |
   *   string |
   *   number
   * ) } Matcher
   */

  /**
   * @template T
   * @template U
   *
   * @typedef { (
   *   ((e: T) => U) | string | number
   * ) } Extractor
   */

  /**
   * @template T
   * @typedef { (val: T, key: any) => boolean } MatchFn
   */

  /**
   * @template T
   * @typedef { T[] } ArrayCollection
   */

  /**
   * @template T
   * @typedef { { [key: string]: T } } StringKeyValueCollection
   */

  /**
   * @template T
   * @typedef { { [key: number]: T } } NumberKeyValueCollection
   */

  /**
   * @template T
   * @typedef { StringKeyValueCollection<T> | NumberKeyValueCollection<T> } KeyValueCollection
   */

  /**
   * @template T
   * @typedef { KeyValueCollection<T> | ArrayCollection<T> } Collection
   */

  /**
   * Find element in collection.
   *
   * @template T
   * @param {Collection<T>} collection
   * @param {Matcher<T>} matcher
   *
   * @return {Object}
   */
  function find$2(collection, matcher) {
    const matchFn = toMatcher$1(matcher);
    let match;
    forEach$3(collection, function (val, key) {
      if (matchFn(val, key)) {
        match = val;
        return false;
      }
    });
    return match;
  }

  /**
   * Filter elements in collection.
   *
   * @template T
   * @param {Collection<T>} collection
   * @param {Matcher<T>} matcher
   *
   * @return {T[]} result
   */
  function filter$1(collection, matcher) {
    const matchFn = toMatcher$1(matcher);
    let result = [];
    forEach$3(collection, function (val, key) {
      if (matchFn(val, key)) {
        result.push(val);
      }
    });
    return result;
  }

  /**
   * Iterate over collection; returning something
   * (non-undefined) will stop iteration.
   *
   * @template T
   * @param {Collection<T>} collection
   * @param { ((item: T, idx: number) => (boolean|void)) | ((item: T, key: string) => (boolean|void)) } iterator
   *
   * @return {T} return result that stopped the iteration
   */
  function forEach$3(collection, iterator) {
    let val, result;
    if (isUndefined$5(collection)) {
      return;
    }
    const convertKey = isArray$5(collection) ? toNum$3 : identity$3;
    for (let key in collection) {
      if (has$3(collection, key)) {
        val = collection[key];
        result = iterator(val, convertKey(key));
        if (result === false) {
          return val;
        }
      }
    }
  }

  /**
   * Reduce collection, returning a single result.
   *
   * @template T
   * @template V
   *
   * @param {Collection<T>} collection
   * @param {(result: V, entry: T, index: any) => V} iterator
   * @param {V} result
   *
   * @return {V} result returned from last iterator
   */
  function reduce(collection, iterator, result) {
    forEach$3(collection, function (value, idx) {
      result = iterator(result, value, idx);
    });
    return result;
  }

  /**
   * Return true if every element in the collection
   * matches the criteria.
   *
   * @param  {Object|Array} collection
   * @param  {Function} matcher
   *
   * @return {Boolean}
   */
  function every(collection, matcher) {
    return !!reduce(collection, function (matches, val, key) {
      return matches && matcher(val, key);
    }, true);
  }

  /**
   * Transform a collection into another collection
   * by piping each member through the given fn.
   *
   * @param  {Object|Array}   collection
   * @param  {Function} fn
   *
   * @return {Array} transformed collection
   */
  function map$1(collection, fn) {
    let result = [];
    forEach$3(collection, function (val, key) {
      result.push(fn(val, key));
    });
    return result;
  }

  /**
   * Sort collection by criteria.
   *
   * @template T
   *
   * @param {Collection<T>} collection
   * @param {Extractor<T, number | string>} extractor
   *
   * @return {Array}
   */
  function sortBy(collection, extractor) {
    extractor = toExtractor(extractor);
    let sorted = [];
    forEach$3(collection, function (value, key) {
      let disc = extractor(value, key);
      let entry = {
        d: disc,
        v: value
      };
      for (var idx = 0; idx < sorted.length; idx++) {
        let {
          d
        } = sorted[idx];
        if (disc < d) {
          sorted.splice(idx, 0, entry);
          return;
        }
      }

      // not inserted, append (!)
      sorted.push(entry);
    });
    return map$1(sorted, e => e.v);
  }

  /**
   * Create an object pattern matcher.
   *
   * @example
   *
   * ```javascript
   * const matcher = matchPattern({ id: 1 });
   *
   * let element = find(elements, matcher);
   * ```
   *
   * @template T
   *
   * @param {T} pattern
   *
   * @return { (el: any) =>  boolean } matcherFn
   */
  function matchPattern(pattern) {
    return function (el) {
      return every(pattern, function (val, key) {
        return el[key] === val;
      });
    };
  }

  /**
   * @param {string | ((e: any) => any) } extractor
   *
   * @return { (e: any) => any }
   */
  function toExtractor(extractor) {
    /**
     * @satisfies { (e: any) => any }
     */
    return isFunction$2(extractor) ? extractor : e => {
      // @ts-ignore: just works
      return e[extractor];
    };
  }

  /**
   * @template T
   * @param {Matcher<T>} matcher
   *
   * @return {MatchFn<T>}
   */
  function toMatcher$1(matcher) {
    return isFunction$2(matcher) ? matcher : e => {
      return e === matcher;
    };
  }
  function identity$3(arg) {
    return arg;
  }
  function toNum$3(arg) {
    return Number(arg);
  }

  /* global setTimeout clearTimeout */

  /**
   * @typedef { {
   *   (...args: any[]): any;
   *   flush: () => void;
   *   cancel: () => void;
   * } } DebouncedFunction
   */

  /**
   * Debounce fn, calling it only once if the given time
   * elapsed between calls.
   *
   * Lodash-style the function exposes methods to `#clear`
   * and `#flush` to control internal behavior.
   *
   * @param  {Function} fn
   * @param  {Number} timeout
   *
   * @return {DebouncedFunction} debounced function
   */
  function debounce(fn, timeout) {
    let timer;
    let lastArgs;
    let lastThis;
    let lastNow;
    function fire(force) {
      let now = Date.now();
      let scheduledDiff = force ? 0 : lastNow + timeout - now;
      if (scheduledDiff > 0) {
        return schedule(scheduledDiff);
      }
      fn.apply(lastThis, lastArgs);
      clear();
    }
    function schedule(timeout) {
      timer = setTimeout(fire, timeout);
    }
    function clear() {
      if (timer) {
        clearTimeout(timer);
      }
      timer = lastNow = lastArgs = lastThis = undefined;
    }
    function flush() {
      if (timer) {
        fire(true);
      }
      clear();
    }

    /**
     * @type { DebouncedFunction }
     */
    function callback(...args) {
      lastNow = Date.now();
      lastArgs = args;
      lastThis = this;

      // ensure an execution is scheduled
      if (!timer) {
        schedule(timeout);
      }
    }
    callback.flush = flush;
    callback.cancel = clear;
    return callback;
  }

  /**
   * Throttle fn, calling at most once
   * in the given interval.
   *
   * @param  {Function} fn
   * @param  {Number} interval
   *
   * @return {Function} throttled function
   */
  function throttle(fn, interval) {
    let throttling = false;
    return function (...args) {
      if (throttling) {
        return;
      }
      fn(...args);
      throttling = true;
      setTimeout(() => {
        throttling = false;
      }, interval);
    };
  }

  /**
   * Bind function against target <this>.
   *
   * @param  {Function} fn
   * @param  {Object}   target
   *
   * @return {Function} bound function
   */
  function bind$3(fn, target) {
    return fn.bind(target);
  }

  /**
   * Convenience wrapper for `Object.assign`.
   *
   * @param {Object} target
   * @param {...Object} others
   *
   * @return {Object} the target
   */
  function assign$4(target, ...others) {
    return Object.assign(target, ...others);
  }

  var FN_REF = '__fn';
  var DEFAULT_PRIORITY$2 = 1000;
  var slice = Array.prototype.slice;

  /**
   * @typedef { {
   *   stopPropagation(): void;
   *   preventDefault(): void;
   *   cancelBubble: boolean;
   *   defaultPrevented: boolean;
   *   returnValue: any;
   * } } Event
   */

  /**
   * @template E
   *
   * @typedef { (event: E & Event, ...any) => any } EventBusEventCallback
   */

  /**
   * @typedef { {
   *  priority: number;
   *  next: EventBusListener | null;
   *  callback: EventBusEventCallback<any>;
   * } } EventBusListener
   */

  /**
   * A general purpose event bus.
   *
   * This component is used to communicate across a diagram instance.
   * Other parts of a diagram can use it to listen to and broadcast events.
   *
   *
   * ## Registering for Events
   *
   * The event bus provides the {@link EventBus#on} and {@link EventBus#once}
   * methods to register for events. {@link EventBus#off} can be used to
   * remove event registrations. Listeners receive an instance of {@link Event}
   * as the first argument. It allows them to hook into the event execution.
   *
   * ```javascript
   *
   * // listen for event
   * eventBus.on('foo', function(event) {
   *
   *   // access event type
   *   event.type; // 'foo'
   *
   *   // stop propagation to other listeners
   *   event.stopPropagation();
   *
   *   // prevent event default
   *   event.preventDefault();
   * });
   *
   * // listen for event with custom payload
   * eventBus.on('bar', function(event, payload) {
   *   console.log(payload);
   * });
   *
   * // listen for event returning value
   * eventBus.on('foobar', function(event) {
   *
   *   // stop event propagation + prevent default
   *   return false;
   *
   *   // stop event propagation + return custom result
   *   return {
   *     complex: 'listening result'
   *   };
   * });
   *
   *
   * // listen with custom priority (default=1000, higher is better)
   * eventBus.on('priorityfoo', 1500, function(event) {
   *   console.log('invoked first!');
   * });
   *
   *
   * // listen for event and pass the context (`this`)
   * eventBus.on('foobar', function(event) {
   *   this.foo();
   * }, this);
   * ```
   *
   *
   * ## Emitting Events
   *
   * Events can be emitted via the event bus using {@link EventBus#fire}.
   *
   * ```javascript
   *
   * // false indicates that the default action
   * // was prevented by listeners
   * if (eventBus.fire('foo') === false) {
   *   console.log('default has been prevented!');
   * };
   *
   *
   * // custom args + return value listener
   * eventBus.on('sum', function(event, a, b) {
   *   return a + b;
   * });
   *
   * // you can pass custom arguments + retrieve result values.
   * var sum = eventBus.fire('sum', 1, 2);
   * console.log(sum); // 3
   * ```
   *
   * @template [EventMap=null]
   */
  function EventBus() {
    /**
     * @type { Record<string, EventBusListener> }
     */
    this._listeners = {};

    // cleanup on destroy on lowest priority to allow
    // message passing until the bitter end
    this.on('diagram.destroy', 1, this._destroy, this);
  }

  /**
   * @overlord
   *
   * Register an event listener for events with the given name.
   *
   * The callback will be invoked with `event, ...additionalArguments`
   * that have been passed to {@link EventBus#fire}.
   *
   * Returning false from a listener will prevent the events default action
   * (if any is specified). To stop an event from being processed further in
   * other listeners execute {@link Event#stopPropagation}.
   *
   * Returning anything but `undefined` from a listener will stop the listener propagation.
   *
   * @template T
   *
   * @param {string|string[]} events to subscribe to
   * @param {number} [priority=1000] listen priority
   * @param {EventBusEventCallback<T>} callback
   * @param {any} [that] callback context
   */
  /**
   * Register an event listener for events with the given name.
   *
   * The callback will be invoked with `event, ...additionalArguments`
   * that have been passed to {@link EventBus#fire}.
   *
   * Returning false from a listener will prevent the events default action
   * (if any is specified). To stop an event from being processed further in
   * other listeners execute {@link Event#stopPropagation}.
   *
   * Returning anything but `undefined` from a listener will stop the listener propagation.
   *
   * @template {keyof EventMap} EventName
   *
   * @param {EventName} events to subscribe to
   * @param {number} [priority=1000] listen priority
   * @param {EventBusEventCallback<EventMap[EventName]>} callback
   * @param {any} [that] callback context
   */
  EventBus.prototype.on = function (events, priority, callback, that) {
    events = isArray$5(events) ? events : [events];
    if (isFunction$2(priority)) {
      that = callback;
      callback = priority;
      priority = DEFAULT_PRIORITY$2;
    }
    if (!isNumber$1(priority)) {
      throw new Error('priority must be a number');
    }
    var actualCallback = callback;
    if (that) {
      actualCallback = bind$3(callback, that);

      // make sure we remember and are able to remove
      // bound callbacks via {@link #off} using the original
      // callback
      actualCallback[FN_REF] = callback[FN_REF] || callback;
    }
    var self = this;
    events.forEach(function (e) {
      self._addListener(e, {
        priority: priority,
        callback: actualCallback,
        next: null
      });
    });
  };

  /**
   * @overlord
   *
   * Register an event listener that is called only once.
   *
   * @template T
   *
   * @param {string|string[]} events to subscribe to
   * @param {number} [priority=1000] the listen priority
   * @param {EventBusEventCallback<T>} callback
   * @param {any} [that] callback context
   */
  /**
   * Register an event listener that is called only once.
   *
   * @template {keyof EventMap} EventName
   *
   * @param {EventName} events to subscribe to
   * @param {number} [priority=1000] listen priority
   * @param {EventBusEventCallback<EventMap[EventName]>} callback
   * @param {any} [that] callback context
   */
  EventBus.prototype.once = function (events, priority, callback, that) {
    var self = this;
    if (isFunction$2(priority)) {
      that = callback;
      callback = priority;
      priority = DEFAULT_PRIORITY$2;
    }
    if (!isNumber$1(priority)) {
      throw new Error('priority must be a number');
    }
    function wrappedCallback() {
      wrappedCallback.__isTomb = true;
      var result = callback.apply(that, arguments);
      self.off(events, wrappedCallback);
      return result;
    }

    // make sure we remember and are able to remove
    // bound callbacks via {@link #off} using the original
    // callback
    wrappedCallback[FN_REF] = callback;
    this.on(events, priority, wrappedCallback);
  };

  /**
   * Removes event listeners by event and callback.
   *
   * If no callback is given, all listeners for a given event name are being removed.
   *
   * @param {string|string[]} events
   * @param {EventBusEventCallback} [callback]
   */
  EventBus.prototype.off = function (events, callback) {
    events = isArray$5(events) ? events : [events];
    var self = this;
    events.forEach(function (event) {
      self._removeListener(event, callback);
    });
  };

  /**
   * Create an event recognized be the event bus.
   *
   * @param {Object} data Event data.
   *
   * @return {Event} An event that will be recognized by the event bus.
   */
  EventBus.prototype.createEvent = function (data) {
    var event = new InternalEvent();
    event.init(data);
    return event;
  };

  /**
   * Fires an event.
   *
   * @example
   *
   * ```javascript
   * // fire event by name
   * events.fire('foo');
   *
   * // fire event object with nested type
   * var event = { type: 'foo' };
   * events.fire(event);
   *
   * // fire event with explicit type
   * var event = { x: 10, y: 20 };
   * events.fire('element.moved', event);
   *
   * // pass additional arguments to the event
   * events.on('foo', function(event, bar) {
   *   alert(bar);
   * });
   *
   * events.fire({ type: 'foo' }, 'I am bar!');
   * ```
   *
   * @param {string} [type] event type
   * @param {Object} [data] event or event data
   * @param {...any} [args] additional arguments the callback will be called with.
   *
   * @return {any} The return value. Will be set to `false` if the default was prevented.
   */
  EventBus.prototype.fire = function (type, data) {
    var event, firstListener, returnValue, args;
    args = slice.call(arguments);
    if (typeof type === 'object') {
      data = type;
      type = data.type;
    }
    if (!type) {
      throw new Error('no event type specified');
    }
    firstListener = this._listeners[type];
    if (!firstListener) {
      return;
    }

    // we make sure we fire instances of our home made
    // events here. We wrap them only once, though
    if (data instanceof InternalEvent) {
      // we are fine, we alread have an event
      event = data;
    } else {
      event = this.createEvent(data);
    }

    // ensure we pass the event as the first parameter
    args[0] = event;

    // original event type (in case we delegate)
    var originalType = event.type;

    // update event type before delegation
    if (type !== originalType) {
      event.type = type;
    }
    try {
      returnValue = this._invokeListeners(event, args, firstListener);
    } finally {
      // reset event type after delegation
      if (type !== originalType) {
        event.type = originalType;
      }
    }

    // set the return value to false if the event default
    // got prevented and no other return value exists
    if (returnValue === undefined && event.defaultPrevented) {
      returnValue = false;
    }
    return returnValue;
  };

  /**
   * Handle an error by firing an event.
   *
   * @param {Error} error The error to be handled.
   *
   * @return {boolean} Whether the error was handled.
   */
  EventBus.prototype.handleError = function (error) {
    return this.fire('error', {
      error: error
    }) === false;
  };
  EventBus.prototype._destroy = function () {
    this._listeners = {};
  };

  /**
   * @param {Event} event
   * @param {any[]} args
   * @param {EventBusListener} listener
   *
   * @return {any}
   */
  EventBus.prototype._invokeListeners = function (event, args, listener) {
    var returnValue;
    while (listener) {
      // handle stopped propagation
      if (event.cancelBubble) {
        break;
      }
      returnValue = this._invokeListener(event, args, listener);
      listener = listener.next;
    }
    return returnValue;
  };

  /**
   * @param {Event} event
   * @param {any[]} args
   * @param {EventBusListener} listener
   *
   * @return {any}
   */
  EventBus.prototype._invokeListener = function (event, args, listener) {
    var returnValue;
    if (listener.callback.__isTomb) {
      return returnValue;
    }
    try {
      // returning false prevents the default action
      returnValue = invokeFunction(listener.callback, args);

      // stop propagation on return value
      if (returnValue !== undefined) {
        event.returnValue = returnValue;
        event.stopPropagation();
      }

      // prevent default on return false
      if (returnValue === false) {
        event.preventDefault();
      }
    } catch (error) {
      if (!this.handleError(error)) {
        console.error('unhandled error in event listener', error);
        throw error;
      }
    }
    return returnValue;
  };

  /**
   * Add new listener with a certain priority to the list
   * of listeners (for the given event).
   *
   * The semantics of listener registration / listener execution are
   * first register, first serve: New listeners will always be inserted
   * after existing listeners with the same priority.
   *
   * Example: Inserting two listeners with priority 1000 and 1300
   *
   *    * before: [ 1500, 1500, 1000, 1000 ]
   *    * after: [ 1500, 1500, (new=1300), 1000, 1000, (new=1000) ]
   *
   * @param {string} event
   * @param {EventBusListener} newListener
   */
  EventBus.prototype._addListener = function (event, newListener) {
    var listener = this._getListeners(event),
      previousListener;

    // no prior listeners
    if (!listener) {
      this._setListeners(event, newListener);
      return;
    }

    // ensure we order listeners by priority from
    // 0 (high) to n > 0 (low)
    while (listener) {
      if (listener.priority < newListener.priority) {
        newListener.next = listener;
        if (previousListener) {
          previousListener.next = newListener;
        } else {
          this._setListeners(event, newListener);
        }
        return;
      }
      previousListener = listener;
      listener = listener.next;
    }

    // add new listener to back
    previousListener.next = newListener;
  };

  /**
   * @param {string} name
   *
   * @return {EventBusListener}
   */
  EventBus.prototype._getListeners = function (name) {
    return this._listeners[name];
  };

  /**
   * @param {string} name
   * @param {EventBusListener} listener
   */
  EventBus.prototype._setListeners = function (name, listener) {
    this._listeners[name] = listener;
  };
  EventBus.prototype._removeListener = function (event, callback) {
    var listener = this._getListeners(event),
      nextListener,
      previousListener,
      listenerCallback;
    if (!callback) {
      // clear listeners
      this._setListeners(event, null);
      return;
    }
    while (listener) {
      nextListener = listener.next;
      listenerCallback = listener.callback;
      if (listenerCallback === callback || listenerCallback[FN_REF] === callback) {
        if (previousListener) {
          previousListener.next = nextListener;
        } else {
          // new first listener
          this._setListeners(event, nextListener);
        }
      }
      previousListener = listener;
      listener = nextListener;
    }
  };

  /**
   * A event that is emitted via the event bus.
   */
  function InternalEvent() {}
  InternalEvent.prototype.stopPropagation = function () {
    this.cancelBubble = true;
  };
  InternalEvent.prototype.preventDefault = function () {
    this.defaultPrevented = true;
  };
  InternalEvent.prototype.init = function (data) {
    assign$4(this, data || {});
  };

  /**
   * Invoke function. Be fast...
   *
   * @param {Function} fn
   * @param {any[]} args
   *
   * @return {any}
   */
  function invokeFunction(fn, args) {
    return fn.apply(null, args);
  }

  /**
   * Flatten array, one level deep.
   *
   * @param {Array<?>} arr
   *
   * @return {Array<?>}
   */
  var nativeToString$3 = Object.prototype.toString;
  function isString$3(obj) {
    return nativeToString$3.call(obj) === '[object String]';
  }
  function _extends$2() {
    _extends$2 = Object.assign || function (target) {
      for (var i = 1; i < arguments.length; i++) {
        var source = arguments[i];
        for (var key in source) {
          if (Object.prototype.hasOwnProperty.call(source, key)) {
            target[key] = source[key];
          }
        }
      }
      return target;
    };
    return _extends$2.apply(this, arguments);
  }

  /**
   * Convenience wrapper for `Object.assign`.
   *
   * @param {Object} target
   * @param {...Object} others
   *
   * @return {Object} the target
   */

  function assign$3(target) {
    for (var _len = arguments.length, others = new Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
      others[_key - 1] = arguments[_key];
    }
    return _extends$2.apply(void 0, [target].concat(others));
  }

  /**
   * Flatten array, one level deep.
   *
   * @param {Array<?>} arr
   *
   * @return {Array<?>}
   */
  var nativeToString$2 = Object.prototype.toString;
  var nativeHasOwnProperty$2 = Object.prototype.hasOwnProperty;
  function isUndefined$4(obj) {
    return obj === undefined;
  }
  function isArray$4(obj) {
    return nativeToString$2.call(obj) === '[object Array]';
  }
  function isObject$1(obj) {
    return nativeToString$2.call(obj) === '[object Object]';
  }
  function isString$2(obj) {
    return nativeToString$2.call(obj) === '[object String]';
  }
  /**
   * Return true, if target owns a property with the given key.
   *
   * @param {Object} target
   * @param {String} key
   *
   * @return {Boolean}
   */

  function has$2(target, key) {
    return nativeHasOwnProperty$2.call(target, key);
  }
  /**
   * Iterate over collection; returning something
   * (non-undefined) will stop iteration.
   *
   * @param  {Array|Object} collection
   * @param  {Function} iterator
   *
   * @return {Object} return result that stopped the iteration
   */

  function forEach$2(collection, iterator) {
    var val, result;
    if (isUndefined$4(collection)) {
      return;
    }
    var convertKey = isArray$4(collection) ? toNum$2 : identity$2;
    for (var key in collection) {
      if (has$2(collection, key)) {
        val = collection[key];
        result = iterator(val, convertKey(key));
        if (result === false) {
          return val;
        }
      }
    }
  }
  function identity$2(arg) {
    return arg;
  }
  function toNum$2(arg) {
    return Number(arg);
  }
  /**
   * Bind function against target <this>.
   *
   * @param  {Function} fn
   * @param  {Object}   target
   *
   * @return {Function} bound function
   */

  function bind$2(fn, target) {
    return fn.bind(target);
  }
  function _extends$1() {
    _extends$1 = Object.assign || function (target) {
      for (var i = 1; i < arguments.length; i++) {
        var source = arguments[i];
        for (var key in source) {
          if (Object.prototype.hasOwnProperty.call(source, key)) {
            target[key] = source[key];
          }
        }
      }
      return target;
    };
    return _extends$1.apply(this, arguments);
  }

  /**
   * Convenience wrapper for `Object.assign`.
   *
   * @param {Object} target
   * @param {...Object} others
   *
   * @return {Object} the target
   */

  function assign$2(target) {
    for (var _len = arguments.length, others = new Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
      others[_key - 1] = arguments[_key];
    }
    return _extends$1.apply(void 0, [target].concat(others));
  }
  /**
   * Pick given properties from the target object.
   *
   * @param {Object} target
   * @param {Array} properties
   *
   * @return {Object} target
   */

  function pick(target, properties) {
    var result = {};
    var obj = Object(target);
    forEach$2(properties, function (prop) {
      if (prop in obj) {
        result[prop] = target[prop];
      }
    });
    return result;
  }

  /**
   * Moddle base element.
   */
  function Base$1() {}
  Base$1.prototype.get = function (name) {
    return this.$model.properties.get(this, name);
  };
  Base$1.prototype.set = function (name, value) {
    this.$model.properties.set(this, name, value);
  };

  /**
   * A model element factory.
   *
   * @param {Moddle} model
   * @param {Properties} properties
   */
  function Factory(model, properties) {
    this.model = model;
    this.properties = properties;
  }
  Factory.prototype.createType = function (descriptor) {
    var model = this.model;
    var props = this.properties,
      prototype = Object.create(Base$1.prototype);

    // initialize default values
    forEach$2(descriptor.properties, function (p) {
      if (!p.isMany && p.default !== undefined) {
        prototype[p.name] = p.default;
      }
    });
    props.defineModel(prototype, model);
    props.defineDescriptor(prototype, descriptor);
    var name = descriptor.ns.name;

    /**
     * The new type constructor
     */
    function ModdleElement(attrs) {
      props.define(this, '$type', {
        value: name,
        enumerable: true
      });
      props.define(this, '$attrs', {
        value: {}
      });
      props.define(this, '$parent', {
        writable: true
      });
      forEach$2(attrs, bind$2(function (val, key) {
        this.set(key, val);
      }, this));
    }
    ModdleElement.prototype = prototype;
    ModdleElement.hasType = prototype.$instanceOf = this.model.hasType;

    // static links
    props.defineModel(ModdleElement, model);
    props.defineDescriptor(ModdleElement, descriptor);
    return ModdleElement;
  };

  /**
   * Built-in moddle types
   */
  var BUILTINS = {
    String: true,
    Boolean: true,
    Integer: true,
    Real: true,
    Element: true
  };

  /**
   * Converters for built in types from string representations
   */
  var TYPE_CONVERTERS = {
    String: function (s) {
      return s;
    },
    Boolean: function (s) {
      return s === 'true';
    },
    Integer: function (s) {
      return parseInt(s, 10);
    },
    Real: function (s) {
      return parseFloat(s);
    }
  };

  /**
   * Convert a type to its real representation
   */
  function coerceType(type, value) {
    var converter = TYPE_CONVERTERS[type];
    if (converter) {
      return converter(value);
    } else {
      return value;
    }
  }

  /**
   * Return whether the given type is built-in
   */
  function isBuiltIn(type) {
    return !!BUILTINS[type];
  }

  /**
   * Return whether the given type is simple
   */
  function isSimple(type) {
    return !!TYPE_CONVERTERS[type];
  }

  /**
   * Parses a namespaced attribute name of the form (ns:)localName to an object,
   * given a default prefix to assume in case no explicit namespace is given.
   *
   * @param {String} name
   * @param {String} [defaultPrefix] the default prefix to take, if none is present.
   *
   * @return {Object} the parsed name
   */
  function parseName(name, defaultPrefix) {
    var parts = name.split(/:/),
      localName,
      prefix;

    // no prefix (i.e. only local name)
    if (parts.length === 1) {
      localName = name;
      prefix = defaultPrefix;
    } else
      // prefix + local name
      if (parts.length === 2) {
        localName = parts[1];
        prefix = parts[0];
      } else {
        throw new Error('expected <prefix:localName> or <localName>, got ' + name);
      }
    name = (prefix ? prefix + ':' : '') + localName;
    return {
      name: name,
      prefix: prefix,
      localName: localName
    };
  }

  /**
   * A utility to build element descriptors.
   */
  function DescriptorBuilder(nameNs) {
    this.ns = nameNs;
    this.name = nameNs.name;
    this.allTypes = [];
    this.allTypesByName = {};
    this.properties = [];
    this.propertiesByName = {};
  }
  DescriptorBuilder.prototype.build = function () {
    return pick(this, ['ns', 'name', 'allTypes', 'allTypesByName', 'properties', 'propertiesByName', 'bodyProperty', 'idProperty']);
  };

  /**
   * Add property at given index.
   *
   * @param {Object} p
   * @param {Number} [idx]
   * @param {Boolean} [validate=true]
   */
  DescriptorBuilder.prototype.addProperty = function (p, idx, validate) {
    if (typeof idx === 'boolean') {
      validate = idx;
      idx = undefined;
    }
    this.addNamedProperty(p, validate !== false);
    var properties = this.properties;
    if (idx !== undefined) {
      properties.splice(idx, 0, p);
    } else {
      properties.push(p);
    }
  };
  DescriptorBuilder.prototype.replaceProperty = function (oldProperty, newProperty, replace) {
    var oldNameNs = oldProperty.ns;
    var props = this.properties,
      propertiesByName = this.propertiesByName,
      rename = oldProperty.name !== newProperty.name;
    if (oldProperty.isId) {
      if (!newProperty.isId) {
        throw new Error('property <' + newProperty.ns.name + '> must be id property ' + 'to refine <' + oldProperty.ns.name + '>');
      }
      this.setIdProperty(newProperty, false);
    }
    if (oldProperty.isBody) {
      if (!newProperty.isBody) {
        throw new Error('property <' + newProperty.ns.name + '> must be body property ' + 'to refine <' + oldProperty.ns.name + '>');
      }

      // TODO: Check compatibility
      this.setBodyProperty(newProperty, false);
    }

    // validate existence and get location of old property
    var idx = props.indexOf(oldProperty);
    if (idx === -1) {
      throw new Error('property <' + oldNameNs.name + '> not found in property list');
    }

    // remove old property
    props.splice(idx, 1);

    // replacing the named property is intentional
    //
    //  * validate only if this is a "rename" operation
    //  * add at specific index unless we "replace"
    //
    this.addProperty(newProperty, replace ? undefined : idx, rename);

    // make new property available under old name
    propertiesByName[oldNameNs.name] = propertiesByName[oldNameNs.localName] = newProperty;
  };
  DescriptorBuilder.prototype.redefineProperty = function (p, targetPropertyName, replace) {
    var nsPrefix = p.ns.prefix;
    var parts = targetPropertyName.split('#');
    var name = parseName(parts[0], nsPrefix);
    var attrName = parseName(parts[1], name.prefix).name;
    var redefinedProperty = this.propertiesByName[attrName];
    if (!redefinedProperty) {
      throw new Error('refined property <' + attrName + '> not found');
    } else {
      this.replaceProperty(redefinedProperty, p, replace);
    }
    delete p.redefines;
  };
  DescriptorBuilder.prototype.addNamedProperty = function (p, validate) {
    var ns = p.ns,
      propsByName = this.propertiesByName;
    if (validate) {
      this.assertNotDefined(p, ns.name);
      this.assertNotDefined(p, ns.localName);
    }
    propsByName[ns.name] = propsByName[ns.localName] = p;
  };
  DescriptorBuilder.prototype.removeNamedProperty = function (p) {
    var ns = p.ns,
      propsByName = this.propertiesByName;
    delete propsByName[ns.name];
    delete propsByName[ns.localName];
  };
  DescriptorBuilder.prototype.setBodyProperty = function (p, validate) {
    if (validate && this.bodyProperty) {
      throw new Error('body property defined multiple times ' + '(<' + this.bodyProperty.ns.name + '>, <' + p.ns.name + '>)');
    }
    this.bodyProperty = p;
  };
  DescriptorBuilder.prototype.setIdProperty = function (p, validate) {
    if (validate && this.idProperty) {
      throw new Error('id property defined multiple times ' + '(<' + this.idProperty.ns.name + '>, <' + p.ns.name + '>)');
    }
    this.idProperty = p;
  };
  DescriptorBuilder.prototype.assertNotDefined = function (p, name) {
    var propertyName = p.name,
      definedProperty = this.propertiesByName[propertyName];
    if (definedProperty) {
      throw new Error('property <' + propertyName + '> already defined; ' + 'override of <' + definedProperty.definedBy.ns.name + '#' + definedProperty.ns.name + '> by ' + '<' + p.definedBy.ns.name + '#' + p.ns.name + '> not allowed without redefines');
    }
  };
  DescriptorBuilder.prototype.hasProperty = function (name) {
    return this.propertiesByName[name];
  };
  DescriptorBuilder.prototype.addTrait = function (t, inherited) {
    var typesByName = this.allTypesByName,
      types = this.allTypes;
    var typeName = t.name;
    if (typeName in typesByName) {
      return;
    }
    forEach$2(t.properties, bind$2(function (p) {
      // clone property to allow extensions
      p = assign$2({}, p, {
        name: p.ns.localName,
        inherited: inherited
      });
      Object.defineProperty(p, 'definedBy', {
        value: t
      });
      var replaces = p.replaces,
        redefines = p.redefines;

      // add replace/redefine support
      if (replaces || redefines) {
        this.redefineProperty(p, replaces || redefines, replaces);
      } else {
        if (p.isBody) {
          this.setBodyProperty(p);
        }
        if (p.isId) {
          this.setIdProperty(p);
        }
        this.addProperty(p);
      }
    }, this));
    types.push(t);
    typesByName[typeName] = t;
  };

  /**
   * A registry of Moddle packages.
   *
   * @param {Array<Package>} packages
   * @param {Properties} properties
   */
  function Registry(packages, properties) {
    this.packageMap = {};
    this.typeMap = {};
    this.packages = [];
    this.properties = properties;
    forEach$2(packages, bind$2(this.registerPackage, this));
  }
  Registry.prototype.getPackage = function (uriOrPrefix) {
    return this.packageMap[uriOrPrefix];
  };
  Registry.prototype.getPackages = function () {
    return this.packages;
  };
  Registry.prototype.registerPackage = function (pkg) {
    // copy package
    pkg = assign$2({}, pkg);
    var pkgMap = this.packageMap;
    ensureAvailable(pkgMap, pkg, 'prefix');
    ensureAvailable(pkgMap, pkg, 'uri');

    // register types
    forEach$2(pkg.types, bind$2(function (descriptor) {
      this.registerType(descriptor, pkg);
    }, this));
    pkgMap[pkg.uri] = pkgMap[pkg.prefix] = pkg;
    this.packages.push(pkg);
  };

  /**
   * Register a type from a specific package with us
   */
  Registry.prototype.registerType = function (type, pkg) {
    type = assign$2({}, type, {
      superClass: (type.superClass || []).slice(),
      extends: (type.extends || []).slice(),
      properties: (type.properties || []).slice(),
      meta: assign$2(type.meta || {})
    });
    var ns = parseName(type.name, pkg.prefix),
      name = ns.name,
      propertiesByName = {};

    // parse properties
    forEach$2(type.properties, bind$2(function (p) {
      // namespace property names
      var propertyNs = parseName(p.name, ns.prefix),
        propertyName = propertyNs.name;

      // namespace property types
      if (!isBuiltIn(p.type)) {
        p.type = parseName(p.type, propertyNs.prefix).name;
      }
      assign$2(p, {
        ns: propertyNs,
        name: propertyName
      });
      propertiesByName[propertyName] = p;
    }, this));

    // update ns + name
    assign$2(type, {
      ns: ns,
      name: name,
      propertiesByName: propertiesByName
    });
    forEach$2(type.extends, bind$2(function (extendsName) {
      var extended = this.typeMap[extendsName];
      extended.traits = extended.traits || [];
      extended.traits.push(name);
    }, this));

    // link to package
    this.definePackage(type, pkg);

    // register
    this.typeMap[name] = type;
  };

  /**
   * Traverse the type hierarchy from bottom to top,
   * calling iterator with (type, inherited) for all elements in
   * the inheritance chain.
   *
   * @param {Object} nsName
   * @param {Function} iterator
   * @param {Boolean} [trait=false]
   */
  Registry.prototype.mapTypes = function (nsName, iterator, trait) {
    var type = isBuiltIn(nsName.name) ? {
      name: nsName.name
    } : this.typeMap[nsName.name];
    var self = this;

    /**
     * Traverse the selected trait.
     *
     * @param {String} cls
     */
    function traverseTrait(cls) {
      return traverseSuper(cls, true);
    }

    /**
     * Traverse the selected super type or trait
     *
     * @param {String} cls
     * @param {Boolean} [trait=false]
     */
    function traverseSuper(cls, trait) {
      var parentNs = parseName(cls, isBuiltIn(cls) ? '' : nsName.prefix);
      self.mapTypes(parentNs, iterator, trait);
    }
    if (!type) {
      throw new Error('unknown type <' + nsName.name + '>');
    }
    forEach$2(type.superClass, trait ? traverseTrait : traverseSuper);

    // call iterator with (type, inherited=!trait)
    iterator(type, !trait);
    forEach$2(type.traits, traverseTrait);
  };

  /**
   * Returns the effective descriptor for a type.
   *
   * @param  {String} type the namespaced name (ns:localName) of the type
   *
   * @return {Descriptor} the resulting effective descriptor
   */
  Registry.prototype.getEffectiveDescriptor = function (name) {
    var nsName = parseName(name);
    var builder = new DescriptorBuilder(nsName);
    this.mapTypes(nsName, function (type, inherited) {
      builder.addTrait(type, inherited);
    });
    var descriptor = builder.build();

    // define package link
    this.definePackage(descriptor, descriptor.allTypes[descriptor.allTypes.length - 1].$pkg);
    return descriptor;
  };
  Registry.prototype.definePackage = function (target, pkg) {
    this.properties.define(target, '$pkg', {
      value: pkg
    });
  };

  ///////// helpers ////////////////////////////

  function ensureAvailable(packageMap, pkg, identifierKey) {
    var value = pkg[identifierKey];
    if (value in packageMap) {
      throw new Error('package with ' + identifierKey + ' <' + value + '> already defined');
    }
  }

  /**
   * A utility that gets and sets properties of model elements.
   *
   * @param {Model} model
   */
  function Properties(model) {
    this.model = model;
  }

  /**
   * Sets a named property on the target element.
   * If the value is undefined, the property gets deleted.
   *
   * @param {Object} target
   * @param {String} name
   * @param {Object} value
   */
  Properties.prototype.set = function (target, name, value) {
    if (!isString$2(name) || !name.length) {
      throw new TypeError('property name must be a non-empty string');
    }
    var property = this.model.getPropertyDescriptor(target, name);
    var propertyName = property && property.name;
    if (isUndefined$3(value)) {
      // unset the property, if the specified value is undefined;
      // delete from $attrs (for extensions) or the target itself
      if (property) {
        delete target[propertyName];
      } else {
        delete target.$attrs[name];
      }
    } else {
      // set the property, defining well defined properties on the fly
      // or simply updating them in target.$attrs (for extensions)
      if (property) {
        if (propertyName in target) {
          target[propertyName] = value;
        } else {
          defineProperty$2(target, property, value);
        }
      } else {
        target.$attrs[name] = value;
      }
    }
  };

  /**
   * Returns the named property of the given element
   *
   * @param  {Object} target
   * @param  {String} name
   *
   * @return {Object}
   */
  Properties.prototype.get = function (target, name) {
    var property = this.model.getPropertyDescriptor(target, name);
    if (!property) {
      return target.$attrs[name];
    }
    var propertyName = property.name;

    // check if access to collection property and lazily initialize it
    if (!target[propertyName] && property.isMany) {
      defineProperty$2(target, property, []);
    }
    return target[propertyName];
  };

  /**
   * Define a property on the target element
   *
   * @param  {Object} target
   * @param  {String} name
   * @param  {Object} options
   */
  Properties.prototype.define = function (target, name, options) {
    if (!options.writable) {
      var value = options.value;

      // use getters for read-only variables to support ES6 proxies
      // cf. https://github.com/bpmn-io/internal-docs/issues/386
      options = assign$2({}, options, {
        get: function () {
          return value;
        }
      });
      delete options.value;
    }
    Object.defineProperty(target, name, options);
  };

  /**
   * Define the descriptor for an element
   */
  Properties.prototype.defineDescriptor = function (target, descriptor) {
    this.define(target, '$descriptor', {
      value: descriptor
    });
  };

  /**
   * Define the model for an element
   */
  Properties.prototype.defineModel = function (target, model) {
    this.define(target, '$model', {
      value: model
    });
  };
  function isUndefined$3(val) {
    return typeof val === 'undefined';
  }
  function defineProperty$2(target, property, value) {
    Object.defineProperty(target, property.name, {
      enumerable: !property.isReference,
      writable: true,
      value: value,
      configurable: true
    });
  }

  //// Moddle implementation /////////////////////////////////////////////////

  /**
   * @class Moddle
   *
   * A model that can be used to create elements of a specific type.
   *
   * @example
   *
   * var Moddle = require('moddle');
   *
   * var pkg = {
   *   name: 'mypackage',
   *   prefix: 'my',
   *   types: [
   *     { name: 'Root' }
   *   ]
   * };
   *
   * var moddle = new Moddle([pkg]);
   *
   * @param {Array<Package>} packages the packages to contain
   */
  function Moddle(packages) {
    this.properties = new Properties(this);
    this.factory = new Factory(this, this.properties);
    this.registry = new Registry(packages, this.properties);
    this.typeCache = {};
  }

  /**
   * Create an instance of the specified type.
   *
   * @method Moddle#create
   *
   * @example
   *
   * var foo = moddle.create('my:Foo');
   * var bar = moddle.create('my:Bar', { id: 'BAR_1' });
   *
   * @param  {String|Object} descriptor the type descriptor or name know to the model
   * @param  {Object} attrs   a number of attributes to initialize the model instance with
   * @return {Object}         model instance
   */
  Moddle.prototype.create = function (descriptor, attrs) {
    var Type = this.getType(descriptor);
    if (!Type) {
      throw new Error('unknown type <' + descriptor + '>');
    }
    return new Type(attrs);
  };

  /**
   * Returns the type representing a given descriptor
   *
   * @method Moddle#getType
   *
   * @example
   *
   * var Foo = moddle.getType('my:Foo');
   * var foo = new Foo({ 'id' : 'FOO_1' });
   *
   * @param  {String|Object} descriptor the type descriptor or name know to the model
   * @return {Object}         the type representing the descriptor
   */
  Moddle.prototype.getType = function (descriptor) {
    var cache = this.typeCache;
    var name = isString$2(descriptor) ? descriptor : descriptor.ns.name;
    var type = cache[name];
    if (!type) {
      descriptor = this.registry.getEffectiveDescriptor(name);
      type = cache[name] = this.factory.createType(descriptor);
    }
    return type;
  };

  /**
   * Creates an any-element type to be used within model instances.
   *
   * This can be used to create custom elements that lie outside the meta-model.
   * The created element contains all the meta-data required to serialize it
   * as part of meta-model elements.
   *
   * @method Moddle#createAny
   *
   * @example
   *
   * var foo = moddle.createAny('vendor:Foo', 'http://vendor', {
   *   value: 'bar'
   * });
   *
   * var container = moddle.create('my:Container', 'http://my', {
   *   any: [ foo ]
   * });
   *
   * // go ahead and serialize the stuff
   *
   *
   * @param  {String} name  the name of the element
   * @param  {String} nsUri the namespace uri of the element
   * @param  {Object} [properties] a map of properties to initialize the instance with
   * @return {Object} the any type instance
   */
  Moddle.prototype.createAny = function (name, nsUri, properties) {
    var nameNs = parseName(name);
    var element = {
      $type: name,
      $instanceOf: function (type) {
        return type === this.$type;
      }
    };
    var descriptor = {
      name: name,
      isGeneric: true,
      ns: {
        prefix: nameNs.prefix,
        localName: nameNs.localName,
        uri: nsUri
      }
    };
    this.properties.defineDescriptor(element, descriptor);
    this.properties.defineModel(element, this);
    this.properties.define(element, '$parent', {
      enumerable: false,
      writable: true
    });
    this.properties.define(element, '$instanceOf', {
      enumerable: false,
      writable: true
    });
    forEach$2(properties, function (a, key) {
      if (isObject$1(a) && a.value !== undefined) {
        element[a.name] = a.value;
      } else {
        element[key] = a;
      }
    });
    return element;
  };

  /**
   * Returns a registered package by uri or prefix
   *
   * @return {Object} the package
   */
  Moddle.prototype.getPackage = function (uriOrPrefix) {
    return this.registry.getPackage(uriOrPrefix);
  };

  /**
   * Returns a snapshot of all known packages
   *
   * @return {Object} the package
   */
  Moddle.prototype.getPackages = function () {
    return this.registry.getPackages();
  };

  /**
   * Returns the descriptor for an element
   */
  Moddle.prototype.getElementDescriptor = function (element) {
    return element.$descriptor;
  };

  /**
   * Returns true if the given descriptor or instance
   * represents the given type.
   *
   * May be applied to this, if element is omitted.
   */
  Moddle.prototype.hasType = function (element, type) {
    if (type === undefined) {
      type = element;
      element = this;
    }
    var descriptor = element.$model.getElementDescriptor(element);
    return type in descriptor.allTypesByName;
  };

  /**
   * Returns the descriptor of an elements named property
   */
  Moddle.prototype.getPropertyDescriptor = function (element, property) {
    return this.getElementDescriptor(element).propertiesByName[property];
  };

  /**
   * Returns a mapped type's descriptor
   */
  Moddle.prototype.getTypeDescriptor = function (type) {
    return this.registry.typeMap[type];
  };

  /**
   * Flatten array, one level deep.
   *
   * @param {Array<?>} arr
   *
   * @return {Array<?>}
   */
  var nativeToString$1 = Object.prototype.toString;
  var nativeHasOwnProperty$1 = Object.prototype.hasOwnProperty;
  function isUndefined$2(obj) {
    return obj === undefined;
  }
  function isArray$3(obj) {
    return nativeToString$1.call(obj) === '[object Array]';
  }
  function isFunction$1(obj) {
    var tag = nativeToString$1.call(obj);
    return tag === '[object Function]' || tag === '[object AsyncFunction]' || tag === '[object GeneratorFunction]' || tag === '[object AsyncGeneratorFunction]' || tag === '[object Proxy]';
  }
  function isString$1(obj) {
    return nativeToString$1.call(obj) === '[object String]';
  }
  /**
   * Return true, if target owns a property with the given key.
   *
   * @param {Object} target
   * @param {String} key
   *
   * @return {Boolean}
   */

  function has$1(target, key) {
    return nativeHasOwnProperty$1.call(target, key);
  }

  /**
   * Find element in collection.
   *
   * @param  {Array|Object} collection
   * @param  {Function|Object} matcher
   *
   * @return {Object}
   */

  function find$1(collection, matcher) {
    matcher = toMatcher(matcher);
    var match;
    forEach$1(collection, function (val, key) {
      if (matcher(val, key)) {
        match = val;
        return false;
      }
    });
    return match;
  }
  /**
   * Find element index in collection.
   *
   * @param  {Array|Object} collection
   * @param  {Function} matcher
   *
   * @return {Object}
   */

  function findIndex(collection, matcher) {
    matcher = toMatcher(matcher);
    var idx = isArray$3(collection) ? -1 : undefined;
    forEach$1(collection, function (val, key) {
      if (matcher(val, key)) {
        idx = key;
        return false;
      }
    });
    return idx;
  }
  /**
   * Find element in collection.
   *
   * @param  {Array|Object} collection
   * @param  {Function} matcher
   *
   * @return {Array} result
   */

  function filter(collection, matcher) {
    var result = [];
    forEach$1(collection, function (val, key) {
      if (matcher(val, key)) {
        result.push(val);
      }
    });
    return result;
  }
  /**
   * Iterate over collection; returning something
   * (non-undefined) will stop iteration.
   *
   * @param  {Array|Object} collection
   * @param  {Function} iterator
   *
   * @return {Object} return result that stopped the iteration
   */

  function forEach$1(collection, iterator) {
    var val, result;
    if (isUndefined$2(collection)) {
      return;
    }
    var convertKey = isArray$3(collection) ? toNum$1 : identity$1;
    for (var key in collection) {
      if (has$1(collection, key)) {
        val = collection[key];
        result = iterator(val, convertKey(key));
        if (result === false) {
          return val;
        }
      }
    }
  }
  function toMatcher(matcher) {
    return isFunction$1(matcher) ? matcher : function (e) {
      return e === matcher;
    };
  }
  function identity$1(arg) {
    return arg;
  }
  function toNum$1(arg) {
    return Number(arg);
  }
  function _extends() {
    _extends = Object.assign || function (target) {
      for (var i = 1; i < arguments.length; i++) {
        var source = arguments[i];
        for (var key in source) {
          if (Object.prototype.hasOwnProperty.call(source, key)) {
            target[key] = source[key];
          }
        }
      }
      return target;
    };
    return _extends.apply(this, arguments);
  }

  /**
   * Convenience wrapper for `Object.assign`.
   *
   * @param {Object} target
   * @param {...Object} others
   *
   * @return {Object} the target
   */

  function assign$1(target) {
    for (var _len = arguments.length, others = new Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
      others[_key - 1] = arguments[_key];
    }
    return _extends.apply(void 0, [target].concat(others));
  }

  var fromCharCode = String.fromCharCode;
  var hasOwnProperty$1 = Object.prototype.hasOwnProperty;
  var ENTITY_PATTERN = /&#(\d+);|&#x([0-9a-f]+);|&(\w+);/ig;
  var ENTITY_MAPPING = {
    'amp': '&',
    'apos': '\'',
    'gt': '>',
    'lt': '<',
    'quot': '"'
  };

  // map UPPERCASE variants of supported special chars
  Object.keys(ENTITY_MAPPING).forEach(function (k) {
    ENTITY_MAPPING[k.toUpperCase()] = ENTITY_MAPPING[k];
  });
  function replaceEntities(_, d, x, z) {
    // reserved names, i.e. &nbsp;
    if (z) {
      if (hasOwnProperty$1.call(ENTITY_MAPPING, z)) {
        return ENTITY_MAPPING[z];
      } else {
        // fall back to original value
        return '&' + z + ';';
      }
    }

    // decimal encoded char
    if (d) {
      return fromCharCode(d);
    }

    // hex encoded char
    return fromCharCode(parseInt(x, 16));
  }

  /**
   * A basic entity decoder that can decode a minimal
   * sub-set of reserved names (&amp;) as well as
   * hex (&#xaaf;) and decimal (&#1231;) encoded characters.
   *
   * @param {string} str
   *
   * @return {string} decoded string
   */
  function decodeEntities(s) {
    if (s.length > 3 && s.indexOf('&') !== -1) {
      return s.replace(ENTITY_PATTERN, replaceEntities);
    }
    return s;
  }
  var XSI_URI = 'http://www.w3.org/2001/XMLSchema-instance';
  var XSI_PREFIX = 'xsi';
  var XSI_TYPE$1 = 'xsi:type';
  var NON_WHITESPACE_OUTSIDE_ROOT_NODE = 'non-whitespace outside of root node';
  function error$1(msg) {
    return new Error(msg);
  }
  function missingNamespaceForPrefix(prefix) {
    return 'missing namespace for prefix <' + prefix + '>';
  }
  function getter(getFn) {
    return {
      'get': getFn,
      'enumerable': true
    };
  }
  function cloneNsMatrix(nsMatrix) {
    var clone = {},
      key;
    for (key in nsMatrix) {
      clone[key] = nsMatrix[key];
    }
    return clone;
  }
  function uriPrefix(prefix) {
    return prefix + '$uri';
  }
  function buildNsMatrix(nsUriToPrefix) {
    var nsMatrix = {},
      uri,
      prefix;
    for (uri in nsUriToPrefix) {
      prefix = nsUriToPrefix[uri];
      nsMatrix[prefix] = prefix;
      nsMatrix[uriPrefix(prefix)] = uri;
    }
    return nsMatrix;
  }
  function noopGetContext() {
    return {
      'line': 0,
      'column': 0
    };
  }
  function throwFunc(err) {
    throw err;
  }

  /**
   * Creates a new parser with the given options.
   *
   * @constructor
   *
   * @param  {!Object<string, ?>=} options
   */
  function Parser$1(options) {
    if (!this) {
      return new Parser$1(options);
    }
    var proxy = options && options['proxy'];
    var onText,
      onOpenTag,
      onCloseTag,
      onCDATA,
      onError = throwFunc,
      onWarning,
      onComment,
      onQuestion,
      onAttention;
    var getContext = noopGetContext;

    /**
     * Do we need to parse the current elements attributes for namespaces?
     *
     * @type {boolean}
     */
    var maybeNS = false;

    /**
     * Do we process namespaces at all?
     *
     * @type {boolean}
     */
    var isNamespace = false;

    /**
     * The caught error returned on parse end
     *
     * @type {Error}
     */
    var returnError = null;

    /**
     * Should we stop parsing?
     *
     * @type {boolean}
     */
    var parseStop = false;

    /**
     * A map of { uri: prefix } used by the parser.
     *
     * This map will ensure we can normalize prefixes during processing;
     * for each uri, only one prefix will be exposed to the handlers.
     *
     * @type {!Object<string, string>}}
     */
    var nsUriToPrefix;

    /**
     * Handle parse error.
     *
     * @param  {string|Error} err
     */
    function handleError(err) {
      if (!(err instanceof Error)) {
        err = error$1(err);
      }
      returnError = err;
      onError(err, getContext);
    }

    /**
     * Handle parse error.
     *
     * @param  {string|Error} err
     */
    function handleWarning(err) {
      if (!onWarning) {
        return;
      }
      if (!(err instanceof Error)) {
        err = error$1(err);
      }
      onWarning(err, getContext);
    }

    /**
     * Register parse listener.
     *
     * @param  {string}   name
     * @param  {Function} cb
     *
     * @return {Parser}
     */
    this['on'] = function (name, cb) {
      if (typeof cb !== 'function') {
        throw error$1('required args <name, cb>');
      }
      switch (name) {
        case 'openTag':
          onOpenTag = cb;
          break;
        case 'text':
          onText = cb;
          break;
        case 'closeTag':
          onCloseTag = cb;
          break;
        case 'error':
          onError = cb;
          break;
        case 'warn':
          onWarning = cb;
          break;
        case 'cdata':
          onCDATA = cb;
          break;
        case 'attention':
          onAttention = cb;
          break;
        // <!XXXXX zzzz="eeee">
        case 'question':
          onQuestion = cb;
          break;
        // <? ....  ?>
        case 'comment':
          onComment = cb;
          break;
        default:
          throw error$1('unsupported event: ' + name);
      }
      return this;
    };

    /**
     * Set the namespace to prefix mapping.
     *
     * @example
     *
     * parser.ns({
     *   'http://foo': 'foo',
     *   'http://bar': 'bar'
     * });
     *
     * @param  {!Object<string, string>} nsMap
     *
     * @return {Parser}
     */
    this['ns'] = function (nsMap) {
      if (typeof nsMap === 'undefined') {
        nsMap = {};
      }
      if (typeof nsMap !== 'object') {
        throw error$1('required args <nsMap={}>');
      }
      var _nsUriToPrefix = {},
        k;
      for (k in nsMap) {
        _nsUriToPrefix[k] = nsMap[k];
      }

      // FORCE default mapping for schema instance
      _nsUriToPrefix[XSI_URI] = XSI_PREFIX;
      isNamespace = true;
      nsUriToPrefix = _nsUriToPrefix;
      return this;
    };

    /**
     * Parse xml string.
     *
     * @param  {string} xml
     *
     * @return {Error} returnError, if not thrown
     */
    this['parse'] = function (xml) {
      if (typeof xml !== 'string') {
        throw error$1('required args <xml=string>');
      }
      returnError = null;
      parse(xml);
      getContext = noopGetContext;
      parseStop = false;
      return returnError;
    };

    /**
     * Stop parsing.
     */
    this['stop'] = function () {
      parseStop = true;
    };

    /**
     * Parse string, invoking configured listeners on element.
     *
     * @param  {string} xml
     */
    function parse(xml) {
      var nsMatrixStack = isNamespace ? [] : null,
        nsMatrix = isNamespace ? buildNsMatrix(nsUriToPrefix) : null,
        _nsMatrix,
        nodeStack = [],
        anonymousNsCount = 0,
        tagStart = false,
        tagEnd = false,
        i = 0,
        j = 0,
        x,
        y,
        q,
        w,
        v,
        xmlns,
        elementName,
        _elementName,
        elementProxy;
      var attrsString = '',
        attrsStart = 0,
        cachedAttrs // false = parsed with errors, null = needs parsing
      ;

      /**
       * Parse attributes on demand and returns the parsed attributes.
       *
       * Return semantics: (1) `false` on attribute parse error,
       * (2) object hash on extracted attrs.
       *
       * @return {boolean|Object}
       */
      function getAttrs() {
        if (cachedAttrs !== null) {
          return cachedAttrs;
        }
        var nsUri,
          nsUriPrefix,
          nsName,
          defaultAlias = isNamespace && nsMatrix['xmlns'],
          attrList = isNamespace && maybeNS ? [] : null,
          i = attrsStart,
          s = attrsString,
          l = s.length,
          hasNewMatrix,
          newalias,
          value,
          alias,
          name,
          attrs = {},
          seenAttrs = {},
          skipAttr,
          w,
          j;
        parseAttr: for (; i < l; i++) {
          skipAttr = false;
          w = s.charCodeAt(i);
          if (w === 32 || w < 14 && w > 8) {
            // WHITESPACE={ \f\n\r\t\v}
            continue;
          }

          // wait for non whitespace character
          if (w < 65 || w > 122 || w > 90 && w < 97) {
            if (w !== 95 && w !== 58) {
              // char 95"_" 58":"
              handleWarning('illegal first char attribute name');
              skipAttr = true;
            }
          }

          // parse attribute name
          for (j = i + 1; j < l; j++) {
            w = s.charCodeAt(j);
            if (w > 96 && w < 123 || w > 64 && w < 91 || w > 47 && w < 59 || w === 46 ||
            // '.'
            w === 45 ||
            // '-'
            w === 95 // '_'
            ) {
              continue;
            }

            // unexpected whitespace
            if (w === 32 || w < 14 && w > 8) {
              // WHITESPACE
              handleWarning('missing attribute value');
              i = j;
              continue parseAttr;
            }

            // expected "="
            if (w === 61) {
              // "=" == 61
              break;
            }
            handleWarning('illegal attribute name char');
            skipAttr = true;
          }
          name = s.substring(i, j);
          if (name === 'xmlns:xmlns') {
            handleWarning('illegal declaration of xmlns');
            skipAttr = true;
          }
          w = s.charCodeAt(j + 1);
          if (w === 34) {
            // '"'
            j = s.indexOf('"', i = j + 2);
            if (j === -1) {
              j = s.indexOf('\'', i);
              if (j !== -1) {
                handleWarning('attribute value quote missmatch');
                skipAttr = true;
              }
            }
          } else if (w === 39) {
            // "'"
            j = s.indexOf('\'', i = j + 2);
            if (j === -1) {
              j = s.indexOf('"', i);
              if (j !== -1) {
                handleWarning('attribute value quote missmatch');
                skipAttr = true;
              }
            }
          } else {
            handleWarning('missing attribute value quotes');
            skipAttr = true;

            // skip to next space
            for (j = j + 1; j < l; j++) {
              w = s.charCodeAt(j + 1);
              if (w === 32 || w < 14 && w > 8) {
                // WHITESPACE
                break;
              }
            }
          }
          if (j === -1) {
            handleWarning('missing closing quotes');
            j = l;
            skipAttr = true;
          }
          if (!skipAttr) {
            value = s.substring(i, j);
          }
          i = j;

          // ensure SPACE follows attribute
          // skip illegal content otherwise
          // example a="b"c
          for (; j + 1 < l; j++) {
            w = s.charCodeAt(j + 1);
            if (w === 32 || w < 14 && w > 8) {
              // WHITESPACE
              break;
            }

            // FIRST ILLEGAL CHAR
            if (i === j) {
              handleWarning('illegal character after attribute end');
              skipAttr = true;
            }
          }

          // advance cursor to next attribute
          i = j + 1;
          if (skipAttr) {
            continue parseAttr;
          }

          // check attribute re-declaration
          if (name in seenAttrs) {
            handleWarning('attribute <' + name + '> already defined');
            continue;
          }
          seenAttrs[name] = true;
          if (!isNamespace) {
            attrs[name] = value;
            continue;
          }

          // try to extract namespace information
          if (maybeNS) {
            newalias = name === 'xmlns' ? 'xmlns' : name.charCodeAt(0) === 120 && name.substr(0, 6) === 'xmlns:' ? name.substr(6) : null;

            // handle xmlns(:alias) assignment
            if (newalias !== null) {
              nsUri = decodeEntities(value);
              nsUriPrefix = uriPrefix(newalias);
              alias = nsUriToPrefix[nsUri];
              if (!alias) {
                // no prefix defined or prefix collision
                if (newalias === 'xmlns' || nsUriPrefix in nsMatrix && nsMatrix[nsUriPrefix] !== nsUri) {
                  // alocate free ns prefix
                  do {
                    alias = 'ns' + anonymousNsCount++;
                  } while (typeof nsMatrix[alias] !== 'undefined');
                } else {
                  alias = newalias;
                }
                nsUriToPrefix[nsUri] = alias;
              }
              if (nsMatrix[newalias] !== alias) {
                if (!hasNewMatrix) {
                  nsMatrix = cloneNsMatrix(nsMatrix);
                  hasNewMatrix = true;
                }
                nsMatrix[newalias] = alias;
                if (newalias === 'xmlns') {
                  nsMatrix[uriPrefix(alias)] = nsUri;
                  defaultAlias = alias;
                }
                nsMatrix[nsUriPrefix] = nsUri;
              }

              // expose xmlns(:asd)="..." in attributes
              attrs[name] = value;
              continue;
            }

            // collect attributes until all namespace
            // declarations are processed
            attrList.push(name, value);
            continue;
          } /** end if (maybeNs) */

          // handle attributes on element without
          // namespace declarations
          w = name.indexOf(':');
          if (w === -1) {
            attrs[name] = value;
            continue;
          }

          // normalize ns attribute name
          if (!(nsName = nsMatrix[name.substring(0, w)])) {
            handleWarning(missingNamespaceForPrefix(name.substring(0, w)));
            continue;
          }
          name = defaultAlias === nsName ? name.substr(w + 1) : nsName + name.substr(w);

          // end: normalize ns attribute name

          // normalize xsi:type ns attribute value
          if (name === XSI_TYPE$1) {
            w = value.indexOf(':');
            if (w !== -1) {
              nsName = value.substring(0, w);

              // handle default prefixes, i.e. xs:String gracefully
              nsName = nsMatrix[nsName] || nsName;
              value = nsName + value.substring(w);
            } else {
              value = defaultAlias + ':' + value;
            }
          }

          // end: normalize xsi:type ns attribute value

          attrs[name] = value;
        }

        // handle deferred, possibly namespaced attributes
        if (maybeNS) {
          // normalize captured attributes
          for (i = 0, l = attrList.length; i < l; i++) {
            name = attrList[i++];
            value = attrList[i];
            w = name.indexOf(':');
            if (w !== -1) {
              // normalize ns attribute name
              if (!(nsName = nsMatrix[name.substring(0, w)])) {
                handleWarning(missingNamespaceForPrefix(name.substring(0, w)));
                continue;
              }
              name = defaultAlias === nsName ? name.substr(w + 1) : nsName + name.substr(w);

              // end: normalize ns attribute name

              // normalize xsi:type ns attribute value
              if (name === XSI_TYPE$1) {
                w = value.indexOf(':');
                if (w !== -1) {
                  nsName = value.substring(0, w);

                  // handle default prefixes, i.e. xs:String gracefully
                  nsName = nsMatrix[nsName] || nsName;
                  value = nsName + value.substring(w);
                } else {
                  value = defaultAlias + ':' + value;
                }
              }

              // end: normalize xsi:type ns attribute value
            }
            attrs[name] = value;
          }

          // end: normalize captured attributes
        }
        return cachedAttrs = attrs;
      }

      /**
       * Extract the parse context { line, column, part }
       * from the current parser position.
       *
       * @return {Object} parse context
       */
      function getParseContext() {
        var splitsRe = /(\r\n|\r|\n)/g;
        var line = 0;
        var column = 0;
        var startOfLine = 0;
        var endOfLine = j;
        var match;
        var data;
        while (i >= startOfLine) {
          match = splitsRe.exec(xml);
          if (!match) {
            break;
          }

          // end of line = (break idx + break chars)
          endOfLine = match[0].length + match.index;
          if (endOfLine > i) {
            break;
          }

          // advance to next line
          line += 1;
          startOfLine = endOfLine;
        }

        // EOF errors
        if (i == -1) {
          column = endOfLine;
          data = xml.substring(j);
        } else
          // start errors
          if (j === 0) {
            data = xml.substring(j, i);
          }

          // other errors
          else {
            column = i - startOfLine;
            data = j == -1 ? xml.substring(i) : xml.substring(i, j + 1);
          }
        return {
          'data': data,
          'line': line,
          'column': column
        };
      }
      getContext = getParseContext;
      if (proxy) {
        elementProxy = Object.create({}, {
          'name': getter(function () {
            return elementName;
          }),
          'originalName': getter(function () {
            return _elementName;
          }),
          'attrs': getter(getAttrs),
          'ns': getter(function () {
            return nsMatrix;
          })
        });
      }

      // actual parse logic
      while (j !== -1) {
        if (xml.charCodeAt(j) === 60) {
          // "<"
          i = j;
        } else {
          i = xml.indexOf('<', j);
        }

        // parse end
        if (i === -1) {
          if (nodeStack.length) {
            return handleError('unexpected end of file');
          }
          if (j === 0) {
            return handleError('missing start tag');
          }
          if (j < xml.length) {
            if (xml.substring(j).trim()) {
              handleWarning(NON_WHITESPACE_OUTSIDE_ROOT_NODE);
            }
          }
          return;
        }

        // parse text
        if (j !== i) {
          if (nodeStack.length) {
            if (onText) {
              onText(xml.substring(j, i), decodeEntities, getContext);
              if (parseStop) {
                return;
              }
            }
          } else {
            if (xml.substring(j, i).trim()) {
              handleWarning(NON_WHITESPACE_OUTSIDE_ROOT_NODE);
              if (parseStop) {
                return;
              }
            }
          }
        }
        w = xml.charCodeAt(i + 1);

        // parse comments + CDATA
        if (w === 33) {
          // "!"
          q = xml.charCodeAt(i + 2);

          // CDATA section
          if (q === 91 && xml.substr(i + 3, 6) === 'CDATA[') {
            // 91 == "["
            j = xml.indexOf(']]>', i);
            if (j === -1) {
              return handleError('unclosed cdata');
            }
            if (onCDATA) {
              onCDATA(xml.substring(i + 9, j), getContext);
              if (parseStop) {
                return;
              }
            }
            j += 3;
            continue;
          }

          // comment
          if (q === 45 && xml.charCodeAt(i + 3) === 45) {
            // 45 == "-"
            j = xml.indexOf('-->', i);
            if (j === -1) {
              return handleError('unclosed comment');
            }
            if (onComment) {
              onComment(xml.substring(i + 4, j), decodeEntities, getContext);
              if (parseStop) {
                return;
              }
            }
            j += 3;
            continue;
          }
        }

        // parse question <? ... ?>
        if (w === 63) {
          // "?"
          j = xml.indexOf('?>', i);
          if (j === -1) {
            return handleError('unclosed question');
          }
          if (onQuestion) {
            onQuestion(xml.substring(i, j + 2), getContext);
            if (parseStop) {
              return;
            }
          }
          j += 2;
          continue;
        }

        // find matching closing tag for attention or standard tags
        // for that we must skip through attribute values
        // (enclosed in single or double quotes)
        for (x = i + 1;; x++) {
          v = xml.charCodeAt(x);
          if (isNaN(v)) {
            j = -1;
            return handleError('unclosed tag');
          }

          // [10] AttValue ::= '"' ([^<&"] | Reference)* '"' | "'" ([^<&'] | Reference)* "'"
          // skips the quoted string
          // (double quotes) does not appear in a literal enclosed by (double quotes)
          // (single quote) does not appear in a literal enclosed by (single quote)
          if (v === 34) {
            //  '"'
            q = xml.indexOf('"', x + 1);
            x = q !== -1 ? q : x;
          } else if (v === 39) {
            // "'"
            q = xml.indexOf("'", x + 1);
            x = q !== -1 ? q : x;
          } else if (v === 62) {
            // '>'
            j = x;
            break;
          }
        }

        // parse attention <! ...>
        // previously comment and CDATA have already been parsed
        if (w === 33) {
          // "!"

          if (onAttention) {
            onAttention(xml.substring(i, j + 1), decodeEntities, getContext);
            if (parseStop) {
              return;
            }
          }
          j += 1;
          continue;
        }

        // don't process attributes;
        // there are none
        cachedAttrs = {};

        // if (xml.charCodeAt(i+1) === 47) { // </...
        if (w === 47) {
          // </...
          tagStart = false;
          tagEnd = true;
          if (!nodeStack.length) {
            return handleError('missing open tag');
          }

          // verify open <-> close tag match
          x = elementName = nodeStack.pop();
          q = i + 2 + x.length;
          if (xml.substring(i + 2, q) !== x) {
            return handleError('closing tag mismatch');
          }

          // verify chars in close tag
          for (; q < j; q++) {
            w = xml.charCodeAt(q);
            if (w === 32 || w > 8 && w < 14) {
              // \f\n\r\t\v space
              continue;
            }
            return handleError('close tag');
          }
        } else {
          if (xml.charCodeAt(j - 1) === 47) {
            // .../>
            x = elementName = xml.substring(i + 1, j - 1);
            tagStart = true;
            tagEnd = true;
          } else {
            x = elementName = xml.substring(i + 1, j);
            tagStart = true;
            tagEnd = false;
          }
          if (!(w > 96 && w < 123 || w > 64 && w < 91 || w === 95 || w === 58)) {
            // char 95"_" 58":"
            return handleError('illegal first char nodeName');
          }
          for (q = 1, y = x.length; q < y; q++) {
            w = x.charCodeAt(q);
            if (w > 96 && w < 123 || w > 64 && w < 91 || w > 47 && w < 59 || w === 45 || w === 95 || w == 46) {
              continue;
            }
            if (w === 32 || w < 14 && w > 8) {
              // \f\n\r\t\v space
              elementName = x.substring(0, q);

              // maybe there are attributes
              cachedAttrs = null;
              break;
            }
            return handleError('invalid nodeName');
          }
          if (!tagEnd) {
            nodeStack.push(elementName);
          }
        }
        if (isNamespace) {
          _nsMatrix = nsMatrix;
          if (tagStart) {
            // remember old namespace
            // unless we're self-closing
            if (!tagEnd) {
              nsMatrixStack.push(_nsMatrix);
            }
            if (cachedAttrs === null) {
              // quick check, whether there may be namespace
              // declarations on the node; if that is the case
              // we need to eagerly parse the node attributes
              if (maybeNS = x.indexOf('xmlns', q) !== -1) {
                attrsStart = q;
                attrsString = x;
                getAttrs();
                maybeNS = false;
              }
            }
          }
          _elementName = elementName;
          w = elementName.indexOf(':');
          if (w !== -1) {
            xmlns = nsMatrix[elementName.substring(0, w)];

            // prefix given; namespace must exist
            if (!xmlns) {
              return handleError('missing namespace on <' + _elementName + '>');
            }
            elementName = elementName.substr(w + 1);
          } else {
            xmlns = nsMatrix['xmlns'];

            // if no default namespace is defined,
            // we'll import the element as anonymous.
            //
            // it is up to users to correct that to the document defined
            // targetNamespace, or whatever their undersanding of the
            // XML spec mandates.
          }

          // adjust namespace prefixs as configured
          if (xmlns) {
            elementName = xmlns + ':' + elementName;
          }
        }
        if (tagStart) {
          attrsStart = q;
          attrsString = x;
          if (onOpenTag) {
            if (proxy) {
              onOpenTag(elementProxy, decodeEntities, tagEnd, getContext);
            } else {
              onOpenTag(elementName, getAttrs, decodeEntities, tagEnd, getContext);
            }
            if (parseStop) {
              return;
            }
          }
        }
        if (tagEnd) {
          if (onCloseTag) {
            onCloseTag(proxy ? elementProxy : elementName, decodeEntities, tagStart, getContext);
            if (parseStop) {
              return;
            }
          }

          // restore old namespace
          if (isNamespace) {
            if (!tagStart) {
              nsMatrix = nsMatrixStack.pop();
            } else {
              nsMatrix = _nsMatrix;
            }
          }
        }
        j += 1;
      }
    } /** end parse */
  }

  function hasLowerCaseAlias(pkg) {
    return pkg.xml && pkg.xml.tagAlias === 'lowerCase';
  }
  var DEFAULT_NS_MAP = {
    'xsi': 'http://www.w3.org/2001/XMLSchema-instance',
    'xml': 'http://www.w3.org/XML/1998/namespace'
  };
  var XSI_TYPE = 'xsi:type';
  function serializeFormat(element) {
    return element.xml && element.xml.serialize;
  }
  function serializeAsType(element) {
    return serializeFormat(element) === XSI_TYPE;
  }
  function serializeAsProperty(element) {
    return serializeFormat(element) === 'property';
  }
  function capitalize(str) {
    return str.charAt(0).toUpperCase() + str.slice(1);
  }
  function aliasToName(aliasNs, pkg) {
    if (!hasLowerCaseAlias(pkg)) {
      return aliasNs.name;
    }
    return aliasNs.prefix + ':' + capitalize(aliasNs.localName);
  }
  function prefixedToName(nameNs, pkg) {
    var name = nameNs.name,
      localName = nameNs.localName;
    var typePrefix = pkg.xml && pkg.xml.typePrefix;
    if (typePrefix && localName.indexOf(typePrefix) === 0) {
      return nameNs.prefix + ':' + localName.slice(typePrefix.length);
    } else {
      return name;
    }
  }
  function normalizeXsiTypeName(name, model) {
    var nameNs = parseName(name);
    var pkg = model.getPackage(nameNs.prefix);
    return prefixedToName(nameNs, pkg);
  }
  function error(message) {
    return new Error(message);
  }

  /**
   * Get the moddle descriptor for a given instance or type.
   *
   * @param  {ModdleElement|Function} element
   *
   * @return {Object} the moddle descriptor
   */
  function getModdleDescriptor(element) {
    return element.$descriptor;
  }

  /**
   * A parse context.
   *
   * @class
   *
   * @param {Object} options
   * @param {ElementHandler} options.rootHandler the root handler for parsing a document
   * @param {boolean} [options.lax=false] whether or not to ignore invalid elements
   */
  function Context(options) {
    /**
     * @property {ElementHandler} rootHandler
     */

    /**
     * @property {Boolean} lax
     */

    assign$1(this, options);
    this.elementsById = {};
    this.references = [];
    this.warnings = [];

    /**
     * Add an unresolved reference.
     *
     * @param {Object} reference
     */
    this.addReference = function (reference) {
      this.references.push(reference);
    };

    /**
     * Add a processed element.
     *
     * @param {ModdleElement} element
     */
    this.addElement = function (element) {
      if (!element) {
        throw error('expected element');
      }
      var elementsById = this.elementsById;
      var descriptor = getModdleDescriptor(element);
      var idProperty = descriptor.idProperty,
        id;
      if (idProperty) {
        id = element.get(idProperty.name);
        if (id) {
          // for QName validation as per http://www.w3.org/TR/REC-xml/#NT-NameChar
          if (!/^([a-z][\w-.]*:)?[a-z_][\w-.]*$/i.test(id)) {
            throw new Error('illegal ID <' + id + '>');
          }
          if (elementsById[id]) {
            throw error('duplicate ID <' + id + '>');
          }
          elementsById[id] = element;
        }
      }
    };

    /**
     * Add an import warning.
     *
     * @param {Object} warning
     * @param {String} warning.message
     * @param {Error} [warning.error]
     */
    this.addWarning = function (warning) {
      this.warnings.push(warning);
    };
  }
  function BaseHandler() {}
  BaseHandler.prototype.handleEnd = function () {};
  BaseHandler.prototype.handleText = function () {};
  BaseHandler.prototype.handleNode = function () {};

  /**
   * A simple pass through handler that does nothing except for
   * ignoring all input it receives.
   *
   * This is used to ignore unknown elements and
   * attributes.
   */
  function NoopHandler() {}
  NoopHandler.prototype = Object.create(BaseHandler.prototype);
  NoopHandler.prototype.handleNode = function () {
    return this;
  };
  function BodyHandler() {}
  BodyHandler.prototype = Object.create(BaseHandler.prototype);
  BodyHandler.prototype.handleText = function (text) {
    this.body = (this.body || '') + text;
  };
  function ReferenceHandler(property, context) {
    this.property = property;
    this.context = context;
  }
  ReferenceHandler.prototype = Object.create(BodyHandler.prototype);
  ReferenceHandler.prototype.handleNode = function (node) {
    if (this.element) {
      throw error('expected no sub nodes');
    } else {
      this.element = this.createReference(node);
    }
    return this;
  };
  ReferenceHandler.prototype.handleEnd = function () {
    this.element.id = this.body;
  };
  ReferenceHandler.prototype.createReference = function (node) {
    return {
      property: this.property.ns.name,
      id: ''
    };
  };
  function ValueHandler(propertyDesc, element) {
    this.element = element;
    this.propertyDesc = propertyDesc;
  }
  ValueHandler.prototype = Object.create(BodyHandler.prototype);
  ValueHandler.prototype.handleEnd = function () {
    var value = this.body || '',
      element = this.element,
      propertyDesc = this.propertyDesc;
    value = coerceType(propertyDesc.type, value);
    if (propertyDesc.isMany) {
      element.get(propertyDesc.name).push(value);
    } else {
      element.set(propertyDesc.name, value);
    }
  };
  function BaseElementHandler() {}
  BaseElementHandler.prototype = Object.create(BodyHandler.prototype);
  BaseElementHandler.prototype.handleNode = function (node) {
    var parser = this,
      element = this.element;
    if (!element) {
      element = this.element = this.createElement(node);
      this.context.addElement(element);
    } else {
      parser = this.handleChild(node);
    }
    return parser;
  };

  /**
   * @class Reader.ElementHandler
   *
   */
  function ElementHandler(model, typeName, context) {
    this.model = model;
    this.type = model.getType(typeName);
    this.context = context;
  }
  ElementHandler.prototype = Object.create(BaseElementHandler.prototype);
  ElementHandler.prototype.addReference = function (reference) {
    this.context.addReference(reference);
  };
  ElementHandler.prototype.handleText = function (text) {
    var element = this.element,
      descriptor = getModdleDescriptor(element),
      bodyProperty = descriptor.bodyProperty;
    if (!bodyProperty) {
      throw error('unexpected body text <' + text + '>');
    }
    BodyHandler.prototype.handleText.call(this, text);
  };
  ElementHandler.prototype.handleEnd = function () {
    var value = this.body,
      element = this.element,
      descriptor = getModdleDescriptor(element),
      bodyProperty = descriptor.bodyProperty;
    if (bodyProperty && value !== undefined) {
      value = coerceType(bodyProperty.type, value);
      element.set(bodyProperty.name, value);
    }
  };

  /**
   * Create an instance of the model from the given node.
   *
   * @param  {Element} node the xml node
   */
  ElementHandler.prototype.createElement = function (node) {
    var attributes = node.attributes,
      Type = this.type,
      descriptor = getModdleDescriptor(Type),
      context = this.context,
      instance = new Type({}),
      model = this.model,
      propNameNs;
    forEach$1(attributes, function (value, name) {
      var prop = descriptor.propertiesByName[name],
        values;
      if (prop && prop.isReference) {
        if (!prop.isMany) {
          context.addReference({
            element: instance,
            property: prop.ns.name,
            id: value
          });
        } else {
          // IDREFS: parse references as whitespace-separated list
          values = value.split(' ');
          forEach$1(values, function (v) {
            context.addReference({
              element: instance,
              property: prop.ns.name,
              id: v
            });
          });
        }
      } else {
        if (prop) {
          value = coerceType(prop.type, value);
        } else if (name !== 'xmlns') {
          propNameNs = parseName(name, descriptor.ns.prefix);

          // check whether attribute is defined in a well-known namespace
          // if that is the case we emit a warning to indicate potential misuse
          if (model.getPackage(propNameNs.prefix)) {
            context.addWarning({
              message: 'unknown attribute <' + name + '>',
              element: instance,
              property: name,
              value: value
            });
          }
        }
        instance.set(name, value);
      }
    });
    return instance;
  };
  ElementHandler.prototype.getPropertyForNode = function (node) {
    var name = node.name;
    var nameNs = parseName(name);
    var type = this.type,
      model = this.model,
      descriptor = getModdleDescriptor(type);
    var propertyName = nameNs.name,
      property = descriptor.propertiesByName[propertyName],
      elementTypeName,
      elementType;

    // search for properties by name first

    if (property && !property.isAttr) {
      if (serializeAsType(property)) {
        elementTypeName = node.attributes[XSI_TYPE];

        // xsi type is optional, if it does not exists the
        // default type is assumed
        if (elementTypeName) {
          // take possible type prefixes from XML
          // into account, i.e.: xsi:type="t{ActualType}"
          elementTypeName = normalizeXsiTypeName(elementTypeName, model);
          elementType = model.getType(elementTypeName);
          return assign$1({}, property, {
            effectiveType: getModdleDescriptor(elementType).name
          });
        }
      }

      // search for properties by name first
      return property;
    }
    var pkg = model.getPackage(nameNs.prefix);
    if (pkg) {
      elementTypeName = aliasToName(nameNs, pkg);
      elementType = model.getType(elementTypeName);

      // search for collection members later
      property = find$1(descriptor.properties, function (p) {
        return !p.isVirtual && !p.isReference && !p.isAttribute && elementType.hasType(p.type);
      });
      if (property) {
        return assign$1({}, property, {
          effectiveType: getModdleDescriptor(elementType).name
        });
      }
    } else {
      // parse unknown element (maybe extension)
      property = find$1(descriptor.properties, function (p) {
        return !p.isReference && !p.isAttribute && p.type === 'Element';
      });
      if (property) {
        return property;
      }
    }
    throw error('unrecognized element <' + nameNs.name + '>');
  };
  ElementHandler.prototype.toString = function () {
    return 'ElementDescriptor[' + getModdleDescriptor(this.type).name + ']';
  };
  ElementHandler.prototype.valueHandler = function (propertyDesc, element) {
    return new ValueHandler(propertyDesc, element);
  };
  ElementHandler.prototype.referenceHandler = function (propertyDesc) {
    return new ReferenceHandler(propertyDesc, this.context);
  };
  ElementHandler.prototype.handler = function (type) {
    if (type === 'Element') {
      return new GenericElementHandler(this.model, type, this.context);
    } else {
      return new ElementHandler(this.model, type, this.context);
    }
  };

  /**
   * Handle the child element parsing
   *
   * @param  {Element} node the xml node
   */
  ElementHandler.prototype.handleChild = function (node) {
    var propertyDesc, type, element, childHandler;
    propertyDesc = this.getPropertyForNode(node);
    element = this.element;
    type = propertyDesc.effectiveType || propertyDesc.type;
    if (isSimple(type)) {
      return this.valueHandler(propertyDesc, element);
    }
    if (propertyDesc.isReference) {
      childHandler = this.referenceHandler(propertyDesc).handleNode(node);
    } else {
      childHandler = this.handler(type).handleNode(node);
    }
    var newElement = childHandler.element;

    // child handles may decide to skip elements
    // by not returning anything
    if (newElement !== undefined) {
      if (propertyDesc.isMany) {
        element.get(propertyDesc.name).push(newElement);
      } else {
        element.set(propertyDesc.name, newElement);
      }
      if (propertyDesc.isReference) {
        assign$1(newElement, {
          element: element
        });
        this.context.addReference(newElement);
      } else {
        // establish child -> parent relationship
        newElement.$parent = element;
      }
    }
    return childHandler;
  };

  /**
   * An element handler that performs special validation
   * to ensure the node it gets initialized with matches
   * the handlers type (namespace wise).
   *
   * @param {Moddle} model
   * @param {String} typeName
   * @param {Context} context
   */
  function RootElementHandler(model, typeName, context) {
    ElementHandler.call(this, model, typeName, context);
  }
  RootElementHandler.prototype = Object.create(ElementHandler.prototype);
  RootElementHandler.prototype.createElement = function (node) {
    var name = node.name,
      nameNs = parseName(name),
      model = this.model,
      type = this.type,
      pkg = model.getPackage(nameNs.prefix),
      typeName = pkg && aliasToName(nameNs, pkg) || name;

    // verify the correct namespace if we parse
    // the first element in the handler tree
    //
    // this ensures we don't mistakenly import wrong namespace elements
    if (!type.hasType(typeName)) {
      throw error('unexpected element <' + node.originalName + '>');
    }
    return ElementHandler.prototype.createElement.call(this, node);
  };
  function GenericElementHandler(model, typeName, context) {
    this.model = model;
    this.context = context;
  }
  GenericElementHandler.prototype = Object.create(BaseElementHandler.prototype);
  GenericElementHandler.prototype.createElement = function (node) {
    var name = node.name,
      ns = parseName(name),
      prefix = ns.prefix,
      uri = node.ns[prefix + '$uri'],
      attributes = node.attributes;
    return this.model.createAny(name, uri, attributes);
  };
  GenericElementHandler.prototype.handleChild = function (node) {
    var handler = new GenericElementHandler(this.model, 'Element', this.context).handleNode(node),
      element = this.element;
    var newElement = handler.element,
      children;
    if (newElement !== undefined) {
      children = element.$children = element.$children || [];
      children.push(newElement);

      // establish child -> parent relationship
      newElement.$parent = element;
    }
    return handler;
  };
  GenericElementHandler.prototype.handleEnd = function () {
    if (this.body) {
      this.element.$body = this.body;
    }
  };

  /**
   * A reader for a meta-model
   *
   * @param {Object} options
   * @param {Model} options.model used to read xml files
   * @param {Boolean} options.lax whether to make parse errors warnings
   */
  function Reader(options) {
    if (options instanceof Moddle) {
      options = {
        model: options
      };
    }
    assign$1(this, {
      lax: false
    }, options);
  }

  /**
   * The fromXML result.
   *
   * @typedef {Object} ParseResult
   *
   * @property {ModdleElement} rootElement
   * @property {Array<Object>} references
   * @property {Array<Error>} warnings
   * @property {Object} elementsById - a mapping containing each ID -> ModdleElement
   */

  /**
   * The fromXML result.
   *
   * @typedef {Error} ParseError
   *
   * @property {Array<Error>} warnings
   */

  /**
   * Parse the given XML into a moddle document tree.
   *
   * @param {String} xml
   * @param {ElementHandler|Object} options or rootHandler
   *
   * @returns {Promise<ParseResult, ParseError>}
   */
  Reader.prototype.fromXML = function (xml, options, done) {
    var rootHandler = options.rootHandler;
    if (options instanceof ElementHandler) {
      // root handler passed via (xml, { rootHandler: ElementHandler }, ...)
      rootHandler = options;
      options = {};
    } else {
      if (typeof options === 'string') {
        // rootHandler passed via (xml, 'someString', ...)
        rootHandler = this.handler(options);
        options = {};
      } else if (typeof rootHandler === 'string') {
        // rootHandler passed via (xml, { rootHandler: 'someString' }, ...)
        rootHandler = this.handler(rootHandler);
      }
    }
    var model = this.model,
      lax = this.lax;
    var context = new Context(assign$1({}, options, {
        rootHandler: rootHandler
      })),
      parser = new Parser$1({
        proxy: true
      }),
      stack = createStack();
    rootHandler.context = context;

    // push root handler
    stack.push(rootHandler);

    /**
     * Handle error.
     *
     * @param  {Error} err
     * @param  {Function} getContext
     * @param  {boolean} lax
     *
     * @return {boolean} true if handled
     */
    function handleError(err, getContext, lax) {
      var ctx = getContext();
      var line = ctx.line,
        column = ctx.column,
        data = ctx.data;

      // we receive the full context data here,
      // for elements trim down the information
      // to the tag name, only
      if (data.charAt(0) === '<' && data.indexOf(' ') !== -1) {
        data = data.slice(0, data.indexOf(' ')) + '>';
      }
      var message = 'unparsable content ' + (data ? data + ' ' : '') + 'detected\n\t' + 'line: ' + line + '\n\t' + 'column: ' + column + '\n\t' + 'nested error: ' + err.message;
      if (lax) {
        context.addWarning({
          message: message,
          error: err
        });
        return true;
      } else {
        throw error(message);
      }
    }
    function handleWarning(err, getContext) {
      // just like handling errors in <lax=true> mode
      return handleError(err, getContext, true);
    }

    /**
     * Resolve collected references on parse end.
     */
    function resolveReferences() {
      var elementsById = context.elementsById;
      var references = context.references;
      var i, r;
      for (i = 0; r = references[i]; i++) {
        var element = r.element;
        var reference = elementsById[r.id];
        var property = getModdleDescriptor(element).propertiesByName[r.property];
        if (!reference) {
          context.addWarning({
            message: 'unresolved reference <' + r.id + '>',
            element: r.element,
            property: r.property,
            value: r.id
          });
        }
        if (property.isMany) {
          var collection = element.get(property.name),
            idx = collection.indexOf(r);

          // we replace an existing place holder (idx != -1) or
          // append to the collection instead
          if (idx === -1) {
            idx = collection.length;
          }
          if (!reference) {
            // remove unresolvable reference
            collection.splice(idx, 1);
          } else {
            // add or update reference in collection
            collection[idx] = reference;
          }
        } else {
          element.set(property.name, reference);
        }
      }
    }
    function handleClose() {
      stack.pop().handleEnd();
    }
    var PREAMBLE_START_PATTERN = /^<\?xml /i;
    var ENCODING_PATTERN = / encoding="([^"]+)"/i;
    var UTF_8_PATTERN = /^utf-8$/i;
    function handleQuestion(question) {
      if (!PREAMBLE_START_PATTERN.test(question)) {
        return;
      }
      var match = ENCODING_PATTERN.exec(question);
      var encoding = match && match[1];
      if (!encoding || UTF_8_PATTERN.test(encoding)) {
        return;
      }
      context.addWarning({
        message: 'unsupported document encoding <' + encoding + '>, ' + 'falling back to UTF-8'
      });
    }
    function handleOpen(node, getContext) {
      var handler = stack.peek();
      try {
        stack.push(handler.handleNode(node));
      } catch (err) {
        if (handleError(err, getContext, lax)) {
          stack.push(new NoopHandler());
        }
      }
    }
    function handleCData(text, getContext) {
      try {
        stack.peek().handleText(text);
      } catch (err) {
        handleWarning(err, getContext);
      }
    }
    function handleText(text, getContext) {
      // strip whitespace only nodes, i.e. before
      // <!CDATA[ ... ]> sections and in between tags

      if (!text.trim()) {
        return;
      }
      handleCData(text, getContext);
    }
    var uriMap = model.getPackages().reduce(function (uriMap, p) {
      uriMap[p.uri] = p.prefix;
      return uriMap;
    }, {
      'http://www.w3.org/XML/1998/namespace': 'xml' // add default xml ns
    });
    parser.ns(uriMap).on('openTag', function (obj, decodeStr, selfClosing, getContext) {
      // gracefully handle unparsable attributes (attrs=false)
      var attrs = obj.attrs || {};
      var decodedAttrs = Object.keys(attrs).reduce(function (d, key) {
        var value = decodeStr(attrs[key]);
        d[key] = value;
        return d;
      }, {});
      var node = {
        name: obj.name,
        originalName: obj.originalName,
        attributes: decodedAttrs,
        ns: obj.ns
      };
      handleOpen(node, getContext);
    }).on('question', handleQuestion).on('closeTag', handleClose).on('cdata', handleCData).on('text', function (text, decodeEntities, getContext) {
      handleText(decodeEntities(text), getContext);
    }).on('error', handleError).on('warn', handleWarning);

    // async XML parsing to make sure the execution environment
    // (node or brower) is kept responsive and that certain optimization
    // strategies can kick in.
    return new Promise(function (resolve, reject) {
      var err;
      try {
        parser.parse(xml);
        resolveReferences();
      } catch (e) {
        err = e;
      }
      var rootElement = rootHandler.element;
      if (!err && !rootElement) {
        err = error('failed to parse document as <' + rootHandler.type.$descriptor.name + '>');
      }
      var warnings = context.warnings;
      var references = context.references;
      var elementsById = context.elementsById;
      if (err) {
        err.warnings = warnings;
        return reject(err);
      } else {
        return resolve({
          rootElement: rootElement,
          elementsById: elementsById,
          references: references,
          warnings: warnings
        });
      }
    });
  };
  Reader.prototype.handler = function (name) {
    return new RootElementHandler(this.model, name);
  };

  // helpers //////////////////////////

  function createStack() {
    var stack = [];
    Object.defineProperty(stack, 'peek', {
      value: function () {
        return this[this.length - 1];
      }
    });
    return stack;
  }
  var XML_PREAMBLE = '<?xml version="1.0" encoding="UTF-8"?>\n';
  var ESCAPE_ATTR_CHARS = /<|>|'|"|&|\n\r|\n/g;
  var ESCAPE_CHARS = /<|>|&/g;
  function Namespaces(parent) {
    var prefixMap = {};
    var uriMap = {};
    var used = {};
    var wellknown = [];
    var custom = [];

    // API

    this.byUri = function (uri) {
      return uriMap[uri] || parent && parent.byUri(uri);
    };
    this.add = function (ns, isWellknown) {
      uriMap[ns.uri] = ns;
      if (isWellknown) {
        wellknown.push(ns);
      } else {
        custom.push(ns);
      }
      this.mapPrefix(ns.prefix, ns.uri);
    };
    this.uriByPrefix = function (prefix) {
      return prefixMap[prefix || 'xmlns'];
    };
    this.mapPrefix = function (prefix, uri) {
      prefixMap[prefix || 'xmlns'] = uri;
    };
    this.getNSKey = function (ns) {
      return ns.prefix !== undefined ? ns.uri + '|' + ns.prefix : ns.uri;
    };
    this.logUsed = function (ns) {
      var uri = ns.uri;
      var nsKey = this.getNSKey(ns);
      used[nsKey] = this.byUri(uri);

      // Inform parent recursively about the usage of this NS
      if (parent) {
        parent.logUsed(ns);
      }
    };
    this.getUsed = function (ns) {
      function isUsed(ns) {
        var nsKey = self.getNSKey(ns);
        return used[nsKey];
      }
      var self = this;
      var allNs = [].concat(wellknown, custom);
      return allNs.filter(isUsed);
    };
  }
  function lower(string) {
    return string.charAt(0).toLowerCase() + string.slice(1);
  }
  function nameToAlias(name, pkg) {
    if (hasLowerCaseAlias(pkg)) {
      return lower(name);
    } else {
      return name;
    }
  }
  function inherits(ctor, superCtor) {
    ctor.super_ = superCtor;
    ctor.prototype = Object.create(superCtor.prototype, {
      constructor: {
        value: ctor,
        enumerable: false,
        writable: true,
        configurable: true
      }
    });
  }
  function nsName(ns) {
    if (isString$1(ns)) {
      return ns;
    } else {
      return (ns.prefix ? ns.prefix + ':' : '') + ns.localName;
    }
  }
  function getNsAttrs(namespaces) {
    return namespaces.getUsed().filter(function (ns) {
      // do not serialize built in <xml> namespace
      return ns.prefix !== 'xml';
    }).map(function (ns) {
      var name = 'xmlns' + (ns.prefix ? ':' + ns.prefix : '');
      return {
        name: name,
        value: ns.uri
      };
    });
  }
  function getElementNs(ns, descriptor) {
    if (descriptor.isGeneric) {
      return assign$1({
        localName: descriptor.ns.localName
      }, ns);
    } else {
      return assign$1({
        localName: nameToAlias(descriptor.ns.localName, descriptor.$pkg)
      }, ns);
    }
  }
  function getPropertyNs(ns, descriptor) {
    return assign$1({
      localName: descriptor.ns.localName
    }, ns);
  }
  function getSerializableProperties(element) {
    var descriptor = element.$descriptor;
    return filter(descriptor.properties, function (p) {
      var name = p.name;
      if (p.isVirtual) {
        return false;
      }

      // do not serialize defaults
      if (!has$1(element, name)) {
        return false;
      }
      var value = element[name];

      // do not serialize default equals
      if (value === p.default) {
        return false;
      }

      // do not serialize null properties
      if (value === null) {
        return false;
      }
      return p.isMany ? value.length : true;
    });
  }
  var ESCAPE_ATTR_MAP = {
    '\n': '#10',
    '\n\r': '#10',
    '"': '#34',
    '\'': '#39',
    '<': '#60',
    '>': '#62',
    '&': '#38'
  };
  var ESCAPE_MAP = {
    '<': 'lt',
    '>': 'gt',
    '&': 'amp'
  };
  function escape$1(str, charPattern, replaceMap) {
    // ensure we are handling strings here
    str = isString$1(str) ? str : '' + str;
    return str.replace(charPattern, function (s) {
      return '&' + replaceMap[s] + ';';
    });
  }

  /**
   * Escape a string attribute to not contain any bad values (line breaks, '"', ...)
   *
   * @param {String} str the string to escape
   * @return {String} the escaped string
   */
  function escapeAttr(str) {
    return escape$1(str, ESCAPE_ATTR_CHARS, ESCAPE_ATTR_MAP);
  }
  function escapeBody(str) {
    return escape$1(str, ESCAPE_CHARS, ESCAPE_MAP);
  }
  function filterAttributes(props) {
    return filter(props, function (p) {
      return p.isAttr;
    });
  }
  function filterContained(props) {
    return filter(props, function (p) {
      return !p.isAttr;
    });
  }
  function ReferenceSerializer(tagName) {
    this.tagName = tagName;
  }
  ReferenceSerializer.prototype.build = function (element) {
    this.element = element;
    return this;
  };
  ReferenceSerializer.prototype.serializeTo = function (writer) {
    writer.appendIndent().append('<' + this.tagName + '>' + this.element.id + '</' + this.tagName + '>').appendNewLine();
  };
  function BodySerializer() {}
  BodySerializer.prototype.serializeValue = BodySerializer.prototype.serializeTo = function (writer) {
    writer.append(this.escape ? escapeBody(this.value) : this.value);
  };
  BodySerializer.prototype.build = function (prop, value) {
    this.value = value;
    if (prop.type === 'String' && value.search(ESCAPE_CHARS) !== -1) {
      this.escape = true;
    }
    return this;
  };
  function ValueSerializer(tagName) {
    this.tagName = tagName;
  }
  inherits(ValueSerializer, BodySerializer);
  ValueSerializer.prototype.serializeTo = function (writer) {
    writer.appendIndent().append('<' + this.tagName + '>');
    this.serializeValue(writer);
    writer.append('</' + this.tagName + '>').appendNewLine();
  };
  function ElementSerializer(parent, propertyDescriptor) {
    this.body = [];
    this.attrs = [];
    this.parent = parent;
    this.propertyDescriptor = propertyDescriptor;
  }
  ElementSerializer.prototype.build = function (element) {
    this.element = element;
    var elementDescriptor = element.$descriptor,
      propertyDescriptor = this.propertyDescriptor;
    var otherAttrs, properties;
    var isGeneric = elementDescriptor.isGeneric;
    if (isGeneric) {
      otherAttrs = this.parseGeneric(element);
    } else {
      otherAttrs = this.parseNsAttributes(element);
    }
    if (propertyDescriptor) {
      this.ns = this.nsPropertyTagName(propertyDescriptor);
    } else {
      this.ns = this.nsTagName(elementDescriptor);
    }

    // compute tag name
    this.tagName = this.addTagName(this.ns);
    if (!isGeneric) {
      properties = getSerializableProperties(element);
      this.parseAttributes(filterAttributes(properties));
      this.parseContainments(filterContained(properties));
    }
    this.parseGenericAttributes(element, otherAttrs);
    return this;
  };
  ElementSerializer.prototype.nsTagName = function (descriptor) {
    var effectiveNs = this.logNamespaceUsed(descriptor.ns);
    return getElementNs(effectiveNs, descriptor);
  };
  ElementSerializer.prototype.nsPropertyTagName = function (descriptor) {
    var effectiveNs = this.logNamespaceUsed(descriptor.ns);
    return getPropertyNs(effectiveNs, descriptor);
  };
  ElementSerializer.prototype.isLocalNs = function (ns) {
    return ns.uri === this.ns.uri;
  };

  /**
   * Get the actual ns attribute name for the given element.
   *
   * @param {Object} element
   * @param {Boolean} [element.inherited=false]
   *
   * @return {Object} nsName
   */
  ElementSerializer.prototype.nsAttributeName = function (element) {
    var ns;
    if (isString$1(element)) {
      ns = parseName(element);
    } else {
      ns = element.ns;
    }

    // return just local name for inherited attributes
    if (element.inherited) {
      return {
        localName: ns.localName
      };
    }

    // parse + log effective ns
    var effectiveNs = this.logNamespaceUsed(ns);

    // LOG ACTUAL namespace use
    this.getNamespaces().logUsed(effectiveNs);

    // strip prefix if same namespace like parent
    if (this.isLocalNs(effectiveNs)) {
      return {
        localName: ns.localName
      };
    } else {
      return assign$1({
        localName: ns.localName
      }, effectiveNs);
    }
  };
  ElementSerializer.prototype.parseGeneric = function (element) {
    var self = this,
      body = this.body;
    var attributes = [];
    forEach$1(element, function (val, key) {
      var nonNsAttr;
      if (key === '$body') {
        body.push(new BodySerializer().build({
          type: 'String'
        }, val));
      } else if (key === '$children') {
        forEach$1(val, function (child) {
          body.push(new ElementSerializer(self).build(child));
        });
      } else if (key.indexOf('$') !== 0) {
        nonNsAttr = self.parseNsAttribute(element, key, val);
        if (nonNsAttr) {
          attributes.push({
            name: key,
            value: val
          });
        }
      }
    });
    return attributes;
  };
  ElementSerializer.prototype.parseNsAttribute = function (element, name, value) {
    var model = element.$model;
    var nameNs = parseName(name);
    var ns;

    // parse xmlns:foo="http://foo.bar"
    if (nameNs.prefix === 'xmlns') {
      ns = {
        prefix: nameNs.localName,
        uri: value
      };
    }

    // parse xmlns="http://foo.bar"
    if (!nameNs.prefix && nameNs.localName === 'xmlns') {
      ns = {
        uri: value
      };
    }
    if (!ns) {
      return {
        name: name,
        value: value
      };
    }
    if (model && model.getPackage(value)) {
      // register well known namespace
      this.logNamespace(ns, true, true);
    } else {
      // log custom namespace directly as used
      var actualNs = this.logNamespaceUsed(ns, true);
      this.getNamespaces().logUsed(actualNs);
    }
  };

  /**
   * Parse namespaces and return a list of left over generic attributes
   *
   * @param  {Object} element
   * @return {Array<Object>}
   */
  ElementSerializer.prototype.parseNsAttributes = function (element, attrs) {
    var self = this;
    var genericAttrs = element.$attrs;
    var attributes = [];

    // parse namespace attributes first
    // and log them. push non namespace attributes to a list
    // and process them later
    forEach$1(genericAttrs, function (value, name) {
      var nonNsAttr = self.parseNsAttribute(element, name, value);
      if (nonNsAttr) {
        attributes.push(nonNsAttr);
      }
    });
    return attributes;
  };
  ElementSerializer.prototype.parseGenericAttributes = function (element, attributes) {
    var self = this;
    forEach$1(attributes, function (attr) {
      // do not serialize xsi:type attribute
      // it is set manually based on the actual implementation type
      if (attr.name === XSI_TYPE) {
        return;
      }
      try {
        self.addAttribute(self.nsAttributeName(attr.name), attr.value);
      } catch (e) {
        console.warn('missing namespace information for ', attr.name, '=', attr.value, 'on', element, e);
      }
    });
  };
  ElementSerializer.prototype.parseContainments = function (properties) {
    var self = this,
      body = this.body,
      element = this.element;
    forEach$1(properties, function (p) {
      var value = element.get(p.name),
        isReference = p.isReference,
        isMany = p.isMany;
      if (!isMany) {
        value = [value];
      }
      if (p.isBody) {
        body.push(new BodySerializer().build(p, value[0]));
      } else if (isSimple(p.type)) {
        forEach$1(value, function (v) {
          body.push(new ValueSerializer(self.addTagName(self.nsPropertyTagName(p))).build(p, v));
        });
      } else if (isReference) {
        forEach$1(value, function (v) {
          body.push(new ReferenceSerializer(self.addTagName(self.nsPropertyTagName(p))).build(v));
        });
      } else {
        // allow serialization via type
        // rather than element name
        var asType = serializeAsType(p),
          asProperty = serializeAsProperty(p);
        forEach$1(value, function (v) {
          var serializer;
          if (asType) {
            serializer = new TypeSerializer(self, p);
          } else if (asProperty) {
            serializer = new ElementSerializer(self, p);
          } else {
            serializer = new ElementSerializer(self);
          }
          body.push(serializer.build(v));
        });
      }
    });
  };
  ElementSerializer.prototype.getNamespaces = function (local) {
    var namespaces = this.namespaces,
      parent = this.parent,
      parentNamespaces;
    if (!namespaces) {
      parentNamespaces = parent && parent.getNamespaces();
      if (local || !parentNamespaces) {
        this.namespaces = namespaces = new Namespaces(parentNamespaces);
      } else {
        namespaces = parentNamespaces;
      }
    }
    return namespaces;
  };
  ElementSerializer.prototype.logNamespace = function (ns, wellknown, local) {
    var namespaces = this.getNamespaces(local);
    var nsUri = ns.uri,
      nsPrefix = ns.prefix;
    var existing = namespaces.byUri(nsUri);
    if (!existing || local) {
      namespaces.add(ns, wellknown);
    }
    namespaces.mapPrefix(nsPrefix, nsUri);
    return ns;
  };
  ElementSerializer.prototype.logNamespaceUsed = function (ns, local) {
    var element = this.element,
      model = element.$model,
      namespaces = this.getNamespaces(local);

    // ns may be
    //
    //   * prefix only
    //   * prefix:uri
    //   * localName only

    var prefix = ns.prefix,
      uri = ns.uri,
      newPrefix,
      idx,
      wellknownUri;

    // handle anonymous namespaces (elementForm=unqualified), cf. #23
    if (!prefix && !uri) {
      return {
        localName: ns.localName
      };
    }
    wellknownUri = DEFAULT_NS_MAP[prefix] || model && (model.getPackage(prefix) || {}).uri;
    uri = uri || wellknownUri || namespaces.uriByPrefix(prefix);
    if (!uri) {
      throw new Error('no namespace uri given for prefix <' + prefix + '>');
    }
    ns = namespaces.byUri(uri);
    if (!ns) {
      newPrefix = prefix;
      idx = 1;

      // find a prefix that is not mapped yet
      while (namespaces.uriByPrefix(newPrefix)) {
        newPrefix = prefix + '_' + idx++;
      }
      ns = this.logNamespace({
        prefix: newPrefix,
        uri: uri
      }, wellknownUri === uri);
    }
    if (prefix) {
      namespaces.mapPrefix(prefix, uri);
    }
    return ns;
  };
  ElementSerializer.prototype.parseAttributes = function (properties) {
    var self = this,
      element = this.element;
    forEach$1(properties, function (p) {
      var value = element.get(p.name);
      if (p.isReference) {
        if (!p.isMany) {
          value = value.id;
        } else {
          var values = [];
          forEach$1(value, function (v) {
            values.push(v.id);
          });

          // IDREFS is a whitespace-separated list of references.
          value = values.join(' ');
        }
      }
      self.addAttribute(self.nsAttributeName(p), value);
    });
  };
  ElementSerializer.prototype.addTagName = function (nsTagName) {
    var actualNs = this.logNamespaceUsed(nsTagName);
    this.getNamespaces().logUsed(actualNs);
    return nsName(nsTagName);
  };
  ElementSerializer.prototype.addAttribute = function (name, value) {
    var attrs = this.attrs;
    if (isString$1(value)) {
      value = escapeAttr(value);
    }

    // de-duplicate attributes
    // https://github.com/bpmn-io/moddle-xml/issues/66
    var idx = findIndex(attrs, function (element) {
      return element.name.localName === name.localName && element.name.uri === name.uri && element.name.prefix === name.prefix;
    });
    var attr = {
      name: name,
      value: value
    };
    if (idx !== -1) {
      attrs.splice(idx, 1, attr);
    } else {
      attrs.push(attr);
    }
  };
  ElementSerializer.prototype.serializeAttributes = function (writer) {
    var attrs = this.attrs,
      namespaces = this.namespaces;
    if (namespaces) {
      attrs = getNsAttrs(namespaces).concat(attrs);
    }
    forEach$1(attrs, function (a) {
      writer.append(' ').append(nsName(a.name)).append('="').append(a.value).append('"');
    });
  };
  ElementSerializer.prototype.serializeTo = function (writer) {
    var firstBody = this.body[0],
      indent = firstBody && firstBody.constructor !== BodySerializer;
    writer.appendIndent().append('<' + this.tagName);
    this.serializeAttributes(writer);
    writer.append(firstBody ? '>' : ' />');
    if (firstBody) {
      if (indent) {
        writer.appendNewLine().indent();
      }
      forEach$1(this.body, function (b) {
        b.serializeTo(writer);
      });
      if (indent) {
        writer.unindent().appendIndent();
      }
      writer.append('</' + this.tagName + '>');
    }
    writer.appendNewLine();
  };

  /**
   * A serializer for types that handles serialization of data types
   */
  function TypeSerializer(parent, propertyDescriptor) {
    ElementSerializer.call(this, parent, propertyDescriptor);
  }
  inherits(TypeSerializer, ElementSerializer);
  TypeSerializer.prototype.parseNsAttributes = function (element) {
    // extracted attributes
    var attributes = ElementSerializer.prototype.parseNsAttributes.call(this, element);
    var descriptor = element.$descriptor;

    // only serialize xsi:type if necessary
    if (descriptor.name === this.propertyDescriptor.type) {
      return attributes;
    }
    var typeNs = this.typeNs = this.nsTagName(descriptor);
    this.getNamespaces().logUsed(this.typeNs);

    // add xsi:type attribute to represent the elements
    // actual type

    var pkg = element.$model.getPackage(typeNs.uri),
      typePrefix = pkg.xml && pkg.xml.typePrefix || '';
    this.addAttribute(this.nsAttributeName(XSI_TYPE), (typeNs.prefix ? typeNs.prefix + ':' : '') + typePrefix + descriptor.ns.localName);
    return attributes;
  };
  TypeSerializer.prototype.isLocalNs = function (ns) {
    return ns.uri === (this.typeNs || this.ns).uri;
  };
  function SavingWriter() {
    this.value = '';
    this.write = function (str) {
      this.value += str;
    };
  }
  function FormatingWriter(out, format) {
    var indent = [''];
    this.append = function (str) {
      out.write(str);
      return this;
    };
    this.appendNewLine = function () {
      if (format) {
        out.write('\n');
      }
      return this;
    };
    this.appendIndent = function () {
      if (format) {
        out.write(indent.join('  '));
      }
      return this;
    };
    this.indent = function () {
      indent.push('');
      return this;
    };
    this.unindent = function () {
      indent.pop();
      return this;
    };
  }

  /**
   * A writer for meta-model backed document trees
   *
   * @param {Object} options output options to pass into the writer
   */
  function Writer(options) {
    options = assign$1({
      format: false,
      preamble: true
    }, options || {});
    function toXML(tree, writer) {
      var internalWriter = writer || new SavingWriter();
      var formatingWriter = new FormatingWriter(internalWriter, options.format);
      if (options.preamble) {
        formatingWriter.append(XML_PREAMBLE);
      }
      new ElementSerializer().build(tree).serializeTo(formatingWriter);
      if (!writer) {
        return internalWriter.value;
      }
    }
    return {
      toXML: toXML
    };
  }

  /**
   * A sub class of {@link Moddle} with support for import and export of DMN xml files.
   *
   * @class DmnModdle
   * @extends Moddle
   *
   * @param {Object|Array} packages to use for instantiating the model
   * @param {Object} [options] additional options to pass over
   */
  function DmnModdle(packages, options) {
    Moddle.call(this, packages, options);
  }
  DmnModdle.prototype = Object.create(Moddle.prototype);

  /**
   * The fromXML result.
   *
   * @typedef {Object} ParseResult
   *
   * @property {ModdleElement} rootElement
   * @property {Array<Object>} references
   * @property {Array<Error>} warnings
   * @property {Object} elementsById - a mapping containing each ID -> ModdleElement
   */

  /**
   * The fromXML error.
   *
   * @typedef {Error} ParseError
   *
   * @property {Array<Error>} warnings
   */

  /**
   * Instantiates a DMN model tree from a given xml string.
   *
   * @param {String}   xmlStr
   * @param {String}   [typeName='dmn:Definitions'] name of the root element
   * @param {Object}   [options]  options to pass to the underlying reader
   *
   * @returns {Promise<ParseResult, ParseError>}
   */
  DmnModdle.prototype.fromXML = function (xmlStr, typeName, options) {
    if (!isString$3(typeName)) {
      options = typeName;
      typeName = 'dmn:Definitions';
    }
    var reader = new Reader(assign$3({
      model: this,
      lax: true
    }, options));
    var rootHandler = reader.handler(typeName);
    return reader.fromXML(xmlStr, rootHandler);
  };

  /**
   * The toXML result.
   *
   * @typedef {Object} SerializationResult
   *
   * @property {String} xml
   */

  /**
   * Serializes a DMN object tree to XML.
   *
   * @param {String}   element    the root element, typically an instance of `Definitions`
   * @param {Object}   [options]  to pass to the underlying writer
   *
   * @returns {Promise<SerializationResult, Error>}
   */
  DmnModdle.prototype.toXML = function (element, options) {
    var writer = new Writer(options);
    return new Promise(function (resolve, reject) {
      try {
        var result = writer.toXML(element);
        return resolve({
          xml: result
        });
      } catch (err) {
        return reject(err);
      }
    });
  };
  var name$4 = "DC";
  var prefix$4 = "dc";
  var uri$4 = "http://www.omg.org/spec/DMN/20180521/DC/";
  var types$4 = [{
    name: "Dimension",
    properties: [{
      name: "width",
      isAttr: true,
      type: "Real"
    }, {
      name: "height",
      isAttr: true,
      type: "Real"
    }]
  }, {
    name: "Bounds",
    properties: [{
      name: "height",
      isAttr: true,
      type: "Real"
    }, {
      name: "width",
      isAttr: true,
      type: "Real"
    }, {
      name: "x",
      isAttr: true,
      type: "Real"
    }, {
      name: "y",
      isAttr: true,
      type: "Real"
    }]
  }, {
    name: "Point",
    properties: [{
      name: "x",
      isAttr: true,
      type: "Real"
    }, {
      name: "y",
      isAttr: true,
      type: "Real"
    }]
  }, {
    name: "Color",
    properties: [{
      name: "red",
      type: "UML_Standard_Profile.mdzip:eee_1045467100323_917313_65"
    }, {
      name: "green",
      type: "UML_Standard_Profile.mdzip:eee_1045467100323_917313_65"
    }, {
      name: "blue",
      type: "UML_Standard_Profile.mdzip:eee_1045467100323_917313_65"
    }]
  }];
  var associations$3 = [];
  var enumerations$3 = [{
    name: "AlignmentKind",
    literalValues: [{
      name: "start"
    }, {
      name: "center"
    }, {
      name: "end"
    }]
  }];
  var DcPackage = {
    name: name$4,
    prefix: prefix$4,
    uri: uri$4,
    types: types$4,
    associations: associations$3,
    enumerations: enumerations$3
  };
  var name$3 = "DI";
  var prefix$3 = "di";
  var uri$3 = "http://www.omg.org/spec/DMN/20180521/DI/";
  var types$3 = [{
    name: "DiagramElement",
    isAbstract: true,
    properties: [{
      name: "extension",
      type: "Extension"
    }, {
      name: "id",
      isAttr: true,
      isId: true,
      type: "String"
    }, {
      name: "style",
      isReference: true,
      type: "Style",
      xml: {
        serialize: "property"
      }
    }, {
      name: "sharedStyle",
      isReference: true,
      isVirtual: true,
      type: "Style"
    }]
  }, {
    name: "Diagram",
    superClass: ["DiagramElement"],
    properties: [{
      name: "name",
      isAttr: true,
      type: "String"
    }, {
      name: "documentation",
      isAttr: true,
      type: "String"
    }, {
      name: "resolution",
      isAttr: true,
      type: "Real"
    }]
  }, {
    name: "Shape",
    isAbstract: true,
    properties: [{
      name: "bounds",
      type: "dc:Bounds"
    }],
    superClass: ["DiagramElement"]
  }, {
    name: "Edge",
    isAbstract: true,
    properties: [{
      name: "waypoint",
      type: "dc:Point",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }],
    superClass: ["DiagramElement"]
  }, {
    name: "Style",
    isAbstract: true,
    properties: [{
      name: "id",
      isAttr: true,
      isId: true,
      type: "String"
    }]
  }, {
    name: "Extension",
    properties: [{
      name: "values",
      isMany: true,
      type: "Element"
    }]
  }];
  var associations$2 = [];
  var enumerations$2 = [];
  var xml$2 = {
    tagAlias: "lowerCase"
  };
  var DiPackage = {
    name: name$3,
    prefix: prefix$3,
    uri: uri$3,
    types: types$3,
    associations: associations$2,
    enumerations: enumerations$2,
    xml: xml$2
  };
  var name$2 = "DMN";
  var prefix$2 = "dmn";
  var uri$2 = "https://www.omg.org/spec/DMN/20191111/MODEL/";
  var types$2 = [{
    name: "AuthorityRequirement",
    superClass: ["DMNElement"],
    properties: [{
      name: "requiredAuthority",
      type: "DMNElementReference",
      xml: {
        serialize: "property"
      }
    }, {
      name: "requiredDecision",
      type: "DMNElementReference",
      xml: {
        serialize: "property"
      }
    }, {
      name: "requiredInput",
      type: "DMNElementReference",
      xml: {
        serialize: "property"
      }
    }]
  }, {
    name: "ItemDefinition",
    superClass: ["NamedElement"],
    properties: [{
      name: "typeRef",
      type: "String"
    }, {
      name: "allowedValues",
      type: "UnaryTests",
      xml: {
        serialize: "property"
      }
    }, {
      name: "typeLanguage",
      type: "String",
      isAttr: true
    }, {
      name: "itemComponent",
      type: "ItemDefinition",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "functionItem",
      type: "FunctionItem"
    }, {
      name: "isCollection",
      isAttr: true,
      type: "Boolean"
    }]
  }, {
    name: "Definitions",
    superClass: ["NamedElement"],
    properties: [{
      name: "import",
      type: "Import",
      isMany: true
    }, {
      name: "itemDefinition",
      type: "ItemDefinition",
      isMany: true
    }, {
      name: "drgElement",
      type: "DRGElement",
      isMany: true
    }, {
      name: "artifact",
      type: "Artifact",
      isMany: true
    }, {
      name: "elementCollection",
      type: "ElementCollection",
      isMany: true
    }, {
      name: "businessContextElement",
      type: "BusinessContextElement",
      isMany: true
    }, {
      name: "namespace",
      type: "String",
      isAttr: true
    }, {
      name: "expressionLanguage",
      type: "String",
      isAttr: true
    }, {
      name: "typeLanguage",
      type: "String",
      isAttr: true
    }, {
      name: "exporter",
      isAttr: true,
      type: "String"
    }, {
      name: "exporterVersion",
      isAttr: true,
      type: "String"
    }, {
      name: "dmnDI",
      type: "dmndi:DMNDI"
    }]
  }, {
    name: "KnowledgeSource",
    superClass: ["DRGElement"],
    properties: [{
      name: "authorityRequirement",
      type: "AuthorityRequirement",
      isMany: true
    }, {
      name: "type",
      type: "String"
    }, {
      name: "owner",
      type: "DMNElementReference",
      xml: {
        serialize: "property"
      }
    }, {
      name: "locationURI",
      type: "String",
      isAttr: true
    }]
  }, {
    name: "DecisionRule",
    superClass: ["DMNElement"],
    properties: [{
      name: "inputEntry",
      type: "UnaryTests",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "outputEntry",
      type: "LiteralExpression",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "annotationEntry",
      type: "RuleAnnotation",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }]
  }, {
    name: "Expression",
    isAbstract: true,
    superClass: ["DMNElement"],
    properties: [{
      name: "typeRef",
      isAttr: true,
      type: "String"
    }]
  }, {
    name: "InformationItem",
    superClass: ["NamedElement"],
    properties: [{
      name: "typeRef",
      isAttr: true,
      type: "String"
    }]
  }, {
    name: "Decision",
    superClass: ["DRGElement"],
    properties: [{
      name: "question",
      type: "String",
      xml: {
        serialize: "property"
      }
    }, {
      name: "allowedAnswers",
      type: "String",
      xml: {
        serialize: "property"
      }
    }, {
      name: "variable",
      type: "InformationItem",
      xml: {
        serialize: "property"
      }
    }, {
      name: "informationRequirement",
      type: "InformationRequirement",
      isMany: true
    }, {
      name: "knowledgeRequirement",
      type: "KnowledgeRequirement",
      isMany: true
    }, {
      name: "authorityRequirement",
      type: "AuthorityRequirement",
      isMany: true
    }, {
      name: "supportedObjective",
      isMany: true,
      type: "DMNElementReference",
      xml: {
        serialize: "property"
      }
    }, {
      name: "impactedPerformanceIndicator",
      type: "DMNElementReference",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "decisionMaker",
      type: "DMNElementReference",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "decisionOwner",
      type: "DMNElementReference",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "usingProcess",
      isMany: true,
      type: "DMNElementReference",
      xml: {
        serialize: "property"
      }
    }, {
      name: "usingTask",
      isMany: true,
      type: "DMNElementReference",
      xml: {
        serialize: "property"
      }
    }, {
      name: "decisionLogic",
      type: "Expression"
    }]
  }, {
    name: "Invocation",
    superClass: ["Expression"],
    properties: [{
      name: "calledFunction",
      type: "Expression"
    }, {
      name: "binding",
      type: "Binding",
      isMany: true
    }]
  }, {
    name: "OrganisationalUnit",
    superClass: ["BusinessContextElement"],
    properties: [{
      name: "decisionMade",
      type: "Decision",
      isReference: true,
      isMany: true
    }, {
      name: "decisionOwned",
      type: "Decision",
      isReference: true,
      isMany: true
    }]
  }, {
    name: "Import",
    superClass: ["NamedElement"],
    properties: [{
      name: "importType",
      type: "String",
      isAttr: true
    }, {
      name: "locationURI",
      type: "String",
      isAttr: true
    }, {
      name: "namespace",
      type: "String",
      isAttr: true
    }]
  }, {
    name: "InformationRequirement",
    superClass: ["DMNElement"],
    properties: [{
      name: "requiredDecision",
      type: "DMNElementReference",
      xml: {
        serialize: "property"
      }
    }, {
      name: "requiredInput",
      type: "DMNElementReference",
      xml: {
        serialize: "property"
      }
    }]
  }, {
    name: "ElementCollection",
    superClass: ["NamedElement"],
    properties: [{
      name: "drgElement",
      type: "DMNElementReference",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }]
  }, {
    name: "DRGElement",
    isAbstract: true,
    superClass: ["NamedElement"],
    properties: []
  }, {
    name: "InputData",
    superClass: ["DRGElement"],
    properties: [{
      name: "variable",
      type: "InformationItem",
      xml: {
        serialize: "property"
      }
    }]
  }, {
    name: "DMNElement",
    isAbstract: true,
    properties: [{
      name: "description",
      type: "String"
    }, {
      name: "extensionElements",
      type: "ExtensionElements"
    }, {
      name: "id",
      type: "String",
      isAttr: true,
      isId: true
    }, {
      name: "extensionAttribute",
      type: "ExtensionAttribute",
      isMany: true
    }, {
      name: "label",
      isAttr: true,
      type: "String"
    }]
  }, {
    name: "InputClause",
    superClass: ["DMNElement"],
    properties: [{
      name: "inputExpression",
      type: "LiteralExpression",
      xml: {
        serialize: "property"
      }
    }, {
      name: "inputValues",
      type: "UnaryTests",
      xml: {
        serialize: "property"
      }
    }]
  }, {
    name: "DecisionTable",
    superClass: ["Expression"],
    properties: [{
      name: "input",
      type: "InputClause",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "output",
      type: "OutputClause",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "annotation",
      type: "RuleAnnotationClause",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "rule",
      type: "DecisionRule",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "hitPolicy",
      type: "HitPolicy",
      isAttr: true,
      "default": "UNIQUE"
    }, {
      name: "aggregation",
      type: "BuiltinAggregator",
      isAttr: true
    }, {
      name: "preferredOrientation",
      type: "DecisionTableOrientation",
      isAttr: true
    }, {
      name: "outputLabel",
      isAttr: true,
      type: "String"
    }]
  }, {
    name: "LiteralExpression",
    superClass: ["Expression"],
    properties: [{
      name: "expressionLanguage",
      type: "String",
      isAttr: true
    }, {
      name: "text",
      type: "String"
    }, {
      name: "importedValues",
      type: "ImportedValues"
    }]
  }, {
    name: "Binding",
    properties: [{
      name: "parameter",
      type: "InformationItem",
      xml: {
        serialize: "property"
      }
    }, {
      name: "bindingFormula",
      type: "Expression"
    }]
  }, {
    name: "KnowledgeRequirement",
    superClass: ["DMNElement"],
    properties: [{
      name: "requiredKnowledge",
      type: "DMNElementReference",
      xml: {
        serialize: "property"
      }
    }]
  }, {
    name: "BusinessKnowledgeModel",
    superClass: ["Invocable"],
    properties: [{
      name: "encapsulatedLogic",
      type: "FunctionDefinition",
      xml: {
        serialize: "property"
      }
    }, {
      name: "knowledgeRequirement",
      type: "KnowledgeRequirement",
      isMany: true
    }, {
      name: "authorityRequirement",
      type: "AuthorityRequirement",
      isMany: true
    }]
  }, {
    name: "BusinessContextElement",
    isAbstract: true,
    superClass: ["NamedElement"],
    properties: [{
      name: "URI",
      type: "String",
      isAttr: true
    }]
  }, {
    name: "PerformanceIndicator",
    superClass: ["BusinessContextElement"],
    properties: [{
      name: "impactingDecision",
      type: "DMNElementReference",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }]
  }, {
    name: "FunctionDefinition",
    superClass: ["Expression"],
    properties: [{
      name: "formalParameter",
      type: "InformationItem",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "body",
      type: "Expression"
    }, {
      name: "kind",
      type: "FunctionKind",
      isAttr: true
    }]
  }, {
    name: "Context",
    superClass: ["Expression"],
    properties: [{
      name: "contextEntry",
      type: "ContextEntry",
      isMany: true
    }]
  }, {
    name: "ContextEntry",
    superClass: ["DMNElement"],
    properties: [{
      name: "variable",
      type: "InformationItem",
      xml: {
        serialize: "property"
      }
    }, {
      name: "value",
      type: "Expression"
    }]
  }, {
    name: "List",
    superClass: ["Expression"],
    properties: [{
      name: "elements",
      isMany: true,
      type: "Expression"
    }]
  }, {
    name: "Relation",
    superClass: ["Expression"],
    properties: [{
      name: "column",
      type: "InformationItem",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "row",
      type: "List",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }]
  }, {
    name: "OutputClause",
    superClass: ["DMNElement"],
    properties: [{
      name: "outputValues",
      type: "UnaryTests",
      xml: {
        serialize: "property"
      }
    }, {
      name: "defaultOutputEntry",
      type: "LiteralExpression",
      xml: {
        serialize: "property"
      }
    }, {
      name: "name",
      isAttr: true,
      type: "String"
    }, {
      name: "typeRef",
      isAttr: true,
      type: "String"
    }]
  }, {
    name: "UnaryTests",
    superClass: ["Expression"],
    properties: [{
      name: "text",
      type: "String"
    }, {
      name: "expressionLanguage",
      type: "String",
      isAttr: true
    }]
  }, {
    name: "NamedElement",
    isAbstract: true,
    superClass: ["DMNElement"],
    properties: [{
      name: "name",
      isAttr: true,
      type: "String"
    }]
  }, {
    name: "ImportedValues",
    superClass: ["Import"],
    properties: [{
      name: "importedElement",
      type: "String"
    }, {
      name: "expressionLanguage",
      type: "String",
      isAttr: true
    }]
  }, {
    name: "DecisionService",
    superClass: ["Invocable"],
    properties: [{
      name: "outputDecision",
      type: "DMNElementReference",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "encapsulatedDecision",
      type: "DMNElementReference",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "inputDecision",
      type: "DMNElementReference",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "inputData",
      type: "DMNElementReference",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }]
  }, {
    name: "ExtensionElements",
    properties: [{
      name: "values",
      type: "Element",
      isMany: true
    }]
  }, {
    name: "ExtensionAttribute",
    properties: [{
      name: "value",
      type: "Element"
    }, {
      name: "valueRef",
      type: "Element",
      isAttr: true,
      isReference: true
    }, {
      name: "name",
      isAttr: true,
      type: "String"
    }]
  }, {
    name: "Element",
    isAbstract: true,
    properties: [{
      name: "extensionAttribute",
      type: "ExtensionAttribute",
      isAttr: true,
      isReference: true
    }, {
      name: "elements",
      type: "ExtensionElements",
      isAttr: true,
      isReference: true
    }]
  }, {
    name: "Artifact",
    isAbstract: true,
    superClass: ["DMNElement"],
    properties: []
  }, {
    name: "Association",
    superClass: ["Artifact"],
    properties: [{
      name: "sourceRef",
      type: "DMNElementReference",
      xml: {
        serialize: "property"
      }
    }, {
      name: "targetRef",
      type: "DMNElementReference",
      xml: {
        serialize: "property"
      }
    }, {
      name: "associationDirection",
      type: "AssociationDirection",
      isAttr: true
    }]
  }, {
    name: "TextAnnotation",
    superClass: ["Artifact"],
    properties: [{
      name: "text",
      type: "String"
    }, {
      name: "textFormat",
      isAttr: true,
      type: "String",
      "default": "text/plain"
    }]
  }, {
    name: "RuleAnnotationClause",
    properties: [{
      name: "name",
      isAttr: true,
      type: "String"
    }]
  }, {
    name: "RuleAnnotation",
    properties: [{
      name: "text",
      type: "String"
    }]
  }, {
    name: "Invocable",
    isAbstract: true,
    superClass: ["DRGElement"],
    properties: [{
      name: "variable",
      type: "InformationItem",
      xml: {
        serialize: "property"
      }
    }]
  }, {
    name: "Group",
    superClass: ["Artifact"],
    properties: [{
      name: "name",
      isAttr: true,
      type: "String"
    }]
  }, {
    name: "FunctionItem",
    superClass: ["DMNElement"],
    properties: [{
      name: "parameters",
      isMany: true,
      type: "InformationItem",
      xml: {
        serialize: "property"
      }
    }, {
      name: "outputTypeRef",
      isAttr: true,
      type: "String"
    }]
  }, {
    name: "DMNElementReference",
    properties: [{
      isAttr: true,
      name: "href",
      type: "String"
    }]
  }];
  var enumerations$1 = [{
    name: "HitPolicy",
    literalValues: [{
      name: "UNIQUE"
    }, {
      name: "FIRST"
    }, {
      name: "PRIORITY"
    }, {
      name: "ANY"
    }, {
      name: "COLLECT"
    }, {
      name: "RULE ORDER"
    }, {
      name: "OUTPUT ORDER"
    }]
  }, {
    name: "BuiltinAggregator",
    literalValues: [{
      name: "SUM"
    }, {
      name: "COUNT"
    }, {
      name: "MIN"
    }, {
      name: "MAX"
    }]
  }, {
    name: "DecisionTableOrientation",
    literalValues: [{
      name: "Rule-as-Row"
    }, {
      name: "Rule-as-Column"
    }, {
      name: "CrossTable"
    }]
  }, {
    name: "AssociationDirection",
    literalValues: [{
      name: "None"
    }, {
      name: "One"
    }, {
      name: "Both"
    }]
  }, {
    name: "FunctionKind",
    literalValues: [{
      name: "FEEL"
    }, {
      name: "Java"
    }, {
      name: "PMML"
    }]
  }];
  var associations$1 = [];
  var xml$1 = {
    tagAlias: "lowerCase"
  };
  var DmnPackage = {
    name: name$2,
    prefix: prefix$2,
    uri: uri$2,
    types: types$2,
    enumerations: enumerations$1,
    associations: associations$1,
    xml: xml$1
  };
  var name$1 = "DMNDI";
  var prefix$1 = "dmndi";
  var uri$1 = "https://www.omg.org/spec/DMN/20191111/DMNDI/";
  var types$1$1 = [{
    name: "DMNDI",
    properties: [{
      name: "diagrams",
      type: "DMNDiagram",
      isMany: true
    }, {
      name: "styles",
      type: "DMNStyle",
      isMany: true
    }]
  }, {
    name: "DMNStyle",
    superClass: ["di:Style"],
    properties: [{
      name: "fillColor",
      type: "dc:Color",
      isAttr: true
    }, {
      name: "strokeColor",
      type: "dc:Color",
      isAttr: true
    }, {
      name: "fontColor",
      type: "dc:Color",
      isAttr: true
    }, {
      name: "fontSize",
      isAttr: true,
      type: "Real"
    }, {
      name: "fontFamily",
      isAttr: true,
      type: "String"
    }, {
      name: "fontItalic",
      isAttr: true,
      type: "Boolean"
    }, {
      name: "fontBold",
      isAttr: true,
      type: "Boolean"
    }, {
      name: "fontUnderline",
      isAttr: true,
      type: "Boolean"
    }, {
      name: "fontStrikeThrough",
      isAttr: true,
      type: "Boolean"
    }, {
      name: "labelHorizontalAlignment",
      type: "dc:AlignmentKind",
      isAttr: true
    }, {
      name: "labelVerticalAlignment",
      type: "dc:AlignmentKind",
      isAttr: true
    }]
  }, {
    name: "DMNDiagram",
    superClass: ["di:Diagram"],
    properties: [{
      name: "dmnElementRef",
      type: "dmn:DMNElement",
      isAttr: true,
      isReference: true
    }, {
      name: "size",
      type: "Size"
    }, {
      name: "localStyle",
      type: "DMNStyle",
      isVirtual: true
    }, {
      name: "sharedStyle",
      type: "DMNStyle",
      isVirtual: true,
      isReference: true,
      redefines: "di:DiagramElement#sharedStyle"
    }, {
      name: "diagramElements",
      type: "DMNDiagramElement",
      isMany: true
    }]
  }, {
    name: "DMNDiagramElement",
    isAbstract: true,
    superClass: ["di:DiagramElement"],
    properties: [{
      name: "dmnElementRef",
      type: "dmn:DMNElement",
      isAttr: true,
      isReference: true
    }, {
      name: "sharedStyle",
      type: "DMNStyle",
      isVirtual: true,
      isReference: true,
      redefines: "di:DiagramElement#sharedStyle"
    }, {
      name: "localStyle",
      type: "DMNStyle",
      isVirtual: true
    }, {
      name: "label",
      type: "DMNLabel"
    }]
  }, {
    name: "DMNLabel",
    superClass: ["di:Shape"],
    properties: [{
      name: "text",
      type: "Text"
    }]
  }, {
    name: "DMNShape",
    superClass: ["di:Shape", "DMNDiagramElement"],
    properties: [{
      name: "isListedInputData",
      isAttr: true,
      type: "Boolean"
    }, {
      name: "decisionServiceDividerLine",
      type: "DMNDecisionServiceDividerLine"
    }, {
      name: "isCollapsed",
      isAttr: true,
      type: "Boolean"
    }]
  }, {
    name: "DMNEdge",
    superClass: ["di:Edge", "DMNDiagramElement"],
    properties: [{
      name: "sourceElement",
      type: "DMNDiagramElement",
      isAttr: true,
      isReference: true
    }, {
      name: "targetElement",
      type: "DMNDiagramElement",
      isAttr: true,
      isReference: true
    }]
  }, {
    name: "DMNDecisionServiceDividerLine",
    superClass: ["di:Edge"]
  }, {
    name: "Text",
    properties: [{
      name: "text",
      isBody: true,
      type: "String"
    }]
  }, {
    name: "Size",
    superClass: ["dc:Dimension"]
  }];
  var associations = [];
  var enumerations = [];
  var DmnDiPackage = {
    name: name$1,
    prefix: prefix$1,
    uri: uri$1,
    types: types$1$1,
    associations: associations,
    enumerations: enumerations
  };
  var name$5 = "bpmn.io DI for DMN";
  var uri = "http://bpmn.io/schema/dmn/biodi/2.0";
  var prefix$5 = "biodi";
  var xml = {
    tagAlias: "lowerCase"
  };
  var types$5 = [{
    name: "DecisionTable",
    isAbstract: true,
    "extends": ["dmn:DecisionTable"],
    properties: [{
      name: "annotationsWidth",
      isAttr: true,
      type: "Integer"
    }]
  }, {
    name: "OutputClause",
    isAbstract: true,
    "extends": ["dmn:OutputClause"],
    properties: [{
      name: "width",
      isAttr: true,
      type: "Integer"
    }]
  }, {
    name: "InputClause",
    isAbstract: true,
    "extends": ["dmn:InputClause"],
    properties: [{
      name: "width",
      isAttr: true,
      type: "Integer"
    }]
  }];
  var BioDiPackage = {
    name: name$5,
    uri: uri,
    prefix: prefix$5,
    xml: xml,
    types: types$5
  };
  var packages = {
    dc: DcPackage,
    di: DiPackage,
    dmn: DmnPackage,
    dmndi: DmnDiPackage,
    biodi: BioDiPackage
  };
  function simple(additionalPackages, options) {
    var pks = assign$3({}, packages, additionalPackages);
    return new DmnModdle(pks, options);
  }

  function _mergeNamespaces$1(n, m) {
    m.forEach(function (e) {
      e && typeof e !== 'string' && !Array.isArray(e) && Object.keys(e).forEach(function (k) {
        if (k !== 'default' && !(k in n)) {
          var d = Object.getOwnPropertyDescriptor(e, k);
          Object.defineProperty(n, k, d.get ? d : {
            enumerable: true,
            get: function () {
              return e[k];
            }
          });
        }
      });
    });
    return Object.freeze(n);
  }

  /**
   * Flatten array, one level deep.
   *
   * @template T
   *
   * @param {T[][] | T[] | null} [arr]
   *
   * @return {T[]}
   */

  const nativeToString = Object.prototype.toString;
  const nativeHasOwnProperty = Object.prototype.hasOwnProperty;
  function isUndefined$1(obj) {
    return obj === undefined;
  }
  function isArray$2(obj) {
    return nativeToString.call(obj) === '[object Array]';
  }

  /**
   * Return true, if target owns a property with the given key.
   *
   * @param {Object} target
   * @param {String} key
   *
   * @return {Boolean}
   */
  function has(target, key) {
    return nativeHasOwnProperty.call(target, key);
  }

  /**
   * Iterate over collection; returning something
   * (non-undefined) will stop iteration.
   *
   * @template T
   * @param {Collection<T>} collection
   * @param { ((item: T, idx: number) => (boolean|void)) | ((item: T, key: string) => (boolean|void)) } iterator
   *
   * @return {T} return result that stopped the iteration
   */
  function forEach(collection, iterator) {
    let val, result;
    if (isUndefined$1(collection)) {
      return;
    }
    const convertKey = isArray$2(collection) ? toNum : identity;
    for (let key in collection) {
      if (has(collection, key)) {
        val = collection[key];
        result = iterator(val, convertKey(key));
        if (result === false) {
          return val;
        }
      }
    }
  }
  function identity(arg) {
    return arg;
  }
  function toNum(arg) {
    return Number(arg);
  }

  /**
   * Assigns style attributes in a style-src compliant way.
   *
   * @param {Element} element
   * @param {...Object} styleSources
   *
   * @return {Element} the element
   */
  function assign(element, ...styleSources) {
    const target = element.style;
    forEach(styleSources, function (style) {
      if (!style) {
        return;
      }
      forEach(style, function (value, key) {
        target[key] = value;
      });
    });
    return element;
  }

  /**
   * Set attribute `name` to `val`, or get attr `name`.
   *
   * @param {Element} el
   * @param {String} name
   * @param {String} [val]
   * @api public
   */
  function attr$1(el, name, val) {
    // get
    if (arguments.length == 2) {
      return el.getAttribute(name);
    }

    // remove
    if (val === null) {
      return el.removeAttribute(name);
    }

    // set
    el.setAttribute(name, val);
    return el;
  }

  /**
   * Taken from https://github.com/component/classes
   *
   * Without the component bits.
   */

  /**
   * toString reference.
   */

  const toString$1 = Object.prototype.toString;

  /**
   * Wrap `el` in a `ClassList`.
   *
   * @param {Element} el
   * @return {ClassList}
   * @api public
   */

  function classes$1(el) {
    return new ClassList$1(el);
  }

  /**
   * Initialize a new ClassList for `el`.
   *
   * @param {Element} el
   * @api private
   */

  function ClassList$1(el) {
    if (!el || !el.nodeType) {
      throw new Error('A DOM element reference is required');
    }
    this.el = el;
    this.list = el.classList;
  }

  /**
   * Add class `name` if not already present.
   *
   * @param {String} name
   * @return {ClassList}
   * @api public
   */

  ClassList$1.prototype.add = function (name) {
    this.list.add(name);
    return this;
  };

  /**
   * Remove class `name` when present, or
   * pass a regular expression to remove
   * any which match.
   *
   * @param {String|RegExp} name
   * @return {ClassList}
   * @api public
   */

  ClassList$1.prototype.remove = function (name) {
    if ('[object RegExp]' == toString$1.call(name)) {
      return this.removeMatching(name);
    }
    this.list.remove(name);
    return this;
  };

  /**
   * Remove all classes matching `re`.
   *
   * @param {RegExp} re
   * @return {ClassList}
   * @api private
   */

  ClassList$1.prototype.removeMatching = function (re) {
    const arr = this.array();
    for (let i = 0; i < arr.length; i++) {
      if (re.test(arr[i])) {
        this.remove(arr[i]);
      }
    }
    return this;
  };

  /**
   * Toggle class `name`, can force state via `force`.
   *
   * For browsers that support classList, but do not support `force` yet,
   * the mistake will be detected and corrected.
   *
   * @param {String} name
   * @param {Boolean} force
   * @return {ClassList}
   * @api public
   */

  ClassList$1.prototype.toggle = function (name, force) {
    if ('undefined' !== typeof force) {
      if (force !== this.list.toggle(name, force)) {
        this.list.toggle(name); // toggle again to correct
      }
    } else {
      this.list.toggle(name);
    }
    return this;
  };

  /**
   * Return an array of classes.
   *
   * @return {Array}
   * @api public
   */

  ClassList$1.prototype.array = function () {
    return Array.from(this.list);
  };

  /**
   * Check if class `name` is present.
   *
   * @param {String} name
   * @return {ClassList}
   * @api public
   */

  ClassList$1.prototype.has = ClassList$1.prototype.contains = function (name) {
    return this.list.contains(name);
  };

  /**
   * Clear utility
   */

  /**
   * Removes all children from the given element
   *
   * @param {Element} element
   *
   * @return {Element} the element (for chaining)
   */
  function clear(element) {
    var child;
    while (child = element.firstChild) {
      element.removeChild(child);
    }
    return element;
  }

  /**
   * Closest
   *
   * @param {Element} el
   * @param {string} selector
   * @param {boolean} checkYourSelf (optional)
   */
  function closest(element, selector, checkYourSelf) {
    var actualElement = checkYourSelf ? element : element.parentNode;
    return actualElement && typeof actualElement.closest === 'function' && actualElement.closest(selector) || null;
  }
  var componentEvent = {};
  var bind$1, unbind$1, prefix;
  function detect() {
    bind$1 = window.addEventListener ? 'addEventListener' : 'attachEvent';
    unbind$1 = window.removeEventListener ? 'removeEventListener' : 'detachEvent';
    prefix = bind$1 !== 'addEventListener' ? 'on' : '';
  }

  /**
   * Bind `el` event `type` to `fn`.
   *
   * @param {Element} el
   * @param {String} type
   * @param {Function} fn
   * @param {Boolean} capture
   * @return {Function}
   * @api public
   */

  var bind_1 = componentEvent.bind = function (el, type, fn, capture) {
    if (!bind$1) detect();
    el[bind$1](prefix + type, fn, capture || false);
    return fn;
  };

  /**
   * Unbind `el` event `type`'s callback `fn`.
   *
   * @param {Element} el
   * @param {String} type
   * @param {Function} fn
   * @param {Boolean} capture
   * @return {Function}
   * @api public
   */

  var unbind_1 = componentEvent.unbind = function (el, type, fn, capture) {
    if (!unbind$1) detect();
    el[unbind$1](prefix + type, fn, capture || false);
    return fn;
  };
  var event = /*#__PURE__*/_mergeNamespaces$1({
    __proto__: null,
    bind: bind_1,
    unbind: unbind_1,
    'default': componentEvent
  }, [componentEvent]);

  /**
   * Module dependencies.
   */

  /**
   * Delegate event `type` to `selector`
   * and invoke `fn(e)`. A callback function
   * is returned which may be passed to `.unbind()`.
   *
   * @param {Element} el
   * @param {String} selector
   * @param {String} type
   * @param {Function} fn
   * @param {Boolean} capture
   * @return {Function}
   * @api public
   */

  // Some events don't bubble, so we want to bind to the capture phase instead
  // when delegating.
  var forceCaptureEvents = ['focus', 'blur'];
  function bind(el, selector, type, fn, capture) {
    if (forceCaptureEvents.indexOf(type) !== -1) {
      capture = true;
    }
    return event.bind(el, type, function (e) {
      var target = e.target || e.srcElement;
      e.delegateTarget = closest(target, selector, true);
      if (e.delegateTarget) {
        fn.call(el, e);
      }
    }, capture);
  }

  /**
   * Unbind event `type`'s callback `fn`.
   *
   * @param {Element} el
   * @param {String} type
   * @param {Function} fn
   * @param {Boolean} capture
   * @api public
   */
  function unbind(el, type, fn, capture) {
    if (forceCaptureEvents.indexOf(type) !== -1) {
      capture = true;
    }
    return event.unbind(el, type, fn, capture);
  }
  var delegate = {
    bind,
    unbind
  };

  /**
   * Expose `parse`.
   */

  var domify = parse$1;

  /**
   * Tests for browser support.
   */

  var innerHTMLBug = false;
  var bugTestDiv;
  if (typeof document !== 'undefined') {
    bugTestDiv = document.createElement('div');
    // Setup
    bugTestDiv.innerHTML = '  <link/><table></table><a href="/a">a</a><input type="checkbox"/>';
    // Make sure that link elements get serialized correctly by innerHTML
    // This requires a wrapper element in IE
    innerHTMLBug = !bugTestDiv.getElementsByTagName('link').length;
    bugTestDiv = undefined;
  }

  /**
   * Wrap map from jquery.
   */

  var map = {
    legend: [1, '<fieldset>', '</fieldset>'],
    tr: [2, '<table><tbody>', '</tbody></table>'],
    col: [2, '<table><tbody></tbody><colgroup>', '</colgroup></table>'],
    // for script/link/style tags to work in IE6-8, you have to wrap
    // in a div with a non-whitespace character in front, ha!
    _default: innerHTMLBug ? [1, 'X<div>', '</div>'] : [0, '', '']
  };
  map.td = map.th = [3, '<table><tbody><tr>', '</tr></tbody></table>'];
  map.option = map.optgroup = [1, '<select multiple="multiple">', '</select>'];
  map.thead = map.tbody = map.colgroup = map.caption = map.tfoot = [1, '<table>', '</table>'];
  map.polyline = map.ellipse = map.polygon = map.circle = map.text = map.line = map.path = map.rect = map.g = [1, '<svg xmlns="http://www.w3.org/2000/svg" version="1.1">', '</svg>'];

  /**
   * Parse `html` and return a DOM Node instance, which could be a TextNode,
   * HTML DOM Node of some kind (<div> for example), or a DocumentFragment
   * instance, depending on the contents of the `html` string.
   *
   * @param {String} html - HTML string to "domify"
   * @param {Document} doc - The `document` instance to create the Node for
   * @return {DOMNode} the TextNode, DOM Node, or DocumentFragment instance
   * @api private
   */

  function parse$1(html, doc) {
    if ('string' != typeof html) throw new TypeError('String expected');

    // default to the global `document` object
    if (!doc) doc = document;

    // tag name
    var m = /<([\w:]+)/.exec(html);
    if (!m) return doc.createTextNode(html);
    html = html.replace(/^\s+|\s+$/g, ''); // Remove leading/trailing whitespace

    var tag = m[1];

    // body support
    if (tag == 'body') {
      var el = doc.createElement('html');
      el.innerHTML = html;
      return el.removeChild(el.lastChild);
    }

    // wrap map
    var wrap = Object.prototype.hasOwnProperty.call(map, tag) ? map[tag] : map._default;
    var depth = wrap[0];
    var prefix = wrap[1];
    var suffix = wrap[2];
    var el = doc.createElement('div');
    el.innerHTML = prefix + html + suffix;
    while (depth--) el = el.lastChild;

    // one element
    if (el.firstChild == el.lastChild) {
      return el.removeChild(el.firstChild);
    }

    // several elements
    var fragment = doc.createDocumentFragment();
    while (el.firstChild) {
      fragment.appendChild(el.removeChild(el.firstChild));
    }
    return fragment;
  }
  var domify$1 = domify;
  function query(selector, el) {
    el = el || document;
    return el.querySelector(selector);
  }
  function all(selector, el) {
    el = el || document;
    return el.querySelectorAll(selector);
  }
  function remove$4(el) {
    el.parentNode && el.parentNode.removeChild(el);
  }

  // TODO: remove with future dmn-js version

  /**
   * Wraps APIs to check:
   *
   * 1) If a callback is passed -> Warn users about callback deprecation.
   * 2) If Promise class is implemented in current environment.
   *
   * @private
   */
  function wrapForCompatibility(api) {
    return function () {
      if (!window.Promise) {
        throw new Error('Promises is not supported in this environment.' + ' Please polyfill Promise.');
      }
      var argLen = arguments.length;
      if (argLen >= 1 && isFunction$2(arguments[argLen - 1])) {
        var callback = arguments[argLen - 1];
        console.warn(new Error('Passing callbacks to ' + replaceBoundPrefix(api.name) + ' is deprecated and will be removed in a future major release. ' + 'Please switch to promises: https://bpmn.io/l/moving-to-promises.html'));
        var argsWithoutCallback = Array.prototype.slice.call(arguments, 0, -1);
        api.apply(this, argsWithoutCallback).then(function (result) {
          var firstKey = Object.keys(result)[0];

          // The APIs we are wrapping all resolve a single item depending on the API.
          // For instance, importXML resolves { warnings } and saveXML returns { xml }.
          // That's why we can call the callback with the first item of result.
          return callback(null, result[firstKey]);

          // Passing a second paramter instead of catch because we don't want to
          // catch errors thrown by callback().
        }, function (err) {
          return callback(err, err.warnings);
        });
      } else {
        return api.apply(this, arguments);
      }
    };
  }

  // helper ////////

  /**
   * replaceBoundPrefix - replace the <bound > prefix from a string. Can be used
   * when logging the name of an API, not being sure whether is was bound or not.
   *
   * @param  {string} string
   * @return {string} the string without the <bound > prefix. If no <bound > prefix
   * was present, the same string will be returned.
   */
  function replaceBoundPrefix(string) {
    return string.replace('bound ', '');
  }

  /**
   * @typedef {import('./View').OpenResult} OpenResult
   */

  /**
   * @typedef {import('./View').OpenError} OpenError
   */

  const DEFAULT_CONTAINER_OPTIONS = {
    width: '100%',
    height: '100%',
    position: 'relative'
  };

  /**
   * The base class for DMN viewers and editors.
   *
   * @abstract
   */
  class Manager {
    /**
     * Create a new instance with the given options.
     *
     * @param  {Object} options
     *
     * @return {Manager}
     */
    constructor(options = {}) {
      this._eventBus = new EventBus();
      this._viewsChanged = debounce(this._viewsChanged, 0);
      this._views = [];
      this._viewers = {};

      // keep support for callbacks
      this.open = wrapForCompatibility(this.open.bind(this));
      this.importXML = wrapForCompatibility(this.importXML.bind(this));
      this.saveXML = wrapForCompatibility(this.saveXML.bind(this));
      this._init(options);
    }

    /**
    * The importXML result.
    *
    * @typedef {Object} ImportXMLResult
    *
    * @property {Array<string>} warnings
    */

    /**
    * The importXML error.
    *
    * @typedef {Error} ImportXMLError
    *
    * @property {Array<string>} warnings
    */

    /**
     * Parse and render a DMN diagram.
     *
     * Once finished the viewer reports back the result to the
     * provided callback function with (err, warnings).
     *
     * ## Life-Cycle Events
     *
     * During import the viewer will fire life-cycle events:
     *
     *   * import.parse.start (about to read model from xml)
     *   * import.parse.complete (model read; may have worked or not)
     *   * import.render.start (graphical import start)
     *   * import.render.complete (graphical import finished)
     *   * import.done (everything done)
     *
     * You can use these events to hook into the life-cycle.
     *
     * @param {string} xml the DMN xml
     * @param {Object} [options]
     * @param {boolean} [options.open=true]
     *
     * @return {Promise<ImportXMLResult, ImportXMLError>}
     */
    importXML(xml, options) {
      var self = this;
      options = options || {
        open: true
      };
      return new Promise(function (resolve, reject) {
        var previousActiveView = self._activeView;

        // clean up previously rendered diagram before new import
        self._clear().then(() => {
          // hook in pre-parse listeners +
          // allow xml manipulation
          xml = self._emit('import.parse.start', {
            xml: xml
          }) || xml;
          var parseWarnings;
          self._moddle.fromXML(xml, 'dmn:Definitions').then(parseResult => {
            var definitions = parseResult.rootElement;
            var references = parseResult.references;
            var elementsById = parseResult.elementsById;
            parseWarnings = parseResult.warnings;

            // hook in post parse listeners +
            // allow definitions manipulation
            definitions = self._emit('import.parse.complete', ParseCompleteEvent({
              error: null,
              definitions: definitions,
              elementsById: elementsById,
              references: references,
              warnings: parseWarnings
            })) || definitions;
            self._setDefinitions(definitions);
            if (!options.open) {
              self._emit('import.done', {
                error: null,
                warnings: parseWarnings
              });
              resolve({
                warnings: parseWarnings
              });
              return;
            }

            // open either previously active view or view of the same type if available
            var view = self._getInitialView(self._views, previousActiveView);
            if (!view) {
              var noDisplayableContentsErr = new Error('no displayable contents');
              self._emit('import.done', {
                error: noDisplayableContentsErr,
                warnings: parseWarnings
              });
              noDisplayableContentsErr.warnings = parseWarnings;
              return reject(noDisplayableContentsErr);
            }
            self.open(view).then(result => ({
              warnings: result.warnings
            })).catch(error => ({
              error: error,
              warnings: error.warnings
            })).then(result => {
              var allWarnings = [].concat(parseWarnings, result.warnings);
              self._emit('import.done', {
                error: result.error,
                warnings: allWarnings
              });
              if (result.error) {
                result.error.warnings = allWarnings;
                reject(result.error);
              } else {
                resolve({
                  warnings: allWarnings
                });
              }
            });
          }).catch(parseError => {
            parseWarnings = parseError.warnings;
            parseError = checkDMNCompatibilityError(parseError, xml) || checkValidationError(parseError) || parseError;
            self._emit('import.parse.complete', ParseCompleteEvent({
              error: parseError,
              warnings: parseWarnings
            }));
            self._emit('import.done', {
              error: parseError,
              warnings: parseWarnings
            });
            parseError.warnings = parseWarnings;
            return reject(parseError);
          });
        }).catch(clearError => {
          self._emit('import.done', {
            error: clearError,
            warnings: []
          });
          clearError.warnings = [];
          return reject(clearError);
        });
      });

      // TODO: remove with future dmn-js version
      function ParseCompleteEvent(data) {
        var event = self._eventBus.createEvent(data);
        Object.defineProperty(event, 'context', {
          enumerable: true,
          get: function () {
            console.warn(new Error('import.parse.complete <context> is deprecated ' + 'and will be removed in future library versions'));
            return {
              warnings: data.warnings,
              references: data.references,
              elementsById: data.elementsById
            };
          }
        });
        return event;
      }
    }
    getDefinitions() {
      return this._definitions;
    }

    /**
     * Return active view.
     *
     * @return {View}
     */
    getActiveView() {
      return this._activeView;
    }

    /**
     * Get the currently active viewer instance.
     *
     * @return {View}
     */
    getActiveViewer() {
      var activeView = this.getActiveView();
      return activeView && this._getViewer(activeView);
    }
    getView(element) {
      return this._views.filter(function (v) {
        return v.element === element;
      })[0];
    }
    getViews() {
      return this._views;
    }

    /**
     * The saveXML result.
     *
     * @typedef {Object} SaveXMLResult
     *
     * @property {string} xml
     */

    /**
     * Export the currently displayed DMN diagram as
     * a DMN XML document.
     *
     * ## Life-Cycle Events
     *
     * During XML saving the viewer will fire life-cycle events:
     *
     *   * saveXML.start (before serialization)
     *   * saveXML.serialized (after xml generation)
     *   * saveXML.done (everything done)
     *
     * You can use these events to hook into the life-cycle.
     *
     * @param {Object} [options] export options
     * @param {boolean} [options.format=false] output formated XML
     * @param {boolean} [options.preamble=true] output preamble
     *
     * @return {Promise<SaveXMLResult, Error>}
     */
    saveXML(options) {
      var self = this;
      options = options || {};
      var definitions = this._definitions;
      return new Promise(function (resolve, reject) {
        if (!definitions) {
          reject(new Error('no definitions loaded'));
          return;
        }

        // allow to fiddle around with definitions
        definitions = self._emit('saveXML.start', {
          definitions: definitions
        }) || definitions;
        self._moddle.toXML(definitions, options).then(function (result) {
          var xml = result.xml;
          xml = self._emit('saveXML.serialized', {
            xml: xml
          }) || xml;
          return {
            xml
          };
        }).catch(error => ({
          error
        })).then(result => {
          self._emit('saveXML.done', result);
          if (result.error) {
            reject(result.error);
          } else {
            resolve({
              xml: result.xml
            });
          }
        });
      });
    }

    /**
     * Register an event listener
     *
     * Remove a previously added listener via {@link #off(event, callback)}.
     *
     * @param {string} event
     * @param {number} [priority]
     * @param {Function} callback
     * @param {Object} [that]
     */
    on(...args) {
      this._eventBus.on(...args);
    }

    /**
     * De-register an event listener
     *
     * @param {string} event
     * @param {Function} callback
     */
    off(...args) {
      this._eventBus.off(...args);
    }

    /**
     * Register a listener to be invoked once only.
     *
     * @param {string} event
     * @param {number} [priority]
     * @param {Function} callback
     * @param {Object} [that]
     */
    once(...args) {
      this._eventBus.once(...args);
    }
    attachTo(parentNode) {
      // unwrap jQuery if provided
      if (parentNode.get && parentNode.constructor.prototype.jquery) {
        parentNode = parentNode.get(0);
      }
      if (typeof parentNode === 'string') {
        parentNode = query(parentNode);
      }
      parentNode.appendChild(this._container);
      this._emit('attach', {});
    }
    detach() {
      this._emit('detach', {});
      remove$4(this._container);
    }
    destroy() {
      Object.keys(this._viewers).forEach(viewerId => {
        var viewer = this._viewers[viewerId];
        safeExecute(viewer, 'destroy');
      });
      remove$4(this._container);
    }
    _init(options) {
      this._options = options;
      this._moddle = this._createModdle(options);
      this._viewers = {};
      this._views = [];
      const container = domify$1('<div class="dmn-js-parent"></div>');
      const containerOptions = assign$4({}, DEFAULT_CONTAINER_OPTIONS, options);
      assign$4(container.style, {
        width: ensureUnit(containerOptions.width),
        height: ensureUnit(containerOptions.height),
        position: containerOptions.position
      });
      this._container = container;
      if (options.container) {
        this.attachTo(options.container);
      }
    }
    _clear() {
      return this._switchView(null);
    }

    /**
     * Open diagram view.
     *
     * @param  {View} view
     * @returns {Promise} Resolves with {OpenResult} when successful
     * or rejects with {OpenError}
     */
    open(view) {
      return this._switchView(view);
    }
    _setDefinitions(definitions) {
      this._definitions = definitions;
      this._updateViews();
    }
    _viewsChanged = () => {
      this._emit('views.changed', {
        views: this._views,
        activeView: this._activeView
      });
    };

    /**
     * Recompute changed views after elements in
     * the DMN diagram have changed.
     */
    _updateViews() {
      var definitions = this._definitions;
      if (!definitions) {
        this._views = [];
        this._switchView(null);
        return;
      }
      var viewProviders = this._getViewProviders();
      var displayableElements = [definitions, ...(definitions.drgElement || [])];

      // compute list of available views
      var views = this._views,
        newViews = [];
      for (var element of displayableElements) {
        var provider = find$2(viewProviders, function (provider) {
          if (typeof provider.opens === 'string') {
            return provider.opens === element.$type;
          } else {
            return provider.opens(element);
          }
        });
        if (!provider) {
          continue;
        }
        var view = {
          element,
          id: element.id,
          name: element.name,
          type: provider.id
        };
        newViews.push(view);
      }
      var activeView = this._activeView,
        newActiveView;
      if (activeView) {
        // check the new active view
        newActiveView = find$2(newViews, function (view) {
          return viewsEqual(activeView, view);
        }) || this._getInitialView(newViews);
        if (!newActiveView) {
          this._switchView(null);
          return;
        }
      }

      // Views have changed if
      // active view has changed OR
      // number of views has changed OR
      // not all views equal
      var activeViewChanged = !viewsEqual(activeView, newActiveView) || viewNameChanged(activeView, newActiveView);
      var viewsChanged = views.length !== newViews.length || !every(newViews, function (newView) {
        return find$2(views, function (view) {
          return viewsEqual(view, newView) && !viewNameChanged(view, newView);
        });
      });
      this._activeView = newActiveView;
      this._views = newViews;
      if (activeViewChanged || viewsChanged) {
        this._viewsChanged();
      }
    }
    _getInitialView(views, preferredView) {
      var initialView;
      if (preferredView) {
        initialView = find$2(views, function (view) {
          return viewsEqual(view, preferredView);
        }) || find$2(views, function (view) {
          return view.type === preferredView;
        });
      }
      return initialView || views[0];
    }

    /**
     * Switch to another view.
     *
     * @param  {View} newView
     * @returns {Promise} Resolves with {OpenResult} when successful
     * or rejects with {OpenError}
     */
    _switchView(newView) {
      var self = this;
      return new Promise(function (resolve, reject) {
        var complete = (openError, openResult) => {
          self._viewsChanged();
          if (openError) {
            reject(openError);
          } else {
            resolve(openResult);
          }
        };
        var activeView = self.getActiveView(),
          activeViewer;
        var newViewer = newView && self._getViewer(newView),
          element = newView && newView.element;
        if (activeView) {
          activeViewer = self._getViewer(activeView);
          if (activeViewer !== newViewer) {
            safeExecute(activeViewer, 'clear');
            activeViewer.detach();
          }
        }
        self._activeView = newView;
        if (newViewer) {
          if (activeViewer !== newViewer) {
            newViewer.attachTo(self._container);
          }
          self._emit('import.render.start', {
            view: newView,
            element: element
          });
          newViewer.open(element).then(result => {
            self._emit('import.render.complete', {
              view: newView,
              error: null,
              warnings: result.warnings
            });
            complete(null, result);
          }).catch(error => {
            self._emit('import.render.complete', {
              view: newView,
              error: error,
              warnings: error.warnings
            });
            complete(error, null);
          });
          return;
        }

        // no active view
        complete();
      });
    }
    _getViewer(view) {
      var type = view.type;
      var viewer = this._viewers[type];
      if (!viewer) {
        viewer = this._viewers[type] = this._createViewer(view.type);
        this._emit('viewer.created', {
          type: type,
          viewer: viewer
        });
      }
      return viewer;
    }
    _createViewer(id) {
      var provider = find$2(this._getViewProviders(), function (provider) {
        return provider.id === id;
      });
      if (!provider) {
        throw new Error('no provider for view type <' + id + '>');
      }
      var Viewer = provider.constructor;
      var providerOptions = this._options[id] || {};
      var commonOptions = this._options.common || {};
      return new Viewer({
        ...commonOptions,
        ...providerOptions,
        additionalModules: [...(providerOptions.additionalModules || []), {
          _parent: ['value', this],
          moddle: ['value', this._moddle]
        }]
      });
    }

    /**
     * Emit an event.
     */
    _emit(...args) {
      return this._eventBus.fire(...args);
    }
    _createModdle(options) {
      return new simple(options.moddleExtensions);
    }

    /**
     * Return the list of available view providers.
     *
     * @abstract
     *
     * @return {Array<ViewProvider>}
     */
    _getViewProviders() {
      return [];
    }
  }

  // helpers //////////////////////

  /**
   * Ensure the passed argument is a proper unit (defaulting to px)
   */
  function ensureUnit(val) {
    return val + (isNumber$1(val) ? 'px' : '');
  }
  function checkDMNCompatibilityError(err, xml) {
    // check if we can indicate opening of old DMN 1.1 or DMN 1.2 diagrams

    if (err.message !== 'failed to parse document as <dmn:Definitions>') {
      return null;
    }
    var olderDMNVersion = xml.indexOf('"http://www.omg.org/spec/DMN/20151101/dmn.xsd"') !== -1 && '1.1' || xml.indexOf('"http://www.omg.org/spec/DMN/20180521/MODEL/"') !== -1 && '1.2';
    if (!olderDMNVersion) {
      return null;
    }
    err = new Error('unsupported DMN ' + olderDMNVersion + ' file detected; ' + 'only DMN 1.3 files can be opened');
    console.error('Cannot open what looks like a DMN ' + olderDMNVersion + ' diagram. ' + 'Please refer to https://bpmn.io/l/dmn-compatibility.html ' + 'to learn how to make the toolkit compatible with older DMN files', err);
    return err;
  }
  function checkValidationError(err) {
    // check if we can help the user by indicating wrong DMN 1.3 xml
    // (in case he or the exporting tool did not get that right)

    var pattern = /unparsable content <([^>]+)> detected([\s\S]*)$/,
      match = pattern.exec(err.message);
    if (!match) {
      return null;
    }
    err.message = 'unparsable content <' + match[1] + '> detected; ' + 'this may indicate an invalid DMN 1.3 diagram file' + match[2];
    return err;
  }
  function viewsEqual(a, b) {
    if (!isDefined(a)) {
      if (!isDefined(b)) {
        return true;
      } else {
        return false;
      }
    }
    if (!isDefined(b)) {
      return false;
    }

    // compare by element OR element ID equality
    return a.element === b.element || a.id === b.id;
  }
  function viewNameChanged(a, b) {
    return !a || !b || a.name !== b.name;
  }
  function safeExecute(viewer, method) {
    if (isFunction$2(viewer[method])) {
      viewer[method]();
    }
  }

  function e(e, t) {
    t && (e.super_ = t, e.prototype = Object.create(t.prototype, {
      constructor: {
        value: e,
        enumerable: !1,
        writable: !0,
        configurable: !0
      }
    }));
  }

  const CLASS_PATTERN = /^class[ {]/;

  /**
   * @param {function} fn
   *
   * @return {boolean}
   */
  function isClass(fn) {
    return CLASS_PATTERN.test(fn.toString());
  }

  /**
   * @param {any} obj
   *
   * @return {boolean}
   */
  function isArray$1(obj) {
    return Array.isArray(obj);
  }

  /**
   * @param {any} obj
   * @param {string} prop
   *
   * @return {boolean}
   */
  function hasOwnProp(obj, prop) {
    return Object.prototype.hasOwnProperty.call(obj, prop);
  }

  /**
   * @typedef {import('./index.js').InjectAnnotated } InjectAnnotated
   */

  /**
   * @template T
   *
   * @params {[...string[], T] | ...string[], T} args
   *
   * @return {T & InjectAnnotated}
   */
  function annotate(...args) {
    if (args.length === 1 && isArray$1(args[0])) {
      args = args[0];
    }
    args = [...args];
    const fn = args.pop();
    fn.$inject = args;
    return fn;
  }

  // Current limitations:
  // - can't put into "function arg" comments
  // function /* (no parenthesis like this) */ (){}
  // function abc( /* xx (no parenthesis like this) */ a, b) {}
  //
  // Just put the comment before function or inside:
  // /* (((this is fine))) */ function(a, b) {}
  // function abc(a) { /* (((this is fine))) */}
  //
  // - can't reliably auto-annotate constructor; we'll match the
  // first constructor(...) pattern found which may be the one
  // of a nested class, too.

  const CONSTRUCTOR_ARGS = /constructor\s*[^(]*\(\s*([^)]*)\)/m;
  const FN_ARGS = /^(?:async\s+)?(?:function\s*[^(]*)?(?:\(\s*([^)]*)\)|(\w+))/m;
  const FN_ARG = /\/\*([^*]*)\*\//m;

  /**
   * @param {unknown} fn
   *
   * @return {string[]}
   */
  function parseAnnotations(fn) {
    if (typeof fn !== 'function') {
      throw new Error(`Cannot annotate "${fn}". Expected a function!`);
    }
    const match = fn.toString().match(isClass(fn) ? CONSTRUCTOR_ARGS : FN_ARGS);

    // may parse class without constructor
    if (!match) {
      return [];
    }
    const args = match[1] || match[2];
    return args && args.split(',').map(arg => {
      const argMatch = arg.match(FN_ARG);
      return (argMatch && argMatch[1] || arg).trim();
    }) || [];
  }

  /**
   * @typedef { import('./index.js').ModuleDeclaration } ModuleDeclaration
   * @typedef { import('./index.js').ModuleDefinition } ModuleDefinition
   * @typedef { import('./index.js').InjectorContext } InjectorContext
   *
   * @typedef { import('./index.js').TypedDeclaration<any, any> } TypedDeclaration
   */

  /**
   * Create a new injector with the given modules.
   *
   * @param {ModuleDefinition[]} modules
   * @param {InjectorContext} [_parent]
   */
  function Injector(modules, _parent) {
    const parent = _parent || (/** @type InjectorContext */{
      get: function (name, strict) {
        currentlyResolving.push(name);
        if (strict === false) {
          return null;
        } else {
          throw error(`No provider for "${name}"!`);
        }
      }
    });
    const currentlyResolving = [];
    const providers = this._providers = Object.create(parent._providers || null);
    const instances = this._instances = Object.create(null);
    const self = instances.injector = this;
    const error = function (msg) {
      const stack = currentlyResolving.join(' -> ');
      currentlyResolving.length = 0;
      return new Error(stack ? `${msg} (Resolving: ${stack})` : msg);
    };

    /**
     * Return a named service.
     *
     * @param {string} name
     * @param {boolean} [strict=true] if false, resolve missing services to null
     *
     * @return {any}
     */
    function get(name, strict) {
      if (!providers[name] && name.includes('.')) {
        const parts = name.split('.');
        let pivot = get(/** @type { string } */parts.shift());
        while (parts.length) {
          pivot = pivot[(/** @type { string } */parts.shift())];
        }
        return pivot;
      }
      if (hasOwnProp(instances, name)) {
        return instances[name];
      }
      if (hasOwnProp(providers, name)) {
        if (currentlyResolving.indexOf(name) !== -1) {
          currentlyResolving.push(name);
          throw error('Cannot resolve circular dependency!');
        }
        currentlyResolving.push(name);
        instances[name] = providers[name][0](providers[name][1]);
        currentlyResolving.pop();
        return instances[name];
      }
      return parent.get(name, strict);
    }
    function fnDef(fn, locals) {
      if (typeof locals === 'undefined') {
        locals = {};
      }
      if (typeof fn !== 'function') {
        if (isArray$1(fn)) {
          fn = annotate(fn.slice());
        } else {
          throw error(`Cannot invoke "${fn}". Expected a function!`);
        }
      }

      /**
       * @type {string[]}
       */
      const inject = fn.$inject || parseAnnotations(fn);
      const dependencies = inject.map(dep => {
        if (hasOwnProp(locals, dep)) {
          return locals[dep];
        } else {
          return get(dep);
        }
      });
      return {
        fn: fn,
        dependencies
      };
    }

    /**
     * Instantiate the given type, injecting dependencies.
     *
     * @template T
     *
     * @param { Function | [...string[], Function ]} type
     *
     * @return T
     */
    function instantiate(type) {
      const {
        fn,
        dependencies
      } = fnDef(type);

      // instantiate var args constructor
      const Constructor = Function.prototype.bind.call(fn, null, ...dependencies);
      return new Constructor();
    }

    /**
     * Invoke the given function, injecting dependencies. Return the result.
     *
     * @template T
     *
     * @param { Function | [...string[], Function ]} func
     * @param { Object } [context]
     * @param { Object } [locals]
     *
     * @return {T} invocation result
     */
    function invoke(func, context, locals) {
      const {
        fn,
        dependencies
      } = fnDef(func, locals);
      return fn.apply(context, dependencies);
    }

    /**
     * @param {Injector} childInjector
     *
     * @return {Function}
     */
    function createPrivateInjectorFactory(childInjector) {
      return annotate(key => childInjector.get(key));
    }

    /**
     * @param {ModuleDefinition[]} modules
     * @param {string[]} [forceNewInstances]
     *
     * @return {Injector}
     */
    function createChild(modules, forceNewInstances) {
      if (forceNewInstances && forceNewInstances.length) {
        const fromParentModule = Object.create(null);
        const matchedScopes = Object.create(null);
        const privateInjectorsCache = [];
        const privateChildInjectors = [];
        const privateChildFactories = [];
        let provider;
        let cacheIdx;
        let privateChildInjector;
        let privateChildInjectorFactory;
        for (let name in providers) {
          provider = providers[name];
          if (forceNewInstances.indexOf(name) !== -1) {
            if (provider[2] === 'private') {
              cacheIdx = privateInjectorsCache.indexOf(provider[3]);
              if (cacheIdx === -1) {
                privateChildInjector = provider[3].createChild([], forceNewInstances);
                privateChildInjectorFactory = createPrivateInjectorFactory(privateChildInjector);
                privateInjectorsCache.push(provider[3]);
                privateChildInjectors.push(privateChildInjector);
                privateChildFactories.push(privateChildInjectorFactory);
                fromParentModule[name] = [privateChildInjectorFactory, name, 'private', privateChildInjector];
              } else {
                fromParentModule[name] = [privateChildFactories[cacheIdx], name, 'private', privateChildInjectors[cacheIdx]];
              }
            } else {
              fromParentModule[name] = [provider[2], provider[1]];
            }
            matchedScopes[name] = true;
          }
          if ((provider[2] === 'factory' || provider[2] === 'type') && provider[1].$scope) {
            /* jshint -W083 */
            forceNewInstances.forEach(scope => {
              if (provider[1].$scope.indexOf(scope) !== -1) {
                fromParentModule[name] = [provider[2], provider[1]];
                matchedScopes[scope] = true;
              }
            });
          }
        }
        forceNewInstances.forEach(scope => {
          if (!matchedScopes[scope]) {
            throw new Error('No provider for "' + scope + '". Cannot use provider from the parent!');
          }
        });
        modules.unshift(fromParentModule);
      }
      return new Injector(modules, self);
    }
    const factoryMap = {
      factory: invoke,
      type: instantiate,
      value: function (value) {
        return value;
      }
    };

    /**
     * @param {ModuleDefinition} moduleDefinition
     * @param {Injector} injector
     */
    function createInitializer(moduleDefinition, injector) {
      const initializers = moduleDefinition.__init__ || [];
      return function () {
        initializers.forEach(initializer => {
          // eagerly resolve component (fn or string)
          if (typeof initializer === 'string') {
            injector.get(initializer);
          } else {
            injector.invoke(initializer);
          }
        });
      };
    }

    /**
     * @param {ModuleDefinition} moduleDefinition
     */
    function loadModule(moduleDefinition) {
      const moduleExports = moduleDefinition.__exports__;

      // private module
      if (moduleExports) {
        const nestedModules = moduleDefinition.__modules__;
        const clonedModule = Object.keys(moduleDefinition).reduce((clonedModule, key) => {
          if (key !== '__exports__' && key !== '__modules__' && key !== '__init__' && key !== '__depends__') {
            clonedModule[key] = moduleDefinition[key];
          }
          return clonedModule;
        }, Object.create(null));
        const childModules = (nestedModules || []).concat(clonedModule);
        const privateInjector = createChild(childModules);
        const getFromPrivateInjector = annotate(function (key) {
          return privateInjector.get(key);
        });
        moduleExports.forEach(function (key) {
          providers[key] = [getFromPrivateInjector, key, 'private', privateInjector];
        });

        // ensure child injector initializes
        const initializers = (moduleDefinition.__init__ || []).slice();
        initializers.unshift(function () {
          privateInjector.init();
        });
        moduleDefinition = Object.assign({}, moduleDefinition, {
          __init__: initializers
        });
        return createInitializer(moduleDefinition, privateInjector);
      }

      // normal module
      Object.keys(moduleDefinition).forEach(function (key) {
        if (key === '__init__' || key === '__depends__') {
          return;
        }
        const typeDeclaration = /** @type { TypedDeclaration } */
        moduleDefinition[key];
        if (typeDeclaration[2] === 'private') {
          providers[key] = typeDeclaration;
          return;
        }
        const type = typeDeclaration[0];
        const value = typeDeclaration[1];
        providers[key] = [factoryMap[type], arrayUnwrap(type, value), type];
      });
      return createInitializer(moduleDefinition, self);
    }

    /**
     * @param {ModuleDefinition[]} moduleDefinitions
     * @param {ModuleDefinition} moduleDefinition
     *
     * @return {ModuleDefinition[]}
     */
    function resolveDependencies(moduleDefinitions, moduleDefinition) {
      if (moduleDefinitions.indexOf(moduleDefinition) !== -1) {
        return moduleDefinitions;
      }
      moduleDefinitions = (moduleDefinition.__depends__ || []).reduce(resolveDependencies, moduleDefinitions);
      if (moduleDefinitions.indexOf(moduleDefinition) !== -1) {
        return moduleDefinitions;
      }
      return moduleDefinitions.concat(moduleDefinition);
    }

    /**
     * @param {ModuleDefinition[]} moduleDefinitions
     *
     * @return { () => void } initializerFn
     */
    function bootstrap(moduleDefinitions) {
      const initializers = moduleDefinitions.reduce(resolveDependencies, []).map(loadModule);
      let initialized = false;
      return function () {
        if (initialized) {
          return;
        }
        initialized = true;
        initializers.forEach(initializer => initializer());
      };
    }

    // public API
    this.get = get;
    this.invoke = invoke;
    this.instantiate = instantiate;
    this.createChild = createChild;

    // setup
    this.init = bootstrap(modules);
  }

  // helpers ///////////////

  function arrayUnwrap(type, value) {
    if (type !== 'value' && isArray$1(value)) {
      value = annotate(value.slice());
    }
    return value;
  }

  var DEFAULT_RENDER_PRIORITY$1 = 1000;

  /**
   * @typedef {import('../core/Types').ElementLike} Element
   * @typedef {import('../core/Types').ConnectionLike} Connection
   * @typedef {import('../core/Types').ShapeLike} Shape
   *
   * @typedef {import('../core/EventBus').default} EventBus
   */

  /**
   * The base implementation of shape and connection renderers.
   *
   * @param {EventBus} eventBus
   * @param {number} [renderPriority=1000]
   */
  function BaseRenderer(eventBus, renderPriority) {
    var self = this;
    renderPriority = renderPriority || DEFAULT_RENDER_PRIORITY$1;
    eventBus.on(['render.shape', 'render.connection'], renderPriority, function (evt, context) {
      var type = evt.type,
        element = context.element,
        visuals = context.gfx,
        attrs = context.attrs;
      if (self.canRender(element)) {
        if (type === 'render.shape') {
          return self.drawShape(visuals, element, attrs);
        } else {
          return self.drawConnection(visuals, element, attrs);
        }
      }
    });
    eventBus.on(['render.getShapePath', 'render.getConnectionPath'], renderPriority, function (evt, element) {
      if (self.canRender(element)) {
        if (evt.type === 'render.getShapePath') {
          return self.getShapePath(element);
        } else {
          return self.getConnectionPath(element);
        }
      }
    });
  }

  /**
   * Checks whether an element can be rendered.
   *
   * @param {Element} element The element to be rendered.
   *
   * @return {boolean} Whether the element can be rendered.
   */
  BaseRenderer.prototype.canRender = function (element) {};

  /**
   * Draws a shape.
   *
   * @param {SVGElement} visuals The SVG element to draw the shape into.
   * @param {Shape} shape The shape to be drawn.
   *
   * @return {SVGElement} The SVG element of the shape drawn.
   */
  BaseRenderer.prototype.drawShape = function (visuals, shape) {};

  /**
   * Draws a connection.
   *
   * @param {SVGElement} visuals The SVG element to draw the connection into.
   * @param {Connection} connection The connection to be drawn.
   *
   * @return {SVGElement} The SVG element of the connection drawn.
   */
  BaseRenderer.prototype.drawConnection = function (visuals, connection) {};

  /**
   * Gets the SVG path of the graphical representation of a shape.
   *
   * @param {Shape} shape The shape.
   *
   * @return {string} The SVG path of the shape.
   */
  BaseRenderer.prototype.getShapePath = function (shape) {};

  /**
   * Gets the SVG path of the graphical representation of a connection.
   *
   * @param {Connection} connection The connection.
   *
   * @return {string} The SVG path of the connection.
   */
  BaseRenderer.prototype.getConnectionPath = function (connection) {};

  function ensureImported(element, target) {
    if (element.ownerDocument !== target.ownerDocument) {
      try {
        // may fail on webkit
        return target.ownerDocument.importNode(element, true);
      } catch (e) {

        // ignore
      }
    }
    return element;
  }

  /**
   * appendTo utility
   */

  /**
   * Append a node to a target element and return the appended node.
   *
   * @param  {SVGElement} element
   * @param  {SVGElement} target
   *
   * @return {SVGElement} the appended node
   */
  function appendTo(element, target) {
    return target.appendChild(ensureImported(element, target));
  }

  /**
   * append utility
   */

  /**
   * Append a node to an element
   *
   * @param  {SVGElement} element
   * @param  {SVGElement} node
   *
   * @return {SVGElement} the element
   */
  function append(target, node) {
    appendTo(node, target);
    return target;
  }

  /**
   * attribute accessor utility
   */

  var LENGTH_ATTR = 2;
  var CSS_PROPERTIES = {
    'alignment-baseline': 1,
    'baseline-shift': 1,
    'clip': 1,
    'clip-path': 1,
    'clip-rule': 1,
    'color': 1,
    'color-interpolation': 1,
    'color-interpolation-filters': 1,
    'color-profile': 1,
    'color-rendering': 1,
    'cursor': 1,
    'direction': 1,
    'display': 1,
    'dominant-baseline': 1,
    'enable-background': 1,
    'fill': 1,
    'fill-opacity': 1,
    'fill-rule': 1,
    'filter': 1,
    'flood-color': 1,
    'flood-opacity': 1,
    'font': 1,
    'font-family': 1,
    'font-size': LENGTH_ATTR,
    'font-size-adjust': 1,
    'font-stretch': 1,
    'font-style': 1,
    'font-variant': 1,
    'font-weight': 1,
    'glyph-orientation-horizontal': 1,
    'glyph-orientation-vertical': 1,
    'image-rendering': 1,
    'kerning': 1,
    'letter-spacing': 1,
    'lighting-color': 1,
    'marker': 1,
    'marker-end': 1,
    'marker-mid': 1,
    'marker-start': 1,
    'mask': 1,
    'opacity': 1,
    'overflow': 1,
    'pointer-events': 1,
    'shape-rendering': 1,
    'stop-color': 1,
    'stop-opacity': 1,
    'stroke': 1,
    'stroke-dasharray': 1,
    'stroke-dashoffset': 1,
    'stroke-linecap': 1,
    'stroke-linejoin': 1,
    'stroke-miterlimit': 1,
    'stroke-opacity': 1,
    'stroke-width': LENGTH_ATTR,
    'text-anchor': 1,
    'text-decoration': 1,
    'text-rendering': 1,
    'unicode-bidi': 1,
    'visibility': 1,
    'word-spacing': 1,
    'writing-mode': 1
  };
  function getAttribute(node, name) {
    if (CSS_PROPERTIES[name]) {
      return node.style[name];
    } else {
      return node.getAttributeNS(null, name);
    }
  }
  function setAttribute(node, name, value) {
    var hyphenated = name.replace(/([a-z])([A-Z])/g, '$1-$2').toLowerCase();
    var type = CSS_PROPERTIES[hyphenated];
    if (type) {
      // append pixel unit, unless present
      if (type === LENGTH_ATTR && typeof value === 'number') {
        value = String(value) + 'px';
      }
      node.style[hyphenated] = value;
    } else {
      node.setAttributeNS(null, name, value);
    }
  }
  function setAttributes(node, attrs) {
    var names = Object.keys(attrs),
      i,
      name;
    for (i = 0, name; name = names[i]; i++) {
      setAttribute(node, name, attrs[name]);
    }
  }

  /**
   * Gets or sets raw attributes on a node.
   *
   * @param  {SVGElement} node
   * @param  {Object} [attrs]
   * @param  {String} [name]
   * @param  {String} [value]
   *
   * @return {String}
   */
  function attr(node, name, value) {
    if (typeof name === 'string') {
      if (value !== undefined) {
        setAttribute(node, name, value);
      } else {
        return getAttribute(node, name);
      }
    } else {
      setAttributes(node, name);
    }
    return node;
  }

  /**
   * Taken from https://github.com/component/classes
   *
   * Without the component bits.
   */

  /**
   * toString reference.
   */

  const toString = Object.prototype.toString;

  /**
    * Wrap `el` in a `ClassList`.
    *
    * @param {Element} el
    * @return {ClassList}
    * @api public
    */

  function classes(el) {
    return new ClassList(el);
  }
  function ClassList(el) {
    if (!el || !el.nodeType) {
      throw new Error('A DOM element reference is required');
    }
    this.el = el;
    this.list = el.classList;
  }

  /**
    * Add class `name` if not already present.
    *
    * @param {String} name
    * @return {ClassList}
    * @api public
    */

  ClassList.prototype.add = function (name) {
    this.list.add(name);
    return this;
  };

  /**
    * Remove class `name` when present, or
    * pass a regular expression to remove
    * any which match.
    *
    * @param {String|RegExp} name
    * @return {ClassList}
    * @api public
    */

  ClassList.prototype.remove = function (name) {
    if ('[object RegExp]' == toString.call(name)) {
      return this.removeMatching(name);
    }
    this.list.remove(name);
    return this;
  };

  /**
    * Remove all classes matching `re`.
    *
    * @param {RegExp} re
    * @return {ClassList}
    * @api private
    */

  ClassList.prototype.removeMatching = function (re) {
    const arr = this.array();
    for (let i = 0; i < arr.length; i++) {
      if (re.test(arr[i])) {
        this.remove(arr[i]);
      }
    }
    return this;
  };

  /**
    * Toggle class `name`, can force state via `force`.
    *
    * For browsers that support classList, but do not support `force` yet,
    * the mistake will be detected and corrected.
    *
    * @param {String} name
    * @param {Boolean} force
    * @return {ClassList}
    * @api public
    */

  ClassList.prototype.toggle = function (name, force) {
    if ('undefined' !== typeof force) {
      if (force !== this.list.toggle(name, force)) {
        this.list.toggle(name); // toggle again to correct
      }
    } else {
      this.list.toggle(name);
    }
    return this;
  };

  /**
    * Return an array of classes.
    *
    * @return {Array}
    * @api public
    */

  ClassList.prototype.array = function () {
    return Array.from(this.list);
  };

  /**
    * Check if class `name` is present.
    *
    * @param {String} name
    * @return {ClassList}
    * @api public
    */

  ClassList.prototype.has = ClassList.prototype.contains = function (name) {
    return this.list.contains(name);
  };
  var ns = {
    svg: 'http://www.w3.org/2000/svg'
  };

  /**
   * DOM parsing utility
   */

  var SVG_START = '<svg xmlns="' + ns.svg + '"';
  function parse(svg) {
    var unwrap = false;

    // ensure we import a valid svg document
    if (svg.substring(0, 4) === '<svg') {
      if (svg.indexOf(ns.svg) === -1) {
        svg = SVG_START + svg.substring(4);
      }
    } else {
      // namespace svg
      svg = SVG_START + '>' + svg + '</svg>';
      unwrap = true;
    }
    var parsed = parseDocument(svg);
    if (!unwrap) {
      return parsed;
    }
    var fragment = document.createDocumentFragment();
    var parent = parsed.firstChild;
    while (parent.firstChild) {
      fragment.appendChild(parent.firstChild);
    }
    return fragment;
  }
  function parseDocument(svg) {
    var parser;

    // parse
    parser = new DOMParser();
    parser.async = false;
    return parser.parseFromString(svg, 'text/xml');
  }

  /**
   * Create utility for SVG elements
   */

  /**
   * Create a specific type from name or SVG markup.
   *
   * @param {String} name the name or markup of the element
   * @param {Object} [attrs] attributes to set on the element
   *
   * @returns {SVGElement}
   */
  function create$2(name, attrs) {
    var element;
    name = name.trim();
    if (name.charAt(0) === '<') {
      element = parse(name).firstChild;
      element = document.importNode(element, true);
    } else {
      element = document.createElementNS(ns.svg, name);
    }
    if (attrs) {
      attr(element, attrs);
    }
    return element;
  }

  /**
   * Geometry helpers
   */

  // fake node used to instantiate svg geometry elements
  var node = null;
  function getNode() {
    if (node === null) {
      node = create$2('svg');
    }
    return node;
  }
  function extend$1(object, props) {
    var i,
      k,
      keys = Object.keys(props);
    for (i = 0; k = keys[i]; i++) {
      object[k] = props[k];
    }
    return object;
  }

  /**
   * Create matrix via args.
   *
   * @example
   *
   * createMatrix({ a: 1, b: 1 });
   * createMatrix();
   * createMatrix(1, 2, 0, 0, 30, 20);
   *
   * @return {SVGMatrix}
   */
  function createMatrix(a, b, c, d, e, f) {
    var matrix = getNode().createSVGMatrix();
    switch (arguments.length) {
      case 0:
        return matrix;
      case 1:
        return extend$1(matrix, a);
      case 6:
        return extend$1(matrix, {
          a: a,
          b: b,
          c: c,
          d: d,
          e: e,
          f: f
        });
    }
  }
  function createTransform(matrix) {
    {
      return getNode().createSVGTransform();
    }
  }

  /**
   * Serialization util
   */

  var TEXT_ENTITIES = /([&<>]{1})/g;
  var ATTR_ENTITIES = /([&<>\n\r"]{1})/g;
  var ENTITY_REPLACEMENT = {
    '&': '&amp;',
    '<': '&lt;',
    '>': '&gt;',
    '"': '\''
  };
  function escape(str, pattern) {
    function replaceFn(match, entity) {
      return ENTITY_REPLACEMENT[entity] || entity;
    }
    return str.replace(pattern, replaceFn);
  }
  function serialize(node, output) {
    var i, len, attrMap, attrNode, childNodes;
    switch (node.nodeType) {
      // TEXT
      case 3:
        // replace special XML characters
        output.push(escape(node.textContent, TEXT_ENTITIES));
        break;

      // ELEMENT
      case 1:
        output.push('<', node.tagName);
        if (node.hasAttributes()) {
          attrMap = node.attributes;
          for (i = 0, len = attrMap.length; i < len; ++i) {
            attrNode = attrMap.item(i);
            output.push(' ', attrNode.name, '="', escape(attrNode.value, ATTR_ENTITIES), '"');
          }
        }
        if (node.hasChildNodes()) {
          output.push('>');
          childNodes = node.childNodes;
          for (i = 0, len = childNodes.length; i < len; ++i) {
            serialize(childNodes.item(i), output);
          }
          output.push('</', node.tagName, '>');
        } else {
          output.push('/>');
        }
        break;

      // COMMENT
      case 8:
        output.push('<!--', escape(node.nodeValue, TEXT_ENTITIES), '-->');
        break;

      // CDATA
      case 4:
        output.push('<![CDATA[', node.nodeValue, ']]>');
        break;
      default:
        throw new Error('unable to handle node ' + node.nodeType);
    }
    return output;
  }
  function get(element) {
    var child = element.firstChild,
      output = [];
    while (child) {
      serialize(child, output);
      child = child.nextSibling;
    }
    return output.join('');
  }
  function innerSVG(element, svg) {
    {
      return get(element);
    }
  }
  function remove$3(element) {
    var parent = element.parentNode;
    if (parent) {
      parent.removeChild(element);
    }
    return element;
  }

  /**
   * transform accessor utility
   */

  function wrapMatrix(transformList, transform) {
    if (transform instanceof SVGMatrix) {
      return transformList.createSVGTransformFromMatrix(transform);
    }
    return transform;
  }
  function setTransforms(transformList, transforms) {
    var i, t;
    transformList.clear();
    for (i = 0; t = transforms[i]; i++) {
      transformList.appendItem(wrapMatrix(transformList, t));
    }
  }

  /**
   * Get or set the transforms on the given node.
   *
   * @param {SVGElement} node
   * @param  {SVGTransform|SVGMatrix|Array<SVGTransform|SVGMatrix>} [transforms]
   *
   * @return {SVGTransform} the consolidated transform
   */
  function transform(node, transforms) {
    var transformList = node.transform.baseVal;
    if (transforms) {
      if (!Array.isArray(transforms)) {
        transforms = [transforms];
      }
      setTransforms(transformList, transforms);
    }
    return transformList.consolidate();
  }

  /**
   * @typedef {(string|number)[]} Component
   *
   * @typedef {import('../util/Types').Point} Point
   */

  /**
   * @param {Component[] | Component[][]} elements
   *
   * @return {string}
   */
  function componentsToPath(elements) {
    return elements.flat().join(',').replace(/,?([A-z]),?/g, '$1');
  }

  /**
   * @param {Point} point
   *
   * @return {Component[]}
   */
  function move(point) {
    return ['M', point.x, point.y];
  }

  /**
   * @param {Point} point
   *
   * @return {Component[]}
   */
  function lineTo(point) {
    return ['L', point.x, point.y];
  }

  /**
   * @param {Point} p1
   * @param {Point} p2
   * @param {Point} p3
   *
   * @return {Component[]}
   */
  function curveTo(p1, p2, p3) {
    return ['C', p1.x, p1.y, p2.x, p2.y, p3.x, p3.y];
  }

  /**
   * @param {Point[]} waypoints
   * @param {number} [cornerRadius]
   * @return {Component[][]}
   */
  function drawPath(waypoints, cornerRadius) {
    const pointCount = waypoints.length;
    const path = [move(waypoints[0])];
    for (let i = 1; i < pointCount; i++) {
      const pointBefore = waypoints[i - 1];
      const point = waypoints[i];
      const pointAfter = waypoints[i + 1];
      if (!pointAfter || !cornerRadius) {
        path.push(lineTo(point));
        continue;
      }
      const effectiveRadius = Math.min(cornerRadius, vectorLength(point.x - pointBefore.x, point.y - pointBefore.y), vectorLength(pointAfter.x - point.x, pointAfter.y - point.y));
      if (!effectiveRadius) {
        path.push(lineTo(point));
        continue;
      }
      const beforePoint = getPointAtLength(point, pointBefore, effectiveRadius);
      const beforePoint2 = getPointAtLength(point, pointBefore, effectiveRadius * .5);
      const afterPoint = getPointAtLength(point, pointAfter, effectiveRadius);
      const afterPoint2 = getPointAtLength(point, pointAfter, effectiveRadius * .5);
      path.push(lineTo(beforePoint));
      path.push(curveTo(beforePoint2, afterPoint2, afterPoint));
    }
    return path;
  }
  function getPointAtLength(start, end, length) {
    const deltaX = end.x - start.x;
    const deltaY = end.y - start.y;
    const totalLength = vectorLength(deltaX, deltaY);
    const percent = length / totalLength;
    return {
      x: start.x + deltaX * percent,
      y: start.y + deltaY * percent
    };
  }
  function vectorLength(x, y) {
    return Math.sqrt(Math.pow(x, 2) + Math.pow(y, 2));
  }

  /**
   * @param {Point[]} points
   * @param {number|Object} [attrs]
   * @param {number} [radius]
   *
   * @return {SVGElement}
   */
  function createLine(points, attrs, radius) {
    if (isNumber$1(attrs)) {
      radius = attrs;
      attrs = null;
    }
    if (!attrs) {
      attrs = {};
    }
    const line = create$2('path', attrs);
    if (isNumber$1(radius)) {
      line.dataset.cornerRadius = String(radius);
    }
    return updateLine(line, points);
  }

  /**
   * @param {SVGElement} gfx
   * @param {Point[]} points
   *
   * @return {SVGElement}
   */
  function updateLine(gfx, points) {
    const cornerRadius = parseInt(gfx.dataset.cornerRadius, 10) || 0;
    attr(gfx, {
      d: componentsToPath(drawPath(points, cornerRadius))
    });
    return gfx;
  }

  /**
   * Returns the surrounding bbox for all elements in
   * the array or the element primitive.
   *
   * @param {Element|Element[]} elements
   * @param {boolean} [stopRecursion=false]
   *
   * @return {Rect}
   */
  function getBBox(elements, stopRecursion) {
    stopRecursion = !!stopRecursion;
    if (!isArray$5(elements)) {
      elements = [elements];
    }
    var minX, minY, maxX, maxY;
    forEach$3(elements, function (element) {
      // If element is a connection the bbox must be computed first
      var bbox = element;
      if (element.waypoints && !stopRecursion) {
        bbox = getBBox(element.waypoints, true);
      }
      var x = bbox.x,
        y = bbox.y,
        height = bbox.height || 0,
        width = bbox.width || 0;
      if (x < minX || minX === undefined) {
        minX = x;
      }
      if (y < minY || minY === undefined) {
        minY = y;
      }
      if (x + width > maxX || maxX === undefined) {
        maxX = x + width;
      }
      if (y + height > maxY || maxY === undefined) {
        maxY = y + height;
      }
    });
    return {
      x: minX,
      y: minY,
      height: maxY - minY,
      width: maxX - minX
    };
  }

  /**
   * Get the element's type
   *
   * @param {Element} element
   *
   * @return {'connection' | 'shape' | 'root'}
   */
  function getType(element) {
    if ('waypoints' in element) {
      return 'connection';
    }
    if ('x' in element) {
      return 'shape';
    }
    return 'root';
  }

  /**
   * @param {Element} element
   *
   * @return {boolean}
   */
  function isFrameElement(element) {
    return !!(element && element.isFrame);
  }

  /**
   * @typedef {import('../core/EventBus').default} EventBus
   * @typedef {import('./Styles').default} Styles
   */

  // apply default renderer with lowest possible priority
  // so that it only kicks in if noone else could render
  var DEFAULT_RENDER_PRIORITY = 1;

  /**
   * The default renderer used for shapes and connections.
   *
   * @param {EventBus} eventBus
   * @param {Styles} styles
   */
  function DefaultRenderer(eventBus, styles) {
    BaseRenderer.call(this, eventBus, DEFAULT_RENDER_PRIORITY);
    this.CONNECTION_STYLE = styles.style(['no-fill'], {
      strokeWidth: 5,
      stroke: 'fuchsia'
    });
    this.SHAPE_STYLE = styles.style({
      fill: 'white',
      stroke: 'fuchsia',
      strokeWidth: 2
    });
    this.FRAME_STYLE = styles.style(['no-fill'], {
      stroke: 'fuchsia',
      strokeDasharray: 4,
      strokeWidth: 2
    });
  }
  e(DefaultRenderer, BaseRenderer);

  /**
   * @private
   */
  DefaultRenderer.prototype.canRender = function () {
    return true;
  };

  /**
   * @private
   */
  DefaultRenderer.prototype.drawShape = function drawShape(visuals, element, attrs) {
    var rect = create$2('rect');
    attr(rect, {
      x: 0,
      y: 0,
      width: element.width || 0,
      height: element.height || 0
    });
    if (isFrameElement(element)) {
      attr(rect, assign$4({}, this.FRAME_STYLE, attrs || {}));
    } else {
      attr(rect, assign$4({}, this.SHAPE_STYLE, attrs || {}));
    }
    append(visuals, rect);
    return rect;
  };

  /**
   * @private
   */
  DefaultRenderer.prototype.drawConnection = function drawConnection(visuals, connection, attrs) {
    var line = createLine(connection.waypoints, assign$4({}, this.CONNECTION_STYLE, attrs || {}));
    append(visuals, line);
    return line;
  };

  /**
   * @private
   */
  DefaultRenderer.prototype.getShapePath = function getShapePath(shape) {
    var x = shape.x,
      y = shape.y,
      width = shape.width,
      height = shape.height;
    var shapePath = [['M', x, y], ['l', width, 0], ['l', 0, height], ['l', -width, 0], ['z']];
    return componentsToPath(shapePath);
  };

  /**
   * @private
   */
  DefaultRenderer.prototype.getConnectionPath = function getConnectionPath(connection) {
    var waypoints = connection.waypoints;
    var idx,
      point,
      connectionPath = [];
    for (idx = 0; point = waypoints[idx]; idx++) {
      // take invisible docking into account
      // when creating the path
      point = point.original || point;
      connectionPath.push([idx === 0 ? 'M' : 'L', point.x, point.y]);
    }
    return componentsToPath(connectionPath);
  };
  DefaultRenderer.$inject = ['eventBus', 'styles'];

  /**
   * A component that manages shape styles
   */
  function Styles() {
    var defaultTraits = {
      'no-fill': {
        fill: 'none'
      },
      'no-border': {
        strokeOpacity: 0.0
      },
      'no-events': {
        pointerEvents: 'none'
      }
    };
    var self = this;

    /**
     * Builds a style definition from a className, a list of traits and an object
     * of additional attributes.
     *
     * @param {string} className
     * @param {string[]} [traits]
     * @param {Object} [additionalAttrs]
     *
     * @return {Object} the style definition
     */
    this.cls = function (className, traits, additionalAttrs) {
      var attrs = this.style(traits, additionalAttrs);
      return assign$4(attrs, {
        'class': className
      });
    };

    /**
     * Builds a style definition from a list of traits and an object of additional
     * attributes.
     *
     * @param {string[]} [traits]
     * @param {Object} additionalAttrs
     *
     * @return {Object} the style definition
     */
    this.style = function (traits, additionalAttrs) {
      if (!isArray$5(traits) && !additionalAttrs) {
        additionalAttrs = traits;
        traits = [];
      }
      var attrs = reduce(traits, function (attrs, t) {
        return assign$4(attrs, defaultTraits[t] || {});
      }, {});
      return additionalAttrs ? assign$4(attrs, additionalAttrs) : attrs;
    };

    /**
     * Computes a style definition from a list of traits and an object of
     * additional attributes, with custom style definition object.
     *
     * @param {Object} custom
     * @param {string[]} [traits]
     * @param {Object} defaultStyles
     *
     * @return {Object} the style definition
     */
    this.computeStyle = function (custom, traits, defaultStyles) {
      if (!isArray$5(traits)) {
        defaultStyles = traits;
        traits = [];
      }
      return self.style(traits || [], assign$4({}, defaultStyles, custom || {}));
    };
  }

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var DrawModule$1 = {
    __init__: ['defaultRenderer'],
    defaultRenderer: ['type', DefaultRenderer],
    styles: ['type', Styles]
  };

  /**
   * Failsafe remove an element from a collection
   *
   * @param {Array<Object>} [collection]
   * @param {Object} [element]
   *
   * @return {number} the previous index of the element
   */
  function remove$2(collection, element) {
    if (!collection || !element) {
      return -1;
    }
    var idx = collection.indexOf(element);
    if (idx !== -1) {
      collection.splice(idx, 1);
    }
    return idx;
  }

  /**
   * Fail save add an element to the given connection, ensuring
   * it does not yet exist.
   *
   * @param {Array<Object>} collection
   * @param {Object} element
   * @param {number} [idx]
   */
  function add$2(collection, element, idx) {
    if (!collection || !element) {
      return;
    }
    if (typeof idx !== 'number') {
      idx = -1;
    }
    var currentIdx = collection.indexOf(element);
    if (currentIdx !== -1) {
      if (currentIdx === idx) {
        // nothing to do, position has not changed
        return;
      } else {
        if (idx !== -1) {
          // remove from current position
          collection.splice(currentIdx, 1);
        } else {
          // already exists in collection
          return;
        }
      }
    }
    if (idx !== -1) {
      // insert at specified position
      collection.splice(idx, 0, element);
    } else {
      // push to end
      collection.push(element);
    }
  }

  /**
   * Convert the given bounds to a { top, left, bottom, right } descriptor.
   *
   * @param {Point|Rect} bounds
   *
   * @return {RectTRBL}
   */
  function asTRBL(bounds) {
    return {
      top: bounds.y,
      right: bounds.x + (bounds.width || 0),
      bottom: bounds.y + (bounds.height || 0),
      left: bounds.x
    };
  }

  /**
   * @typedef {import('./Types').ConnectionLike} ConnectionLike
   * @typedef {import('./Types').RootLike} RootLike
   * @typedef {import('./Types').ParentLike } ParentLike
   * @typedef {import('./Types').ShapeLike} ShapeLike
   *
   * @typedef { {
   *   container?: HTMLElement;
   *   deferUpdate?: boolean;
   *   width?: number;
   *   height?: number;
   * } } CanvasConfig
   * @typedef { {
   *   group: SVGElement;
   *   index: number;
   *   visible: boolean;
   * } } CanvasLayer
   * @typedef { {
   *   [key: string]: CanvasLayer;
   * } } CanvasLayers
   * @typedef { {
   *   rootElement: ShapeLike;
   *   layer: CanvasLayer;
   * } } CanvasPlane
   * @typedef { {
   *   scale: number;
   *   inner: Rect;
   *   outer: Dimensions;
   * } & Rect } CanvasViewbox
   *
   * @typedef {import('./ElementRegistry').default} ElementRegistry
   * @typedef {import('./EventBus').default} EventBus
   * @typedef {import('./GraphicsFactory').default} GraphicsFactory
   *
   * @typedef {import('../util/Types').Dimensions} Dimensions
   * @typedef {import('../util/Types').Point} Point
   * @typedef {import('../util/Types').Rect} Rect
   * @typedef {import('../util/Types').RectTRBL} RectTRBL
   * @typedef {import('../util/Types').ScrollDelta} ScrollDelta
   */

  function round(number, resolution) {
    return Math.round(number * resolution) / resolution;
  }
  function ensurePx(number) {
    return isNumber$1(number) ? number + 'px' : number;
  }
  function findRoot(element) {
    while (element.parent) {
      element = element.parent;
    }
    return element;
  }

  /**
   * Creates a HTML container element for a SVG element with
   * the given configuration
   *
   * @param {CanvasConfig} options
   *
   * @return {HTMLElement} the container element
   */
  function createContainer(options) {
    options = assign$4({}, {
      width: '100%',
      height: '100%'
    }, options);
    const container = options.container || document.body;

    // create a <div> around the svg element with the respective size
    // this way we can always get the correct container size
    // (this is impossible for <svg> elements at the moment)
    const parent = document.createElement('div');
    parent.setAttribute('class', 'djs-container djs-parent');
    assign(parent, {
      position: 'relative',
      overflow: 'hidden',
      width: ensurePx(options.width),
      height: ensurePx(options.height)
    });
    container.appendChild(parent);
    return parent;
  }
  function createGroup(parent, cls, childIndex) {
    const group = create$2('g');
    classes(group).add(cls);
    const index = childIndex !== undefined ? childIndex : parent.childNodes.length - 1;

    // must ensure second argument is node or _null_
    // cf. https://developer.mozilla.org/en-US/docs/Web/API/Node/insertBefore
    parent.insertBefore(group, parent.childNodes[index] || null);
    return group;
  }
  const BASE_LAYER = 'base';

  // render plane contents behind utility layers
  const PLANE_LAYER_INDEX = 0;
  const UTILITY_LAYER_INDEX = 1;
  const REQUIRED_MODEL_ATTRS = {
    shape: ['x', 'y', 'width', 'height'],
    connection: ['waypoints']
  };

  /**
   * The main drawing canvas.
   *
   * @class
   * @constructor
   *
   * @emits Canvas#canvas.init
   *
   * @param {CanvasConfig|null} config
   * @param {EventBus} eventBus
   * @param {GraphicsFactory} graphicsFactory
   * @param {ElementRegistry} elementRegistry
   */
  function Canvas(config, eventBus, graphicsFactory, elementRegistry) {
    this._eventBus = eventBus;
    this._elementRegistry = elementRegistry;
    this._graphicsFactory = graphicsFactory;

    /**
     * @type {number}
     */
    this._rootsIdx = 0;

    /**
     * @type {CanvasLayers}
     */
    this._layers = {};

    /**
     * @type {CanvasPlane[]}
     */
    this._planes = [];

    /**
     * @type {RootLike|null}
     */
    this._rootElement = null;

    /**
     * @type {boolean}
     */
    this._focused = false;
    this._init(config || {});
  }
  Canvas.$inject = ['config.canvas', 'eventBus', 'graphicsFactory', 'elementRegistry'];

  /**
   * Creates a <svg> element that is wrapped into a <div>.
   * This way we are always able to correctly figure out the size of the svg element
   * by querying the parent node.

   * (It is not possible to get the size of a svg element cross browser @ 2014-04-01)

   * <div class="djs-container" style="width: {desired-width}, height: {desired-height}">
   *   <svg width="100%" height="100%">
   *    ...
   *   </svg>
   * </div>
   *
   * @param {CanvasConfig} config
   */
  Canvas.prototype._init = function (config) {
    const eventBus = this._eventBus;

    // html container
    const container = this._container = createContainer(config);
    const svg = this._svg = create$2('svg');
    attr(svg, {
      width: '100%',
      height: '100%'
    });
    attr$1(svg, 'tabindex', 0);
    eventBus.on('element.hover', () => {
      this.restoreFocus();
    });
    svg.addEventListener('focusin', () => {
      this._focused = true;
      eventBus.fire('canvas.focus.changed', {
        focused: true
      });
    });
    svg.addEventListener('focusout', () => {
      this._focused = false;
      eventBus.fire('canvas.focus.changed', {
        focused: false
      });
    });
    append(container, svg);
    const viewport = this._viewport = createGroup(svg, 'viewport');

    // debounce canvas.viewbox.changed events when deferUpdate is set
    // to help with potential performance issues
    if (config.deferUpdate) {
      this._viewboxChanged = debounce(bind$3(this._viewboxChanged, this), 300);
    }
    eventBus.on('diagram.init', () => {
      /**
       * An event indicating that the canvas is ready to be drawn on.
       *
       * @memberOf Canvas
       *
       * @event canvas.init
       *
       * @type {Object}
       * @property {SVGElement} svg the created svg element
       * @property {SVGElement} viewport the direct parent of diagram elements and shapes
       */
      eventBus.fire('canvas.init', {
        svg: svg,
        viewport: viewport
      });
    });

    // reset viewbox on shape changes to
    // recompute the viewbox
    eventBus.on(['shape.added', 'connection.added', 'shape.removed', 'connection.removed', 'elements.changed', 'root.set'], () => {
      delete this._cachedViewbox;
    });
    eventBus.on('diagram.destroy', 500, this._destroy, this);
    eventBus.on('diagram.clear', 500, this._clear, this);
  };
  Canvas.prototype._destroy = function () {
    this._eventBus.fire('canvas.destroy', {
      svg: this._svg,
      viewport: this._viewport
    });
    const parent = this._container.parentNode;
    if (parent) {
      parent.removeChild(this._container);
    }
    delete this._svg;
    delete this._container;
    delete this._layers;
    delete this._planes;
    delete this._rootElement;
    delete this._viewport;
  };
  Canvas.prototype._clear = function () {
    const allElements = this._elementRegistry.getAll();

    // remove all elements
    allElements.forEach(element => {
      const type = getType(element);
      if (type === 'root') {
        this.removeRootElement(element);
      } else {
        this._removeElement(element, type);
      }
    });

    // remove all planes
    this._planes = [];
    this._rootElement = null;

    // force recomputation of view box
    delete this._cachedViewbox;
  };

  /**
  * Sets focus on the canvas SVG element.
  */
  Canvas.prototype.focus = function () {
    this._svg.focus({
      preventScroll: true
    });
  };

  /**
  * Sets focus on the canvas SVG element if `document.body` is currently focused.
  */
  Canvas.prototype.restoreFocus = function () {
    if (document.activeElement === document.body) {
      this.focus();
    }
  };

  /**
  * Returns true if the canvas is focused.
  *
  * @return {boolean}
  */
  Canvas.prototype.isFocused = function () {
    return this._focused;
  };

  /**
   * Returns the default layer on which
   * all elements are drawn.
   *
   * @return {SVGElement}  The SVG element of the layer.
   */
  Canvas.prototype.getDefaultLayer = function () {
    return this.getLayer(BASE_LAYER, PLANE_LAYER_INDEX);
  };

  /**
   * Returns a layer that is used to draw elements
   * or annotations on it.
   *
   * Non-existing layers retrieved through this method
   * will be created. During creation, the optional index
   * may be used to create layers below or above existing layers.
   * A layer with a certain index is always created above all
   * existing layers with the same index.
   *
   * @param {string} name The name of the layer.
   * @param {number} [index] The index of the layer.
   *
   * @return {SVGElement} The SVG element of the layer.
   */
  Canvas.prototype.getLayer = function (name, index) {
    if (!name) {
      throw new Error('must specify a name');
    }
    let layer = this._layers[name];
    if (!layer) {
      layer = this._layers[name] = this._createLayer(name, index);
    }

    // throw an error if layer creation / retrival is
    // requested on different index
    if (typeof index !== 'undefined' && layer.index !== index) {
      throw new Error('layer <' + name + '> already created at index <' + index + '>');
    }
    return layer.group;
  };

  /**
   * For a given index, return the number of layers that have a higher index and
   * are visible.
   *
   * This is used to determine the node a layer should be inserted at.
   *
   * @param {number} index
   *
   * @return {number}
   */
  Canvas.prototype._getChildIndex = function (index) {
    return reduce(this._layers, function (childIndex, layer) {
      if (layer.visible && index >= layer.index) {
        childIndex++;
      }
      return childIndex;
    }, 0);
  };

  /**
   * Creates a given layer and returns it.
   *
   * @param {string} name
   * @param {number} [index=0]
   *
   * @return {CanvasLayer}
   */
  Canvas.prototype._createLayer = function (name, index) {
    if (typeof index === 'undefined') {
      index = UTILITY_LAYER_INDEX;
    }
    const childIndex = this._getChildIndex(index);
    return {
      group: createGroup(this._viewport, 'layer-' + name, childIndex),
      index: index,
      visible: true
    };
  };

  /**
   * Shows a given layer.
   *
   * @param {string} name The name of the layer.
   *
   * @return {SVGElement} The SVG element of the layer.
   */
  Canvas.prototype.showLayer = function (name) {
    if (!name) {
      throw new Error('must specify a name');
    }
    const layer = this._layers[name];
    if (!layer) {
      throw new Error('layer <' + name + '> does not exist');
    }
    const viewport = this._viewport;
    const group = layer.group;
    const index = layer.index;
    if (layer.visible) {
      return group;
    }
    const childIndex = this._getChildIndex(index);
    viewport.insertBefore(group, viewport.childNodes[childIndex] || null);
    layer.visible = true;
    return group;
  };

  /**
   * Hides a given layer.
   *
   * @param {string} name The name of the layer.
   *
   * @return {SVGElement} The SVG element of the layer.
   */
  Canvas.prototype.hideLayer = function (name) {
    if (!name) {
      throw new Error('must specify a name');
    }
    const layer = this._layers[name];
    if (!layer) {
      throw new Error('layer <' + name + '> does not exist');
    }
    const group = layer.group;
    if (!layer.visible) {
      return group;
    }
    remove$3(group);
    layer.visible = false;
    return group;
  };
  Canvas.prototype._removeLayer = function (name) {
    const layer = this._layers[name];
    if (layer) {
      delete this._layers[name];
      remove$3(layer.group);
    }
  };

  /**
   * Returns the currently active layer. Can be null.
   *
   * @return {CanvasLayer|null} The active layer of `null`.
   */
  Canvas.prototype.getActiveLayer = function () {
    const plane = this._findPlaneForRoot(this.getRootElement());
    if (!plane) {
      return null;
    }
    return plane.layer;
  };

  /**
   * Returns the plane which contains the given element.
   *
   * @param {ShapeLike|ConnectionLike|string} element The element or its ID.
   *
   * @return {RootLike|undefined} The root of the element.
   */
  Canvas.prototype.findRoot = function (element) {
    if (typeof element === 'string') {
      element = this._elementRegistry.get(element);
    }
    if (!element) {
      return;
    }
    const plane = this._findPlaneForRoot(findRoot(element)) || {};
    return plane.rootElement;
  };

  /**
   * Return a list of all root elements on the diagram.
   *
   * @return {(RootLike)[]} The list of root elements.
   */
  Canvas.prototype.getRootElements = function () {
    return this._planes.map(function (plane) {
      return plane.rootElement;
    });
  };
  Canvas.prototype._findPlaneForRoot = function (rootElement) {
    return find$2(this._planes, function (plane) {
      return plane.rootElement === rootElement;
    });
  };

  /**
   * Returns the html element that encloses the
   * drawing canvas.
   *
   * @return {HTMLElement} The HTML element of the container.
   */
  Canvas.prototype.getContainer = function () {
    return this._container;
  };

  // markers //////////////////////

  Canvas.prototype._updateMarker = function (element, marker, add) {
    let container;
    if (!element.id) {
      element = this._elementRegistry.get(element);
    }
    element.markers = element.markers || new Set();

    // we need to access all
    container = this._elementRegistry._elements[element.id];
    if (!container) {
      return;
    }
    forEach$3([container.gfx, container.secondaryGfx], function (gfx) {
      if (gfx) {
        // invoke either addClass or removeClass based on mode
        if (add) {
          element.markers.add(marker);
          classes(gfx).add(marker);
        } else {
          element.markers.delete(marker);
          classes(gfx).remove(marker);
        }
      }
    });

    /**
     * An event indicating that a marker has been updated for an element
     *
     * @event element.marker.update
     * @type {Object}
     * @property {Element} element the shape
     * @property {SVGElement} gfx the graphical representation of the shape
     * @property {string} marker
     * @property {boolean} add true if the marker was added, false if it got removed
     */
    this._eventBus.fire('element.marker.update', {
      element: element,
      gfx: container.gfx,
      marker: marker,
      add: !!add
    });
  };

  /**
   * Adds a marker to an element (basically a css class).
   *
   * Fires the element.marker.update event, making it possible to
   * integrate extension into the marker life-cycle, too.
   *
   * @example
   *
   * ```javascript
   * canvas.addMarker('foo', 'some-marker');
   *
   * const fooGfx = canvas.getGraphics('foo');
   *
   * fooGfx; // <g class="... some-marker"> ... </g>
   * ```
   *
   * @param {ShapeLike|ConnectionLike|string} element The element or its ID.
   * @param {string} marker The marker.
   */
  Canvas.prototype.addMarker = function (element, marker) {
    this._updateMarker(element, marker, true);
  };

  /**
   * Remove a marker from an element.
   *
   * Fires the element.marker.update event, making it possible to
   * integrate extension into the marker life-cycle, too.
   *
   * @param {ShapeLike|ConnectionLike|string} element The element or its ID.
   * @param {string} marker The marker.
   */
  Canvas.prototype.removeMarker = function (element, marker) {
    this._updateMarker(element, marker, false);
  };

  /**
   * Check whether an element has a given marker.
   *
   * @param {ShapeLike|ConnectionLike|string} element The element or its ID.
   * @param {string} marker The marker.
   */
  Canvas.prototype.hasMarker = function (element, marker) {
    if (!element.id) {
      element = this._elementRegistry.get(element);
    }
    if (!element.markers) {
      return false;
    }
    return element.markers.has(marker);
  };

  /**
   * Toggles a marker on an element.
   *
   * Fires the element.marker.update event, making it possible to
   * integrate extension into the marker life-cycle, too.
   *
   * @param {ShapeLike|ConnectionLike|string} element The element or its ID.
   * @param {string} marker The marker.
   */
  Canvas.prototype.toggleMarker = function (element, marker) {
    if (this.hasMarker(element, marker)) {
      this.removeMarker(element, marker);
    } else {
      this.addMarker(element, marker);
    }
  };

  /**
   * Returns the current root element.
   *
   * Supports two different modes for handling root elements:
   *
   * 1. if no root element has been added before, an implicit root will be added
   * and returned. This is used in applications that don't require explicit
   * root elements.
   *
   * 2. when root elements have been added before calling `getRootElement`,
   * root elements can be null. This is used for applications that want to manage
   * root elements themselves.
   *
   * @return {RootLike} The current root element.
   */
  Canvas.prototype.getRootElement = function () {
    const rootElement = this._rootElement;

    // can return null if root elements are present but none was set yet
    if (rootElement || this._planes.length) {
      return rootElement;
    }
    return this.setRootElement(this.addRootElement(null));
  };

  /**
   * Adds a given root element and returns it.
   *
   * @param {RootLike} [rootElement] The root element to be added.
   *
   * @return {RootLike} The added root element or an implicit root element.
   */
  Canvas.prototype.addRootElement = function (rootElement) {
    const idx = this._rootsIdx++;
    if (!rootElement) {
      rootElement = {
        id: '__implicitroot_' + idx,
        children: [],
        isImplicit: true
      };
    }
    const layerName = rootElement.layer = 'root-' + idx;
    this._ensureValid('root', rootElement);
    const layer = this.getLayer(layerName, PLANE_LAYER_INDEX);
    this.hideLayer(layerName);
    this._addRoot(rootElement, layer);
    this._planes.push({
      rootElement: rootElement,
      layer: layer
    });
    return rootElement;
  };

  /**
   * Removes a given root element and returns it.
   *
   * @param {RootLike|string} rootElement element or element ID
   *
   * @return {RootLike|undefined} removed element
   */
  Canvas.prototype.removeRootElement = function (rootElement) {
    if (typeof rootElement === 'string') {
      rootElement = this._elementRegistry.get(rootElement);
    }
    const plane = this._findPlaneForRoot(rootElement);
    if (!plane) {
      return;
    }

    // hook up life-cycle events
    this._removeRoot(rootElement);

    // clean up layer
    this._removeLayer(rootElement.layer);

    // clean up plane
    this._planes = this._planes.filter(function (plane) {
      return plane.rootElement !== rootElement;
    });

    // clean up active root
    if (this._rootElement === rootElement) {
      this._rootElement = null;
    }
    return rootElement;
  };

  /**
   * Sets a given element as the new root element for the canvas
   * and returns the new root element.
   *
   * @param {RootLike} rootElement The root element to be set.
   *
   * @return {RootLike} The set root element.
   */
  Canvas.prototype.setRootElement = function (rootElement) {
    if (rootElement === this._rootElement) {
      return rootElement;
    }
    let plane;
    if (!rootElement) {
      throw new Error('rootElement required');
    }
    plane = this._findPlaneForRoot(rootElement);

    // give set add semantics for backwards compatibility
    if (!plane) {
      rootElement = this.addRootElement(rootElement);
    }
    this._setRoot(rootElement);
    return rootElement;
  };
  Canvas.prototype._removeRoot = function (element) {
    const elementRegistry = this._elementRegistry,
      eventBus = this._eventBus;

    // simulate element remove event sequence
    eventBus.fire('root.remove', {
      element: element
    });
    eventBus.fire('root.removed', {
      element: element
    });
    elementRegistry.remove(element);
  };
  Canvas.prototype._addRoot = function (element, gfx) {
    const elementRegistry = this._elementRegistry,
      eventBus = this._eventBus;

    // resemble element add event sequence
    eventBus.fire('root.add', {
      element: element
    });
    elementRegistry.add(element, gfx);
    eventBus.fire('root.added', {
      element: element,
      gfx: gfx
    });
  };
  Canvas.prototype._setRoot = function (rootElement, layer) {
    const currentRoot = this._rootElement;
    if (currentRoot) {
      // un-associate previous root element <svg>
      this._elementRegistry.updateGraphics(currentRoot, null, true);

      // hide previous layer
      this.hideLayer(currentRoot.layer);
    }
    if (rootElement) {
      if (!layer) {
        layer = this._findPlaneForRoot(rootElement).layer;
      }

      // associate element with <svg>
      this._elementRegistry.updateGraphics(rootElement, this._svg, true);

      // show root layer
      this.showLayer(rootElement.layer);
    }
    this._rootElement = rootElement;
    this._eventBus.fire('root.set', {
      element: rootElement
    });
  };
  Canvas.prototype._ensureValid = function (type, element) {
    if (!element.id) {
      throw new Error('element must have an id');
    }
    if (this._elementRegistry.get(element.id)) {
      throw new Error('element <' + element.id + '> already exists');
    }
    const requiredAttrs = REQUIRED_MODEL_ATTRS[type];
    const valid = every(requiredAttrs, function (attr) {
      return typeof element[attr] !== 'undefined';
    });
    if (!valid) {
      throw new Error('must supply { ' + requiredAttrs.join(', ') + ' } with ' + type);
    }
  };
  Canvas.prototype._setParent = function (element, parent, parentIndex) {
    add$2(parent.children, element, parentIndex);
    element.parent = parent;
  };

  /**
   * Adds an element to the canvas.
   *
   * This wires the parent <-> child relationship between the element and
   * a explicitly specified parent or an implicit root element.
   *
   * During add it emits the events
   *
   *  * <{type}.add> (element, parent)
   *  * <{type}.added> (element, gfx)
   *
   * Extensions may hook into these events to perform their magic.
   *
   * @param {string} type
   * @param {ConnectionLike|ShapeLike} element
   * @param {ShapeLike} [parent]
   * @param {number} [parentIndex]
   *
   * @return {ConnectionLike|ShapeLike} The added element.
   */
  Canvas.prototype._addElement = function (type, element, parent, parentIndex) {
    parent = parent || this.getRootElement();
    const eventBus = this._eventBus,
      graphicsFactory = this._graphicsFactory;
    this._ensureValid(type, element);
    eventBus.fire(type + '.add', {
      element: element,
      parent: parent
    });
    this._setParent(element, parent, parentIndex);

    // create graphics
    const gfx = graphicsFactory.create(type, element, parentIndex);
    this._elementRegistry.add(element, gfx);

    // update its visual
    graphicsFactory.update(type, element, gfx);
    eventBus.fire(type + '.added', {
      element: element,
      gfx: gfx
    });
    return element;
  };

  /**
   * Adds a shape to the canvas.
   *
   * @param {ShapeLike} shape The shape to be added
   * @param {ParentLike} [parent] The shape's parent.
   * @param {number} [parentIndex] The index at which to add the shape to the parent's children.
   *
   * @return {ShapeLike} The added shape.
   */
  Canvas.prototype.addShape = function (shape, parent, parentIndex) {
    return this._addElement('shape', shape, parent, parentIndex);
  };

  /**
   * Adds a connection to the canvas.
   *
   * @param {ConnectionLike} connection The connection to be added.
   * @param {ParentLike} [parent] The connection's parent.
   * @param {number} [parentIndex] The index at which to add the connection to the parent's children.
   *
   * @return {ConnectionLike} The added connection.
   */
  Canvas.prototype.addConnection = function (connection, parent, parentIndex) {
    return this._addElement('connection', connection, parent, parentIndex);
  };

  /**
   * Internal remove element
   */
  Canvas.prototype._removeElement = function (element, type) {
    const elementRegistry = this._elementRegistry,
      graphicsFactory = this._graphicsFactory,
      eventBus = this._eventBus;
    element = elementRegistry.get(element.id || element);
    if (!element) {
      // element was removed already
      return;
    }
    eventBus.fire(type + '.remove', {
      element: element
    });
    graphicsFactory.remove(element);

    // unset parent <-> child relationship
    remove$2(element.parent && element.parent.children, element);
    element.parent = null;
    eventBus.fire(type + '.removed', {
      element: element
    });
    elementRegistry.remove(element);
    return element;
  };

  /**
   * Removes a shape from the canvas.
   *
   * @fires ShapeRemoveEvent
   * @fires ShapeRemovedEvent
   *
   * @param {ShapeLike|string} shape The shape or its ID.
   *
   * @return {ShapeLike} The removed shape.
   */
  Canvas.prototype.removeShape = function (shape) {
    /**
     * An event indicating that a shape is about to be removed from the canvas.
     *
     * @memberOf Canvas
     *
     * @event ShapeRemoveEvent
     * @type {Object}
     * @property {ShapeLike} element The shape.
     * @property {SVGElement} gfx The graphical element.
     */

    /**
     * An event indicating that a shape has been removed from the canvas.
     *
     * @memberOf Canvas
     *
     * @event ShapeRemovedEvent
     * @type {Object}
     * @property {ShapeLike} element The shape.
     * @property {SVGElement} gfx The graphical element.
     */
    return this._removeElement(shape, 'shape');
  };

  /**
   * Removes a connection from the canvas.
   *
   * @fires ConnectionRemoveEvent
   * @fires ConnectionRemovedEvent
   *
   * @param {ConnectionLike|string} connection The connection or its ID.
   *
   * @return {ConnectionLike} The removed connection.
   */
  Canvas.prototype.removeConnection = function (connection) {
    /**
     * An event indicating that a connection is about to be removed from the canvas.
     *
     * @memberOf Canvas
     *
     * @event ConnectionRemoveEvent
     * @type {Object}
     * @property {ConnectionLike} element The connection.
     * @property {SVGElement} gfx The graphical element.
     */

    /**
     * An event indicating that a connection has been removed from the canvas.
     *
     * @memberOf Canvas
     *
     * @event ConnectionRemovedEvent
     * @type {Object}
     * @property {ConnectionLike} element The connection.
     * @property {SVGElement} gfx The graphical element.
     */
    return this._removeElement(connection, 'connection');
  };

  /**
   * Returns the graphical element of an element.
   *
   * @param {ShapeLike|ConnectionLike|string} element The element or its ID.
   * @param {boolean} [secondary=false] Whether to return the secondary graphical element.
   *
   * @return {SVGElement} The graphical element.
   */
  Canvas.prototype.getGraphics = function (element, secondary) {
    return this._elementRegistry.getGraphics(element, secondary);
  };

  /**
   * Perform a viewbox update via a given change function.
   *
   * @param {Function} changeFn
   */
  Canvas.prototype._changeViewbox = function (changeFn) {
    // notify others of the upcoming viewbox change
    this._eventBus.fire('canvas.viewbox.changing');

    // perform actual change
    changeFn.apply(this);

    // reset the cached viewbox so that
    // a new get operation on viewbox or zoom
    // triggers a viewbox re-computation
    this._cachedViewbox = null;

    // notify others of the change; this step
    // may or may not be debounced
    this._viewboxChanged();
  };
  Canvas.prototype._viewboxChanged = function () {
    this._eventBus.fire('canvas.viewbox.changed', {
      viewbox: this.viewbox()
    });
  };

  /**
   * Gets or sets the view box of the canvas, i.e. the
   * area that is currently displayed.
   *
   * The getter may return a cached viewbox (if it is currently
   * changing). To force a recomputation, pass `false` as the first argument.
   *
   * @example
   *
   * ```javascript
   * canvas.viewbox({ x: 100, y: 100, width: 500, height: 500 })
   *
   * // sets the visible area of the diagram to (100|100) -> (600|100)
   * // and and scales it according to the diagram width
   *
   * const viewbox = canvas.viewbox(); // pass `false` to force recomputing the box.
   *
   * console.log(viewbox);
   * // {
   * //   inner: Dimensions,
   * //   outer: Dimensions,
   * //   scale,
   * //   x, y,
   * //   width, height
   * // }
   *
   * // if the current diagram is zoomed and scrolled, you may reset it to the
   * // default zoom via this method, too:
   *
   * const zoomedAndScrolledViewbox = canvas.viewbox();
   *
   * canvas.viewbox({
   *   x: 0,
   *   y: 0,
   *   width: zoomedAndScrolledViewbox.outer.width,
   *   height: zoomedAndScrolledViewbox.outer.height
   * });
   * ```
   *
   * @param {Rect} [box] The viewbox to be set.
   *
   * @return {CanvasViewbox} The set viewbox.
   */
  Canvas.prototype.viewbox = function (box) {
    if (box === undefined && this._cachedViewbox) {
      return structuredClone(this._cachedViewbox);
    }
    const viewport = this._viewport,
      outerBox = this.getSize();
    let innerBox, matrix, activeLayer, transform$1, scale, x, y;
    if (!box) {
      // compute the inner box based on the
      // diagrams active layer. This allows us to exclude
      // external components, such as overlays

      activeLayer = this._rootElement ? this.getActiveLayer() : null;
      innerBox = activeLayer && activeLayer.getBBox() || {};
      transform$1 = transform(viewport);
      matrix = transform$1 ? transform$1.matrix : createMatrix();
      scale = round(matrix.a, 1000);
      x = round(-matrix.e || 0, 1000);
      y = round(-matrix.f || 0, 1000);
      box = this._cachedViewbox = {
        x: x ? x / scale : 0,
        y: y ? y / scale : 0,
        width: outerBox.width / scale,
        height: outerBox.height / scale,
        scale: scale,
        inner: {
          width: innerBox.width || 0,
          height: innerBox.height || 0,
          x: innerBox.x || 0,
          y: innerBox.y || 0
        },
        outer: outerBox
      };
      return box;
    } else {
      this._changeViewbox(function () {
        scale = Math.min(outerBox.width / box.width, outerBox.height / box.height);
        const matrix = this._svg.createSVGMatrix().scale(scale).translate(-box.x, -box.y);
        transform(viewport, matrix);
      });
    }
    return box;
  };

  /**
   * Gets or sets the scroll of the canvas.
   *
   * @param {ScrollDelta} [delta] The scroll to be set.
   *
   * @return {Point}
   */
  Canvas.prototype.scroll = function (delta) {
    const node = this._viewport;
    let matrix = node.getCTM();
    if (delta) {
      this._changeViewbox(function () {
        delta = assign$4({
          dx: 0,
          dy: 0
        }, delta || {});
        matrix = this._svg.createSVGMatrix().translate(delta.dx, delta.dy).multiply(matrix);
        setCTM(node, matrix);
      });
    }
    return {
      x: matrix.e,
      y: matrix.f
    };
  };

  /**
   * Scrolls the viewbox to contain the given element.
   * Optionally specify a padding to be applied to the edges.
   *
   * @param {ShapeLike|ConnectionLike|string} element The element to scroll to or its ID.
   * @param {RectTRBL|number} [padding=100] The padding to be applied. Can also specify top, bottom, left and right.
   */
  Canvas.prototype.scrollToElement = function (element, padding) {
    let defaultPadding = 100;
    if (typeof element === 'string') {
      element = this._elementRegistry.get(element);
    }

    // set to correct rootElement
    const rootElement = this.findRoot(element);
    if (rootElement !== this.getRootElement()) {
      this.setRootElement(rootElement);
    }

    // element is rootElement, do not change viewport
    if (rootElement === element) {
      return;
    }
    if (!padding) {
      padding = {};
    }
    if (typeof padding === 'number') {
      defaultPadding = padding;
    }
    padding = {
      top: padding.top || defaultPadding,
      right: padding.right || defaultPadding,
      bottom: padding.bottom || defaultPadding,
      left: padding.left || defaultPadding
    };
    const elementBounds = getBBox(element),
      elementTrbl = asTRBL(elementBounds),
      viewboxBounds = this.viewbox(),
      zoom = this.zoom();
    let dx, dy;

    // shrink viewboxBounds with padding
    viewboxBounds.y += padding.top / zoom;
    viewboxBounds.x += padding.left / zoom;
    viewboxBounds.width -= (padding.right + padding.left) / zoom;
    viewboxBounds.height -= (padding.bottom + padding.top) / zoom;
    const viewboxTrbl = asTRBL(viewboxBounds);
    const canFit = elementBounds.width < viewboxBounds.width && elementBounds.height < viewboxBounds.height;
    if (!canFit) {
      // top-left when element can't fit
      dx = elementBounds.x - viewboxBounds.x;
      dy = elementBounds.y - viewboxBounds.y;
    } else {
      const dRight = Math.max(0, elementTrbl.right - viewboxTrbl.right),
        dLeft = Math.min(0, elementTrbl.left - viewboxTrbl.left),
        dBottom = Math.max(0, elementTrbl.bottom - viewboxTrbl.bottom),
        dTop = Math.min(0, elementTrbl.top - viewboxTrbl.top);
      dx = dRight || dLeft;
      dy = dBottom || dTop;
    }
    this.scroll({
      dx: -dx * zoom,
      dy: -dy * zoom
    });
  };

  /**
   * Gets or sets the current zoom of the canvas, optionally zooming to the
   * specified position.
   *
   * The getter may return a cached zoom level. Call it with `false` as the first
   * argument to force recomputation of the current level.
   *
   * @param {number|'fit-viewport'} [newScale] The new zoom level, either a number,
   * i.e. 0.9, or `fit-viewport` to adjust the size to fit the current viewport.
   * @param {Point} [center] The reference point { x: ..., y: ...} to zoom to.
   *
   * @return {number} The set zoom level.
   */
  Canvas.prototype.zoom = function (newScale, center) {
    if (!newScale) {
      return this.viewbox(newScale).scale;
    }
    if (newScale === 'fit-viewport') {
      return this._fitViewport(center);
    }
    let outer, matrix;
    this._changeViewbox(function () {
      if (typeof center !== 'object') {
        outer = this.viewbox().outer;
        center = {
          x: outer.width / 2,
          y: outer.height / 2
        };
      }
      matrix = this._setZoom(newScale, center);
    });
    return round(matrix.a, 1000);
  };
  function setCTM(node, m) {
    const mstr = 'matrix(' + m.a + ',' + m.b + ',' + m.c + ',' + m.d + ',' + m.e + ',' + m.f + ')';
    node.setAttribute('transform', mstr);
  }
  Canvas.prototype._fitViewport = function (center) {
    const vbox = this.viewbox(),
      outer = vbox.outer,
      inner = vbox.inner;
    let newScale, newViewbox;

    // display the complete diagram without zooming in.
    // instead of relying on internal zoom, we perform a
    // hard reset on the canvas viewbox to realize this
    //
    // if diagram does not need to be zoomed in, we focus it around
    // the diagram origin instead

    if (inner.x >= 0 && inner.y >= 0 && inner.x + inner.width <= outer.width && inner.y + inner.height <= outer.height && !center) {
      newViewbox = {
        x: 0,
        y: 0,
        width: Math.max(inner.width + inner.x, outer.width),
        height: Math.max(inner.height + inner.y, outer.height)
      };
    } else {
      newScale = Math.min(1, outer.width / inner.width, outer.height / inner.height);
      newViewbox = {
        x: inner.x + (center ? inner.width / 2 - outer.width / newScale / 2 : 0),
        y: inner.y + (center ? inner.height / 2 - outer.height / newScale / 2 : 0),
        width: outer.width / newScale,
        height: outer.height / newScale
      };
    }
    this.viewbox(newViewbox);
    return this.viewbox(false).scale;
  };
  Canvas.prototype._setZoom = function (scale, center) {
    const svg = this._svg,
      viewport = this._viewport;
    const matrix = svg.createSVGMatrix();
    const point = svg.createSVGPoint();
    let centerPoint, originalPoint, currentMatrix, scaleMatrix, newMatrix;
    currentMatrix = viewport.getCTM();
    const currentScale = currentMatrix.a;
    if (center) {
      centerPoint = assign$4(point, center);

      // revert applied viewport transformations
      originalPoint = centerPoint.matrixTransform(currentMatrix.inverse());

      // create scale matrix
      scaleMatrix = matrix.translate(originalPoint.x, originalPoint.y).scale(1 / currentScale * scale).translate(-originalPoint.x, -originalPoint.y);
      newMatrix = currentMatrix.multiply(scaleMatrix);
    } else {
      newMatrix = matrix.scale(scale);
    }
    setCTM(this._viewport, newMatrix);
    return newMatrix;
  };

  /**
   * Returns the size of the canvas.
   *
   * @return {Dimensions} The size of the canvas.
   */
  Canvas.prototype.getSize = function () {
    return {
      width: this._container.clientWidth,
      height: this._container.clientHeight
    };
  };

  /**
   * Returns the absolute bounding box of an element.
   *
   * The absolute bounding box may be used to display overlays in the callers
   * (browser) coordinate system rather than the zoomed in/out canvas coordinates.
   *
   * @param {ShapeLike|ConnectionLike} element The element.
   *
   * @return {Rect} The element's absolute bounding box.
   */
  Canvas.prototype.getAbsoluteBBox = function (element) {
    const vbox = this.viewbox();
    let bbox;

    // connection
    // use svg bbox
    if (element.waypoints) {
      const gfx = this.getGraphics(element);
      bbox = gfx.getBBox();
    }

    // shapes
    // use data
    else {
      bbox = element;
    }
    const x = bbox.x * vbox.scale - vbox.x * vbox.scale;
    const y = bbox.y * vbox.scale - vbox.y * vbox.scale;
    const width = bbox.width * vbox.scale;
    const height = bbox.height * vbox.scale;
    return {
      x: x,
      y: y,
      width: width,
      height: height
    };
  };

  /**
   * Fires an event so other modules can react to the canvas resizing.
   */
  Canvas.prototype.resized = function () {
    // force recomputation of view box
    delete this._cachedViewbox;
    this._eventBus.fire('canvas.resized');
  };

  var ELEMENT_ID = 'data-element-id';

  /**
   * @typedef {import('./Types').ElementLike} ElementLike
   *
   * @typedef {import('./EventBus').default} EventBus
   *
   * @typedef { (element: ElementLike, gfx: SVGElement) => boolean|any } ElementRegistryFilterCallback
   * @typedef { (element: ElementLike, gfx: SVGElement) => any } ElementRegistryForEachCallback
   */

  /**
   * A registry that keeps track of all shapes in the diagram.
   *
   * @class
   * @constructor
   *
   * @param {EventBus} eventBus
   */
  function ElementRegistry$2(eventBus) {
    /**
     * @type { {
     *   [id: string]: {
     *     element: ElementLike;
     *     gfx?: SVGElement;
     *     secondaryGfx?: SVGElement;
     *   }
     * } }
     */
    this._elements = {};
    this._eventBus = eventBus;
  }
  ElementRegistry$2.$inject = ['eventBus'];

  /**
   * Add an element and its graphical representation(s) to the registry.
   *
   * @param {ElementLike} element The element to be added.
   * @param {SVGElement} gfx The primary graphical representation.
   * @param {SVGElement} [secondaryGfx] The secondary graphical representation.
   */
  ElementRegistry$2.prototype.add = function (element, gfx, secondaryGfx) {
    var id = element.id;
    this._validateId(id);

    // associate dom node with element
    attr(gfx, ELEMENT_ID, id);
    if (secondaryGfx) {
      attr(secondaryGfx, ELEMENT_ID, id);
    }
    this._elements[id] = {
      element: element,
      gfx: gfx,
      secondaryGfx: secondaryGfx
    };
  };

  /**
   * Remove an element from the registry.
   *
   * @param {ElementLike|string} element
   */
  ElementRegistry$2.prototype.remove = function (element) {
    var elements = this._elements,
      id = element.id || element,
      container = id && elements[id];
    if (container) {
      // unset element id on gfx
      attr(container.gfx, ELEMENT_ID, '');
      if (container.secondaryGfx) {
        attr(container.secondaryGfx, ELEMENT_ID, '');
      }
      delete elements[id];
    }
  };

  /**
   * Update an elements ID.
   *
   * @param {ElementLike|string} element The element or its ID.
   * @param {string} newId The new ID.
   */
  ElementRegistry$2.prototype.updateId = function (element, newId) {
    this._validateId(newId);
    if (typeof element === 'string') {
      element = this.get(element);
    }
    this._eventBus.fire('element.updateId', {
      element: element,
      newId: newId
    });
    var gfx = this.getGraphics(element),
      secondaryGfx = this.getGraphics(element, true);
    this.remove(element);
    element.id = newId;
    this.add(element, gfx, secondaryGfx);
  };

  /**
   * Update the graphical representation of an element.
   *
   * @param {ElementLike|string} filter The element or its ID.
   * @param {SVGElement} gfx The new graphical representation.
   * @param {boolean} [secondary=false] Whether to update the secondary graphical representation.
   */
  ElementRegistry$2.prototype.updateGraphics = function (filter, gfx, secondary) {
    var id = filter.id || filter;
    var container = this._elements[id];
    if (secondary) {
      container.secondaryGfx = gfx;
    } else {
      container.gfx = gfx;
    }
    if (gfx) {
      attr(gfx, ELEMENT_ID, id);
    }
    return gfx;
  };

  /**
   * Get the element with the given ID or graphical representation.
   *
   * @example
   *
   * ```javascript
   * elementRegistry.get('SomeElementId_1');
   *
   * elementRegistry.get(gfx);
   * ```
   *
   * @param {string|SVGElement} filter The elements ID or graphical representation.
   *
   * @return {ElementLike|undefined} The element.
   */
  ElementRegistry$2.prototype.get = function (filter) {
    var id;
    if (typeof filter === 'string') {
      id = filter;
    } else {
      id = filter && attr(filter, ELEMENT_ID);
    }
    var container = this._elements[id];
    return container && container.element;
  };

  /**
   * Return all elements that match a given filter function.
   *
   * @param {ElementRegistryFilterCallback} fn The filter function.
   *
   * @return {ElementLike[]} The matching elements.
   */
  ElementRegistry$2.prototype.filter = function (fn) {
    var filtered = [];
    this.forEach(function (element, gfx) {
      if (fn(element, gfx)) {
        filtered.push(element);
      }
    });
    return filtered;
  };

  /**
   * Return the first element that matches the given filter function.
   *
   * @param {ElementRegistryFilterCallback} fn The filter function.
   *
   * @return {ElementLike|undefined} The matching element.
   */
  ElementRegistry$2.prototype.find = function (fn) {
    var map = this._elements,
      keys = Object.keys(map);
    for (var i = 0; i < keys.length; i++) {
      var id = keys[i],
        container = map[id],
        element = container.element,
        gfx = container.gfx;
      if (fn(element, gfx)) {
        return element;
      }
    }
  };

  /**
   * Get all elements.
   *
   * @return {ElementLike[]} All elements.
   */
  ElementRegistry$2.prototype.getAll = function () {
    return this.filter(function (e) {
      return e;
    });
  };

  /**
   * Execute a given function for each element.
   *
   * @param {ElementRegistryForEachCallback} fn The function to execute.
   */
  ElementRegistry$2.prototype.forEach = function (fn) {
    var map = this._elements;
    Object.keys(map).forEach(function (id) {
      var container = map[id],
        element = container.element,
        gfx = container.gfx;
      return fn(element, gfx);
    });
  };

  /**
   * Return the graphical representation of an element.
   *
   * @example
   *
   * ```javascript
   * elementRegistry.getGraphics('SomeElementId_1');
   *
   * elementRegistry.getGraphics(rootElement); // <g ...>
   *
   * elementRegistry.getGraphics(rootElement, true); // <svg ...>
   * ```
   *
   * @param {ElementLike|string} filter The element or its ID.
   * @param {boolean} [secondary=false] Whether to return the secondary graphical representation.
   *
   * @return {SVGElement} The graphical representation.
   */
  ElementRegistry$2.prototype.getGraphics = function (filter, secondary) {
    var id = filter.id || filter;
    var container = this._elements[id];
    return container && (secondary ? container.secondaryGfx : container.gfx);
  };

  /**
   * Validate an ID and throw an error if invalid.
   *
   * @param {string} id
   *
   * @throws {Error} Error indicating that the ID is invalid or already assigned.
   */
  ElementRegistry$2.prototype._validateId = function (id) {
    if (!id) {
      throw new Error('element must have an id');
    }
    if (this._elements[id]) {
      throw new Error('element with id ' + id + ' already added');
    }
  };

  /**
   * Extends a collection with {@link Refs} aware methods
   *
   * @param {Array<Object>} collection
   * @param {Refs} refs instance
   * @param {Object} property represented by the collection
   * @param {Object} target object the collection is attached to
   *
   * @return {RefsCollection<Object>} the extended array
   */
  function extend(collection, refs, property, target) {
    var inverseProperty = property.inverse;

    /**
     * Removes the given element from the array and returns it.
     *
     * @method RefsCollection#remove
     *
     * @param {Object} element the element to remove
     */
    Object.defineProperty(collection, 'remove', {
      value: function (element) {
        var idx = this.indexOf(element);
        if (idx !== -1) {
          this.splice(idx, 1);

          // unset inverse
          refs.unset(element, inverseProperty, target);
        }
        return element;
      }
    });

    /**
     * Returns true if the collection contains the given element
     *
     * @method RefsCollection#contains
     *
     * @param {Object} element the element to check for
     */
    Object.defineProperty(collection, 'contains', {
      value: function (element) {
        return this.indexOf(element) !== -1;
      }
    });

    /**
     * Adds an element to the array, unless it exists already (set semantics).
     *
     * @method RefsCollection#add
     *
     * @param {Object} element the element to add
     * @param {Number} optional index to add element to
     *                 (possibly moving other elements around)
     */
    Object.defineProperty(collection, 'add', {
      value: function (element, idx) {
        var currentIdx = this.indexOf(element);
        if (typeof idx === 'undefined') {
          if (currentIdx !== -1) {
            // element already in collection (!)
            return;
          }

          // add to end of array, as no idx is specified
          idx = this.length;
        }

        // handle already in collection
        if (currentIdx !== -1) {
          // remove element from currentIdx
          this.splice(currentIdx, 1);
        }

        // add element at idx
        this.splice(idx, 0, element);
        if (currentIdx === -1) {
          // set inverse, unless element was
          // in collection already
          refs.set(element, inverseProperty, target);
        }
      }
    });

    // a simple marker, identifying this element
    // as being a refs collection
    Object.defineProperty(collection, '__refs_collection', {
      value: true
    });
    return collection;
  }

  /**
   * Checks if a given collection is extended
   *
   * @param {Array<Object>} collection
   *
   * @return {boolean}
   */
  function isExtended(collection) {
    return collection.__refs_collection === true;
  }
  function hasOwnProperty(e, property) {
    return Object.prototype.hasOwnProperty.call(e, property.name || property);
  }
  function defineCollectionProperty(ref, property, target) {
    var collection = extend(target[property.name] || [], ref, property, target);
    Object.defineProperty(target, property.name, {
      enumerable: property.enumerable,
      value: collection
    });
    if (collection.length) {
      collection.forEach(function (o) {
        ref.set(o, property.inverse, target);
      });
    }
  }
  function defineProperty$1(ref, property, target) {
    var inverseProperty = property.inverse;
    var _value = target[property.name];
    Object.defineProperty(target, property.name, {
      configurable: property.configurable,
      enumerable: property.enumerable,
      get: function () {
        return _value;
      },
      set: function (value) {
        // return if we already performed all changes
        if (value === _value) {
          return;
        }
        var old = _value;

        // temporary set null
        _value = null;
        if (old) {
          ref.unset(old, inverseProperty, target);
        }

        // set new value
        _value = value;

        // set inverse value
        ref.set(_value, inverseProperty, target);
      }
    });
  }

  /**
   * Creates a new references object defining two inversly related
   * attribute descriptors a and b.
   *
   * <p>
   *   When bound to an object using {@link Refs#bind} the references
   *   get activated and ensure that add and remove operations are applied
   *   reversely, too.
   * </p>
   *
   * <p>
   *   For attributes represented as collections {@link Refs} provides the
   *   {@link RefsCollection#add}, {@link RefsCollection#remove} and {@link RefsCollection#contains} extensions
   *   that must be used to properly hook into the inverse change mechanism.
   * </p>
   *
   * @class Refs
   *
   * @classdesc A bi-directional reference between two attributes.
   *
   * @param {Refs.AttributeDescriptor} a property descriptor
   * @param {Refs.AttributeDescriptor} b property descriptor
   *
   * @example
   *
   * var refs = Refs({ name: 'wheels', collection: true, enumerable: true }, { name: 'car' });
   *
   * var car = { name: 'toyota' };
   * var wheels = [{ pos: 'front-left' }, { pos: 'front-right' }];
   *
   * refs.bind(car, 'wheels');
   *
   * car.wheels // []
   * car.wheels.add(wheels[0]);
   * car.wheels.add(wheels[1]);
   *
   * car.wheels // [{ pos: 'front-left' }, { pos: 'front-right' }]
   *
   * wheels[0].car // { name: 'toyota' };
   * car.wheels.remove(wheels[0]);
   *
   * wheels[0].car // undefined
   */
  function Refs(a, b) {
    if (!(this instanceof Refs)) {
      return new Refs(a, b);
    }

    // link
    a.inverse = b;
    b.inverse = a;
    this.props = {};
    this.props[a.name] = a;
    this.props[b.name] = b;
  }

  /**
   * Binds one side of a bi-directional reference to a
   * target object.
   *
   * @memberOf Refs
   *
   * @param  {Object} target
   * @param  {String} property
   */
  Refs.prototype.bind = function (target, property) {
    if (typeof property === 'string') {
      if (!this.props[property]) {
        throw new Error('no property <' + property + '> in ref');
      }
      property = this.props[property];
    }
    if (property.collection) {
      defineCollectionProperty(this, property, target);
    } else {
      defineProperty$1(this, property, target);
    }
  };
  Refs.prototype.ensureRefsCollection = function (target, property) {
    var collection = target[property.name];
    if (!isExtended(collection)) {
      defineCollectionProperty(this, property, target);
    }
    return collection;
  };
  Refs.prototype.ensureBound = function (target, property) {
    if (!hasOwnProperty(target, property)) {
      this.bind(target, property);
    }
  };
  Refs.prototype.unset = function (target, property, value) {
    if (target) {
      this.ensureBound(target, property);
      if (property.collection) {
        this.ensureRefsCollection(target, property).remove(value);
      } else {
        target[property.name] = undefined;
      }
    }
  };
  Refs.prototype.set = function (target, property, value) {
    if (target) {
      this.ensureBound(target, property);
      if (property.collection) {
        this.ensureRefsCollection(target, property).add(value);
      } else {
        target[property.name] = value;
      }
    }
  };

  var parentRefs = new Refs({
      name: 'children',
      enumerable: true,
      collection: true
    }, {
      name: 'parent'
    }),
    labelRefs = new Refs({
      name: 'labels',
      enumerable: true,
      collection: true
    }, {
      name: 'labelTarget'
    }),
    attacherRefs = new Refs({
      name: 'attachers',
      collection: true
    }, {
      name: 'host'
    }),
    outgoingRefs = new Refs({
      name: 'outgoing',
      collection: true
    }, {
      name: 'source'
    }),
    incomingRefs = new Refs({
      name: 'incoming',
      collection: true
    }, {
      name: 'target'
    });

  /**
   * @typedef {import('./Types').Element} Element
   * @typedef {import('./Types').Shape} Shape
   * @typedef {import('./Types').Root} Root
   * @typedef {import('./Types').Label} Label
   * @typedef {import('./Types').Connection} Connection
   */

  /**
   * The basic graphical representation
   *
   * @class
   * @constructor
   */
  function ElementImpl() {
    /**
     * The object that backs up the shape
     *
     * @name Element#businessObject
     * @type Object
     */
    Object.defineProperty(this, 'businessObject', {
      writable: true
    });

    /**
     * Single label support, will mapped to multi label array
     *
     * @name Element#label
     * @type Object
     */
    Object.defineProperty(this, 'label', {
      get: function () {
        return this.labels[0];
      },
      set: function (newLabel) {
        var label = this.label,
          labels = this.labels;
        if (!newLabel && label) {
          labels.remove(label);
        } else {
          labels.add(newLabel, 0);
        }
      }
    });

    /**
     * The parent shape
     *
     * @name Element#parent
     * @type Shape
     */
    parentRefs.bind(this, 'parent');

    /**
     * The list of labels
     *
     * @name Element#labels
     * @type Label
     */
    labelRefs.bind(this, 'labels');

    /**
     * The list of outgoing connections
     *
     * @name Element#outgoing
     * @type Array<Connection>
     */
    outgoingRefs.bind(this, 'outgoing');

    /**
     * The list of incoming connections
     *
     * @name Element#incoming
     * @type Array<Connection>
     */
    incomingRefs.bind(this, 'incoming');
  }

  /**
   * A graphical object
   *
   * @class
   * @constructor
   *
   * @extends ElementImpl
   */
  function ShapeImpl() {
    ElementImpl.call(this);

    /**
     * Indicates frame shapes
     *
     * @name ShapeImpl#isFrame
     * @type boolean
     */

    /**
     * The list of children
     *
     * @name ShapeImpl#children
     * @type Element[]
     */
    parentRefs.bind(this, 'children');

    /**
     * @name ShapeImpl#host
     * @type Shape
     */
    attacherRefs.bind(this, 'host');

    /**
     * @name ShapeImpl#attachers
     * @type Shape
     */
    attacherRefs.bind(this, 'attachers');
  }
  e(ShapeImpl, ElementImpl);

  /**
   * A root graphical object
   *
   * @class
   * @constructor
   *
   * @extends ElementImpl
   */
  function RootImpl() {
    ElementImpl.call(this);

    /**
     * The list of children
     *
     * @name RootImpl#children
     * @type Element[]
     */
    parentRefs.bind(this, 'children');
  }
  e(RootImpl, ShapeImpl);

  /**
   * A label for an element
   *
   * @class
   * @constructor
   *
   * @extends ShapeImpl
   */
  function LabelImpl() {
    ShapeImpl.call(this);

    /**
     * The labeled element
     *
     * @name LabelImpl#labelTarget
     * @type Element
     */
    labelRefs.bind(this, 'labelTarget');
  }
  e(LabelImpl, ShapeImpl);

  /**
   * A connection between two elements
   *
   * @class
   * @constructor
   *
   * @extends ElementImpl
   */
  function ConnectionImpl() {
    ElementImpl.call(this);

    /**
     * The element this connection originates from
     *
     * @name ConnectionImpl#source
     * @type Element
     */
    outgoingRefs.bind(this, 'source');

    /**
     * The element this connection points to
     *
     * @name ConnectionImpl#target
     * @type Element
     */
    incomingRefs.bind(this, 'target');
  }
  e(ConnectionImpl, ElementImpl);
  var types$1 = {
    connection: ConnectionImpl,
    shape: ShapeImpl,
    label: LabelImpl,
    root: RootImpl
  };

  /**
   * Creates a root element.
   *
   * @overlord
   *
   * @example
   *
   * ```javascript
   * import * as Model from 'diagram-js/lib/model';
   *
   * const root = Model.create('root', {
   *   x: 100,
   *   y: 100,
   *   width: 100,
   *   height: 100
   * });
   * ```
   *
   * @param {'root'} type
   * @param {any} [attrs]
   *
   * @return {Root}
   */

  /**
   * Creates a connection.
   *
   * @overlord
   *
   * @example
   *
   * ```javascript
   * import * as Model from 'diagram-js/lib/model';
   *
   * const connection = Model.create('connection', {
   *   waypoints: [
   *     { x: 100, y: 100 },
   *     { x: 200, y: 100 }
   *   ]
   * });
   * ```
   *
   * @param {'connection'} type
   * @param {any} [attrs]
   *
   * @return {Connection}
   */

  /**
   * Creates a shape.
   *
   * @overlord
   *
   * @example
   *
   * ```javascript
   * import * as Model from 'diagram-js/lib/model';
   *
   * const shape = Model.create('shape', {
   *   x: 100,
   *   y: 100,
   *   width: 100,
   *   height: 100
   * });
   * ```
   *
   * @param {'shape'} type
   * @param {any} [attrs]
   *
   * @return {Shape}
   */

  /**
   * Creates a label.
   *
   * @example
   *
   * ```javascript
   * import * as Model from 'diagram-js/lib/model';
   *
   * const label = Model.create('label', {
   *   x: 100,
   *   y: 100,
   *   width: 100,
   *   height: 100,
   *   labelTarget: shape
   * });
   * ```
   *
   * @param {'label'} type
   * @param {Object} [attrs]
   *
   * @return {Label}
   */
  function create$1(type, attrs) {
    var Type = types$1[type];
    if (!Type) {
      throw new Error('unknown type: <' + type + '>');
    }
    return assign$4(new Type(), attrs);
  }

  /**
   * @typedef {import('../model/Types').Element} Element
   * @typedef {import('../model/Types').Connection} Connection
   * @typedef {import('../model/Types').Label} Label
   * @typedef {import('../model/Types').Root} Root
   * @typedef {import('../model/Types').Shape} Shape
   */

  /**
   * A factory for model elements.
   *
   * @template {Connection} [T=Connection]
   * @template {Label} [U=Label]
   * @template {Root} [V=Root]
   * @template {Shape} [W=Shape]
   */
  function ElementFactory$1() {
    this._uid = 12;
  }

  /**
   * Create a root element.
   *
   * @param {Partial<Root>} [attrs]
   *
   * @return {V} The created root element.
   */
  ElementFactory$1.prototype.createRoot = function (attrs) {
    return this.create('root', attrs);
  };

  /**
   * Create a label.
   *
   * @param {Partial<Label>} [attrs]
   *
   * @return {U} The created label.
   */
  ElementFactory$1.prototype.createLabel = function (attrs) {
    return this.create('label', attrs);
  };

  /**
   * Create a shape.
   *
   * @param {Partial<Shape>} [attrs]
   *
   * @return {W} The created shape.
   */
  ElementFactory$1.prototype.createShape = function (attrs) {
    return this.create('shape', attrs);
  };

  /**
   * Create a connection.
   *
   * @param {Partial<Connection>} [attrs]
   *
   * @return {T} The created connection.
   */
  ElementFactory$1.prototype.createConnection = function (attrs) {
    return this.create('connection', attrs);
  };

  /**
   * Create a root element.
   *
   * @overlord
   * @param {'root'} type
   * @param {Partial<Root>} [attrs]
   * @return {V}
   */
  /**
   * Create a shape.
   *
   * @overlord
   * @param {'shape'} type
   * @param {Partial<Shape>} [attrs]
   * @return {W}
   */
  /**
   * Create a connection.
   *
   * @overlord
   * @param {'connection'} type
   * @param {Partial<Connection>} [attrs]
   * @return {T}
   */
  /**
   * Create a label.
   *
   * @param {'label'} type
   * @param {Partial<Label>} [attrs]
   * @return {U}
   */
  ElementFactory$1.prototype.create = function (type, attrs) {
    attrs = assign$4({}, attrs || {});
    if (!attrs.id) {
      attrs.id = type + '_' + this._uid++;
    }
    return create$1(type, attrs);
  };

  /**
   * SVGs for elements are generated by the {@link GraphicsFactory}.
   *
   * This utility gives quick access to the important semantic
   * parts of an element.
   */

  /**
   * Returns the visual part of a diagram element.
   *
   * @param {SVGElement} gfx
   *
   * @return {SVGElement}
   */
  function getVisual(gfx) {
    return gfx.childNodes[0];
  }

  /**
   * Returns the children for a given diagram element.
   *
   * @param {SVGElement} gfx
   * @return {SVGElement}
   */
  function getChildren$1(gfx) {
    return gfx.parentNode.childNodes[1];
  }

  /**
   * @param {SVGElement} gfx
   * @param {number} x
   * @param {number} y
   */
  function translate$1(gfx, x, y) {
    var translate = createTransform();
    translate.setTranslate(x, y);
    transform(gfx, translate);
  }

  /**
   * @typedef {import('./Types').ConnectionLike} ConnectionLike
   * @typedef {import('./Types').ElementLike} ElementLike
   * @typedef {import('./Types').ShapeLike} ShapeLike
   *
   * @typedef {import('./ElementRegistry').default} ElementRegistry
   * @typedef {import('./EventBus').default} EventBus
   */

  /**
   * A factory that creates graphical elements.
   *
   * @param {EventBus} eventBus
   * @param {ElementRegistry} elementRegistry
   */
  function GraphicsFactory(eventBus, elementRegistry) {
    this._eventBus = eventBus;
    this._elementRegistry = elementRegistry;
  }
  GraphicsFactory.$inject = ['eventBus', 'elementRegistry'];

  /**
   * @param { { parent?: any } } element
   * @return {SVGElement}
   */
  GraphicsFactory.prototype._getChildrenContainer = function (element) {
    var gfx = this._elementRegistry.getGraphics(element);
    var childrenGfx;

    // root element
    if (!element.parent) {
      childrenGfx = gfx;
    } else {
      childrenGfx = getChildren$1(gfx);
      if (!childrenGfx) {
        childrenGfx = create$2('g');
        classes(childrenGfx).add('djs-children');
        append(gfx.parentNode, childrenGfx);
      }
    }
    return childrenGfx;
  };

  /**
   * Clears the graphical representation of the element and returns the
   * cleared visual (the <g class="djs-visual" /> element).
   */
  GraphicsFactory.prototype._clear = function (gfx) {
    var visual = getVisual(gfx);
    clear(visual);
    return visual;
  };

  /**
   * Creates a gfx container for shapes and connections
   *
   * The layout is as follows:
   *
   * <g class="djs-group">
   *
   *   <!-- the gfx -->
   *   <g class="djs-element djs-(shape|connection|frame)">
   *     <g class="djs-visual">
   *       <!-- the renderer draws in here -->
   *     </g>
   *
   *     <!-- extensions (overlays, click box, ...) goes here
   *   </g>
   *
   *   <!-- the gfx child nodes -->
   *   <g class="djs-children"></g>
   * </g>
   *
   * @param {string} type the type of the element, i.e. shape | connection
   * @param {SVGElement} childrenGfx
   * @param {number} [parentIndex] position to create container in parent
   * @param {boolean} [isFrame] is frame element
   *
   * @return {SVGElement}
   */
  GraphicsFactory.prototype._createContainer = function (type, childrenGfx, parentIndex, isFrame) {
    var outerGfx = create$2('g');
    classes(outerGfx).add('djs-group');

    // insert node at position
    if (typeof parentIndex !== 'undefined') {
      prependTo(outerGfx, childrenGfx, childrenGfx.childNodes[parentIndex]);
    } else {
      append(childrenGfx, outerGfx);
    }
    var gfx = create$2('g');
    classes(gfx).add('djs-element');
    classes(gfx).add('djs-' + type);
    if (isFrame) {
      classes(gfx).add('djs-frame');
    }
    append(outerGfx, gfx);

    // create visual
    var visual = create$2('g');
    classes(visual).add('djs-visual');
    append(gfx, visual);
    return gfx;
  };

  /**
   * Create a graphical element.
   *
   * @param { 'shape' | 'connection' | 'label' | 'root' } type The type of the element.
   * @param {ElementLike} element The element.
   * @param {number} [parentIndex] The index at which to add the graphical element to its parent's children.
   *
   * @return {SVGElement} The graphical element.
   */
  GraphicsFactory.prototype.create = function (type, element, parentIndex) {
    var childrenGfx = this._getChildrenContainer(element.parent);
    return this._createContainer(type, childrenGfx, parentIndex, isFrameElement(element));
  };

  /**
   * Update the containments of the given elements.
   *
   * @param {ElementLike[]} elements The elements.
   */
  GraphicsFactory.prototype.updateContainments = function (elements) {
    var self = this,
      elementRegistry = this._elementRegistry,
      parents;
    parents = reduce(elements, function (map, e) {
      if (e.parent) {
        map[e.parent.id] = e.parent;
      }
      return map;
    }, {});

    // update all parents of changed and reorganized their children
    // in the correct order (as indicated in our model)
    forEach$3(parents, function (parent) {
      var children = parent.children;
      if (!children) {
        return;
      }
      var childrenGfx = self._getChildrenContainer(parent);
      forEach$3(children.slice().reverse(), function (child) {
        var childGfx = elementRegistry.getGraphics(child);
        prependTo(childGfx.parentNode, childrenGfx);
      });
    });
  };

  /**
   * Draw a shape.
   *
   * @param {SVGElement} visual The graphical element.
   * @param {ShapeLike} element The shape.
   * @param {Object} attrs Optional attributes.
   *
   * @return {SVGElement}
   */
  GraphicsFactory.prototype.drawShape = function (visual, element, attrs = {}) {
    var eventBus = this._eventBus;
    return eventBus.fire('render.shape', {
      gfx: visual,
      element,
      attrs
    });
  };

  /**
   * Get the path of a shape.
   *
   * @param {ShapeLike} element The shape.
   *
   * @return {string} The path of the shape.
   */
  GraphicsFactory.prototype.getShapePath = function (element) {
    var eventBus = this._eventBus;
    return eventBus.fire('render.getShapePath', element);
  };

  /**
   * Draw a connection.
   *
   * @param {SVGElement} visual The graphical element.
   * @param {ConnectionLike} element The connection.
   * @param {Object} attrs Optional attributes.
   *
   * @return {SVGElement}
   */
  GraphicsFactory.prototype.drawConnection = function (visual, element, attrs = {}) {
    var eventBus = this._eventBus;
    return eventBus.fire('render.connection', {
      gfx: visual,
      element,
      attrs
    });
  };

  /**
   * Get the path of a connection.
   *
   * @param {ConnectionLike} connection The connection.
   *
   * @return {string} The path of the connection.
   */
  GraphicsFactory.prototype.getConnectionPath = function (connection) {
    var eventBus = this._eventBus;
    return eventBus.fire('render.getConnectionPath', connection);
  };

  /**
   * Update an elements graphical representation.
   *
   * @param {'shape'|'connection'} type
   * @param {ElementLike} element
   * @param {SVGElement} gfx
   */
  GraphicsFactory.prototype.update = function (type, element, gfx) {
    // do NOT update root element
    if (!element.parent) {
      return;
    }
    var visual = this._clear(gfx);

    // redraw
    if (type === 'shape') {
      this.drawShape(visual, element);

      // update positioning
      translate$1(gfx, element.x, element.y);
    } else if (type === 'connection') {
      this.drawConnection(visual, element);
    } else {
      throw new Error('unknown type: ' + type);
    }
    if (element.hidden) {
      attr(gfx, 'display', 'none');
    } else {
      attr(gfx, 'display', 'block');
    }
  };

  /**
   * Remove a graphical element.
   *
   * @param {ElementLike} element The element.
   */
  GraphicsFactory.prototype.remove = function (element) {
    var gfx = this._elementRegistry.getGraphics(element);

    // remove
    remove$3(gfx.parentNode);
  };

  // helpers //////////

  function prependTo(newNode, parentNode, siblingNode) {
    var node = siblingNode || parentNode.firstChild;

    // do not prepend node to itself to prevent IE from crashing
    // https://github.com/bpmn-io/bpmn-js/issues/746
    if (newNode === node) {
      return;
    }
    parentNode.insertBefore(newNode, node);
  }

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var CoreModule$2 = {
    __depends__: [DrawModule$1],
    __init__: ['canvas'],
    canvas: ['type', Canvas],
    elementRegistry: ['type', ElementRegistry$2],
    elementFactory: ['type', ElementFactory$1],
    eventBus: ['type', EventBus],
    graphicsFactory: ['type', GraphicsFactory]
  };

  /**
   * @typedef {import('didi').InjectionContext} InjectionContext
   * @typedef {import('didi').LocalsMap} LocalsMap
   * @typedef {import('didi').ModuleDeclaration} ModuleDeclaration
   *
   * @typedef { {
   *   modules?: ModuleDeclaration[];
   * } & Record<string, any> } DiagramOptions
   */

  /**
   * @template T
   * @typedef {import('didi').FactoryFunction<T>} FactoryFunction
   */

  /**
   * @template T
   * @typedef {import('didi').ArrayFunc<T>} ArrayFunc
   */

  /**
   * Bootstrap an injector from a list of modules, instantiating a number of default components
   *
   * @param {ModuleDeclaration[]} modules
   *
   * @return {Injector} a injector to use to access the components
   */
  function bootstrap(modules) {
    var injector = new Injector(modules);
    injector.init();
    return injector;
  }

  /**
   * Creates an injector from passed options.
   *
   * @template ServiceMap
   * @param {DiagramOptions} [options]
   *
   * @return {Injector<ServiceMap>}
   */
  function createInjector$2(options) {
    options = options || {};

    /**
     * @type { ModuleDeclaration }
     */
    var configModule = {
      'config': ['value', options]
    };
    var modules = [configModule, CoreModule$2].concat(options.modules || []);
    return bootstrap(modules);
  }

  /**
   * The main diagram-js entry point that bootstraps the diagram with the given
   * configuration.
   *
   * To register extensions with the diagram, pass them as Array<Module> to the constructor.
   *
   * @class
   * @constructor
   * @template [ServiceMap=null]
   *
   * @example Creating a plug-in that logs whenever a shape is added to the canvas.
   *
   * ```javascript
   * // plug-in implementation
   * function MyLoggingPlugin(eventBus) {
   *   eventBus.on('shape.added', function(event) {
   *     console.log('shape ', event.shape, ' was added to the diagram');
   *   });
   * }
   *
   * // export as module
   * export default {
   *   __init__: [ 'myLoggingPlugin' ],
   *     myLoggingPlugin: [ 'type', MyLoggingPlugin ]
   * };
   * ```
   *
   * Use the plug-in in a Diagram instance:
   *
   * ```javascript
   * import MyLoggingModule from 'path-to-my-logging-plugin';
   *
   * var diagram = new Diagram({
   *   modules: [
   *     MyLoggingModule
   *   ]
   * });
   *
   * diagram.invoke([ 'canvas', function(canvas) {
   *   // add shape to drawing canvas
   *   canvas.addShape({ x: 10, y: 10 });
   * });
   *
   * // 'shape ... was added to the diagram' logged to console
   * ```
   *
   * @param {DiagramOptions} [options]
   * @param {Injector<ServiceMap>} [injector] An (optional) injector to bootstrap the diagram with.
   */
  function Diagram(options, injector) {
    /**
     * @type {Injector<ServiceMap>}
     */
    this._injector = injector || createInjector$2(options);

    // init

    /**
     * An event indicating that all plug-ins are loaded.
     *
     * Use this event to fire other events to interested plug-ins
     *
     * @memberOf Diagram
     *
     * @event diagram.init
     *
     * @example
     *
     * ```javascript
     * eventBus.on('diagram.init', function() {
     *   eventBus.fire('my-custom-event', { foo: 'BAR' });
     * });
     * ```
     *
     * @type {Object}
     */
    this.get('eventBus').fire('diagram.init');
  }

  /**
   * @overlord
   *
   * Resolves a diagram service.
   *
   * @template T
   *
   * @param {string} name The name of the service to get.
   *
   * @return {T}
   */
  /**
   * @overlord
   *
   * Resolves a diagram service.
   *
   * @template T
   *
   * @param {string} name The name of the service to get.
   * @param {true} strict If false, resolve missing services to null.
   *
   * @return {T}
   */
  /**
   * @overlord
   *
   * Resolves a diagram service.
   *
   * @template T
   *
   * @param {string} name The name of the service to get.
   * @param {boolean} strict If false, resolve missing services to null.
   *
   * @return {T|null}
   */
  /**
   * Resolves a diagram service.
   *
   * @template {keyof ServiceMap} Name
   *
   * @param {Name} name The name of the service to get.
   *
   * @return {ServiceMap[Name]}
   */
  Diagram.prototype.get = function (name, strict) {
    return this._injector.get(name, strict);
  };

  /**
   * @overlord
   *
   * Invoke the given function, injecting dependencies. Return the result.
   *
   * @template T
   *
   * @param {FactoryFunction<T>} func
   * @param {InjectionContext} [context]
   * @param {LocalsMap} [locals]
   *
   * @return {T}
   */
  /**
   * Invoke the given function, injecting dependencies provided in
   * array notation. Return the result.
   *
   * @template T
   *
   * @param {ArrayFunc<T>} func function to be invoked
   * @param {InjectionContext} [context] context of the invocation
   * @param {LocalsMap} [locals] locals provided
   *
   * @return {T}
   */
  Diagram.prototype.invoke = function (func, context, locals) {
    return this._injector.invoke(func, context, locals);
  };

  /**
   * Destroys the diagram
   */
  Diagram.prototype.destroy = function () {
    this.get('eventBus').fire('diagram.destroy');
  };

  /**
   * Clear the diagram, removing all contents.
   */
  Diagram.prototype.clear = function () {
    this.get('eventBus').fire('diagram.clear');
  };

  /**
   * Is an element of the given DMN type?
   *
   * @param  {tjs.model.Base|ModdleElement} element
   * @param  {string} type
   *
   * @return {boolean}
   */
  function is(element, type) {
    var bo = getBusinessObject(element);
    return bo && typeof bo.$instanceOf === 'function' && bo.$instanceOf(type);
  }

  /**
   * Return the business object for a given element.
   *
   * @param  {tjs.model.Base|ModdleElement} element
   *
   * @return {ModdleElement}
   */
  function getBusinessObject(element) {
    return element && element.businessObject || element;
  }
  function getName(element) {
    return getBusinessObject(element).name;
  }

  /**
   * Return main boxed expression of a given decision or BKM.
   *
   * @param {ModdleElement} decisionOrBkm - the decision or business knowledge model
   * @returns {ModdleElement|undefined}
   */
  function getBoxedExpression(decisionOrBkm) {
    var bo = getBusinessObject(decisionOrBkm);
    if (is(bo, 'dmn:Decision')) {
      return bo.get('decisionLogic');
    } else if (is(bo, 'dmn:BusinessKnowledgeModel')) {
      var encapsulatedLogic = bo.get('encapsulatedLogic');
      return encapsulatedLogic && encapsulatedLogic.get('body');
    }
  }

  var diRefs = new Refs({
    name: 'dmnElementRef',
    enumerable: true
  }, {
    name: 'di',
    configurable: true
  });
  function DRDTreeWalker(handler, options) {
    // list of elements to handle deferred to ensure
    // prerequisites are drawn
    var deferred = [];
    function visit(element) {
      var gfx = element.gfx;

      // avoid multiple rendering of elements
      if (gfx) {
        throw new Error('already rendered ' + element.id);
      }

      // call handler
      return handler.element(element);
    }
    function visitRoot(element) {
      return handler.root(element);
    }
    function visitIfDi(element) {
      try {
        var gfx = element.di && visit(element);
        return gfx;
      } catch (e) {
        logError(e.message, {
          element: element,
          error: e
        });
      }
    }

    // Semantic handling //////////////////////

    /**
     * Handle definitions and return the rendered diagram (if any)
     *
     * @param {ModdleElement} definitions to walk and import
     * @param {ModdleElement} [diagram] specific diagram to import and display
     *
     * @throws {Error} if no diagram to display could be found
     */
    function handleDefinitions(definitions, diagram) {
      // make sure we walk the correct dmnElement
      var dmnDI = definitions.dmnDI;
      if (!dmnDI) {
        throw new Error('no dmndi:DMNDI');
      }
      var diagrams = dmnDI.diagrams || [];
      if (diagram && diagrams.indexOf(diagram) === -1) {
        throw new Error('diagram not part of dmndi:DMNDI');
      }
      if (!diagram && diagrams && diagrams.length) {
        diagram = diagrams[0];
      }

      // no diagram -> nothing to import
      if (!diagram) {
        throw new Error('no diagram to display');
      }

      // assign current diagram to definitions so that it can accessed later
      definitions.di = diagram;

      // load DI from selected diagram only
      handleDiagram(diagram);
      visitRoot(definitions);
      handleDrgElements(definitions.get('drgElement'));
      handleArtifacts(definitions.get('artifact'));
      handleDeferred();
    }
    function handleDrgElements(elements) {
      forEach$3(elements, function (element) {
        visitIfDi(element);
        handleRequirements(element);
      });
    }
    function handleArtifacts(elements) {
      forEach$3(elements, function (element) {
        if (is(element, 'dmn:Association')) {
          handleAssociation(element);
        } else {
          visitIfDi(element);
        }
      });
    }

    /**
     * Defer association visit until all shapes are visited.
     *
     * @param {ModdleElement} element
     */
    function handleAssociation(element) {
      defer(function () {
        visitIfDi(element);
      });
    }

    /**
     * Defer requirements visiting until all shapes are visited.
     *
     * @param {ModdleElement} element
     */
    function handleRequirements(element) {
      forEach$3(['informationRequirement', 'knowledgeRequirement', 'authorityRequirement'], function (requirements) {
        forEach$3(element[requirements], function (requirement) {
          defer(function () {
            visitIfDi(requirement);
          });
        });
      });
    }

    // DI handling //////////////////////
    function handleDiagram(diagram) {
      forEach$3(diagram.diagramElements, handleDiagramElement);
    }
    function handleDiagramElement(diagramElement) {
      registerDi(diagramElement);
    }
    function registerDi(di) {
      var dmnElement = di.dmnElementRef;
      if (dmnElement) {
        if (dmnElement.di) {
          logError('multiple DI elements defined for element', {
            element: dmnElement
          });
        } else {
          diRefs.bind(dmnElement, 'di');
          dmnElement.di = di;
        }
      } else {
        logError('no DMN element referenced in element', {
          element: di
        });
      }
    }
    function defer(fn) {
      deferred.push(fn);
    }
    function handleDeferred() {
      forEach$3(deferred, function (d) {
        d();
      });
    }
    function logError(message, context) {
      handler.error(message, context);
    }

    // API //////////////////////

    return {
      handleDefinitions: handleDefinitions
    };
  }

  /**
   * Import the definitions into a diagram.
   *
   * Errors and warnings are reported through the specified callback.
   *
   * @param  {Drd} drd
   * @param  {ModdleElement} definitions
   * @param  {Function} done
   *         the callback, invoked with (err, [ warning ]) once the import is done
   */
  function importDRD(drd, definitions, done) {
    var importer = drd.get('drdImporter'),
      eventBus = drd.get('eventBus');
    var error,
      warnings = [];
    function render(definitions) {
      var visitor = {
        root: function (element) {
          return importer.root(element);
        },
        element: function (element, di) {
          return importer.add(element, di);
        },
        error: function (message, context) {
          warnings.push({
            message: message,
            context: context
          });
        }
      };
      var walker = new DRDTreeWalker(visitor);

      // import
      walker.handleDefinitions(definitions);
    }
    eventBus.fire('import.start', {
      definitions: definitions
    });
    try {
      render(definitions);
    } catch (e) {
      error = e;
    }
    eventBus.fire('import.done', {
      error: error,
      warnings: warnings
    });
    done(error, warnings);
  }

  var NO_OP = '$NO_OP';
  var ERROR_MSG = 'a runtime error occured! Use Inferno in development environment to find the error.';
  var isBrowser = !!(typeof window !== 'undefined' && window.document);
  var isArray = Array.isArray;
  function isStringOrNumber(o) {
    var type = typeof o;
    return type === 'string' || type === 'number';
  }
  function isNullOrUndef(o) {
    return isUndefined(o) || isNull(o);
  }
  function isInvalid(o) {
    return isNull(o) || o === false || isTrue(o) || isUndefined(o);
  }
  function isFunction(o) {
    return typeof o === 'function';
  }
  function isString(o) {
    return typeof o === 'string';
  }
  function isNumber(o) {
    return typeof o === 'number';
  }
  function isNull(o) {
    return o === null;
  }
  function isTrue(o) {
    return o === true;
  }
  function isUndefined(o) {
    return o === void 0;
  }
  function isObject(o) {
    return typeof o === 'object';
  }
  function throwError(message) {
    if (!message) {
      message = ERROR_MSG;
    }
    throw new Error("Inferno Error: " + message);
  }
  function warning(message) {
    // tslint:disable-next-line:no-console
    console.error(message);
  }
  function combineFrom(first, second) {
    var out = {};
    if (first) {
      for (var key in first) {
        out[key] = first[key];
      }
    }
    if (second) {
      for (var _key in second) {
        out[_key] = second[_key];
      }
    }
    return out;
  }
  function getTagName(input) {
    var tagName;
    if (isArray(input)) {
      var arrayText = input.length > 3 ? input.slice(0, 3).toString() + ',...' : input.toString();
      tagName = 'Array(' + arrayText + ')';
    } else if (isStringOrNumber(input)) {
      tagName = 'Text(' + input + ')';
    } else if (isInvalid(input)) {
      tagName = 'InvalidVNode(' + input + ')';
    } else {
      var flags = input.flags;
      if (flags & 481 /* VNodeFlags.Element */) {
        tagName = "<" + input.type + (input.className ? ' class="' + input.className + '"' : '') + ">";
      } else if (flags & 16 /* VNodeFlags.Text */) {
        tagName = "Text(" + input.children + ")";
      } else if (flags & 1024 /* VNodeFlags.Portal */) {
        tagName = "Portal*";
      } else {
        var type = input.type;
        // Fallback for IE
        var componentName = type.name || type.displayName || type.constructor.name || (type.toString().match(/^function\s*([^\s(]+)/) || [])[1];
        tagName = "<" + componentName + " />";
      }
    }
    return '>> ' + tagName + '\n';
  }
  function DEV_ValidateKeys(vNodeTree, forceKeyed) {
    var foundKeys = {};
    for (var i = 0, len = vNodeTree.length; i < len; i++) {
      var childNode = vNodeTree[i];
      if (isArray(childNode)) {
        return 'Encountered ARRAY in mount, array must be flattened, or normalize used. Location: \n' + getTagName(childNode);
      }
      if (isInvalid(childNode)) {
        if (forceKeyed) {
          return 'Encountered invalid node when preparing to keyed algorithm. Location: \n' + getTagName(childNode);
        } else if (Object.keys(foundKeys).length !== 0) {
          return 'Encountered invalid node with mixed keys. Location: \n' + getTagName(childNode);
        }
        continue;
      }
      if (typeof childNode === 'object') {
        childNode.isValidated = true;
      }
      // Key can be undefined, null too. But typescript complains for no real reason
      var key = childNode.key;
      if (!isNullOrUndef(key) && !isStringOrNumber(key)) {
        return 'Encountered child vNode where key property is not string or number. Location: \n' + getTagName(childNode);
      }
      var children = childNode.children;
      var childFlags = childNode.childFlags;
      if (!isInvalid(children)) {
        var val = void 0;
        if (childFlags & 12 /* ChildFlags.MultipleChildren */) {
          val = DEV_ValidateKeys(children, childNode.childFlags & 8 /* ChildFlags.HasKeyedChildren */);
        } else if (childFlags === 2 /* ChildFlags.HasVNodeChildren */) {
          val = DEV_ValidateKeys([children], childNode.childFlags & 8 /* ChildFlags.HasKeyedChildren */);
        }
        if (val) {
          val += getTagName(childNode);
          return val;
        }
      }
      if (forceKeyed && isNullOrUndef(key)) {
        return 'Encountered child without key during keyed algorithm. If this error points to Array make sure children is flat list. Location: \n' + getTagName(childNode);
      } else if (!forceKeyed && isNullOrUndef(key)) {
        if (Object.keys(foundKeys).length !== 0) {
          return 'Encountered children with key missing. Location: \n' + getTagName(childNode);
        }
        continue;
      }
      if (foundKeys[key]) {
        return 'Encountered two children with same key: {' + key + '}. Location: \n' + getTagName(childNode);
      }
      foundKeys[key] = true;
    }
  }
  function validateVNodeElementChildren(vNode) {
    {
      if (vNode.childFlags & 1 /* ChildFlags.HasInvalidChildren */) {
        return;
      }
      if (vNode.flags & 64 /* VNodeFlags.InputElement */) {
        throwError("input elements can't have children.");
      }
      if (vNode.flags & 128 /* VNodeFlags.TextareaElement */) {
        throwError("textarea elements can't have children.");
      }
      if (vNode.flags & 481 /* VNodeFlags.Element */) {
        var voidTypes = ['area', 'base', 'br', 'col', 'command', 'embed', 'hr', 'img', 'input', 'keygen', 'link', 'meta', 'param', 'source', 'track', 'wbr'];
        var tag = vNode.type.toLowerCase();
        if (tag === 'media') {
          throwError("media elements can't have children.");
        }
        var idx = voidTypes.indexOf(tag);
        if (idx !== -1) {
          throwError(voidTypes[idx] + " elements can't have children.");
        }
      }
    }
  }
  function validateKeys(vNode) {
    {
      // Checks if there is any key missing or duplicate keys
      if (vNode.isValidated === false && vNode.children && vNode.flags & 481 /* VNodeFlags.Element */) {
        var error = DEV_ValidateKeys(Array.isArray(vNode.children) ? vNode.children : [vNode.children], (vNode.childFlags & 8 /* ChildFlags.HasKeyedChildren */) > 0);
        if (error) {
          throwError(error + getTagName(vNode));
        }
      }
      vNode.isValidated = true;
    }
  }
  var keyPrefix = '$';
  function getVNode(childFlags, children, className, flags, key, props, ref, type) {
    {
      return {
        childFlags: childFlags,
        children: children,
        className: className,
        dom: null,
        flags: flags,
        isValidated: false,
        key: key === void 0 ? null : key,
        parentVNode: null,
        props: props === void 0 ? null : props,
        ref: ref === void 0 ? null : ref,
        type: type
      };
    }
  }
  function createVNode(flags, type, className, children, childFlags, props, key, ref) {
    {
      if (flags & 14 /* VNodeFlags.Component */) {
        throwError('Creating Component vNodes using createVNode is not allowed. Use Inferno.createComponentVNode method.');
      }
    }
    var childFlag = childFlags === void 0 ? 1 /* ChildFlags.HasInvalidChildren */ : childFlags;
    var vNode = getVNode(childFlag, children, className, flags, key, props, ref, type);
    if (childFlag === 0 /* ChildFlags.UnknownChildren */) {
      normalizeChildren(vNode, vNode.children);
    }
    {
      validateVNodeElementChildren(vNode);
    }
    return vNode;
  }
  function createComponentVNode(flags, type, props, key, ref) {
    {
      if (flags & 1 /* VNodeFlags.HtmlElement */) {
        throwError('Creating element vNodes using createComponentVNode is not allowed. Use Inferno.createVNode method.');
      }
    }
    if ((flags & 2 /* VNodeFlags.ComponentUnknown */) > 0) {
      flags = type.prototype && isFunction(type.prototype.render) ? 4 /* VNodeFlags.ComponentClass */ : 8 /* VNodeFlags.ComponentFunction */;
    }
    // set default props
    var defaultProps = type.defaultProps;
    if (!isNullOrUndef(defaultProps)) {
      if (!props) {
        props = {}; // Props can be referenced and modified at application level so always create new object
      }
      for (var prop in defaultProps) {
        if (isUndefined(props[prop])) {
          props[prop] = defaultProps[prop];
        }
      }
    }
    if ((flags & 8 /* VNodeFlags.ComponentFunction */) > 0) {
      var defaultHooks = type.defaultHooks;
      if (!isNullOrUndef(defaultHooks)) {
        if (!ref) {
          // As ref cannot be referenced from application level, we can use the same refs object
          ref = defaultHooks;
        } else {
          for (var _prop in defaultHooks) {
            if (isUndefined(ref[_prop])) {
              ref[_prop] = defaultHooks[_prop];
            }
          }
        }
      }
    }
    var vNode = getVNode(1 /* ChildFlags.HasInvalidChildren */, null, null, flags, key, props, ref, type);
    var optsVNode = options.createVNode;
    if (isFunction(optsVNode)) {
      optsVNode(vNode);
    }
    return vNode;
  }
  function createTextVNode(text, key) {
    return getVNode(1 /* ChildFlags.HasInvalidChildren */, isNullOrUndef(text) ? '' : text, null, 16 /* VNodeFlags.Text */, key, null, null, null);
  }
  function normalizeProps(vNode) {
    var props = vNode.props;
    if (props) {
      var flags = vNode.flags;
      if (flags & 481 /* VNodeFlags.Element */) {
        if (props.children !== void 0 && isNullOrUndef(vNode.children)) {
          normalizeChildren(vNode, props.children);
        }
        if (props.className !== void 0) {
          vNode.className = props.className || null;
          props.className = undefined;
        }
      }
      if (props.key !== void 0) {
        vNode.key = props.key;
        props.key = undefined;
      }
      if (props.ref !== void 0) {
        if (flags & 8 /* VNodeFlags.ComponentFunction */) {
          vNode.ref = combineFrom(vNode.ref, props.ref);
        } else {
          vNode.ref = props.ref;
        }
        props.ref = undefined;
      }
    }
    return vNode;
  }
  function directClone(vNodeToClone) {
    var newVNode;
    var flags = vNodeToClone.flags;
    if (flags & 14 /* VNodeFlags.Component */) {
      var props;
      var propsToClone = vNodeToClone.props;
      if (!isNull(propsToClone)) {
        props = {};
        for (var key in propsToClone) {
          props[key] = propsToClone[key];
        }
      }
      newVNode = createComponentVNode(flags, vNodeToClone.type, props, vNodeToClone.key, vNodeToClone.ref);
    } else if (flags & 481 /* VNodeFlags.Element */) {
      newVNode = createVNode(flags, vNodeToClone.type, vNodeToClone.className, vNodeToClone.children, vNodeToClone.childFlags, vNodeToClone.props, vNodeToClone.key, vNodeToClone.ref);
    } else if (flags & 16 /* VNodeFlags.Text */) {
      newVNode = createTextVNode(vNodeToClone.children, vNodeToClone.key);
    } else if (flags & 1024 /* VNodeFlags.Portal */) {
      newVNode = vNodeToClone;
    }
    return newVNode;
  }
  function createVoidVNode() {
    return createTextVNode('', null);
  }
  function _normalizeVNodes(nodes, result, index, currentKey) {
    for (var len = nodes.length; index < len; index++) {
      var n = nodes[index];
      if (!isInvalid(n)) {
        var newKey = currentKey + keyPrefix + index;
        if (isArray(n)) {
          _normalizeVNodes(n, result, 0, newKey);
        } else {
          if (isStringOrNumber(n)) {
            n = createTextVNode(n, newKey);
          } else {
            var oldKey = n.key;
            var isPrefixedKey = isString(oldKey) && oldKey[0] === keyPrefix;
            if (!isNull(n.dom) || isPrefixedKey) {
              n = directClone(n);
            }
            if (isNull(oldKey) || isPrefixedKey) {
              n.key = newKey;
            } else {
              n.key = currentKey + oldKey;
            }
          }
          result.push(n);
        }
      }
    }
  }
  function normalizeChildren(vNode, children) {
    var newChildren;
    var newChildFlags = 1 /* ChildFlags.HasInvalidChildren */;
    // Don't change children to match strict equal (===) true in patching
    if (isInvalid(children)) {
      newChildren = children;
    } else if (isString(children)) {
      newChildFlags = 2 /* ChildFlags.HasVNodeChildren */;
      newChildren = createTextVNode(children);
    } else if (isNumber(children)) {
      newChildFlags = 2 /* ChildFlags.HasVNodeChildren */;
      newChildren = createTextVNode(children + '');
    } else if (isArray(children)) {
      var len = children.length;
      if (len === 0) {
        newChildren = null;
        newChildFlags = 1 /* ChildFlags.HasInvalidChildren */;
      } else {
        // we assign $ which basically means we've flagged this array for future note
        // if it comes back again, we need to clone it, as people are using it
        // in an immutable way
        // tslint:disable-next-line
        if (Object.isFrozen(children) || children['$'] === true) {
          children = children.slice();
        }
        newChildFlags = 8 /* ChildFlags.HasKeyedChildren */;
        for (var i = 0; i < len; i++) {
          var n = children[i];
          if (isInvalid(n) || isArray(n)) {
            newChildren = newChildren || children.slice(0, i);
            _normalizeVNodes(children, newChildren, i, '');
            break;
          } else if (isStringOrNumber(n)) {
            newChildren = newChildren || children.slice(0, i);
            newChildren.push(createTextVNode(n, keyPrefix + i));
          } else {
            var key = n.key;
            var isNullDom = isNull(n.dom);
            var isNullKey = isNull(key);
            var isPrefixed = !isNullKey && isString(key) && key[0] === keyPrefix;
            if (!isNullDom || isNullKey || isPrefixed) {
              newChildren = newChildren || children.slice(0, i);
              if (!isNullDom || isPrefixed) {
                n = directClone(n);
              }
              if (isNullKey || isPrefixed) {
                n.key = keyPrefix + i;
              }
              newChildren.push(n);
            } else if (newChildren) {
              newChildren.push(n);
            }
          }
        }
        newChildren = newChildren || children;
        newChildren.$ = true;
      }
    } else {
      newChildren = children;
      if (!isNull(children.dom)) {
        newChildren = directClone(children);
      }
      newChildFlags = 2 /* ChildFlags.HasVNodeChildren */;
    }
    vNode.children = newChildren;
    vNode.childFlags = newChildFlags;
    {
      validateVNodeElementChildren(vNode);
    }
    return vNode;
  }
  var options = {
    afterRender: null,
    beforeRender: null,
    createVNode: null,
    renderComplete: null
  };
  var xlinkNS = 'http://www.w3.org/1999/xlink';
  var xmlNS = 'http://www.w3.org/XML/1998/namespace';
  var svgNS = 'http://www.w3.org/2000/svg';
  var namespaces = {
    'xlink:actuate': xlinkNS,
    'xlink:arcrole': xlinkNS,
    'xlink:href': xlinkNS,
    'xlink:role': xlinkNS,
    'xlink:show': xlinkNS,
    'xlink:title': xlinkNS,
    'xlink:type': xlinkNS,
    'xml:base': xmlNS,
    'xml:lang': xmlNS,
    'xml:space': xmlNS
  };

  // We need EMPTY_OBJ defined in one place.
  // Its used for comparison so we cant inline it into shared
  var EMPTY_OBJ = {};
  var LIFECYCLE = [];
  {
    Object.freeze(EMPTY_OBJ);
  }
  function appendChild(parentDom, dom) {
    parentDom.appendChild(dom);
  }
  function insertOrAppend(parentDom, newNode, nextNode) {
    if (isNullOrUndef(nextNode)) {
      appendChild(parentDom, newNode);
    } else {
      parentDom.insertBefore(newNode, nextNode);
    }
  }
  function documentCreateElement(tag, isSVG) {
    if (isSVG) {
      return document.createElementNS(svgNS, tag);
    }
    return document.createElement(tag);
  }
  function replaceChild(parentDom, newDom, lastDom) {
    parentDom.replaceChild(newDom, lastDom);
  }
  function removeChild(parentDom, dom) {
    parentDom.removeChild(dom);
  }
  function callAll(arrayFn) {
    var listener;
    while ((listener = arrayFn.shift()) !== undefined) {
      listener();
    }
  }
  var attachedEventCounts = {};
  var attachedEvents = {};
  function handleEvent(name, nextEvent, dom) {
    var eventsLeft = attachedEventCounts[name];
    var eventsObject = dom.$EV;
    if (nextEvent) {
      if (!eventsLeft) {
        attachedEvents[name] = attachEventToDocument(name);
        attachedEventCounts[name] = 0;
      }
      if (!eventsObject) {
        eventsObject = dom.$EV = {};
      }
      if (!eventsObject[name]) {
        attachedEventCounts[name]++;
      }
      eventsObject[name] = nextEvent;
    } else if (eventsObject && eventsObject[name]) {
      attachedEventCounts[name]--;
      if (eventsLeft === 1) {
        document.removeEventListener(normalizeEventName(name), attachedEvents[name]);
        attachedEvents[name] = null;
      }
      eventsObject[name] = nextEvent;
    }
  }
  // When browsers fully support event.composedPath we could loop it through instead of using parentNode property
  function getTargetNode(event) {
    return isFunction(event.composedPath) ? event.composedPath()[0] : event.target;
  }
  function dispatchEvents(event, isClick, name, eventData) {
    var dom = getTargetNode(event);
    while (!isNull(dom)) {
      // Html Nodes can be nested fe: span inside button in that scenario browser does not handle disabled attribute on parent,
      // because the event listener is on document.body
      // Don't process clicks on disabled elements
      if (isClick && dom.disabled) {
        return;
      }
      var eventsObject = dom.$EV;
      if (eventsObject) {
        var currentEvent = eventsObject[name];
        if (currentEvent) {
          // linkEvent object
          eventData.dom = dom;
          if (currentEvent.event) {
            currentEvent.event(currentEvent.data, event);
          } else {
            currentEvent(event);
          }
          if (event.cancelBubble) {
            return;
          }
        }
      }
      dom = dom.parentNode;
    }
  }
  function normalizeEventName(name) {
    return name.substr(2).toLowerCase();
  }
  function stopPropagation$1() {
    this.cancelBubble = true;
    if (!this.immediatePropagationStopped) {
      this.stopImmediatePropagation();
    }
  }
  function attachEventToDocument(name) {
    var docEvent = function docEvent(event) {
      var type = event.type;
      var isClick = type === 'click' || type === 'dblclick';
      if (isClick && event.button !== 0) {
        // Firefox incorrectly triggers click event for mid/right mouse buttons.
        // This bug has been active for 12 years.
        // https://bugzilla.mozilla.org/show_bug.cgi?id=184051
        event.stopPropagation();
        return false;
      }
      event.stopPropagation = stopPropagation$1;
      // Event data needs to be object to save reference to currentTarget getter
      var eventData = {
        dom: document
      };
      Object.defineProperty(event, 'currentTarget', {
        configurable: true,
        get: function get() {
          return eventData.dom;
        }
      });
      dispatchEvents(event, isClick, name, eventData);
      return;
    };
    document.addEventListener(normalizeEventName(name), docEvent);
    return docEvent;
  }
  function isSameInnerHTML(dom, innerHTML) {
    var tempdom = document.createElement('i');
    tempdom.innerHTML = innerHTML;
    return tempdom.innerHTML === dom.innerHTML;
  }
  function isSamePropsInnerHTML(dom, props) {
    return Boolean(props && props.dangerouslySetInnerHTML && props.dangerouslySetInnerHTML.__html && isSameInnerHTML(dom, props.dangerouslySetInnerHTML.__html));
  }
  function triggerEventListener(props, methodName, e) {
    if (props[methodName]) {
      var listener = props[methodName];
      if (listener.event) {
        listener.event(listener.data, e);
      } else {
        listener(e);
      }
    } else {
      var nativeListenerName = methodName.toLowerCase();
      if (props[nativeListenerName]) {
        props[nativeListenerName](e);
      }
    }
  }
  function createWrappedFunction(methodName, applyValue) {
    var fnMethod = function fnMethod(e) {
      e.stopPropagation();
      var vNode = this.$V;
      // If vNode is gone by the time event fires, no-op
      if (!vNode) {
        return;
      }
      var props = vNode.props || EMPTY_OBJ;
      var dom = vNode.dom;
      if (isString(methodName)) {
        triggerEventListener(props, methodName, e);
      } else {
        for (var i = 0; i < methodName.length; i++) {
          triggerEventListener(props, methodName[i], e);
        }
      }
      if (isFunction(applyValue)) {
        var newVNode = this.$V;
        var newProps = newVNode.props || EMPTY_OBJ;
        applyValue(newProps, dom, false, newVNode);
      }
    };
    Object.defineProperty(fnMethod, 'wrapped', {
      configurable: false,
      enumerable: false,
      value: true,
      writable: false
    });
    return fnMethod;
  }
  function isCheckedType(type) {
    return type === 'checkbox' || type === 'radio';
  }
  var onTextInputChange = createWrappedFunction('onInput', applyValueInput);
  var wrappedOnChange$1 = createWrappedFunction(['onClick', 'onChange'], applyValueInput);
  /* tslint:disable-next-line:no-empty */
  function emptywrapper(event) {
    event.stopPropagation();
  }
  emptywrapper.wrapped = true;
  function inputEvents(dom, nextPropsOrEmpty) {
    if (isCheckedType(nextPropsOrEmpty.type)) {
      dom.onchange = wrappedOnChange$1;
      dom.onclick = emptywrapper;
    } else {
      dom.oninput = onTextInputChange;
    }
  }
  function applyValueInput(nextPropsOrEmpty, dom) {
    var type = nextPropsOrEmpty.type;
    var value = nextPropsOrEmpty.value;
    var checked = nextPropsOrEmpty.checked;
    var multiple = nextPropsOrEmpty.multiple;
    var defaultValue = nextPropsOrEmpty.defaultValue;
    var hasValue = !isNullOrUndef(value);
    if (type && type !== dom.type) {
      dom.setAttribute('type', type);
    }
    if (!isNullOrUndef(multiple) && multiple !== dom.multiple) {
      dom.multiple = multiple;
    }
    if (!isNullOrUndef(defaultValue) && !hasValue) {
      dom.defaultValue = defaultValue + '';
    }
    if (isCheckedType(type)) {
      if (hasValue) {
        dom.value = value;
      }
      if (!isNullOrUndef(checked)) {
        dom.checked = checked;
      }
    } else {
      if (hasValue && dom.value !== value) {
        dom.defaultValue = value;
        dom.value = value;
      } else if (!isNullOrUndef(checked)) {
        dom.checked = checked;
      }
    }
  }
  function updateChildOptionGroup(vNode, value) {
    var type = vNode.type;
    if (type === 'optgroup') {
      var children = vNode.children;
      var childFlags = vNode.childFlags;
      if (childFlags & 12 /* ChildFlags.MultipleChildren */) {
        for (var i = 0, len = children.length; i < len; i++) {
          updateChildOption(children[i], value);
        }
      } else if (childFlags === 2 /* ChildFlags.HasVNodeChildren */) {
        updateChildOption(children, value);
      }
    } else {
      updateChildOption(vNode, value);
    }
  }
  function updateChildOption(vNode, value) {
    var props = vNode.props || EMPTY_OBJ;
    var dom = vNode.dom;
    // we do this as multiple may have changed
    dom.value = props.value;
    if (isArray(value) && value.indexOf(props.value) !== -1 || props.value === value) {
      dom.selected = true;
    } else if (!isNullOrUndef(value) || !isNullOrUndef(props.selected)) {
      dom.selected = props.selected || false;
    }
  }
  var onSelectChange = createWrappedFunction('onChange', applyValueSelect);
  function selectEvents(dom) {
    dom.onchange = onSelectChange;
  }
  function applyValueSelect(nextPropsOrEmpty, dom, mounting, vNode) {
    var multiplePropInBoolean = Boolean(nextPropsOrEmpty.multiple);
    if (!isNullOrUndef(nextPropsOrEmpty.multiple) && multiplePropInBoolean !== dom.multiple) {
      dom.multiple = multiplePropInBoolean;
    }
    var childFlags = vNode.childFlags;
    if ((childFlags & 1 /* ChildFlags.HasInvalidChildren */) === 0) {
      var children = vNode.children;
      var value = nextPropsOrEmpty.value;
      if (mounting && isNullOrUndef(value)) {
        value = nextPropsOrEmpty.defaultValue;
      }
      if (childFlags & 12 /* ChildFlags.MultipleChildren */) {
        for (var i = 0, len = children.length; i < len; i++) {
          updateChildOptionGroup(children[i], value);
        }
      } else if (childFlags === 2 /* ChildFlags.HasVNodeChildren */) {
        updateChildOptionGroup(children, value);
      }
    }
  }
  var onTextareaInputChange = createWrappedFunction('onInput', applyValueTextArea);
  var wrappedOnChange = createWrappedFunction('onChange');
  function textAreaEvents(dom, nextPropsOrEmpty) {
    dom.oninput = onTextareaInputChange;
    if (nextPropsOrEmpty.onChange) {
      dom.onchange = wrappedOnChange;
    }
  }
  function applyValueTextArea(nextPropsOrEmpty, dom, mounting) {
    var value = nextPropsOrEmpty.value;
    var domValue = dom.value;
    if (isNullOrUndef(value)) {
      if (mounting) {
        var defaultValue = nextPropsOrEmpty.defaultValue;
        if (!isNullOrUndef(defaultValue) && defaultValue !== domValue) {
          dom.defaultValue = defaultValue;
          dom.value = defaultValue;
        }
      }
    } else if (domValue !== value) {
      /* There is value so keep it controlled */
      dom.defaultValue = value;
      dom.value = value;
    }
  }

  /**
   * There is currently no support for switching same input between controlled and nonControlled
   * If that ever becomes a real issue, then re design controlled elements
   * Currently user must choose either controlled or non-controlled and stick with that
   */
  function processElement(flags, vNode, dom, nextPropsOrEmpty, mounting, isControlled) {
    if (flags & 64 /* VNodeFlags.InputElement */) {
      applyValueInput(nextPropsOrEmpty, dom);
    } else if (flags & 256 /* VNodeFlags.SelectElement */) {
      applyValueSelect(nextPropsOrEmpty, dom, mounting, vNode);
    } else if (flags & 128 /* VNodeFlags.TextareaElement */) {
      applyValueTextArea(nextPropsOrEmpty, dom, mounting);
    }
    if (isControlled) {
      dom.$V = vNode;
    }
  }
  function addFormElementEventHandlers(flags, dom, nextPropsOrEmpty) {
    if (flags & 64 /* VNodeFlags.InputElement */) {
      inputEvents(dom, nextPropsOrEmpty);
    } else if (flags & 256 /* VNodeFlags.SelectElement */) {
      selectEvents(dom);
    } else if (flags & 128 /* VNodeFlags.TextareaElement */) {
      textAreaEvents(dom, nextPropsOrEmpty);
    }
  }
  function isControlledFormElement(nextPropsOrEmpty) {
    return nextPropsOrEmpty.type && isCheckedType(nextPropsOrEmpty.type) ? !isNullOrUndef(nextPropsOrEmpty.checked) : !isNullOrUndef(nextPropsOrEmpty.value);
  }
  function remove$1(vNode, parentDom) {
    unmount(vNode);
    if (parentDom && vNode.dom) {
      removeChild(parentDom, vNode.dom);
      // Let carbage collector free memory
      vNode.dom = null;
    }
  }
  function unmount(vNode) {
    var flags = vNode.flags;
    if (flags & 481 /* VNodeFlags.Element */) {
      var ref = vNode.ref;
      var props = vNode.props;
      if (isFunction(ref)) {
        ref(null);
      }
      var children = vNode.children;
      var childFlags = vNode.childFlags;
      if (childFlags & 12 /* ChildFlags.MultipleChildren */) {
        unmountAllChildren(children);
      } else if (childFlags === 2 /* ChildFlags.HasVNodeChildren */) {
        unmount(children);
      }
      if (!isNull(props)) {
        for (var name in props) {
          switch (name) {
            case 'onClick':
            case 'onDblClick':
            case 'onFocusIn':
            case 'onFocusOut':
            case 'onKeyDown':
            case 'onKeyPress':
            case 'onKeyUp':
            case 'onMouseDown':
            case 'onMouseMove':
            case 'onMouseUp':
            case 'onSubmit':
            case 'onTouchEnd':
            case 'onTouchMove':
            case 'onTouchStart':
              handleEvent(name, null, vNode.dom);
              break;
          }
        }
      }
    } else {
      var _children = vNode.children;
      // Safe guard for crashed VNode
      if (_children) {
        if (flags & 14 /* VNodeFlags.Component */) {
          var _ref = vNode.ref;
          if (flags & 4 /* VNodeFlags.ComponentClass */) {
            if (isFunction(_children.componentWillUnmount)) {
              _children.componentWillUnmount();
            }
            if (isFunction(_ref)) {
              _ref(null);
            }
            _children.$UN = true;
            if (_children.$LI) {
              unmount(_children.$LI);
            }
          } else {
            if (!isNullOrUndef(_ref) && isFunction(_ref.onComponentWillUnmount)) {
              _ref.onComponentWillUnmount(vNode.dom, vNode.props || EMPTY_OBJ);
            }
            unmount(_children);
          }
        } else if (flags & 1024 /* VNodeFlags.Portal */) {
          remove$1(_children, vNode.type);
        }
      }
    }
  }
  function unmountAllChildren(children) {
    for (var i = 0, len = children.length; i < len; i++) {
      unmount(children[i]);
    }
  }
  function removeAllChildren(dom, children) {
    unmountAllChildren(children);
    dom.textContent = '';
  }
  function createLinkEvent(linkEvent, nextValue) {
    return function (e) {
      linkEvent(nextValue.data, e);
    };
  }
  function patchEvent(name, nextValue, dom) {
    var nameLowerCase = name.toLowerCase();
    if (!isFunction(nextValue) && !isNullOrUndef(nextValue)) {
      var linkEvent = nextValue.event;
      if (linkEvent && isFunction(linkEvent)) {
        dom[nameLowerCase] = createLinkEvent(linkEvent, nextValue);
      } else {
        // Development warning
        {
          throwError("an event on a VNode \"" + name + "\". was not a function or a valid linkEvent.");
        }
      }
    } else {
      var domEvent = dom[nameLowerCase];
      // if the function is wrapped, that means it's been controlled by a wrapper
      if (!domEvent || !domEvent.wrapped) {
        dom[nameLowerCase] = nextValue;
      }
    }
  }
  function getNumberStyleValue(style, value) {
    switch (style) {
      case 'animationIterationCount':
      case 'borderImageOutset':
      case 'borderImageSlice':
      case 'borderImageWidth':
      case 'boxFlex':
      case 'boxFlexGroup':
      case 'boxOrdinalGroup':
      case 'columnCount':
      case 'fillOpacity':
      case 'flex':
      case 'flexGrow':
      case 'flexNegative':
      case 'flexOrder':
      case 'flexPositive':
      case 'flexShrink':
      case 'floodOpacity':
      case 'fontWeight':
      case 'gridColumn':
      case 'gridRow':
      case 'lineClamp':
      case 'lineHeight':
      case 'opacity':
      case 'order':
      case 'orphans':
      case 'stopOpacity':
      case 'strokeDasharray':
      case 'strokeDashoffset':
      case 'strokeMiterlimit':
      case 'strokeOpacity':
      case 'strokeWidth':
      case 'tabSize':
      case 'widows':
      case 'zIndex':
      case 'zoom':
        return value;
      default:
        return value + 'px';
    }
  }
  // We are assuming here that we come from patchProp routine
  // -nextAttrValue cannot be null or undefined
  function patchStyle(lastAttrValue, nextAttrValue, dom) {
    var domStyle = dom.style;
    var style;
    var value;
    if (isString(nextAttrValue)) {
      domStyle.cssText = nextAttrValue;
      return;
    }
    if (!isNullOrUndef(lastAttrValue) && !isString(lastAttrValue)) {
      for (style in nextAttrValue) {
        // do not add a hasOwnProperty check here, it affects performance
        value = nextAttrValue[style];
        if (value !== lastAttrValue[style]) {
          domStyle[style] = isNumber(value) ? getNumberStyleValue(style, value) : value;
        }
      }
      for (style in lastAttrValue) {
        if (isNullOrUndef(nextAttrValue[style])) {
          domStyle[style] = '';
        }
      }
    } else {
      for (style in nextAttrValue) {
        value = nextAttrValue[style];
        domStyle[style] = isNumber(value) ? getNumberStyleValue(style, value) : value;
      }
    }
  }
  function patchProp(prop, lastValue, nextValue, dom, isSVG, hasControlledValue, lastVNode) {
    switch (prop) {
      case 'onClick':
      case 'onDblClick':
      case 'onFocusIn':
      case 'onFocusOut':
      case 'onKeyDown':
      case 'onKeyPress':
      case 'onKeyUp':
      case 'onMouseDown':
      case 'onMouseMove':
      case 'onMouseUp':
      case 'onSubmit':
      case 'onTouchEnd':
      case 'onTouchMove':
      case 'onTouchStart':
        handleEvent(prop, nextValue, dom);
        break;
      case 'children':
      case 'childrenType':
      case 'className':
      case 'defaultValue':
      case 'key':
      case 'multiple':
      case 'ref':
        break;
      case 'autoFocus':
        dom.autofocus = !!nextValue;
        break;
      case 'allowfullscreen':
      case 'autoplay':
      case 'capture':
      case 'checked':
      case 'controls':
      case 'default':
      case 'disabled':
      case 'hidden':
      case 'indeterminate':
      case 'loop':
      case 'muted':
      case 'novalidate':
      case 'open':
      case 'readOnly':
      case 'required':
      case 'reversed':
      case 'scoped':
      case 'seamless':
      case 'selected':
        dom[prop] = !!nextValue;
        break;
      case 'defaultChecked':
      case 'value':
      case 'volume':
        if (hasControlledValue && prop === 'value') {
          return;
        }
        var value = isNullOrUndef(nextValue) ? '' : nextValue;
        if (dom[prop] !== value) {
          dom[prop] = value;
        }
        break;
      case 'dangerouslySetInnerHTML':
        var lastHtml = lastValue && lastValue.__html || '';
        var nextHtml = nextValue && nextValue.__html || '';
        if (lastHtml !== nextHtml) {
          if (!isNullOrUndef(nextHtml) && !isSameInnerHTML(dom, nextHtml)) {
            if (!isNull(lastVNode)) {
              if (lastVNode.childFlags & 12 /* ChildFlags.MultipleChildren */) {
                unmountAllChildren(lastVNode.children);
              } else if (lastVNode.childFlags === 2 /* ChildFlags.HasVNodeChildren */) {
                unmount(lastVNode.children);
              }
              lastVNode.children = null;
              lastVNode.childFlags = 1 /* ChildFlags.HasInvalidChildren */;
            }
            dom.innerHTML = nextHtml;
          }
        }
        break;
      default:
        if (prop[0] === 'o' && prop[1] === 'n') {
          patchEvent(prop, nextValue, dom);
        } else if (isNullOrUndef(nextValue)) {
          dom.removeAttribute(prop);
        } else if (prop === 'style') {
          patchStyle(lastValue, nextValue, dom);
        } else if (isSVG && namespaces[prop]) {
          // We optimize for isSVG being false
          // If we end up in this path we can read property again
          dom.setAttributeNS(namespaces[prop], prop, nextValue);
        } else {
          dom.setAttribute(prop, nextValue);
        }
        break;
    }
  }
  function mountProps(vNode, flags, props, dom, isSVG) {
    var hasControlledValue = false;
    var isFormElement = (flags & 448 /* VNodeFlags.FormElement */) > 0;
    if (isFormElement) {
      hasControlledValue = isControlledFormElement(props);
      if (hasControlledValue) {
        addFormElementEventHandlers(flags, dom, props);
      }
    }
    for (var prop in props) {
      // do not add a hasOwnProperty check here, it affects performance
      patchProp(prop, null, props[prop], dom, isSVG, hasControlledValue, null);
    }
    if (isFormElement) {
      processElement(flags, vNode, dom, props, true, hasControlledValue);
    }
  }
  function createClassComponentInstance(vNode, Component, props, context) {
    var instance = new Component(props, context);
    vNode.children = instance;
    instance.$V = vNode;
    instance.$BS = false;
    instance.context = context;
    if (instance.props === EMPTY_OBJ) {
      instance.props = props;
    }
    instance.$UN = false;
    if (isFunction(instance.componentWillMount)) {
      instance.$BR = true;
      instance.componentWillMount();
      if (instance.$PSS) {
        var state = instance.state;
        var pending = instance.$PS;
        if (isNull(state)) {
          instance.state = pending;
        } else {
          for (var key in pending) {
            state[key] = pending[key];
          }
        }
        instance.$PSS = false;
        instance.$PS = null;
      }
      instance.$BR = false;
    }
    if (isFunction(options.beforeRender)) {
      options.beforeRender(instance);
    }
    var input = handleComponentInput(instance.render(props, instance.state, context), vNode);
    var childContext;
    if (isFunction(instance.getChildContext)) {
      childContext = instance.getChildContext();
    }
    if (isNullOrUndef(childContext)) {
      instance.$CX = context;
    } else {
      instance.$CX = combineFrom(context, childContext);
    }
    if (isFunction(options.afterRender)) {
      options.afterRender(instance);
    }
    instance.$LI = input;
    return instance;
  }
  function handleComponentInput(input, componentVNode) {
    // Development validation
    {
      if (isArray(input)) {
        throwError('a valid Inferno VNode (or null) must be returned from a component render. You may have returned an array or an invalid object.');
      }
    }
    if (isInvalid(input)) {
      input = createVoidVNode();
    } else if (isStringOrNumber(input)) {
      input = createTextVNode(input, null);
    } else {
      if (input.dom) {
        input = directClone(input);
      }
      if (input.flags & 14 /* VNodeFlags.Component */) {
        // if we have an input that is also a component, we run into a tricky situation
        // where the root vNode needs to always have the correct DOM entry
        // we can optimise this in the future, but this gets us out of a lot of issues
        input.parentVNode = componentVNode;
      }
    }
    return input;
  }
  function mount(vNode, parentDom, context, isSVG) {
    var flags = vNode.flags;
    if (flags & 481 /* VNodeFlags.Element */) {
      return mountElement(vNode, parentDom, context, isSVG);
    }
    if (flags & 14 /* VNodeFlags.Component */) {
      return mountComponent(vNode, parentDom, context, isSVG, (flags & 4 /* VNodeFlags.ComponentClass */) > 0);
    }
    if (flags & 512 /* VNodeFlags.Void */ || flags & 16 /* VNodeFlags.Text */) {
      return mountText(vNode, parentDom);
    }
    if (flags & 1024 /* VNodeFlags.Portal */) {
      mount(vNode.children, vNode.type, context, false);
      return vNode.dom = mountText(createVoidVNode(), parentDom);
    }
    // Development validation, in production we don't need to throw because it crashes anyway
    {
      if (typeof vNode === 'object') {
        throwError("mount() received an object that's not a valid VNode, you should stringify it first, fix createVNode flags or call normalizeChildren. Object: \"" + JSON.stringify(vNode) + "\".");
      } else {
        throwError("mount() expects a valid VNode, instead it received an object with the type \"" + typeof vNode + "\".");
      }
    }
  }
  function mountText(vNode, parentDom) {
    var dom = vNode.dom = document.createTextNode(vNode.children);
    if (!isNull(parentDom)) {
      appendChild(parentDom, dom);
    }
    return dom;
  }
  function mountElement(vNode, parentDom, context, isSVG) {
    var flags = vNode.flags;
    var children = vNode.children;
    var props = vNode.props;
    var className = vNode.className;
    var ref = vNode.ref;
    var childFlags = vNode.childFlags;
    isSVG = isSVG || (flags & 32 /* VNodeFlags.SvgElement */) > 0;
    var dom = documentCreateElement(vNode.type, isSVG);
    vNode.dom = dom;
    if (!isNullOrUndef(className) && className !== '') {
      if (isSVG) {
        dom.setAttribute('class', className);
      } else {
        dom.className = className;
      }
    }
    {
      validateKeys(vNode);
    }
    if (!isNull(parentDom)) {
      appendChild(parentDom, dom);
    }
    if ((childFlags & 1 /* ChildFlags.HasInvalidChildren */) === 0) {
      var childrenIsSVG = isSVG === true && vNode.type !== 'foreignObject';
      if (childFlags === 2 /* ChildFlags.HasVNodeChildren */) {
        mount(children, dom, context, childrenIsSVG);
      } else if (childFlags & 12 /* ChildFlags.MultipleChildren */) {
        mountArrayChildren(children, dom, context, childrenIsSVG);
      }
    }
    if (!isNull(props)) {
      mountProps(vNode, flags, props, dom, isSVG);
    }
    {
      if (isString(ref)) {
        throwError('string "refs" are not supported in Inferno 1.0. Use callback "refs" instead.');
      }
    }
    if (isFunction(ref)) {
      mountRef(dom, ref);
    }
    return dom;
  }
  function mountArrayChildren(children, dom, context, isSVG) {
    for (var i = 0, len = children.length; i < len; i++) {
      var child = children[i];
      if (!isNull(child.dom)) {
        children[i] = child = directClone(child);
      }
      mount(child, dom, context, isSVG);
    }
  }
  function mountComponent(vNode, parentDom, context, isSVG, isClass) {
    var dom;
    var type = vNode.type;
    var props = vNode.props || EMPTY_OBJ;
    var ref = vNode.ref;
    if (isClass) {
      var instance = createClassComponentInstance(vNode, type, props, context);
      vNode.dom = dom = mount(instance.$LI, null, instance.$CX, isSVG);
      mountClassComponentCallbacks(vNode, ref, instance);
      instance.$UPD = false;
    } else {
      var input = handleComponentInput(type(props, context), vNode);
      vNode.children = input;
      vNode.dom = dom = mount(input, null, context, isSVG);
      mountFunctionalComponentCallbacks(props, ref, dom);
    }
    if (!isNull(parentDom)) {
      appendChild(parentDom, dom);
    }
    return dom;
  }
  function createClassMountCallback(instance) {
    return function () {
      instance.$UPD = true;
      instance.componentDidMount();
      instance.$UPD = false;
    };
  }
  function mountClassComponentCallbacks(vNode, ref, instance) {
    if (isFunction(ref)) {
      ref(instance);
    } else {
      {
        if (isStringOrNumber(ref)) {
          throwError('string "refs" are not supported in Inferno 1.0. Use callback "refs" instead.');
        } else if (!isNullOrUndef(ref) && isObject(ref) && vNode.flags & 4 /* VNodeFlags.ComponentClass */) {
          throwError('functional component lifecycle events are not supported on ES2015 class components.');
        }
      }
    }
    if (isFunction(instance.componentDidMount)) {
      LIFECYCLE.push(createClassMountCallback(instance));
    }
  }
  function createOnMountCallback(ref, dom, props) {
    return function () {
      return ref.onComponentDidMount(dom, props);
    };
  }
  function mountFunctionalComponentCallbacks(props, ref, dom) {
    if (!isNullOrUndef(ref)) {
      if (isFunction(ref.onComponentWillMount)) {
        ref.onComponentWillMount(props);
      }
      if (isFunction(ref.onComponentDidMount)) {
        LIFECYCLE.push(createOnMountCallback(ref, dom, props));
      }
    }
  }
  function mountRef(dom, value) {
    LIFECYCLE.push(function () {
      return value(dom);
    });
  }
  function hydrateComponent(vNode, dom, context, isSVG, isClass) {
    var type = vNode.type;
    var ref = vNode.ref;
    var props = vNode.props || EMPTY_OBJ;
    if (isClass) {
      var instance = createClassComponentInstance(vNode, type, props, context);
      var input = instance.$LI;
      hydrateVNode(input, dom, instance.$CX, isSVG);
      vNode.dom = input.dom;
      mountClassComponentCallbacks(vNode, ref, instance);
      instance.$UPD = false; // Mount finished allow going sync
    } else {
      var _input = handleComponentInput(type(props, context), vNode);
      hydrateVNode(_input, dom, context, isSVG);
      vNode.children = _input;
      vNode.dom = _input.dom;
      mountFunctionalComponentCallbacks(props, ref, dom);
    }
  }
  function hydrateElement(vNode, dom, context, isSVG) {
    var children = vNode.children;
    var props = vNode.props;
    var className = vNode.className;
    var flags = vNode.flags;
    var ref = vNode.ref;
    isSVG = isSVG || (flags & 32 /* VNodeFlags.SvgElement */) > 0;
    if (dom.nodeType !== 1 || dom.tagName.toLowerCase() !== vNode.type) {
      {
        warning("Inferno hydration: Server-side markup doesn't match client-side markup or Initial render target is not empty");
      }
      var newDom = mountElement(vNode, null, context, isSVG);
      vNode.dom = newDom;
      replaceChild(dom.parentNode, newDom, dom);
    } else {
      vNode.dom = dom;
      var childNode = dom.firstChild;
      var childFlags = vNode.childFlags;
      if ((childFlags & 1 /* ChildFlags.HasInvalidChildren */) === 0) {
        var nextSibling = null;
        while (childNode) {
          nextSibling = childNode.nextSibling;
          if (childNode.nodeType === 8) {
            if (childNode.data === '!') {
              dom.replaceChild(document.createTextNode(''), childNode);
            } else {
              dom.removeChild(childNode);
            }
          }
          childNode = nextSibling;
        }
        childNode = dom.firstChild;
        if (childFlags === 2 /* ChildFlags.HasVNodeChildren */) {
          if (isNull(childNode)) {
            mount(children, dom, context, isSVG);
          } else {
            nextSibling = childNode.nextSibling;
            hydrateVNode(children, childNode, context, isSVG);
            childNode = nextSibling;
          }
        } else if (childFlags & 12 /* ChildFlags.MultipleChildren */) {
          for (var i = 0, len = children.length; i < len; i++) {
            var child = children[i];
            if (isNull(childNode)) {
              mount(child, dom, context, isSVG);
            } else {
              nextSibling = childNode.nextSibling;
              hydrateVNode(child, childNode, context, isSVG);
              childNode = nextSibling;
            }
          }
        }
        // clear any other DOM nodes, there should be only a single entry for the root
        while (childNode) {
          nextSibling = childNode.nextSibling;
          dom.removeChild(childNode);
          childNode = nextSibling;
        }
      } else if (!isNull(dom.firstChild) && !isSamePropsInnerHTML(dom, props)) {
        dom.textContent = ''; // dom has content, but VNode has no children remove everything from DOM
        if (flags & 448 /* VNodeFlags.FormElement */) {
          // If element is form element, we need to clear defaultValue also
          dom.defaultValue = '';
        }
      }
      if (!isNull(props)) {
        mountProps(vNode, flags, props, dom, isSVG);
      }
      if (isNullOrUndef(className)) {
        if (dom.className !== '') {
          dom.removeAttribute('class');
        }
      } else if (isSVG) {
        dom.setAttribute('class', className);
      } else {
        dom.className = className;
      }
      if (isFunction(ref)) {
        mountRef(dom, ref);
      } else {
        {
          if (isString(ref)) {
            throwError('string "refs" are not supported in Inferno 1.0. Use callback "refs" instead.');
          }
        }
      }
    }
  }
  function hydrateText(vNode, dom) {
    if (dom.nodeType !== 3) {
      var newDom = mountText(vNode, null);
      vNode.dom = newDom;
      replaceChild(dom.parentNode, newDom, dom);
    } else {
      var text = vNode.children;
      if (dom.nodeValue !== text) {
        dom.nodeValue = text;
      }
      vNode.dom = dom;
    }
  }
  function hydrateVNode(vNode, dom, context, isSVG) {
    var flags = vNode.flags;
    if (flags & 14 /* VNodeFlags.Component */) {
      hydrateComponent(vNode, dom, context, isSVG, (flags & 4 /* VNodeFlags.ComponentClass */) > 0);
    } else if (flags & 481 /* VNodeFlags.Element */) {
      hydrateElement(vNode, dom, context, isSVG);
    } else if (flags & 16 /* VNodeFlags.Text */) {
      hydrateText(vNode, dom);
    } else if (flags & 512 /* VNodeFlags.Void */) {
      vNode.dom = dom;
    } else {
      {
        throwError("hydrate() expects a valid VNode, instead it received an object with the type \"" + typeof vNode + "\".");
      }
      throwError();
    }
  }
  function hydrate(input, parentDom, callback) {
    var dom = parentDom.firstChild;
    if (!isNull(dom)) {
      if (!isInvalid(input)) {
        hydrateVNode(input, dom, EMPTY_OBJ, false);
      }
      dom = parentDom.firstChild;
      // clear any other DOM nodes, there should be only a single entry for the root
      while (dom = dom.nextSibling) {
        parentDom.removeChild(dom);
      }
    }
    if (LIFECYCLE.length > 0) {
      callAll(LIFECYCLE);
    }
    parentDom.$V = input;
    if (isFunction(callback)) {
      callback();
    }
  }
  function replaceWithNewNode(lastNode, nextNode, parentDom, context, isSVG) {
    unmount(lastNode);
    replaceChild(parentDom, mount(nextNode, null, context, isSVG), lastNode.dom);
  }
  function patch(lastVNode, nextVNode, parentDom, context, isSVG) {
    var nextFlags = nextVNode.flags | 0;
    if (lastVNode.flags !== nextFlags || nextFlags & 2048 /* VNodeFlags.ReCreate */) {
      replaceWithNewNode(lastVNode, nextVNode, parentDom, context, isSVG);
    } else if (nextFlags & 481 /* VNodeFlags.Element */) {
      patchElement(lastVNode, nextVNode, parentDom, context, isSVG, nextFlags);
    } else if (nextFlags & 14 /* VNodeFlags.Component */) {
      patchComponent(lastVNode, nextVNode, parentDom, context, isSVG, (nextFlags & 4 /* VNodeFlags.ComponentClass */) > 0);
    } else if (nextFlags & 16 /* VNodeFlags.Text */) {
      patchText(lastVNode, nextVNode);
    } else if (nextFlags & 512 /* VNodeFlags.Void */) {
      nextVNode.dom = lastVNode.dom;
    } else {
      patchPortal(lastVNode, nextVNode, context);
    }
  }
  function patchContentEditableChildren(dom, nextVNode) {
    if (dom.textContent !== nextVNode.children) {
      dom.textContent = nextVNode.children;
    }
  }
  function patchPortal(lastVNode, nextVNode, context) {
    var lastContainer = lastVNode.type;
    var nextContainer = nextVNode.type;
    var nextChildren = nextVNode.children;
    patchChildren(lastVNode.childFlags, nextVNode.childFlags, lastVNode.children, nextChildren, lastContainer, context, false);
    nextVNode.dom = lastVNode.dom;
    if (lastContainer !== nextContainer && !isInvalid(nextChildren)) {
      var node = nextChildren.dom;
      lastContainer.removeChild(node);
      nextContainer.appendChild(node);
    }
  }
  function patchElement(lastVNode, nextVNode, parentDom, context, isSVG, nextFlags) {
    var nextTag = nextVNode.type;
    if (lastVNode.type !== nextTag) {
      replaceWithNewNode(lastVNode, nextVNode, parentDom, context, isSVG);
    } else {
      var dom = lastVNode.dom;
      var lastProps = lastVNode.props;
      var nextProps = nextVNode.props;
      var isFormElement = false;
      var hasControlledValue = false;
      var nextPropsOrEmpty;
      nextVNode.dom = dom;
      isSVG = isSVG || (nextFlags & 32 /* VNodeFlags.SvgElement */) > 0;
      // inlined patchProps  -- starts --
      if (lastProps !== nextProps) {
        var lastPropsOrEmpty = lastProps || EMPTY_OBJ;
        nextPropsOrEmpty = nextProps || EMPTY_OBJ;
        if (nextPropsOrEmpty !== EMPTY_OBJ) {
          isFormElement = (nextFlags & 448 /* VNodeFlags.FormElement */) > 0;
          if (isFormElement) {
            hasControlledValue = isControlledFormElement(nextPropsOrEmpty);
          }
          for (var prop in nextPropsOrEmpty) {
            var lastValue = lastPropsOrEmpty[prop];
            var nextValue = nextPropsOrEmpty[prop];
            if (lastValue !== nextValue) {
              patchProp(prop, lastValue, nextValue, dom, isSVG, hasControlledValue, lastVNode);
            }
          }
        }
        if (lastPropsOrEmpty !== EMPTY_OBJ) {
          for (var _prop in lastPropsOrEmpty) {
            if (!nextPropsOrEmpty.hasOwnProperty(_prop) && !isNullOrUndef(lastPropsOrEmpty[_prop])) {
              patchProp(_prop, lastPropsOrEmpty[_prop], null, dom, isSVG, hasControlledValue, lastVNode);
            }
          }
        }
      }
      var lastChildren = lastVNode.children;
      var nextChildren = nextVNode.children;
      var nextRef = nextVNode.ref;
      var lastClassName = lastVNode.className;
      var nextClassName = nextVNode.className;
      {
        validateKeys(nextVNode);
      }
      if (nextFlags & 4096 /* VNodeFlags.ContentEditable */) {
        patchContentEditableChildren(dom, nextChildren);
      } else {
        patchChildren(lastVNode.childFlags, nextVNode.childFlags, lastChildren, nextChildren, dom, context, isSVG && nextTag !== 'foreignObject');
      }
      if (isFormElement) {
        processElement(nextFlags, nextVNode, dom, nextPropsOrEmpty, false, hasControlledValue);
      }
      // inlined patchProps  -- ends --
      if (lastClassName !== nextClassName) {
        if (isNullOrUndef(nextClassName)) {
          dom.removeAttribute('class');
        } else if (isSVG) {
          dom.setAttribute('class', nextClassName);
        } else {
          dom.className = nextClassName;
        }
      }
      if (isFunction(nextRef) && lastVNode.ref !== nextRef) {
        mountRef(dom, nextRef);
      } else {
        {
          if (isString(nextRef)) {
            throwError('string "refs" are not supported in Inferno 1.0. Use callback "refs" instead.');
          }
        }
      }
    }
  }
  function patchChildren(lastChildFlags, nextChildFlags, lastChildren, nextChildren, parentDOM, context, isSVG) {
    switch (lastChildFlags) {
      case 2 /* ChildFlags.HasVNodeChildren */:
        switch (nextChildFlags) {
          case 2 /* ChildFlags.HasVNodeChildren */:
            patch(lastChildren, nextChildren, parentDOM, context, isSVG);
            break;
          case 1 /* ChildFlags.HasInvalidChildren */:
            remove$1(lastChildren, parentDOM);
            break;
          default:
            remove$1(lastChildren, parentDOM);
            mountArrayChildren(nextChildren, parentDOM, context, isSVG);
            break;
        }
        break;
      case 1 /* ChildFlags.HasInvalidChildren */:
        switch (nextChildFlags) {
          case 2 /* ChildFlags.HasVNodeChildren */:
            mount(nextChildren, parentDOM, context, isSVG);
            break;
          case 1 /* ChildFlags.HasInvalidChildren */:
            break;
          default:
            mountArrayChildren(nextChildren, parentDOM, context, isSVG);
            break;
        }
        break;
      default:
        if (nextChildFlags & 12 /* ChildFlags.MultipleChildren */) {
          var lastLength = lastChildren.length;
          var nextLength = nextChildren.length;
          // Fast path's for both algorithms
          if (lastLength === 0) {
            if (nextLength > 0) {
              mountArrayChildren(nextChildren, parentDOM, context, isSVG);
            }
          } else if (nextLength === 0) {
            removeAllChildren(parentDOM, lastChildren);
          } else if (nextChildFlags === 8 /* ChildFlags.HasKeyedChildren */ && lastChildFlags === 8 /* ChildFlags.HasKeyedChildren */) {
            patchKeyedChildren(lastChildren, nextChildren, parentDOM, context, isSVG, lastLength, nextLength);
          } else {
            patchNonKeyedChildren(lastChildren, nextChildren, parentDOM, context, isSVG, lastLength, nextLength);
          }
        } else if (nextChildFlags === 1 /* ChildFlags.HasInvalidChildren */) {
          removeAllChildren(parentDOM, lastChildren);
        } else if (nextChildFlags === 2 /* ChildFlags.HasVNodeChildren */) {
          removeAllChildren(parentDOM, lastChildren);
          mount(nextChildren, parentDOM, context, isSVG);
        }
        break;
    }
  }
  function updateClassComponent(instance, nextState, nextVNode, nextProps, parentDom, context, isSVG, force, fromSetState) {
    var lastState = instance.state;
    var lastProps = instance.props;
    nextVNode.children = instance;
    var renderOutput;
    if (instance.$UN) {
      {
        warning('Inferno Error: Can only update a mounted or mounting component. This usually means you called setState() or forceUpdate() on an unmounted component. This is a no-op.');
      }
      return;
    }
    if (lastProps !== nextProps || nextProps === EMPTY_OBJ) {
      if (!fromSetState && isFunction(instance.componentWillReceiveProps)) {
        instance.$BR = true;
        instance.componentWillReceiveProps(nextProps, context);
        // If instance component was removed during its own update do nothing.
        if (instance.$UN) {
          return;
        }
        instance.$BR = false;
      }
      if (instance.$PSS) {
        nextState = combineFrom(nextState, instance.$PS);
        instance.$PSS = false;
        instance.$PS = null;
      }
    }
    /* Update if scu is not defined, or it returns truthy value or force */
    var hasSCU = Boolean(instance.shouldComponentUpdate);
    if (force || !hasSCU || hasSCU && instance.shouldComponentUpdate(nextProps, nextState, context)) {
      if (isFunction(instance.componentWillUpdate)) {
        instance.$BS = true;
        instance.componentWillUpdate(nextProps, nextState, context);
        instance.$BS = false;
      }
      instance.props = nextProps;
      instance.state = nextState;
      instance.context = context;
      if (isFunction(options.beforeRender)) {
        options.beforeRender(instance);
      }
      renderOutput = instance.render(nextProps, nextState, context);
      if (isFunction(options.afterRender)) {
        options.afterRender(instance);
      }
      var didUpdate = renderOutput !== NO_OP;
      var childContext;
      if (isFunction(instance.getChildContext)) {
        childContext = instance.getChildContext();
      }
      if (isNullOrUndef(childContext)) {
        childContext = context;
      } else {
        childContext = combineFrom(context, childContext);
      }
      instance.$CX = childContext;
      if (didUpdate) {
        var lastInput = instance.$LI;
        var nextInput = handleComponentInput(renderOutput, nextVNode);
        patch(lastInput, nextInput, parentDom, childContext, isSVG);
        instance.$LI = nextInput;
        if (isFunction(instance.componentDidUpdate)) {
          instance.componentDidUpdate(lastProps, lastState);
        }
      }
    } else {
      instance.props = nextProps;
      instance.state = nextState;
      instance.context = context;
    }
    nextVNode.dom = instance.$LI.dom;
  }
  function patchComponent(lastVNode, nextVNode, parentDom, context, isSVG, isClass) {
    var nextType = nextVNode.type;
    var lastKey = lastVNode.key;
    var nextKey = nextVNode.key;
    if (lastVNode.type !== nextType || lastKey !== nextKey) {
      replaceWithNewNode(lastVNode, nextVNode, parentDom, context, isSVG);
    } else {
      var nextProps = nextVNode.props || EMPTY_OBJ;
      if (isClass) {
        var instance = lastVNode.children;
        instance.$UPD = true;
        instance.$V = nextVNode;
        updateClassComponent(instance, instance.state, nextVNode, nextProps, parentDom, context, isSVG, false, false);
        instance.$UPD = false;
      } else {
        var shouldUpdate = true;
        var lastProps = lastVNode.props;
        var nextHooks = nextVNode.ref;
        var nextHooksDefined = !isNullOrUndef(nextHooks);
        var lastInput = lastVNode.children;
        nextVNode.dom = lastVNode.dom;
        nextVNode.children = lastInput;
        if (nextHooksDefined && isFunction(nextHooks.onComponentShouldUpdate)) {
          shouldUpdate = nextHooks.onComponentShouldUpdate(lastProps, nextProps);
        }
        if (shouldUpdate !== false) {
          if (nextHooksDefined && isFunction(nextHooks.onComponentWillUpdate)) {
            nextHooks.onComponentWillUpdate(lastProps, nextProps);
          }
          var nextInput = nextType(nextProps, context);
          if (nextInput !== NO_OP) {
            nextInput = handleComponentInput(nextInput, nextVNode);
            patch(lastInput, nextInput, parentDom, context, isSVG);
            nextVNode.children = nextInput;
            nextVNode.dom = nextInput.dom;
            if (nextHooksDefined && isFunction(nextHooks.onComponentDidUpdate)) {
              nextHooks.onComponentDidUpdate(lastProps, nextProps);
            }
          }
        } else if (lastInput.flags & 14 /* VNodeFlags.Component */) {
          lastInput.parentVNode = nextVNode;
        }
      }
    }
  }
  function patchText(lastVNode, nextVNode) {
    var nextText = nextVNode.children;
    var dom = lastVNode.dom;
    if (nextText !== lastVNode.children) {
      dom.nodeValue = nextText;
    }
    nextVNode.dom = dom;
  }
  function patchNonKeyedChildren(lastChildren, nextChildren, dom, context, isSVG, lastChildrenLength, nextChildrenLength) {
    var commonLength = lastChildrenLength > nextChildrenLength ? nextChildrenLength : lastChildrenLength;
    var i = 0;
    var nextChild;
    var lastChild;
    for (; i < commonLength; i++) {
      nextChild = nextChildren[i];
      lastChild = lastChildren[i];
      if (nextChild.dom) {
        nextChild = nextChildren[i] = directClone(nextChild);
      }
      patch(lastChild, nextChild, dom, context, isSVG);
      lastChildren[i] = nextChild;
    }
    if (lastChildrenLength < nextChildrenLength) {
      for (i = commonLength; i < nextChildrenLength; i++) {
        nextChild = nextChildren[i];
        if (nextChild.dom) {
          nextChild = nextChildren[i] = directClone(nextChild);
        }
        mount(nextChild, dom, context, isSVG);
      }
    } else if (lastChildrenLength > nextChildrenLength) {
      for (i = commonLength; i < lastChildrenLength; i++) {
        remove$1(lastChildren[i], dom);
      }
    }
  }
  function patchKeyedChildren(a, b, dom, context, isSVG, aLength, bLength) {
    var aEnd = aLength - 1;
    var bEnd = bLength - 1;
    var i;
    var j = 0;
    var aNode = a[j];
    var bNode = b[j];
    var nextPos;
    // Step 1
    // tslint:disable-next-line
    outer: {
      // Sync nodes with the same key at the beginning.
      while (aNode.key === bNode.key) {
        if (bNode.dom) {
          b[j] = bNode = directClone(bNode);
        }
        patch(aNode, bNode, dom, context, isSVG);
        a[j] = bNode;
        j++;
        if (j > aEnd || j > bEnd) {
          break outer;
        }
        aNode = a[j];
        bNode = b[j];
      }
      aNode = a[aEnd];
      bNode = b[bEnd];
      // Sync nodes with the same key at the end.
      while (aNode.key === bNode.key) {
        if (bNode.dom) {
          b[bEnd] = bNode = directClone(bNode);
        }
        patch(aNode, bNode, dom, context, isSVG);
        a[aEnd] = bNode;
        aEnd--;
        bEnd--;
        if (j > aEnd || j > bEnd) {
          break outer;
        }
        aNode = a[aEnd];
        bNode = b[bEnd];
      }
    }
    if (j > aEnd) {
      if (j <= bEnd) {
        nextPos = bEnd + 1;
        var nextNode = nextPos < bLength ? b[nextPos].dom : null;
        while (j <= bEnd) {
          bNode = b[j];
          if (bNode.dom) {
            b[j] = bNode = directClone(bNode);
          }
          j++;
          insertOrAppend(dom, mount(bNode, null, context, isSVG), nextNode);
        }
      }
    } else if (j > bEnd) {
      while (j <= aEnd) {
        remove$1(a[j++], dom);
      }
    } else {
      var aStart = j;
      var bStart = j;
      var aLeft = aEnd - j + 1;
      var bLeft = bEnd - j + 1;
      var sources = [];
      for (i = 0; i < bLeft; i++) {
        sources.push(0);
      }
      // Keep track if its possible to remove whole DOM using textContent = '';
      var canRemoveWholeContent = aLeft === aLength;
      var moved = false;
      var pos = 0;
      var patched = 0;
      // When sizes are small, just loop them through
      if (bLength < 4 || (aLeft | bLeft) < 32) {
        for (i = aStart; i <= aEnd; i++) {
          aNode = a[i];
          if (patched < bLeft) {
            for (j = bStart; j <= bEnd; j++) {
              bNode = b[j];
              if (aNode.key === bNode.key) {
                sources[j - bStart] = i + 1;
                if (canRemoveWholeContent) {
                  canRemoveWholeContent = false;
                  while (i > aStart) {
                    remove$1(a[aStart++], dom);
                  }
                }
                if (pos > j) {
                  moved = true;
                } else {
                  pos = j;
                }
                if (bNode.dom) {
                  b[j] = bNode = directClone(bNode);
                }
                patch(aNode, bNode, dom, context, isSVG);
                patched++;
                break;
              }
            }
            if (!canRemoveWholeContent && j > bEnd) {
              remove$1(aNode, dom);
            }
          } else if (!canRemoveWholeContent) {
            remove$1(aNode, dom);
          }
        }
      } else {
        var keyIndex = {};
        // Map keys by their index
        for (i = bStart; i <= bEnd; i++) {
          keyIndex[b[i].key] = i;
        }
        // Try to patch same keys
        for (i = aStart; i <= aEnd; i++) {
          aNode = a[i];
          if (patched < bLeft) {
            j = keyIndex[aNode.key];
            if (j !== void 0) {
              if (canRemoveWholeContent) {
                canRemoveWholeContent = false;
                while (i > aStart) {
                  remove$1(a[aStart++], dom);
                }
              }
              bNode = b[j];
              sources[j - bStart] = i + 1;
              if (pos > j) {
                moved = true;
              } else {
                pos = j;
              }
              if (bNode.dom) {
                b[j] = bNode = directClone(bNode);
              }
              patch(aNode, bNode, dom, context, isSVG);
              patched++;
            } else if (!canRemoveWholeContent) {
              remove$1(aNode, dom);
            }
          } else if (!canRemoveWholeContent) {
            remove$1(aNode, dom);
          }
        }
      }
      // fast-path: if nothing patched remove all old and add all new
      if (canRemoveWholeContent) {
        removeAllChildren(dom, a);
        mountArrayChildren(b, dom, context, isSVG);
      } else {
        if (moved) {
          var seq = lis_algorithm(sources);
          j = seq.length - 1;
          for (i = bLeft - 1; i >= 0; i--) {
            if (sources[i] === 0) {
              pos = i + bStart;
              bNode = b[pos];
              if (bNode.dom) {
                b[pos] = bNode = directClone(bNode);
              }
              nextPos = pos + 1;
              insertOrAppend(dom, mount(bNode, null, context, isSVG), nextPos < bLength ? b[nextPos].dom : null);
            } else if (j < 0 || i !== seq[j]) {
              pos = i + bStart;
              bNode = b[pos];
              nextPos = pos + 1;
              insertOrAppend(dom, bNode.dom, nextPos < bLength ? b[nextPos].dom : null);
            } else {
              j--;
            }
          }
        } else if (patched !== bLeft) {
          // when patched count doesn't match b length we need to insert those new ones
          // loop backwards so we can use insertBefore
          for (i = bLeft - 1; i >= 0; i--) {
            if (sources[i] === 0) {
              pos = i + bStart;
              bNode = b[pos];
              if (bNode.dom) {
                b[pos] = bNode = directClone(bNode);
              }
              nextPos = pos + 1;
              insertOrAppend(dom, mount(bNode, null, context, isSVG), nextPos < bLength ? b[nextPos].dom : null);
            }
          }
        }
      }
    }
  }
  // https://en.wikipedia.org/wiki/Longest_increasing_subsequence
  function lis_algorithm(arr) {
    var p = arr.slice();
    var result = [0];
    var i;
    var j;
    var u;
    var v;
    var c;
    var len = arr.length;
    for (i = 0; i < len; i++) {
      var arrI = arr[i];
      if (arrI !== 0) {
        j = result[result.length - 1];
        if (arr[j] < arrI) {
          p[i] = j;
          result.push(i);
          continue;
        }
        u = 0;
        v = result.length - 1;
        while (u < v) {
          c = (u + v) / 2 | 0;
          if (arr[result[c]] < arrI) {
            u = c + 1;
          } else {
            v = c;
          }
        }
        if (arrI < arr[result[u]]) {
          if (u > 0) {
            p[i] = result[u - 1];
          }
          result[u] = i;
        }
      }
    }
    u = result.length;
    v = result[u - 1];
    while (u-- > 0) {
      result[u] = v;
      v = p[v];
    }
    return result;
  }
  {
    if (isBrowser && document.body === null) {
      warning('Inferno warning: you cannot initialize inferno without "document.body". Wait on "DOMContentLoaded" event, add script to bottom of body, or use async/defer attributes on script tag.');
    }
  }
  var documentBody = isBrowser ? document.body : null;
  function render(input, parentDom, callback) {
    // Development warning
    {
      if (documentBody === parentDom) {
        throwError('you cannot render() to the "document.body". Use an empty element as a container instead.');
      }
    }
    if (input === NO_OP) {
      return;
    }
    var rootInput = parentDom.$V;
    if (isNullOrUndef(rootInput)) {
      if (!isInvalid(input)) {
        if (input.dom) {
          input = directClone(input);
        }
        if (isNull(parentDom.firstChild)) {
          mount(input, parentDom, EMPTY_OBJ, false);
          parentDom.$V = input;
        } else {
          hydrate(input, parentDom);
        }
        rootInput = input;
      }
    } else {
      if (isNullOrUndef(input)) {
        remove$1(rootInput, parentDom);
        parentDom.$V = null;
      } else {
        if (input.dom) {
          input = directClone(input);
        }
        patch(rootInput, input, parentDom, EMPTY_OBJ, false);
        rootInput = parentDom.$V = input;
      }
    }
    if (LIFECYCLE.length > 0) {
      callAll(LIFECYCLE);
    }
    if (isFunction(callback)) {
      callback();
    }
    if (isFunction(options.renderComplete)) {
      options.renderComplete(rootInput);
    }
    if (rootInput && rootInput.flags & 14 /* VNodeFlags.Component */) {
      return rootInput.children;
    }
  }
  function createPortal(children, container) {
    return createVNode(1024 /* VNodeFlags.Portal */, container, null, children, 0 /* ChildFlags.UnknownChildren */, null, isInvalid(children) ? null : children.key, null);
  }
  var resolvedPromise = typeof Promise === 'undefined' ? null : Promise.resolve();
  // raf.bind(window) is needed to work around bug in IE10-IE11 strict mode (TypeError: Invalid calling object)
  var fallbackMethod = typeof requestAnimationFrame === 'undefined' ? setTimeout : requestAnimationFrame.bind(window);
  function nextTick(fn) {
    if (resolvedPromise) {
      return resolvedPromise.then(fn);
    }
    return fallbackMethod(fn);
  }
  function queueStateChanges(component, newState, callback, force) {
    if (isFunction(newState)) {
      newState = newState(component.state, component.props, component.context);
    }
    var pending = component.$PS;
    if (isNullOrUndef(pending)) {
      component.$PS = newState;
    } else {
      for (var stateKey in newState) {
        pending[stateKey] = newState[stateKey];
      }
    }
    if (!component.$PSS && !component.$BR) {
      if (!component.$UPD) {
        component.$PSS = true;
        component.$UPD = true;
        applyState(component, force, callback);
        component.$UPD = false;
      } else {
        // Async
        var queue = component.$QU;
        if (isNull(queue)) {
          queue = component.$QU = [];
          nextTick(promiseCallback(component, queue));
        }
        if (isFunction(callback)) {
          queue.push(callback);
        }
      }
    } else {
      component.$PSS = true;
      if (component.$BR && isFunction(callback)) {
        LIFECYCLE.push(callback.bind(component));
      }
    }
  }
  function promiseCallback(component, queue) {
    return function () {
      component.$QU = null;
      component.$UPD = true;
      applyState(component, false, function () {
        for (var i = 0, len = queue.length; i < len; i++) {
          queue[i].call(component);
        }
      });
      component.$UPD = false;
    };
  }
  function applyState(component, force, callback) {
    if (component.$UN) {
      return;
    }
    if (force || !component.$BR) {
      component.$PSS = false;
      var pendingState = component.$PS;
      var prevState = component.state;
      var nextState = combineFrom(prevState, pendingState);
      var props = component.props;
      var context = component.context;
      component.$PS = null;
      var vNode = component.$V;
      var lastInput = component.$LI;
      var parentDom = lastInput.dom && lastInput.dom.parentNode;
      updateClassComponent(component, nextState, vNode, props, parentDom, context, (vNode.flags & 32 /* VNodeFlags.SvgElement */) > 0, force, true);
      if (component.$UN) {
        return;
      }
      if ((component.$LI.flags & 1024 /* VNodeFlags.Portal */) === 0) {
        var dom = component.$LI.dom;
        while (!isNull(vNode = vNode.parentVNode)) {
          if ((vNode.flags & 14 /* VNodeFlags.Component */) > 0) {
            vNode.dom = dom;
          }
        }
      }
      if (LIFECYCLE.length > 0) {
        callAll(LIFECYCLE);
      }
    } else {
      component.state = component.$PS;
      component.$PS = null;
    }
    if (isFunction(callback)) {
      callback.call(component);
    }
  }
  var Component = /*#__PURE__*/function () {
    // QUEUE
    function Component(props, context) {
      this.state = null;
      this.props = void 0;
      this.context = void 0;
      this.refs = void 0;
      // Internal properties
      this.$BR = false;
      // BLOCK RENDER
      this.$BS = true;
      // BLOCK STATE
      this.$PSS = false;
      // PENDING SET STATE
      this.$PS = null;
      // PENDING STATE (PARTIAL or FULL)
      this.$LI = null;
      // LAST INPUT
      this.$V = null;
      // VNODE
      this.$UN = false;
      // UNMOUNTED
      this.$CX = null;
      // CHILDCONTEXT
      this.$UPD = true;
      // UPDATING
      this.$QU = null;
      /** @type {object} */
      this.props = props || EMPTY_OBJ;
      /** @type {object} */
      this.context = context || EMPTY_OBJ; // context should not be mutable
    }
    var _proto = Component.prototype;
    _proto.forceUpdate = function forceUpdate(callback) {
      if (this.$UN) {
        return;
      }
      // Do not allow double render during force update
      queueStateChanges(this, {}, callback, true);
    };
    _proto.setState = function setState(newState, callback) {
      if (this.$UN) {
        return;
      }
      if (!this.$BS) {
        queueStateChanges(this, newState, callback, false);
      } else {
        // Development warning
        {
          throwError('cannot update state via setState() in componentWillUpdate() or constructor.');
        }
        return;
      }
    }
    // tslint:disable-next-line:no-empty
    ;
    _proto.render = function render(_nextProps, _nextState, _nextContext) {};
    return Component;
  }();
  // Public
  Component.defaultProps = void 0;
  {
    /* tslint:disable-next-line:no-empty */
    var testFunc = function testFn() {};
    /* tslint:disable-next-line*/
    console.info('Inferno is in development mode.');
    if ((testFunc.name || testFunc.toString()).indexOf('testFn') === -1) {
      warning("It looks like you're using a minified copy of the development build " + 'of Inferno. When deploying Inferno apps to production, make sure to use ' + 'the production build which skips development warnings and is faster. ' + 'See http://infernojs.org for more details.');
    }
  }

  // inlined ../../../../resources/logo.svg
  // eslint-disable-next-line
  var BPMNIO_IMG = '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 14.02 5.57" width="53" height="21" style="vertical-align:middle"><path fill="#000000" d="M1.88.92v.14c0 .41-.13.68-.4.8.33.14.46.44.46.86v.33c0 .61-.33.95-.95.95H0V0h.95c.65 0 .93.3.93.92zM.63.57v1.06h.24c.24 0 .38-.1.38-.43V.98c0-.28-.1-.4-.32-.4zm0 1.63v1.22h.36c.2 0 .32-.1.32-.39v-.35c0-.37-.12-.48-.4-.48H.63zM4.18.99v.52c0 .64-.31.98-.94.98h-.3V4h-.62V0h.92c.63 0 .94.35.94.99zM2.94.57v1.35h.3c.2 0 .3-.09.3-.37v-.6c0-.29-.1-.38-.3-.38h-.3zm2.89 2.27L6.25 0h.88v4h-.6V1.12L6.1 3.99h-.6l-.46-2.82v2.82h-.55V0h.87zM8.14 1.1V4h-.56V0h.79L9 2.4V0h.56v4h-.64zm2.49 2.29v.6h-.6v-.6zM12.12 1c0-.63.33-1 .95-1 .61 0 .95.37.95 1v2.04c0 .64-.34 1-.95 1-.62 0-.95-.37-.95-1zm.62 2.08c0 .28.13.39.33.39s.32-.1.32-.4V.98c0-.29-.12-.4-.32-.4s-.33.11-.33.4z"/><path fill="#000000" d="M0 4.53h14.02v1.04H0zM11.08 0h.63v.62h-.63zm.63 4V1h-.63v2.98z"/></svg>';

  /**
   * Adds the project logo to the diagram container as
   * required by the bpmn.io license.
   *
   * @see http://bpmn.io/license
   *
   * @param {Element} container
   */
  function addProjectLogo(container) {
    var linkMarkup = '<a href="http://bpmn.io" ' + 'target="_blank" ' + 'class="bjs-powered-by" ' + 'title="Powered by bpmn.io" ' + 'style="position: absolute; bottom: 15px; right: 15px; z-index: 100;">' + BPMNIO_IMG + '</a>';
    var linkElement = domify$1(linkMarkup);
    container.appendChild(linkElement);
    event.bind(linkElement, 'click', function (event) {
      open();
      event.preventDefault();
    });
  }
  class PoweredByComponent extends Component {
    constructor(props, context) {
      super(props, context);
      this.node = null;
    }
    componentDidMount() {
      addProjectLogo(this.node);
    }
    render() {
      return createVNode(1, "div", null, null, 1, null, null, node => this.node = node);
    }
  }
  function css(attrs) {
    return attrs.join(';');
  }
  var LIGHTBOX_STYLES = css(['z-index: 1001', 'position: fixed', 'top: 0', 'left: 0', 'right: 0', 'bottom: 0']);
  var BACKDROP_STYLES = css(['width: 100%', 'height: 100%', 'background: rgba(40,40,40,0.2)']);
  var NOTICE_STYLES = css(['position: absolute', 'left: 50%', 'top: 40%', 'transform: translate(-50%)', 'width: 260px', 'padding: 10px', 'background: white', 'box-shadow: 0 1px 4px rgba(0,0,0,0.3)', 'font-family: Helvetica, Arial, sans-serif', 'font-size: 14px', 'display: flex', 'line-height: 1.3']);

  /* eslint-disable max-len */
  var LIGHTBOX_MARKUP = '<div class="bjs-powered-by-lightbox" style="' + LIGHTBOX_STYLES + '">' + '<div class="backdrop" style="' + BACKDROP_STYLES + '"></div>' + '<div class="notice" style="' + NOTICE_STYLES + '">' + '<a href="https://bpmn.io" target="_blank" rel="noopener" style="margin: 15px 20px 15px 10px; align-self: center;' + '">' + BPMNIO_IMG + '</a>' + '<span>' + 'Web-based tooling for BPMN, DMN and CMMN diagrams ' + 'powered by <a href="https://bpmn.io" target="_blank" rel="noopener">bpmn.io</a>.' + '</span>' + '</div>' + '</div>';
  /* eslint-enable */

  var lightbox;
  function open() {
    if (!lightbox) {
      lightbox = domify$1(LIGHTBOX_MARKUP);
      delegate.bind(lightbox, '.backdrop', 'click', function (event) {
        document.body.removeChild(lightbox);
      });
    }
    document.body.appendChild(lightbox);
  }

  function createCommonjsModule(fn, module) {
    return module = {
      exports: {}
    }, fn(module, module.exports), module.exports;
  }
  var hat_1 = createCommonjsModule(function (module) {
    var hat = module.exports = function (bits, base) {
      if (!base) base = 16;
      if (bits === undefined) bits = 128;
      if (bits <= 0) return '0';
      var digits = Math.log(Math.pow(2, bits)) / Math.log(base);
      for (var i = 2; digits === Infinity; i *= 2) {
        digits = Math.log(Math.pow(2, bits / i)) / Math.log(base) * i;
      }
      var rem = digits - Math.floor(digits);
      var res = '';
      for (var i = 0; i < Math.floor(digits); i++) {
        var x = Math.floor(Math.random() * base).toString(base);
        res = x + res;
      }
      if (rem) {
        var b = Math.pow(base, rem);
        var x = Math.floor(Math.random() * b).toString(base);
        res = x + res;
      }
      var parsed = parseInt(res, base);
      if (parsed !== Infinity && parsed >= Math.pow(2, bits)) {
        return hat(bits, base);
      } else return res;
    };
    hat.rack = function (bits, base, expandBy) {
      var fn = function (data) {
        var iters = 0;
        do {
          if (iters++ > 10) {
            if (expandBy) bits += expandBy;else throw new Error('too many ID collisions, use more bits');
          }
          var id = hat(bits, base);
        } while (Object.hasOwnProperty.call(hats, id));
        hats[id] = data;
        return id;
      };
      var hats = fn.hats = {};
      fn.get = function (id) {
        return fn.hats[id];
      };
      fn.set = function (id, value) {
        fn.hats[id] = value;
        return fn;
      };
      fn.bits = bits || 128;
      fn.base = base || 16;
      return fn;
    };
  });

  /**
   * Create a new id generator / cache instance.
   *
   * You may optionally provide a seed that is used internally.
   *
   * @param {Seed} seed
   */
  function Ids(seed) {
    if (!(this instanceof Ids)) {
      return new Ids(seed);
    }
    seed = seed || [128, 36, 1];
    this._seed = seed.length ? hat_1.rack(seed[0], seed[1], seed[2]) : seed;
  }

  /**
   * Generate a next id.
   *
   * @param {Object} [element] element to bind the id to
   *
   * @return {String} id
   */
  Ids.prototype.next = function (element) {
    return this._seed(element || true);
  };

  /**
   * Generate a next id with a given prefix.
   *
   * @param {Object} [element] element to bind the id to
   *
   * @return {String} id
   */
  Ids.prototype.nextPrefixed = function (prefix, element) {
    var id;
    do {
      id = prefix + this.next(true);
    } while (this.assigned(id));

    // claim {prefix}{random}
    this.claim(id, element);

    // return
    return id;
  };

  /**
   * Manually claim an existing id.
   *
   * @param {String} id
   * @param {String} [element] element the id is claimed by
   */
  Ids.prototype.claim = function (id, element) {
    this._seed.set(id, element || true);
  };

  /**
   * Returns true if the given id has already been assigned.
   *
   * @param  {String} id
   * @return {Boolean}
   */
  Ids.prototype.assigned = function (id) {
    return this._seed.get(id) || false;
  };

  /**
   * Unclaim an id.
   *
   * @param  {String} id the id to unclaim
   */
  Ids.prototype.unclaim = function (id) {
    delete this._seed.hats[id];
  };

  /**
   * Clear all claimed ids.
   */
  Ids.prototype.clear = function () {
    var hats = this._seed.hats,
      id;
    for (id in hats) {
      this.unclaim(id);
    }
  };

  var RENDERER_IDS = new Ids();
  var black = 'hsl(225, 10%, 15%)';

  /**
   * Renderer for the DRD view. The default colors are configurable.
   * When default label color is not provided, it will take the default
   * stroke color.
   *
   * @example
   * ```javascript
   * // for simple DRD viewer
   * const viewer = new DrdViewer({
   *   drdRenderer: {
   *     defaultFillColor: '#ffd700',
   *     defaultStrokeColor: '#0057b8',
   *     defaultLabelColor: '#0057b8'
   *   }
   * });
   *
   * // in dmn-js
   * const modeler = new DmnModeler({
   *   drd: {
   *     drdRenderer: {
   *       defaultFillColor: '#ffd700',
   *       defaultStrokeColor: '#0057b8',
   *       defaultLabelColor: '#0057b8'
   *     }
   *   }
   * });
   * ```
   */
  function DrdRenderer(config, eventBus, pathMap, styles, textRenderer, canvas) {
    BaseRenderer.call(this, eventBus);
    var rendererId = RENDERER_IDS.next();
    var computeStyle = styles.computeStyle;
    var markers = {};
    var defaultFillColor = config && config.defaultFillColor || 'white',
      defaultStrokeColor = config && config.defaultStrokeColor || black,
      defaultLabelColor = config && config.defaultLabelColor;
    function marker(type, fill, stroke) {
      var id = type + '-' + colorEscape(fill) + '-' + colorEscape(stroke) + '-' + rendererId;
      if (!markers[id]) {
        createMarker(id, type, fill, stroke);
      }
      return 'url(#' + id + ')';
    }
    function addMarker(id, options) {
      var attrs = assign$4({
        strokeWidth: 1,
        strokeLinecap: 'round',
        strokeDasharray: 'none'
      }, options.attrs);
      var ref = options.ref || {
        x: 0,
        y: 0
      };
      var scale = options.scale || 1;

      // fix for safari / chrome / firefox bug not correctly
      // resetting stroke dash array
      if (attrs.strokeDasharray === 'none') {
        attrs.strokeDasharray = [10000, 1];
      }
      var marker = create$2('marker');
      attr(options.element, attrs);
      append(marker, options.element);
      attr(marker, {
        id: id,
        viewBox: '0 0 20 20',
        refX: ref.x,
        refY: ref.y,
        markerWidth: 20 * scale,
        markerHeight: 20 * scale,
        orient: 'auto'
      });
      var defs = query('defs', canvas._svg);
      if (!defs) {
        defs = create$2('defs');
        append(canvas._svg, defs);
      }
      append(defs, marker);
      markers[id] = marker;
    }
    function createMarker(id, type, fill, stroke) {
      if (type === 'association-start') {
        var associationStart = create$2('path');
        attr(associationStart, {
          d: 'M 11 5 L 1 10 L 11 15'
        });
        addMarker(id, {
          element: associationStart,
          attrs: {
            fill: 'none',
            stroke: stroke,
            strokeWidth: 1.5
          },
          ref: {
            x: 1,
            y: 10
          },
          scale: 0.5
        });
      } else if (type === 'association-end') {
        var associationEnd = create$2('path');
        attr(associationEnd, {
          d: 'M 1 5 L 11 10 L 1 15'
        });
        addMarker(id, {
          element: associationEnd,
          attrs: {
            fill: 'none',
            stroke: stroke,
            strokeWidth: 1.5
          },
          ref: {
            x: 12,
            y: 10
          },
          scale: 0.5
        });
      } else if (type === 'information-requirement-end') {
        var informationRequirementEnd = create$2('path');
        attr(informationRequirementEnd, {
          d: 'M 1 5 L 11 10 L 1 15 Z'
        });
        addMarker(id, {
          element: informationRequirementEnd,
          attrs: {
            fill: stroke,
            stroke: 'none'
          },
          ref: {
            x: 11,
            y: 10
          },
          scale: 1
        });
      } else if (type === 'knowledge-requirement-end') {
        var knowledgeRequirementEnd = create$2('path');
        attr(knowledgeRequirementEnd, {
          d: 'M 1 3 L 11 10 L 1 17'
        });
        addMarker(id, {
          element: knowledgeRequirementEnd,
          attrs: {
            fill: 'none',
            stroke: stroke,
            strokeWidth: 2
          },
          ref: {
            x: 11,
            y: 10
          },
          scale: 0.8
        });
      } else if (type === 'authority-requirement-end') {
        var authorityRequirementEnd = create$2('circle');
        attr(authorityRequirementEnd, {
          cx: 3,
          cy: 3,
          r: 3
        });
        addMarker(id, {
          element: authorityRequirementEnd,
          attrs: {
            fill: stroke,
            stroke: 'none'
          },
          ref: {
            x: 3,
            y: 3
          },
          scale: 0.9
        });
      }
    }
    function drawRect(p, width, height, r, offset, attrs) {
      if (isObject$2(offset)) {
        attrs = offset;
        offset = 0;
      }
      offset = offset || 0;
      attrs = computeStyle(attrs, {
        stroke: black,
        strokeWidth: 2,
        fill: 'white'
      });
      var rect = create$2('rect');
      attr(rect, {
        x: offset,
        y: offset,
        width: width - offset * 2,
        height: height - offset * 2,
        rx: r,
        ry: r
      });
      attr(rect, attrs);
      append(p, rect);
      return rect;
    }
    function renderLabel(p, label, options) {
      var text = textRenderer.createText(label || '', options);
      attr$1(text, 'class', 'djs-label');
      append(p, text);
      return text;
    }
    function renderEmbeddedLabel(p, element, align, options) {
      var name = getName(element);
      options = assign$4({
        box: element,
        align: align,
        padding: 5,
        style: {
          fill: getLabelColor(element, defaultLabelColor, defaultStrokeColor)
        }
      }, options);
      return renderLabel(p, name, options);
    }
    function drawPath(p, d, attrs) {
      attrs = computeStyle(attrs, ['no-fill'], {
        strokeWidth: 2,
        stroke: black
      });
      var path = create$2('path');
      attr(path, {
        d: d
      });
      attr(path, attrs);
      append(p, path);
      return path;
    }
    var handlers = {
      'dmn:Decision': function (p, element) {
        var rect = drawRect(p, element.width, element.height, 0, {
          stroke: getStrokeColor(element, defaultStrokeColor),
          fill: getFillColor(element, defaultFillColor)
        });
        renderEmbeddedLabel(p, element, 'center-middle');
        return rect;
      },
      'dmn:KnowledgeSource': function (p, element) {
        var pathData = pathMap.getScaledPath('KNOWLEDGE_SOURCE', {
          xScaleFactor: 1.021,
          yScaleFactor: 1,
          containerWidth: element.width,
          containerHeight: element.height,
          position: {
            mx: 0.0,
            my: 0.075
          }
        });
        var knowledgeSource = drawPath(p, pathData, {
          strokeWidth: 2,
          fill: getFillColor(element, defaultFillColor),
          stroke: getStrokeColor(element, defaultStrokeColor)
        });
        renderEmbeddedLabel(p, element, 'center-middle');
        return knowledgeSource;
      },
      'dmn:BusinessKnowledgeModel': function (p, element) {
        var pathData = pathMap.getScaledPath('BUSINESS_KNOWLEDGE_MODEL', {
          xScaleFactor: 1,
          yScaleFactor: 1,
          containerWidth: element.width,
          containerHeight: element.height,
          position: {
            mx: 0.0,
            my: 0.3
          }
        });
        var businessKnowledge = drawPath(p, pathData, {
          strokeWidth: 2,
          fill: getFillColor(element, defaultFillColor),
          stroke: getStrokeColor(element, defaultStrokeColor)
        });
        renderEmbeddedLabel(p, element, 'center-middle');
        return businessKnowledge;
      },
      'dmn:InputData': function (p, element) {
        var rect = drawRect(p, element.width, element.height, 22, {
          stroke: getStrokeColor(element, defaultStrokeColor),
          fill: getFillColor(element, defaultFillColor)
        });
        renderEmbeddedLabel(p, element, 'center-middle');
        return rect;
      },
      'dmn:TextAnnotation': function (p, element) {
        var style = {
          'fill': 'none',
          'stroke': 'none'
        };
        var textElement = drawRect(p, element.width, element.height, 0, 0, style);
        var textPathData = pathMap.getScaledPath('TEXT_ANNOTATION', {
          xScaleFactor: 1,
          yScaleFactor: 1,
          containerWidth: element.width,
          containerHeight: element.height,
          position: {
            mx: 0.0,
            my: 0.0
          }
        });
        drawPath(p, textPathData, {
          stroke: getStrokeColor(element, defaultStrokeColor)
        });
        var text = getSemantic(element).text || '';
        renderLabel(p, text, {
          style: {
            fill: getLabelColor(element, defaultLabelColor, defaultStrokeColor)
          },
          box: element,
          align: 'left-top',
          padding: 5
        });
        return textElement;
      },
      'dmn:Association': function (p, element) {
        var semantic = getSemantic(element);
        var fill = getFillColor(element, defaultFillColor),
          stroke = getStrokeColor(element, defaultStrokeColor),
          attrs = {
            stroke: stroke,
            strokeDasharray: '0.5, 5',
            strokeLinecap: 'round',
            strokeLinejoin: 'round',
            fill: 'none'
          };
        if (semantic.associationDirection === 'One' || semantic.associationDirection === 'Both') {
          attrs.markerEnd = marker('association-end', fill, stroke);
        }
        if (semantic.associationDirection === 'Both') {
          attrs.markerStart = marker('association-start', fill, stroke);
        }
        return drawLine(p, element.waypoints, attrs);
      },
      'dmn:InformationRequirement': function (p, element) {
        var fill = getFillColor(element, defaultFillColor),
          stroke = getStrokeColor(element, defaultStrokeColor),
          attrs = {
            stroke: stroke,
            strokeWidth: 1,
            strokeLinecap: 'round',
            strokeLinejoin: 'round',
            markerEnd: marker('information-requirement-end', fill, stroke)
          };
        return drawLine(p, element.waypoints, attrs);
      },
      'dmn:KnowledgeRequirement': function (p, element) {
        var fill = getFillColor(element, defaultFillColor),
          stroke = getStrokeColor(element, defaultStrokeColor);
        var attrs = {
          stroke: stroke,
          strokeWidth: 1,
          strokeDasharray: 5,
          strokeLinecap: 'round',
          strokeLinejoin: 'round',
          markerEnd: marker('knowledge-requirement-end', fill, stroke)
        };
        return drawLine(p, element.waypoints, attrs);
      },
      'dmn:AuthorityRequirement': function (p, element) {
        var fill = getFillColor(element, defaultFillColor),
          stroke = getStrokeColor(element, defaultStrokeColor),
          attrs = {
            stroke: stroke,
            strokeWidth: 1.5,
            strokeDasharray: 5,
            strokeLinecap: 'round',
            strokeLinejoin: 'round',
            markerEnd: marker('authority-requirement-end', fill, stroke)
          };
        return drawLine(p, element.waypoints, attrs);
      }
    };

    // draw shape and connection //////////////////

    function drawShape(parent, element) {
      var h = handlers[element.type];
      if (!h) {
        return BaseRenderer.prototype.drawShape.apply(this, [parent, element]);
      } else {
        return h(parent, element);
      }
    }
    function drawConnection(parent, element) {
      var type = element.type;
      var h = handlers[type];
      if (!h) {
        return BaseRenderer.prototype.drawConnection.apply(this, [parent, element]);
      } else {
        return h(parent, element);
      }
    }
    function drawLine(p, waypoints, attrs) {
      attrs = computeStyle(attrs, ['no-fill'], {
        stroke: black,
        strokeWidth: 2,
        fill: 'none'
      });
      var line = createLine(waypoints, attrs);
      append(p, line);
      return line;
    }
    this.canRender = function (element) {
      return is(element, 'dmn:DMNElement') || is(element, 'dmn:InformationRequirement') || is(element, 'dmn:KnowledgeRequirement') || is(element, 'dmn:AuthorityRequirement');
    };
    this.drawShape = drawShape;
    this.drawConnection = drawConnection;
  }
  e(DrdRenderer, BaseRenderer);
  DrdRenderer.$inject = ['config.drdRenderer', 'eventBus', 'pathMap', 'styles', 'textRenderer', 'canvas'];

  // helper functions //////////////////////

  function getSemantic(element) {
    return element.businessObject;
  }
  function colorEscape(str) {
    // only allow characters and numbers
    return str.replace(/[^0-9a-zA-z]+/g, '_');
  }
  function getStrokeColor(element, defaultColor) {
    return defaultColor;
  }
  function getFillColor(element, defaultColor) {
    return defaultColor;
  }
  function getLabelColor(element, defaultColor, defaultStrokeColor) {
    return defaultColor || getStrokeColor(element, defaultStrokeColor);
  }

  /**
   * @typedef {import('../util/Types').Dimensions} Dimensions
   *
   * @typedef { {
   *   top: number;
   *   left: number;
   *   right: number;
   *   bottom: number;
   * } } Padding
   *
   * @typedef { number | Partial<Padding> } PaddingConfig
   *
   * @typedef { {
   *   horizontal: 'center' | 'left' | 'right';
   *   vertical: 'top' | 'middle';
   * } } Alignment
   *
   *  @typedef { 'center-middle' | 'center-top' } AlignmentConfig
   *
   * @typedef { Partial<{
   *   align: AlignmentConfig;
   *   style: Record<string, number | string>;
   *   padding: PaddingConfig;
   * }> } BaseTextConfig
   *
   * @typedef { BaseTextConfig & Partial<{
   *   size: Dimensions;
   * }> } TextConfig
   *
   * @typedef { BaseTextConfig & Partial<{
   *   box: Dimensions;
   *   fitBox: boolean;
   * }> } TextLayoutConfig
   *
   *  @typedef { Dimensions & {
   *  text: string;
   * } } LineDescriptor
   */

  var DEFAULT_BOX_PADDING = 0;
  var DEFAULT_LABEL_SIZE = {
    width: 150,
    height: 50
  };

  /**
   * @param {AlignmentConfig} align
   * @return {Alignment}
   */
  function parseAlign(align) {
    var parts = align.split('-');
    return {
      horizontal: parts[0] || 'center',
      vertical: parts[1] || 'top'
    };
  }

  /**
   * @param {PaddingConfig} padding
   *
   * @return {Padding}
   */
  function parsePadding(padding) {
    if (isObject$2(padding)) {
      return assign$4({
        top: 0,
        left: 0,
        right: 0,
        bottom: 0
      }, padding);
    } else {
      return {
        top: padding,
        left: padding,
        right: padding,
        bottom: padding
      };
    }
  }

  /**
   * @param {string} text
   * @param {SVGTextElement} fakeText
   *
   * @return {import('../util/Types').Dimensions}
   */
  function getTextBBox(text, fakeText) {
    fakeText.textContent = text;
    var textBBox;
    try {
      var bbox,
        emptyLine = text === '';

      // add dummy text, when line is empty to
      // determine correct height
      fakeText.textContent = emptyLine ? 'dummy' : text;
      textBBox = fakeText.getBBox();

      // take text rendering related horizontal
      // padding into account
      bbox = {
        width: textBBox.width + textBBox.x * 2,
        height: textBBox.height
      };
      if (emptyLine) {
        // correct width
        bbox.width = 0;
      }
      return bbox;
    } catch (e) {
      console.log(e);
      return {
        width: 0,
        height: 0
      };
    }
  }

  /**
   * Layout the next line and return the layouted element.
   *
   * Alters the lines passed.
   *
   * @param {string[]} lines
   * @param {number} maxWidth
   * @param {SVGTextElement} fakeText
   *
   * @return {LineDescriptor} the line descriptor
   */
  function layoutNext(lines, maxWidth, fakeText) {
    var originalLine = lines.shift(),
      fitLine = originalLine;
    var textBBox;
    for (;;) {
      textBBox = getTextBBox(fitLine, fakeText);
      textBBox.width = fitLine ? textBBox.width : 0;

      // try to fit
      if (fitLine === ' ' || fitLine === '' || textBBox.width < Math.round(maxWidth) || fitLine.length < 2) {
        return fit(lines, fitLine, originalLine, textBBox);
      }
      fitLine = shortenLine(fitLine, textBBox.width, maxWidth);
    }
  }

  /**
   * @param {string[]} lines
   * @param {string} fitLine
   * @param {string} originalLine
   * @param {Dimensions} textBBox
   *
   * @return {LineDescriptor}
   */
  function fit(lines, fitLine, originalLine, textBBox) {
    if (fitLine.length < originalLine.length) {
      var remainder = originalLine.slice(fitLine.length).trim();
      lines.unshift(remainder);
    }
    return {
      width: textBBox.width,
      height: textBBox.height,
      text: fitLine
    };
  }
  var SOFT_BREAK = '\u00AD';

  /**
   * Shortens a line based on spacing and hyphens.
   * Returns the shortened result on success.
   *
   * @param {string} line
   * @param {number} maxLength the maximum characters of the string
   *
   * @return {string} the shortened string
   */
  function semanticShorten(line, maxLength) {
    var parts = line.split(/(\s|-|\u00AD)/g),
      part,
      shortenedParts = [],
      length = 0;

    // try to shorten via break chars
    if (parts.length > 1) {
      while (part = parts.shift()) {
        if (part.length + length < maxLength) {
          shortenedParts.push(part);
          length += part.length;
        } else {
          // remove previous part, too if hyphen does not fit anymore
          if (part === '-' || part === SOFT_BREAK) {
            shortenedParts.pop();
          }
          break;
        }
      }
    }
    var last = shortenedParts[shortenedParts.length - 1];

    // translate trailing soft break to actual hyphen
    if (last && last === SOFT_BREAK) {
      shortenedParts[shortenedParts.length - 1] = '-';
    }
    return shortenedParts.join('');
  }

  /**
   * @param {string} line
   * @param {number} width
   * @param {number} maxWidth
   *
   * @return {string}
   */
  function shortenLine(line, width, maxWidth) {
    var length = Math.max(line.length * (maxWidth / width), 1);

    // try to shorten semantically (i.e. based on spaces and hyphens)
    var shortenedLine = semanticShorten(line, length);
    if (!shortenedLine) {
      // force shorten by cutting the long word
      shortenedLine = line.slice(0, Math.max(Math.round(length - 1), 1));
    }
    return shortenedLine;
  }

  /**
   * @return {SVGSVGElement}
   */
  function getHelperSvg() {
    var helperSvg = document.getElementById('helper-svg');
    if (!helperSvg) {
      helperSvg = create$2('svg');
      attr(helperSvg, {
        id: 'helper-svg'
      });
      assign(helperSvg, {
        visibility: 'hidden',
        position: 'fixed',
        width: 0,
        height: 0
      });
      document.body.appendChild(helperSvg);
    }
    return helperSvg;
  }

  /**
   * Creates a new label utility
   *
   * @param {TextConfig} [config]
   */
  function Text$1(config) {
    this._config = assign$4({}, {
      size: DEFAULT_LABEL_SIZE,
      padding: DEFAULT_BOX_PADDING,
      style: {},
      align: 'center-top'
    }, config || {});
  }

  /**
   * Returns the layouted text as an SVG element.
   *
   * @param {string} text
   * @param {TextLayoutConfig} options
   *
   * @return {SVGElement}
   */
  Text$1.prototype.createText = function (text, options) {
    return this.layoutText(text, options).element;
  };

  /**
   * Returns a labels layouted dimensions.
   *
   * @param {string} text to layout
   * @param {TextLayoutConfig} options
   *
   * @return {Dimensions}
   */
  Text$1.prototype.getDimensions = function (text, options) {
    return this.layoutText(text, options).dimensions;
  };

  /**
   * Creates and returns a label and its bounding box.
   *
   * @param {string} text the text to render on the label
   * @param {TextLayoutConfig} options
   *
   * @return { {
   *   element: SVGElement,
   *   dimensions: Dimensions
   * } }
   */
  Text$1.prototype.layoutText = function (text, options) {
    var box = assign$4({}, this._config.size, options.box),
      style = assign$4({}, this._config.style, options.style),
      align = parseAlign(options.align || this._config.align),
      padding = parsePadding(options.padding !== undefined ? options.padding : this._config.padding),
      fitBox = options.fitBox || false;
    var lineHeight = getLineHeight(style);

    // we split text by lines and normalize
    // {soft break} + {line break} => { line break }
    var lines = text.split(/\u00AD?\r?\n/),
      layouted = [];
    var maxWidth = box.width - padding.left - padding.right;

    // ensure correct rendering by attaching helper text node to invisible SVG
    var helperText = create$2('text');
    attr(helperText, {
      x: 0,
      y: 0
    });
    attr(helperText, style);
    var helperSvg = getHelperSvg();
    append(helperSvg, helperText);
    while (lines.length) {
      layouted.push(layoutNext(lines, maxWidth, helperText));
    }
    if (align.vertical === 'middle') {
      padding.top = padding.bottom = 0;
    }
    var totalHeight = reduce(layouted, function (sum, line, idx) {
      return sum + (lineHeight || line.height);
    }, 0) + padding.top + padding.bottom;
    var maxLineWidth = reduce(layouted, function (sum, line, idx) {
      return line.width > sum ? line.width : sum;
    }, 0);

    // the y position of the next line
    var y = padding.top;
    if (align.vertical === 'middle') {
      y += (box.height - totalHeight) / 2;
    }

    // magic number initial offset
    y -= (lineHeight || layouted[0].height) / 4;
    var textElement = create$2('text');
    attr(textElement, style);

    // layout each line taking into account that parent
    // shape might resize to fit text size
    forEach$3(layouted, function (line) {
      var x;
      y += lineHeight || line.height;
      switch (align.horizontal) {
        case 'left':
          x = padding.left;
          break;
        case 'right':
          x = (fitBox ? maxLineWidth : maxWidth) - padding.right - line.width;
          break;
        default:
          // aka center
          x = Math.max(((fitBox ? maxLineWidth : maxWidth) - line.width) / 2 + padding.left, 0);
      }
      var tspan = create$2('tspan');
      attr(tspan, {
        x: x,
        y: y
      });
      tspan.textContent = line.text;
      append(textElement, tspan);
    });
    remove$3(helperText);
    var dimensions = {
      width: maxLineWidth,
      height: totalHeight
    };
    return {
      dimensions: dimensions,
      element: textElement
    };
  };
  function getLineHeight(style) {
    if ('fontSize' in style && 'lineHeight' in style) {
      return style.lineHeight * parseInt(style.fontSize, 10);
    }
  }

  var DEFAULT_FONT_SIZE = 12;
  var LINE_HEIGHT_RATIO = 1.2;
  var MIN_TEXT_ANNOTATION_HEIGHT = 30;
  function TextRenderer(config) {
    var defaultStyle = assign$4({
      fontFamily: 'Arial, sans-serif',
      fontSize: DEFAULT_FONT_SIZE,
      fontWeight: 'normal',
      lineHeight: LINE_HEIGHT_RATIO
    }, config && config.defaultStyle || {});
    var fontSize = parseInt(defaultStyle.fontSize, 10) - 1;
    var externalStyle = assign$4({}, defaultStyle, {
      fontSize: fontSize
    }, config && config.externalStyle || {});
    var textUtil = new Text$1({
      style: defaultStyle
    });

    /**
     * Get the new bounds of an externally rendered,
     * layouted label.
     *
     * @param  {Bounds} bounds
     * @param  {string} text
     *
     * @return {Bounds}
     */
    this.getExternalLabelBounds = function (bounds, text) {
      var layoutedDimensions = textUtil.getDimensions(text, {
        box: {
          width: 90,
          height: 30,
          x: bounds.width / 2 + bounds.x,
          y: bounds.height / 2 + bounds.y
        },
        style: externalStyle
      });

      // resize label shape to fit label text
      return {
        x: Math.round(bounds.x + bounds.width / 2 - layoutedDimensions.width / 2),
        y: Math.round(bounds.y),
        width: Math.ceil(layoutedDimensions.width),
        height: Math.ceil(layoutedDimensions.height)
      };
    };

    /**
     * Get the new bounds of text annotation.
     *
     * @param  {Bounds} bounds
     * @param  {string} text
     *
     * @return {Bounds}
     */
    this.getTextAnnotationBounds = function (bounds, text) {
      var layoutedDimensions = textUtil.getDimensions(text, {
        box: bounds,
        style: defaultStyle,
        align: 'left-top',
        padding: 5
      });
      return {
        x: bounds.x,
        y: bounds.y,
        width: bounds.width,
        height: Math.max(MIN_TEXT_ANNOTATION_HEIGHT, Math.round(layoutedDimensions.height))
      };
    };

    /**
     * Create a layouted text element.
     *
     * @param {string} text
     * @param {Object} [options]
     *
     * @return {SVGElement} rendered text
     */
    this.createText = function (text, options) {
      return textUtil.createText(text, options || {});
    };

    /**
     * Get default text style.
     */
    this.getDefaultStyle = function () {
      return defaultStyle;
    };

    /**
     * Get the external text style.
     */
    this.getExternalStyle = function () {
      return externalStyle;
    };
  }
  TextRenderer.$inject = ['config.textRenderer'];

  /* eslint-disable max-len */

  /**
   * Map containing SVG paths needed by BpmnRenderer.
   */

  function PathMap() {
    /**
     * Contains a map of path elements
     *
     * <h1>Path definition</h1>
     * A parameterized path is defined like this:
     * <pre>
     * 'GATEWAY_PARALLEL': {
     *   d: 'm {mx},{my} {e.x0},0 0,{e.x1} {e.x1},0 0,{e.y0} -{e.x1},0 0,{e.y1} ' +
            '-{e.x0},0 0,-{e.y1} -{e.x1},0 0,-{e.y0} {e.x1},0 z',
     *   height: 17.5,
     *   width:  17.5,
     *   heightElements: [2.5, 7.5],
     *   widthElements: [2.5, 7.5]
     * }
     * </pre>
     * <p>It's important to specify a correct <b>height and width</b> for the path as the scaling
     * is based on the ratio between the specified height and width in this object and the
     * height and width that is set as scale target (Note x,y coordinates will be scaled with
     * individual ratios).</p>
     * <p>The '<b>heightElements</b>' and '<b>widthElements</b>' array must contain the values that will be scaled.
     * The scaling is based on the computed ratios.
     * Coordinates on the y axis should be in the <b>heightElement</b>'s array, they will be scaled using
     * the computed ratio coefficient.
     * In the parameterized path the scaled values can be accessed through the 'e' object in {} brackets.
     *   <ul>
     *    <li>The values for the y axis can be accessed in the path string using {e.y0}, {e.y1}, ....</li>
     *    <li>The values for the x axis can be accessed in the path string using {e.x0}, {e.x1}, ....</li>
     *   </ul>
     *   The numbers x0, x1 respectively y0, y1, ... map to the corresponding array index.
     * </p>
      m1,1
      l 0,55.3
      c 29.8,19.7 48.4,-4.2 67.2,-6.7
      c 12.2,-2.3 19.8,1.6 30.8,6.2
      l 0,-54.6
      z
      */
    this.pathMap = {
      'KNOWLEDGE_SOURCE': {
        d: 'm {mx},{my} ' + 'l 0,{e.y0} ' + 'c {e.x0},{e.y1} {e.x1},-{e.y2} {e.x2},-{e.y3} ' + 'c {e.x3},-{e.y4} {e.x4},{e.y5} {e.x5},{e.y6} ' + 'l 0,-{e.y7}z',
        width: 100,
        height: 65,
        widthElements: [29.8, 48.4, 67.2, 12.2, 19.8, 30.8],
        heightElements: [55.3, 19.7, 4.2, 6.7, 2.3, 1.6, 6.2, 54.6]
      },
      'BUSINESS_KNOWLEDGE_MODEL': {
        d: 'm {mx},{my} l {e.x0},-{e.y0} l {e.x1},0 l 0,{e.y1} l -{e.x2},{e.y2} l -{e.x3},0z',
        width: 125,
        height: 45,
        widthElements: [13.8, 109.2, 13.8, 109.1],
        heightElements: [13.2, 29.8, 13.2]
      },
      'TEXT_ANNOTATION': {
        d: 'm {mx}, {my} m 10,0 l -10,0 l 0,{e.y0} l 10,0',
        width: 10,
        height: 30,
        widthElements: [10],
        heightElements: [30]
      }
    };
    this.getRawPath = function getRawPath(pathId) {
      return this.pathMap[pathId].d;
    };

    /**
     * Scales the path to the given height and width.
     * <h1>Use case</h1>
     * <p>Use case is to scale the content of elements (event, gateways) based
     * on the element bounding box's size.
     * </p>
     * <h1>Why not transform</h1>
     * <p>Scaling a path with transform() will also scale the stroke and IE does not support
     * the option 'non-scaling-stroke' to prevent this.
     * Also there are use cases where only some parts of a path should be
     * scaled.</p>
     *
     * @param {string} pathId The ID of the path.
     * @param {Object} param <p>
     *   Example param object scales the path to 60% size of the container (data.width, data.height).
     *   <pre>
     *   {
     *     xScaleFactor: 0.6,
     *     yScaleFactor:0.6,
     *     containerWidth: data.width,
     *     containerHeight: data.height,
     *     position: {
     *       mx: 0.46,
     *       my: 0.2,
     *     }
     *   }
     *   </pre>
     *   <ul>
     *    <li>targetpathwidth = xScaleFactor * containerWidth</li>
     *    <li>targetpathheight = yScaleFactor * containerHeight</li>
     *    <li>Position is used to set the starting coordinate of the path. M is computed:
      *    <ul>
      *      <li>position.x * containerWidth</li>
      *      <li>position.y * containerHeight</li>
      *    </ul>
      *    Center of the container <pre> position: {
     *       mx: 0.5,
     *       my: 0.5,
     *     }</pre>
     *     Upper left corner of the container
     *     <pre> position: {
     *       mx: 0.0,
     *       my: 0.0,
     *     }</pre>
     *    </li>
     *   </ul>
     * </p>
     *
     */
    this.getScaledPath = function getScaledPath(pathId, param) {
      var rawPath = this.pathMap[pathId];

      // positioning
      // compute the start point of the path
      var mx, my;
      if (param.abspos) {
        mx = param.abspos.x;
        my = param.abspos.y;
      } else {
        mx = param.containerWidth * param.position.mx;
        my = param.containerHeight * param.position.my;
      }
      var coordinates = {}; // map for the scaled coordinates
      if (param.position) {
        // path
        var heightRatio = param.containerHeight / rawPath.height * param.yScaleFactor;
        var widthRatio = param.containerWidth / rawPath.width * param.xScaleFactor;

        // Apply height ratio
        for (var heightIndex = 0; heightIndex < rawPath.heightElements.length; heightIndex++) {
          coordinates['y' + heightIndex] = rawPath.heightElements[heightIndex] * heightRatio;
        }

        // Apply width ratio
        for (var widthIndex = 0; widthIndex < rawPath.widthElements.length; widthIndex++) {
          coordinates['x' + widthIndex] = rawPath.widthElements[widthIndex] * widthRatio;
        }
      }

      // Apply value to raw path
      var path = format(rawPath.d, {
        mx: mx,
        my: my,
        e: coordinates
      });
      return path;
    };
  }

  // helpers //////////////////////

  // copied and adjusted from https://github.com/adobe-webplatform/Snap.svg/blob/master/src/svg.js
  var tokenRegex = /\{([^{}]+)\}/g,
    objNotationRegex = /(?:(?:^|\.)(.+?)(?=\[|\.|$|\()|\[('|")(.+?)\2\])(\(\))?/g; // matches .xxxxx or ["xxxxx"] to run over object properties

  function replacer(all, key, obj) {
    var res = obj;
    key.replace(objNotationRegex, function (all, name, quote, quotedName, isFunc) {
      name = name || quotedName;
      if (res) {
        if (name in res) {
          res = res[name];
        }
        typeof res == 'function' && isFunc && (res = res());
      }
    });
    res = (res == null || res == obj ? all : res) + '';
    return res;
  }
  function format(str, obj) {
    return String(str).replace(tokenRegex, function (all, key) {
      return replacer(all, key, obj);
    });
  }

  var DrawModule = {
    __init__: ['drdRenderer'],
    drdRenderer: ['type', DrdRenderer],
    textRenderer: ['type', TextRenderer],
    pathMap: ['type', PathMap]
  };

  function DrdImporter(eventBus, canvas, elementFactory, elementRegistry) {
    this._eventBus = eventBus;
    this._canvas = canvas;
    this._elementRegistry = elementRegistry;
    this._elementFactory = elementFactory;
  }
  DrdImporter.$inject = ['eventBus', 'canvas', 'elementFactory', 'elementRegistry'];
  DrdImporter.prototype.root = function (semantic) {
    var element = this._elementFactory.createRoot(elementData$1(semantic));
    this._canvas.setRootElement(element);
    return element;
  };

  /**
   * Add drd element (semantic) to the canvas.
   */
  DrdImporter.prototype.add = function (semantic) {
    var elementFactory = this._elementFactory,
      canvas = this._canvas,
      eventBus = this._eventBus,
      di = semantic.di;
    var element, waypoints, source, target, elementDefinition, bounds;
    if (di.$instanceOf('dmndi:DMNShape')) {
      bounds = di.bounds;
      elementDefinition = elementData$1(semantic, {
        x: Math.round(bounds.x),
        y: Math.round(bounds.y),
        width: Math.round(bounds.width),
        height: Math.round(bounds.height)
      });
      element = elementFactory.createShape(elementDefinition);
      canvas.addShape(element);
      eventBus.fire('drdElement.added', {
        element: element,
        di: di
      });
    } else if (di.$instanceOf('dmndi:DMNEdge')) {
      waypoints = collectWaypoints(di);
      source = this._getSource(semantic);
      target = this._getTarget(semantic);
      if (source && target) {
        elementDefinition = elementData$1(semantic, {
          hidden: false,
          source: source,
          target: target,
          waypoints: waypoints
        });
        element = elementFactory.createConnection(elementDefinition);
        canvas.addConnection(element);
        eventBus.fire('drdElement.added', {
          element: element,
          di: di
        });
      }
    } else {
      throw new Error('unknown di for element ' + semantic.id);
    }
    return element;
  };
  DrdImporter.prototype._getSource = function (semantic) {
    var href, elementReference;
    if (is(semantic, 'dmn:Association')) {
      elementReference = semantic.sourceRef;
    } else if (is(semantic, 'dmn:InformationRequirement')) {
      elementReference = semantic.requiredDecision || semantic.requiredInput;
    } else if (is(semantic, 'dmn:KnowledgeRequirement')) {
      elementReference = semantic.requiredKnowledge;
    } else if (is(semantic, 'dmn:AuthorityRequirement')) {
      elementReference = semantic.requiredDecision || semantic.requiredInput || semantic.requiredAuthority;
    }
    if (elementReference) {
      href = elementReference.href;
    }
    if (href) {
      return this._getShape(getIdFromHref(href));
    }
  };
  DrdImporter.prototype._getTarget = function (semantic) {
    if (is(semantic, 'dmn:Association')) {
      return semantic.targetRef && this._getShape(getIdFromHref(semantic.targetRef.href));
    }
    return this._getShape(semantic.$parent.id);
  };
  DrdImporter.prototype._getShape = function (id) {
    return this._elementRegistry.get(id);
  };

  // helper /////
  function elementData$1(semantic, attrs) {
    return assign$4({
      id: semantic.id,
      type: semantic.$type,
      businessObject: semantic
    }, attrs);
  }
  function collectWaypoints(edge) {
    var waypoints = edge.waypoint;
    if (waypoints) {
      return map$1(waypoints, function (waypoint) {
        var position = {
          x: waypoint.x,
          y: waypoint.y
        };
        return assign$4({
          original: position
        }, position);
      });
    }
  }
  function getIdFromHref(href) {
    return href.split('#').pop();
  }

  var ImportModule = {
    drdImporter: ['type', DrdImporter]
  };

  var CoreModule$1 = {
    __depends__: [DrawModule, ImportModule]
  };

  /**
   * @typedef { {
   *   [key: string]: string;
   * } } TranslateReplacements
   */

  /**
   * A simple translation stub to be used for multi-language support
   * in diagrams. Can be easily replaced with a more sophisticated
   * solution.
   *
   * @example
   *
   * ```javascript
   * // use it inside any diagram component by injecting `translate`.
   *
   * function MyService(translate) {
   *   alert(translate('HELLO {you}', { you: 'You!' }));
   * }
   * ```
   *
   * @param {string} template to interpolate
   * @param {TranslateReplacements} [replacements] a map with substitutes
   *
   * @return {string} the translated string
   */
  function translate(template, replacements) {
    replacements = replacements || {};
    return template.replace(/{([^}]+)}/g, function (_, key) {
      return replacements[key] || '{' + key + '}';
    });
  }

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var TranslateModule = {
    translate: ['value', translate]
  };

  /**
   * @typedef {import('../util/Types').Point} Point
   */


  /**
   * @param {import('../core/EventBus').Event} event
   *
   * @return {Event}
   */
  function getOriginal(event) {
    return event.originalEvent || event.srcEvent;
  }

  /**
   * @param {Event} event
   *
   * @return {Point|null}
   */
  function toPoint(event) {
    if (event.pointers && event.pointers.length) {
      event = event.pointers[0];
    }
    if (event.touches && event.touches.length) {
      event = event.touches[0];
    }
    return event ? {
      x: event.clientX,
      y: event.clientY
    } : null;
  }

  function isMac() {
    return /mac/i.test(navigator.platform);
  }

  /**
   * @param {MouseEvent} event
   * @param {string} button
   *
   * @return {boolean}
   */
  function isButton(event, button) {
    return (getOriginal(event) || event).button === button;
  }

  /**
   * @param {MouseEvent} event
   *
   * @return {boolean}
   */
  function isPrimaryButton(event) {
    // button === 0 -> left áka primary mouse button
    return isButton(event, 0);
  }

  /**
   * @param {MouseEvent} event
   *
   * @return {boolean}
   */
  function isAuxiliaryButton(event) {
    // button === 1 -> auxiliary áka wheel button
    return isButton(event, 1);
  }

  /**
   * @param {MouseEvent} event
   *
   * @return {boolean}
   */
  function hasSecondaryModifier(event) {
    var originalEvent = getOriginal(event) || event;
    return isPrimaryButton(event) && originalEvent.shiftKey;
  }

  /**
   * @typedef {import('../../model/Types').Element} Element
   *
   * @typedef {import('../../core/ElementRegistry').default} ElementRegistry
   * @typedef {import('../../core/EventBus').default} EventBus
   * @typedef {import('../../draw/Styles').default} Styles
   *
   * @typedef {import('../../util/Types').Point} Point
   */

  function allowAll(event) {
    return true;
  }
  function allowPrimaryAndAuxiliary(event) {
    return isPrimaryButton(event) || isAuxiliaryButton(event);
  }
  var LOW_PRIORITY$3 = 500;

  /**
   * A plugin that provides interaction events for diagram elements.
   *
   * It emits the following events:
   *
   *   * element.click
   *   * element.contextmenu
   *   * element.dblclick
   *   * element.hover
   *   * element.mousedown
   *   * element.mousemove
   *   * element.mouseup
   *   * element.out
   *
   * Each event is a tuple { element, gfx, originalEvent }.
   *
   * Canceling the event via Event#preventDefault()
   * prevents the original DOM operation.
   *
   * @param {EventBus} eventBus
   * @param {ElementRegistry} elementRegistry
   * @param {Styles} styles
   */
  function InteractionEvents(eventBus, elementRegistry, styles) {
    var self = this;

    /**
     * Fire an interaction event.
     *
     * @param {string} type local event name, e.g. element.click.
     * @param {MouseEvent|TouchEvent} event native event
     * @param {Element} [element] the diagram element to emit the event on;
     *                                   defaults to the event target
     */
    function fire(type, event, element) {
      if (isIgnored(type, event)) {
        return;
      }
      var target, gfx, returnValue;
      if (!element) {
        target = event.delegateTarget || event.target;
        if (target) {
          gfx = target;
          element = elementRegistry.get(gfx);
        }
      } else {
        gfx = elementRegistry.getGraphics(element);
      }
      if (!gfx || !element) {
        return;
      }
      returnValue = eventBus.fire(type, {
        element: element,
        gfx: gfx,
        originalEvent: event
      });
      if (returnValue === false) {
        event.stopPropagation();
        event.preventDefault();
      }
    }

    // TODO(nikku): document this
    var handlers = {};
    function mouseHandler(localEventName) {
      return handlers[localEventName];
    }
    function isIgnored(localEventName, event) {
      var filter = ignoredFilters[localEventName] || isPrimaryButton;

      // only react on left mouse button interactions
      // except for interaction events that are enabled
      // for secundary mouse button
      return !filter(event);
    }
    var bindings = {
      click: 'element.click',
      contextmenu: 'element.contextmenu',
      dblclick: 'element.dblclick',
      mousedown: 'element.mousedown',
      mousemove: 'element.mousemove',
      mouseover: 'element.hover',
      mouseout: 'element.out',
      mouseup: 'element.mouseup'
    };
    var ignoredFilters = {
      'element.contextmenu': allowAll,
      'element.mousedown': allowPrimaryAndAuxiliary,
      'element.mouseup': allowPrimaryAndAuxiliary,
      'element.click': allowPrimaryAndAuxiliary,
      'element.dblclick': allowPrimaryAndAuxiliary
    };

    // manual event trigger //////////

    /**
     * Trigger an interaction event (based on a native dom event)
     * on the target shape or connection.
     *
     * @param {string} eventName the name of the triggered DOM event
     * @param {MouseEvent|TouchEvent} event
     * @param {Element} targetElement
     */
    function triggerMouseEvent(eventName, event, targetElement) {
      // i.e. element.mousedown...
      var localEventName = bindings[eventName];
      if (!localEventName) {
        throw new Error('unmapped DOM event name <' + eventName + '>');
      }
      return fire(localEventName, event, targetElement);
    }
    var ELEMENT_SELECTOR = 'svg, .djs-element';

    // event handling ///////

    function registerEvent(node, event, localEvent, ignoredFilter) {
      var handler = handlers[localEvent] = function (event) {
        fire(localEvent, event);
      };
      if (ignoredFilter) {
        ignoredFilters[localEvent] = ignoredFilter;
      }
      handler.$delegate = delegate.bind(node, ELEMENT_SELECTOR, event, handler);
    }
    function unregisterEvent(node, event, localEvent) {
      var handler = mouseHandler(localEvent);
      if (!handler) {
        return;
      }
      delegate.unbind(node, event, handler.$delegate);
    }
    function registerEvents(svg) {
      forEach$3(bindings, function (val, key) {
        registerEvent(svg, key, val);
      });
    }
    function unregisterEvents(svg) {
      forEach$3(bindings, function (val, key) {
        unregisterEvent(svg, key, val);
      });
    }
    eventBus.on('canvas.destroy', function (event) {
      unregisterEvents(event.svg);
    });
    eventBus.on('canvas.init', function (event) {
      registerEvents(event.svg);
    });

    // hit box updating ////////////////

    eventBus.on(['shape.added', 'connection.added'], function (event) {
      var element = event.element,
        gfx = event.gfx;
      eventBus.fire('interactionEvents.createHit', {
        element: element,
        gfx: gfx
      });
    });

    // Update djs-hit on change.
    // A low priortity is necessary, because djs-hit of labels has to be updated
    // after the label bounds have been updated in the renderer.
    eventBus.on(['shape.changed', 'connection.changed'], LOW_PRIORITY$3, function (event) {
      var element = event.element,
        gfx = event.gfx;
      eventBus.fire('interactionEvents.updateHit', {
        element: element,
        gfx: gfx
      });
    });
    eventBus.on('interactionEvents.createHit', LOW_PRIORITY$3, function (event) {
      var element = event.element,
        gfx = event.gfx;
      self.createDefaultHit(element, gfx);
    });
    eventBus.on('interactionEvents.updateHit', function (event) {
      var element = event.element,
        gfx = event.gfx;
      self.updateDefaultHit(element, gfx);
    });

    // hit styles ////////////

    var STROKE_HIT_STYLE = createHitStyle('djs-hit djs-hit-stroke');
    var CLICK_STROKE_HIT_STYLE = createHitStyle('djs-hit djs-hit-click-stroke');
    var ALL_HIT_STYLE = createHitStyle('djs-hit djs-hit-all');
    var NO_MOVE_HIT_STYLE = createHitStyle('djs-hit djs-hit-no-move');
    var HIT_TYPES = {
      'all': ALL_HIT_STYLE,
      'click-stroke': CLICK_STROKE_HIT_STYLE,
      'stroke': STROKE_HIT_STYLE,
      'no-move': NO_MOVE_HIT_STYLE
    };
    function createHitStyle(classNames, attrs) {
      attrs = assign$4({
        stroke: 'white',
        strokeWidth: 15
      }, attrs || {});
      return styles.cls(classNames, ['no-fill', 'no-border'], attrs);
    }

    // style helpers ///////////////

    function applyStyle(hit, type) {
      var attrs = HIT_TYPES[type];
      if (!attrs) {
        throw new Error('invalid hit type <' + type + '>');
      }
      attr(hit, attrs);
      return hit;
    }
    function appendHit(gfx, hit) {
      append(gfx, hit);
    }

    // API

    /**
     * Remove hints on the given graphics.
     *
     * @param {SVGElement} gfx
     */
    this.removeHits = function (gfx) {
      var hits = all('.djs-hit', gfx);
      forEach$3(hits, remove$3);
    };

    /**
     * Create default hit for the given element.
     *
     * @param {Element} element
     * @param {SVGElement} gfx
     *
     * @return {SVGElement} created hit
     */
    this.createDefaultHit = function (element, gfx) {
      var waypoints = element.waypoints,
        isFrame = element.isFrame,
        boxType;
      if (waypoints) {
        return this.createWaypointsHit(gfx, waypoints);
      } else {
        boxType = isFrame ? 'stroke' : 'all';
        return this.createBoxHit(gfx, boxType, {
          width: element.width,
          height: element.height
        });
      }
    };

    /**
     * Create hits for the given waypoints.
     *
     * @param {SVGElement} gfx
     * @param {Point[]} waypoints
     *
     * @return {SVGElement}
     */
    this.createWaypointsHit = function (gfx, waypoints) {
      var hit = createLine(waypoints);
      applyStyle(hit, 'stroke');
      appendHit(gfx, hit);
      return hit;
    };

    /**
     * Create hits for a box.
     *
     * @param {SVGElement} gfx
     * @param {string} type
     * @param {Object} attrs
     *
     * @return {SVGElement}
     */
    this.createBoxHit = function (gfx, type, attrs) {
      attrs = assign$4({
        x: 0,
        y: 0
      }, attrs);
      var hit = create$2('rect');
      applyStyle(hit, type);
      attr(hit, attrs);
      appendHit(gfx, hit);
      return hit;
    };

    /**
     * Update default hit of the element.
     *
     * @param {Element} element
     * @param {SVGElement} gfx
     *
     * @return {SVGElement} updated hit
     */
    this.updateDefaultHit = function (element, gfx) {
      var hit = query('.djs-hit', gfx);
      if (!hit) {
        return;
      }
      if (element.waypoints) {
        updateLine(hit, element.waypoints);
      } else {
        attr(hit, {
          width: element.width,
          height: element.height
        });
      }
      return hit;
    };
    this.fire = fire;
    this.triggerMouseEvent = triggerMouseEvent;
    this.mouseHandler = mouseHandler;
    this.registerEvent = registerEvent;
    this.unregisterEvent = unregisterEvent;
  }
  InteractionEvents.$inject = ['eventBus', 'elementRegistry', 'styles'];

  /**
   * An event indicating that the mouse hovered over an element
   *
   * @event element.hover
   *
   * @type {Object}
   * @property {Element} element
   * @property {SVGElement} gfx
   * @property {Event} originalEvent
   */

  /**
   * An event indicating that the mouse has left an element
   *
   * @event element.out
   *
   * @type {Object}
   * @property {Element} element
   * @property {SVGElement} gfx
   * @property {Event} originalEvent
   */

  /**
   * An event indicating that the mouse has clicked an element
   *
   * @event element.click
   *
   * @type {Object}
   * @property {Element} element
   * @property {SVGElement} gfx
   * @property {Event} originalEvent
   */

  /**
   * An event indicating that the mouse has double clicked an element
   *
   * @event element.dblclick
   *
   * @type {Object}
   * @property {Element} element
   * @property {SVGElement} gfx
   * @property {Event} originalEvent
   */

  /**
   * An event indicating that the mouse has gone down on an element.
   *
   * @event element.mousedown
   *
   * @type {Object}
   * @property {Element} element
   * @property {SVGElement} gfx
   * @property {Event} originalEvent
   */

  /**
   * An event indicating that the mouse has gone up on an element.
   *
   * @event element.mouseup
   *
   * @type {Object}
   * @property {Element} element
   * @property {SVGElement} gfx
   * @property {Event} originalEvent
   */

  /**
   * An event indicating that the context menu action is triggered
   * via mouse or touch controls.
   *
   * @event element.contextmenu
   *
   * @type {Object}
   * @property {Element} element
   * @property {SVGElement} gfx
   * @property {Event} originalEvent
   */

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var InteractionEventsModule = {
    __init__: ['interactionEvents'],
    interactionEvents: ['type', InteractionEvents]
  };

  /**
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../../core/EventBus').default} EventBus
   */

  /**
   * A service that offers the current selection in a diagram.
   * Offers the api to control the selection, too.
   *
   * @param {EventBus} eventBus
   * @param {Canvas} canvas
   */
  function Selection(eventBus, canvas) {
    this._eventBus = eventBus;
    this._canvas = canvas;

    /**
     * @type {Object[]}
     */
    this._selectedElements = [];
    var self = this;
    eventBus.on(['shape.remove', 'connection.remove'], function (e) {
      var element = e.element;
      self.deselect(element);
    });
    eventBus.on(['diagram.clear', 'root.set'], function (e) {
      self.select(null);
    });
  }
  Selection.$inject = ['eventBus', 'canvas'];

  /**
   * Deselect an element.
   *
   * @param {Object} element The element to deselect.
   */
  Selection.prototype.deselect = function (element) {
    var selectedElements = this._selectedElements;
    var idx = selectedElements.indexOf(element);
    if (idx !== -1) {
      var oldSelection = selectedElements.slice();
      selectedElements.splice(idx, 1);
      this._eventBus.fire('selection.changed', {
        oldSelection: oldSelection,
        newSelection: selectedElements
      });
    }
  };

  /**
   * Get the selected elements.
   *
   * @return {Object[]} The selected elements.
   */
  Selection.prototype.get = function () {
    return this._selectedElements;
  };

  /**
   * Check whether an element is selected.
   *
   * @param {Object} element The element.
   *
   * @return {boolean} Whether the element is selected.
   */
  Selection.prototype.isSelected = function (element) {
    return this._selectedElements.indexOf(element) !== -1;
  };

  /**
   * Select one or many elements.
   *
   * @param {Object|Object[]} elements The element(s) to select.
   * @param {boolean} [add] Whether to add the element(s) to the selected elements.
   * Defaults to `false`.
   */
  Selection.prototype.select = function (elements, add) {
    var selectedElements = this._selectedElements,
      oldSelection = selectedElements.slice();
    if (!isArray$5(elements)) {
      elements = elements ? [elements] : [];
    }
    var canvas = this._canvas;
    var rootElement = canvas.getRootElement();
    elements = elements.filter(function (element) {
      var elementRoot = canvas.findRoot(element);
      return rootElement === elementRoot;
    });

    // selection may be cleared by passing an empty array or null
    // to the method
    if (add) {
      forEach$3(elements, function (element) {
        if (selectedElements.indexOf(element) !== -1) {
          // already selected
          return;
        } else {
          selectedElements.push(element);
        }
      });
    } else {
      this._selectedElements = selectedElements = elements.slice();
    }
    this._eventBus.fire('selection.changed', {
      oldSelection: oldSelection,
      newSelection: selectedElements
    });
  };

  /**
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../../core/EventBus').default} EventBus
   */

  var MARKER_HOVER = 'hover',
    MARKER_SELECTED = 'selected';

  /**
   * A plugin that adds a visible selection UI to shapes and connections
   * by appending the <code>hover</code> and <code>selected</code> classes to them.
   *
   * @class
   *
   * Makes elements selectable, too.
   *
   * @param {Canvas} canvas
   * @param {EventBus} eventBus
   */
  function SelectionVisuals(canvas, eventBus) {
    this._canvas = canvas;
    function addMarker(e, cls) {
      canvas.addMarker(e, cls);
    }
    function removeMarker(e, cls) {
      canvas.removeMarker(e, cls);
    }
    eventBus.on('element.hover', function (event) {
      addMarker(event.element, MARKER_HOVER);
    });
    eventBus.on('element.out', function (event) {
      removeMarker(event.element, MARKER_HOVER);
    });
    eventBus.on('selection.changed', function (event) {
      function deselect(s) {
        removeMarker(s, MARKER_SELECTED);
      }
      function select(s) {
        addMarker(s, MARKER_SELECTED);
      }
      var oldSelection = event.oldSelection,
        newSelection = event.newSelection;
      forEach$3(oldSelection, function (e) {
        if (newSelection.indexOf(e) === -1) {
          deselect(e);
        }
      });
      forEach$3(newSelection, function (e) {
        if (oldSelection.indexOf(e) === -1) {
          select(e);
        }
      });
    });
  }
  SelectionVisuals.$inject = ['canvas', 'eventBus'];

  /**
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../../core/ElementRegistry').default} ElementRegistry
   * @typedef {import('../../core/EventBus').default} EventBus
   * @typedef {import('./Selection').default} Selection
   */

  /**
   * @param {EventBus} eventBus
   * @param {Selection} selection
   * @param {Canvas} canvas
   * @param {ElementRegistry} elementRegistry
   */
  function SelectionBehavior(eventBus, selection, canvas, elementRegistry) {
    // Select elements on create
    eventBus.on('create.end', 500, function (event) {
      var context = event.context,
        canExecute = context.canExecute,
        elements = context.elements,
        hints = context.hints || {},
        autoSelect = hints.autoSelect;
      if (canExecute) {
        if (autoSelect === false) {
          // Select no elements
          return;
        }
        if (isArray$5(autoSelect)) {
          selection.select(autoSelect);
        } else {
          // Select all elements by default
          selection.select(elements.filter(isShown));
        }
      }
    });

    // Select connection targets on connect
    eventBus.on('connect.end', 500, function (event) {
      var context = event.context,
        connection = context.connection;
      if (connection) {
        selection.select(connection);
      }
    });

    // Select shapes on move
    eventBus.on('shape.move.end', 500, function (event) {
      var previousSelection = event.previousSelection || [];
      var shape = elementRegistry.get(event.context.shape.id);

      // Always select main shape on move
      var isSelected = find$2(previousSelection, function (selectedShape) {
        return shape.id === selectedShape.id;
      });
      if (!isSelected) {
        selection.select(shape);
      }
    });

    // Select elements on click
    eventBus.on('element.click', function (event) {
      if (!isPrimaryButton(event)) {
        return;
      }
      var element = event.element;
      if (element === canvas.getRootElement()) {
        element = null;
      }
      var isSelected = selection.isSelected(element),
        isMultiSelect = selection.get().length > 1;

      // Add to selection if SHIFT pressed
      var add = hasSecondaryModifier(event);
      if (isSelected && isMultiSelect) {
        if (add) {
          // Deselect element
          return selection.deselect(element);
        } else {
          // Select element only
          return selection.select(element);
        }
      } else if (!isSelected) {
        // Select element
        selection.select(element, add);
      } else {
        // Deselect element
        selection.deselect(element);
      }
    });
  }
  SelectionBehavior.$inject = ['eventBus', 'selection', 'canvas', 'elementRegistry'];
  function isShown(element) {
    return !element.hidden;
  }

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var SelectionModule = {
    __init__: ['selectionVisuals', 'selectionBehavior'],
    __depends__: [InteractionEventsModule],
    selection: ['type', Selection],
    selectionVisuals: ['type', SelectionVisuals],
    selectionBehavior: ['type', SelectionBehavior]
  };

  /**
   * Util that provides unique IDs.
   *
   * @class
   * @constructor
   *
   * The ids can be customized via a given prefix and contain a random value to avoid collisions.
   *
   * @param {string} [prefix] a prefix to prepend to generated ids (for better readability)
   */
  function IdGenerator(prefix) {
    this._counter = 0;
    this._prefix = (prefix ? prefix + '-' : '') + Math.floor(Math.random() * 1000000000) + '-';
  }

  /**
   * Returns a next unique ID.
   *
   * @return {string} the id
   */
  IdGenerator.prototype.next = function () {
    return this._prefix + ++this._counter;
  };

  // document wide unique overlay ids
  var ids = new IdGenerator('ov');
  var LOW_PRIORITY$2 = 500;

  /**
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../../core/ElementRegistry').default} ElementRegistry
   * @typedef {import('../../core/EventBus').default} EventBus
   *
   * @typedef {import('../../model/Types').Element} Element
   *
   * @typedef { {
   *   minZoom?: number,
   *   maxZoom?: number
   * } } OverlaysConfigShow
   *
   * @typedef { {
   *   min?: number,
   *   max?: number
   * } } OverlaysConfigScale
   *
   * @typedef { {
  *   id: string,
  *   type: string | null,
  *   element: Element | string
  * } & OverlayAttrs } Overlay
  *
   * @typedef { {
   *   html: HTMLElement | string,
   *   position: {
   *     top?: number,
   *     right?: number,
   *     bottom?: number,
   *     left?: number
   *   }
   * } & OverlaysConfigDefault } OverlayAttrs
   *
   * @typedef { {
   *   html: HTMLElement,
   *   element: Element,
   *   overlays: Overlay[]
   * } } OverlayContainer
   *
   * @typedef {{
   *   defaults?: OverlaysConfigDefault
   * }} OverlaysConfig
   *
   * @typedef { {
   *  show?: OverlaysConfigShow,
   *  scale?: OverlaysConfigScale | boolean
   * } } OverlaysConfigDefault
   *
   * @typedef { {
   *   id?: string;
   *   element?: Element | string;
   *   type?: string;
   * } | string } OverlaysFilter
   */

  /**
   * A service that allows users to attach overlays to diagram elements.
   *
   * The overlay service will take care of overlay positioning during updates.
   *
   * @example
   *
   * ```javascript
   * // add a pink badge on the top left of the shape
   *
   * overlays.add(someShape, {
   *   position: {
   *     top: -5,
   *     left: -5
   *   },
   *   html: '<div style="width: 10px; background: fuchsia; color: white;">0</div>'
   * });
   *
   * // or add via shape id
   *
   * overlays.add('some-element-id', {
   *   position: {
   *     top: -5,
   *     left: -5
   *   }
   *   html: '<div style="width: 10px; background: fuchsia; color: white;">0</div>'
   * });
   *
   * // or add with optional type
   *
   * overlays.add(someShape, 'badge', {
   *   position: {
   *     top: -5,
   *     left: -5
   *   }
   *   html: '<div style="width: 10px; background: fuchsia; color: white;">0</div>'
   * });
   * ```
   *
   * ```javascript
   * // remove an overlay
   *
   * var id = overlays.add(...);
   * overlays.remove(id);
   *
   *
   * You may configure overlay defaults during tool by providing a `config` module
   * with `overlays.defaults` as an entry:
   *
   * {
   *   overlays: {
   *     defaults: {
   *       show: {
   *         minZoom: 0.7,
   *         maxZoom: 5.0
   *       },
   *       scale: {
   *         min: 1
   *       }
   *     }
   * }
   * ```
   *
   * @param {OverlaysConfig} config
   * @param {EventBus} eventBus
   * @param {Canvas} canvas
   * @param {ElementRegistry} elementRegistry
   */
  function Overlays(config, eventBus, canvas, elementRegistry) {
    this._eventBus = eventBus;
    this._canvas = canvas;
    this._elementRegistry = elementRegistry;
    this._ids = ids;

    /**
     * @type {OverlaysConfigDefault}
     */
    this._overlayDefaults = assign$4({
      // no show constraints
      show: null,
      // always scale
      scale: true
    }, config && config.defaults);

    /**
     * @type {Record<string, Overlay>}
     */
    this._overlays = {};

    /**
     * @type {OverlayContainer[]}
     */
    this._overlayContainers = [];

    /**
     * @type {HTMLElement}
     */
    this._overlayRoot = createRoot(canvas.getContainer());
    this._init();
  }
  Overlays.$inject = ['config.overlays', 'eventBus', 'canvas', 'elementRegistry'];

  /**
   * Returns the overlay with the specified ID or a list of overlays
   * for an element with a given type.
   *
   * @example
   *
   * ```javascript
   * // return the single overlay with the given ID
   * overlays.get('some-id');
   *
   * // return all overlays for the shape
   * overlays.get({ element: someShape });
   *
   * // return all overlays on shape with type 'badge'
   * overlays.get({ element: someShape, type: 'badge' });
   *
   * // shape can also be specified as ID
   * overlays.get({ element: 'element-id', type: 'badge' });
   * ```
   *
   * @param {OverlaysFilter} search The filter to be used to find the overlay(s).
   *
   * @return {Overlay|Overlay[]} The overlay(s).
   */
  Overlays.prototype.get = function (search) {
    if (isString$4(search)) {
      search = {
        id: search
      };
    }
    if (isString$4(search.element)) {
      search.element = this._elementRegistry.get(search.element);
    }
    if (search.element) {
      var container = this._getOverlayContainer(search.element, true);

      // return a list of overlays when searching by element (+type)
      if (container) {
        return search.type ? filter$1(container.overlays, matchPattern({
          type: search.type
        })) : container.overlays.slice();
      } else {
        return [];
      }
    } else if (search.type) {
      return filter$1(this._overlays, matchPattern({
        type: search.type
      }));
    } else {
      // return single element when searching by id
      return search.id ? this._overlays[search.id] : null;
    }
  };

  /**
   * Adds an HTML overlay to an element.
   *
   * @param {Element|string} element The element to add the overlay to.
   * @param {string} [type] An optional type that can be used to filter.
   * @param {OverlayAttrs} overlay The overlay.
   *
   * @return {string} The overlay's ID that can be used to get or remove it.
   */
  Overlays.prototype.add = function (element, type, overlay) {
    if (isObject$2(type)) {
      overlay = type;
      type = null;
    }
    if (!element.id) {
      element = this._elementRegistry.get(element);
    }
    if (!overlay.position) {
      throw new Error('must specifiy overlay position');
    }
    if (!overlay.html) {
      throw new Error('must specifiy overlay html');
    }
    if (!element) {
      throw new Error('invalid element specified');
    }
    var id = this._ids.next();
    overlay = assign$4({}, this._overlayDefaults, overlay, {
      id: id,
      type: type,
      element: element,
      html: overlay.html
    });
    this._addOverlay(overlay);
    return id;
  };

  /**
   * Remove an overlay with the given ID or all overlays matching the given filter.
   *
   * @see Overlays#get for filter options.
   *
   * @param {OverlaysFilter} filter The filter to be used to find the overlay.
   */
  Overlays.prototype.remove = function (filter) {
    var overlays = this.get(filter) || [];
    if (!isArray$5(overlays)) {
      overlays = [overlays];
    }
    var self = this;
    forEach$3(overlays, function (overlay) {
      var container = self._getOverlayContainer(overlay.element, true);
      if (overlay) {
        remove$4(overlay.html);
        remove$4(overlay.htmlContainer);
        delete overlay.htmlContainer;
        delete overlay.element;
        delete self._overlays[overlay.id];
      }
      if (container) {
        var idx = container.overlays.indexOf(overlay);
        if (idx !== -1) {
          container.overlays.splice(idx, 1);
        }
      }
    });
  };

  /**
   * Checks whether overlays are shown.
   *
   * @return {boolean} Whether overlays are shown.
   */
  Overlays.prototype.isShown = function () {
    return this._overlayRoot.style.display !== 'none';
  };

  /**
   * Show all overlays.
   */
  Overlays.prototype.show = function () {
    setVisible(this._overlayRoot);
  };

  /**
   * Hide all overlays.
   */
  Overlays.prototype.hide = function () {
    setVisible(this._overlayRoot, false);
  };

  /**
   * Remove all overlays and their container.
   */
  Overlays.prototype.clear = function () {
    this._overlays = {};
    this._overlayContainers = [];
    clear(this._overlayRoot);
  };
  Overlays.prototype._updateOverlayContainer = function (container) {
    var element = container.element,
      html = container.html;

    // update container left,top according to the elements x,y coordinates
    // this ensures we can attach child elements relative to this container

    var x = element.x,
      y = element.y;
    if (element.waypoints) {
      var bbox = getBBox(element);
      x = bbox.x;
      y = bbox.y;
    }
    setPosition(html, x, y);
    attr$1(container.html, 'data-container-id', element.id);
  };
  Overlays.prototype._updateOverlay = function (overlay) {
    var position = overlay.position,
      htmlContainer = overlay.htmlContainer,
      element = overlay.element;

    // update overlay html relative to shape because
    // it is already positioned on the element

    // update relative
    var left = position.left,
      top = position.top;
    if (position.right !== undefined) {
      var width;
      if (element.waypoints) {
        width = getBBox(element).width;
      } else {
        width = element.width;
      }
      left = position.right * -1 + width;
    }
    if (position.bottom !== undefined) {
      var height;
      if (element.waypoints) {
        height = getBBox(element).height;
      } else {
        height = element.height;
      }
      top = position.bottom * -1 + height;
    }
    setPosition(htmlContainer, left || 0, top || 0);
    this._updateOverlayVisibilty(overlay, this._canvas.viewbox());
  };
  Overlays.prototype._createOverlayContainer = function (element) {
    var html = domify$1('<div class="djs-overlays" />');
    assign(html, {
      position: 'absolute'
    });
    this._overlayRoot.appendChild(html);
    var container = {
      html: html,
      element: element,
      overlays: []
    };
    this._updateOverlayContainer(container);
    this._overlayContainers.push(container);
    return container;
  };
  Overlays.prototype._updateRoot = function (viewbox) {
    var scale = viewbox.scale || 1;
    var matrix = 'matrix(' + [scale, 0, 0, scale, -1 * viewbox.x * scale, -1 * viewbox.y * scale].join(',') + ')';
    setTransform(this._overlayRoot, matrix);
  };
  Overlays.prototype._getOverlayContainer = function (element, raw) {
    var container = find$2(this._overlayContainers, function (c) {
      return c.element === element;
    });
    if (!container && !raw) {
      return this._createOverlayContainer(element);
    }
    return container;
  };
  Overlays.prototype._addOverlay = function (overlay) {
    var id = overlay.id,
      element = overlay.element,
      html = overlay.html,
      htmlContainer,
      overlayContainer;

    // unwrap jquery (for those who need it)
    if (html.get && html.constructor.prototype.jquery) {
      html = html.get(0);
    }

    // create proper html elements from
    // overlay HTML strings
    if (isString$4(html)) {
      html = domify$1(html);
    }
    overlayContainer = this._getOverlayContainer(element);
    htmlContainer = domify$1('<div class="djs-overlay" data-overlay-id="' + id + '">');
    assign(htmlContainer, {
      position: 'absolute'
    });
    htmlContainer.appendChild(html);
    if (overlay.type) {
      classes$1(htmlContainer).add('djs-overlay-' + overlay.type);
    }
    var elementRoot = this._canvas.findRoot(element);
    var activeRoot = this._canvas.getRootElement();
    setVisible(htmlContainer, elementRoot === activeRoot);
    overlay.htmlContainer = htmlContainer;
    overlayContainer.overlays.push(overlay);
    overlayContainer.html.appendChild(htmlContainer);
    this._overlays[id] = overlay;
    this._updateOverlay(overlay);
    this._updateOverlayVisibilty(overlay, this._canvas.viewbox());
  };
  Overlays.prototype._updateOverlayVisibilty = function (overlay, viewbox) {
    var show = overlay.show,
      rootElement = this._canvas.findRoot(overlay.element),
      minZoom = show && show.minZoom,
      maxZoom = show && show.maxZoom,
      htmlContainer = overlay.htmlContainer,
      activeRootElement = this._canvas.getRootElement(),
      visible = true;
    if (rootElement !== activeRootElement) {
      visible = false;
    } else if (show) {
      if (isDefined(minZoom) && minZoom > viewbox.scale || isDefined(maxZoom) && maxZoom < viewbox.scale) {
        visible = false;
      }
    }
    setVisible(htmlContainer, visible);
    this._updateOverlayScale(overlay, viewbox);
  };
  Overlays.prototype._updateOverlayScale = function (overlay, viewbox) {
    var shouldScale = overlay.scale,
      minScale,
      maxScale,
      htmlContainer = overlay.htmlContainer;
    var scale,
      transform = '';
    if (shouldScale !== true) {
      if (shouldScale === false) {
        minScale = 1;
        maxScale = 1;
      } else {
        minScale = shouldScale.min;
        maxScale = shouldScale.max;
      }
      if (isDefined(minScale) && viewbox.scale < minScale) {
        scale = (1 / viewbox.scale || 1) * minScale;
      }
      if (isDefined(maxScale) && viewbox.scale > maxScale) {
        scale = (1 / viewbox.scale || 1) * maxScale;
      }
    }
    if (isDefined(scale)) {
      transform = 'scale(' + scale + ',' + scale + ')';
    }
    setTransform(htmlContainer, transform);
  };
  Overlays.prototype._updateOverlaysVisibilty = function (viewbox) {
    var self = this;
    forEach$3(this._overlays, function (overlay) {
      self._updateOverlayVisibilty(overlay, viewbox);
    });
  };
  Overlays.prototype._init = function () {
    var eventBus = this._eventBus;
    var self = this;

    // scroll/zoom integration

    function updateViewbox(viewbox) {
      self._updateRoot(viewbox);
      self._updateOverlaysVisibilty(viewbox);
      self.show();
    }
    eventBus.on('canvas.viewbox.changing', function (event) {
      self.hide();
    });
    eventBus.on('canvas.viewbox.changed', function (event) {
      updateViewbox(event.viewbox);
    });

    // remove integration

    eventBus.on(['shape.remove', 'connection.remove'], function (e) {
      var element = e.element;
      var overlays = self.get({
        element: element
      });
      forEach$3(overlays, function (o) {
        self.remove(o.id);
      });
      var container = self._getOverlayContainer(element);
      if (container) {
        remove$4(container.html);
        var i = self._overlayContainers.indexOf(container);
        if (i !== -1) {
          self._overlayContainers.splice(i, 1);
        }
      }
    });

    // move integration

    eventBus.on('element.changed', LOW_PRIORITY$2, function (e) {
      var element = e.element;
      var container = self._getOverlayContainer(element, true);
      if (container) {
        forEach$3(container.overlays, function (overlay) {
          self._updateOverlay(overlay);
        });
        self._updateOverlayContainer(container);
      }
    });

    // marker integration, simply add them on the overlays as classes, too.

    eventBus.on('element.marker.update', function (e) {
      var container = self._getOverlayContainer(e.element, true);
      if (container) {
        classes$1(container.html)[e.add ? 'add' : 'remove'](e.marker);
      }
    });
    eventBus.on('root.set', function () {
      self._updateOverlaysVisibilty(self._canvas.viewbox());
    });

    // clear overlays with diagram

    eventBus.on('diagram.clear', this.clear, this);
  };

  // helpers /////////////////////////////

  function createRoot(parentNode) {
    var root = domify$1('<div class="djs-overlay-container" />');
    assign(root, {
      position: 'absolute',
      width: 0,
      height: 0
    });
    parentNode.insertBefore(root, parentNode.firstChild);
    return root;
  }
  function setPosition(el, x, y) {
    assign(el, {
      left: x + 'px',
      top: y + 'px'
    });
  }

  /**
   * Set element visible
   *
   * @param {DOMElement} el
   * @param {boolean} [visible=true]
   */
  function setVisible(el, visible) {
    el.style.display = visible === false ? 'none' : '';
  }
  function setTransform(el, transform) {
    el.style['transform-origin'] = 'top left';
    ['', '-ms-', '-webkit-'].forEach(function (prefix) {
      el.style[prefix + 'transform'] = transform;
    });
  }

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var OverlaysModule = {
    __init__: ['overlays'],
    overlays: ['type', Overlays]
  };

  function DefinitionPropertiesView(eventBus, canvas, translate) {
    this._eventBus = eventBus;
    this._canvas = canvas;
    this._translate = translate;
    eventBus.on('diagram.init', function () {
      this._init();
    }, this);
    eventBus.on('import.done', function (event) {
      if (!event.error) {
        this.update();
      }
    }, this);

    /* markup definition */

    this.HTML_MARKUP = '<div class="dmn-definitions">' + '<div class="dmn-definitions-name" title="' + this._translate('Definition name') + '" spellcheck="false">' + '</div>' + '<div class="dmn-definitions-id" title="' + this._translate('Definition ID') + '" spellcheck="false">' + '</div>' + '</div>';
  }
  DefinitionPropertiesView.$inject = ['eventBus', 'canvas', 'translate'];

  /**
   * Initialize
   */
  DefinitionPropertiesView.prototype._init = function () {
    var canvas = this._canvas,
      eventBus = this._eventBus;
    var parent = canvas.getContainer(),
      container = this._container = domify$1(this.HTML_MARKUP);
    parent.appendChild(container);
    this.nameElement = query('.dmn-definitions-name', this._container);
    this.idElement = query('.dmn-definitions-id', this._container);
    delegate.bind(container, '.dmn-definitions-name, .dmn-definitions-id', 'mousedown', function (event) {
      event.stopPropagation();
    });
    eventBus.fire('definitionIdView.create', {
      html: container
    });
  };
  DefinitionPropertiesView.prototype.update = function () {
    var businessObject = this._canvas.getRootElement().businessObject;
    this.nameElement.textContent = businessObject.name;
    this.idElement.textContent = businessObject.id;
  };

  function PaletteAdapter(eventBus, canvas) {
    function toggleMarker(cls, on) {
      var container = canvas.getContainer();
      classes$1(container).toggle(cls, on);
    }
    eventBus.on('palette.create', function () {
      toggleMarker('with-palette', true);
    });
    eventBus.on('palette.changed', function (event) {
      toggleMarker('with-palette-two-column', event.twoColumn);
    });
  }
  PaletteAdapter.$inject = ['eventBus', 'canvas'];

  var DefinitionPropertiesModule = {
    __depends__: [TranslateModule],
    __init__: ['definitionPropertiesView', 'definitionPropertiesPaletteAdapter'],
    definitionPropertiesView: ['type', DefinitionPropertiesView],
    definitionPropertiesPaletteAdapter: ['type', PaletteAdapter]
  };

  var PROVIDERS = [{
    className: 'dmn-icon-decision-table',
    matches: function (el) {
      var businessObject = getBusinessObject(el);
      return is(businessObject, 'dmn:Decision') && is(businessObject.decisionLogic, 'dmn:DecisionTable');
    },
    title: 'Open decision table'
  }, {
    className: 'dmn-icon-literal-expression',
    matches: function (el) {
      var boxedExpression = getBoxedExpression(el);
      return is(boxedExpression, 'dmn:LiteralExpression');
    },
    title: 'Open literal expression'
  }];

  /**
   * Displays overlays that can be clicked in order to drill
   * down into a DMN element.
   */
  class DrillDown {
    constructor(injector, eventBus, overlays, config, translate) {
      this._injector = injector;
      this._eventBus = eventBus;
      this._overlays = overlays;
      this._translate = translate;
      this._config = config || {
        enabled: true
      };
      eventBus.on(['shape.added'], ({
        element
      }) => {
        for (let i = 0; i < PROVIDERS.length; i++) {
          const {
            matches,
            className,
            title
          } = PROVIDERS[i];
          var editable = matches && matches(element);
          if (editable) {
            this.addOverlay(element, className, title);
          }
        }
      });
    }

    /**
     * Add overlay to an element that enables drill down.
     *
     * @param {Object} element Element to add overlay to.
     * @param {string} className
     *        CSS class that will be added to overlay in order to display icon.
     * @param {string} title added to the button
     */
    addOverlay(element, className, title) {
      const enabled = this._config.enabled !== false;
      const node = this._getOverlayNode(className, title, enabled);
      const overlayId = this._overlays.add(element, {
        position: {
          top: 2,
          left: 2
        },
        html: node
      });

      // TODO(nikku): can we remove renamed to drillDown.enabled
      if (enabled) {
        classes$1(node).add('interactive');
        this.bindEventListener(element, node, overlayId);
      }
    }
    _getOverlayNode(className, title, enabled) {
      const container = document.createElement('div');
      container.className = 'drill-down-overlay';
      if (!enabled) {
        const icon = document.createElement('span');
        icon.className = className;
        container.appendChild(icon);
        return container;
      }
      const button = document.createElement('button');
      button.type = 'button';
      button.className = className;
      button.title = this._translate(title);
      container.appendChild(button);
      return container;
    }

    /**
     * @param {Object} element
     * @param {Object} overlay
     * @param {string} id
     */
    bindEventListener(element, overlay, id) {
      const overlays = this._overlays,
        eventBus = this._eventBus;
      const overlaysRoot = overlays._overlayRoot;
      delegate.bind(overlaysRoot, '[data-overlay-id="' + id + '"]', 'click', () => {
        const triggerDefault = eventBus.fire('drillDown.click', {
          element
        });
        if (triggerDefault === false) {
          return;
        }
        this.drillDown(element);
      });
    }

    /**
     * Drill down into the specific element.
     *
     * @param  {djs.model.Base} element
     *
     * @return {boolean} whether drill down was executed
     */
    drillDown(element) {
      const parent = this._injector.get('_parent', false);

      // no parent; skip drill down
      if (!parent) {
        return false;
      }
      const view = parent.getView(element.businessObject);

      // no view to drill down to
      if (!view) {
        return false;
      }
      parent.open(view);
      return true;
    }
  }
  DrillDown.$inject = ['injector', 'eventBus', 'overlays', 'config.drillDown', 'translate'];

  var DrillDownModule = {
    __depends__: [OverlaysModule, TranslateModule],
    __init__: ['drillDown'],
    drillDown: ['type', DrillDown]
  };

  /**
   * The code in the <project-logo></project-logo> area
   * must not be changed.
   *
   * @see http://bpmn.io/license for more information.
   */


  /**
   * @typedef {import('dmn-js-shared/lib/base/View).OpenResult} OpenResult
   */

  /**
   * @typedef {import('dmn-js-shared/lib/base/View).OpenError} OpenError
   */

  /**
   * A viewer for DMN diagrams.
   *
   * Have a look at {@link NavigatedViewer} or {@link Modeler} for bundles that include
   * additional features.
   *
   *
   * ## Extending the Viewer
   *
   * In order to extend the viewer pass extension modules to bootstrap via the
   * `additionalModules` option. An extension module is an object that exposes
   * named services.
   *
   * The following example depicts the integration of a simple
   * logging component that integrates with interaction events:
   *
   *
   * ```javascript
   *
   * // logging component
   * function InteractionLogger(eventBus) {
   *   eventBus.on('element.hover', function(event) {
   *     console.log()
   *   })
   * }
   *
   * InteractionLogger.$inject = [ 'eventBus' ]; // minification save
   *
   * // extension module
   * var extensionModule = {
   *   __init__: [ 'interactionLogger' ],
   *   interactionLogger: [ 'type', InteractionLogger ]
   * };
   *
   * // extend the viewer
   * var drdViewer = new Viewer({ additionalModules: [ extensionModule ] });
   * drdViewer.importXML(...);
   * ```
   *
   * @param {Object} options configuration options to pass to the viewer
   * @param {DOMElement} [options.container]
   *        the container to render the viewer in, defaults to body
   * @param {Array<didi.Module>} [options.modules]
   *        a list of modules to override the default modules
   * @param {Array<didi.Module>} [options.additionalModules]
   *        a list of modules to use with the default modules
   */
  function Viewer$5(options) {
    this._container = this._createContainer();

    /* <project-logo> */

    addProjectLogo(this._container);

    /* </project-logo> */

    this._init(this._container, options);
  }
  e(Viewer$5, Diagram);

  /**
   * The saveSVG result.
   *
   * @typedef {Object} SaveSVGResult
   *
   * @property {string} svg
   */

  /**
   * Export the currently displayed DMN diagram as
   * an SVG image.
   *
   * @param {Object} [options]
   *
   * @return {Promise<SaveSVGResult>}
   */
  Viewer$5.prototype.saveSVG = wrapForCompatibility(function (options) {
    var self = this;
    return new Promise(function (resolve) {
      var canvas = self.get('canvas');
      var contentNode = canvas.getActiveLayer(),
        defsNode = query('defs', canvas._svg);
      var contents = innerSVG(contentNode),
        defs = defsNode && defsNode.outerHTML || '';
      var bbox = contentNode.getBBox();

      /* eslint-disable max-len */
      var svg = '<?xml version="1.0" encoding="utf-8"?>\n' + '<!-- created with dmn-js / http://bpmn.io -->\n' + '<!DOCTYPE svg PUBLIC "-//W3C//DTD SVG 1.1//EN" "http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd">\n' + '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" ' + 'width="' + bbox.width + '" height="' + bbox.height + '" ' + 'viewBox="' + bbox.x + ' ' + bbox.y + ' ' + bbox.width + ' ' + bbox.height + '" version="1.1">' + defs + contents + '</svg>';
      /* eslint-enable */

      resolve({
        svg
      });
    });
  });
  Viewer$5.prototype.getModules = function () {
    return this._modules;
  };

  /**
   * Destroy the viewer instance and remove all its
   * remainders from the document tree.
   */
  Viewer$5.prototype.destroy = function () {
    // diagram destroy
    Diagram.prototype.destroy.call(this);

    // dom detach
    remove$4(this._container);
  };

  /**
   * Register an event listener
   *
   * Remove a previously added listener via {@link #off(event, callback)}.
   *
   * @param {string} event
   * @param {number} [priority]
   * @param {Function} callback
   * @param {Object} [that]
   */
  Viewer$5.prototype.on = function (event, priority, callback, target) {
    return this.get('eventBus').on(event, priority, callback, target);
  };

  /**
   * De-register an event listener
   *
   * @param {string} event
   * @param {Function} callback
   */
  Viewer$5.prototype.off = function (event, callback) {
    this.get('eventBus').off(event, callback);
  };
  Viewer$5.prototype._init = function (container, options) {
    var {
      additionalModules,
      canvas,
      ...additionalOptions
    } = options;
    var baseModules = options.modules || this.getModules(),
      staticModules = [{
        drd: ['value', this]
      }];
    var modules = [...staticModules, ...baseModules, ...(additionalModules || [])];
    var diagramOptions = {
      ...additionalOptions,
      canvas: {
        ...canvas,
        container
      },
      modules
    };

    // invoke diagram constructor
    Diagram.call(this, diagramOptions);
    if (options && options.container) {
      this.attachTo(options.container);
    }
  };

  /**
   * Emit an event on the underlying {@link EventBus}
   *
   * @param  {string} type
   * @param  {Object} event
   *
   * @return {Object} event processing result (if any)
   */
  Viewer$5.prototype._emit = function (type, event) {
    return this.get('eventBus').fire(type, event);
  };
  Viewer$5.prototype._createContainer = function () {
    return domify$1('<div class="dmn-drd-container"></div>');
  };

  /**
   * Open diagram element.
   *
   * @param  {ModdleElement} definitions
   * @returns {Promise} Resolves with {OpenResult} when successful
   * or rejects with {OpenError}
   */
  Viewer$5.prototype.open = function (definitions) {
    var self = this;
    return new Promise((resolve, reject) => {
      var err;

      // use try/catch to not swallow synchronous exceptions
      // that may be raised during model parsing
      try {
        if (self._definitions) {
          // clear existing rendered diagram
          self.clear();
        }

        // update definitions
        self._definitions = definitions;

        // perform graphical import
        return importDRD(self, definitions, function (err, warnings) {
          if (err) {
            err.warnings = warnings || [];
            reject(err);
          } else {
            resolve({
              warnings: warnings || []
            });
          }
        });
      } catch (e) {
        err = e;
      }
      if (err) {
        err.warnings = err.warnings || [];
        reject(err);
      } else {
        resolve({
          warnings: []
        });
      }
    });
  };

  /**
   * Attach viewer to given parent node.
   *
   * @param  {Element} parentNode
   */
  Viewer$5.prototype.attachTo = function (parentNode) {
    if (!parentNode) {
      throw new Error('parentNode required');
    }

    // ensure we detach from the
    // previous, old parent
    this.detach();
    var container = this._container;
    parentNode.appendChild(container);
    this._emit('attach', {});
    this.get('canvas').resized();
  };

  /**
   * Detach viewer from parent node, if attached.
   */
  Viewer$5.prototype.detach = function () {
    var container = this._container,
      parentNode = container.parentNode;
    if (!parentNode) {
      return;
    }
    this._emit('detach', {});
    parentNode.removeChild(container);
  };
  Viewer$5.prototype._modules = [CoreModule$1, TranslateModule, SelectionModule, OverlaysModule, DefinitionPropertiesModule, DrillDownModule];

  /**
   * @typedef {import('../util/Types').Point} Point
   * @typedef {import('../util/Types').Rect} Rect
   */


  /**
   * @param {Point} a
   * @param {Point} b
   * @return {Point}
   */
  function delta(a, b) {
    return {
      x: a.x - b.x,
      y: a.y - b.y
    };
  }

  /**
   * Get the logarithm of x with base 10.
   *
   * @param {number} x
   */
  function log10(x) {
    return Math.log(x) / Math.log(10);
  }

  /**
   * Get step size for given range and number of steps.
   *
   * @param {Object} range
   * @param {number} range.min
   * @param {number} range.max
   * @param {number} steps
   */
  function getStepSize(range, steps) {
    var minLinearRange = log10(range.min),
      maxLinearRange = log10(range.max);
    var absoluteLinearRange = Math.abs(minLinearRange) + Math.abs(maxLinearRange);
    return absoluteLinearRange / steps;
  }

  /**
   * @param {Object} range
   * @param {number} range.min
   * @param {number} range.max
   * @param {number} scale
   */
  function cap(range, scale) {
    return Math.max(range.min, Math.min(range.max, scale));
  }

  /**
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../../core/EventBus').default} EventBus
   *
   * @typedef {import('../../util/Types').Point} Point
   * @typedef {import('../../util/Types').ScrollDelta} ScrollDelta
   */

  var sign = Math.sign || function (n) {
    return n >= 0 ? 1 : -1;
  };
  var RANGE = {
      min: 0.2,
      max: 4
    },
    NUM_STEPS = 10;
  var DELTA_THRESHOLD = 0.1;
  var DEFAULT_SCALE = 0.75;

  /**
   * An implementation of zooming and scrolling within the
   * {@link Canvas} via the mouse wheel.
   *
   * Mouse wheel zooming / scrolling may be disabled using
   * the {@link toggle(enabled)} method.
   *
   * @param {Object} [config]
   * @param {boolean} [config.enabled=true] default enabled state
   * @param {number} [config.scale=.75] scroll sensivity
   * @param {EventBus} eventBus
   * @param {Canvas} canvas
   */
  function ZoomScroll$1(config, eventBus, canvas) {
    config = config || {};
    this._enabled = false;
    this._canvas = canvas;
    this._container = canvas._container;
    this._handleWheel = bind$3(this._handleWheel, this);
    this._totalDelta = 0;
    this._scale = config.scale || DEFAULT_SCALE;
    var self = this;
    eventBus.on('canvas.init', function (e) {
      self._init(config.enabled !== false);
    });
  }
  ZoomScroll$1.$inject = ['config.zoomScroll', 'eventBus', 'canvas'];

  /**
   * @param {ScrollDelta} delta
   */
  ZoomScroll$1.prototype.scroll = function scroll(delta) {
    this._canvas.scroll(delta);
  };
  ZoomScroll$1.prototype.reset = function reset() {
    this._canvas.zoom('fit-viewport');
  };

  /**
   * Zoom depending on delta.
   *
   * @param {number} delta
   * @param {Point} position
   */
  ZoomScroll$1.prototype.zoom = function zoom(delta, position) {
    // zoom with half the step size of stepZoom
    var stepSize = getStepSize(RANGE, NUM_STEPS * 2);

    // add until threshold reached
    this._totalDelta += delta;
    if (Math.abs(this._totalDelta) > DELTA_THRESHOLD) {
      this._zoom(delta, position, stepSize);

      // reset
      this._totalDelta = 0;
    }
  };
  ZoomScroll$1.prototype._handleWheel = function handleWheel(event) {
    // event is already handled by '.djs-scrollable'
    if (closest(event.target, '.djs-scrollable', true)) {
      return;
    }
    var element = this._container;
    event.preventDefault();

    // pinch to zoom is mapped to wheel + ctrlKey = true
    // in modern browsers (!)

    var isZoom = event.ctrlKey || isMac() && event.metaKey;
    var isHorizontalScroll = event.shiftKey;
    var factor = -1 * this._scale,
      delta;
    if (isZoom) {
      factor *= event.deltaMode === 0 ? 0.020 : 0.32;
    } else {
      factor *= event.deltaMode === 0 ? 1.0 : 16.0;
    }
    if (isZoom) {
      var elementRect = element.getBoundingClientRect();
      var offset = {
        x: event.clientX - elementRect.left,
        y: event.clientY - elementRect.top
      };
      delta = Math.sqrt(Math.pow(event.deltaY, 2) + Math.pow(event.deltaX, 2)) * sign(event.deltaY) * factor;

      // zoom in relative to diagram {x,y} coordinates
      this.zoom(delta, offset);
    } else {
      if (isHorizontalScroll) {
        delta = {
          dx: factor * event.deltaY,
          dy: 0
        };
      } else {
        delta = {
          dx: factor * event.deltaX,
          dy: factor * event.deltaY
        };
      }
      this.scroll(delta);
    }
  };

  /**
   * Zoom with fixed step size.
   *
   * @param {number} delta Zoom delta (1 for zooming in, -1 for zooming out).
   * @param {Point} [position]
   */
  ZoomScroll$1.prototype.stepZoom = function stepZoom(delta, position) {
    var stepSize = getStepSize(RANGE, NUM_STEPS);
    this._zoom(delta, position, stepSize);
  };

  /**
   * Zoom in/out given a step size.
   *
   * @param {number} delta
   * @param {Point} [position]
   * @param {number} stepSize
   */
  ZoomScroll$1.prototype._zoom = function (delta, position, stepSize) {
    var canvas = this._canvas;
    var direction = delta > 0 ? 1 : -1;
    var currentLinearZoomLevel = log10(canvas.zoom());

    // snap to a proximate zoom step
    var newLinearZoomLevel = Math.round(currentLinearZoomLevel / stepSize) * stepSize;

    // increase or decrease one zoom step in the given direction
    newLinearZoomLevel += stepSize * direction;

    // calculate the absolute logarithmic zoom level based on the linear zoom level
    // (e.g. 2 for an absolute x2 zoom)
    var newLogZoomLevel = Math.pow(10, newLinearZoomLevel);
    canvas.zoom(cap(RANGE, newLogZoomLevel), position);
  };

  /**
   * Toggle the zoom scroll ability via mouse wheel.
   *
   * @param {boolean} [newEnabled] new enabled state
   */
  ZoomScroll$1.prototype.toggle = function toggle(newEnabled) {
    var element = this._container;
    var handleWheel = this._handleWheel;
    var oldEnabled = this._enabled;
    if (typeof newEnabled === 'undefined') {
      newEnabled = !oldEnabled;
    }

    // only react on actual changes
    if (oldEnabled !== newEnabled) {
      // add or remove wheel listener based on
      // changed enabled state
      event[newEnabled ? 'bind' : 'unbind'](element, 'wheel', handleWheel, false);
    }
    this._enabled = newEnabled;
    return newEnabled;
  };
  ZoomScroll$1.prototype._init = function (newEnabled) {
    this.toggle(newEnabled);
  };

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var ZoomScroll = {
    __init__: ['zoomScroll'],
    zoomScroll: ['type', ZoomScroll$1]
  };

  var CURSOR_CLS_PATTERN = /^djs-cursor-.*$/;

  /**
   * @param {string} mode
   */
  function set(mode) {
    var classes = classes$1(document.body);
    classes.removeMatching(CURSOR_CLS_PATTERN);
    if (mode) {
      classes.add('djs-cursor-' + mode);
    }
  }
  function unset() {
    set(null);
  }

  /**
   * @typedef {import('../core/EventBus').default} EventBus
   */

  var TRAP_PRIORITY = 5000;

  /**
   * Installs a click trap that prevents a ghost click following a dragging operation.
   *
   * @param {EventBus} eventBus
   * @param {string} [eventName='element.click']
   *
   * @return {() => void} a function to immediately remove the installed trap.
   */
  function install(eventBus, eventName) {
    eventName = eventName || 'element.click';
    function trap() {
      return false;
    }
    eventBus.once(eventName, TRAP_PRIORITY, trap);
    return function () {
      eventBus.off(eventName, trap);
    };
  }

  /**
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../../core/EventBus').default} EventBus
   */

  var THRESHOLD = 15;

  /**
   * Move the canvas via mouse.
   *
   * @param {EventBus} eventBus
   * @param {Canvas} canvas
   */
  function MoveCanvas$1(eventBus, canvas) {
    var context;

    // listen for move on element mouse down;
    // allow others to hook into the event before us though
    // (dragging / element moving will do this)
    eventBus.on('element.mousedown', 500, function (e) {
      return handleStart(e.originalEvent);
    });
    function handleMove(event) {
      var start = context.start,
        button = context.button,
        position = toPoint(event),
        delta$1 = delta(position, start);
      if (!context.dragging && length(delta$1) > THRESHOLD) {
        context.dragging = true;
        if (button === 0) {
          install(eventBus);
        }
        set('grab');
      }
      if (context.dragging) {
        var lastPosition = context.last || context.start;
        delta$1 = delta(position, lastPosition);
        canvas.scroll({
          dx: delta$1.x,
          dy: delta$1.y
        });
        context.last = position;
      }

      // prevent select
      event.preventDefault();
    }
    function handleEnd(event$1) {
      event.unbind(document, 'mousemove', handleMove);
      event.unbind(document, 'mouseup', handleEnd);
      context = null;
      unset();
    }
    function handleStart(event$1) {
      // event is already handled by '.djs-draggable'
      if (closest(event$1.target, '.djs-draggable')) {
        return;
      }
      var button = event$1.button;

      // reject right mouse button or modifier key
      if (button >= 2 || event$1.ctrlKey || event$1.shiftKey || event$1.altKey) {
        return;
      }
      context = {
        button: button,
        start: toPoint(event$1)
      };
      event.bind(document, 'mousemove', handleMove);
      event.bind(document, 'mouseup', handleEnd);

      // we've handled the event
      return true;
    }
    this.isActive = function () {
      return !!context;
    };
  }
  MoveCanvas$1.$inject = ['eventBus', 'canvas'];

  // helpers ///////

  function length(point) {
    return Math.sqrt(Math.pow(point.x, 2) + Math.pow(point.y, 2));
  }

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var MoveCanvas = {
    __init__: ['moveCanvas'],
    moveCanvas: ['type', MoveCanvas$1]
  };

  /**
   * @param {string} str
   *
   * @return {string}
   */
  var HTML_ESCAPE_MAP = {
    '&': '&amp;',
    '<': '&lt;',
    '>': '&gt;',
    '"': '&quot;',
    '\'': '&#39;'
  };

  /**
   * @param {string} str
   *
   * @return {string}
   */
  function escapeHTML(str) {
    str = '' + str;
    return str && str.replace(/[&<>"']/g, function (match) {
      return HTML_ESCAPE_MAP[match];
    });
  }

  /**
   * Checks if key pressed is one of provided keys.
   *
   * @param {string|string[]} keys
   * @param {KeyboardEvent} event
   * @return {boolean}
   */
  function isKey(keys, event) {
    keys = isArray$5(keys) ? keys : [keys];
    return keys.indexOf(event.key) !== -1 || keys.indexOf(event.code) !== -1;
  }

  /**
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../../core/EventBus').default} EventBus
   * @typedef {import('../selection/Selection').default} Selection
   * @typedef {import('../../i18n/translate/translate.js').default} Translate
   *
   * @typedef {import('../../util/Types').Dimensions} Dimensions
   *
   * @typedef {import('./SearchPadProvider').default} SearchPadProvider
   * @typedef {import('./SearchPadProvider').SearchResult} SearchResult
   * @typedef {import('./SearchPadProvider').Token} Token
   */

  var SCROLL_TO_ELEMENT_PADDING = 300;

  /**
   * Provides searching infrastructure.
   *
   * @param {Canvas} canvas
   * @param {EventBus} eventBus
   * @param {Selection} selection
   * @param {Translate} translate
   */
  function SearchPad(canvas, eventBus, selection, translate) {
    this._open = false;
    this._results = {};
    this._eventMaps = [];
    this._cachedRootElement = null;
    this._cachedSelection = null;
    this._cachedViewbox = null;
    this._canvas = canvas;
    this._eventBus = eventBus;
    this._selection = selection;
    this._translate = translate;

    // setup elements
    this._container = this._getBoxHtml();
    this._searchInput = query(SearchPad.INPUT_SELECTOR, this._container);
    this._resultsContainer = query(SearchPad.RESULTS_CONTAINER_SELECTOR, this._container);

    // attach search pad
    this._canvas.getContainer().appendChild(this._container);

    // cleanup whenever appropriate
    eventBus.on(['canvas.destroy', 'diagram.destroy', 'drag.init', 'elements.changed'], this.close, this);
  }
  SearchPad.$inject = ['canvas', 'eventBus', 'selection', 'translate'];

  /**
   * Binds and keeps track of all event listereners
   */
  SearchPad.prototype._bindEvents = function () {
    var self = this;
    function listen(el, selector, type, fn) {
      self._eventMaps.push({
        el: el,
        type: type,
        listener: delegate.bind(el, selector, type, fn)
      });
    }

    // close search on clicking anywhere outside
    listen(document, 'html', 'click', function (e) {
      self.close(false);
    });

    // stop event from propagating and closing search
    // focus on input
    listen(this._container, SearchPad.INPUT_SELECTOR, 'click', function (e) {
      e.stopPropagation();
      e.delegateTarget.focus();
    });

    // preselect result on hover
    listen(this._container, SearchPad.RESULT_SELECTOR, 'mouseover', function (e) {
      e.stopPropagation();
      self._scrollToNode(e.delegateTarget);
      self._preselect(e.delegateTarget);
    });

    // selects desired result on mouse click
    listen(this._container, SearchPad.RESULT_SELECTOR, 'click', function (e) {
      e.stopPropagation();
      self._select(e.delegateTarget);
    });

    // prevent cursor in input from going left and right when using up/down to
    // navigate results
    listen(this._container, SearchPad.INPUT_SELECTOR, 'keydown', function (e) {
      if (isKey('ArrowUp', e)) {
        e.preventDefault();
      }
      if (isKey('ArrowDown', e)) {
        e.preventDefault();
      }
    });

    // handle keyboard input
    listen(this._container, SearchPad.INPUT_SELECTOR, 'keyup', function (e) {
      if (isKey('Escape', e)) {
        return self.close();
      }
      if (isKey('Enter', e)) {
        var selected = self._getCurrentResult();
        return selected ? self._select(selected) : self.close(false);
      }
      if (isKey('ArrowUp', e)) {
        return self._scrollToDirection(true);
      }
      if (isKey('ArrowDown', e)) {
        return self._scrollToDirection();
      }

      // do not search while navigating text input
      if (isKey(['ArrowLeft', 'ArrowRight'], e)) {
        return;
      }

      // anything else
      self._search(e.delegateTarget.value);
    });
  };

  /**
   * Unbinds all previously established listeners
   */
  SearchPad.prototype._unbindEvents = function () {
    this._eventMaps.forEach(function (m) {
      delegate.unbind(m.el, m.type, m.listener);
    });
  };

  /**
   * Performs a search for the given pattern.
   *
   * @param {string} pattern
   */
  SearchPad.prototype._search = function (pattern) {
    var self = this;
    this._clearResults();

    // do not search on empty query
    if (!pattern || pattern === '') {
      return;
    }
    var searchResults = this._searchProvider.find(pattern);
    searchResults = searchResults.filter(function (searchResult) {
      return !self._canvas.getRootElements().includes(searchResult.element);
    });
    if (!searchResults.length) {
      this._selection.select(null);
      return;
    }

    // append new results
    searchResults.forEach(function (result) {
      var id = result.element.id;
      var node = self._createResultNode(result, id);
      self._results[id] = {
        element: result.element,
        node: node
      };
    });

    // preselect first result
    var node = query(SearchPad.RESULT_SELECTOR, this._resultsContainer);
    this._scrollToNode(node);
    this._preselect(node);
  };

  /**
   * Navigate to the previous/next result. Defaults to next result.
   *
   * @param {boolean} previous
   */
  SearchPad.prototype._scrollToDirection = function (previous) {
    var selected = this._getCurrentResult();
    if (!selected) {
      return;
    }
    var node = previous ? selected.previousElementSibling : selected.nextElementSibling;
    if (node) {
      this._scrollToNode(node);
      this._preselect(node);
    }
  };

  /**
   * Scroll to the node if it is not visible.
   *
   * @param {HTMLElement} node
   */
  SearchPad.prototype._scrollToNode = function (node) {
    if (!node || node === this._getCurrentResult()) {
      return;
    }
    var nodeOffset = node.offsetTop;
    var containerScroll = this._resultsContainer.scrollTop;
    var bottomScroll = nodeOffset - this._resultsContainer.clientHeight + node.clientHeight;
    if (nodeOffset < containerScroll) {
      this._resultsContainer.scrollTop = nodeOffset;
    } else if (containerScroll < bottomScroll) {
      this._resultsContainer.scrollTop = bottomScroll;
    }
  };

  /**
   * Clears all results data.
   */
  SearchPad.prototype._clearResults = function () {
    clear(this._resultsContainer);
    this._results = {};
    this._eventBus.fire('searchPad.cleared');
  };

  /**
   * Get currently selected result.
   *
   * @return {HTMLElement}
   */
  SearchPad.prototype._getCurrentResult = function () {
    return query(SearchPad.RESULT_SELECTED_SELECTOR, this._resultsContainer);
  };

  /**
   * Create result DOM element within results container
   * that corresponds to a search result.
   *
   * 'result' : one of the elements returned by Pad
   * 'id' : id attribute value to assign to the new DOM node
   * return : created DOM element
   *
   * @param {SearchResult} result
   * @param {string} id
   *
   * @return {HTMLElement}
   */
  SearchPad.prototype._createResultNode = function (result, id) {
    var node = domify$1(SearchPad.RESULT_HTML);

    // create only if available
    if (result.primaryTokens.length > 0) {
      createInnerTextNode(node, result.primaryTokens, SearchPad.RESULT_PRIMARY_HTML);
    }

    // secondary tokens (represent element ID) are allways available
    createInnerTextNode(node, result.secondaryTokens, SearchPad.RESULT_SECONDARY_HTML);
    attr$1(node, SearchPad.RESULT_ID_ATTRIBUTE, id);
    this._resultsContainer.appendChild(node);
    return node;
  };

  /**
   * Register search element provider.
   *
   * @param {SearchPadProvider} provider
   */
  SearchPad.prototype.registerProvider = function (provider) {
    this._searchProvider = provider;
  };

  /**
   * Open search pad.
   */
  SearchPad.prototype.open = function () {
    if (!this._searchProvider) {
      throw new Error('no search provider registered');
    }
    if (this.isOpen()) {
      return;
    }
    this._cachedRootElement = this._canvas.getRootElement();
    this._cachedSelection = this._selection.get();
    this._cachedViewbox = this._canvas.viewbox();
    this._selection.select(null);
    this._bindEvents();
    this._open = true;
    classes$1(this._canvas.getContainer()).add('djs-search-open');
    classes$1(this._container).add('open');
    this._searchInput.focus();
    this._eventBus.fire('searchPad.opened');
  };

  /**
   * Close search pad.
   */
  SearchPad.prototype.close = function (restoreCached = true) {
    if (!this.isOpen()) {
      return;
    }
    if (restoreCached) {
      if (this._cachedRootElement) {
        this._canvas.setRootElement(this._cachedRootElement);
      }
      if (this._cachedSelection) {
        this._selection.select(this._cachedSelection);
      }
      if (this._cachedViewbox) {
        this._canvas.viewbox(this._cachedViewbox);
      }
      this._eventBus.fire('searchPad.restored');
    }
    this._cachedRootElement = null;
    this._cachedSelection = null;
    this._cachedViewbox = null;
    this._unbindEvents();
    this._open = false;
    classes$1(this._canvas.getContainer()).remove('djs-search-open');
    classes$1(this._container).remove('open');
    this._clearResults();
    this._searchInput.value = '';
    this._searchInput.blur();
    this._eventBus.fire('searchPad.closed');
    this._canvas.restoreFocus();
  };

  /**
   * Toggles search pad on/off.
   */
  SearchPad.prototype.toggle = function () {
    this.isOpen() ? this.close() : this.open();
  };

  /**
   * Report state of search pad.
   */
  SearchPad.prototype.isOpen = function () {
    return this._open;
  };

  /**
   * Preselect result entry.
   *
   * @param {HTMLElement} element
   */
  SearchPad.prototype._preselect = function (node) {
    var selectedNode = this._getCurrentResult();

    // already selected
    if (node === selectedNode) {
      return;
    }

    // removing preselection from current node
    if (selectedNode) {
      classes$1(selectedNode).remove(SearchPad.RESULT_SELECTED_CLASS);
    }
    var id = attr$1(node, SearchPad.RESULT_ID_ATTRIBUTE);
    var element = this._results[id].element;
    classes$1(node).add(SearchPad.RESULT_SELECTED_CLASS);
    this._canvas.scrollToElement(element, {
      top: SCROLL_TO_ELEMENT_PADDING
    });
    this._selection.select(element);
    this._eventBus.fire('searchPad.preselected', element);
  };

  /**
   * Select result node.
   *
   * @param {HTMLElement} element
   */
  SearchPad.prototype._select = function (node) {
    var id = attr$1(node, SearchPad.RESULT_ID_ATTRIBUTE);
    var element = this._results[id].element;
    this._cachedSelection = null;
    this._cachedViewbox = null;
    this.close(false);
    this._canvas.scrollToElement(element, {
      top: SCROLL_TO_ELEMENT_PADDING
    });
    this._selection.select(element);
    this._eventBus.fire('searchPad.selected', element);
  };
  SearchPad.prototype._getBoxHtml = function () {
    const box = domify$1(SearchPad.BOX_HTML);
    const input = query(SearchPad.INPUT_SELECTOR, box);
    if (input) {
      input.setAttribute('aria-label', this._translate('Search in diagram'));
    }
    return box;
  };

  /**
   * Creates and appends child node from result tokens and HTML template.
   *
   * @param {HTMLElement} node
   * @param {Token[]} tokens
   * @param {string} template
   */
  function createInnerTextNode(parentNode, tokens, template) {
    var text = createHtmlText(tokens);
    var childNode = domify$1(template);
    childNode.innerHTML = text;
    parentNode.appendChild(childNode);
  }

  /**
   * Create internal HTML markup from result tokens.
   * Caters for highlighting pattern matched tokens.
   *
   * @param {Token[]} tokens
   *
   * @return {string|null}
   */
  function createHtmlText(tokens) {
    var htmlText = '';
    tokens.forEach(function (t) {
      var text = escapeHTML(t.value || t.matched || t.normal);
      var match = t.match || t.matched;
      if (match) {
        htmlText += '<b class="' + SearchPad.RESULT_HIGHLIGHT_CLASS + '">' + text + '</b>';
      } else {
        htmlText += text;
      }
    });
    return htmlText !== '' ? htmlText : null;
  }

  /**
   * CONSTANTS
   */
  SearchPad.CONTAINER_SELECTOR = '.djs-search-container';
  SearchPad.INPUT_SELECTOR = '.djs-search-input input';
  SearchPad.RESULTS_CONTAINER_SELECTOR = '.djs-search-results';
  SearchPad.RESULT_SELECTOR = '.djs-search-result';
  SearchPad.RESULT_SELECTED_CLASS = 'djs-search-result-selected';
  SearchPad.RESULT_SELECTED_SELECTOR = '.' + SearchPad.RESULT_SELECTED_CLASS;
  SearchPad.RESULT_ID_ATTRIBUTE = 'data-result-id';
  SearchPad.RESULT_HIGHLIGHT_CLASS = 'djs-search-highlight';
  SearchPad.BOX_HTML = `<div class="djs-search-container djs-scrollable">
  <div class="djs-search-input">
    <svg class="djs-search-icon" width="14" height="14" viewBox="0 0 14 14" fill="none" xmlns="http://www.w3.org/2000/svg">
      <path fill-rule="evenodd" clip-rule="evenodd" d="M9.0325 8.5H9.625L13.3675 12.25L12.25 13.3675L8.5 9.625V9.0325L8.2975 8.8225C7.4425 9.5575 6.3325 10 5.125 10C2.4325 10 0.25 7.8175 0.25 5.125C0.25 2.4325 2.4325 0.25 5.125 0.25C7.8175 0.25 10 2.4325 10 5.125C10 6.3325 9.5575 7.4425 8.8225 8.2975L9.0325 8.5ZM1.75 5.125C1.75 6.9925 3.2575 8.5 5.125 8.5C6.9925 8.5 8.5 6.9925 8.5 5.125C8.5 3.2575 6.9925 1.75 5.125 1.75C3.2575 1.75 1.75 3.2575 1.75 5.125Z" fill="#22242A"/>
    </svg>
    <input type="text" spellcheck="false" />
  </div>
  <div class="djs-search-results" />
</div>`;
  SearchPad.RESULT_HTML = '<div class="djs-search-result"></div>';
  SearchPad.RESULT_PRIMARY_HTML = '<div class="djs-search-result-primary"></div>';
  SearchPad.RESULT_SECONDARY_HTML = '<p class="djs-search-result-secondary"></p>';

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var SearchPadModule = {
    __depends__: [TranslateModule, OverlaysModule, SelectionModule],
    searchPad: ['type', SearchPad]
  };

  function getLabelAttr(semantic) {
    if (is(semantic, 'dmn:Decision') || is(semantic, 'dmn:BusinessKnowledgeModel') || is(semantic, 'dmn:InputData') || is(semantic, 'dmn:KnowledgeSource')) {
      return 'name';
    }
    if (is(semantic, 'dmn:TextAnnotation')) {
      return 'text';
    }
  }
  function getLabel(element) {
    var semantic = element.businessObject,
      attr = getLabelAttr(semantic);
    if (attr) {
      return semantic[attr] || '';
    }
  }

  /**
   * @typedef {import('diagram-js/lib/core/Canvas').default} Canvas
   * @typedef {import('diagram-js/lib/core/ElementRegistry').default} ElementRegistry
   * @typedef {import('diagram-js/lib/features/search-pad/SearchPad').default} SearchPad
   *
   * @typedef {import('diagram-js/lib/features/search-pad/SearchPadProvider').default
   * } SearchPadProvider
   * @typedef {import('diagram-js/lib/features/search-pad/SearchPadProvider').SearchResult
   * } SearchResult
   */

  /**
   * Provides ability to search for DMN elements.
   *
   * @implements {SearchPadProvider}
   *
   * @param {ElementRegistry} elementRegistry
   * @param {SearchPad} searchPad
   * @param {Canvas} canvas
   */
  function DmnSearchProvider(elementRegistry, searchPad, canvas) {
    this._elementRegistry = elementRegistry;
    this._canvas = canvas;
    searchPad.registerProvider(this);
  }
  DmnSearchProvider.$inject = ['elementRegistry', 'searchPad', 'canvas'];

  /**
   * @param {string} pattern
   *
   * @return {SearchResult[]}
   */
  DmnSearchProvider.prototype.find = function (pattern) {
    const rootElement = this._canvas.getRootElement();
    let elements = this._elementRegistry.filter(function (element) {
      if (element.labelTarget) {
        return false;
      }
      return true;
    });

    // do not include root element
    elements = filter$1(elements, function (element) {
      return element !== rootElement;
    });
    elements = map$1(elements, function (element) {
      return {
        primaryTokens: matchAndSplit(getLabel(element), pattern),
        secondaryTokens: matchAndSplit(element.id, pattern),
        element: element
      };
    });

    // exclude non-matched elements
    elements = filter$1(elements, function (element) {
      return hasMatched(element.primaryTokens) || hasMatched(element.secondaryTokens);
    });
    elements = sortBy(elements, function (element) {
      return getLabel(element.element) + element.element.id;
    });
    return elements;
  };

  /**
   * @param {Token[]} tokens
   *
   * @return {boolean}
   */
  function hasMatched(tokens) {
    const matched = filter$1(tokens, function (token) {
      return !!token.matched;
    });
    return matched.length > 0;
  }

  /**
   * @param {string} text
   * @param {string} pattern
   *
   * @return {Token[]}
   */
  function matchAndSplit(text, pattern) {
    const tokens = [],
      originalText = text;
    if (!text) {
      return tokens;
    }
    text = text.toLowerCase();
    pattern = pattern.toLowerCase();
    const i = text.indexOf(pattern);
    if (i > -1) {
      if (i !== 0) {
        tokens.push({
          normal: originalText.substr(0, i)
        });
      }
      tokens.push({
        matched: originalText.substr(i, pattern.length)
      });
      if (pattern.length + i < text.length) {
        tokens.push({
          normal: originalText.substr(pattern.length + i, text.length)
        });
      }
    } else {
      tokens.push({
        normal: originalText
      });
    }
    return tokens;
  }

  var DmnSearchModule = {
    __depends__: [SearchPadModule],
    __init__: ['dmnSearch'],
    dmnSearch: ['type', DmnSearchProvider]
  };

  /**
   * A viewer that includes mouse navigation facilities
   *
   * @param {Object} options
   */
  function NavigatedViewer(options) {
    Viewer$5.call(this, options);
  }
  e(NavigatedViewer, Viewer$5);
  NavigatedViewer.prototype._navigationModules = [ZoomScroll, MoveCanvas, DmnSearchModule];
  NavigatedViewer.prototype._modules = [].concat(NavigatedViewer.prototype._modules, NavigatedViewer.prototype._navigationModules);

  class Base {
    constructor(attrs) {
      assign$4(this, attrs);
      /**
       * The object that backs up the shape
       *
       * @name Base#businessObject
       * @type Object
       */

      defineProperty(this, 'businessObject', {
        writable: true
      });
    }
  }
  class Root extends Base {
    constructor(attrs) {
      super(attrs);
      /**
       * The tables rows
       *
       * @name Root#rows
       * @type Row
       */

      defineProperty(this, 'rows', {
        enumerable: true,
        value: this.rows || []
      });
      /**
       * The tables columns
       *
       * @name Root#cols
       * @type Col
       */

      defineProperty(this, 'cols', {
        enumerable: true,
        value: this.cols || []
      });
    }
  }
  class Row extends Base {
    constructor(attrs) {
      super(attrs);
      /**
       * Reference to the table
       *
       * @name Row#root
       * @type Root
       */

      defineProperty(this, 'root', {
        writable: true
      });
      /**
       * Reference to contained cells
       *
       * @name Row#cells
       * @type Cell
       */

      defineProperty(this, 'cells', {
        enumerable: true,
        value: this.cells || []
      });
    }
  }
  class Col extends Base {
    constructor(attrs) {
      super(attrs);
      /**
       * Reference to the table
       *
       * @name Col#table
       * @type Root
       */

      defineProperty(this, 'root', {
        writable: true
      });
      /**
       * Reference to contained cells
       *
       * @name Row#cells
       * @type Cell
       */

      defineProperty(this, 'cells', {
        enumerable: true,
        value: this.cells || []
      });
    }
  }
  class Cell extends Base {
    constructor(attrs) {
      super(attrs);
      /**
       * Reference to the row
       *
       * @name Cell#row
       * @type Row
       */

      defineProperty(this, 'row', {
        writable: true
      });
      /**
       * Reference to the col
       *
       * @name Cell#col
       * @type Col
       */

      defineProperty(this, 'col', {
        writable: true
      });
    }
  }
  const TYPES = {
    root: Root,
    row: Row,
    col: Col,
    cell: Cell
  };
  function create(type, attrs) {
    const Type = TYPES[type];
    if (!Type) {
      throw new Error('unknown type ' + type);
    }
    return new Type(attrs);
  } // helpers /////////////

  function defineProperty(el, prop, options) {
    Object.defineProperty(el, prop, options);
  }

  class ElementFactory {
    constructor() {
      this._uid = 12;
    }
    create(type, attrs = {}) {
      if (!attrs.id) {
        attrs.id = type + '_' + this._uid++;
      }
      return create(type, attrs);
    }
    createRoot(attrs) {
      return this.create('root', attrs);
    }
    createRow(attrs) {
      return this.create('row', attrs);
    }
    createCol(attrs) {
      return this.create('col', attrs);
    }
    createCell(attrs) {
      return this.create('cell', attrs);
    }
  }

  let ElementRegistry$1 = class ElementRegistry {
    constructor(eventBus) {
      this._eventBus = eventBus;
      this._elements = {};
      eventBus.on('table.clear', this.clear.bind(this));
    }
    add(element, type) {
      const {
        id
      } = element;
      this._elements[id] = element;
    }
    remove(element) {
      const id = element.id || element;
      delete this._elements[id];
    }
    get(id) {
      return this._elements[id];
    }
    getAll() {
      return values(this._elements);
    }
    forEach(fn) {
      values(this._elements).forEach(element => fn(element));
    }
    filter(fn) {
      return values(this._elements).filter(element => fn(element));
    }
    clear() {
      this._elements = {};
    }
    updateId(element, newId) {
      this._validateId(newId);
      if (typeof element === 'string') {
        element = this.get(element);
      }
      this._eventBus.fire('element.updateId', {
        element: element,
        newId: newId
      });
      this.remove(element);
      element.id = newId;
      this.add(element);
    }
    /**
    * Validate the suitability of the given id and signals a problem
    * with an exception.
    *
    * @param {String} id
    *
    * @throws {Error} if id is empty or already assigned
    */

    _validateId(id) {
      if (!id) {
        throw new Error('element must have an id');
      }
      if (this._elements[id]) {
        throw new Error('element with id ' + id + ' already added');
      }
    }
  };
  ElementRegistry$1.$inject = ['eventBus']; // helpers

  function values(obj) {
    return Object.keys(obj).map(function (k) {
      return obj[k];
    });
  }

  let ChangeSupport$1 = class ChangeSupport {
    constructor(eventBus) {
      this._listeners = {};
      eventBus.on('elements.changed', ({
        elements
      }) => {
        this.elementsChanged(elements);
      });
      eventBus.on('root.remove', context => {
        const oldRootId = context.root.id;
        if (this._listeners[oldRootId]) {
          eventBus.once('root.add', context => {
            const newRootId = context.root.id;
            this.updateId(oldRootId, newRootId);
          });
        }
      });
      eventBus.on('element.updateId', ({
        element,
        newId
      }) => {
        this.updateId(element.id, newId);
      });
    }
    elementsChanged(elements) {
      const invoked = {};
      const elementsLength = elements.length;
      for (let i = 0; i < elementsLength; i++) {
        const {
          id
        } = elements[i];
        if (invoked[id]) {
          return;
        }
        invoked[id] = true;
        const listenersLength = this._listeners[id] && this._listeners[id].length;
        if (listenersLength) {
          for (let j = 0; j < listenersLength; j++) {
            // listeners might remove themselves before they get called
            this._listeners[id][j] && this._listeners[id][j]();
          }
        }
      }
    }
    onElementsChanged(id, listener) {
      if (!this._listeners[id]) {
        this._listeners[id] = [];
      } // avoid push for better performance

      this._listeners[id][this._listeners[id].length] = listener;
    }
    offElementsChanged(id, listener) {
      if (!this._listeners[id]) {
        return;
      }
      if (listener) {
        const idx = this._listeners[id].indexOf(listener);
        if (idx !== -1) {
          this._listeners[id].splice(idx, 1);
        }
      } else {
        this._listeners[id].length = 0;
      }
    }
    updateId(oldId, newId) {
      if (this._listeners[oldId]) {
        this._listeners[newId] = this._listeners[oldId];
        delete this._listeners[oldId];
      }
    }
  };
  ChangeSupport$1.$inject = ['eventBus'];

  const DEFAULT_PRIORITY$1 = 1000;
  let Components$1 = class Components {
    constructor() {
      this._listeners = {};
    }
    getComponent(type, context) {
      const listeners = this._listeners[type];
      if (!listeners) {
        return;
      }
      let component;
      for (let i = 0; i < listeners.length; i++) {
        component = listeners[i].callback(context);
        if (component) {
          break;
        }
      }
      return component;
    }
    getComponents(type, context) {
      const listeners = this._listeners[type];
      const components = [];
      if (!listeners) {
        return components;
      }
      for (let i = 0; i < listeners.length; i++) {
        const component = listeners[i].callback(context);
        if (component) {
          components.push(component);
        }
      }
      if (!components.length) {
        return components;
      }
      return components;
    }
    onGetComponent(type, priority, callback) {
      if (isFunction$2(priority)) {
        callback = priority;
        priority = DEFAULT_PRIORITY$1;
      }
      if (!isNumber$1(priority)) {
        throw new Error('priority must be a number');
      }
      const listeners = this._getListeners(type);
      let existingListener, idx;
      const newListener = {
        priority,
        callback
      };
      for (idx = 0; existingListener = listeners[idx]; idx++) {
        if (existingListener.priority < priority) {
          // prepend newListener at before existingListener
          listeners.splice(idx, 0, newListener);
          return;
        }
      }
      listeners.push(newListener);
    }
    offGetComponent(type, callback) {
      const listeners = this._getListeners(type);
      let listener, listenerCallback, idx;
      if (callback) {
        // move through listeners from back to front
        // and remove matching listeners
        for (idx = listeners.length - 1; listener = listeners[idx]; idx--) {
          listenerCallback = listener.callback;
          if (listenerCallback === callback) {
            listeners.splice(idx, 1);
          }
        }
      } else {
        // clear listeners
        listeners.length = 0;
      }
    }
    _getListeners(type) {
      let listeners = this._listeners[type];
      if (!listeners) {
        this._listeners[type] = listeners = [];
      }
      return listeners;
    }
  };

  class TableComponent extends Component {
    constructor(props) {
      super(props);
      const injector = this._injector = props.injector;
      this._sheet = injector.get('sheet');
      this._changeSupport = injector.get('changeSupport');
      this._components = injector.get('components');
      this._eventBus = injector.get('eventBus');
      const throttle = injector.get('throttle');
      this.onElementsChanged = this.onElementsChanged.bind(this);
      this.onScroll = throttle(this.onScroll.bind(this));
    }
    onElementsChanged() {
      this.forceUpdate();
    }
    onScroll() {
      this._eventBus.fire('sheet.scroll');
    }
    getChildContext() {
      return {
        changeSupport: this._changeSupport,
        components: this._components,
        injector: this._injector
      };
    }
    componentWillMount() {
      const {
        id
      } = this._sheet.getRoot();
      this._changeSupport.onElementsChanged(id, this.onElementsChanged);
    }
    componentWillUnmount() {
      const {
        id
      } = this._sheet.getRoot();
      this._changeSupport.offElementsChanged(id, this.onElementsChanged);
    }
    render() {
      const {
        rows,
        cols
      } = this._sheet.getRoot();
      const beforeTableComponents = this._components.getComponents('table.before');
      const afterTableComponents = this._components.getComponents('table.after');
      const Head = this._components.getComponent('table.head');
      const Body = this._components.getComponent('table.body');
      const Foot = this._components.getComponent('table.foot');
      return createVNode(1, "div", "tjs-container", [beforeTableComponents && beforeTableComponents.map((Component, index) => createComponentVNode(2, Component, null, index)), createVNode(1, "div", "tjs-table-container", createVNode(1, "table", "tjs-table", [Head && createComponentVNode(2, Head, {
        "rows": rows,
        "cols": cols
      }), Body && createComponentVNode(2, Body, {
        "rows": rows,
        "cols": cols
      }), Foot && createComponentVNode(2, Foot, {
        "rows": rows,
        "cols": cols
      })], 0), 2, {
        "onScroll": this.onScroll
      }), afterTableComponents && afterTableComponents.map((Component, index) => createComponentVNode(2, Component, null, index))], 0);
    }
  }

  let Renderer$1 = class Renderer {
    constructor(changeSupport, components, config, eventBus, injector) {
      const {
        container
      } = config;
      this._container = container;
      eventBus.on('root.added', () => {
        render(createComponentVNode(2, TableComponent, {
          "injector": injector
        }), container);
      });
      eventBus.on('root.remove', () => {
        render(null, container);
      });
    }
    getContainer() {
      return this._container;
    }
  };
  Renderer$1.$inject = ['changeSupport', 'components', 'config.renderer', 'eventBus', 'injector'];

  var renderModule = {
    __init__: ['changeSupport', 'components', 'renderer'],
    changeSupport: ['type', ChangeSupport$1],
    components: ['type', Components$1],
    renderer: ['type', Renderer$1]
  };

  class Sheet {
    constructor(elementRegistry, eventBus) {
      this._elementRegistry = elementRegistry;
      this._eventBus = eventBus;
      this._root = null;
      eventBus.on('table.clear', () => {
        this.setRoot(null);
      });
    }
    setRoot(root) {
      if (this._root) {
        const oldRoot = this._root;
        this._eventBus.fire('root.remove', {
          root: oldRoot
        });
        this._root = null;
        this._eventBus.fire('root.removed', {
          root: oldRoot
        });
      }
      if (root) {
        this._eventBus.fire('root.add', {
          root
        });
      }
      this._root = root;
      if (root) {
        this._eventBus.fire('root.added', {
          root
        });
      }
    }
    getRoot() {
      if (!this._root) {
        this.setRoot({
          id: '__implicitroot',
          rows: [],
          cols: []
        });
      }
      return this._root;
    }
    /**
     * Add row to sheet.
     *
     * @param {Object} row - Row.
     */

    addRow(row, index) {
      const root = this.getRoot();
      if (root.cols.length != row.cells.length) {
        throw new Error('number of cells is not equal to number of cols');
      }
      if (typeof index === 'undefined') {
        index = root.rows.length;
      }
      addAtIndex(index, root.rows, row);
      row.root = root;
      this._elementRegistry.add(row);
      row.cells.forEach((cell, idx) => {
        this._elementRegistry.add(cell);
        cell.row = row;
        cell.col = root.cols[idx];
        addAtIndex(index, root.cols[idx].cells, cell);
      });
      this._eventBus.fire('row.add', {
        row
      });
      return row;
    }
    /**
     * Remove row from sheet.
     *
     * @param {Object|string} row - Row or row ID.
     */

    removeRow(row) {
      const root = this.getRoot();
      if (typeof row === 'string') {
        row = this._elementRegistry.get(row);
      }
      const index = root.rows.indexOf(row);
      if (index === -1) {
        return;
      }
      removeAtIndex(index, root.rows);
      row.root = undefined;
      this._elementRegistry.remove(row);
      row.cells.forEach((cell, idx) => {
        this._elementRegistry.remove(cell);
        cell.col = undefined;
        removeAtIndex(index, root.cols[idx].cells);
      });
      this._eventBus.fire('row.remove', {
        row
      });
    }
    /**
     * Add col to sheet.
     *
     * @param {Object} col
     * @param {Number} [index]
     */

    addCol(col, index) {
      const root = this.getRoot();
      this._elementRegistry.add(col);
      if (root.rows.length != col.cells.length) {
        throw new Error('number of cells is not equal to number of rows');
      }
      if (typeof index === 'undefined') {
        index = root.cols.length;
      }
      addAtIndex(index, root.cols, col);
      col.root = root;
      col.cells.forEach((cell, idx) => {
        this._elementRegistry.add(cell);
        cell.col = col;
        cell.row = root.rows[idx];
        addAtIndex(index, root.rows[idx].cells, cell);
      });
      this._eventBus.fire('col.add', {
        col
      });
      return col;
    }
    /**
     * Remove col from sheet.
     *
     * @param {Object|string} col - Col or col ID.
     */

    removeCol(col) {
      const root = this.getRoot();
      if (typeof col === 'string') {
        col = this._elementRegistry.get(col);
      }
      const index = root.cols.indexOf(col);
      if (index === -1) {
        return;
      }
      removeAtIndex(index, root.cols);
      col.root = undefined;
      this._elementRegistry.remove(col);
      col.cells.forEach((cell, idx) => {
        this._elementRegistry.remove(cell);
        cell.row = undefined;
        removeAtIndex(index, root.rows[idx].cells);
      });
      this._eventBus.fire('col.remove', {
        col
      });
    }
    resized() {
      this._eventBus.fire('sheet.resized');
    }
  }
  Sheet.$inject = ['elementRegistry', 'eventBus']; // helpers /////////////

  /**
   * Insert value
   *
   * @param {number} index - Index to insert value at.
   * @param {Array} array - Array to insert value into.
   * @param {*} value - Value to insert.
   */

  function addAtIndex(index, array, value) {
    return array.splice(index, 0, value);
  }
  /**
   *
   * @param {number} index - Index to remove.
   * @param {Array} array - Array to remove from.
   */

  function removeAtIndex(index, array) {
    return array.splice(index, 1);
  }

  /**
   * A factory to create a configurable throttler.
   *
   * @param {number|boolean} [config=true]
   */

  function ThrottleFactory(config = true) {
    const timeout = typeof config === 'number' ? config : config ? 300 : 0;
    if (timeout) {
      return fn => throttle(fn, timeout);
    } else {
      return fn => fn;
    }
  }
  ThrottleFactory.$inject = ['config.throttle'];

  var core$1 = {
    __depends__: [renderModule],
    __init__: ['elementFactory', 'sheet'],
    elementFactory: ['type', ElementFactory],
    elementRegistry: ['type', ElementRegistry$1],
    eventBus: ['type', EventBus],
    sheet: ['type', Sheet],
    throttle: ['factory', ThrottleFactory]
  };

  class Table {
    constructor(options = {}) {
      let {
        injector
      } = options;
      if (!injector) {
        let {
          modules,
          config
        } = this._init(options);
        injector = createInjector$1(config, modules);
      }
      this.get = injector.get;
      this.invoke = injector.invoke;
      this.get('eventBus').fire('table.init');
      this.get('eventBus').fire('diagram.init');
    }
    /**
     * Intialize table and return modules and config used for creation.
     *
     * @param  {Object} options
     *
     * @return {Object} { modules=[], config }
     */

    _init(options) {
      let {
        modules,
        ...config
      } = options;
      return {
        modules,
        config
      };
    }
    /**
     * Destroys the table. This results in removing the attachment from the container.
     */

    destroy() {
      const eventBus = this.get('eventBus');
      eventBus.fire('table.destroy');
      eventBus.fire('diagram.destroy');
    }
    /**
     * Clears the table. Should be used to reset the state of any stateful services.
     */

    clear() {
      const eventBus = this.get('eventBus');
      eventBus.fire('table.clear');
      eventBus.fire('diagram.clear');
    }
  }
  function createInjector$1(config, modules) {
    const bootstrapModules = [{
      config: ['value', config]
    }, core$1].concat(modules || []);
    const injector = new Injector(bootstrapModules);
    injector.init();
    return injector;
  }

  function elementToString(element) {
    if (!element) {
      return '<null>';
    }
    const id = element.id ? ` id="${element.id}"` : '';
    return `<${element.$type}${id} />`;
  }

  function TableTreeWalker(handler, options) {
    function visit(element, ctx, definitions) {
      var gfx = element.gfx;

      // avoid multiple rendering of elements
      if (gfx) {
        throw new Error(`already rendered ${elementToString(element)}`);
      }

      // call handler
      return handler.element(element, ctx, definitions);
    }
    function visitTable(element) {
      return handler.table(element);
    }

    // Semantic handling //////////////////////

    function handleDecision(decision) {
      if (!decision.id) {
        decision.id = 'decision';
      }
      const table = decision.decisionLogic;
      if (table) {
        if (!table.output) {
          throw new Error(`missing output for ${elementToString(table)}`);
        }
        const ctx = visitTable(table);
        if (table.input) {
          handleClauses(table.input, ctx, table);
        }
        handleClauses(table.output, ctx, table);

        // if any input or output clauses (columns) were added
        // make sure that for each rule the according input/output entry is created
        handleRules(table.rule, ctx, table);
      } else {
        throw new Error(`no table for ${elementToString(decision)}`);
      }
    }
    function handleClauses(clauses, context, definitions) {
      forEach$3(clauses, function (e) {
        visit(e, context, definitions);
      });
    }
    function handleRules(rules, context, definitions) {
      forEach$3(rules, function (e) {
        visit(e, context, definitions);
        handleEntry(e.inputEntry, e);
        handleEntry(e.outputEntry, e);
      });
    }
    function handleEntry(entry, context, definitions) {
      forEach$3(entry, function (e) {
        visit(e, context, definitions);
      });
    }

    // API //////////////////////

    return {
      handleDecision: handleDecision
    };
  }

  /**
   * Import the decision table into a table.
   *
   * Errors and warnings are reported through the specified callback.
   *
   * @param  {decisionTable} decisionTable instance of DecisionTable
   * @param  {ModdleElement} decision moddle element
   * @param  {Function} done
   *         the callback, invoked with (err, [ warning ]) once the import is done
   */
  function importDecision(decisionTable, decision, done) {
    const importer = decisionTable.get('tableImporter'),
      eventBus = decisionTable.get('eventBus'),
      sheet = decisionTable.get('sheet');
    decisionTable.get('modeling', false);
    let error,
      warnings = [];
    function render(decision) {
      const visitor = {
        create(type, parent, clause, rule) {
          return importer.create(type, parent, clause, rule);
        },
        table(element) {
          return importer.add(element);
        },
        element(element, parentShape, definitions) {
          return importer.add(element, parentShape, definitions);
        },
        error(message, context) {
          warnings.push({
            message: message,
            context: context
          });
        }
      };
      const walker = new TableTreeWalker(visitor);

      // import
      walker.handleDecision(decision);
    }
    eventBus.fire('import.render.start', {
      decision: decision
    });
    try {
      render(decision);
    } catch (e) {
      error = e;
    }
    eventBus.fire('import.render.complete', {
      error: error,
      warnings: warnings
    });
    eventBus.fire('elements.changed', {
      elements: [sheet.getRoot()]
    });
    done(error, warnings);
  }

  function newSet() {
    return {
      elements: [],
      index: {}
    };
  }
  function add$1(set, element) {
    const {
      elements,
      index
    } = set;
    if (index[element]) {
      return set;
    } else {
      return {
        elements: [...elements, element],
        index: {
          ...index,
          [element]: true
        }
      };
    }
  }
  function join(set, separator) {
    return set.elements.join(separator);
  }
  function classNames(...args) {
    let set = newSet();
    args.forEach(function (item) {
      const type = typeof item;
      if (type === 'string' && item.length > 0) {
        set = add$1(set, item);
      } else if (type === 'object' && item !== null) {
        Object.keys(item).forEach(function (key) {
          const value = item[key];
          if (value) {
            set = add$1(set, key);
          }
        });
      }
    });
    return join(set, ' ');
  }

  function inject(component) {
    const Type = component.constructor;
    return injectType(Type, component);
  }
  function injectType(Type, component) {
    const annotation = Type.$inject;
    if (!annotation) {
      return;
    }
    const {
      injector
    } = component.context;
    const setupFn = [...annotation, function (...args) {
      for (const idx in args) {
        const name = annotation[idx];
        const value = args[idx];
        component[name] = value;
      }
    }];
    injector.invoke(setupFn);
  }

  /**
   * Composes a number of functions.
   *
   * All receive the the same arguments; the chain is interruped as soon
   * as one function returns a value.
   *
   * @param  {Object}    self
   * @param  {...Function} fns
   *
   * @return {Object}
   */
  function compose(self, ...fns) {
    return function (...args) {
      let result;
      fns.forEach(function (fn) {
        result = fn.call(self, ...args);
        if (typeof result !== 'undefined') {
          return false;
        }
      });
      return result;
    }.bind(self);
  }

  /**
   * A Component and injection aware mixin mechanism.
   *
   * @param {Component} component
   * @param {Object|Function} mixinDef
   */

  function mixin(component, mixinDef) {
    Object.keys(mixinDef).forEach(function (key) {
      if (key === '$inject' || key === '__init') {
        return;
      }
      const mixinFn = mixinDef[key];
      if (key === 'constructor') {
        mixinFn.call(component, component.props, component.context);
      }
      const componentFn = component[key];
      if (typeof componentFn !== 'undefined') {
        if (typeof componentFn !== 'function') {
          throw new Error(`failed to mixin <${key}>: cannot combine with non-fn component value`);
        }
        component[key] = compose(component, componentFn, mixinFn);
      } else {
        component[key] = mixinFn.bind(component);
      }
    });
    if ('$inject' in mixinDef) {
      injectType(mixinDef, component);
    } // call initializer

    if ('__init' in mixinDef) {
      mixinDef.__init.call(component, component.props, component.context);
    }
  }

  /**
   * A mixin to make an element _selection aware_.
   */

  const SelectionAware = {
    getSelectionClasses() {
      const {
        selected,
        selectedSecondary,
        focussed
      } = this.state;
      return classNames({
        'selected': selected,
        'selected-secondary': selectedSecondary,
        'focussed': focussed
      });
    },
    selectionChanged(newSelection) {
      // newSelection = { selected, selectedSecondary, focussed }
      this.setState(newSelection);
    },
    componentWillUpdate(newProps) {
      if (newProps.elementId !== this.props.elementId) {
        this.updateSelectionSubscription(false);
      }
    },
    componentDidUpdate(oldProps) {
      if (oldProps.elementId !== this.props.elementId) {
        this.updateSelectionSubscription(true);
      }
    },
    componentDidMount() {
      this.updateSelectionSubscription(true);
    },
    componentWillUnmount() {
      this.updateSelectionSubscription(false);
    },
    updateSelectionSubscription(enable) {
      const {
        elementId
      } = this.props;
      if (!elementId) {
        return;
      }
      if (elementId) {
        this.eventBus[enable ? 'on' : 'off'](`selection.${elementId}.changed`, this.selectionChanged);
      }
    }
  };
  SelectionAware.$inject = ['eventBus'];

  class BaseCell extends Component {
    constructor(props, context) {
      super(props, context);
      mixin(this, SelectionAware);
      inject(this);
    }
    getRenderProps(...cls) {
      const {
        className,
        elementId,
        coords,
        ...props
      } = this.props;
      const baseProps = {
        className: classNames(...cls, this.getSelectionClasses(), className)
      };
      if (elementId) {
        baseProps['data-element-id'] = elementId;
      }
      if (coords) {
        baseProps['data-coords'] = coords;
      }
      return {
        ...baseProps,
        ...props
      };
    }
  }

  class HeaderCell extends BaseCell {
    constructor(props, context) {
      super(props, context);
      this.state = {};
    }
    render() {
      const {
        children
      } = this.props;
      const props = this.getRenderProps('cell');
      return normalizeProps(createVNode(1, "td", null, children, 0, {
        ...props
      }));
    }
  }

  /**
   * A simple slot extension, built upon the components service.
   *
   * @type {Object}
   */
  const ComponentWithSlots = {
    slotFill(slotProps, DefaultFill) {
      const {
        type,
        context,
        ...props
      } = slotProps;
      const Fill = this.components.getComponent(type, context) || DefaultFill;
      if (Fill) {
        return normalizeProps(createComponentVNode(2, Fill, {
          ...context,
          ...props
        }));
      }
      return null;
    },
    slotFills(slotProps) {
      const {
        type,
        context,
        ...props
      } = slotProps;
      const fills = this.components.getComponents(type, context);
      return fills.map(Fill => normalizeProps(createComponentVNode(2, Fill, {
        ...context,
        ...props
      })));
    }
  };
  ComponentWithSlots.$inject = ['components'];

  const MIN_WIDTH = 400;
  class AnnotationHeader extends Component {
    constructor(props, context) {
      super(props, context);
      mixin(this, ComponentWithSlots);
      inject(this);
    }
    componentDidMount() {
      this.changeSupport.onElementsChanged(this.getRoot(), this.onElementsChanged);
    }
    componentWillUnmount() {
      this.changeSupport.offElementsChanged(this.getRoot(), this.onElementsChanged);
    }
    onElementsChanged = () => {
      this.forceUpdate();
    };
    getRoot() {
      return this.sheet.getRoot();
    }
    render() {
      const decisionTable = this.getRoot();
      const annotationsWidth = decisionTable.businessObject.get('annotationsWidth');
      const width = (annotationsWidth || MIN_WIDTH) + 'px';
      return createVNode(1, "th", "annotation header", [this.slotFills({
        type: 'cell-inner',
        context: {
          cellType: 'annotations',
          col: this.sheet.getRoot(),
          minWidth: MIN_WIDTH
        }
      }), this.translate('Annotations')], 0, {
        "style": {
          width
        }
      });
    }
  }
  AnnotationHeader.$inject = ['changeSupport', 'sheet', 'translate'];

  function AnnotationCell(props) {
    const {
      row
    } = props;
    const {
      id,
      description
    } = row.businessObject;
    return createComponentVNode(2, HeaderCell, {
      "className": "annotation",
      "elementId": id,
      children: description || '-'
    });
  }

  function AnnotationsProvider(components) {
    components.onGetComponent('cell', ({
      cellType
    }) => {
      if (cellType === 'after-label-cells') {
        return AnnotationHeader;
      } else if (cellType === 'after-rule-cells') {
        return AnnotationCell;
      }
    });
  }
  AnnotationsProvider.$inject = ['components'];

  var annotationsModule = {
    __init__: ['annotationsProvider'],
    annotationsProvider: ['type', AnnotationsProvider]
  };

  function elementData(semantic, attrs) {
    return assign$4({
      id: semantic.id,
      type: semantic.$type,
      businessObject: semantic
    }, attrs);
  }
  class TableImporter {
    constructor(elementFactory, eventBus, sheet) {
      this._elementFactory = elementFactory;
      this._eventBus = eventBus;
      this._sheet = sheet;
    }

    /**
     * Add DMN element.
     */
    add(semantic) {
      let element;

      // decision table
      if (is(semantic, 'dmn:DecisionTable')) {
        element = this._elementFactory.createRoot(elementData(semantic));
        this._sheet.setRoot(element);
      }

      // input clause
      else if (is(semantic, 'dmn:InputClause')) {
        element = this._elementFactory.createCol(elementData(semantic));
        this._sheet.addCol(element);
      }

      // output clause
      else if (is(semantic, 'dmn:OutputClause')) {
        element = this._elementFactory.createCol(elementData(semantic));
        this._sheet.addCol(element);
      }

      // rule
      else if (is(semantic, 'dmn:DecisionRule')) {
        if (!semantic.inputEntry) {
          semantic.inputEntry = [];
        }
        if (!semantic.outputEntry) {
          semantic.outputEntry = [];
        }
        const cells = [...semantic.inputEntry, ...semantic.outputEntry].map(entry => {
          return this._elementFactory.createCell(elementData(entry));
        });
        element = this._elementFactory.createRow(assign$4(elementData(semantic), {
          cells
        }));
        this._sheet.addRow(element);
      }
      this._eventBus.fire('dmnElement.added', {
        element: element
      });
      return element;
    }
  }
  TableImporter.$inject = ['elementFactory', 'eventBus', 'sheet'];

  var importModule = {
    __depends__: [TranslateModule],
    tableImporter: ['type', TableImporter]
  };

  var coreModule = {
    __depends__: [importModule, renderModule]
  };

  class DecisionTableHead extends Component {
    constructor(props, context) {
      super(props, context);
      mixin(this, ComponentWithSlots);
      this._sheet = context.injector.get('sheet');
      this._changeSupport = context.changeSupport;
    }
    onElementsChanged = () => {
      this.forceUpdate();
    };
    componentWillMount() {
      const root = this._sheet.getRoot();
      this._changeSupport.onElementsChanged(root.id, this.onElementsChanged);
    }
    componentWillUnmount() {
      const root = this._sheet.getRoot();
      this._changeSupport.offElementsChanged(root.id, this.onElementsChanged);
    }
    render() {
      const root = this._sheet.getRoot();
      if (!is(root, 'dmn:DMNElement')) {
        return null;
      }
      const businessObject = getBusinessObject(root);
      const inputs = businessObject.input,
        outputs = businessObject.output;
      return createVNode(1, "thead", null, createVNode(1, "tr", null, [createVNode(1, "th", "index-column"), this.slotFills({
        type: 'cell',
        context: {
          cellType: 'before-label-cells'
        }
      }), inputs && inputs.map((input, index) => {
        const width = input.width || '192px';
        return this.slotFill({
          type: 'cell',
          context: {
            cellType: 'input-header',
            input,
            index,
            inputsLength: inputs.length,
            width
          },
          key: input.id
        }, DefaultInputHeaderCell);
      }), outputs.map((output, index) => {
        return this.slotFill({
          type: 'cell',
          context: {
            cellType: 'output-header',
            output,
            index,
            outputsLength: outputs.length
          },
          key: output.id
        }, DefaultOutputHeaderCell);
      }), this.slotFills({
        type: 'cell',
        context: {
          cellType: 'after-label-cells'
        }
      })], 0), 2);
    }
  }

  // default components ///////////////////////

  function DefaultInputHeaderCell(props, context) {
    const {
      input,
      className,
      index
    } = props;
    const {
      label,
      inputExpression,
      inputValues
    } = input;
    const translate = context.injector.get('translate');
    const actualClassName = (className || '') + ' input-cell';
    return createVNode(1, "th", actualClassName, [createVNode(1, "div", "clause", index === 0 ? translate('When') : translate('And'), 0), label ? createVNode(1, "div", "input-label", label, 0, {
      "title": translate('Input label: ') + label
    }) : createVNode(1, "div", "input-expression", inputExpression.text, 0, {
      "title": translate('Input expression: ') + inputExpression.text
    }), createVNode(1, "div", "input-variable", inputValues && inputValues.text || inputExpression.typeRef, 0, {
      "title": inputValues && inputValues.text ? translate('Input values') : translate('Input type')
    })], 0, {
      "data-col-id": input.id
    }, input.id);
  }
  function DefaultOutputHeaderCell(props, context) {
    const {
      output,
      className,
      index
    } = props;
    const {
      label,
      name,
      outputValues,
      typeRef
    } = output;
    const translate = context.injector.get('translate');
    const actualClassName = (className || '') + ' output-cell';
    return createVNode(1, "th", actualClassName, [createVNode(1, "div", "clause", index === 0 ? translate('Then') : translate('And'), 0), label ? createVNode(1, "div", "output-label", label, 0, {
      "title": translate('Output label')
    }) : createVNode(1, "div", "output-name", name, 0, {
      "title": translate('Output name')
    }), createVNode(1, "div", "output-variable", outputValues && outputValues.text || typeRef, 0, {
      "title": outputValues && outputValues.text ? translate('Output values') : translate('Output type')
    })], 0, null, output.id);
  }

  function DecisionTableHeadProvider(components) {
    components.onGetComponent('table.head', () => DecisionTableHead);
  }
  DecisionTableHeadProvider.$inject = ['components'];

  var decisionTableHeadModule = {
    __init__: ['decisionTableHeadProvider'],
    decisionTableHeadProvider: ['type', DecisionTableHeadProvider]
  };

  class DecisionTablePropertiesComponent extends Component {
    constructor(props, context) {
      super(props, context);
      this._translate = context.injector.get('translate');
      inject(this);
    }
    render() {
      const root = this.sheet.getRoot();
      if (!is(root, 'dmn:DMNElement')) {
        return null;
      }
      const {
        name
      } = root.businessObject.$parent;
      const HitPolicy = this.components.getComponent('hit-policy') || NullComponent;
      return createVNode(1, "div", "decision-table-properties", [createVNode(1, "div", "decision-table-name", name, 0, {
        "title": this._translate('Decision name: ') + name
      }), createVNode(1, "div", "decision-table-header-separator"), createComponentVNode(2, HitPolicy)], 4);
    }
  }
  DecisionTablePropertiesComponent.$inject = ['sheet', 'components'];
  function NullComponent() {
    return null;
  }

  const LOW_PRIORITY$1 = 500;
  class DecisionTableProperties {
    constructor(components) {
      components.onGetComponent('table.before', LOW_PRIORITY$1, () => {
        return DecisionTablePropertiesComponent;
      });
    }
  }
  DecisionTableProperties.$inject = ['components'];

  var decisionTablePropertiesModule = {
    __init__: ['decisionTableProperties'],
    decisionTableProperties: ['type', DecisionTableProperties]
  };

  class DecisionRulesIndexCellComponent extends Component {
    render() {
      const {
        row,
        rowIndex
      } = this.props;
      const {
        components
      } = this.context;
      const innerComponents = components.getComponents('cell-inner', {
        cellType: 'rule-index',
        row,
        rowIndex
      });
      return createVNode(1, "td", "rule-index", [innerComponents && innerComponents.map(InnerComponent => createComponentVNode(2, InnerComponent, {
        "row": row,
        "rowIndex": rowIndex
      })), rowIndex + 1], 0, {
        "data-element-id": row.id,
        "data-row-id": row.id
      });
    }
  }

  class DecisionRuleIndices {
    constructor(components) {
      components.onGetComponent('cell', ({
        cellType
      }) => {
        if (cellType === 'before-rule-cells') {
          return DecisionRulesIndexCellComponent;
        }
      });
    }
  }
  DecisionRuleIndices.$inject = ['components'];

  var decisionRuleIndicesModule = {
    __init__: ['decisionRuleIndices'],
    decisionRuleIndices: ['type', DecisionRuleIndices]
  };

  const EXPRESSION_LANGUAGE_OPTIONS = [{
    label: 'FEEL',
    value: 'feel'
  }];

  /**
   * @typedef ExpressionLanguageDescriptor
   * @property {string} value - value inserted into XML
   * @property {string} label - human-readable label
   */

  /**
   * Provide options and defaults of expression languages via config.
   *
   * @example
   *
   * // there will be two languages available with FEEL as default
   * const editor = new DmnJS({
   *   common: {
   *     expressionLanguages: {
   *       options: [{
   *         value: 'feel',
   *         label: 'FEEL'
   *       }, {
   *         value: 'juel',
   *         label: 'JUEL'
   *       }],
   *       defaults: {
   *         editor: 'feel'
   *       }
   *     }
   *   }
   * })
   */
  class ExpressionLanguages {
    constructor(injector) {
      this._injector = injector;
      const config = injector.get('config.expressionLanguages') || {};
      this._config = {
        options: EXPRESSION_LANGUAGE_OPTIONS,
        defaults: {
          editor: 'feel'
        }
      };

      // first assign the list of languages as it might be required for the legacy defaults
      if (config.options) {
        this._config.options = config.options;
      }
      const legacyDefaults = this._getLegacyDefaults();
      assign$4(this._config.defaults, legacyDefaults, config.defaults);
    }

    /**
     * Get default expression language for a component or the editor if `componentName`
     * is not provided.
     *
     * @param {string} [componentName]
     * @returns {ExpressionLanguageDescriptor}
     */
    getDefault(componentName) {
      const {
        defaults
      } = this._config;
      const defaultFromConfig = defaults[componentName] || defaults.editor;
      return this._getLanguageByValue(defaultFromConfig) || this.getAll()[0];
    }

    /**
     * Get label for provided expression language.
     *
     * @param {string} expressionLanguageValue - value from XML
     * @returns {string}
     */
    getLabel(expressionLanguageValue) {
      const langauge = this._getLanguageByValue(expressionLanguageValue);
      return langauge ? langauge.label : expressionLanguageValue;
    }

    /**
     * Get list of configured expression languages.
     *
     * @returns {ExpressionLanguageDescriptor[]}
     */
    getAll() {
      return this._config.options;
    }
    _getLegacyDefaults() {
      const defaults = {},
        injector = this._injector;
      const inputCellValue = injector.get('config.defaultInputExpressionLanguage');
      const outputCellValue = injector.get('config.defaultOutputExpressionLanguage');
      if (inputCellValue) {
        defaults.inputCell = inputCellValue;
      }
      if (outputCellValue) {
        defaults.outputCell = outputCellValue;
      }
      return defaults;
    }
    _getLanguageByValue(value) {
      return find$2(this.getAll(), language => value === language.value);
    }
  }
  ExpressionLanguages.$inject = ['injector'];

  var ExpressionLanguagesModule = {
    __init__: ['expressionLanguages'],
    expressionLanguages: ['type', ExpressionLanguages]
  };

  class DecisionRulesBodyComponent extends Component {
    render({
      rows,
      cols
    }) {
      const {
        components
      } = this.context;
      return createVNode(1, "tbody", null, rows.map((row, rowIndex) => {
        const RowComponent = components.getComponent('row', {
          rowType: 'rule'
        });
        return RowComponent && createComponentVNode(2, RowComponent, {
          "row": row,
          "rowIndex": rowIndex,
          "cols": cols
        }, row.id);
      }), 0);
    }
  }

  class DecisionRulesRowComponent extends Component {
    constructor(props, context) {
      super(props, context);
      mixin(this, ComponentWithSlots);
    }
    render() {
      const {
        row,
        rowIndex,
        cols
      } = this.props;
      const {
        cells
      } = row;
      return createVNode(1, "tr", null, [this.slotFills({
        type: 'cell',
        context: {
          cellType: 'before-rule-cells',
          row,
          rowIndex
        }
      }), cells.map((cell, colIndex) => {
        return this.slotFill({
          type: 'cell',
          context: {
            cellType: 'rule',
            cell,
            rowIndex: rowIndex,
            colIndex: colIndex
          },
          key: cell.id,
          row,
          col: cols[colIndex]
        });
      }), this.slotFills({
        type: 'cell',
        context: {
          cellType: 'after-rule-cells',
          row,
          rowIndex
        }
      })], 0);
    }
  }

  class DecisionRulesCellComponent extends Component {
    render() {
      const {
        cell,
        row,
        col
      } = this.props;
      if (is(cell, 'dmn:UnaryTests')) {
        return createComponentVNode(2, HeaderCell, {
          "className": "input-cell",
          "elementId": cell.id,
          "data-row-id": row.id,
          "data-col-id": col.id,
          children: cell.businessObject.text
        });
      } else {
        return createComponentVNode(2, HeaderCell, {
          "className": "output-cell",
          "elementId": cell.id,
          "data-row-id": row.id,
          "data-col-id": col.id,
          children: cell.businessObject.text
        });
      }
    }
  }

  class Rules {
    constructor(components) {
      components.onGetComponent('table.body', () => DecisionRulesBodyComponent);
      components.onGetComponent('row', ({
        rowType
      }) => {
        if (rowType === 'rule') {
          return DecisionRulesRowComponent;
        }
      });
      components.onGetComponent('cell', ({
        cellType
      }) => {
        if (cellType === 'rule') {
          return DecisionRulesCellComponent;
        }
      });
    }
  }
  Rules.$inject = ['components'];

  var decisionRulesModule = {
    __depends__: [ExpressionLanguagesModule],
    __init__: ['decisionRules'],
    decisionRules: ['type', Rules]
  };

  /* eslint max-len: 0 */
  const HIT_POLICIES = [{
    label: 'Unique',
    value: {
      hitPolicy: 'UNIQUE',
      aggregation: undefined
    },
    explanation: 'No overlap is possible and all rules are disjoint. Only a single rule can be matched'
  }, {
    label: 'First',
    value: {
      hitPolicy: 'FIRST',
      aggregation: undefined
    },
    explanation: 'Rules may overlap. The first matching rule will be chosen'
  }, {
    label: 'Priority',
    value: {
      hitPolicy: 'PRIORITY',
      aggregation: undefined
    },
    explanation: 'Rules may overlap. The one with the highest priority will be chosen'
  }, {
    label: 'Any',
    value: {
      hitPolicy: 'ANY',
      aggregation: undefined
    },
    explanation: 'Rules may overlap. Their output have to match'
  }, {
    label: 'Collect',
    value: {
      hitPolicy: 'COLLECT',
      aggregation: undefined
    },
    explanation: 'Collects the values of all matching rules'
  }, {
    label: 'Collect (Sum)',
    value: {
      hitPolicy: 'COLLECT',
      aggregation: 'SUM'
    },
    explanation: 'Collects the values of all matching rules and sums up to a single value'
  }, {
    label: 'Collect (Min)',
    value: {
      hitPolicy: 'COLLECT',
      aggregation: 'MIN'
    },
    explanation: 'Collects the values of all matching rules and uses the lowest value'
  }, {
    label: 'Collect (Max)',
    value: {
      hitPolicy: 'COLLECT',
      aggregation: 'MAX'
    },
    explanation: 'Collects the values of all matching rules and uses the highest value'
  }, {
    label: 'Collect (Count)',
    value: {
      hitPolicy: 'COLLECT',
      aggregation: 'COUNT'
    },
    explanation: 'Collects the values of all matching rules and counts the number of them'
  }, {
    label: 'Rule order',
    value: {
      hitPolicy: 'RULE ORDER',
      aggregation: undefined
    },
    explanation: 'Collects the values of all matching rules in rule order'
  }, {
    label: 'Output order',
    value: {
      hitPolicy: 'OUTPUT ORDER',
      aggregation: undefined
    },
    explanation: 'Collects the values of all matching rules in decreasing output priority order'
  }];

  class HitPolicy extends Component {
    constructor(props, context) {
      super(props, context);
      this._translate = context.injector.get('translate');
      inject(this);
    }
    getRoot() {
      return this.sheet.getRoot();
    }
    render() {
      const root = this.getRoot(),
        businessObject = root.businessObject;
      const {
        aggregation,
        hitPolicy
      } = businessObject;
      const hitPolicyEntry = find$2(HIT_POLICIES, entry => {
        return isEqualHitPolicy(entry.value, {
          aggregation,
          hitPolicy
        });
      });
      return createVNode(1, "div", "hit-policy header", [createVNode(1, "label", "dms-label", this._translate('Hit policy:'), 0), createVNode(1, "span", "hit-policy-value", this._translate(hitPolicyEntry.label), 0)], 4, {
        "title": this._translate(hitPolicyEntry.explanation)
      });
    }
  }
  HitPolicy.$inject = ['sheet'];

  // helpers //////////////////////
  function isEqualHitPolicy(a, b) {
    return a.hitPolicy === b.hitPolicy && a.aggregation === b.aggregation;
  }

  function HitPolicyProvider(components) {
    components.onGetComponent('hit-policy', () => {
      return HitPolicy;
    });
  }
  HitPolicyProvider.$inject = ['components'];

  var hitPolicyModule = {
    __init__: ['hitPolicyProvider'],
    hitPolicyProvider: ['type', HitPolicyProvider]
  };

  let ViewDrdComponent$2 = class ViewDrdComponent extends Component {
    constructor(props, context) {
      super(props, context);
      const {
        injector
      } = context;
      this._translate = injector.get('translate');
      this._eventBus = injector.get('eventBus');
    }
    onClick = () => {
      this._eventBus.fire('showDrd');
    };
    render() {
      return createVNode(1, "div", "view-drd", createVNode(1, "button", "view-drd-button", this._translate('View DRD'), 0, {
        "type": "button",
        "onClick": this.onClick
      }), 2, null, null, node => this.node = node);
    }
  };

  let ViewDrd$2 = class ViewDrd {
    constructor(components, eventBus, injector, sheet) {
      this._injector = injector;
      this._sheet = sheet;
      components.onGetComponent('table.before', () => {
        if (this.canViewDrd()) {
          return ViewDrdComponent$2;
        }
      });
      eventBus.on('showDrd', () => {
        const parent = injector.get('_parent', false);
        const root = sheet.getRoot();
        const definitions = getDefinitions$1(root);
        if (!definitions) {
          return;
        }

        // open definitions
        const view = parent.getView(definitions);
        parent.open(view);
      });
    }
    canViewDrd() {
      const parent = this._injector.get('_parent', false);
      if (!parent) {
        return false;
      }
      const root = this._sheet.getRoot();
      const definitions = getDefinitions$1(root);
      return !!parent.getView(definitions);
    }
  };
  ViewDrd$2.$inject = ['components', 'eventBus', 'injector', 'sheet'];

  // helpers //////////////////////

  function getDefinitions$1(root) {
    const {
      businessObject
    } = root;

    // root might not have business object
    if (!businessObject) {
      return;
    }
    const decision = businessObject.$parent;
    const definitions = decision.$parent;
    return definitions;
  }

  var viewDrdModule = {
    __init__: ['viewDrd'],
    viewDrd: ['type', ViewDrd$2]
  };

  let PoweredBy$2 = class PoweredBy {
    constructor(components) {
      components.onGetComponent('table.before', () => {
        return PoweredByComponent;
      });
    }
  };
  PoweredBy$2.$inject = ['components'];

  var PoweredByModule$2 = {
    __init__: ['poweredBy'],
    poweredBy: ['type', PoweredBy$2]
  };

  /**
   * @typedef {import('dmn-js-shared/lib/base/View).OpenResult} OpenResult
   */

  /**
   * @typedef {import('dmn-js-shared/lib/base/View).OpenError} OpenError
   */

  let Viewer$4 = class Viewer extends Table {
    constructor(options = {}) {
      const container = Viewer._createContainer();
      super(assign$4(options, {
        renderer: {
          container
        }
      }));
      this._container = container;
    }

    /**
     * Open diagram element.
     *
     * @param  {ModdleElement} decision
     * @returns {Promise} Resolves with {OpenResult} when successful
     * or rejects with {OpenError}
     */
    open(decision) {
      var self = this;
      return new Promise((resolve, reject) => {
        var err;

        // use try/catch to not swallow synchronous exceptions
        // that may be raised during model parsing
        try {
          if (self._decision) {
            // clear existing rendered diagram
            self.clear();
          }

          // update decision
          self._decision = decision;

          // perform import
          return importDecision(self, decision, function (err, warnings) {
            if (err) {
              err.warnings = warnings || [];
              reject(err);
            } else {
              resolve({
                warnings: warnings || []
              });
            }
          });
        } catch (e) {
          err = e;
        }

        // handle synchronously thrown exception
        if (err) {
          err.warnings = err.warnings || [];
          reject(err);
        } else {
          resolve({
            warnings: []
          });
        }
      });
    }

    /**
     * Initialize the table, returning { modules: [], config }.
     *
     * @param  {Object} options
     *
     * @return {Object} init config
     */
    _init(options) {
      let {
        modules,
        additionalModules,
        ...config
      } = options;
      let baseModules = modules || this.getModules();
      let extraModules = additionalModules || [];
      let staticModules = [{
        decisionTable: ['value', this]
      }];
      let allModules = [PoweredByModule$2, ...baseModules, ...extraModules, ...staticModules];
      return {
        modules: allModules,
        config
      };
    }

    /**
     * Register an event listener
     *
     * Remove a previously added listener via {@link #off(event, callback)}.
     *
     * @param {string} event
     * @param {number} [priority]
     * @param {Function} callback
     * @param {Object} [that]
     */
    on(event, priority, callback, target) {
      return this.get('eventBus').on(event, priority, callback, target);
    }

    /**
     * De-register an event listener
     *
     * @param {string} event
     * @param {Function} callback
     */
    off(event, callback) {
      this.get('eventBus').off(event, callback);
    }

    /**
     * Emit an event on the underlying {@link EventBus}
     *
     * @param  {string} type
     * @param  {Object} event
     *
     * @return {Object} event processing result (if any)
     */
    _emit(type, event) {
      return this.get('eventBus').fire(type, event);
    }

    /**
     * Attach viewer to given parent node.
     *
     * @param  {Element} parentNode
     */
    attachTo(parentNode) {
      if (!parentNode) {
        throw new Error('parentNode required');
      }

      // ensure we detach from the
      // previous, old parent
      this.detach();
      const container = this._container;
      parentNode.appendChild(container);
      this._emit('attach', {});
    }

    /**
     * Detach viewer from parent node, if attached.
     */
    detach() {
      const container = this._container,
        parentNode = container.parentNode;
      if (!parentNode) {
        return;
      }
      this._emit('detach', {});
      remove$4(container);
    }
    destroy() {
      super.destroy();
      this.detach();
    }
    getModules() {
      return Viewer._getModules();
    }
    static _getModules() {
      return [annotationsModule, coreModule, TranslateModule, decisionTableHeadModule, decisionTablePropertiesModule, decisionRuleIndicesModule, decisionRulesModule, hitPolicyModule, viewDrdModule];
    }
    static _createContainer() {
      return domify$1('<div class="dmn-decision-table-container"></div>');
    }
  };

  class ChangeSupport {
    constructor(eventBus) {
      this._listeners = {};
      eventBus.on('elements.changed', ({
        elements
      }) => {
        this.elementsChanged(elements);
      });
      eventBus.on('element.updateId', ({
        element,
        newId
      }) => {
        this.updateId(element.id, newId);
      });
    }
    elementsChanged(elements) {
      const invoked = {};
      const elementsLength = elements.length;
      for (let i = 0; i < elementsLength; i++) {
        const {
          id
        } = elements[i];
        if (invoked[id]) {
          return;
        }
        invoked[id] = true;
        const listenersLength = this._listeners[id] && this._listeners[id].length;
        if (listenersLength) {
          for (let j = 0; j < listenersLength; j++) {
            // listeners might remove themselves before they get called
            this._listeners[id][j] && this._listeners[id][j]();
          }
        }
      }
    }
    onElementsChanged(id, listener) {
      if (!this._listeners[id]) {
        this._listeners[id] = [];
      }

      // avoid push for better performance
      this._listeners[id][this._listeners[id].length] = listener;
    }
    offElementsChanged(id, listener) {
      if (!this._listeners[id]) {
        return;
      }
      if (listener) {
        const idx = this._listeners[id].indexOf(listener);
        if (idx !== -1) {
          this._listeners[id].splice(idx, 1);
        }
      } else {
        this._listeners[id].length = 0;
      }
    }
    updateId(oldId, newId) {
      if (this._listeners[oldId]) {
        this._listeners[newId] = this._listeners[oldId];
        delete this._listeners[oldId];
      }
    }
  }
  ChangeSupport.$inject = ['eventBus'];

  const DEFAULT_PRIORITY = 1000;
  class Components {
    constructor() {
      this._listeners = {};
    }
    getComponent(type, context) {
      const listeners = this._listeners[type];
      if (!listeners) {
        return;
      }
      let component;
      for (let i = 0; i < listeners.length; i++) {
        component = listeners[i].callback(context);
        if (component) {
          break;
        }
      }
      return component;
    }
    getComponents(type, context) {
      const listeners = this._listeners[type];
      if (!listeners) {
        return;
      }
      const components = [];
      for (let i = 0; i < listeners.length; i++) {
        const component = listeners[i].callback(context);
        if (component) {
          components.push(component);
        }
      }
      if (!components.length) {
        return;
      }
      return components;
    }
    onGetComponent(type, priority, callback) {
      if (isFunction$2(priority)) {
        callback = priority;
        priority = DEFAULT_PRIORITY;
      }
      if (!isNumber$1(priority)) {
        throw new Error('priority must be a number');
      }
      const listeners = this._getListeners(type);
      let existingListener, idx;
      const newListener = {
        priority,
        callback
      };
      for (idx = 0; existingListener = listeners[idx]; idx++) {
        if (existingListener.priority < priority) {
          // prepend newListener at before existingListener
          listeners.splice(idx, 0, newListener);
          return;
        }
      }
      listeners.push(newListener);
    }
    offGetComponent(type, callback) {
      const listeners = this._getListeners(type);
      let listener, listenerCallback, idx;
      if (callback) {
        // move through listeners from back to front
        // and remove matching listeners
        for (idx = listeners.length - 1; listener = listeners[idx]; idx--) {
          listenerCallback = listener.callback;
          if (listenerCallback === callback) {
            listeners.splice(idx, 1);
          }
        }
      } else {
        // clear listeners
        listeners.length = 0;
      }
    }
    _getListeners(type) {
      let listeners = this._listeners[type];
      if (!listeners) {
        this._listeners[type] = listeners = [];
      }
      return listeners;
    }
  }

  class ViewerComponent extends Component {
    constructor(props) {
      super(props);
      const injector = this._injector = props.injector;
      this._changeSupport = injector.get('changeSupport');
      this._components = injector.get('components');
      this._renderer = injector.get('renderer');
    }
    getChildContext() {
      return {
        changeSupport: this._changeSupport,
        components: this._components,
        renderer: this._renderer,
        injector: this._injector
      };
    }
    render() {
      const components = this._components.getComponents('viewer');
      return createVNode(1, "div", "viewer-container", components && components.map((Component, index) => createComponentVNode(2, Component, null, index)), 0);
    }
  }

  class Renderer {
    constructor(changeSupport, components, config, eventBus, injector) {
      const {
        container
      } = config;
      this._container = container;
      eventBus.on('renderer.mount', () => {
        render(createComponentVNode(2, ViewerComponent, {
          "injector": injector
        }), container);
      });
      eventBus.on('renderer.unmount', () => {
        render(null, container);
      });
    }
    getContainer() {
      return this._container;
    }
  }
  Renderer.$inject = ['changeSupport', 'components', 'config.renderer', 'eventBus', 'injector'];

  var core = {
    __init__: ['changeSupport', 'components', 'renderer'],
    changeSupport: ['type', ChangeSupport],
    components: ['type', Components],
    eventBus: ['type', EventBus],
    renderer: ['type', Renderer]
  };

  /**
   * A base for React-style viewers.
   */
  let Viewer$3 = class Viewer {
    constructor(options = {}) {
      let {
        injector
      } = options;
      if (!injector) {
        let {
          modules,
          config
        } = this._init(options);
        injector = createInjector(config, modules);
      }
      this.get = injector.get;
      this.invoke = injector.invoke;
      this.get('eventBus').fire('viewer.init');
    }

    /**
     * Intialize and return modules and config used for creation.
     *
     * @param  {Object} options
     *
     * @return {Object} { modules=[], config }
     */
    _init(options) {
      let {
        modules,
        ...config
      } = options;
      return {
        modules,
        config
      };
    }

    /**
     * Destroy. This results in removing the attachment from the container.
     */
    destroy() {
      const eventBus = this.get('eventBus');
      eventBus.fire('viewer.destroy');
    }

    /**
     * Clear. Should be used to reset the state of any stateful services.
     */
    clear() {
      const eventBus = this.get('eventBus');
      eventBus.fire('viewer.clear');
    }
  };

  // helpers //////////////////////

  function createInjector(config, modules) {
    const bootstrapModules = [{
      config: ['value', config]
    }, core].concat(modules || []);
    const injector = new Injector(bootstrapModules);
    injector.init();
    return injector;
  }

  /**
   * A single decision element registry.
   *
   * The sole purpose of this service is to provide the necessary API
   * to serve shared components, i.e. the UpdatePropertiesHandler.
   */
  class ElementRegistry {
    constructor(viewer, eventBus) {
      this._eventBus = eventBus;
      this._viewer = viewer;
    }
    getDecision() {
      return this._viewer.getDecision();
    }
    updateId(element, newId) {
      var decision = this.getDecision();
      if (element !== decision) {
        throw new Error('element !== decision');
      }
      this._eventBus.fire('element.updateId', {
        element: element,
        newId: newId
      });
      element.id = newId;
    }
  }
  ElementRegistry.$inject = ['viewer', 'eventBus'];

  var CoreModule = {
    __init__: ['elementRegistry'],
    elementRegistry: ['type', ElementRegistry]
  };

  class DecisionPropertiesComponent extends Component {
    constructor(props, context) {
      super(props, context);
      this._viewer = context.injector.get('viewer');
    }
    render() {
      // there is only one single element
      const {
        name
      } = this._viewer.getDecision();
      return createVNode(1, "div", "decision-properties", createVNode(1, "h3", "decision-name", name, 0), 2);
    }
  }

  const HIGH_PRIORITY = 1500;
  let DecisionProperties$1 = class DecisionProperties {
    constructor(components) {
      components.onGetComponent('viewer', HIGH_PRIORITY, () => DecisionPropertiesComponent);
    }
  };
  DecisionProperties$1.$inject = ['components'];

  var DecisionPropertiesModule = {
    __init__: ['decisionProperties'],
    decisionProperties: ['type', DecisionProperties$1]
  };

  class LiteralExpressionPropertiesComponent extends Component {
    constructor(props, context) {
      super(props, context);
      this._translate = context.injector.get('translate');
      this._viewer = context.injector.get('viewer');
    }
    render() {
      const {
        decisionLogic: literalExpression,
        variable
      } = this._viewer.getDecision();
      return createVNode(1, "div", "literal-expression-properties", createVNode(1, "table", null, [createVNode(1, "tr", null, [createVNode(1, "td", null, this._translate('Variable name:'), 0), createVNode(1, "td", null, createVNode(1, "span", null, variable.name || '-', 0), 2)], 4), createVNode(1, "tr", null, [createVNode(1, "td", null, this._translate('Variable type:'), 0), createVNode(1, "td", null, createVNode(1, "span", null, this._translate(variable.typeRef || '') || '-', 0), 2)], 4), createVNode(1, "tr", null, [createVNode(1, "td", null, this._translate('Expression language:'), 0), createVNode(1, "td", null, createVNode(1, "span", null, literalExpression.expressionLanguage || '-', 0), 2)], 4)], 4), 2);
    }
  }

  const LOW_PRIORITY = 500;
  class DecisionProperties {
    constructor(components) {
      components.onGetComponent('viewer', LOW_PRIORITY, () => {
        return LiteralExpressionPropertiesComponent;
      });
    }
  }
  DecisionProperties.$inject = ['components'];

  var LiteralExpressionPropertiesModule = {
    __depends__: [],
    __init__: ['literalExpressionProperties'],
    literalExpressionProperties: ['type', DecisionProperties]
  };

  const HIGHER_PRIORITY$1 = 2000;
  let PoweredBy$1 = class PoweredBy {
    constructor(components) {
      components.onGetComponent('viewer', HIGHER_PRIORITY$1, () => {
        return PoweredByComponent;
      });
    }
  };
  PoweredBy$1.$inject = ['components'];

  var PoweredByModule$1 = {
    __init__: ['poweredBy'],
    poweredBy: ['type', PoweredBy$1]
  };

  class TextareaComponent extends Component {
    constructor(props, context) {
      super(props, context);
      this._viewer = context.injector.get('viewer');
    }
    render() {
      const {
        text
      } = this._viewer.getDecision().decisionLogic;
      return createVNode(1, "div", "textarea", createVNode(1, "div", "content", text, 0), 2);
    }
  }

  class Textarea {
    constructor(components) {
      components.onGetComponent('viewer', () => TextareaComponent);
    }
  }
  Textarea.$inject = ['components'];

  var TextareaModule = {
    __init__: ['textarea'],
    textarea: ['type', Textarea]
  };

  let ViewDrdComponent$1 = class ViewDrdComponent extends Component {
    constructor(props, context) {
      super(props, context);
      const {
        injector
      } = context;
      this._translate = injector.get('translate');
      this._eventBus = injector.get('eventBus');
    }
    onClick = () => {
      this._eventBus.fire('showDrd');
    };
    render() {
      return createVNode(1, "div", "view-drd", createVNode(1, "button", "view-drd-button", this._translate('View DRD'), 0, {
        "type": "button",
        "onClick": this.onClick
      }), 2, null, null, node => this.node = node);
    }
  };
  ViewDrdComponent$1.$inject = ['translate'];

  const VERY_HIGH_PRIORITY$1 = 2000;
  let ViewDrd$1 = class ViewDrd {
    constructor(components, viewer, eventBus, injector) {
      this._injector = injector;
      this._viewer = viewer;
      components.onGetComponent('viewer', VERY_HIGH_PRIORITY$1, () => {
        if (this.canViewDrd()) {
          return ViewDrdComponent$1;
        }
      });
      eventBus.on('showDrd', () => {
        const parent = injector.get('_parent', false);

        // there is only one single element
        const definitions = this.getDefinitions();

        // open definitions
        const view = parent.getView(definitions);
        parent.open(view);
      });
    }
    canViewDrd() {
      const parent = this._injector.get('_parent', false);
      if (!parent) {
        return;
      }

      // there is only one single element
      const definitions = this.getDefinitions();
      return !!parent.getView(definitions);
    }
    getDefinitions() {
      return getDefinitions(this._viewer.getDecision());
    }
  };
  ViewDrd$1.$inject = ['components', 'viewer', 'eventBus', 'injector'];

  // helpers //////////////////////

  function getDefinitions(decision) {
    const definitions = decision.$parent;
    return definitions;
  }

  var ViewDrdModule$1 = {
    __init__: ['viewDrd'],
    viewDrd: ['type', ViewDrd$1]
  };

  /**
   * @typedef {import('dmn-js-shared/lib/base/View).OpenResult} OpenResult
   */

  /**
   * @typedef {import('dmn-js-shared/lib/base/View).OpenError} OpenError
   */

  let Viewer$2 = class Viewer extends Viewer$3 {
    constructor(options = {}) {
      const container = Viewer._createContainer();
      super(assign$4(options, {
        renderer: {
          container
        }
      }));
      this._container = container;
    }

    /**
     * Open diagram element.
     *
     * @param  {ModdleElement} decision
     * @returns {Promise} Resolves with {OpenResult} when successful
     * or rejects with {OpenError}
     */
    open(decision) {
      var self = this;
      return new Promise(function (resolve, reject) {
        var err;

        // use try/catch to not swallow synchronous exceptions
        // that may be raised during model parsing
        try {
          if (self._decision) {
            // clear existing literal expression
            self.clear();

            // unmount first
            self.get('eventBus').fire('renderer.unmount');
          }

          // update literal expression
          self._decision = decision;

          // let others know about import
          self.get('eventBus').fire('import', decision);
          self.get('eventBus').fire('renderer.mount');
        } catch (e) {
          err = e;
        }

        // handle synchronously thrown exception
        if (err) {
          err.warnings = err.warnings || [];
          reject(err);
        } else {
          resolve({
            warnings: []
          });
        }
      });
    }

    /**
     * Initialize the literal expression, returning { modules: [], config }.
     *
     * @param  {Object} options
     *
     * @return {Object} init config
     */
    _init(options) {
      let {
        modules,
        additionalModules,
        ...config
      } = options;
      let baseModules = modules || this.getModules();
      let extraModules = additionalModules || [];
      let staticModules = [{
        viewer: ['value', this]
      }];
      let allModules = [...baseModules, ...extraModules, ...staticModules];
      return {
        modules: allModules,
        config
      };
    }

    /**
     * Register an event listener
     *
     * Remove a previously added listener via {@link #off(event, callback)}.
     *
     * @param {string} event
     * @param {number} [priority]
     * @param {Function} callback
     * @param {Object} [that]
     */
    on(event, priority, callback, target) {
      return this.get('eventBus').on(event, priority, callback, target);
    }

    /**
     * De-register an event listener
     *
     * @param {string} event
     * @param {Function} callback
     */
    off(event, callback) {
      this.get('eventBus').off(event, callback);
    }

    /**
     * Emit an event on the underlying {@link EventBus}
     *
     * @param  {string} type
     * @param  {Object} event
     *
     * @return {Object} event processing result (if any)
     */
    _emit(type, event) {
      return this.get('eventBus').fire(type, event);
    }

    /**
     * Returns the currently displayed decision.
     *
     * @return {ModdleElement}
     */
    getDecision() {
      return this._decision;
    }

    /**
     * Attach viewer to given parent node.
     *
     * @param  {Element} parentNode
     */
    attachTo(parentNode) {
      if (!parentNode) {
        throw new Error('parentNode required');
      }

      // ensure we detach from the
      // previous, old parent
      this.detach();
      parentNode.appendChild(this._container);
      this._emit('attach', {});
    }

    /**
     * Detach viewer from parent node, if attached.
     */
    detach() {
      const container = this._container,
        parentNode = container.parentNode;
      if (!parentNode) {
        return;
      }
      this._emit('detach', {});
      remove$4(container);
    }
    destroy() {
      super.destroy();
      this.detach();
    }
    getModules() {
      return Viewer._getModules();
    }
    static _getModules() {
      return [CoreModule, TranslateModule, DecisionPropertiesModule, LiteralExpressionPropertiesModule, PoweredByModule$1, TextareaModule, ViewDrdModule$1];
    }
    static _createContainer() {
      return domify$1('<div class="dmn-literal-expression-container"></div>');
    }
  };

  class ViewRenderer {
    static $inject = ['components'];
    constructor(components) {
      components.onGetComponent('viewer', () => {
        return Header;
      });
      components.onGetComponent('viewer', () => {
        return Body;
      });
      components.onGetComponent('viewer', () => {
        return Footer;
      });
    }
  }
  function Header(_, {
    injector
  }) {
    const components = injector.get('components');
    const headerComponents = components.getComponents('header');
    return createVNode(1, "div", "dmn-boxed-expression-section dmn-boxed-expression-header", headerComponents && headerComponents.map((Component, index) => {
      return createComponentVNode(2, Component, null, index);
    }), 0);
  }
  function Body(_, {
    injector
  }) {
    const components = injector.get('components');
    const bodyComponents = components.getComponents('body');
    return createVNode(1, "div", "dmn-boxed-expression-section dmn-boxed-expression-body", bodyComponents && bodyComponents.map((Component, index) => {
      return createComponentVNode(2, Component, null, index);
    }), 0);
  }
  function Footer(_, {
    injector
  }) {
    const components = injector.get('components');
    const footerComponents = components.getComponents('footer');
    return createVNode(1, "div", "dmn-boxed-expression-section dmn-boxed-expression-footer", footerComponents && footerComponents.map((Component, index) => {
      return createComponentVNode(2, Component, null, index);
    }), 0);
  }

  /**
   * Allows to use modules from `table-js` which depend on `table.*` components.
   * @TODO(barmac): This is a temporary solution until we move context menu out of table-js.
   */
  class TableJsSupport {
    static $inject = ['components'];
    constructor(components) {
      components.onGetComponent('viewer', () => {
        const children = components.getComponents('table.before') || [];
        return () => {
          return createVNode(1, "div", null, children.map((Component, index) => createComponentVNode(2, Component, null, index)), 0);
        };
      });
    }
  }

  var RenderModule = {
    __init__: ['viewRenderer', 'tableJsSupport'],
    'viewRenderer': ['type', ViewRenderer],
    'tableJsSupport': ['type', TableJsSupport]
  };

  const HIGHER_PRIORITY = 2000;
  class PoweredBy {
    constructor(components) {
      components.onGetComponent('viewer', HIGHER_PRIORITY, () => {
        return PoweredByComponent;
      });
    }
  }
  PoweredBy.$inject = ['components'];

  var PoweredByModule = {
    __init__: ['poweredBy'],
    poweredBy: ['type', PoweredBy]
  };

  class LiteralExpressionComponentProvider {
    static $inject = ['components'];
    constructor(components) {
      components.onGetComponent('expression', ({
        expression
      }) => {
        if (is(expression, 'dmn:LiteralExpression')) {
          return LiteralExpressionComponent;
        }
      });
    }
  }
  function LiteralExpressionComponent({
    expression
  }, context) {
    const literalExpression = context.injector.get('literalExpression');
    const text = literalExpression.getText(expression);
    return createVNode(1, "div", "textarea", createVNode(1, "div", "content", text, 0), 2);
  }

  class LiteralExpression {
    getText(literalExpression) {
      return literalExpression.get('text');
    }
  }

  var LiteralExpressionModule = {
    __init__: ['literalExpressionComponent'],
    literalExpressionComponent: ['type', LiteralExpressionComponentProvider],
    literalExpression: ['type', LiteralExpression]
  };

  class FunctionDefinition {
    getParameters(element) {
      return element.get('formalParameter');
    }
    getBody(element) {
      return element.get('body');
    }
    getKind(element) {
      return element.get('kind') || 'FEEL';
    }
  }

  class FunctionDefinitionComponentProvider {
    static $inject = ['components'];
    constructor(components) {
      components.onGetComponent('expression', ({
        expression
      }) => {
        if (is(expression, 'dmn:FunctionDefinition')) {
          return FunctionDefinitionComponent;
        }
      });
    }
  }
  function FunctionDefinitionComponent({
    expression
  }, context) {
    const functionDefinition = context.injector.get('functionDefinition');
    const kind = functionDefinition.getKind(expression);
    const parameters = functionDefinition.getParameters(expression);
    const body = functionDefinition.getBody(expression);
    return createVNode(1, "div", "function-definition", [createComponentVNode(2, Kind, {
      "kind": kind
    }), createComponentVNode(2, FormalParameters, {
      "parameters": parameters
    }), createComponentVNode(2, BodyExpression, {
      "expression": body
    })], 4);
  }
  const KIND_MAP = {
    'FEEL': 'F',
    'Java': 'J',
    'PMML': 'P'
  };
  function Kind({
    kind
  }, context) {
    const translate = context.injector.get('translate');
    return createVNode(1, "div", "function-definition-kind", KIND_MAP[kind], 0, {
      "title": translate('Function kind: {kind}', {
        kind
      })
    });
  }
  function FormalParameters({
    parameters
  }) {
    return createVNode(1, "div", "function-definition-parameters", createVNode(1, "div", null, [createTextVNode("("), parameters.reduce((acc, parameter) => {
      return acc.concat(createComponentVNode(2, Parameter, {
        "parameter": parameter
      }), ', ');
    }, []).slice(0, -1), createTextVNode(")")], 0), 2);
  }
  function Parameter({
    parameter
  }) {
    const {
      name,
      typeRef
    } = parameter;
    const displayedName = name || '<unnamed>';
    return createVNode(1, "span", null, typeRef ? `${displayedName}: ${typeRef}` : displayedName, 0);
  }
  function BodyExpression({
    expression
  }, context) {
    const Expression = context.components.getComponent('expression', {
      expression
    });
    return createVNode(1, "div", "function-definition-body", createComponentVNode(2, Expression, {
      "expression": expression
    }), 2);
  }

  var FunctionDefinitionEditorModule = {
    __init__: ['functionDefinitionComponent'],
    functionDefinition: ['type', FunctionDefinition],
    functionDefinitionComponent: ['type', FunctionDefinitionComponentProvider]
  };

  class ViewDrdComponent extends Component {
    constructor(props, context) {
      super(props, context);
      const {
        injector
      } = context;
      this._translate = injector.get('translate');
      this._eventBus = injector.get('eventBus');
    }
    onClick = () => {
      this._eventBus.fire('showDrd');
    };
    render() {
      return createVNode(1, "div", "view-drd", createVNode(1, "button", "view-drd-button", this._translate('View DRD'), 0, {
        "type": "button",
        "onClick": this.onClick
      }), 2, null, null, node => this.node = node);
    }
  }
  ViewDrdComponent.$inject = ['translate'];

  const VERY_HIGH_PRIORITY = 2000;
  class ViewDrd {
    constructor(components, eventBus, injector) {
      this._injector = injector;
      components.onGetComponent('viewer', VERY_HIGH_PRIORITY, () => {
        if (this.canViewDrd()) {
          return ViewDrdComponent;
        }
      });
      eventBus.on('showDrd', () => {
        const parent = injector.get('_parent', false);

        // there is only one single element
        const definitions = parent.getDefinitions();

        // open definitions
        const view = parent.getView(definitions);
        parent.open(view);
      });
    }
    canViewDrd() {
      const parent = this._injector.get('_parent', false);
      if (!parent) {
        return;
      }

      // there is only one single element
      const definitions = parent.getDefinitions();
      return !!parent.getView(definitions);
    }
  }
  ViewDrd.$inject = ['components', 'eventBus', 'injector'];

  var ViewDrdModule = {
    __init__: ['viewDrd'],
    viewDrd: ['type', ViewDrd]
  };

  function ElementPropertiesComponent(_, context) {
    const viewer = context.injector.get('viewer');

    // there is only one single element
    const {
      name
    } = viewer.getRootElement();
    return createVNode(1, "div", "element-properties", createVNode(1, "h2", "element-name", name, 0), 2);
  }

  class ElementProperties {
    constructor(components) {
      components.onGetComponent('header', () => ElementPropertiesComponent);
    }
  }
  ElementProperties.$inject = ['components'];

  var ElementPropertiesModule = {
    __init__: ['elementProperties'],
    elementProperties: ['type', ElementProperties]
  };

  const FALLBACK_PRIORITY = 100;
  class ElementLogic {
    static $inject = ['components'];
    constructor(components) {
      components.onGetComponent('body', () => {
        return LogicComponent;
      });
      components.onGetComponent('expression', FALLBACK_PRIORITY, () => FallbackExpression);
    }
  }
  function LogicComponent(_, {
    injector
  }) {
    const components = injector.get('components');
    const viewer = injector.get('viewer');
    const rootElement = viewer.getRootElement();
    const expression = getLogic(rootElement);
    const Expression = components.getComponent('expression', {
      expression
    });
    return createComponentVNode(2, Expression, {
      "expression": expression
    });
  }
  function getLogic(element) {
    if (is(element, 'dmn:Decision')) {
      return element.get('decisionLogic');
    } else if (is(element, 'dmn:BusinessKnowledgeModel')) {
      return element.get('encapsulatedLogic');
    }
  }
  function FallbackExpression({
    expression
  }) {
    return createVNode(1, "div", null, createVNode(1, "span", null, [createTextVNode("Expression of type "), expression.$type, createTextVNode(" is not supported.")], 0, {
      "style": "color:red;"
    }), 2);
  }

  var ElementLogicModule = {
    __init__: ['elementLogic'],
    elementLogic: ['type', ElementLogic]
  };

  class ElementVariable {
    static $inject = ['viewer'];
    constructor(viewer) {
      this._viewer = viewer;
    }
    getName() {
      const variable = this.getVariable(),
        element = this._getElement();
      const variableName = variable ? variable.get('name') : null;
      return variableName || element.get('name');
    }
    getType() {
      const variable = this.getVariable();
      return variable ? variable.get('typeRef') : 'Any';
    }
    _getElement() {
      return this._viewer.getRootElement();
    }
    getVariable() {
      return this._getElement().get('variable');
    }
  }

  class ElementVariableComponentProvider {
    static $inject = ['components'];
    constructor(components) {
      components.onGetComponent('footer', () => ElementVariableComponent);
    }
  }
  function ElementVariableComponent(_, context) {
    const elementVariable = context.injector.get('elementVariable');
    const translate = context.injector.get('translate');

    // there is only one single element
    const name = elementVariable.getName();
    const type = elementVariable.getType();
    return createVNode(1, "div", "element-variable", [createVNode(1, "h2", null, createTextVNode("Result"), 2), createVNode(1, "div", "element-variable-name", [createVNode(1, "span", "element-variable-name-label", translate('Variable name'), 0), createVNode(1, "span", null, name, 0)], 4), createVNode(1, "div", "element-variable-type", [createVNode(1, "span", "element-variable-type-label", translate('Variable type'), 0), createVNode(1, "span", null, type, 0)], 4)], 4);
  }

  var ElementVariableModule = {
    __init__: ['elementVariableComponent'],
    elementVariable: ['type', ElementVariable],
    elementVariableComponent: ['type', ElementVariableComponentProvider]
  };

  /**
   * @typedef {import('dmn-js-shared/lib/base/View).OpenResult} OpenResult
   */

  /**
   * @typedef {import('dmn-js-shared/lib/base/View).OpenError} OpenError
   */

  let Viewer$1 = class Viewer extends Viewer$3 {
    constructor(options = {}) {
      const container = Viewer._createContainer();
      super(assign$4(options, {
        renderer: {
          container
        }
      }));
      this._container = container;
    }

    /**
     * Open diagram element.
     *
     * @param  {ModdleElement} element
     * @returns {Promise} Resolves with {OpenResult} when successful
     * or rejects with {OpenError}
     */
    open(element) {
      const eventBus = this.get('eventBus');
      return new Promise((resolve, reject) => {
        let err;

        // use try/catch to not swallow synchronous exceptions
        // that may be raised during model parsing
        try {
          const rootElement = this.getRootElement();
          if (rootElement) {
            // clear existing literal expression
            this.clear();

            // unmount first
            eventBus.fire('renderer.unmount');
          }

          // update literal expression
          this._setRootElement(element);

          // let others know about import
          eventBus.fire('import', element);
          eventBus.fire('renderer.mount');
        } catch (e) {
          err = e;
        }

        // handle synchronously thrown exception
        if (err) {
          err.warnings = err.warnings || [];
          reject(err);
        } else {
          resolve({
            warnings: []
          });
        }
      });
    }

    /**
     * Initialize the viewer, returning { modules: [], config }.
     *
     * @param  {Object} options
     *
     * @return {Object} init config
     */
    _init(options) {
      let {
        modules,
        additionalModules,
        ...config
      } = options;
      let baseModules = modules || this.getModules();
      let extraModules = additionalModules || [];
      let staticModules = [{
        viewer: ['value', this]
      }];
      let allModules = [...baseModules, ...extraModules, ...staticModules];
      return {
        modules: allModules,
        config
      };
    }

    /**
     * Register an event listener
     *
     * Remove a previously added listener via {@link #off(event, callback)}.
     *
     * @param {string} event
     * @param {number} [priority]
     * @param {Function} callback
     * @param {Object} [that]
     */
    on(event, priority, callback, target) {
      return this.get('eventBus').on(event, priority, callback, target);
    }

    /**
     * De-register an event listener
     *
     * @param {string} event
     * @param {Function} callback
     */
    off(event, callback) {
      this.get('eventBus').off(event, callback);
    }

    /**
     * Emit an event on the underlying {@link EventBus}
     *
     * @param  {string} type
     * @param  {Object} event
     *
     * @return {Object} event processing result (if any)
     */
    _emit(type, event) {
      return this.get('eventBus').fire(type, event);
    }

    /**
     * Returns the currently displayed element.
     *
     * @return {ModdleElement}
     */
    getRootElement() {
      return this._root;
    }
    _setRootElement(element) {
      this._root = element;
    }

    /**
     * Attach viewer to given parent node.
     *
     * @param  {Element} parentNode
     */
    attachTo(parentNode) {
      if (!parentNode) {
        throw new Error('parentNode required');
      }

      // ensure we detach from the
      // previous, old parent
      this.detach();
      parentNode.appendChild(this._container);
      this._emit('attach', {});
    }

    /**
     * Detach viewer from parent node, if attached.
     */
    detach() {
      const container = this._container,
        parentNode = container.parentNode;
      if (!parentNode) {
        return;
      }
      this._emit('detach', {});
      remove$4(container);
    }
    destroy() {
      super.destroy();
      this.detach();
    }
    getModules() {
      return [RenderModule, TranslateModule, PoweredByModule, ViewDrdModule, ElementPropertiesModule, ElementLogicModule, FunctionDefinitionEditorModule, LiteralExpressionModule, ElementVariableModule];
    }
    static _createContainer() {
      return domify$1('<div class="dmn-boxed-expression-container"></div>');
    }
  };

  const ID = 'id';

  /**
   * A generic handler that implements property editing.
   */
  class EditPropertiesHandler {
    constructor(elementRegistry, moddle) {
      this._elementRegistry = elementRegistry;
      this._moddle = moddle;
    }

    /**
     * <do>
     */
    execute(context) {
      const {
        element,
        properties
      } = context;
      const bo = getBusinessObject(element);
      const {
        changed,
        oldProperties
      } = this.updateProperties(bo, properties);
      context.oldProperties = oldProperties;
      return [...changed, element];
    }

    /**
     * <undo>
     */
    revert(context) {
      const {
        element,
        oldProperties
      } = context;
      var bo = getBusinessObject(element);
      var {
        changed
      } = this.updateProperties(bo, oldProperties);
      return [...changed, element];
    }

    /**
     * Update properties of the given business object
     * and return { changed, oldProperties }.
     */
    updateProperties(bo, newProps) {
      const ids = this._moddle.ids;

      // Reduce over all new properties and return
      //
      // {
      //  changed,
      //  oldProperties
      // }
      return reduce(newProps, (result, value, key) => {
        const propertyValue = bo.get(key);

        // handle nested update
        if (isContainer(value)) {
          if (!isContainer(propertyValue)) {
            throw new Error(`non-existing property <${key}>: cannot update values`);
          }
          let {
            changed,
            oldProperties
          } = this.updateProperties(propertyValue, value);
          return {
            changed: [...result.changed, ...changed, propertyValue],
            oldProperties: {
              ...result.oldProperties,
              [key]: oldProperties
            }
          };
        }

        // handle ID change
        if (key === ID && isIdChange(bo, value)) {
          ids.unclaim(bo[ID]);
          this._elementRegistry.updateId(bo, value);
          ids.claim(value, bo);
        }

        // handle plain update
        bo.set(key, value);
        return {
          changed: result.changed,
          oldProperties: {
            ...result.oldProperties,
            [key]: propertyValue
          }
        };
      }, {
        changed: [],
        oldProperties: {}
      });
    }
  }
  EditPropertiesHandler.$inject = ['elementRegistry', 'moddle'];

  // helpers //////////////////////

  function isIdChange(element, newId) {
    return element[ID] !== newId;
  }
  function isContainer(o) {
    return isDefined(o) && isObject$2(o);
  }

  /**
   * Expose `xor`
   */

  window.getSelection();

  var selectionUpdate = {exports: {}};

  var hasRequiredSelectionUpdate;
  function requireSelectionUpdate() {
    if (hasRequiredSelectionUpdate) return selectionUpdate.exports;
    hasRequiredSelectionUpdate = 1;

    /**
     * Calculate the selection update for the given
     * current and new input values.
     *
     * @param {Object} currentSelection as {start, end}
     * @param {String} currentValue
     * @param {String} newValue
     *
     * @return {Object} newSelection as {start, end}
     */
    function calculateUpdate(currentSelection, currentValue, newValue) {
      var currentCursor = currentSelection.start,
        newCursor = currentCursor,
        diff = newValue.length - currentValue.length;
      newValue.length - currentValue.length;
      var currentTail = currentValue.substring(currentCursor);

      // check if we can remove common ending from the equation
      // to be able to properly detect a selection change for
      // the following scenarios:
      //
      //  * (AAATTT|TF) => (AAAT|TF)
      //  * (AAAT|TF) =>  (AAATTT|TF)
      //
      if (newValue.lastIndexOf(currentTail) === newValue.length - currentTail.length) {
        currentValue = currentValue.substring(0, currentValue.length - currentTail.length);
        newValue = newValue.substring(0, newValue.length - currentTail.length);
      }

      // diff
      var diff = createDiff(currentValue, newValue);
      if (diff) {
        if (diff.type === 'remove') {
          newCursor = diff.newStart;
        } else {
          newCursor = diff.newEnd;
        }
      }
      return range(newCursor);
    }
    selectionUpdate.exports = calculateUpdate;
    function createDiff(currentValue, newValue) {
      var insert;
      var l_str,
        l_char,
        l_idx = 0,
        s_str,
        s_char,
        s_idx = 0;
      if (newValue.length > currentValue.length) {
        l_str = newValue;
        s_str = currentValue;
      } else {
        l_str = currentValue;
        s_str = newValue;
      }

      // assume there will be only one insert / remove and
      // detect that _first_ edit operation only
      while (l_idx < l_str.length) {
        l_char = l_str.charAt(l_idx);
        s_char = s_str.charAt(s_idx);

        // chars no not equal
        if (l_char !== s_char) {
          if (!insert) {
            insert = {
              l_start: l_idx,
              s_start: s_idx
            };
          }
          l_idx++;
        }

        // chars equal (again?)
        else {
          if (insert && !insert.complete) {
            insert.l_end = l_idx;
            insert.s_end = s_idx;
            insert.complete = true;
          }
          s_idx++;
          l_idx++;
        }
      }
      if (insert && !insert.complete) {
        insert.complete = true;
        insert.s_end = s_str.length;
        insert.l_end = l_str.length;
      }

      // no diff
      if (!insert) {
        return;
      }
      if (newValue.length > currentValue.length) {
        return {
          newStart: insert.l_start,
          newEnd: insert.l_end,
          type: 'add'
        };
      } else {
        return {
          newStart: insert.s_start,
          newEnd: insert.s_end,
          type: newValue.length < currentValue.length ? 'remove' : 'replace'
        };
      }
    }

    /**
     * Utility method for creating a new selection range {start, end} object.
     *
     * @param {Number} start
     * @param {Number} [end]
     *
     * @return {Object} selection range as {start, end}
     */
    function range(start, end) {
      return {
        start: start,
        end: end === undefined ? start : end
      };
    }
    selectionUpdate.exports.range = range;
    return selectionUpdate.exports;
  }

  requireSelectionUpdate();

  // These are filled with ranges (rangeFrom[i] up to but not including
  // rangeTo[i]) of code points that count as extending characters.
  let rangeFrom = [],
    rangeTo = [];
  (() => {
    // Compressed representation of the Grapheme_Cluster_Break=Extend
    // information from
    // http://www.unicode.org/Public/16.0.0/ucd/auxiliary/GraphemeBreakProperty.txt.
    // Each pair of elements represents a range, as an offet from the
    // previous range and a length. Numbers are in base-36, with the empty
    // string being a shorthand for 1.
    let numbers = "lc,34,7n,7,7b,19,,,,2,,2,,,20,b,1c,l,g,,2t,7,2,6,2,2,,4,z,,u,r,2j,b,1m,9,9,,o,4,,9,,3,,5,17,3,3b,f,,w,1j,,,,4,8,4,,3,7,a,2,t,,1m,,,,2,4,8,,9,,a,2,q,,2,2,1l,,4,2,4,2,2,3,3,,u,2,3,,b,2,1l,,4,5,,2,4,,k,2,m,6,,,1m,,,2,,4,8,,7,3,a,2,u,,1n,,,,c,,9,,14,,3,,1l,3,5,3,,4,7,2,b,2,t,,1m,,2,,2,,3,,5,2,7,2,b,2,s,2,1l,2,,,2,4,8,,9,,a,2,t,,20,,4,,2,3,,,8,,29,,2,7,c,8,2q,,2,9,b,6,22,2,r,,,,,,1j,e,,5,,2,5,b,,10,9,,2u,4,,6,,2,2,2,p,2,4,3,g,4,d,,2,2,6,,f,,jj,3,qa,3,t,3,t,2,u,2,1s,2,,7,8,,2,b,9,,19,3,3b,2,y,,3a,3,4,2,9,,6,3,63,2,2,,1m,,,7,,,,,2,8,6,a,2,,1c,h,1r,4,1c,7,,,5,,14,9,c,2,w,4,2,2,,3,1k,,,2,3,,,3,1m,8,2,2,48,3,,d,,7,4,,6,,3,2,5i,1m,,5,ek,,5f,x,2da,3,3x,,2o,w,fe,6,2x,2,n9w,4,,a,w,2,28,2,7k,,3,,4,,p,2,5,,47,2,q,i,d,,12,8,p,b,1a,3,1c,,2,4,2,2,13,,1v,6,2,2,2,2,c,,8,,1b,,1f,,,3,2,2,5,2,,,16,2,8,,6m,,2,,4,,fn4,,kh,g,g,g,a6,2,gt,,6a,,45,5,1ae,3,,2,5,4,14,3,4,,4l,2,fx,4,ar,2,49,b,4w,,1i,f,1k,3,1d,4,2,2,1x,3,10,5,,8,1q,,c,2,1g,9,a,4,2,,2n,3,2,,,2,6,,4g,,3,8,l,2,1l,2,,,,,m,,e,7,3,5,5f,8,2,3,,,n,,29,,2,6,,,2,,,2,,2,6j,,2,4,6,2,,2,r,2,2d,8,2,,,2,2y,,,,2,6,,,2t,3,2,4,,5,77,9,,2,6t,,a,2,,,4,,40,4,2,2,4,,w,a,14,6,2,4,8,,9,6,2,3,1a,d,,2,ba,7,,6,,,2a,m,2,7,,2,,2,3e,6,3,,,2,,7,,,20,2,3,,,,9n,2,f0b,5,1n,7,t4,,1r,4,29,,f5k,2,43q,,,3,4,5,8,8,2,7,u,4,44,3,1iz,1j,4,1e,8,,e,,m,5,,f,11s,7,,h,2,7,,2,,5,79,7,c5,4,15s,7,31,7,240,5,gx7k,2o,3k,6o".split(",").map(s => s ? parseInt(s, 36) : 1);
    for (let i = 0, n = 0; i < numbers.length; i++) (i % 2 ? rangeTo : rangeFrom).push(n = n + numbers[i]);
  })();
  function isExtendingChar(code) {
    if (code < 768) return false;
    for (let from = 0, to = rangeFrom.length;;) {
      let mid = from + to >> 1;
      if (code < rangeFrom[mid]) to = mid;else if (code >= rangeTo[mid]) from = mid + 1;else return true;
      if (from == to) return false;
    }
  }
  function isRegionalIndicator(code) {
    return code >= 0x1F1E6 && code <= 0x1F1FF;
  }
  const ZWJ = 0x200d;
  function findClusterBreak$1(str, pos, forward = true, includeExtending = true) {
    return (forward ? nextClusterBreak : prevClusterBreak)(str, pos, includeExtending);
  }
  function nextClusterBreak(str, pos, includeExtending) {
    if (pos == str.length) return pos;
    // If pos is in the middle of a surrogate pair, move to its start
    if (pos && surrogateLow(str.charCodeAt(pos)) && surrogateHigh(str.charCodeAt(pos - 1))) pos--;
    let prev = codePointAt(str, pos);
    pos += codePointSize(prev);
    while (pos < str.length) {
      let next = codePointAt(str, pos);
      if (prev == ZWJ || next == ZWJ || includeExtending && isExtendingChar(next)) {
        pos += codePointSize(next);
        prev = next;
      } else if (isRegionalIndicator(next)) {
        let countBefore = 0,
          i = pos - 2;
        while (i >= 0 && isRegionalIndicator(codePointAt(str, i))) {
          countBefore++;
          i -= 2;
        }
        if (countBefore % 2 == 0) break;else pos += 2;
      } else {
        break;
      }
    }
    return pos;
  }
  function prevClusterBreak(str, pos, includeExtending) {
    while (pos > 0) {
      let found = nextClusterBreak(str, pos - 2, includeExtending);
      if (found < pos) return found;
      pos--;
    }
    return 0;
  }
  function codePointAt(str, pos) {
    let code0 = str.charCodeAt(pos);
    if (!surrogateHigh(code0) || pos + 1 == str.length) return code0;
    let code1 = str.charCodeAt(pos + 1);
    if (!surrogateLow(code1)) return code0;
    return (code0 - 0xd800 << 10) + (code1 - 0xdc00) + 0x10000;
  }
  function surrogateLow(ch) {
    return ch >= 0xDC00 && ch < 0xE000;
  }
  function surrogateHigh(ch) {
    return ch >= 0xD800 && ch < 0xDC00;
  }
  function codePointSize(code) {
    return code < 0x10000 ? 1 : 2;
  }

  /**
  The data structure for documents. @nonabstract
  */
  class Text {
    /**
    Get the line description around the given position.
    */
    lineAt(pos) {
      if (pos < 0 || pos > this.length) throw new RangeError(`Invalid position ${pos} in document of length ${this.length}`);
      return this.lineInner(pos, false, 1, 0);
    }
    /**
    Get the description for the given (1-based) line number.
    */
    line(n) {
      if (n < 1 || n > this.lines) throw new RangeError(`Invalid line number ${n} in ${this.lines}-line document`);
      return this.lineInner(n, true, 1, 0);
    }
    /**
    Replace a range of the text with the given content.
    */
    replace(from, to, text) {
      [from, to] = clip(this, from, to);
      let parts = [];
      this.decompose(0, from, parts, 2 /* Open.To */);
      if (text.length) text.decompose(0, text.length, parts, 1 /* Open.From */ | 2 /* Open.To */);
      this.decompose(to, this.length, parts, 1 /* Open.From */);
      return TextNode.from(parts, this.length - (to - from) + text.length);
    }
    /**
    Append another document to this one.
    */
    append(other) {
      return this.replace(this.length, this.length, other);
    }
    /**
    Retrieve the text between the given points.
    */
    slice(from, to = this.length) {
      [from, to] = clip(this, from, to);
      let parts = [];
      this.decompose(from, to, parts, 0);
      return TextNode.from(parts, to - from);
    }
    /**
    Test whether this text is equal to another instance.
    */
    eq(other) {
      if (other == this) return true;
      if (other.length != this.length || other.lines != this.lines) return false;
      let start = this.scanIdentical(other, 1),
        end = this.length - this.scanIdentical(other, -1);
      let a = new RawTextCursor(this),
        b = new RawTextCursor(other);
      for (let skip = start, pos = start;;) {
        a.next(skip);
        b.next(skip);
        skip = 0;
        if (a.lineBreak != b.lineBreak || a.done != b.done || a.value != b.value) return false;
        pos += a.value.length;
        if (a.done || pos >= end) return true;
      }
    }
    /**
    Iterate over the text. When `dir` is `-1`, iteration happens
    from end to start. This will return lines and the breaks between
    them as separate strings.
    */
    iter(dir = 1) {
      return new RawTextCursor(this, dir);
    }
    /**
    Iterate over a range of the text. When `from` > `to`, the
    iterator will run in reverse.
    */
    iterRange(from, to = this.length) {
      return new PartialTextCursor(this, from, to);
    }
    /**
    Return a cursor that iterates over the given range of lines,
    _without_ returning the line breaks between, and yielding empty
    strings for empty lines.
    
    When `from` and `to` are given, they should be 1-based line numbers.
    */
    iterLines(from, to) {
      let inner;
      if (from == null) {
        inner = this.iter();
      } else {
        if (to == null) to = this.lines + 1;
        let start = this.line(from).from;
        inner = this.iterRange(start, Math.max(start, to == this.lines + 1 ? this.length : to <= 1 ? 0 : this.line(to - 1).to));
      }
      return new LineCursor(inner);
    }
    /**
    Return the document as a string, using newline characters to
    separate lines.
    */
    toString() {
      return this.sliceString(0);
    }
    /**
    Convert the document to an array of lines (which can be
    deserialized again via [`Text.of`](https://codemirror.net/6/docs/ref/#state.Text^of)).
    */
    toJSON() {
      let lines = [];
      this.flatten(lines);
      return lines;
    }
    /**
    @internal
    */
    constructor() {}
    /**
    Create a `Text` instance for the given array of lines.
    */
    static of(text) {
      if (text.length == 0) throw new RangeError("A document must have at least one line");
      if (text.length == 1 && !text[0]) return Text.empty;
      return text.length <= 32 /* Tree.Branch */ ? new TextLeaf(text) : TextNode.from(TextLeaf.split(text, []));
    }
  }
  // Leaves store an array of line strings. There are always line breaks
  // between these strings. Leaves are limited in size and have to be
  // contained in TextNode instances for bigger documents.
  class TextLeaf extends Text {
    constructor(text, length = textLength(text)) {
      super();
      this.text = text;
      this.length = length;
    }
    get lines() {
      return this.text.length;
    }
    get children() {
      return null;
    }
    lineInner(target, isLine, line, offset) {
      for (let i = 0;; i++) {
        let string = this.text[i],
          end = offset + string.length;
        if ((isLine ? line : end) >= target) return new Line(offset, end, line, string);
        offset = end + 1;
        line++;
      }
    }
    decompose(from, to, target, open) {
      let text = from <= 0 && to >= this.length ? this : new TextLeaf(sliceText(this.text, from, to), Math.min(to, this.length) - Math.max(0, from));
      if (open & 1 /* Open.From */) {
        let prev = target.pop();
        let joined = appendText(text.text, prev.text.slice(), 0, text.length);
        if (joined.length <= 32 /* Tree.Branch */) {
          target.push(new TextLeaf(joined, prev.length + text.length));
        } else {
          let mid = joined.length >> 1;
          target.push(new TextLeaf(joined.slice(0, mid)), new TextLeaf(joined.slice(mid)));
        }
      } else {
        target.push(text);
      }
    }
    replace(from, to, text) {
      if (!(text instanceof TextLeaf)) return super.replace(from, to, text);
      [from, to] = clip(this, from, to);
      let lines = appendText(this.text, appendText(text.text, sliceText(this.text, 0, from)), to);
      let newLen = this.length + text.length - (to - from);
      if (lines.length <= 32 /* Tree.Branch */) return new TextLeaf(lines, newLen);
      return TextNode.from(TextLeaf.split(lines, []), newLen);
    }
    sliceString(from, to = this.length, lineSep = "\n") {
      [from, to] = clip(this, from, to);
      let result = "";
      for (let pos = 0, i = 0; pos <= to && i < this.text.length; i++) {
        let line = this.text[i],
          end = pos + line.length;
        if (pos > from && i) result += lineSep;
        if (from < end && to > pos) result += line.slice(Math.max(0, from - pos), to - pos);
        pos = end + 1;
      }
      return result;
    }
    flatten(target) {
      for (let line of this.text) target.push(line);
    }
    scanIdentical() {
      return 0;
    }
    static split(text, target) {
      let part = [],
        len = -1;
      for (let line of text) {
        part.push(line);
        len += line.length + 1;
        if (part.length == 32 /* Tree.Branch */) {
          target.push(new TextLeaf(part, len));
          part = [];
          len = -1;
        }
      }
      if (len > -1) target.push(new TextLeaf(part, len));
      return target;
    }
  }
  // Nodes provide the tree structure of the `Text` type. They store a
  // number of other nodes or leaves, taking care to balance themselves
  // on changes. There are implied line breaks _between_ the children of
  // a node (but not before the first or after the last child).
  class TextNode extends Text {
    constructor(children, length) {
      super();
      this.children = children;
      this.length = length;
      this.lines = 0;
      for (let child of children) this.lines += child.lines;
    }
    lineInner(target, isLine, line, offset) {
      for (let i = 0;; i++) {
        let child = this.children[i],
          end = offset + child.length,
          endLine = line + child.lines - 1;
        if ((isLine ? endLine : end) >= target) return child.lineInner(target, isLine, line, offset);
        offset = end + 1;
        line = endLine + 1;
      }
    }
    decompose(from, to, target, open) {
      for (let i = 0, pos = 0; pos <= to && i < this.children.length; i++) {
        let child = this.children[i],
          end = pos + child.length;
        if (from <= end && to >= pos) {
          let childOpen = open & ((pos <= from ? 1 /* Open.From */ : 0) | (end >= to ? 2 /* Open.To */ : 0));
          if (pos >= from && end <= to && !childOpen) target.push(child);else child.decompose(from - pos, to - pos, target, childOpen);
        }
        pos = end + 1;
      }
    }
    replace(from, to, text) {
      [from, to] = clip(this, from, to);
      if (text.lines < this.lines) for (let i = 0, pos = 0; i < this.children.length; i++) {
        let child = this.children[i],
          end = pos + child.length;
        // Fast path: if the change only affects one child and the
        // child's size remains in the acceptable range, only update
        // that child
        if (from >= pos && to <= end) {
          let updated = child.replace(from - pos, to - pos, text);
          let totalLines = this.lines - child.lines + updated.lines;
          if (updated.lines < totalLines >> 5 /* Tree.BranchShift */ - 1 && updated.lines > totalLines >> 5 /* Tree.BranchShift */ + 1) {
            let copy = this.children.slice();
            copy[i] = updated;
            return new TextNode(copy, this.length - (to - from) + text.length);
          }
          return super.replace(pos, end, updated);
        }
        pos = end + 1;
      }
      return super.replace(from, to, text);
    }
    sliceString(from, to = this.length, lineSep = "\n") {
      [from, to] = clip(this, from, to);
      let result = "";
      for (let i = 0, pos = 0; i < this.children.length && pos <= to; i++) {
        let child = this.children[i],
          end = pos + child.length;
        if (pos > from && i) result += lineSep;
        if (from < end && to > pos) result += child.sliceString(from - pos, to - pos, lineSep);
        pos = end + 1;
      }
      return result;
    }
    flatten(target) {
      for (let child of this.children) child.flatten(target);
    }
    scanIdentical(other, dir) {
      if (!(other instanceof TextNode)) return 0;
      let length = 0;
      let [iA, iB, eA, eB] = dir > 0 ? [0, 0, this.children.length, other.children.length] : [this.children.length - 1, other.children.length - 1, -1, -1];
      for (;; iA += dir, iB += dir) {
        if (iA == eA || iB == eB) return length;
        let chA = this.children[iA],
          chB = other.children[iB];
        if (chA != chB) return length + chA.scanIdentical(chB, dir);
        length += chA.length + 1;
      }
    }
    static from(children, length = children.reduce((l, ch) => l + ch.length + 1, -1)) {
      let lines = 0;
      for (let ch of children) lines += ch.lines;
      if (lines < 32 /* Tree.Branch */) {
        let flat = [];
        for (let ch of children) ch.flatten(flat);
        return new TextLeaf(flat, length);
      }
      let chunk = Math.max(32 /* Tree.Branch */, lines >> 5 /* Tree.BranchShift */),
        maxChunk = chunk << 1,
        minChunk = chunk >> 1;
      let chunked = [],
        currentLines = 0,
        currentLen = -1,
        currentChunk = [];
      function add(child) {
        let last;
        if (child.lines > maxChunk && child instanceof TextNode) {
          for (let node of child.children) add(node);
        } else if (child.lines > minChunk && (currentLines > minChunk || !currentLines)) {
          flush();
          chunked.push(child);
        } else if (child instanceof TextLeaf && currentLines && (last = currentChunk[currentChunk.length - 1]) instanceof TextLeaf && child.lines + last.lines <= 32 /* Tree.Branch */) {
          currentLines += child.lines;
          currentLen += child.length + 1;
          currentChunk[currentChunk.length - 1] = new TextLeaf(last.text.concat(child.text), last.length + 1 + child.length);
        } else {
          if (currentLines + child.lines > chunk) flush();
          currentLines += child.lines;
          currentLen += child.length + 1;
          currentChunk.push(child);
        }
      }
      function flush() {
        if (currentLines == 0) return;
        chunked.push(currentChunk.length == 1 ? currentChunk[0] : TextNode.from(currentChunk, currentLen));
        currentLen = -1;
        currentLines = currentChunk.length = 0;
      }
      for (let child of children) add(child);
      flush();
      return chunked.length == 1 ? chunked[0] : new TextNode(chunked, length);
    }
  }
  Text.empty = /*@__PURE__*/new TextLeaf([""], 0);
  function textLength(text) {
    let length = -1;
    for (let line of text) length += line.length + 1;
    return length;
  }
  function appendText(text, target, from = 0, to = 1e9) {
    for (let pos = 0, i = 0, first = true; i < text.length && pos <= to; i++) {
      let line = text[i],
        end = pos + line.length;
      if (end >= from) {
        if (end > to) line = line.slice(0, to - pos);
        if (pos < from) line = line.slice(from - pos);
        if (first) {
          target[target.length - 1] += line;
          first = false;
        } else target.push(line);
      }
      pos = end + 1;
    }
    return target;
  }
  function sliceText(text, from, to) {
    return appendText(text, [""], from, to);
  }
  class RawTextCursor {
    constructor(text, dir = 1) {
      this.dir = dir;
      this.done = false;
      this.lineBreak = false;
      this.value = "";
      this.nodes = [text];
      this.offsets = [dir > 0 ? 1 : (text instanceof TextLeaf ? text.text.length : text.children.length) << 1];
    }
    nextInner(skip, dir) {
      this.done = this.lineBreak = false;
      for (;;) {
        let last = this.nodes.length - 1;
        let top = this.nodes[last],
          offsetValue = this.offsets[last],
          offset = offsetValue >> 1;
        let size = top instanceof TextLeaf ? top.text.length : top.children.length;
        if (offset == (dir > 0 ? size : 0)) {
          if (last == 0) {
            this.done = true;
            this.value = "";
            return this;
          }
          if (dir > 0) this.offsets[last - 1]++;
          this.nodes.pop();
          this.offsets.pop();
        } else if ((offsetValue & 1) == (dir > 0 ? 0 : 1)) {
          this.offsets[last] += dir;
          if (skip == 0) {
            this.lineBreak = true;
            this.value = "\n";
            return this;
          }
          skip--;
        } else if (top instanceof TextLeaf) {
          // Move to the next string
          let next = top.text[offset + (dir < 0 ? -1 : 0)];
          this.offsets[last] += dir;
          if (next.length > Math.max(0, skip)) {
            this.value = skip == 0 ? next : dir > 0 ? next.slice(skip) : next.slice(0, next.length - skip);
            return this;
          }
          skip -= next.length;
        } else {
          let next = top.children[offset + (dir < 0 ? -1 : 0)];
          if (skip > next.length) {
            skip -= next.length;
            this.offsets[last] += dir;
          } else {
            if (dir < 0) this.offsets[last]--;
            this.nodes.push(next);
            this.offsets.push(dir > 0 ? 1 : (next instanceof TextLeaf ? next.text.length : next.children.length) << 1);
          }
        }
      }
    }
    next(skip = 0) {
      if (skip < 0) {
        this.nextInner(-skip, -this.dir);
        skip = this.value.length;
      }
      return this.nextInner(skip, this.dir);
    }
  }
  class PartialTextCursor {
    constructor(text, start, end) {
      this.value = "";
      this.done = false;
      this.cursor = new RawTextCursor(text, start > end ? -1 : 1);
      this.pos = start > end ? text.length : 0;
      this.from = Math.min(start, end);
      this.to = Math.max(start, end);
    }
    nextInner(skip, dir) {
      if (dir < 0 ? this.pos <= this.from : this.pos >= this.to) {
        this.value = "";
        this.done = true;
        return this;
      }
      skip += Math.max(0, dir < 0 ? this.pos - this.to : this.from - this.pos);
      let limit = dir < 0 ? this.pos - this.from : this.to - this.pos;
      if (skip > limit) skip = limit;
      limit -= skip;
      let {
        value
      } = this.cursor.next(skip);
      this.pos += (value.length + skip) * dir;
      this.value = value.length <= limit ? value : dir < 0 ? value.slice(value.length - limit) : value.slice(0, limit);
      this.done = !this.value;
      return this;
    }
    next(skip = 0) {
      if (skip < 0) skip = Math.max(skip, this.from - this.pos);else if (skip > 0) skip = Math.min(skip, this.to - this.pos);
      return this.nextInner(skip, this.cursor.dir);
    }
    get lineBreak() {
      return this.cursor.lineBreak && this.value != "";
    }
  }
  class LineCursor {
    constructor(inner) {
      this.inner = inner;
      this.afterBreak = true;
      this.value = "";
      this.done = false;
    }
    next(skip = 0) {
      let {
        done,
        lineBreak,
        value
      } = this.inner.next(skip);
      if (done && this.afterBreak) {
        this.value = "";
        this.afterBreak = false;
      } else if (done) {
        this.done = true;
        this.value = "";
      } else if (lineBreak) {
        if (this.afterBreak) {
          this.value = "";
        } else {
          this.afterBreak = true;
          this.next();
        }
      } else {
        this.value = value;
        this.afterBreak = false;
      }
      return this;
    }
    get lineBreak() {
      return false;
    }
  }
  if (typeof Symbol != "undefined") {
    Text.prototype[Symbol.iterator] = function () {
      return this.iter();
    };
    RawTextCursor.prototype[Symbol.iterator] = PartialTextCursor.prototype[Symbol.iterator] = LineCursor.prototype[Symbol.iterator] = function () {
      return this;
    };
  }
  /**
  This type describes a line in the document. It is created
  on-demand when lines are [queried](https://codemirror.net/6/docs/ref/#state.Text.lineAt).
  */
  class Line {
    /**
    @internal
    */
    constructor(
    /**
    The position of the start of the line.
    */
    from,
    /**
    The position at the end of the line (_before_ the line break,
    or at the end of document for the last line).
    */
    to,
    /**
    This line's line number (1-based).
    */
    number,
    /**
    The line's content.
    */
    text) {
      this.from = from;
      this.to = to;
      this.number = number;
      this.text = text;
    }
    /**
    The length of the line (not including any line break after it).
    */
    get length() {
      return this.to - this.from;
    }
  }
  function clip(text, from, to) {
    from = Math.max(0, Math.min(text.length, from));
    return [from, Math.max(from, Math.min(text.length, to))];
  }

  /**
  Returns a next grapheme cluster break _after_ (not equal to)
  `pos`, if `forward` is true, or before otherwise. Returns `pos`
  itself if no further cluster break is available in the string.
  Moves across surrogate pairs, extending characters (when
  `includeExtending` is true), characters joined with zero-width
  joiners, and flag emoji.
  */
  function findClusterBreak(str, pos, forward = true, includeExtending = true) {
    return findClusterBreak$1(str, pos, forward, includeExtending);
  }
  const DefaultSplit = /\r\n?|\n/;
  /**
  Distinguishes different ways in which positions can be mapped.
  */
  var MapMode = /*@__PURE__*/function (MapMode) {
    /**
    Map a position to a valid new position, even when its context
    was deleted.
    */
    MapMode[MapMode["Simple"] = 0] = "Simple";
    /**
    Return null if deletion happens across the position.
    */
    MapMode[MapMode["TrackDel"] = 1] = "TrackDel";
    /**
    Return null if the character _before_ the position is deleted.
    */
    MapMode[MapMode["TrackBefore"] = 2] = "TrackBefore";
    /**
    Return null if the character _after_ the position is deleted.
    */
    MapMode[MapMode["TrackAfter"] = 3] = "TrackAfter";
    return MapMode;
  }(MapMode || (MapMode = {}));
  /**
  A change description is a variant of [change set](https://codemirror.net/6/docs/ref/#state.ChangeSet)
  that doesn't store the inserted text. As such, it can't be
  applied, but is cheaper to store and manipulate.
  */
  class ChangeDesc {
    // Sections are encoded as pairs of integers. The first is the
    // length in the current document, and the second is -1 for
    // unaffected sections, and the length of the replacement content
    // otherwise. So an insertion would be (0, n>0), a deletion (n>0,
    // 0), and a replacement two positive numbers.
    /**
    @internal
    */
    constructor(
    /**
    @internal
    */
    sections) {
      this.sections = sections;
    }
    /**
    The length of the document before the change.
    */
    get length() {
      let result = 0;
      for (let i = 0; i < this.sections.length; i += 2) result += this.sections[i];
      return result;
    }
    /**
    The length of the document after the change.
    */
    get newLength() {
      let result = 0;
      for (let i = 0; i < this.sections.length; i += 2) {
        let ins = this.sections[i + 1];
        result += ins < 0 ? this.sections[i] : ins;
      }
      return result;
    }
    /**
    False when there are actual changes in this set.
    */
    get empty() {
      return this.sections.length == 0 || this.sections.length == 2 && this.sections[1] < 0;
    }
    /**
    Iterate over the unchanged parts left by these changes. `posA`
    provides the position of the range in the old document, `posB`
    the new position in the changed document.
    */
    iterGaps(f) {
      for (let i = 0, posA = 0, posB = 0; i < this.sections.length;) {
        let len = this.sections[i++],
          ins = this.sections[i++];
        if (ins < 0) {
          f(posA, posB, len);
          posB += len;
        } else {
          posB += ins;
        }
        posA += len;
      }
    }
    /**
    Iterate over the ranges changed by these changes. (See
    [`ChangeSet.iterChanges`](https://codemirror.net/6/docs/ref/#state.ChangeSet.iterChanges) for a
    variant that also provides you with the inserted text.)
    `fromA`/`toA` provides the extent of the change in the starting
    document, `fromB`/`toB` the extent of the replacement in the
    changed document.
    
    When `individual` is true, adjacent changes (which are kept
    separate for [position mapping](https://codemirror.net/6/docs/ref/#state.ChangeDesc.mapPos)) are
    reported separately.
    */
    iterChangedRanges(f, individual = false) {
      iterChanges(this, f, individual);
    }
    /**
    Get a description of the inverted form of these changes.
    */
    get invertedDesc() {
      let sections = [];
      for (let i = 0; i < this.sections.length;) {
        let len = this.sections[i++],
          ins = this.sections[i++];
        if (ins < 0) sections.push(len, ins);else sections.push(ins, len);
      }
      return new ChangeDesc(sections);
    }
    /**
    Compute the combined effect of applying another set of changes
    after this one. The length of the document after this set should
    match the length before `other`.
    */
    composeDesc(other) {
      return this.empty ? other : other.empty ? this : composeSets(this, other);
    }
    /**
    Map this description, which should start with the same document
    as `other`, over another set of changes, so that it can be
    applied after it. When `before` is true, map as if the changes
    in `this` happened before the ones in `other`.
    */
    mapDesc(other, before = false) {
      return other.empty ? this : mapSet(this, other, before);
    }
    mapPos(pos, assoc = -1, mode = MapMode.Simple) {
      let posA = 0,
        posB = 0;
      for (let i = 0; i < this.sections.length;) {
        let len = this.sections[i++],
          ins = this.sections[i++],
          endA = posA + len;
        if (ins < 0) {
          if (endA > pos) return posB + (pos - posA);
          posB += len;
        } else {
          if (mode != MapMode.Simple && endA >= pos && (mode == MapMode.TrackDel && posA < pos && endA > pos || mode == MapMode.TrackBefore && posA < pos || mode == MapMode.TrackAfter && endA > pos)) return null;
          if (endA > pos || endA == pos && assoc < 0 && !len) return pos == posA || assoc < 0 ? posB : posB + ins;
          posB += ins;
        }
        posA = endA;
      }
      if (pos > posA) throw new RangeError(`Position ${pos} is out of range for changeset of length ${posA}`);
      return posB;
    }
    /**
    Check whether these changes touch a given range. When one of the
    changes entirely covers the range, the string `"cover"` is
    returned.
    */
    touchesRange(from, to = from) {
      for (let i = 0, pos = 0; i < this.sections.length && pos <= to;) {
        let len = this.sections[i++],
          ins = this.sections[i++],
          end = pos + len;
        if (ins >= 0 && pos <= to && end >= from) return pos < from && end > to ? "cover" : true;
        pos = end;
      }
      return false;
    }
    /**
    @internal
    */
    toString() {
      let result = "";
      for (let i = 0; i < this.sections.length;) {
        let len = this.sections[i++],
          ins = this.sections[i++];
        result += (result ? " " : "") + len + (ins >= 0 ? ":" + ins : "");
      }
      return result;
    }
    /**
    Serialize this change desc to a JSON-representable value.
    */
    toJSON() {
      return this.sections;
    }
    /**
    Create a change desc from its JSON representation (as produced
    by [`toJSON`](https://codemirror.net/6/docs/ref/#state.ChangeDesc.toJSON).
    */
    static fromJSON(json) {
      if (!Array.isArray(json) || json.length % 2 || json.some(a => typeof a != "number")) throw new RangeError("Invalid JSON representation of ChangeDesc");
      return new ChangeDesc(json);
    }
    /**
    @internal
    */
    static create(sections) {
      return new ChangeDesc(sections);
    }
  }
  /**
  A change set represents a group of modifications to a document. It
  stores the document length, and can only be applied to documents
  with exactly that length.
  */
  class ChangeSet extends ChangeDesc {
    constructor(sections,
    /**
    @internal
    */
    inserted) {
      super(sections);
      this.inserted = inserted;
    }
    /**
    Apply the changes to a document, returning the modified
    document.
    */
    apply(doc) {
      if (this.length != doc.length) throw new RangeError("Applying change set to a document with the wrong length");
      iterChanges(this, (fromA, toA, fromB, _toB, text) => doc = doc.replace(fromB, fromB + (toA - fromA), text), false);
      return doc;
    }
    mapDesc(other, before = false) {
      return mapSet(this, other, before, true);
    }
    /**
    Given the document as it existed _before_ the changes, return a
    change set that represents the inverse of this set, which could
    be used to go from the document created by the changes back to
    the document as it existed before the changes.
    */
    invert(doc) {
      let sections = this.sections.slice(),
        inserted = [];
      for (let i = 0, pos = 0; i < sections.length; i += 2) {
        let len = sections[i],
          ins = sections[i + 1];
        if (ins >= 0) {
          sections[i] = ins;
          sections[i + 1] = len;
          let index = i >> 1;
          while (inserted.length < index) inserted.push(Text.empty);
          inserted.push(len ? doc.slice(pos, pos + len) : Text.empty);
        }
        pos += len;
      }
      return new ChangeSet(sections, inserted);
    }
    /**
    Combine two subsequent change sets into a single set. `other`
    must start in the document produced by `this`. If `this` goes
    `docA` → `docB` and `other` represents `docB` → `docC`, the
    returned value will represent the change `docA` → `docC`.
    */
    compose(other) {
      return this.empty ? other : other.empty ? this : composeSets(this, other, true);
    }
    /**
    Given another change set starting in the same document, maps this
    change set over the other, producing a new change set that can be
    applied to the document produced by applying `other`. When
    `before` is `true`, order changes as if `this` comes before
    `other`, otherwise (the default) treat `other` as coming first.
    
    Given two changes `A` and `B`, `A.compose(B.map(A))` and
    `B.compose(A.map(B, true))` will produce the same document. This
    provides a basic form of [operational
    transformation](https://en.wikipedia.org/wiki/Operational_transformation),
    and can be used for collaborative editing.
    */
    map(other, before = false) {
      return other.empty ? this : mapSet(this, other, before, true);
    }
    /**
    Iterate over the changed ranges in the document, calling `f` for
    each, with the range in the original document (`fromA`-`toA`)
    and the range that replaces it in the new document
    (`fromB`-`toB`).
    
    When `individual` is true, adjacent changes are reported
    separately.
    */
    iterChanges(f, individual = false) {
      iterChanges(this, f, individual);
    }
    /**
    Get a [change description](https://codemirror.net/6/docs/ref/#state.ChangeDesc) for this change
    set.
    */
    get desc() {
      return ChangeDesc.create(this.sections);
    }
    /**
    @internal
    */
    filter(ranges) {
      let resultSections = [],
        resultInserted = [],
        filteredSections = [];
      let iter = new SectionIter(this);
      done: for (let i = 0, pos = 0;;) {
        let next = i == ranges.length ? 1e9 : ranges[i++];
        while (pos < next || pos == next && iter.len == 0) {
          if (iter.done) break done;
          let len = Math.min(iter.len, next - pos);
          addSection(filteredSections, len, -1);
          let ins = iter.ins == -1 ? -1 : iter.off == 0 ? iter.ins : 0;
          addSection(resultSections, len, ins);
          if (ins > 0) addInsert(resultInserted, resultSections, iter.text);
          iter.forward(len);
          pos += len;
        }
        let end = ranges[i++];
        while (pos < end) {
          if (iter.done) break done;
          let len = Math.min(iter.len, end - pos);
          addSection(resultSections, len, -1);
          addSection(filteredSections, len, iter.ins == -1 ? -1 : iter.off == 0 ? iter.ins : 0);
          iter.forward(len);
          pos += len;
        }
      }
      return {
        changes: new ChangeSet(resultSections, resultInserted),
        filtered: ChangeDesc.create(filteredSections)
      };
    }
    /**
    Serialize this change set to a JSON-representable value.
    */
    toJSON() {
      let parts = [];
      for (let i = 0; i < this.sections.length; i += 2) {
        let len = this.sections[i],
          ins = this.sections[i + 1];
        if (ins < 0) parts.push(len);else if (ins == 0) parts.push([len]);else parts.push([len].concat(this.inserted[i >> 1].toJSON()));
      }
      return parts;
    }
    /**
    Create a change set for the given changes, for a document of the
    given length, using `lineSep` as line separator.
    */
    static of(changes, length, lineSep) {
      let sections = [],
        inserted = [],
        pos = 0;
      let total = null;
      function flush(force = false) {
        if (!force && !sections.length) return;
        if (pos < length) addSection(sections, length - pos, -1);
        let set = new ChangeSet(sections, inserted);
        total = total ? total.compose(set.map(total)) : set;
        sections = [];
        inserted = [];
        pos = 0;
      }
      function process(spec) {
        if (Array.isArray(spec)) {
          for (let sub of spec) process(sub);
        } else if (spec instanceof ChangeSet) {
          if (spec.length != length) throw new RangeError(`Mismatched change set length (got ${spec.length}, expected ${length})`);
          flush();
          total = total ? total.compose(spec.map(total)) : spec;
        } else {
          let {
            from,
            to = from,
            insert
          } = spec;
          if (from > to || from < 0 || to > length) throw new RangeError(`Invalid change range ${from} to ${to} (in doc of length ${length})`);
          let insText = !insert ? Text.empty : typeof insert == "string" ? Text.of(insert.split(lineSep || DefaultSplit)) : insert;
          let insLen = insText.length;
          if (from == to && insLen == 0) return;
          if (from < pos) flush();
          if (from > pos) addSection(sections, from - pos, -1);
          addSection(sections, to - from, insLen);
          addInsert(inserted, sections, insText);
          pos = to;
        }
      }
      process(changes);
      flush(!total);
      return total;
    }
    /**
    Create an empty changeset of the given length.
    */
    static empty(length) {
      return new ChangeSet(length ? [length, -1] : [], []);
    }
    /**
    Create a changeset from its JSON representation (as produced by
    [`toJSON`](https://codemirror.net/6/docs/ref/#state.ChangeSet.toJSON).
    */
    static fromJSON(json) {
      if (!Array.isArray(json)) throw new RangeError("Invalid JSON representation of ChangeSet");
      let sections = [],
        inserted = [];
      for (let i = 0; i < json.length; i++) {
        let part = json[i];
        if (typeof part == "number") {
          sections.push(part, -1);
        } else if (!Array.isArray(part) || typeof part[0] != "number" || part.some((e, i) => i && typeof e != "string")) {
          throw new RangeError("Invalid JSON representation of ChangeSet");
        } else if (part.length == 1) {
          sections.push(part[0], 0);
        } else {
          while (inserted.length < i) inserted.push(Text.empty);
          inserted[i] = Text.of(part.slice(1));
          sections.push(part[0], inserted[i].length);
        }
      }
      return new ChangeSet(sections, inserted);
    }
    /**
    @internal
    */
    static createSet(sections, inserted) {
      return new ChangeSet(sections, inserted);
    }
  }
  function addSection(sections, len, ins, forceJoin = false) {
    if (len == 0 && ins <= 0) return;
    let last = sections.length - 2;
    if (last >= 0 && ins <= 0 && ins == sections[last + 1]) sections[last] += len;else if (last >= 0 && len == 0 && sections[last] == 0) sections[last + 1] += ins;else if (forceJoin) {
      sections[last] += len;
      sections[last + 1] += ins;
    } else sections.push(len, ins);
  }
  function addInsert(values, sections, value) {
    if (value.length == 0) return;
    let index = sections.length - 2 >> 1;
    if (index < values.length) {
      values[values.length - 1] = values[values.length - 1].append(value);
    } else {
      while (values.length < index) values.push(Text.empty);
      values.push(value);
    }
  }
  function iterChanges(desc, f, individual) {
    let inserted = desc.inserted;
    for (let posA = 0, posB = 0, i = 0; i < desc.sections.length;) {
      let len = desc.sections[i++],
        ins = desc.sections[i++];
      if (ins < 0) {
        posA += len;
        posB += len;
      } else {
        let endA = posA,
          endB = posB,
          text = Text.empty;
        for (;;) {
          endA += len;
          endB += ins;
          if (ins && inserted) text = text.append(inserted[i - 2 >> 1]);
          if (individual || i == desc.sections.length || desc.sections[i + 1] < 0) break;
          len = desc.sections[i++];
          ins = desc.sections[i++];
        }
        f(posA, endA, posB, endB, text);
        posA = endA;
        posB = endB;
      }
    }
  }
  function mapSet(setA, setB, before, mkSet = false) {
    // Produce a copy of setA that applies to the document after setB
    // has been applied (assuming both start at the same document).
    let sections = [],
      insert = mkSet ? [] : null;
    let a = new SectionIter(setA),
      b = new SectionIter(setB);
    // Iterate over both sets in parallel. inserted tracks, for changes
    // in A that have to be processed piece-by-piece, whether their
    // content has been inserted already, and refers to the section
    // index.
    for (let inserted = -1;;) {
      if (a.done && b.len || b.done && a.len) {
        throw new Error("Mismatched change set lengths");
      } else if (a.ins == -1 && b.ins == -1) {
        // Move across ranges skipped by both sets.
        let len = Math.min(a.len, b.len);
        addSection(sections, len, -1);
        a.forward(len);
        b.forward(len);
      } else if (b.ins >= 0 && (a.ins < 0 || inserted == a.i || a.off == 0 && (b.len < a.len || b.len == a.len && !before))) {
        // If there's a change in B that comes before the next change in
        // A (ordered by start pos, then len, then before flag), skip
        // that (and process any changes in A it covers).
        let len = b.len;
        addSection(sections, b.ins, -1);
        while (len) {
          let piece = Math.min(a.len, len);
          if (a.ins >= 0 && inserted < a.i && a.len <= piece) {
            addSection(sections, 0, a.ins);
            if (insert) addInsert(insert, sections, a.text);
            inserted = a.i;
          }
          a.forward(piece);
          len -= piece;
        }
        b.next();
      } else if (a.ins >= 0) {
        // Process the part of a change in A up to the start of the next
        // non-deletion change in B (if overlapping).
        let len = 0,
          left = a.len;
        while (left) {
          if (b.ins == -1) {
            let piece = Math.min(left, b.len);
            len += piece;
            left -= piece;
            b.forward(piece);
          } else if (b.ins == 0 && b.len < left) {
            left -= b.len;
            b.next();
          } else {
            break;
          }
        }
        addSection(sections, len, inserted < a.i ? a.ins : 0);
        if (insert && inserted < a.i) addInsert(insert, sections, a.text);
        inserted = a.i;
        a.forward(a.len - left);
      } else if (a.done && b.done) {
        return insert ? ChangeSet.createSet(sections, insert) : ChangeDesc.create(sections);
      } else {
        throw new Error("Mismatched change set lengths");
      }
    }
  }
  function composeSets(setA, setB, mkSet = false) {
    let sections = [];
    let insert = mkSet ? [] : null;
    let a = new SectionIter(setA),
      b = new SectionIter(setB);
    for (let open = false;;) {
      if (a.done && b.done) {
        return insert ? ChangeSet.createSet(sections, insert) : ChangeDesc.create(sections);
      } else if (a.ins == 0) {
        // Deletion in A
        addSection(sections, a.len, 0, open);
        a.next();
      } else if (b.len == 0 && !b.done) {
        // Insertion in B
        addSection(sections, 0, b.ins, open);
        if (insert) addInsert(insert, sections, b.text);
        b.next();
      } else if (a.done || b.done) {
        throw new Error("Mismatched change set lengths");
      } else {
        let len = Math.min(a.len2, b.len),
          sectionLen = sections.length;
        if (a.ins == -1) {
          let insB = b.ins == -1 ? -1 : b.off ? 0 : b.ins;
          addSection(sections, len, insB, open);
          if (insert && insB) addInsert(insert, sections, b.text);
        } else if (b.ins == -1) {
          addSection(sections, a.off ? 0 : a.len, len, open);
          if (insert) addInsert(insert, sections, a.textBit(len));
        } else {
          addSection(sections, a.off ? 0 : a.len, b.off ? 0 : b.ins, open);
          if (insert && !b.off) addInsert(insert, sections, b.text);
        }
        open = (a.ins > len || b.ins >= 0 && b.len > len) && (open || sections.length > sectionLen);
        a.forward2(len);
        b.forward(len);
      }
    }
  }
  class SectionIter {
    constructor(set) {
      this.set = set;
      this.i = 0;
      this.next();
    }
    next() {
      let {
        sections
      } = this.set;
      if (this.i < sections.length) {
        this.len = sections[this.i++];
        this.ins = sections[this.i++];
      } else {
        this.len = 0;
        this.ins = -2;
      }
      this.off = 0;
    }
    get done() {
      return this.ins == -2;
    }
    get len2() {
      return this.ins < 0 ? this.len : this.ins;
    }
    get text() {
      let {
          inserted
        } = this.set,
        index = this.i - 2 >> 1;
      return index >= inserted.length ? Text.empty : inserted[index];
    }
    textBit(len) {
      let {
          inserted
        } = this.set,
        index = this.i - 2 >> 1;
      return index >= inserted.length && !len ? Text.empty : inserted[index].slice(this.off, len == null ? undefined : this.off + len);
    }
    forward(len) {
      if (len == this.len) this.next();else {
        this.len -= len;
        this.off += len;
      }
    }
    forward2(len) {
      if (this.ins == -1) this.forward(len);else if (len == this.ins) this.next();else {
        this.ins -= len;
        this.off += len;
      }
    }
  }

  /**
  A single selection range. When
  [`allowMultipleSelections`](https://codemirror.net/6/docs/ref/#state.EditorState^allowMultipleSelections)
  is enabled, a [selection](https://codemirror.net/6/docs/ref/#state.EditorSelection) may hold
  multiple ranges. By default, selections hold exactly one range.
  */
  class SelectionRange {
    constructor(
    /**
    The lower boundary of the range.
    */
    from,
    /**
    The upper boundary of the range.
    */
    to, flags) {
      this.from = from;
      this.to = to;
      this.flags = flags;
    }
    /**
    The anchor of the range—the side that doesn't move when you
    extend it.
    */
    get anchor() {
      return this.flags & 32 /* RangeFlag.Inverted */ ? this.to : this.from;
    }
    /**
    The head of the range, which is moved when the range is
    [extended](https://codemirror.net/6/docs/ref/#state.SelectionRange.extend).
    */
    get head() {
      return this.flags & 32 /* RangeFlag.Inverted */ ? this.from : this.to;
    }
    /**
    True when `anchor` and `head` are at the same position.
    */
    get empty() {
      return this.from == this.to;
    }
    /**
    If this is a cursor that is explicitly associated with the
    character on one of its sides, this returns the side. -1 means
    the character before its position, 1 the character after, and 0
    means no association.
    */
    get assoc() {
      return this.flags & 8 /* RangeFlag.AssocBefore */ ? -1 : this.flags & 16 /* RangeFlag.AssocAfter */ ? 1 : 0;
    }
    /**
    The bidirectional text level associated with this cursor, if
    any.
    */
    get bidiLevel() {
      let level = this.flags & 7 /* RangeFlag.BidiLevelMask */;
      return level == 7 ? null : level;
    }
    /**
    The goal column (stored vertical offset) associated with a
    cursor. This is used to preserve the vertical position when
    [moving](https://codemirror.net/6/docs/ref/#view.EditorView.moveVertically) across
    lines of different length.
    */
    get goalColumn() {
      let value = this.flags >> 6 /* RangeFlag.GoalColumnOffset */;
      return value == 16777215 /* RangeFlag.NoGoalColumn */ ? undefined : value;
    }
    /**
    Map this range through a change, producing a valid range in the
    updated document.
    */
    map(change, assoc = -1) {
      let from, to;
      if (this.empty) {
        from = to = change.mapPos(this.from, assoc);
      } else {
        from = change.mapPos(this.from, 1);
        to = change.mapPos(this.to, -1);
      }
      return from == this.from && to == this.to ? this : new SelectionRange(from, to, this.flags);
    }
    /**
    Extend this range to cover at least `from` to `to`.
    */
    extend(from, to = from) {
      if (from <= this.anchor && to >= this.anchor) return EditorSelection.range(from, to);
      let head = Math.abs(from - this.anchor) > Math.abs(to - this.anchor) ? from : to;
      return EditorSelection.range(this.anchor, head);
    }
    /**
    Compare this range to another range.
    */
    eq(other, includeAssoc = false) {
      return this.anchor == other.anchor && this.head == other.head && (!includeAssoc || !this.empty || this.assoc == other.assoc);
    }
    /**
    Return a JSON-serializable object representing the range.
    */
    toJSON() {
      return {
        anchor: this.anchor,
        head: this.head
      };
    }
    /**
    Convert a JSON representation of a range to a `SelectionRange`
    instance.
    */
    static fromJSON(json) {
      if (!json || typeof json.anchor != "number" || typeof json.head != "number") throw new RangeError("Invalid JSON representation for SelectionRange");
      return EditorSelection.range(json.anchor, json.head);
    }
    /**
    @internal
    */
    static create(from, to, flags) {
      return new SelectionRange(from, to, flags);
    }
  }
  /**
  An editor selection holds one or more selection ranges.
  */
  class EditorSelection {
    constructor(
    /**
    The ranges in the selection, sorted by position. Ranges cannot
    overlap (but they may touch, if they aren't empty).
    */
    ranges,
    /**
    The index of the _main_ range in the selection (which is
    usually the range that was added last).
    */
    mainIndex) {
      this.ranges = ranges;
      this.mainIndex = mainIndex;
    }
    /**
    Map a selection through a change. Used to adjust the selection
    position for changes.
    */
    map(change, assoc = -1) {
      if (change.empty) return this;
      return EditorSelection.create(this.ranges.map(r => r.map(change, assoc)), this.mainIndex);
    }
    /**
    Compare this selection to another selection. By default, ranges
    are compared only by position. When `includeAssoc` is true,
    cursor ranges must also have the same
    [`assoc`](https://codemirror.net/6/docs/ref/#state.SelectionRange.assoc) value.
    */
    eq(other, includeAssoc = false) {
      if (this.ranges.length != other.ranges.length || this.mainIndex != other.mainIndex) return false;
      for (let i = 0; i < this.ranges.length; i++) if (!this.ranges[i].eq(other.ranges[i], includeAssoc)) return false;
      return true;
    }
    /**
    Get the primary selection range. Usually, you should make sure
    your code applies to _all_ ranges, by using methods like
    [`changeByRange`](https://codemirror.net/6/docs/ref/#state.EditorState.changeByRange).
    */
    get main() {
      return this.ranges[this.mainIndex];
    }
    /**
    Make sure the selection only has one range. Returns a selection
    holding only the main range from this selection.
    */
    asSingle() {
      return this.ranges.length == 1 ? this : new EditorSelection([this.main], 0);
    }
    /**
    Extend this selection with an extra range.
    */
    addRange(range, main = true) {
      return EditorSelection.create([range].concat(this.ranges), main ? 0 : this.mainIndex + 1);
    }
    /**
    Replace a given range with another range, and then normalize the
    selection to merge and sort ranges if necessary.
    */
    replaceRange(range, which = this.mainIndex) {
      let ranges = this.ranges.slice();
      ranges[which] = range;
      return EditorSelection.create(ranges, this.mainIndex);
    }
    /**
    Convert this selection to an object that can be serialized to
    JSON.
    */
    toJSON() {
      return {
        ranges: this.ranges.map(r => r.toJSON()),
        main: this.mainIndex
      };
    }
    /**
    Create a selection from a JSON representation.
    */
    static fromJSON(json) {
      if (!json || !Array.isArray(json.ranges) || typeof json.main != "number" || json.main >= json.ranges.length) throw new RangeError("Invalid JSON representation for EditorSelection");
      return new EditorSelection(json.ranges.map(r => SelectionRange.fromJSON(r)), json.main);
    }
    /**
    Create a selection holding a single range.
    */
    static single(anchor, head = anchor) {
      return new EditorSelection([EditorSelection.range(anchor, head)], 0);
    }
    /**
    Sort and merge the given set of ranges, creating a valid
    selection.
    */
    static create(ranges, mainIndex = 0) {
      if (ranges.length == 0) throw new RangeError("A selection needs at least one range");
      for (let pos = 0, i = 0; i < ranges.length; i++) {
        let range = ranges[i];
        if (range.empty ? range.from <= pos : range.from < pos) return EditorSelection.normalized(ranges.slice(), mainIndex);
        pos = range.to;
      }
      return new EditorSelection(ranges, mainIndex);
    }
    /**
    Create a cursor selection range at the given position. You can
    safely ignore the optional arguments in most situations.
    */
    static cursor(pos, assoc = 0, bidiLevel, goalColumn) {
      return SelectionRange.create(pos, pos, (assoc == 0 ? 0 : assoc < 0 ? 8 /* RangeFlag.AssocBefore */ : 16 /* RangeFlag.AssocAfter */) | (bidiLevel == null ? 7 : Math.min(6, bidiLevel)) | (goalColumn !== null && goalColumn !== void 0 ? goalColumn : 16777215 /* RangeFlag.NoGoalColumn */) << 6 /* RangeFlag.GoalColumnOffset */);
    }
    /**
    Create a selection range.
    */
    static range(anchor, head, goalColumn, bidiLevel) {
      let flags = (goalColumn !== null && goalColumn !== void 0 ? goalColumn : 16777215 /* RangeFlag.NoGoalColumn */) << 6 /* RangeFlag.GoalColumnOffset */ | (bidiLevel == null ? 7 : Math.min(6, bidiLevel));
      return head < anchor ? SelectionRange.create(head, anchor, 32 /* RangeFlag.Inverted */ | 16 /* RangeFlag.AssocAfter */ | flags) : SelectionRange.create(anchor, head, (head > anchor ? 8 /* RangeFlag.AssocBefore */ : 0) | flags);
    }
    /**
    @internal
    */
    static normalized(ranges, mainIndex = 0) {
      let main = ranges[mainIndex];
      ranges.sort((a, b) => a.from - b.from);
      mainIndex = ranges.indexOf(main);
      for (let i = 1; i < ranges.length; i++) {
        let range = ranges[i],
          prev = ranges[i - 1];
        if (range.empty ? range.from <= prev.to : range.from < prev.to) {
          let from = prev.from,
            to = Math.max(range.to, prev.to);
          if (i <= mainIndex) mainIndex--;
          ranges.splice(--i, 2, range.anchor > range.head ? EditorSelection.range(to, from) : EditorSelection.range(from, to));
        }
      }
      return new EditorSelection(ranges, mainIndex);
    }
  }
  function checkSelection(selection, docLength) {
    for (let range of selection.ranges) if (range.to > docLength) throw new RangeError("Selection points outside of document");
  }
  let nextID = 0;
  /**
  A facet is a labeled value that is associated with an editor
  state. It takes inputs from any number of extensions, and combines
  those into a single output value.

  Examples of uses of facets are the [tab
  size](https://codemirror.net/6/docs/ref/#state.EditorState^tabSize), [editor
  attributes](https://codemirror.net/6/docs/ref/#view.EditorView^editorAttributes), and [update
  listeners](https://codemirror.net/6/docs/ref/#view.EditorView^updateListener).

  Note that `Facet` instances can be used anywhere where
  [`FacetReader`](https://codemirror.net/6/docs/ref/#state.FacetReader) is expected.
  */
  class Facet {
    constructor(
    /**
    @internal
    */
    combine,
    /**
    @internal
    */
    compareInput,
    /**
    @internal
    */
    compare, isStatic, enables) {
      this.combine = combine;
      this.compareInput = compareInput;
      this.compare = compare;
      this.isStatic = isStatic;
      /**
      @internal
      */
      this.id = nextID++;
      this.default = combine([]);
      this.extensions = typeof enables == "function" ? enables(this) : enables;
    }
    /**
    Returns a facet reader for this facet, which can be used to
    [read](https://codemirror.net/6/docs/ref/#state.EditorState.facet) it but not to define values for it.
    */
    get reader() {
      return this;
    }
    /**
    Define a new facet.
    */
    static define(config = {}) {
      return new Facet(config.combine || (a => a), config.compareInput || ((a, b) => a === b), config.compare || (!config.combine ? sameArray$1 : (a, b) => a === b), !!config.static, config.enables);
    }
    /**
    Returns an extension that adds the given value to this facet.
    */
    of(value) {
      return new FacetProvider([], this, 0 /* Provider.Static */, value);
    }
    /**
    Create an extension that computes a value for the facet from a
    state. You must take care to declare the parts of the state that
    this value depends on, since your function is only called again
    for a new state when one of those parts changed.
    
    In cases where your value depends only on a single field, you'll
    want to use the [`from`](https://codemirror.net/6/docs/ref/#state.Facet.from) method instead.
    */
    compute(deps, get) {
      if (this.isStatic) throw new Error("Can't compute a static facet");
      return new FacetProvider(deps, this, 1 /* Provider.Single */, get);
    }
    /**
    Create an extension that computes zero or more values for this
    facet from a state.
    */
    computeN(deps, get) {
      if (this.isStatic) throw new Error("Can't compute a static facet");
      return new FacetProvider(deps, this, 2 /* Provider.Multi */, get);
    }
    from(field, get) {
      if (!get) get = x => x;
      return this.compute([field], state => get(state.field(field)));
    }
  }
  function sameArray$1(a, b) {
    return a == b || a.length == b.length && a.every((e, i) => e === b[i]);
  }
  class FacetProvider {
    constructor(dependencies, facet, type, value) {
      this.dependencies = dependencies;
      this.facet = facet;
      this.type = type;
      this.value = value;
      this.id = nextID++;
    }
    dynamicSlot(addresses) {
      var _a;
      let getter = this.value;
      let compare = this.facet.compareInput;
      let id = this.id,
        idx = addresses[id] >> 1,
        multi = this.type == 2 /* Provider.Multi */;
      let depDoc = false,
        depSel = false,
        depAddrs = [];
      for (let dep of this.dependencies) {
        if (dep == "doc") depDoc = true;else if (dep == "selection") depSel = true;else if ((((_a = addresses[dep.id]) !== null && _a !== void 0 ? _a : 1) & 1) == 0) depAddrs.push(addresses[dep.id]);
      }
      return {
        create(state) {
          state.values[idx] = getter(state);
          return 1 /* SlotStatus.Changed */;
        },
        update(state, tr) {
          if (depDoc && tr.docChanged || depSel && (tr.docChanged || tr.selection) || ensureAll(state, depAddrs)) {
            let newVal = getter(state);
            if (multi ? !compareArray(newVal, state.values[idx], compare) : !compare(newVal, state.values[idx])) {
              state.values[idx] = newVal;
              return 1 /* SlotStatus.Changed */;
            }
          }
          return 0;
        },
        reconfigure: (state, oldState) => {
          let newVal,
            oldAddr = oldState.config.address[id];
          if (oldAddr != null) {
            let oldVal = getAddr(oldState, oldAddr);
            if (this.dependencies.every(dep => {
              return dep instanceof Facet ? oldState.facet(dep) === state.facet(dep) : dep instanceof StateField ? oldState.field(dep, false) == state.field(dep, false) : true;
            }) || (multi ? compareArray(newVal = getter(state), oldVal, compare) : compare(newVal = getter(state), oldVal))) {
              state.values[idx] = oldVal;
              return 0;
            }
          } else {
            newVal = getter(state);
          }
          state.values[idx] = newVal;
          return 1 /* SlotStatus.Changed */;
        }
      };
    }
  }
  function compareArray(a, b, compare) {
    if (a.length != b.length) return false;
    for (let i = 0; i < a.length; i++) if (!compare(a[i], b[i])) return false;
    return true;
  }
  function ensureAll(state, addrs) {
    let changed = false;
    for (let addr of addrs) if (ensureAddr(state, addr) & 1 /* SlotStatus.Changed */) changed = true;
    return changed;
  }
  function dynamicFacetSlot(addresses, facet, providers) {
    let providerAddrs = providers.map(p => addresses[p.id]);
    let providerTypes = providers.map(p => p.type);
    let dynamic = providerAddrs.filter(p => !(p & 1));
    let idx = addresses[facet.id] >> 1;
    function get(state) {
      let values = [];
      for (let i = 0; i < providerAddrs.length; i++) {
        let value = getAddr(state, providerAddrs[i]);
        if (providerTypes[i] == 2 /* Provider.Multi */) for (let val of value) values.push(val);else values.push(value);
      }
      return facet.combine(values);
    }
    return {
      create(state) {
        for (let addr of providerAddrs) ensureAddr(state, addr);
        state.values[idx] = get(state);
        return 1 /* SlotStatus.Changed */;
      },
      update(state, tr) {
        if (!ensureAll(state, dynamic)) return 0;
        let value = get(state);
        if (facet.compare(value, state.values[idx])) return 0;
        state.values[idx] = value;
        return 1 /* SlotStatus.Changed */;
      },
      reconfigure(state, oldState) {
        let depChanged = ensureAll(state, providerAddrs);
        let oldProviders = oldState.config.facets[facet.id],
          oldValue = oldState.facet(facet);
        if (oldProviders && !depChanged && sameArray$1(providers, oldProviders)) {
          state.values[idx] = oldValue;
          return 0;
        }
        let value = get(state);
        if (facet.compare(value, oldValue)) {
          state.values[idx] = oldValue;
          return 0;
        }
        state.values[idx] = value;
        return 1 /* SlotStatus.Changed */;
      }
    };
  }
  const initField = /*@__PURE__*/Facet.define({
    static: true
  });
  /**
  Fields can store additional information in an editor state, and
  keep it in sync with the rest of the state.
  */
  class StateField {
    constructor(
    /**
    @internal
    */
    id, createF, updateF, compareF,
    /**
    @internal
    */
    spec) {
      this.id = id;
      this.createF = createF;
      this.updateF = updateF;
      this.compareF = compareF;
      this.spec = spec;
      /**
      @internal
      */
      this.provides = undefined;
    }
    /**
    Define a state field.
    */
    static define(config) {
      let field = new StateField(nextID++, config.create, config.update, config.compare || ((a, b) => a === b), config);
      if (config.provide) field.provides = config.provide(field);
      return field;
    }
    create(state) {
      let init = state.facet(initField).find(i => i.field == this);
      return ((init === null || init === void 0 ? void 0 : init.create) || this.createF)(state);
    }
    /**
    @internal
    */
    slot(addresses) {
      let idx = addresses[this.id] >> 1;
      return {
        create: state => {
          state.values[idx] = this.create(state);
          return 1 /* SlotStatus.Changed */;
        },
        update: (state, tr) => {
          let oldVal = state.values[idx];
          let value = this.updateF(oldVal, tr);
          if (this.compareF(oldVal, value)) return 0;
          state.values[idx] = value;
          return 1 /* SlotStatus.Changed */;
        },
        reconfigure: (state, oldState) => {
          if (oldState.config.address[this.id] != null) {
            state.values[idx] = oldState.field(this);
            return 0;
          }
          state.values[idx] = this.create(state);
          return 1 /* SlotStatus.Changed */;
        }
      };
    }
    /**
    Returns an extension that enables this field and overrides the
    way it is initialized. Can be useful when you need to provide a
    non-default starting value for the field.
    */
    init(create) {
      return [this, initField.of({
        field: this,
        create
      })];
    }
    /**
    State field instances can be used as
    [`Extension`](https://codemirror.net/6/docs/ref/#state.Extension) values to enable the field in a
    given state.
    */
    get extension() {
      return this;
    }
  }
  const Prec_ = {
    lowest: 4,
    low: 3,
    default: 2,
    high: 1,
    highest: 0
  };
  function prec(value) {
    return ext => new PrecExtension(ext, value);
  }
  /**
  By default extensions are registered in the order they are found
  in the flattened form of nested array that was provided.
  Individual extension values can be assigned a precedence to
  override this. Extensions that do not have a precedence set get
  the precedence of the nearest parent with a precedence, or
  [`default`](https://codemirror.net/6/docs/ref/#state.Prec.default) if there is no such parent. The
  final ordering of extensions is determined by first sorting by
  precedence and then by order within each precedence.
  */
  const Prec = {
    /**
    The highest precedence level, for extensions that should end up
    near the start of the precedence ordering.
    */
    highest: /*@__PURE__*/prec(Prec_.highest),
    /**
    A higher-than-default precedence, for extensions that should
    come before those with default precedence.
    */
    high: /*@__PURE__*/prec(Prec_.high),
    /**
    The default precedence, which is also used for extensions
    without an explicit precedence.
    */
    default: /*@__PURE__*/prec(Prec_.default),
    /**
    A lower-than-default precedence.
    */
    low: /*@__PURE__*/prec(Prec_.low),
    /**
    The lowest precedence level. Meant for things that should end up
    near the end of the extension order.
    */
    lowest: /*@__PURE__*/prec(Prec_.lowest)
  };
  class PrecExtension {
    constructor(inner, prec) {
      this.inner = inner;
      this.prec = prec;
    }
  }
  /**
  Extension compartments can be used to make a configuration
  dynamic. By [wrapping](https://codemirror.net/6/docs/ref/#state.Compartment.of) part of your
  configuration in a compartment, you can later
  [replace](https://codemirror.net/6/docs/ref/#state.Compartment.reconfigure) that part through a
  transaction.
  */
  class Compartment {
    /**
    Create an instance of this compartment to add to your [state
    configuration](https://codemirror.net/6/docs/ref/#state.EditorStateConfig.extensions).
    */
    of(ext) {
      return new CompartmentInstance(this, ext);
    }
    /**
    Create an [effect](https://codemirror.net/6/docs/ref/#state.TransactionSpec.effects) that
    reconfigures this compartment.
    */
    reconfigure(content) {
      return Compartment.reconfigure.of({
        compartment: this,
        extension: content
      });
    }
    /**
    Get the current content of the compartment in the state, or
    `undefined` if it isn't present.
    */
    get(state) {
      return state.config.compartments.get(this);
    }
  }
  class CompartmentInstance {
    constructor(compartment, inner) {
      this.compartment = compartment;
      this.inner = inner;
    }
  }
  class Configuration {
    constructor(base, compartments, dynamicSlots, address, staticValues, facets) {
      this.base = base;
      this.compartments = compartments;
      this.dynamicSlots = dynamicSlots;
      this.address = address;
      this.staticValues = staticValues;
      this.facets = facets;
      this.statusTemplate = [];
      while (this.statusTemplate.length < dynamicSlots.length) this.statusTemplate.push(0 /* SlotStatus.Unresolved */);
    }
    staticFacet(facet) {
      let addr = this.address[facet.id];
      return addr == null ? facet.default : this.staticValues[addr >> 1];
    }
    static resolve(base, compartments, oldState) {
      let fields = [];
      let facets = Object.create(null);
      let newCompartments = new Map();
      for (let ext of flatten(base, compartments, newCompartments)) {
        if (ext instanceof StateField) fields.push(ext);else (facets[ext.facet.id] || (facets[ext.facet.id] = [])).push(ext);
      }
      let address = Object.create(null);
      let staticValues = [];
      let dynamicSlots = [];
      for (let field of fields) {
        address[field.id] = dynamicSlots.length << 1;
        dynamicSlots.push(a => field.slot(a));
      }
      let oldFacets = oldState === null || oldState === void 0 ? void 0 : oldState.config.facets;
      for (let id in facets) {
        let providers = facets[id],
          facet = providers[0].facet;
        let oldProviders = oldFacets && oldFacets[id] || [];
        if (providers.every(p => p.type == 0 /* Provider.Static */)) {
          address[facet.id] = staticValues.length << 1 | 1;
          if (sameArray$1(oldProviders, providers)) {
            staticValues.push(oldState.facet(facet));
          } else {
            let value = facet.combine(providers.map(p => p.value));
            staticValues.push(oldState && facet.compare(value, oldState.facet(facet)) ? oldState.facet(facet) : value);
          }
        } else {
          for (let p of providers) {
            if (p.type == 0 /* Provider.Static */) {
              address[p.id] = staticValues.length << 1 | 1;
              staticValues.push(p.value);
            } else {
              address[p.id] = dynamicSlots.length << 1;
              dynamicSlots.push(a => p.dynamicSlot(a));
            }
          }
          address[facet.id] = dynamicSlots.length << 1;
          dynamicSlots.push(a => dynamicFacetSlot(a, facet, providers));
        }
      }
      let dynamic = dynamicSlots.map(f => f(address));
      return new Configuration(base, newCompartments, dynamic, address, staticValues, facets);
    }
  }
  function flatten(extension, compartments, newCompartments) {
    let result = [[], [], [], [], []];
    let seen = new Map();
    function inner(ext, prec) {
      let known = seen.get(ext);
      if (known != null) {
        if (known <= prec) return;
        let found = result[known].indexOf(ext);
        if (found > -1) result[known].splice(found, 1);
        if (ext instanceof CompartmentInstance) newCompartments.delete(ext.compartment);
      }
      seen.set(ext, prec);
      if (Array.isArray(ext)) {
        for (let e of ext) inner(e, prec);
      } else if (ext instanceof CompartmentInstance) {
        if (newCompartments.has(ext.compartment)) throw new RangeError(`Duplicate use of compartment in extensions`);
        let content = compartments.get(ext.compartment) || ext.inner;
        newCompartments.set(ext.compartment, content);
        inner(content, prec);
      } else if (ext instanceof PrecExtension) {
        inner(ext.inner, ext.prec);
      } else if (ext instanceof StateField) {
        result[prec].push(ext);
        if (ext.provides) inner(ext.provides, prec);
      } else if (ext instanceof FacetProvider) {
        result[prec].push(ext);
        if (ext.facet.extensions) inner(ext.facet.extensions, Prec_.default);
      } else {
        let content = ext.extension;
        if (!content) throw new Error(`Unrecognized extension value in extension set (${ext}). This sometimes happens because multiple instances of @codemirror/state are loaded, breaking instanceof checks.`);
        inner(content, prec);
      }
    }
    inner(extension, Prec_.default);
    return result.reduce((a, b) => a.concat(b));
  }
  function ensureAddr(state, addr) {
    if (addr & 1) return 2 /* SlotStatus.Computed */;
    let idx = addr >> 1;
    let status = state.status[idx];
    if (status == 4 /* SlotStatus.Computing */) throw new Error("Cyclic dependency between fields and/or facets");
    if (status & 2 /* SlotStatus.Computed */) return status;
    state.status[idx] = 4 /* SlotStatus.Computing */;
    let changed = state.computeSlot(state, state.config.dynamicSlots[idx]);
    return state.status[idx] = 2 /* SlotStatus.Computed */ | changed;
  }
  function getAddr(state, addr) {
    return addr & 1 ? state.config.staticValues[addr >> 1] : state.values[addr >> 1];
  }
  const languageData = /*@__PURE__*/Facet.define();
  const allowMultipleSelections = /*@__PURE__*/Facet.define({
    combine: values => values.some(v => v),
    static: true
  });
  const lineSeparator = /*@__PURE__*/Facet.define({
    combine: values => values.length ? values[0] : undefined,
    static: true
  });
  const changeFilter = /*@__PURE__*/Facet.define();
  const transactionFilter = /*@__PURE__*/Facet.define();
  const transactionExtender = /*@__PURE__*/Facet.define();
  const readOnly = /*@__PURE__*/Facet.define({
    combine: values => values.length ? values[0] : false
  });

  /**
  Annotations are tagged values that are used to add metadata to
  transactions in an extensible way. They should be used to model
  things that effect the entire transaction (such as its [time
  stamp](https://codemirror.net/6/docs/ref/#state.Transaction^time) or information about its
  [origin](https://codemirror.net/6/docs/ref/#state.Transaction^userEvent)). For effects that happen
  _alongside_ the other changes made by the transaction, [state
  effects](https://codemirror.net/6/docs/ref/#state.StateEffect) are more appropriate.
  */
  class Annotation {
    /**
    @internal
    */
    constructor(
    /**
    The annotation type.
    */
    type,
    /**
    The value of this annotation.
    */
    value) {
      this.type = type;
      this.value = value;
    }
    /**
    Define a new type of annotation.
    */
    static define() {
      return new AnnotationType();
    }
  }
  /**
  Marker that identifies a type of [annotation](https://codemirror.net/6/docs/ref/#state.Annotation).
  */
  class AnnotationType {
    /**
    Create an instance of this annotation.
    */
    of(value) {
      return new Annotation(this, value);
    }
  }
  /**
  Representation of a type of state effect. Defined with
  [`StateEffect.define`](https://codemirror.net/6/docs/ref/#state.StateEffect^define).
  */
  class StateEffectType {
    /**
    @internal
    */
    constructor(
    // The `any` types in these function types are there to work
    // around TypeScript issue #37631, where the type guard on
    // `StateEffect.is` mysteriously stops working when these properly
    // have type `Value`.
    /**
    @internal
    */
    map) {
      this.map = map;
    }
    /**
    Create a [state effect](https://codemirror.net/6/docs/ref/#state.StateEffect) instance of this
    type.
    */
    of(value) {
      return new StateEffect(this, value);
    }
  }
  /**
  State effects can be used to represent additional effects
  associated with a [transaction](https://codemirror.net/6/docs/ref/#state.Transaction.effects). They
  are often useful to model changes to custom [state
  fields](https://codemirror.net/6/docs/ref/#state.StateField), when those changes aren't implicit in
  document or selection changes.
  */
  class StateEffect {
    /**
    @internal
    */
    constructor(
    /**
    @internal
    */
    type,
    /**
    The value of this effect.
    */
    value) {
      this.type = type;
      this.value = value;
    }
    /**
    Map this effect through a position mapping. Will return
    `undefined` when that ends up deleting the effect.
    */
    map(mapping) {
      let mapped = this.type.map(this.value, mapping);
      return mapped === undefined ? undefined : mapped == this.value ? this : new StateEffect(this.type, mapped);
    }
    /**
    Tells you whether this effect object is of a given
    [type](https://codemirror.net/6/docs/ref/#state.StateEffectType).
    */
    is(type) {
      return this.type == type;
    }
    /**
    Define a new effect type. The type parameter indicates the type
    of values that his effect holds. It should be a type that
    doesn't include `undefined`, since that is used in
    [mapping](https://codemirror.net/6/docs/ref/#state.StateEffect.map) to indicate that an effect is
    removed.
    */
    static define(spec = {}) {
      return new StateEffectType(spec.map || (v => v));
    }
    /**
    Map an array of effects through a change set.
    */
    static mapEffects(effects, mapping) {
      if (!effects.length) return effects;
      let result = [];
      for (let effect of effects) {
        let mapped = effect.map(mapping);
        if (mapped) result.push(mapped);
      }
      return result;
    }
  }
  /**
  This effect can be used to reconfigure the root extensions of
  the editor. Doing this will discard any extensions
  [appended](https://codemirror.net/6/docs/ref/#state.StateEffect^appendConfig), but does not reset
  the content of [reconfigured](https://codemirror.net/6/docs/ref/#state.Compartment.reconfigure)
  compartments.
  */
  StateEffect.reconfigure = /*@__PURE__*/StateEffect.define();
  /**
  Append extensions to the top-level configuration of the editor.
  */
  StateEffect.appendConfig = /*@__PURE__*/StateEffect.define();
  /**
  Changes to the editor state are grouped into transactions.
  Typically, a user action creates a single transaction, which may
  contain any number of document changes, may change the selection,
  or have other effects. Create a transaction by calling
  [`EditorState.update`](https://codemirror.net/6/docs/ref/#state.EditorState.update), or immediately
  dispatch one by calling
  [`EditorView.dispatch`](https://codemirror.net/6/docs/ref/#view.EditorView.dispatch).
  */
  class Transaction {
    constructor(
    /**
    The state from which the transaction starts.
    */
    startState,
    /**
    The document changes made by this transaction.
    */
    changes,
    /**
    The selection set by this transaction, or undefined if it
    doesn't explicitly set a selection.
    */
    selection,
    /**
    The effects added to the transaction.
    */
    effects,
    /**
    @internal
    */
    annotations,
    /**
    Whether the selection should be scrolled into view after this
    transaction is dispatched.
    */
    scrollIntoView) {
      this.startState = startState;
      this.changes = changes;
      this.selection = selection;
      this.effects = effects;
      this.annotations = annotations;
      this.scrollIntoView = scrollIntoView;
      /**
      @internal
      */
      this._doc = null;
      /**
      @internal
      */
      this._state = null;
      if (selection) checkSelection(selection, changes.newLength);
      if (!annotations.some(a => a.type == Transaction.time)) this.annotations = annotations.concat(Transaction.time.of(Date.now()));
    }
    /**
    @internal
    */
    static create(startState, changes, selection, effects, annotations, scrollIntoView) {
      return new Transaction(startState, changes, selection, effects, annotations, scrollIntoView);
    }
    /**
    The new document produced by the transaction. Contrary to
    [`.state`](https://codemirror.net/6/docs/ref/#state.Transaction.state)`.doc`, accessing this won't
    force the entire new state to be computed right away, so it is
    recommended that [transaction
    filters](https://codemirror.net/6/docs/ref/#state.EditorState^transactionFilter) use this getter
    when they need to look at the new document.
    */
    get newDoc() {
      return this._doc || (this._doc = this.changes.apply(this.startState.doc));
    }
    /**
    The new selection produced by the transaction. If
    [`this.selection`](https://codemirror.net/6/docs/ref/#state.Transaction.selection) is undefined,
    this will [map](https://codemirror.net/6/docs/ref/#state.EditorSelection.map) the start state's
    current selection through the changes made by the transaction.
    */
    get newSelection() {
      return this.selection || this.startState.selection.map(this.changes);
    }
    /**
    The new state created by the transaction. Computed on demand
    (but retained for subsequent access), so it is recommended not to
    access it in [transaction
    filters](https://codemirror.net/6/docs/ref/#state.EditorState^transactionFilter) when possible.
    */
    get state() {
      if (!this._state) this.startState.applyTransaction(this);
      return this._state;
    }
    /**
    Get the value of the given annotation type, if any.
    */
    annotation(type) {
      for (let ann of this.annotations) if (ann.type == type) return ann.value;
      return undefined;
    }
    /**
    Indicates whether the transaction changed the document.
    */
    get docChanged() {
      return !this.changes.empty;
    }
    /**
    Indicates whether this transaction reconfigures the state
    (through a [configuration compartment](https://codemirror.net/6/docs/ref/#state.Compartment) or
    with a top-level configuration
    [effect](https://codemirror.net/6/docs/ref/#state.StateEffect^reconfigure).
    */
    get reconfigured() {
      return this.startState.config != this.state.config;
    }
    /**
    Returns true if the transaction has a [user
    event](https://codemirror.net/6/docs/ref/#state.Transaction^userEvent) annotation that is equal to
    or more specific than `event`. For example, if the transaction
    has `"select.pointer"` as user event, `"select"` and
    `"select.pointer"` will match it.
    */
    isUserEvent(event) {
      let e = this.annotation(Transaction.userEvent);
      return !!(e && (e == event || e.length > event.length && e.slice(0, event.length) == event && e[event.length] == "."));
    }
  }
  /**
  Annotation used to store transaction timestamps. Automatically
  added to every transaction, holding `Date.now()`.
  */
  Transaction.time = /*@__PURE__*/Annotation.define();
  /**
  Annotation used to associate a transaction with a user interface
  event. Holds a string identifying the event, using a
  dot-separated format to support attaching more specific
  information. The events used by the core libraries are:

   - `"input"` when content is entered
     - `"input.type"` for typed input
       - `"input.type.compose"` for composition
     - `"input.paste"` for pasted input
     - `"input.drop"` when adding content with drag-and-drop
     - `"input.complete"` when autocompleting
   - `"delete"` when the user deletes content
     - `"delete.selection"` when deleting the selection
     - `"delete.forward"` when deleting forward from the selection
     - `"delete.backward"` when deleting backward from the selection
     - `"delete.cut"` when cutting to the clipboard
   - `"move"` when content is moved
     - `"move.drop"` when content is moved within the editor through drag-and-drop
   - `"select"` when explicitly changing the selection
     - `"select.pointer"` when selecting with a mouse or other pointing device
   - `"undo"` and `"redo"` for history actions

  Use [`isUserEvent`](https://codemirror.net/6/docs/ref/#state.Transaction.isUserEvent) to check
  whether the annotation matches a given event.
  */
  Transaction.userEvent = /*@__PURE__*/Annotation.define();
  /**
  Annotation indicating whether a transaction should be added to
  the undo history or not.
  */
  Transaction.addToHistory = /*@__PURE__*/Annotation.define();
  /**
  Annotation indicating (when present and true) that a transaction
  represents a change made by some other actor, not the user. This
  is used, for example, to tag other people's changes in
  collaborative editing.
  */
  Transaction.remote = /*@__PURE__*/Annotation.define();
  function joinRanges(a, b) {
    let result = [];
    for (let iA = 0, iB = 0;;) {
      let from, to;
      if (iA < a.length && (iB == b.length || b[iB] >= a[iA])) {
        from = a[iA++];
        to = a[iA++];
      } else if (iB < b.length) {
        from = b[iB++];
        to = b[iB++];
      } else return result;
      if (!result.length || result[result.length - 1] < from) result.push(from, to);else if (result[result.length - 1] < to) result[result.length - 1] = to;
    }
  }
  function mergeTransaction(a, b, sequential) {
    var _a;
    let mapForA, mapForB, changes;
    if (sequential) {
      mapForA = b.changes;
      mapForB = ChangeSet.empty(b.changes.length);
      changes = a.changes.compose(b.changes);
    } else {
      mapForA = b.changes.map(a.changes);
      mapForB = a.changes.mapDesc(b.changes, true);
      changes = a.changes.compose(mapForA);
    }
    return {
      changes,
      selection: b.selection ? b.selection.map(mapForB) : (_a = a.selection) === null || _a === void 0 ? void 0 : _a.map(mapForA),
      effects: StateEffect.mapEffects(a.effects, mapForA).concat(StateEffect.mapEffects(b.effects, mapForB)),
      annotations: a.annotations.length ? a.annotations.concat(b.annotations) : b.annotations,
      scrollIntoView: a.scrollIntoView || b.scrollIntoView
    };
  }
  function resolveTransactionInner(state, spec, docSize) {
    let sel = spec.selection,
      annotations = asArray(spec.annotations);
    if (spec.userEvent) annotations = annotations.concat(Transaction.userEvent.of(spec.userEvent));
    return {
      changes: spec.changes instanceof ChangeSet ? spec.changes : ChangeSet.of(spec.changes || [], docSize, state.facet(lineSeparator)),
      selection: sel && (sel instanceof EditorSelection ? sel : EditorSelection.single(sel.anchor, sel.head)),
      effects: asArray(spec.effects),
      annotations,
      scrollIntoView: !!spec.scrollIntoView
    };
  }
  function resolveTransaction(state, specs, filter) {
    let s = resolveTransactionInner(state, specs.length ? specs[0] : {}, state.doc.length);
    if (specs.length && specs[0].filter === false) filter = false;
    for (let i = 1; i < specs.length; i++) {
      if (specs[i].filter === false) filter = false;
      let seq = !!specs[i].sequential;
      s = mergeTransaction(s, resolveTransactionInner(state, specs[i], seq ? s.changes.newLength : state.doc.length), seq);
    }
    let tr = Transaction.create(state, s.changes, s.selection, s.effects, s.annotations, s.scrollIntoView);
    return extendTransaction(filter ? filterTransaction(tr) : tr);
  }
  // Finish a transaction by applying filters if necessary.
  function filterTransaction(tr) {
    let state = tr.startState;
    // Change filters
    let result = true;
    for (let filter of state.facet(changeFilter)) {
      let value = filter(tr);
      if (value === false) {
        result = false;
        break;
      }
      if (Array.isArray(value)) result = result === true ? value : joinRanges(result, value);
    }
    if (result !== true) {
      let changes, back;
      if (result === false) {
        back = tr.changes.invertedDesc;
        changes = ChangeSet.empty(state.doc.length);
      } else {
        let filtered = tr.changes.filter(result);
        changes = filtered.changes;
        back = filtered.filtered.mapDesc(filtered.changes).invertedDesc;
      }
      tr = Transaction.create(state, changes, tr.selection && tr.selection.map(back), StateEffect.mapEffects(tr.effects, back), tr.annotations, tr.scrollIntoView);
    }
    // Transaction filters
    let filters = state.facet(transactionFilter);
    for (let i = filters.length - 1; i >= 0; i--) {
      let filtered = filters[i](tr);
      if (filtered instanceof Transaction) tr = filtered;else if (Array.isArray(filtered) && filtered.length == 1 && filtered[0] instanceof Transaction) tr = filtered[0];else tr = resolveTransaction(state, asArray(filtered), false);
    }
    return tr;
  }
  function extendTransaction(tr) {
    let state = tr.startState,
      extenders = state.facet(transactionExtender),
      spec = tr;
    for (let i = extenders.length - 1; i >= 0; i--) {
      let extension = extenders[i](tr);
      if (extension && Object.keys(extension).length) spec = mergeTransaction(spec, resolveTransactionInner(state, extension, tr.changes.newLength), true);
    }
    return spec == tr ? tr : Transaction.create(state, tr.changes, tr.selection, spec.effects, spec.annotations, spec.scrollIntoView);
  }
  const none = [];
  function asArray(value) {
    return value == null ? none : Array.isArray(value) ? value : [value];
  }

  /**
  The categories produced by a [character
  categorizer](https://codemirror.net/6/docs/ref/#state.EditorState.charCategorizer). These are used
  do things like selecting by word.
  */
  var CharCategory = /*@__PURE__*/function (CharCategory) {
    /**
    Word characters.
    */
    CharCategory[CharCategory["Word"] = 0] = "Word";
    /**
    Whitespace.
    */
    CharCategory[CharCategory["Space"] = 1] = "Space";
    /**
    Anything else.
    */
    CharCategory[CharCategory["Other"] = 2] = "Other";
    return CharCategory;
  }(CharCategory || (CharCategory = {}));
  const nonASCIISingleCaseWordChar = /[\u00df\u0587\u0590-\u05f4\u0600-\u06ff\u3040-\u309f\u30a0-\u30ff\u3400-\u4db5\u4e00-\u9fcc\uac00-\ud7af]/;
  let wordChar;
  try {
    wordChar = /*@__PURE__*/new RegExp("[\\p{Alphabetic}\\p{Number}_]", "u");
  } catch (_) {}
  function hasWordChar(str) {
    if (wordChar) return wordChar.test(str);
    for (let i = 0; i < str.length; i++) {
      let ch = str[i];
      if (/\w/.test(ch) || ch > "\x80" && (ch.toUpperCase() != ch.toLowerCase() || nonASCIISingleCaseWordChar.test(ch))) return true;
    }
    return false;
  }
  function makeCategorizer(wordChars) {
    return char => {
      if (!/\S/.test(char)) return CharCategory.Space;
      if (hasWordChar(char)) return CharCategory.Word;
      for (let i = 0; i < wordChars.length; i++) if (char.indexOf(wordChars[i]) > -1) return CharCategory.Word;
      return CharCategory.Other;
    };
  }

  /**
  The editor state class is a persistent (immutable) data structure.
  To update a state, you [create](https://codemirror.net/6/docs/ref/#state.EditorState.update) a
  [transaction](https://codemirror.net/6/docs/ref/#state.Transaction), which produces a _new_ state
  instance, without modifying the original object.

  As such, _never_ mutate properties of a state directly. That'll
  just break things.
  */
  class EditorState {
    constructor(
    /**
    @internal
    */
    config,
    /**
    The current document.
    */
    doc,
    /**
    The current selection.
    */
    selection,
    /**
    @internal
    */
    values, computeSlot, tr) {
      this.config = config;
      this.doc = doc;
      this.selection = selection;
      this.values = values;
      this.status = config.statusTemplate.slice();
      this.computeSlot = computeSlot;
      // Fill in the computed state immediately, so that further queries
      // for it made during the update return this state
      if (tr) tr._state = this;
      for (let i = 0; i < this.config.dynamicSlots.length; i++) ensureAddr(this, i << 1);
      this.computeSlot = null;
    }
    field(field, require = true) {
      let addr = this.config.address[field.id];
      if (addr == null) {
        if (require) throw new RangeError("Field is not present in this state");
        return undefined;
      }
      ensureAddr(this, addr);
      return getAddr(this, addr);
    }
    /**
    Create a [transaction](https://codemirror.net/6/docs/ref/#state.Transaction) that updates this
    state. Any number of [transaction specs](https://codemirror.net/6/docs/ref/#state.TransactionSpec)
    can be passed. Unless
    [`sequential`](https://codemirror.net/6/docs/ref/#state.TransactionSpec.sequential) is set, the
    [changes](https://codemirror.net/6/docs/ref/#state.TransactionSpec.changes) (if any) of each spec
    are assumed to start in the _current_ document (not the document
    produced by previous specs), and its
    [selection](https://codemirror.net/6/docs/ref/#state.TransactionSpec.selection) and
    [effects](https://codemirror.net/6/docs/ref/#state.TransactionSpec.effects) are assumed to refer
    to the document created by its _own_ changes. The resulting
    transaction contains the combined effect of all the different
    specs. For [selection](https://codemirror.net/6/docs/ref/#state.TransactionSpec.selection), later
    specs take precedence over earlier ones.
    */
    update(...specs) {
      return resolveTransaction(this, specs, true);
    }
    /**
    @internal
    */
    applyTransaction(tr) {
      let conf = this.config,
        {
          base,
          compartments
        } = conf;
      for (let effect of tr.effects) {
        if (effect.is(Compartment.reconfigure)) {
          if (conf) {
            compartments = new Map();
            conf.compartments.forEach((val, key) => compartments.set(key, val));
            conf = null;
          }
          compartments.set(effect.value.compartment, effect.value.extension);
        } else if (effect.is(StateEffect.reconfigure)) {
          conf = null;
          base = effect.value;
        } else if (effect.is(StateEffect.appendConfig)) {
          conf = null;
          base = asArray(base).concat(effect.value);
        }
      }
      let startValues;
      if (!conf) {
        conf = Configuration.resolve(base, compartments, this);
        let intermediateState = new EditorState(conf, this.doc, this.selection, conf.dynamicSlots.map(() => null), (state, slot) => slot.reconfigure(state, this), null);
        startValues = intermediateState.values;
      } else {
        startValues = tr.startState.values.slice();
      }
      let selection = tr.startState.facet(allowMultipleSelections) ? tr.newSelection : tr.newSelection.asSingle();
      new EditorState(conf, tr.newDoc, selection, startValues, (state, slot) => slot.update(state, tr), tr);
    }
    /**
    Create a [transaction spec](https://codemirror.net/6/docs/ref/#state.TransactionSpec) that
    replaces every selection range with the given content.
    */
    replaceSelection(text) {
      if (typeof text == "string") text = this.toText(text);
      return this.changeByRange(range => ({
        changes: {
          from: range.from,
          to: range.to,
          insert: text
        },
        range: EditorSelection.cursor(range.from + text.length)
      }));
    }
    /**
    Create a set of changes and a new selection by running the given
    function for each range in the active selection. The function
    can return an optional set of changes (in the coordinate space
    of the start document), plus an updated range (in the coordinate
    space of the document produced by the call's own changes). This
    method will merge all the changes and ranges into a single
    changeset and selection, and return it as a [transaction
    spec](https://codemirror.net/6/docs/ref/#state.TransactionSpec), which can be passed to
    [`update`](https://codemirror.net/6/docs/ref/#state.EditorState.update).
    */
    changeByRange(f) {
      let sel = this.selection;
      let result1 = f(sel.ranges[0]);
      let changes = this.changes(result1.changes),
        ranges = [result1.range];
      let effects = asArray(result1.effects);
      for (let i = 1; i < sel.ranges.length; i++) {
        let result = f(sel.ranges[i]);
        let newChanges = this.changes(result.changes),
          newMapped = newChanges.map(changes);
        for (let j = 0; j < i; j++) ranges[j] = ranges[j].map(newMapped);
        let mapBy = changes.mapDesc(newChanges, true);
        ranges.push(result.range.map(mapBy));
        changes = changes.compose(newMapped);
        effects = StateEffect.mapEffects(effects, newMapped).concat(StateEffect.mapEffects(asArray(result.effects), mapBy));
      }
      return {
        changes,
        selection: EditorSelection.create(ranges, sel.mainIndex),
        effects
      };
    }
    /**
    Create a [change set](https://codemirror.net/6/docs/ref/#state.ChangeSet) from the given change
    description, taking the state's document length and line
    separator into account.
    */
    changes(spec = []) {
      if (spec instanceof ChangeSet) return spec;
      return ChangeSet.of(spec, this.doc.length, this.facet(EditorState.lineSeparator));
    }
    /**
    Using the state's [line
    separator](https://codemirror.net/6/docs/ref/#state.EditorState^lineSeparator), create a
    [`Text`](https://codemirror.net/6/docs/ref/#state.Text) instance from the given string.
    */
    toText(string) {
      return Text.of(string.split(this.facet(EditorState.lineSeparator) || DefaultSplit));
    }
    /**
    Return the given range of the document as a string.
    */
    sliceDoc(from = 0, to = this.doc.length) {
      return this.doc.sliceString(from, to, this.lineBreak);
    }
    /**
    Get the value of a state [facet](https://codemirror.net/6/docs/ref/#state.Facet).
    */
    facet(facet) {
      let addr = this.config.address[facet.id];
      if (addr == null) return facet.default;
      ensureAddr(this, addr);
      return getAddr(this, addr);
    }
    /**
    Convert this state to a JSON-serializable object. When custom
    fields should be serialized, you can pass them in as an object
    mapping property names (in the resulting object, which should
    not use `doc` or `selection`) to fields.
    */
    toJSON(fields) {
      let result = {
        doc: this.sliceDoc(),
        selection: this.selection.toJSON()
      };
      if (fields) for (let prop in fields) {
        let value = fields[prop];
        if (value instanceof StateField && this.config.address[value.id] != null) result[prop] = value.spec.toJSON(this.field(fields[prop]), this);
      }
      return result;
    }
    /**
    Deserialize a state from its JSON representation. When custom
    fields should be deserialized, pass the same object you passed
    to [`toJSON`](https://codemirror.net/6/docs/ref/#state.EditorState.toJSON) when serializing as
    third argument.
    */
    static fromJSON(json, config = {}, fields) {
      if (!json || typeof json.doc != "string") throw new RangeError("Invalid JSON representation for EditorState");
      let fieldInit = [];
      if (fields) for (let prop in fields) {
        if (Object.prototype.hasOwnProperty.call(json, prop)) {
          let field = fields[prop],
            value = json[prop];
          fieldInit.push(field.init(state => field.spec.fromJSON(value, state)));
        }
      }
      return EditorState.create({
        doc: json.doc,
        selection: EditorSelection.fromJSON(json.selection),
        extensions: config.extensions ? fieldInit.concat([config.extensions]) : fieldInit
      });
    }
    /**
    Create a new state. You'll usually only need this when
    initializing an editor—updated states are created by applying
    transactions.
    */
    static create(config = {}) {
      let configuration = Configuration.resolve(config.extensions || [], new Map());
      let doc = config.doc instanceof Text ? config.doc : Text.of((config.doc || "").split(configuration.staticFacet(EditorState.lineSeparator) || DefaultSplit));
      let selection = !config.selection ? EditorSelection.single(0) : config.selection instanceof EditorSelection ? config.selection : EditorSelection.single(config.selection.anchor, config.selection.head);
      checkSelection(selection, doc.length);
      if (!configuration.staticFacet(allowMultipleSelections)) selection = selection.asSingle();
      return new EditorState(configuration, doc, selection, configuration.dynamicSlots.map(() => null), (state, slot) => slot.create(state), null);
    }
    /**
    The size (in columns) of a tab in the document, determined by
    the [`tabSize`](https://codemirror.net/6/docs/ref/#state.EditorState^tabSize) facet.
    */
    get tabSize() {
      return this.facet(EditorState.tabSize);
    }
    /**
    Get the proper [line-break](https://codemirror.net/6/docs/ref/#state.EditorState^lineSeparator)
    string for this state.
    */
    get lineBreak() {
      return this.facet(EditorState.lineSeparator) || "\n";
    }
    /**
    Returns true when the editor is
    [configured](https://codemirror.net/6/docs/ref/#state.EditorState^readOnly) to be read-only.
    */
    get readOnly() {
      return this.facet(readOnly);
    }
    /**
    Look up a translation for the given phrase (via the
    [`phrases`](https://codemirror.net/6/docs/ref/#state.EditorState^phrases) facet), or return the
    original string if no translation is found.
    
    If additional arguments are passed, they will be inserted in
    place of markers like `$1` (for the first value) and `$2`, etc.
    A single `$` is equivalent to `$1`, and `$$` will produce a
    literal dollar sign.
    */
    phrase(phrase, ...insert) {
      for (let map of this.facet(EditorState.phrases)) if (Object.prototype.hasOwnProperty.call(map, phrase)) {
        phrase = map[phrase];
        break;
      }
      if (insert.length) phrase = phrase.replace(/\$(\$|\d*)/g, (m, i) => {
        if (i == "$") return "$";
        let n = +(i || 1);
        return !n || n > insert.length ? m : insert[n - 1];
      });
      return phrase;
    }
    /**
    Find the values for a given language data field, provided by the
    the [`languageData`](https://codemirror.net/6/docs/ref/#state.EditorState^languageData) facet.
    
    Examples of language data fields are...
    
    - [`"commentTokens"`](https://codemirror.net/6/docs/ref/#commands.CommentTokens) for specifying
      comment syntax.
    - [`"autocomplete"`](https://codemirror.net/6/docs/ref/#autocomplete.autocompletion^config.override)
      for providing language-specific completion sources.
    - [`"wordChars"`](https://codemirror.net/6/docs/ref/#state.EditorState.charCategorizer) for adding
      characters that should be considered part of words in this
      language.
    - [`"closeBrackets"`](https://codemirror.net/6/docs/ref/#autocomplete.CloseBracketConfig) controls
      bracket closing behavior.
    */
    languageDataAt(name, pos, side = -1) {
      let values = [];
      for (let provider of this.facet(languageData)) {
        for (let result of provider(this, pos, side)) {
          if (Object.prototype.hasOwnProperty.call(result, name)) values.push(result[name]);
        }
      }
      return values;
    }
    /**
    Return a function that can categorize strings (expected to
    represent a single [grapheme cluster](https://codemirror.net/6/docs/ref/#state.findClusterBreak))
    into one of:
    
     - Word (contains an alphanumeric character or a character
       explicitly listed in the local language's `"wordChars"`
       language data, which should be a string)
     - Space (contains only whitespace)
     - Other (anything else)
    */
    charCategorizer(at) {
      return makeCategorizer(this.languageDataAt("wordChars", at).join(""));
    }
    /**
    Find the word at the given position, meaning the range
    containing all [word](https://codemirror.net/6/docs/ref/#state.CharCategory.Word) characters
    around it. If no word characters are adjacent to the position,
    this returns null.
    */
    wordAt(pos) {
      let {
        text,
        from,
        length
      } = this.doc.lineAt(pos);
      let cat = this.charCategorizer(pos);
      let start = pos - from,
        end = pos - from;
      while (start > 0) {
        let prev = findClusterBreak(text, start, false);
        if (cat(text.slice(prev, start)) != CharCategory.Word) break;
        start = prev;
      }
      while (end < length) {
        let next = findClusterBreak(text, end);
        if (cat(text.slice(end, next)) != CharCategory.Word) break;
        end = next;
      }
      return start == end ? null : EditorSelection.range(start + from, end + from);
    }
  }
  /**
  A facet that, when enabled, causes the editor to allow multiple
  ranges to be selected. Be careful though, because by default the
  editor relies on the native DOM selection, which cannot handle
  multiple selections. An extension like
  [`drawSelection`](https://codemirror.net/6/docs/ref/#view.drawSelection) can be used to make
  secondary selections visible to the user.
  */
  EditorState.allowMultipleSelections = allowMultipleSelections;
  /**
  Configures the tab size to use in this state. The first
  (highest-precedence) value of the facet is used. If no value is
  given, this defaults to 4.
  */
  EditorState.tabSize = /*@__PURE__*/Facet.define({
    combine: values => values.length ? values[0] : 4
  });
  /**
  The line separator to use. By default, any of `"\n"`, `"\r\n"`
  and `"\r"` is treated as a separator when splitting lines, and
  lines are joined with `"\n"`.

  When you configure a value here, only that precise separator
  will be used, allowing you to round-trip documents through the
  editor without normalizing line separators.
  */
  EditorState.lineSeparator = lineSeparator;
  /**
  This facet controls the value of the
  [`readOnly`](https://codemirror.net/6/docs/ref/#state.EditorState.readOnly) getter, which is
  consulted by commands and extensions that implement editing
  functionality to determine whether they should apply. It
  defaults to false, but when its highest-precedence value is
  `true`, such functionality disables itself.

  Not to be confused with
  [`EditorView.editable`](https://codemirror.net/6/docs/ref/#view.EditorView^editable), which
  controls whether the editor's DOM is set to be editable (and
  thus focusable).
  */
  EditorState.readOnly = readOnly;
  /**
  Registers translation phrases. The
  [`phrase`](https://codemirror.net/6/docs/ref/#state.EditorState.phrase) method will look through
  all objects registered with this facet to find translations for
  its argument.
  */
  EditorState.phrases = /*@__PURE__*/Facet.define({
    compare(a, b) {
      let kA = Object.keys(a),
        kB = Object.keys(b);
      return kA.length == kB.length && kA.every(k => a[k] == b[k]);
    }
  });
  /**
  A facet used to register [language
  data](https://codemirror.net/6/docs/ref/#state.EditorState.languageDataAt) providers.
  */
  EditorState.languageData = languageData;
  /**
  Facet used to register change filters, which are called for each
  transaction (unless explicitly
  [disabled](https://codemirror.net/6/docs/ref/#state.TransactionSpec.filter)), and can suppress
  part of the transaction's changes.

  Such a function can return `true` to indicate that it doesn't
  want to do anything, `false` to completely stop the changes in
  the transaction, or a set of ranges in which changes should be
  suppressed. Such ranges are represented as an array of numbers,
  with each pair of two numbers indicating the start and end of a
  range. So for example `[10, 20, 100, 110]` suppresses changes
  between 10 and 20, and between 100 and 110.
  */
  EditorState.changeFilter = changeFilter;
  /**
  Facet used to register a hook that gets a chance to update or
  replace transaction specs before they are applied. This will
  only be applied for transactions that don't have
  [`filter`](https://codemirror.net/6/docs/ref/#state.TransactionSpec.filter) set to `false`. You
  can either return a single transaction spec (possibly the input
  transaction), or an array of specs (which will be combined in
  the same way as the arguments to
  [`EditorState.update`](https://codemirror.net/6/docs/ref/#state.EditorState.update)).

  When possible, it is recommended to avoid accessing
  [`Transaction.state`](https://codemirror.net/6/docs/ref/#state.Transaction.state) in a filter,
  since it will force creation of a state that will then be
  discarded again, if the transaction is actually filtered.

  (This functionality should be used with care. Indiscriminately
  modifying transaction is likely to break something or degrade
  the user experience.)
  */
  EditorState.transactionFilter = transactionFilter;
  /**
  This is a more limited form of
  [`transactionFilter`](https://codemirror.net/6/docs/ref/#state.EditorState^transactionFilter),
  which can only add
  [annotations](https://codemirror.net/6/docs/ref/#state.TransactionSpec.annotations) and
  [effects](https://codemirror.net/6/docs/ref/#state.TransactionSpec.effects). _But_, this type
  of filter runs even if the transaction has disabled regular
  [filtering](https://codemirror.net/6/docs/ref/#state.TransactionSpec.filter), making it suitable
  for effects that don't need to touch the changes or selection,
  but do want to process every transaction.

  Extenders run _after_ filters, when both are present.
  */
  EditorState.transactionExtender = transactionExtender;
  Compartment.reconfigure = /*@__PURE__*/StateEffect.define();

  /**
  Utility function for combining behaviors to fill in a config
  object from an array of provided configs. `defaults` should hold
  default values for all optional fields in `Config`.

  The function will, by default, error
  when a field gets two values that aren't `===`-equal, but you can
  provide combine functions per field to do something else.
  */
  function combineConfig(configs, defaults,
  // Should hold only the optional properties of Config, but I haven't managed to express that
  combine = {}) {
    let result = {};
    for (let config of configs) for (let key of Object.keys(config)) {
      let value = config[key],
        current = result[key];
      if (current === undefined) result[key] = value;else if (current === value || value === undefined) ; // No conflict
      else if (Object.hasOwnProperty.call(combine, key)) result[key] = combine[key](current, value);else throw new Error("Config merge conflict for field " + key);
    }
    for (let key in defaults) if (result[key] === undefined) result[key] = defaults[key];
    return result;
  }

  /**
  Each range is associated with a value, which must inherit from
  this class.
  */
  class RangeValue {
    /**
    Compare this value with another value. Used when comparing
    rangesets. The default implementation compares by identity.
    Unless you are only creating a fixed number of unique instances
    of your value type, it is a good idea to implement this
    properly.
    */
    eq(other) {
      return this == other;
    }
    /**
    Create a [range](https://codemirror.net/6/docs/ref/#state.Range) with this value.
    */
    range(from, to = from) {
      return Range$1.create(from, to, this);
    }
  }
  RangeValue.prototype.startSide = RangeValue.prototype.endSide = 0;
  RangeValue.prototype.point = false;
  RangeValue.prototype.mapMode = MapMode.TrackDel;
  /**
  A range associates a value with a range of positions.
  */
  let Range$1 = class Range {
    constructor(
    /**
    The range's start position.
    */
    from,
    /**
    Its end position.
    */
    to,
    /**
    The value associated with this range.
    */
    value) {
      this.from = from;
      this.to = to;
      this.value = value;
    }
    /**
    @internal
    */
    static create(from, to, value) {
      return new Range(from, to, value);
    }
  };
  function cmpRange(a, b) {
    return a.from - b.from || a.value.startSide - b.value.startSide;
  }
  class Chunk {
    constructor(from, to, value,
    // Chunks are marked with the largest point that occurs
    // in them (or -1 for no points), so that scans that are
    // only interested in points (such as the
    // heightmap-related logic) can skip range-only chunks.
    maxPoint) {
      this.from = from;
      this.to = to;
      this.value = value;
      this.maxPoint = maxPoint;
    }
    get length() {
      return this.to[this.to.length - 1];
    }
    // Find the index of the given position and side. Use the ranges'
    // `from` pos when `end == false`, `to` when `end == true`.
    findIndex(pos, side, end, startAt = 0) {
      let arr = end ? this.to : this.from;
      for (let lo = startAt, hi = arr.length;;) {
        if (lo == hi) return lo;
        let mid = lo + hi >> 1;
        let diff = arr[mid] - pos || (end ? this.value[mid].endSide : this.value[mid].startSide) - side;
        if (mid == lo) return diff >= 0 ? lo : hi;
        if (diff >= 0) hi = mid;else lo = mid + 1;
      }
    }
    between(offset, from, to, f) {
      for (let i = this.findIndex(from, -1000000000 /* C.Far */, true), e = this.findIndex(to, 1000000000 /* C.Far */, false, i); i < e; i++) if (f(this.from[i] + offset, this.to[i] + offset, this.value[i]) === false) return false;
    }
    map(offset, changes) {
      let value = [],
        from = [],
        to = [],
        newPos = -1,
        maxPoint = -1;
      for (let i = 0; i < this.value.length; i++) {
        let val = this.value[i],
          curFrom = this.from[i] + offset,
          curTo = this.to[i] + offset,
          newFrom,
          newTo;
        if (curFrom == curTo) {
          let mapped = changes.mapPos(curFrom, val.startSide, val.mapMode);
          if (mapped == null) continue;
          newFrom = newTo = mapped;
          if (val.startSide != val.endSide) {
            newTo = changes.mapPos(curFrom, val.endSide);
            if (newTo < newFrom) continue;
          }
        } else {
          newFrom = changes.mapPos(curFrom, val.startSide);
          newTo = changes.mapPos(curTo, val.endSide);
          if (newFrom > newTo || newFrom == newTo && val.startSide > 0 && val.endSide <= 0) continue;
        }
        if ((newTo - newFrom || val.endSide - val.startSide) < 0) continue;
        if (newPos < 0) newPos = newFrom;
        if (val.point) maxPoint = Math.max(maxPoint, newTo - newFrom);
        value.push(val);
        from.push(newFrom - newPos);
        to.push(newTo - newPos);
      }
      return {
        mapped: value.length ? new Chunk(from, to, value, maxPoint) : null,
        pos: newPos
      };
    }
  }
  /**
  A range set stores a collection of [ranges](https://codemirror.net/6/docs/ref/#state.Range) in a
  way that makes them efficient to [map](https://codemirror.net/6/docs/ref/#state.RangeSet.map) and
  [update](https://codemirror.net/6/docs/ref/#state.RangeSet.update). This is an immutable data
  structure.
  */
  class RangeSet {
    constructor(
    /**
    @internal
    */
    chunkPos,
    /**
    @internal
    */
    chunk,
    /**
    @internal
    */
    nextLayer,
    /**
    @internal
    */
    maxPoint) {
      this.chunkPos = chunkPos;
      this.chunk = chunk;
      this.nextLayer = nextLayer;
      this.maxPoint = maxPoint;
    }
    /**
    @internal
    */
    static create(chunkPos, chunk, nextLayer, maxPoint) {
      return new RangeSet(chunkPos, chunk, nextLayer, maxPoint);
    }
    /**
    @internal
    */
    get length() {
      let last = this.chunk.length - 1;
      return last < 0 ? 0 : Math.max(this.chunkEnd(last), this.nextLayer.length);
    }
    /**
    The number of ranges in the set.
    */
    get size() {
      if (this.isEmpty) return 0;
      let size = this.nextLayer.size;
      for (let chunk of this.chunk) size += chunk.value.length;
      return size;
    }
    /**
    @internal
    */
    chunkEnd(index) {
      return this.chunkPos[index] + this.chunk[index].length;
    }
    /**
    Update the range set, optionally adding new ranges or filtering
    out existing ones.
    
    (Note: The type parameter is just there as a kludge to work
    around TypeScript variance issues that prevented `RangeSet<X>`
    from being a subtype of `RangeSet<Y>` when `X` is a subtype of
    `Y`.)
    */
    update(updateSpec) {
      let {
        add = [],
        sort = false,
        filterFrom = 0,
        filterTo = this.length
      } = updateSpec;
      let filter = updateSpec.filter;
      if (add.length == 0 && !filter) return this;
      if (sort) add = add.slice().sort(cmpRange);
      if (this.isEmpty) return add.length ? RangeSet.of(add) : this;
      let cur = new LayerCursor(this, null, -1).goto(0),
        i = 0,
        spill = [];
      let builder = new RangeSetBuilder();
      while (cur.value || i < add.length) {
        if (i < add.length && (cur.from - add[i].from || cur.startSide - add[i].value.startSide) >= 0) {
          let range = add[i++];
          if (!builder.addInner(range.from, range.to, range.value)) spill.push(range);
        } else if (cur.rangeIndex == 1 && cur.chunkIndex < this.chunk.length && (i == add.length || this.chunkEnd(cur.chunkIndex) < add[i].from) && (!filter || filterFrom > this.chunkEnd(cur.chunkIndex) || filterTo < this.chunkPos[cur.chunkIndex]) && builder.addChunk(this.chunkPos[cur.chunkIndex], this.chunk[cur.chunkIndex])) {
          cur.nextChunk();
        } else {
          if (!filter || filterFrom > cur.to || filterTo < cur.from || filter(cur.from, cur.to, cur.value)) {
            if (!builder.addInner(cur.from, cur.to, cur.value)) spill.push(Range$1.create(cur.from, cur.to, cur.value));
          }
          cur.next();
        }
      }
      return builder.finishInner(this.nextLayer.isEmpty && !spill.length ? RangeSet.empty : this.nextLayer.update({
        add: spill,
        filter,
        filterFrom,
        filterTo
      }));
    }
    /**
    Map this range set through a set of changes, return the new set.
    */
    map(changes) {
      if (changes.empty || this.isEmpty) return this;
      let chunks = [],
        chunkPos = [],
        maxPoint = -1;
      for (let i = 0; i < this.chunk.length; i++) {
        let start = this.chunkPos[i],
          chunk = this.chunk[i];
        let touch = changes.touchesRange(start, start + chunk.length);
        if (touch === false) {
          maxPoint = Math.max(maxPoint, chunk.maxPoint);
          chunks.push(chunk);
          chunkPos.push(changes.mapPos(start));
        } else if (touch === true) {
          let {
            mapped,
            pos
          } = chunk.map(start, changes);
          if (mapped) {
            maxPoint = Math.max(maxPoint, mapped.maxPoint);
            chunks.push(mapped);
            chunkPos.push(pos);
          }
        }
      }
      let next = this.nextLayer.map(changes);
      return chunks.length == 0 ? next : new RangeSet(chunkPos, chunks, next || RangeSet.empty, maxPoint);
    }
    /**
    Iterate over the ranges that touch the region `from` to `to`,
    calling `f` for each. There is no guarantee that the ranges will
    be reported in any specific order. When the callback returns
    `false`, iteration stops.
    */
    between(from, to, f) {
      if (this.isEmpty) return;
      for (let i = 0; i < this.chunk.length; i++) {
        let start = this.chunkPos[i],
          chunk = this.chunk[i];
        if (to >= start && from <= start + chunk.length && chunk.between(start, from - start, to - start, f) === false) return;
      }
      this.nextLayer.between(from, to, f);
    }
    /**
    Iterate over the ranges in this set, in order, including all
    ranges that end at or after `from`.
    */
    iter(from = 0) {
      return HeapCursor.from([this]).goto(from);
    }
    /**
    @internal
    */
    get isEmpty() {
      return this.nextLayer == this;
    }
    /**
    Iterate over the ranges in a collection of sets, in order,
    starting from `from`.
    */
    static iter(sets, from = 0) {
      return HeapCursor.from(sets).goto(from);
    }
    /**
    Iterate over two groups of sets, calling methods on `comparator`
    to notify it of possible differences.
    */
    static compare(oldSets, newSets,
    /**
    This indicates how the underlying data changed between these
    ranges, and is needed to synchronize the iteration.
    */
    textDiff, comparator,
    /**
    Can be used to ignore all non-point ranges, and points below
    the given size. When -1, all ranges are compared.
    */
    minPointSize = -1) {
      let a = oldSets.filter(set => set.maxPoint > 0 || !set.isEmpty && set.maxPoint >= minPointSize);
      let b = newSets.filter(set => set.maxPoint > 0 || !set.isEmpty && set.maxPoint >= minPointSize);
      let sharedChunks = findSharedChunks(a, b, textDiff);
      let sideA = new SpanCursor(a, sharedChunks, minPointSize);
      let sideB = new SpanCursor(b, sharedChunks, minPointSize);
      textDiff.iterGaps((fromA, fromB, length) => compare(sideA, fromA, sideB, fromB, length, comparator));
      if (textDiff.empty && textDiff.length == 0) compare(sideA, 0, sideB, 0, 0, comparator);
    }
    /**
    Compare the contents of two groups of range sets, returning true
    if they are equivalent in the given range.
    */
    static eq(oldSets, newSets, from = 0, to) {
      if (to == null) to = 1000000000 /* C.Far */ - 1;
      let a = oldSets.filter(set => !set.isEmpty && newSets.indexOf(set) < 0);
      let b = newSets.filter(set => !set.isEmpty && oldSets.indexOf(set) < 0);
      if (a.length != b.length) return false;
      if (!a.length) return true;
      let sharedChunks = findSharedChunks(a, b);
      let sideA = new SpanCursor(a, sharedChunks, 0).goto(from),
        sideB = new SpanCursor(b, sharedChunks, 0).goto(from);
      for (;;) {
        if (sideA.to != sideB.to || !sameValues(sideA.active, sideB.active) || sideA.point && (!sideB.point || !sideA.point.eq(sideB.point))) return false;
        if (sideA.to > to) return true;
        sideA.next();
        sideB.next();
      }
    }
    /**
    Iterate over a group of range sets at the same time, notifying
    the iterator about the ranges covering every given piece of
    content. Returns the open count (see
    [`SpanIterator.span`](https://codemirror.net/6/docs/ref/#state.SpanIterator.span)) at the end
    of the iteration.
    */
    static spans(sets, from, to, iterator,
    /**
    When given and greater than -1, only points of at least this
    size are taken into account.
    */
    minPointSize = -1) {
      let cursor = new SpanCursor(sets, null, minPointSize).goto(from),
        pos = from;
      let openRanges = cursor.openStart;
      for (;;) {
        let curTo = Math.min(cursor.to, to);
        if (cursor.point) {
          let active = cursor.activeForPoint(cursor.to);
          let openCount = cursor.pointFrom < from ? active.length + 1 : cursor.point.startSide < 0 ? active.length : Math.min(active.length, openRanges);
          iterator.point(pos, curTo, cursor.point, active, openCount, cursor.pointRank);
          openRanges = Math.min(cursor.openEnd(curTo), active.length);
        } else if (curTo > pos) {
          iterator.span(pos, curTo, cursor.active, openRanges);
          openRanges = cursor.openEnd(curTo);
        }
        if (cursor.to > to) return openRanges + (cursor.point && cursor.to > to ? 1 : 0);
        pos = cursor.to;
        cursor.next();
      }
    }
    /**
    Create a range set for the given range or array of ranges. By
    default, this expects the ranges to be _sorted_ (by start
    position and, if two start at the same position,
    `value.startSide`). You can pass `true` as second argument to
    cause the method to sort them.
    */
    static of(ranges, sort = false) {
      let build = new RangeSetBuilder();
      for (let range of ranges instanceof Range$1 ? [ranges] : sort ? lazySort(ranges) : ranges) build.add(range.from, range.to, range.value);
      return build.finish();
    }
    /**
    Join an array of range sets into a single set.
    */
    static join(sets) {
      if (!sets.length) return RangeSet.empty;
      let result = sets[sets.length - 1];
      for (let i = sets.length - 2; i >= 0; i--) {
        for (let layer = sets[i]; layer != RangeSet.empty; layer = layer.nextLayer) result = new RangeSet(layer.chunkPos, layer.chunk, result, Math.max(layer.maxPoint, result.maxPoint));
      }
      return result;
    }
  }
  /**
  The empty set of ranges.
  */
  RangeSet.empty = /*@__PURE__*/new RangeSet([], [], null, -1);
  function lazySort(ranges) {
    if (ranges.length > 1) for (let prev = ranges[0], i = 1; i < ranges.length; i++) {
      let cur = ranges[i];
      if (cmpRange(prev, cur) > 0) return ranges.slice().sort(cmpRange);
      prev = cur;
    }
    return ranges;
  }
  RangeSet.empty.nextLayer = RangeSet.empty;
  /**
  A range set builder is a data structure that helps build up a
  [range set](https://codemirror.net/6/docs/ref/#state.RangeSet) directly, without first allocating
  an array of [`Range`](https://codemirror.net/6/docs/ref/#state.Range) objects.
  */
  class RangeSetBuilder {
    finishChunk(newArrays) {
      this.chunks.push(new Chunk(this.from, this.to, this.value, this.maxPoint));
      this.chunkPos.push(this.chunkStart);
      this.chunkStart = -1;
      this.setMaxPoint = Math.max(this.setMaxPoint, this.maxPoint);
      this.maxPoint = -1;
      if (newArrays) {
        this.from = [];
        this.to = [];
        this.value = [];
      }
    }
    /**
    Create an empty builder.
    */
    constructor() {
      this.chunks = [];
      this.chunkPos = [];
      this.chunkStart = -1;
      this.last = null;
      this.lastFrom = -1000000000 /* C.Far */;
      this.lastTo = -1000000000 /* C.Far */;
      this.from = [];
      this.to = [];
      this.value = [];
      this.maxPoint = -1;
      this.setMaxPoint = -1;
      this.nextLayer = null;
    }
    /**
    Add a range. Ranges should be added in sorted (by `from` and
    `value.startSide`) order.
    */
    add(from, to, value) {
      if (!this.addInner(from, to, value)) (this.nextLayer || (this.nextLayer = new RangeSetBuilder())).add(from, to, value);
    }
    /**
    @internal
    */
    addInner(from, to, value) {
      let diff = from - this.lastTo || value.startSide - this.last.endSide;
      if (diff <= 0 && (from - this.lastFrom || value.startSide - this.last.startSide) < 0) throw new Error("Ranges must be added sorted by `from` position and `startSide`");
      if (diff < 0) return false;
      if (this.from.length == 250 /* C.ChunkSize */) this.finishChunk(true);
      if (this.chunkStart < 0) this.chunkStart = from;
      this.from.push(from - this.chunkStart);
      this.to.push(to - this.chunkStart);
      this.last = value;
      this.lastFrom = from;
      this.lastTo = to;
      this.value.push(value);
      if (value.point) this.maxPoint = Math.max(this.maxPoint, to - from);
      return true;
    }
    /**
    @internal
    */
    addChunk(from, chunk) {
      if ((from - this.lastTo || chunk.value[0].startSide - this.last.endSide) < 0) return false;
      if (this.from.length) this.finishChunk(true);
      this.setMaxPoint = Math.max(this.setMaxPoint, chunk.maxPoint);
      this.chunks.push(chunk);
      this.chunkPos.push(from);
      let last = chunk.value.length - 1;
      this.last = chunk.value[last];
      this.lastFrom = chunk.from[last] + from;
      this.lastTo = chunk.to[last] + from;
      return true;
    }
    /**
    Finish the range set. Returns the new set. The builder can't be
    used anymore after this has been called.
    */
    finish() {
      return this.finishInner(RangeSet.empty);
    }
    /**
    @internal
    */
    finishInner(next) {
      if (this.from.length) this.finishChunk(false);
      if (this.chunks.length == 0) return next;
      let result = RangeSet.create(this.chunkPos, this.chunks, this.nextLayer ? this.nextLayer.finishInner(next) : next, this.setMaxPoint);
      this.from = null; // Make sure further `add` calls produce errors
      return result;
    }
  }
  function findSharedChunks(a, b, textDiff) {
    let inA = new Map();
    for (let set of a) for (let i = 0; i < set.chunk.length; i++) if (set.chunk[i].maxPoint <= 0) inA.set(set.chunk[i], set.chunkPos[i]);
    let shared = new Set();
    for (let set of b) for (let i = 0; i < set.chunk.length; i++) {
      let known = inA.get(set.chunk[i]);
      if (known != null && (textDiff ? textDiff.mapPos(known) : known) == set.chunkPos[i] && !(textDiff === null || textDiff === void 0 ? void 0 : textDiff.touchesRange(known, known + set.chunk[i].length))) shared.add(set.chunk[i]);
    }
    return shared;
  }
  class LayerCursor {
    constructor(layer, skip, minPoint, rank = 0) {
      this.layer = layer;
      this.skip = skip;
      this.minPoint = minPoint;
      this.rank = rank;
    }
    get startSide() {
      return this.value ? this.value.startSide : 0;
    }
    get endSide() {
      return this.value ? this.value.endSide : 0;
    }
    goto(pos, side = -1000000000 /* C.Far */) {
      this.chunkIndex = this.rangeIndex = 0;
      this.gotoInner(pos, side, false);
      return this;
    }
    gotoInner(pos, side, forward) {
      while (this.chunkIndex < this.layer.chunk.length) {
        let next = this.layer.chunk[this.chunkIndex];
        if (!(this.skip && this.skip.has(next) || this.layer.chunkEnd(this.chunkIndex) < pos || next.maxPoint < this.minPoint)) break;
        this.chunkIndex++;
        forward = false;
      }
      if (this.chunkIndex < this.layer.chunk.length) {
        let rangeIndex = this.layer.chunk[this.chunkIndex].findIndex(pos - this.layer.chunkPos[this.chunkIndex], side, true);
        if (!forward || this.rangeIndex < rangeIndex) this.setRangeIndex(rangeIndex);
      }
      this.next();
    }
    forward(pos, side) {
      if ((this.to - pos || this.endSide - side) < 0) this.gotoInner(pos, side, true);
    }
    next() {
      for (;;) {
        if (this.chunkIndex == this.layer.chunk.length) {
          this.from = this.to = 1000000000 /* C.Far */;
          this.value = null;
          break;
        } else {
          let chunkPos = this.layer.chunkPos[this.chunkIndex],
            chunk = this.layer.chunk[this.chunkIndex];
          let from = chunkPos + chunk.from[this.rangeIndex];
          this.from = from;
          this.to = chunkPos + chunk.to[this.rangeIndex];
          this.value = chunk.value[this.rangeIndex];
          this.setRangeIndex(this.rangeIndex + 1);
          if (this.minPoint < 0 || this.value.point && this.to - this.from >= this.minPoint) break;
        }
      }
    }
    setRangeIndex(index) {
      if (index == this.layer.chunk[this.chunkIndex].value.length) {
        this.chunkIndex++;
        if (this.skip) {
          while (this.chunkIndex < this.layer.chunk.length && this.skip.has(this.layer.chunk[this.chunkIndex])) this.chunkIndex++;
        }
        this.rangeIndex = 0;
      } else {
        this.rangeIndex = index;
      }
    }
    nextChunk() {
      this.chunkIndex++;
      this.rangeIndex = 0;
      this.next();
    }
    compare(other) {
      return this.from - other.from || this.startSide - other.startSide || this.rank - other.rank || this.to - other.to || this.endSide - other.endSide;
    }
  }
  class HeapCursor {
    constructor(heap) {
      this.heap = heap;
    }
    static from(sets, skip = null, minPoint = -1) {
      let heap = [];
      for (let i = 0; i < sets.length; i++) {
        for (let cur = sets[i]; !cur.isEmpty; cur = cur.nextLayer) {
          if (cur.maxPoint >= minPoint) heap.push(new LayerCursor(cur, skip, minPoint, i));
        }
      }
      return heap.length == 1 ? heap[0] : new HeapCursor(heap);
    }
    get startSide() {
      return this.value ? this.value.startSide : 0;
    }
    goto(pos, side = -1000000000 /* C.Far */) {
      for (let cur of this.heap) cur.goto(pos, side);
      for (let i = this.heap.length >> 1; i >= 0; i--) heapBubble(this.heap, i);
      this.next();
      return this;
    }
    forward(pos, side) {
      for (let cur of this.heap) cur.forward(pos, side);
      for (let i = this.heap.length >> 1; i >= 0; i--) heapBubble(this.heap, i);
      if ((this.to - pos || this.value.endSide - side) < 0) this.next();
    }
    next() {
      if (this.heap.length == 0) {
        this.from = this.to = 1000000000 /* C.Far */;
        this.value = null;
        this.rank = -1;
      } else {
        let top = this.heap[0];
        this.from = top.from;
        this.to = top.to;
        this.value = top.value;
        this.rank = top.rank;
        if (top.value) top.next();
        heapBubble(this.heap, 0);
      }
    }
  }
  function heapBubble(heap, index) {
    for (let cur = heap[index];;) {
      let childIndex = (index << 1) + 1;
      if (childIndex >= heap.length) break;
      let child = heap[childIndex];
      if (childIndex + 1 < heap.length && child.compare(heap[childIndex + 1]) >= 0) {
        child = heap[childIndex + 1];
        childIndex++;
      }
      if (cur.compare(child) < 0) break;
      heap[childIndex] = cur;
      heap[index] = child;
      index = childIndex;
    }
  }
  class SpanCursor {
    constructor(sets, skip, minPoint) {
      this.minPoint = minPoint;
      this.active = [];
      this.activeTo = [];
      this.activeRank = [];
      this.minActive = -1;
      // A currently active point range, if any
      this.point = null;
      this.pointFrom = 0;
      this.pointRank = 0;
      this.to = -1000000000 /* C.Far */;
      this.endSide = 0;
      // The amount of open active ranges at the start of the iterator.
      // Not including points.
      this.openStart = -1;
      this.cursor = HeapCursor.from(sets, skip, minPoint);
    }
    goto(pos, side = -1000000000 /* C.Far */) {
      this.cursor.goto(pos, side);
      this.active.length = this.activeTo.length = this.activeRank.length = 0;
      this.minActive = -1;
      this.to = pos;
      this.endSide = side;
      this.openStart = -1;
      this.next();
      return this;
    }
    forward(pos, side) {
      while (this.minActive > -1 && (this.activeTo[this.minActive] - pos || this.active[this.minActive].endSide - side) < 0) this.removeActive(this.minActive);
      this.cursor.forward(pos, side);
    }
    removeActive(index) {
      remove(this.active, index);
      remove(this.activeTo, index);
      remove(this.activeRank, index);
      this.minActive = findMinIndex(this.active, this.activeTo);
    }
    addActive(trackOpen) {
      let i = 0,
        {
          value,
          to,
          rank
        } = this.cursor;
      // Organize active marks by rank first, then by size
      while (i < this.activeRank.length && (rank - this.activeRank[i] || to - this.activeTo[i]) > 0) i++;
      insert(this.active, i, value);
      insert(this.activeTo, i, to);
      insert(this.activeRank, i, rank);
      if (trackOpen) insert(trackOpen, i, this.cursor.from);
      this.minActive = findMinIndex(this.active, this.activeTo);
    }
    // After calling this, if `this.point` != null, the next range is a
    // point. Otherwise, it's a regular range, covered by `this.active`.
    next() {
      let from = this.to,
        wasPoint = this.point;
      this.point = null;
      let trackOpen = this.openStart < 0 ? [] : null;
      for (;;) {
        let a = this.minActive;
        if (a > -1 && (this.activeTo[a] - this.cursor.from || this.active[a].endSide - this.cursor.startSide) < 0) {
          if (this.activeTo[a] > from) {
            this.to = this.activeTo[a];
            this.endSide = this.active[a].endSide;
            break;
          }
          this.removeActive(a);
          if (trackOpen) remove(trackOpen, a);
        } else if (!this.cursor.value) {
          this.to = this.endSide = 1000000000 /* C.Far */;
          break;
        } else if (this.cursor.from > from) {
          this.to = this.cursor.from;
          this.endSide = this.cursor.startSide;
          break;
        } else {
          let nextVal = this.cursor.value;
          if (!nextVal.point) {
            // Opening a range
            this.addActive(trackOpen);
            this.cursor.next();
          } else if (wasPoint && this.cursor.to == this.to && this.cursor.from < this.cursor.to) {
            // Ignore any non-empty points that end precisely at the end of the prev point
            this.cursor.next();
          } else {
            // New point
            this.point = nextVal;
            this.pointFrom = this.cursor.from;
            this.pointRank = this.cursor.rank;
            this.to = this.cursor.to;
            this.endSide = nextVal.endSide;
            this.cursor.next();
            this.forward(this.to, this.endSide);
            break;
          }
        }
      }
      if (trackOpen) {
        this.openStart = 0;
        for (let i = trackOpen.length - 1; i >= 0 && trackOpen[i] < from; i--) this.openStart++;
      }
    }
    activeForPoint(to) {
      if (!this.active.length) return this.active;
      let active = [];
      for (let i = this.active.length - 1; i >= 0; i--) {
        if (this.activeRank[i] < this.pointRank) break;
        if (this.activeTo[i] > to || this.activeTo[i] == to && this.active[i].endSide >= this.point.endSide) active.push(this.active[i]);
      }
      return active.reverse();
    }
    openEnd(to) {
      let open = 0;
      for (let i = this.activeTo.length - 1; i >= 0 && this.activeTo[i] > to; i--) open++;
      return open;
    }
  }
  function compare(a, startA, b, startB, length, comparator) {
    a.goto(startA);
    b.goto(startB);
    let endB = startB + length;
    let pos = startB,
      dPos = startB - startA;
    for (;;) {
      let dEnd = a.to + dPos - b.to,
        diff = dEnd || a.endSide - b.endSide;
      let end = diff < 0 ? a.to + dPos : b.to,
        clipEnd = Math.min(end, endB);
      if (a.point || b.point) {
        if (!(a.point && b.point && (a.point == b.point || a.point.eq(b.point)) && sameValues(a.activeForPoint(a.to), b.activeForPoint(b.to)))) comparator.comparePoint(pos, clipEnd, a.point, b.point);
      } else {
        if (clipEnd > pos && !sameValues(a.active, b.active)) comparator.compareRange(pos, clipEnd, a.active, b.active);
      }
      if (end > endB) break;
      if ((dEnd || a.openEnd != b.openEnd) && comparator.boundChange) comparator.boundChange(end);
      pos = end;
      if (diff <= 0) a.next();
      if (diff >= 0) b.next();
    }
  }
  function sameValues(a, b) {
    if (a.length != b.length) return false;
    for (let i = 0; i < a.length; i++) if (a[i] != b[i] && !a[i].eq(b[i])) return false;
    return true;
  }
  function remove(array, index) {
    for (let i = index, e = array.length - 1; i < e; i++) array[i] = array[i + 1];
    array.pop();
  }
  function insert(array, index, value) {
    for (let i = array.length - 1; i >= index; i--) array[i + 1] = array[i];
    array[index] = value;
  }
  function findMinIndex(value, array) {
    let found = -1,
      foundPos = 1000000000 /* C.Far */;
    for (let i = 0; i < array.length; i++) if ((array[i] - foundPos || value[i].endSide - value[found].endSide) < 0) {
      found = i;
      foundPos = array[i];
    }
    return found;
  }
  /**
  Find the offset that corresponds to the given column position in a
  string, taking extending characters and tab size into account. By
  default, the string length is returned when it is too short to
  reach the column. Pass `strict` true to make it return -1 in that
  situation.
  */
  function findColumn(string, col, tabSize, strict) {
    for (let i = 0, n = 0;;) {
      if (n >= col) return i;
      if (i == string.length) break;
      n += string.charCodeAt(i) == 9 ? tabSize - n % tabSize : 1;
      i = findClusterBreak(string, i);
    }
    return strict === true ? -1 : string.length;
  }

  const C = "\u037c";
  const COUNT = typeof Symbol == "undefined" ? "__" + C : Symbol.for(C);
  const SET = typeof Symbol == "undefined" ? "__styleSet" + Math.floor(Math.random() * 1e8) : Symbol("styleSet");
  const top = typeof globalThis != "undefined" ? globalThis : typeof window != "undefined" ? window : {};

  // :: - Style modules encapsulate a set of CSS rules defined from
  // JavaScript. Their definitions are only available in a given DOM
  // root after it has been _mounted_ there with `StyleModule.mount`.
  //
  // Style modules should be created once and stored somewhere, as
  // opposed to re-creating them every time you need them. The amount of
  // CSS rules generated for a given DOM root is bounded by the amount
  // of style modules that were used. So to avoid leaking rules, don't
  // create these dynamically, but treat them as one-time allocations.
  class StyleModule {
    // :: (Object<Style>, ?{finish: ?(string) → string})
    // Create a style module from the given spec.
    //
    // When `finish` is given, it is called on regular (non-`@`)
    // selectors (after `&` expansion) to compute the final selector.
    constructor(spec, options) {
      this.rules = [];
      let {
        finish
      } = options || {};
      function splitSelector(selector) {
        return /^@/.test(selector) ? [selector] : selector.split(/,\s*/);
      }
      function render(selectors, spec, target, isKeyframes) {
        let local = [],
          isAt = /^@(\w+)\b/.exec(selectors[0]),
          keyframes = isAt && isAt[1] == "keyframes";
        if (isAt && spec == null) return target.push(selectors[0] + ";");
        for (let prop in spec) {
          let value = spec[prop];
          if (/&/.test(prop)) {
            render(prop.split(/,\s*/).map(part => selectors.map(sel => part.replace(/&/, sel))).reduce((a, b) => a.concat(b)), value, target);
          } else if (value && typeof value == "object") {
            if (!isAt) throw new RangeError("The value of a property (" + prop + ") should be a primitive value.");
            render(splitSelector(prop), value, local, keyframes);
          } else if (value != null) {
            local.push(prop.replace(/_.*/, "").replace(/[A-Z]/g, l => "-" + l.toLowerCase()) + ": " + value + ";");
          }
        }
        if (local.length || keyframes) {
          target.push((finish && !isAt && !isKeyframes ? selectors.map(finish) : selectors).join(", ") + " {" + local.join(" ") + "}");
        }
      }
      for (let prop in spec) render(splitSelector(prop), spec[prop], this.rules);
    }

    // :: () → string
    // Returns a string containing the module's CSS rules.
    getRules() {
      return this.rules.join("\n");
    }

    // :: () → string
    // Generate a new unique CSS class name.
    static newName() {
      let id = top[COUNT] || 1;
      top[COUNT] = id + 1;
      return C + id.toString(36);
    }

    // :: (union<Document, ShadowRoot>, union<[StyleModule], StyleModule>, ?{nonce: ?string})
    //
    // Mount the given set of modules in the given DOM root, which ensures
    // that the CSS rules defined by the module are available in that
    // context.
    //
    // Rules are only added to the document once per root.
    //
    // Rule order will follow the order of the modules, so that rules from
    // modules later in the array take precedence of those from earlier
    // modules. If you call this function multiple times for the same root
    // in a way that changes the order of already mounted modules, the old
    // order will be changed.
    //
    // If a Content Security Policy nonce is provided, it is added to
    // the `<style>` tag generated by the library.
    static mount(root, modules, options) {
      let set = root[SET],
        nonce = options && options.nonce;
      if (!set) set = new StyleSet(root, nonce);else if (nonce) set.setNonce(nonce);
      set.mount(Array.isArray(modules) ? modules : [modules], root);
    }
  }
  let adoptedSet = new Map(); //<Document, StyleSet>

  class StyleSet {
    constructor(root, nonce) {
      let doc = root.ownerDocument || root,
        win = doc.defaultView;
      if (!root.head && root.adoptedStyleSheets && win.CSSStyleSheet) {
        let adopted = adoptedSet.get(doc);
        if (adopted) return root[SET] = adopted;
        this.sheet = new win.CSSStyleSheet();
        adoptedSet.set(doc, this);
      } else {
        this.styleTag = doc.createElement("style");
        if (nonce) this.styleTag.setAttribute("nonce", nonce);
      }
      this.modules = [];
      root[SET] = this;
    }
    mount(modules, root) {
      let sheet = this.sheet;
      let pos = 0 /* Current rule offset */,
        j = 0; /* Index into this.modules */
      for (let i = 0; i < modules.length; i++) {
        let mod = modules[i],
          index = this.modules.indexOf(mod);
        if (index < j && index > -1) {
          // Ordering conflict
          this.modules.splice(index, 1);
          j--;
          index = -1;
        }
        if (index == -1) {
          this.modules.splice(j++, 0, mod);
          if (sheet) for (let k = 0; k < mod.rules.length; k++) sheet.insertRule(mod.rules[k], pos++);
        } else {
          while (j < index) pos += this.modules[j++].rules.length;
          pos += mod.rules.length;
          j++;
        }
      }
      if (sheet) {
        if (root.adoptedStyleSheets.indexOf(this.sheet) < 0) root.adoptedStyleSheets = [this.sheet, ...root.adoptedStyleSheets];
      } else {
        let text = "";
        for (let i = 0; i < this.modules.length; i++) text += this.modules[i].getRules() + "\n";
        this.styleTag.textContent = text;
        let target = root.head || root;
        if (this.styleTag.parentNode != target) target.insertBefore(this.styleTag, target.firstChild);
      }
    }
    setNonce(nonce) {
      if (this.styleTag && this.styleTag.getAttribute("nonce") != nonce) this.styleTag.setAttribute("nonce", nonce);
    }
  }

  // Style::Object<union<Style,string>>
  //
  // A style is an object that, in the simple case, maps CSS property
  // names to strings holding their values, as in `{color: "red",
  // fontWeight: "bold"}`. The property names can be given in
  // camel-case—the library will insert a dash before capital letters
  // when converting them to CSS.
  //
  // If you include an underscore in a property name, it and everything
  // after it will be removed from the output, which can be useful when
  // providing a property multiple times, for browser compatibility
  // reasons.
  //
  // A property in a style object can also be a sub-selector, which
  // extends the current context to add a pseudo-selector or a child
  // selector. Such a property should contain a `&` character, which
  // will be replaced by the current selector. For example `{"&:before":
  // {content: '"hi"'}}`. Sub-selectors and regular properties can
  // freely be mixed in a given object. Any property containing a `&` is
  // assumed to be a sub-selector.
  //
  // Finally, a property can specify an @-block to be wrapped around the
  // styles defined inside the object that's the property's value. For
  // example to create a media query you can do `{"@media screen and
  // (min-width: 400px)": {...}}`.

  var base = {
    8: "Backspace",
    9: "Tab",
    10: "Enter",
    12: "NumLock",
    13: "Enter",
    16: "Shift",
    17: "Control",
    18: "Alt",
    20: "CapsLock",
    27: "Escape",
    32: " ",
    33: "PageUp",
    34: "PageDown",
    35: "End",
    36: "Home",
    37: "ArrowLeft",
    38: "ArrowUp",
    39: "ArrowRight",
    40: "ArrowDown",
    44: "PrintScreen",
    45: "Insert",
    46: "Delete",
    59: ";",
    61: "=",
    91: "Meta",
    92: "Meta",
    106: "*",
    107: "+",
    108: ",",
    109: "-",
    110: ".",
    111: "/",
    144: "NumLock",
    145: "ScrollLock",
    160: "Shift",
    161: "Shift",
    162: "Control",
    163: "Control",
    164: "Alt",
    165: "Alt",
    173: "-",
    186: ";",
    187: "=",
    188: ",",
    189: "-",
    190: ".",
    191: "/",
    192: "`",
    219: "[",
    220: "\\",
    221: "]",
    222: "'"
  };
  var shift = {
    48: ")",
    49: "!",
    50: "@",
    51: "#",
    52: "$",
    53: "%",
    54: "^",
    55: "&",
    56: "*",
    57: "(",
    59: ":",
    61: "+",
    173: "_",
    186: ":",
    187: "+",
    188: "<",
    189: "_",
    190: ">",
    191: "?",
    192: "~",
    219: "{",
    220: "|",
    221: "}",
    222: "\""
  };

  // Fill in the digit keys
  for (var i = 0; i < 10; i++) base[48 + i] = base[96 + i] = String(i);

  // The function keys
  for (var i = 1; i <= 24; i++) base[i + 111] = "F" + i;

  // And the alphabetic keys
  for (var i = 65; i <= 90; i++) {
    base[i] = String.fromCharCode(i + 32);
    shift[i] = String.fromCharCode(i);
  }

  // For each code that doesn't have a shift-equivalent, copy the base name
  for (var code in base) if (!shift.hasOwnProperty(code)) shift[code] = base[code];

  function getSelection(root) {
    let target;
    // Browsers differ on whether shadow roots have a getSelection
    // method. If it exists, use that, otherwise, call it on the
    // document.
    if (root.nodeType == 11) {
      // Shadow root
      target = root.getSelection ? root : root.ownerDocument;
    } else {
      target = root;
    }
    return target.getSelection();
  }
  function contains(dom, node) {
    return node ? dom == node || dom.contains(node.nodeType != 1 ? node.parentNode : node) : false;
  }
  function hasSelection(dom, selection) {
    if (!selection.anchorNode) return false;
    try {
      // Firefox will raise 'permission denied' errors when accessing
      // properties of `sel.anchorNode` when it's in a generated CSS
      // element.
      return contains(dom, selection.anchorNode);
    } catch (_) {
      return false;
    }
  }
  function clientRectsFor(dom) {
    if (dom.nodeType == 3) return textRange(dom, 0, dom.nodeValue.length).getClientRects();else if (dom.nodeType == 1) return dom.getClientRects();else return [];
  }
  // Scans forward and backward through DOM positions equivalent to the
  // given one to see if the two are in the same place (i.e. after a
  // text node vs at the end of that text node)
  function isEquivalentPosition(node, off, targetNode, targetOff) {
    return targetNode ? scanFor(node, off, targetNode, targetOff, -1) || scanFor(node, off, targetNode, targetOff, 1) : false;
  }
  function domIndex(node) {
    for (var index = 0;; index++) {
      node = node.previousSibling;
      if (!node) return index;
    }
  }
  function isBlockElement(node) {
    return node.nodeType == 1 && /^(DIV|P|LI|UL|OL|BLOCKQUOTE|DD|DT|H\d|SECTION|PRE)$/.test(node.nodeName);
  }
  function scanFor(node, off, targetNode, targetOff, dir) {
    for (;;) {
      if (node == targetNode && off == targetOff) return true;
      if (off == (dir < 0 ? 0 : maxOffset(node))) {
        if (node.nodeName == "DIV") return false;
        let parent = node.parentNode;
        if (!parent || parent.nodeType != 1) return false;
        off = domIndex(node) + (dir < 0 ? 0 : 1);
        node = parent;
      } else if (node.nodeType == 1) {
        node = node.childNodes[off + (dir < 0 ? -1 : 0)];
        if (node.nodeType == 1 && node.contentEditable == "false") return false;
        off = dir < 0 ? maxOffset(node) : 0;
      } else {
        return false;
      }
    }
  }
  function maxOffset(node) {
    return node.nodeType == 3 ? node.nodeValue.length : node.childNodes.length;
  }
  function flattenRect(rect, left) {
    let x = left ? rect.left : rect.right;
    return {
      left: x,
      right: x,
      top: rect.top,
      bottom: rect.bottom
    };
  }
  function windowRect(win) {
    let vp = win.visualViewport;
    if (vp) return {
      left: 0,
      right: vp.width,
      top: 0,
      bottom: vp.height
    };
    return {
      left: 0,
      right: win.innerWidth,
      top: 0,
      bottom: win.innerHeight
    };
  }
  function getScale(elt, rect) {
    let scaleX = rect.width / elt.offsetWidth;
    let scaleY = rect.height / elt.offsetHeight;
    if (scaleX > 0.995 && scaleX < 1.005 || !isFinite(scaleX) || Math.abs(rect.width - elt.offsetWidth) < 1) scaleX = 1;
    if (scaleY > 0.995 && scaleY < 1.005 || !isFinite(scaleY) || Math.abs(rect.height - elt.offsetHeight) < 1) scaleY = 1;
    return {
      scaleX,
      scaleY
    };
  }
  function scrollRectIntoView(dom, rect, side, x, y, xMargin, yMargin, ltr) {
    let doc = dom.ownerDocument,
      win = doc.defaultView || window;
    for (let cur = dom, stop = false; cur && !stop;) {
      if (cur.nodeType == 1) {
        // Element
        let bounding,
          top = cur == doc.body;
        let scaleX = 1,
          scaleY = 1;
        if (top) {
          bounding = windowRect(win);
        } else {
          if (/^(fixed|sticky)$/.test(getComputedStyle(cur).position)) stop = true;
          if (cur.scrollHeight <= cur.clientHeight && cur.scrollWidth <= cur.clientWidth) {
            cur = cur.assignedSlot || cur.parentNode;
            continue;
          }
          let rect = cur.getBoundingClientRect();
          ({
            scaleX,
            scaleY
          } = getScale(cur, rect));
          // Make sure scrollbar width isn't included in the rectangle
          bounding = {
            left: rect.left,
            right: rect.left + cur.clientWidth * scaleX,
            top: rect.top,
            bottom: rect.top + cur.clientHeight * scaleY
          };
        }
        let moveX = 0,
          moveY = 0;
        if (y == "nearest") {
          if (rect.top < bounding.top) {
            moveY = -(bounding.top - rect.top + yMargin);
            if (side > 0 && rect.bottom > bounding.bottom + moveY) moveY = rect.bottom - bounding.bottom + moveY + yMargin;
          } else if (rect.bottom > bounding.bottom) {
            moveY = rect.bottom - bounding.bottom + yMargin;
            if (side < 0 && rect.top - moveY < bounding.top) moveY = -(bounding.top + moveY - rect.top + yMargin);
          }
        } else {
          let rectHeight = rect.bottom - rect.top,
            boundingHeight = bounding.bottom - bounding.top;
          let targetTop = y == "center" && rectHeight <= boundingHeight ? rect.top + rectHeight / 2 - boundingHeight / 2 : y == "start" || y == "center" && side < 0 ? rect.top - yMargin : rect.bottom - boundingHeight + yMargin;
          moveY = targetTop - bounding.top;
        }
        if (x == "nearest") {
          if (rect.left < bounding.left) {
            moveX = -(bounding.left - rect.left + xMargin);
            if (side > 0 && rect.right > bounding.right + moveX) moveX = rect.right - bounding.right + moveX + xMargin;
          } else if (rect.right > bounding.right) {
            moveX = rect.right - bounding.right + xMargin;
            if (side < 0 && rect.left < bounding.left + moveX) moveX = -(bounding.left + moveX - rect.left + xMargin);
          }
        } else {
          let targetLeft = x == "center" ? rect.left + (rect.right - rect.left) / 2 - (bounding.right - bounding.left) / 2 : x == "start" == ltr ? rect.left - xMargin : rect.right - (bounding.right - bounding.left) + xMargin;
          moveX = targetLeft - bounding.left;
        }
        if (moveX || moveY) {
          if (top) {
            win.scrollBy(moveX, moveY);
          } else {
            let movedX = 0,
              movedY = 0;
            if (moveY) {
              let start = cur.scrollTop;
              cur.scrollTop += moveY / scaleY;
              movedY = (cur.scrollTop - start) * scaleY;
            }
            if (moveX) {
              let start = cur.scrollLeft;
              cur.scrollLeft += moveX / scaleX;
              movedX = (cur.scrollLeft - start) * scaleX;
            }
            rect = {
              left: rect.left - movedX,
              top: rect.top - movedY,
              right: rect.right - movedX,
              bottom: rect.bottom - movedY
            };
            if (movedX && Math.abs(movedX - moveX) < 1) x = "nearest";
            if (movedY && Math.abs(movedY - moveY) < 1) y = "nearest";
          }
        }
        if (top) break;
        cur = cur.assignedSlot || cur.parentNode;
      } else if (cur.nodeType == 11) {
        // A shadow root
        cur = cur.host;
      } else {
        break;
      }
    }
  }
  function scrollableParents(dom) {
    let doc = dom.ownerDocument,
      x,
      y;
    for (let cur = dom.parentNode; cur;) {
      if (cur == doc.body || x && y) {
        break;
      } else if (cur.nodeType == 1) {
        if (!y && cur.scrollHeight > cur.clientHeight) y = cur;
        if (!x && cur.scrollWidth > cur.clientWidth) x = cur;
        cur = cur.assignedSlot || cur.parentNode;
      } else if (cur.nodeType == 11) {
        cur = cur.host;
      } else {
        break;
      }
    }
    return {
      x,
      y
    };
  }
  class DOMSelectionState {
    constructor() {
      this.anchorNode = null;
      this.anchorOffset = 0;
      this.focusNode = null;
      this.focusOffset = 0;
    }
    eq(domSel) {
      return this.anchorNode == domSel.anchorNode && this.anchorOffset == domSel.anchorOffset && this.focusNode == domSel.focusNode && this.focusOffset == domSel.focusOffset;
    }
    setRange(range) {
      let {
        anchorNode,
        focusNode
      } = range;
      // Clip offsets to node size to avoid crashes when Safari reports bogus offsets (#1152)
      this.set(anchorNode, Math.min(range.anchorOffset, anchorNode ? maxOffset(anchorNode) : 0), focusNode, Math.min(range.focusOffset, focusNode ? maxOffset(focusNode) : 0));
    }
    set(anchorNode, anchorOffset, focusNode, focusOffset) {
      this.anchorNode = anchorNode;
      this.anchorOffset = anchorOffset;
      this.focusNode = focusNode;
      this.focusOffset = focusOffset;
    }
  }
  let preventScrollSupported = null;
  // Feature-detects support for .focus({preventScroll: true}), and uses
  // a fallback kludge when not supported.
  function focusPreventScroll(dom) {
    if (dom.setActive) return dom.setActive(); // in IE
    if (preventScrollSupported) return dom.focus(preventScrollSupported);
    let stack = [];
    for (let cur = dom; cur; cur = cur.parentNode) {
      stack.push(cur, cur.scrollTop, cur.scrollLeft);
      if (cur == cur.ownerDocument) break;
    }
    dom.focus(preventScrollSupported == null ? {
      get preventScroll() {
        preventScrollSupported = {
          preventScroll: true
        };
        return true;
      }
    } : undefined);
    if (!preventScrollSupported) {
      preventScrollSupported = false;
      for (let i = 0; i < stack.length;) {
        let elt = stack[i++],
          top = stack[i++],
          left = stack[i++];
        if (elt.scrollTop != top) elt.scrollTop = top;
        if (elt.scrollLeft != left) elt.scrollLeft = left;
      }
    }
  }
  let scratchRange;
  function textRange(node, from, to = from) {
    let range = scratchRange || (scratchRange = document.createRange());
    range.setEnd(node, to);
    range.setStart(node, from);
    return range;
  }
  function dispatchKey(elt, name, code, mods) {
    let options = {
      key: name,
      code: name,
      keyCode: code,
      which: code,
      cancelable: true
    };
    if (mods) ({
      altKey: options.altKey,
      ctrlKey: options.ctrlKey,
      shiftKey: options.shiftKey,
      metaKey: options.metaKey
    } = mods);
    let down = new KeyboardEvent("keydown", options);
    down.synthetic = true;
    elt.dispatchEvent(down);
    let up = new KeyboardEvent("keyup", options);
    up.synthetic = true;
    elt.dispatchEvent(up);
    return down.defaultPrevented || up.defaultPrevented;
  }
  function getRoot(node) {
    while (node) {
      if (node && (node.nodeType == 9 || node.nodeType == 11 && node.host)) return node;
      node = node.assignedSlot || node.parentNode;
    }
    return null;
  }
  function clearAttributes(node) {
    while (node.attributes.length) node.removeAttributeNode(node.attributes[0]);
  }
  function atElementStart(doc, selection) {
    let node = selection.focusNode,
      offset = selection.focusOffset;
    if (!node || selection.anchorNode != node || selection.anchorOffset != offset) return false;
    // Safari can report bogus offsets (#1152)
    offset = Math.min(offset, maxOffset(node));
    for (;;) {
      if (offset) {
        if (node.nodeType != 1) return false;
        let prev = node.childNodes[offset - 1];
        if (prev.contentEditable == "false") offset--;else {
          node = prev;
          offset = maxOffset(node);
        }
      } else if (node == doc) {
        return true;
      } else {
        offset = domIndex(node);
        node = node.parentNode;
      }
    }
  }
  function isScrolledToBottom(elt) {
    return elt.scrollTop > Math.max(1, elt.scrollHeight - elt.clientHeight - 4);
  }
  function textNodeBefore(startNode, startOffset) {
    for (let node = startNode, offset = startOffset;;) {
      if (node.nodeType == 3 && offset > 0) {
        return {
          node: node,
          offset: offset
        };
      } else if (node.nodeType == 1 && offset > 0) {
        if (node.contentEditable == "false") return null;
        node = node.childNodes[offset - 1];
        offset = maxOffset(node);
      } else if (node.parentNode && !isBlockElement(node)) {
        offset = domIndex(node);
        node = node.parentNode;
      } else {
        return null;
      }
    }
  }
  function textNodeAfter(startNode, startOffset) {
    for (let node = startNode, offset = startOffset;;) {
      if (node.nodeType == 3 && offset < node.nodeValue.length) {
        return {
          node: node,
          offset: offset
        };
      } else if (node.nodeType == 1 && offset < node.childNodes.length) {
        if (node.contentEditable == "false") return null;
        node = node.childNodes[offset];
        offset = 0;
      } else if (node.parentNode && !isBlockElement(node)) {
        offset = domIndex(node) + 1;
        node = node.parentNode;
      } else {
        return null;
      }
    }
  }
  class DOMPos {
    constructor(node, offset, precise = true) {
      this.node = node;
      this.offset = offset;
      this.precise = precise;
    }
    static before(dom, precise) {
      return new DOMPos(dom.parentNode, domIndex(dom), precise);
    }
    static after(dom, precise) {
      return new DOMPos(dom.parentNode, domIndex(dom) + 1, precise);
    }
  }
  const noChildren = [];
  class ContentView {
    constructor() {
      this.parent = null;
      this.dom = null;
      this.flags = 2 /* ViewFlag.NodeDirty */;
    }
    get overrideDOMText() {
      return null;
    }
    get posAtStart() {
      return this.parent ? this.parent.posBefore(this) : 0;
    }
    get posAtEnd() {
      return this.posAtStart + this.length;
    }
    posBefore(view) {
      let pos = this.posAtStart;
      for (let child of this.children) {
        if (child == view) return pos;
        pos += child.length + child.breakAfter;
      }
      throw new RangeError("Invalid child in posBefore");
    }
    posAfter(view) {
      return this.posBefore(view) + view.length;
    }
    sync(view, track) {
      if (this.flags & 2 /* ViewFlag.NodeDirty */) {
        let parent = this.dom;
        let prev = null,
          next;
        for (let child of this.children) {
          if (child.flags & 7 /* ViewFlag.Dirty */) {
            if (!child.dom && (next = prev ? prev.nextSibling : parent.firstChild)) {
              let contentView = ContentView.get(next);
              if (!contentView || !contentView.parent && contentView.canReuseDOM(child)) child.reuseDOM(next);
            }
            child.sync(view, track);
            child.flags &= ~7 /* ViewFlag.Dirty */;
          }
          next = prev ? prev.nextSibling : parent.firstChild;
          if (track && !track.written && track.node == parent && next != child.dom) track.written = true;
          if (child.dom.parentNode == parent) {
            while (next && next != child.dom) next = rm$1(next);
          } else {
            parent.insertBefore(child.dom, next);
          }
          prev = child.dom;
        }
        next = prev ? prev.nextSibling : parent.firstChild;
        if (next && track && track.node == parent) track.written = true;
        while (next) next = rm$1(next);
      } else if (this.flags & 1 /* ViewFlag.ChildDirty */) {
        for (let child of this.children) if (child.flags & 7 /* ViewFlag.Dirty */) {
          child.sync(view, track);
          child.flags &= ~7 /* ViewFlag.Dirty */;
        }
      }
    }
    reuseDOM(_dom) {}
    localPosFromDOM(node, offset) {
      let after;
      if (node == this.dom) {
        after = this.dom.childNodes[offset];
      } else {
        let bias = maxOffset(node) == 0 ? 0 : offset == 0 ? -1 : 1;
        for (;;) {
          let parent = node.parentNode;
          if (parent == this.dom) break;
          if (bias == 0 && parent.firstChild != parent.lastChild) {
            if (node == parent.firstChild) bias = -1;else bias = 1;
          }
          node = parent;
        }
        if (bias < 0) after = node;else after = node.nextSibling;
      }
      if (after == this.dom.firstChild) return 0;
      while (after && !ContentView.get(after)) after = after.nextSibling;
      if (!after) return this.length;
      for (let i = 0, pos = 0;; i++) {
        let child = this.children[i];
        if (child.dom == after) return pos;
        pos += child.length + child.breakAfter;
      }
    }
    domBoundsAround(from, to, offset = 0) {
      let fromI = -1,
        fromStart = -1,
        toI = -1,
        toEnd = -1;
      for (let i = 0, pos = offset, prevEnd = offset; i < this.children.length; i++) {
        let child = this.children[i],
          end = pos + child.length;
        if (pos < from && end > to) return child.domBoundsAround(from, to, pos);
        if (end >= from && fromI == -1) {
          fromI = i;
          fromStart = pos;
        }
        if (pos > to && child.dom.parentNode == this.dom) {
          toI = i;
          toEnd = prevEnd;
          break;
        }
        prevEnd = end;
        pos = end + child.breakAfter;
      }
      return {
        from: fromStart,
        to: toEnd < 0 ? offset + this.length : toEnd,
        startDOM: (fromI ? this.children[fromI - 1].dom.nextSibling : null) || this.dom.firstChild,
        endDOM: toI < this.children.length && toI >= 0 ? this.children[toI].dom : null
      };
    }
    markDirty(andParent = false) {
      this.flags |= 2 /* ViewFlag.NodeDirty */;
      this.markParentsDirty(andParent);
    }
    markParentsDirty(childList) {
      for (let parent = this.parent; parent; parent = parent.parent) {
        if (childList) parent.flags |= 2 /* ViewFlag.NodeDirty */;
        if (parent.flags & 1 /* ViewFlag.ChildDirty */) return;
        parent.flags |= 1 /* ViewFlag.ChildDirty */;
        childList = false;
      }
    }
    setParent(parent) {
      if (this.parent != parent) {
        this.parent = parent;
        if (this.flags & 7 /* ViewFlag.Dirty */) this.markParentsDirty(true);
      }
    }
    setDOM(dom) {
      if (this.dom == dom) return;
      if (this.dom) this.dom.cmView = null;
      this.dom = dom;
      dom.cmView = this;
    }
    get rootView() {
      for (let v = this;;) {
        let parent = v.parent;
        if (!parent) return v;
        v = parent;
      }
    }
    replaceChildren(from, to, children = noChildren) {
      this.markDirty();
      for (let i = from; i < to; i++) {
        let child = this.children[i];
        if (child.parent == this && children.indexOf(child) < 0) child.destroy();
      }
      if (children.length < 250) this.children.splice(from, to - from, ...children);else this.children = [].concat(this.children.slice(0, from), children, this.children.slice(to));
      for (let i = 0; i < children.length; i++) children[i].setParent(this);
    }
    ignoreMutation(_rec) {
      return false;
    }
    ignoreEvent(_event) {
      return false;
    }
    childCursor(pos = this.length) {
      return new ChildCursor(this.children, pos, this.children.length);
    }
    childPos(pos, bias = 1) {
      return this.childCursor().findPos(pos, bias);
    }
    toString() {
      let name = this.constructor.name.replace("View", "");
      return name + (this.children.length ? "(" + this.children.join() + ")" : this.length ? "[" + (name == "Text" ? this.text : this.length) + "]" : "") + (this.breakAfter ? "#" : "");
    }
    static get(node) {
      return node.cmView;
    }
    get isEditable() {
      return true;
    }
    get isWidget() {
      return false;
    }
    get isHidden() {
      return false;
    }
    merge(from, to, source, hasStart, openStart, openEnd) {
      return false;
    }
    become(other) {
      return false;
    }
    canReuseDOM(other) {
      return other.constructor == this.constructor && !((this.flags | other.flags) & 8 /* ViewFlag.Composition */);
    }
    // When this is a zero-length view with a side, this should return a
    // number <= 0 to indicate it is before its position, or a
    // number > 0 when after its position.
    getSide() {
      return 0;
    }
    destroy() {
      for (let child of this.children) if (child.parent == this) child.destroy();
      this.parent = null;
    }
  }
  ContentView.prototype.breakAfter = 0;
  // Remove a DOM node and return its next sibling.
  function rm$1(dom) {
    let next = dom.nextSibling;
    dom.parentNode.removeChild(dom);
    return next;
  }
  class ChildCursor {
    constructor(children, pos, i) {
      this.children = children;
      this.pos = pos;
      this.i = i;
      this.off = 0;
    }
    findPos(pos, bias = 1) {
      for (;;) {
        if (pos > this.pos || pos == this.pos && (bias > 0 || this.i == 0 || this.children[this.i - 1].breakAfter)) {
          this.off = pos - this.pos;
          return this;
        }
        let next = this.children[--this.i];
        this.pos -= next.length + next.breakAfter;
      }
    }
  }
  function replaceRange(parent, fromI, fromOff, toI, toOff, insert, breakAtStart, openStart, openEnd) {
    let {
      children
    } = parent;
    let before = children.length ? children[fromI] : null;
    let last = insert.length ? insert[insert.length - 1] : null;
    let breakAtEnd = last ? last.breakAfter : breakAtStart;
    // Change within a single child
    if (fromI == toI && before && !breakAtStart && !breakAtEnd && insert.length < 2 && before.merge(fromOff, toOff, insert.length ? last : null, fromOff == 0, openStart, openEnd)) return;
    if (toI < children.length) {
      let after = children[toI];
      // Make sure the end of the child after the update is preserved in `after`
      if (after && (toOff < after.length || after.breakAfter && (last === null || last === void 0 ? void 0 : last.breakAfter))) {
        // If we're splitting a child, separate part of it to avoid that
        // being mangled when updating the child before the update.
        if (fromI == toI) {
          after = after.split(toOff);
          toOff = 0;
        }
        // If the element after the replacement should be merged with
        // the last replacing element, update `content`
        if (!breakAtEnd && last && after.merge(0, toOff, last, true, 0, openEnd)) {
          insert[insert.length - 1] = after;
        } else {
          // Remove the start of the after element, if necessary, and
          // add it to `content`.
          if (toOff || after.children.length && !after.children[0].length) after.merge(0, toOff, null, false, 0, openEnd);
          insert.push(after);
        }
      } else if (after === null || after === void 0 ? void 0 : after.breakAfter) {
        // The element at `toI` is entirely covered by this range.
        // Preserve its line break, if any.
        if (last) last.breakAfter = 1;else breakAtStart = 1;
      }
      // Since we've handled the next element from the current elements
      // now, make sure `toI` points after that.
      toI++;
    }
    if (before) {
      before.breakAfter = breakAtStart;
      if (fromOff > 0) {
        if (!breakAtStart && insert.length && before.merge(fromOff, before.length, insert[0], false, openStart, 0)) {
          before.breakAfter = insert.shift().breakAfter;
        } else if (fromOff < before.length || before.children.length && before.children[before.children.length - 1].length == 0) {
          before.merge(fromOff, before.length, null, false, openStart, 0);
        }
        fromI++;
      }
    }
    // Try to merge widgets on the boundaries of the replacement
    while (fromI < toI && insert.length) {
      if (children[toI - 1].become(insert[insert.length - 1])) {
        toI--;
        insert.pop();
        openEnd = insert.length ? 0 : openStart;
      } else if (children[fromI].become(insert[0])) {
        fromI++;
        insert.shift();
        openStart = insert.length ? 0 : openEnd;
      } else {
        break;
      }
    }
    if (!insert.length && fromI && toI < children.length && !children[fromI - 1].breakAfter && children[toI].merge(0, 0, children[fromI - 1], false, openStart, openEnd)) fromI--;
    if (fromI < toI || insert.length) parent.replaceChildren(fromI, toI, insert);
  }
  function mergeChildrenInto(parent, from, to, insert, openStart, openEnd) {
    let cur = parent.childCursor();
    let {
      i: toI,
      off: toOff
    } = cur.findPos(to, 1);
    let {
      i: fromI,
      off: fromOff
    } = cur.findPos(from, -1);
    let dLen = from - to;
    for (let view of insert) dLen += view.length;
    parent.length += dLen;
    replaceRange(parent, fromI, fromOff, toI, toOff, insert, 0, openStart, openEnd);
  }
  let nav = typeof navigator != "undefined" ? navigator : {
    userAgent: "",
    vendor: "",
    platform: ""
  };
  let doc = typeof document != "undefined" ? document : {
    documentElement: {
      style: {}
    }
  };
  const ie_edge = /*@__PURE__*//Edge\/(\d+)/.exec(nav.userAgent);
  const ie_upto10 = /*@__PURE__*//MSIE \d/.test(nav.userAgent);
  const ie_11up = /*@__PURE__*//Trident\/(?:[7-9]|\d{2,})\..*rv:(\d+)/.exec(nav.userAgent);
  const ie = !!(ie_upto10 || ie_11up || ie_edge);
  const gecko = !ie && /*@__PURE__*//gecko\/(\d+)/i.test(nav.userAgent);
  const chrome = !ie && /*@__PURE__*//Chrome\/(\d+)/.exec(nav.userAgent);
  const webkit = "webkitFontSmoothing" in doc.documentElement.style;
  const safari = !ie && /*@__PURE__*//Apple Computer/.test(nav.vendor);
  const ios = safari && (/*@__PURE__*//Mobile\/\w+/.test(nav.userAgent) || nav.maxTouchPoints > 2);
  var browser = {
    mac: ios || /*@__PURE__*//Mac/.test(nav.platform),
    windows: /*@__PURE__*//Win/.test(nav.platform),
    linux: /*@__PURE__*//Linux|X11/.test(nav.platform),
    ie,
    ie_version: ie_upto10 ? doc.documentMode || 6 : ie_11up ? +ie_11up[1] : ie_edge ? +ie_edge[1] : 0,
    gecko,
    gecko_version: gecko ? +(/*@__PURE__*//Firefox\/(\d+)/.exec(nav.userAgent) || [0, 0])[1] : 0,
    chrome: !!chrome,
    chrome_version: chrome ? +chrome[1] : 0,
    ios,
    android: /*@__PURE__*//Android\b/.test(nav.userAgent),
    webkit,
    safari,
    webkit_version: webkit ? +(/*@__PURE__*//\bAppleWebKit\/(\d+)/.exec(nav.userAgent) || [0, 0])[1] : 0,
    tabSize: doc.documentElement.style.tabSize != null ? "tab-size" : "-moz-tab-size"
  };
  const MaxJoinLen = 256;
  class TextView extends ContentView {
    constructor(text) {
      super();
      this.text = text;
    }
    get length() {
      return this.text.length;
    }
    createDOM(textDOM) {
      this.setDOM(textDOM || document.createTextNode(this.text));
    }
    sync(view, track) {
      if (!this.dom) this.createDOM();
      if (this.dom.nodeValue != this.text) {
        if (track && track.node == this.dom) track.written = true;
        this.dom.nodeValue = this.text;
      }
    }
    reuseDOM(dom) {
      if (dom.nodeType == 3) this.createDOM(dom);
    }
    merge(from, to, source) {
      if (this.flags & 8 /* ViewFlag.Composition */ || source && (!(source instanceof TextView) || this.length - (to - from) + source.length > MaxJoinLen || source.flags & 8 /* ViewFlag.Composition */)) return false;
      this.text = this.text.slice(0, from) + (source ? source.text : "") + this.text.slice(to);
      this.markDirty();
      return true;
    }
    split(from) {
      let result = new TextView(this.text.slice(from));
      this.text = this.text.slice(0, from);
      this.markDirty();
      result.flags |= this.flags & 8 /* ViewFlag.Composition */;
      return result;
    }
    localPosFromDOM(node, offset) {
      return node == this.dom ? offset : offset ? this.text.length : 0;
    }
    domAtPos(pos) {
      return new DOMPos(this.dom, pos);
    }
    domBoundsAround(_from, _to, offset) {
      return {
        from: offset,
        to: offset + this.length,
        startDOM: this.dom,
        endDOM: this.dom.nextSibling
      };
    }
    coordsAt(pos, side) {
      return textCoords(this.dom, pos, side);
    }
  }
  class MarkView extends ContentView {
    constructor(mark, children = [], length = 0) {
      super();
      this.mark = mark;
      this.children = children;
      this.length = length;
      for (let ch of children) ch.setParent(this);
    }
    setAttrs(dom) {
      clearAttributes(dom);
      if (this.mark.class) dom.className = this.mark.class;
      if (this.mark.attrs) for (let name in this.mark.attrs) dom.setAttribute(name, this.mark.attrs[name]);
      return dom;
    }
    canReuseDOM(other) {
      return super.canReuseDOM(other) && !((this.flags | other.flags) & 8 /* ViewFlag.Composition */);
    }
    reuseDOM(node) {
      if (node.nodeName == this.mark.tagName.toUpperCase()) {
        this.setDOM(node);
        this.flags |= 4 /* ViewFlag.AttrsDirty */ | 2 /* ViewFlag.NodeDirty */;
      }
    }
    sync(view, track) {
      if (!this.dom) this.setDOM(this.setAttrs(document.createElement(this.mark.tagName)));else if (this.flags & 4 /* ViewFlag.AttrsDirty */) this.setAttrs(this.dom);
      super.sync(view, track);
    }
    merge(from, to, source, _hasStart, openStart, openEnd) {
      if (source && (!(source instanceof MarkView && source.mark.eq(this.mark)) || from && openStart <= 0 || to < this.length && openEnd <= 0)) return false;
      mergeChildrenInto(this, from, to, source ? source.children.slice() : [], openStart - 1, openEnd - 1);
      this.markDirty();
      return true;
    }
    split(from) {
      let result = [],
        off = 0,
        detachFrom = -1,
        i = 0;
      for (let elt of this.children) {
        let end = off + elt.length;
        if (end > from) result.push(off < from ? elt.split(from - off) : elt);
        if (detachFrom < 0 && off >= from) detachFrom = i;
        off = end;
        i++;
      }
      let length = this.length - from;
      this.length = from;
      if (detachFrom > -1) {
        this.children.length = detachFrom;
        this.markDirty();
      }
      return new MarkView(this.mark, result, length);
    }
    domAtPos(pos) {
      return inlineDOMAtPos(this, pos);
    }
    coordsAt(pos, side) {
      return coordsInChildren(this, pos, side);
    }
  }
  function textCoords(text, pos, side) {
    let length = text.nodeValue.length;
    if (pos > length) pos = length;
    let from = pos,
      to = pos,
      flatten = 0;
    if (pos == 0 && side < 0 || pos == length && side >= 0) {
      if (!(browser.chrome || browser.gecko)) {
        // These browsers reliably return valid rectangles for empty ranges
        if (pos) {
          from--;
          flatten = 1;
        } // FIXME this is wrong in RTL text
        else if (to < length) {
          to++;
          flatten = -1;
        }
      }
    } else {
      if (side < 0) from--;else if (to < length) to++;
    }
    let rects = textRange(text, from, to).getClientRects();
    if (!rects.length) return null;
    let rect = rects[(flatten ? flatten < 0 : side >= 0) ? 0 : rects.length - 1];
    if (browser.safari && !flatten && rect.width == 0) rect = Array.prototype.find.call(rects, r => r.width) || rect;
    return flatten ? flattenRect(rect, flatten < 0) : rect || null;
  }
  // Also used for collapsed ranges that don't have a placeholder widget!
  class WidgetView extends ContentView {
    static create(widget, length, side) {
      return new WidgetView(widget, length, side);
    }
    constructor(widget, length, side) {
      super();
      this.widget = widget;
      this.length = length;
      this.side = side;
      this.prevWidget = null;
    }
    split(from) {
      let result = WidgetView.create(this.widget, this.length - from, this.side);
      this.length -= from;
      return result;
    }
    sync(view) {
      if (!this.dom || !this.widget.updateDOM(this.dom, view)) {
        if (this.dom && this.prevWidget) this.prevWidget.destroy(this.dom);
        this.prevWidget = null;
        this.setDOM(this.widget.toDOM(view));
        if (!this.widget.editable) this.dom.contentEditable = "false";
      }
    }
    getSide() {
      return this.side;
    }
    merge(from, to, source, hasStart, openStart, openEnd) {
      if (source && (!(source instanceof WidgetView) || !this.widget.compare(source.widget) || from > 0 && openStart <= 0 || to < this.length && openEnd <= 0)) return false;
      this.length = from + (source ? source.length : 0) + (this.length - to);
      return true;
    }
    become(other) {
      if (other instanceof WidgetView && other.side == this.side && this.widget.constructor == other.widget.constructor) {
        if (!this.widget.compare(other.widget)) this.markDirty(true);
        if (this.dom && !this.prevWidget) this.prevWidget = this.widget;
        this.widget = other.widget;
        this.length = other.length;
        return true;
      }
      return false;
    }
    ignoreMutation() {
      return true;
    }
    ignoreEvent(event) {
      return this.widget.ignoreEvent(event);
    }
    get overrideDOMText() {
      if (this.length == 0) return Text.empty;
      let top = this;
      while (top.parent) top = top.parent;
      let {
          view
        } = top,
        text = view && view.state.doc,
        start = this.posAtStart;
      return text ? text.slice(start, start + this.length) : Text.empty;
    }
    domAtPos(pos) {
      return (this.length ? pos == 0 : this.side > 0) ? DOMPos.before(this.dom) : DOMPos.after(this.dom, pos == this.length);
    }
    domBoundsAround() {
      return null;
    }
    coordsAt(pos, side) {
      let custom = this.widget.coordsAt(this.dom, pos, side);
      if (custom) return custom;
      let rects = this.dom.getClientRects(),
        rect = null;
      if (!rects.length) return null;
      let fromBack = this.side ? this.side < 0 : pos > 0;
      for (let i = fromBack ? rects.length - 1 : 0;; i += fromBack ? -1 : 1) {
        rect = rects[i];
        if (pos > 0 ? i == 0 : i == rects.length - 1 || rect.top < rect.bottom) break;
      }
      return flattenRect(rect, !fromBack);
    }
    get isEditable() {
      return false;
    }
    get isWidget() {
      return true;
    }
    get isHidden() {
      return this.widget.isHidden;
    }
    destroy() {
      super.destroy();
      if (this.dom) this.widget.destroy(this.dom);
    }
  }
  // These are drawn around uneditable widgets to avoid a number of
  // browser bugs that show up when the cursor is directly next to
  // uneditable inline content.
  class WidgetBufferView extends ContentView {
    constructor(side) {
      super();
      this.side = side;
    }
    get length() {
      return 0;
    }
    merge() {
      return false;
    }
    become(other) {
      return other instanceof WidgetBufferView && other.side == this.side;
    }
    split() {
      return new WidgetBufferView(this.side);
    }
    sync() {
      if (!this.dom) {
        let dom = document.createElement("img");
        dom.className = "cm-widgetBuffer";
        dom.setAttribute("aria-hidden", "true");
        this.setDOM(dom);
      }
    }
    getSide() {
      return this.side;
    }
    domAtPos(pos) {
      return this.side > 0 ? DOMPos.before(this.dom) : DOMPos.after(this.dom);
    }
    localPosFromDOM() {
      return 0;
    }
    domBoundsAround() {
      return null;
    }
    coordsAt(pos) {
      return this.dom.getBoundingClientRect();
    }
    get overrideDOMText() {
      return Text.empty;
    }
    get isHidden() {
      return true;
    }
  }
  TextView.prototype.children = WidgetView.prototype.children = WidgetBufferView.prototype.children = noChildren;
  function inlineDOMAtPos(parent, pos) {
    let dom = parent.dom,
      {
        children
      } = parent,
      i = 0;
    for (let off = 0; i < children.length; i++) {
      let child = children[i],
        end = off + child.length;
      if (end == off && child.getSide() <= 0) continue;
      if (pos > off && pos < end && child.dom.parentNode == dom) return child.domAtPos(pos - off);
      if (pos <= off) break;
      off = end;
    }
    for (let j = i; j > 0; j--) {
      let prev = children[j - 1];
      if (prev.dom.parentNode == dom) return prev.domAtPos(prev.length);
    }
    for (let j = i; j < children.length; j++) {
      let next = children[j];
      if (next.dom.parentNode == dom) return next.domAtPos(0);
    }
    return new DOMPos(dom, 0);
  }
  // Assumes `view`, if a mark view, has precisely 1 child.
  function joinInlineInto(parent, view, open) {
    let last,
      {
        children
      } = parent;
    if (open > 0 && view instanceof MarkView && children.length && (last = children[children.length - 1]) instanceof MarkView && last.mark.eq(view.mark)) {
      joinInlineInto(last, view.children[0], open - 1);
    } else {
      children.push(view);
      view.setParent(parent);
    }
    parent.length += view.length;
  }
  function coordsInChildren(view, pos, side) {
    let before = null,
      beforePos = -1,
      after = null,
      afterPos = -1;
    function scan(view, pos) {
      for (let i = 0, off = 0; i < view.children.length && off <= pos; i++) {
        let child = view.children[i],
          end = off + child.length;
        if (end >= pos) {
          if (child.children.length) {
            scan(child, pos - off);
          } else if ((!after || after.isHidden && side > 0) && (end > pos || off == end && child.getSide() > 0)) {
            after = child;
            afterPos = pos - off;
          } else if (off < pos || off == end && child.getSide() < 0 && !child.isHidden) {
            before = child;
            beforePos = pos - off;
          }
        }
        off = end;
      }
    }
    scan(view, pos);
    let target = (side < 0 ? before : after) || before || after;
    if (target) return target.coordsAt(Math.max(0, target == before ? beforePos : afterPos), side);
    return fallbackRect(view);
  }
  function fallbackRect(view) {
    let last = view.dom.lastChild;
    if (!last) return view.dom.getBoundingClientRect();
    let rects = clientRectsFor(last);
    return rects[rects.length - 1] || null;
  }
  function combineAttrs(source, target) {
    for (let name in source) {
      if (name == "class" && target.class) target.class += " " + source.class;else if (name == "style" && target.style) target.style += ";" + source.style;else target[name] = source[name];
    }
    return target;
  }
  const noAttrs = /*@__PURE__*/Object.create(null);
  function attrsEq(a, b, ignore) {
    if (a == b) return true;
    if (!a) a = noAttrs;
    if (!b) b = noAttrs;
    let keysA = Object.keys(a),
      keysB = Object.keys(b);
    if (keysA.length - (ignore && keysA.indexOf(ignore) > -1 ? 1 : 0) != keysB.length - (ignore && keysB.indexOf(ignore) > -1 ? 1 : 0)) return false;
    for (let key of keysA) {
      if (key != ignore && (keysB.indexOf(key) == -1 || a[key] !== b[key])) return false;
    }
    return true;
  }
  function updateAttrs(dom, prev, attrs) {
    let changed = false;
    if (prev) for (let name in prev) if (!(attrs && name in attrs)) {
      changed = true;
      if (name == "style") dom.style.cssText = "";else dom.removeAttribute(name);
    }
    if (attrs) for (let name in attrs) if (!(prev && prev[name] == attrs[name])) {
      changed = true;
      if (name == "style") dom.style.cssText = attrs[name];else dom.setAttribute(name, attrs[name]);
    }
    return changed;
  }
  function getAttrs(dom) {
    let attrs = Object.create(null);
    for (let i = 0; i < dom.attributes.length; i++) {
      let attr = dom.attributes[i];
      attrs[attr.name] = attr.value;
    }
    return attrs;
  }

  /**
  Widgets added to the content are described by subclasses of this
  class. Using a description object like that makes it possible to
  delay creating of the DOM structure for a widget until it is
  needed, and to avoid redrawing widgets even if the decorations
  that define them are recreated.
  */
  class WidgetType {
    /**
    Compare this instance to another instance of the same type.
    (TypeScript can't express this, but only instances of the same
    specific class will be passed to this method.) This is used to
    avoid redrawing widgets when they are replaced by a new
    decoration of the same type. The default implementation just
    returns `false`, which will cause new instances of the widget to
    always be redrawn.
    */
    eq(widget) {
      return false;
    }
    /**
    Update a DOM element created by a widget of the same type (but
    different, non-`eq` content) to reflect this widget. May return
    true to indicate that it could update, false to indicate it
    couldn't (in which case the widget will be redrawn). The default
    implementation just returns false.
    */
    updateDOM(dom, view) {
      return false;
    }
    /**
    @internal
    */
    compare(other) {
      return this == other || this.constructor == other.constructor && this.eq(other);
    }
    /**
    The estimated height this widget will have, to be used when
    estimating the height of content that hasn't been drawn. May
    return -1 to indicate you don't know. The default implementation
    returns -1.
    */
    get estimatedHeight() {
      return -1;
    }
    /**
    For inline widgets that are displayed inline (as opposed to
    `inline-block`) and introduce line breaks (through `<br>` tags
    or textual newlines), this must indicate the amount of line
    breaks they introduce. Defaults to 0.
    */
    get lineBreaks() {
      return 0;
    }
    /**
    Can be used to configure which kinds of events inside the widget
    should be ignored by the editor. The default is to ignore all
    events.
    */
    ignoreEvent(event) {
      return true;
    }
    /**
    Override the way screen coordinates for positions at/in the
    widget are found. `pos` will be the offset into the widget, and
    `side` the side of the position that is being queried—less than
    zero for before, greater than zero for after, and zero for
    directly at that position.
    */
    coordsAt(dom, pos, side) {
      return null;
    }
    /**
    @internal
    */
    get isHidden() {
      return false;
    }
    /**
    @internal
    */
    get editable() {
      return false;
    }
    /**
    This is called when the an instance of the widget is removed
    from the editor view.
    */
    destroy(dom) {}
  }
  /**
  The different types of blocks that can occur in an editor view.
  */
  var BlockType = /*@__PURE__*/function (BlockType) {
    /**
    A line of text.
    */
    BlockType[BlockType["Text"] = 0] = "Text";
    /**
    A block widget associated with the position after it.
    */
    BlockType[BlockType["WidgetBefore"] = 1] = "WidgetBefore";
    /**
    A block widget associated with the position before it.
    */
    BlockType[BlockType["WidgetAfter"] = 2] = "WidgetAfter";
    /**
    A block widget [replacing](https://codemirror.net/6/docs/ref/#view.Decoration^replace) a range of content.
    */
    BlockType[BlockType["WidgetRange"] = 3] = "WidgetRange";
    return BlockType;
  }(BlockType || (BlockType = {}));
  /**
  A decoration provides information on how to draw or style a piece
  of content. You'll usually use it wrapped in a
  [`Range`](https://codemirror.net/6/docs/ref/#state.Range), which adds a start and end position.
  @nonabstract
  */
  class Decoration extends RangeValue {
    constructor(
    /**
    @internal
    */
    startSide,
    /**
    @internal
    */
    endSide,
    /**
    @internal
    */
    widget,
    /**
    The config object used to create this decoration. You can
    include additional properties in there to store metadata about
    your decoration.
    */
    spec) {
      super();
      this.startSide = startSide;
      this.endSide = endSide;
      this.widget = widget;
      this.spec = spec;
    }
    /**
    @internal
    */
    get heightRelevant() {
      return false;
    }
    /**
    Create a mark decoration, which influences the styling of the
    content in its range. Nested mark decorations will cause nested
    DOM elements to be created. Nesting order is determined by
    precedence of the [facet](https://codemirror.net/6/docs/ref/#view.EditorView^decorations), with
    the higher-precedence decorations creating the inner DOM nodes.
    Such elements are split on line boundaries and on the boundaries
    of lower-precedence decorations.
    */
    static mark(spec) {
      return new MarkDecoration(spec);
    }
    /**
    Create a widget decoration, which displays a DOM element at the
    given position.
    */
    static widget(spec) {
      let side = Math.max(-10000, Math.min(10000, spec.side || 0)),
        block = !!spec.block;
      side += block && !spec.inlineOrder ? side > 0 ? 300000000 /* Side.BlockAfter */ : -400000000 /* Side.BlockBefore */ : side > 0 ? 100000000 /* Side.InlineAfter */ : -100000000 /* Side.InlineBefore */;
      return new PointDecoration(spec, side, side, block, spec.widget || null, false);
    }
    /**
    Create a replace decoration which replaces the given range with
    a widget, or simply hides it.
    */
    static replace(spec) {
      let block = !!spec.block,
        startSide,
        endSide;
      if (spec.isBlockGap) {
        startSide = -500000000 /* Side.GapStart */;
        endSide = 400000000 /* Side.GapEnd */;
      } else {
        let {
          start,
          end
        } = getInclusive(spec, block);
        startSide = (start ? block ? -300000000 /* Side.BlockIncStart */ : -1 /* Side.InlineIncStart */ : 500000000 /* Side.NonIncStart */) - 1;
        endSide = (end ? block ? 200000000 /* Side.BlockIncEnd */ : 1 /* Side.InlineIncEnd */ : -600000000 /* Side.NonIncEnd */) + 1;
      }
      return new PointDecoration(spec, startSide, endSide, block, spec.widget || null, true);
    }
    /**
    Create a line decoration, which can add DOM attributes to the
    line starting at the given position.
    */
    static line(spec) {
      return new LineDecoration(spec);
    }
    /**
    Build a [`DecorationSet`](https://codemirror.net/6/docs/ref/#view.DecorationSet) from the given
    decorated range or ranges. If the ranges aren't already sorted,
    pass `true` for `sort` to make the library sort them for you.
    */
    static set(of, sort = false) {
      return RangeSet.of(of, sort);
    }
    /**
    @internal
    */
    hasHeight() {
      return this.widget ? this.widget.estimatedHeight > -1 : false;
    }
  }
  /**
  The empty set of decorations.
  */
  Decoration.none = RangeSet.empty;
  class MarkDecoration extends Decoration {
    constructor(spec) {
      let {
        start,
        end
      } = getInclusive(spec);
      super(start ? -1 /* Side.InlineIncStart */ : 500000000 /* Side.NonIncStart */, end ? 1 /* Side.InlineIncEnd */ : -600000000 /* Side.NonIncEnd */, null, spec);
      this.tagName = spec.tagName || "span";
      this.class = spec.class || "";
      this.attrs = spec.attributes || null;
    }
    eq(other) {
      var _a, _b;
      return this == other || other instanceof MarkDecoration && this.tagName == other.tagName && (this.class || ((_a = this.attrs) === null || _a === void 0 ? void 0 : _a.class)) == (other.class || ((_b = other.attrs) === null || _b === void 0 ? void 0 : _b.class)) && attrsEq(this.attrs, other.attrs, "class");
    }
    range(from, to = from) {
      if (from >= to) throw new RangeError("Mark decorations may not be empty");
      return super.range(from, to);
    }
  }
  MarkDecoration.prototype.point = false;
  class LineDecoration extends Decoration {
    constructor(spec) {
      super(-200000000 /* Side.Line */, -200000000 /* Side.Line */, null, spec);
    }
    eq(other) {
      return other instanceof LineDecoration && this.spec.class == other.spec.class && attrsEq(this.spec.attributes, other.spec.attributes);
    }
    range(from, to = from) {
      if (to != from) throw new RangeError("Line decoration ranges must be zero-length");
      return super.range(from, to);
    }
  }
  LineDecoration.prototype.mapMode = MapMode.TrackBefore;
  LineDecoration.prototype.point = true;
  class PointDecoration extends Decoration {
    constructor(spec, startSide, endSide, block, widget, isReplace) {
      super(startSide, endSide, widget, spec);
      this.block = block;
      this.isReplace = isReplace;
      this.mapMode = !block ? MapMode.TrackDel : startSide <= 0 ? MapMode.TrackBefore : MapMode.TrackAfter;
    }
    // Only relevant when this.block == true
    get type() {
      return this.startSide != this.endSide ? BlockType.WidgetRange : this.startSide <= 0 ? BlockType.WidgetBefore : BlockType.WidgetAfter;
    }
    get heightRelevant() {
      return this.block || !!this.widget && (this.widget.estimatedHeight >= 5 || this.widget.lineBreaks > 0);
    }
    eq(other) {
      return other instanceof PointDecoration && widgetsEq(this.widget, other.widget) && this.block == other.block && this.startSide == other.startSide && this.endSide == other.endSide;
    }
    range(from, to = from) {
      if (this.isReplace && (from > to || from == to && this.startSide > 0 && this.endSide <= 0)) throw new RangeError("Invalid range for replacement decoration");
      if (!this.isReplace && to != from) throw new RangeError("Widget decorations can only have zero-length ranges");
      return super.range(from, to);
    }
  }
  PointDecoration.prototype.point = true;
  function getInclusive(spec, block = false) {
    let {
      inclusiveStart: start,
      inclusiveEnd: end
    } = spec;
    if (start == null) start = spec.inclusive;
    if (end == null) end = spec.inclusive;
    return {
      start: start !== null && start !== void 0 ? start : block,
      end: end !== null && end !== void 0 ? end : block
    };
  }
  function widgetsEq(a, b) {
    return a == b || !!(a && b && a.compare(b));
  }
  function addRange(from, to, ranges, margin = 0) {
    let last = ranges.length - 1;
    if (last >= 0 && ranges[last] + margin >= from) ranges[last] = Math.max(ranges[last], to);else ranges.push(from, to);
  }
  class LineView extends ContentView {
    constructor() {
      super(...arguments);
      this.children = [];
      this.length = 0;
      this.prevAttrs = undefined;
      this.attrs = null;
      this.breakAfter = 0;
    }
    // Consumes source
    merge(from, to, source, hasStart, openStart, openEnd) {
      if (source) {
        if (!(source instanceof LineView)) return false;
        if (!this.dom) source.transferDOM(this); // Reuse source.dom when appropriate
      }
      if (hasStart) this.setDeco(source ? source.attrs : null);
      mergeChildrenInto(this, from, to, source ? source.children.slice() : [], openStart, openEnd);
      return true;
    }
    split(at) {
      let end = new LineView();
      end.breakAfter = this.breakAfter;
      if (this.length == 0) return end;
      let {
        i,
        off
      } = this.childPos(at);
      if (off) {
        end.append(this.children[i].split(off), 0);
        this.children[i].merge(off, this.children[i].length, null, false, 0, 0);
        i++;
      }
      for (let j = i; j < this.children.length; j++) end.append(this.children[j], 0);
      while (i > 0 && this.children[i - 1].length == 0) this.children[--i].destroy();
      this.children.length = i;
      this.markDirty();
      this.length = at;
      return end;
    }
    transferDOM(other) {
      if (!this.dom) return;
      this.markDirty();
      other.setDOM(this.dom);
      other.prevAttrs = this.prevAttrs === undefined ? this.attrs : this.prevAttrs;
      this.prevAttrs = undefined;
      this.dom = null;
    }
    setDeco(attrs) {
      if (!attrsEq(this.attrs, attrs)) {
        if (this.dom) {
          this.prevAttrs = this.attrs;
          this.markDirty();
        }
        this.attrs = attrs;
      }
    }
    append(child, openStart) {
      joinInlineInto(this, child, openStart);
    }
    // Only called when building a line view in ContentBuilder
    addLineDeco(deco) {
      let attrs = deco.spec.attributes,
        cls = deco.spec.class;
      if (attrs) this.attrs = combineAttrs(attrs, this.attrs || {});
      if (cls) this.attrs = combineAttrs({
        class: cls
      }, this.attrs || {});
    }
    domAtPos(pos) {
      return inlineDOMAtPos(this, pos);
    }
    reuseDOM(node) {
      if (node.nodeName == "DIV") {
        this.setDOM(node);
        this.flags |= 4 /* ViewFlag.AttrsDirty */ | 2 /* ViewFlag.NodeDirty */;
      }
    }
    sync(view, track) {
      var _a;
      if (!this.dom) {
        this.setDOM(document.createElement("div"));
        this.dom.className = "cm-line";
        this.prevAttrs = this.attrs ? null : undefined;
      } else if (this.flags & 4 /* ViewFlag.AttrsDirty */) {
        clearAttributes(this.dom);
        this.dom.className = "cm-line";
        this.prevAttrs = this.attrs ? null : undefined;
      }
      if (this.prevAttrs !== undefined) {
        updateAttrs(this.dom, this.prevAttrs, this.attrs);
        this.dom.classList.add("cm-line");
        this.prevAttrs = undefined;
      }
      super.sync(view, track);
      let last = this.dom.lastChild;
      while (last && ContentView.get(last) instanceof MarkView) last = last.lastChild;
      if (!last || !this.length || last.nodeName != "BR" && ((_a = ContentView.get(last)) === null || _a === void 0 ? void 0 : _a.isEditable) == false && (!browser.ios || !this.children.some(ch => ch instanceof TextView))) {
        let hack = document.createElement("BR");
        hack.cmIgnore = true;
        this.dom.appendChild(hack);
      }
    }
    measureTextSize() {
      if (this.children.length == 0 || this.length > 20) return null;
      let totalWidth = 0,
        textHeight;
      for (let child of this.children) {
        if (!(child instanceof TextView) || /[^ -~]/.test(child.text)) return null;
        let rects = clientRectsFor(child.dom);
        if (rects.length != 1) return null;
        totalWidth += rects[0].width;
        textHeight = rects[0].height;
      }
      return !totalWidth ? null : {
        lineHeight: this.dom.getBoundingClientRect().height,
        charWidth: totalWidth / this.length,
        textHeight
      };
    }
    coordsAt(pos, side) {
      let rect = coordsInChildren(this, pos, side);
      // Correct rectangle height for empty lines when the returned
      // height is larger than the text height.
      if (!this.children.length && rect && this.parent) {
        let {
            heightOracle
          } = this.parent.view.viewState,
          height = rect.bottom - rect.top;
        if (Math.abs(height - heightOracle.lineHeight) < 2 && heightOracle.textHeight < height) {
          let dist = (height - heightOracle.textHeight) / 2;
          return {
            top: rect.top + dist,
            bottom: rect.bottom - dist,
            left: rect.left,
            right: rect.left
          };
        }
      }
      return rect;
    }
    become(other) {
      return other instanceof LineView && this.children.length == 0 && other.children.length == 0 && attrsEq(this.attrs, other.attrs) && this.breakAfter == other.breakAfter;
    }
    covers() {
      return true;
    }
    static find(docView, pos) {
      for (let i = 0, off = 0; i < docView.children.length; i++) {
        let block = docView.children[i],
          end = off + block.length;
        if (end >= pos) {
          if (block instanceof LineView) return block;
          if (end > pos) break;
        }
        off = end + block.breakAfter;
      }
      return null;
    }
  }
  class BlockWidgetView extends ContentView {
    constructor(widget, length, deco) {
      super();
      this.widget = widget;
      this.length = length;
      this.deco = deco;
      this.breakAfter = 0;
      this.prevWidget = null;
    }
    merge(from, to, source, _takeDeco, openStart, openEnd) {
      if (source && (!(source instanceof BlockWidgetView) || !this.widget.compare(source.widget) || from > 0 && openStart <= 0 || to < this.length && openEnd <= 0)) return false;
      this.length = from + (source ? source.length : 0) + (this.length - to);
      return true;
    }
    domAtPos(pos) {
      return pos == 0 ? DOMPos.before(this.dom) : DOMPos.after(this.dom, pos == this.length);
    }
    split(at) {
      let len = this.length - at;
      this.length = at;
      let end = new BlockWidgetView(this.widget, len, this.deco);
      end.breakAfter = this.breakAfter;
      return end;
    }
    get children() {
      return noChildren;
    }
    sync(view) {
      if (!this.dom || !this.widget.updateDOM(this.dom, view)) {
        if (this.dom && this.prevWidget) this.prevWidget.destroy(this.dom);
        this.prevWidget = null;
        this.setDOM(this.widget.toDOM(view));
        if (!this.widget.editable) this.dom.contentEditable = "false";
      }
    }
    get overrideDOMText() {
      return this.parent ? this.parent.view.state.doc.slice(this.posAtStart, this.posAtEnd) : Text.empty;
    }
    domBoundsAround() {
      return null;
    }
    become(other) {
      if (other instanceof BlockWidgetView && other.widget.constructor == this.widget.constructor) {
        if (!other.widget.compare(this.widget)) this.markDirty(true);
        if (this.dom && !this.prevWidget) this.prevWidget = this.widget;
        this.widget = other.widget;
        this.length = other.length;
        this.deco = other.deco;
        this.breakAfter = other.breakAfter;
        return true;
      }
      return false;
    }
    ignoreMutation() {
      return true;
    }
    ignoreEvent(event) {
      return this.widget.ignoreEvent(event);
    }
    get isEditable() {
      return false;
    }
    get isWidget() {
      return true;
    }
    coordsAt(pos, side) {
      let custom = this.widget.coordsAt(this.dom, pos, side);
      if (custom) return custom;
      if (this.widget instanceof BlockGapWidget) return null;
      return flattenRect(this.dom.getBoundingClientRect(), this.length ? pos == 0 : side <= 0);
    }
    destroy() {
      super.destroy();
      if (this.dom) this.widget.destroy(this.dom);
    }
    covers(side) {
      let {
        startSide,
        endSide
      } = this.deco;
      return startSide == endSide ? false : side < 0 ? startSide < 0 : endSide > 0;
    }
  }
  class BlockGapWidget extends WidgetType {
    constructor(height) {
      super();
      this.height = height;
    }
    toDOM() {
      let elt = document.createElement("div");
      elt.className = "cm-gap";
      this.updateDOM(elt);
      return elt;
    }
    eq(other) {
      return other.height == this.height;
    }
    updateDOM(elt) {
      elt.style.height = this.height + "px";
      return true;
    }
    get editable() {
      return true;
    }
    get estimatedHeight() {
      return this.height;
    }
    ignoreEvent() {
      return false;
    }
  }
  class ContentBuilder {
    constructor(doc, pos, end, disallowBlockEffectsFor) {
      this.doc = doc;
      this.pos = pos;
      this.end = end;
      this.disallowBlockEffectsFor = disallowBlockEffectsFor;
      this.content = [];
      this.curLine = null;
      this.breakAtStart = 0;
      this.pendingBuffer = 0 /* Buf.No */;
      this.bufferMarks = [];
      // Set to false directly after a widget that covers the position after it
      this.atCursorPos = true;
      this.openStart = -1;
      this.openEnd = -1;
      this.text = "";
      this.textOff = 0;
      this.cursor = doc.iter();
      this.skip = pos;
    }
    posCovered() {
      if (this.content.length == 0) return !this.breakAtStart && this.doc.lineAt(this.pos).from != this.pos;
      let last = this.content[this.content.length - 1];
      return !(last.breakAfter || last instanceof BlockWidgetView && last.deco.endSide < 0);
    }
    getLine() {
      if (!this.curLine) {
        this.content.push(this.curLine = new LineView());
        this.atCursorPos = true;
      }
      return this.curLine;
    }
    flushBuffer(active = this.bufferMarks) {
      if (this.pendingBuffer) {
        this.curLine.append(wrapMarks(new WidgetBufferView(-1), active), active.length);
        this.pendingBuffer = 0 /* Buf.No */;
      }
    }
    addBlockWidget(view) {
      this.flushBuffer();
      this.curLine = null;
      this.content.push(view);
    }
    finish(openEnd) {
      if (this.pendingBuffer && openEnd <= this.bufferMarks.length) this.flushBuffer();else this.pendingBuffer = 0 /* Buf.No */;
      if (!this.posCovered() && !(openEnd && this.content.length && this.content[this.content.length - 1] instanceof BlockWidgetView)) this.getLine();
    }
    buildText(length, active, openStart) {
      while (length > 0) {
        if (this.textOff == this.text.length) {
          let {
            value,
            lineBreak,
            done
          } = this.cursor.next(this.skip);
          this.skip = 0;
          if (done) throw new Error("Ran out of text content when drawing inline views");
          if (lineBreak) {
            if (!this.posCovered()) this.getLine();
            if (this.content.length) this.content[this.content.length - 1].breakAfter = 1;else this.breakAtStart = 1;
            this.flushBuffer();
            this.curLine = null;
            this.atCursorPos = true;
            length--;
            continue;
          } else {
            this.text = value;
            this.textOff = 0;
          }
        }
        let take = Math.min(this.text.length - this.textOff, length, 512 /* T.Chunk */);
        this.flushBuffer(active.slice(active.length - openStart));
        this.getLine().append(wrapMarks(new TextView(this.text.slice(this.textOff, this.textOff + take)), active), openStart);
        this.atCursorPos = true;
        this.textOff += take;
        length -= take;
        openStart = 0;
      }
    }
    span(from, to, active, openStart) {
      this.buildText(to - from, active, openStart);
      this.pos = to;
      if (this.openStart < 0) this.openStart = openStart;
    }
    point(from, to, deco, active, openStart, index) {
      if (this.disallowBlockEffectsFor[index] && deco instanceof PointDecoration) {
        if (deco.block) throw new RangeError("Block decorations may not be specified via plugins");
        if (to > this.doc.lineAt(this.pos).to) throw new RangeError("Decorations that replace line breaks may not be specified via plugins");
      }
      let len = to - from;
      if (deco instanceof PointDecoration) {
        if (deco.block) {
          if (deco.startSide > 0 && !this.posCovered()) this.getLine();
          this.addBlockWidget(new BlockWidgetView(deco.widget || NullWidget.block, len, deco));
        } else {
          let view = WidgetView.create(deco.widget || NullWidget.inline, len, len ? 0 : deco.startSide);
          let cursorBefore = this.atCursorPos && !view.isEditable && openStart <= active.length && (from < to || deco.startSide > 0);
          let cursorAfter = !view.isEditable && (from < to || openStart > active.length || deco.startSide <= 0);
          let line = this.getLine();
          if (this.pendingBuffer == 2 /* Buf.IfCursor */ && !cursorBefore && !view.isEditable) this.pendingBuffer = 0 /* Buf.No */;
          this.flushBuffer(active);
          if (cursorBefore) {
            line.append(wrapMarks(new WidgetBufferView(1), active), openStart);
            openStart = active.length + Math.max(0, openStart - active.length);
          }
          line.append(wrapMarks(view, active), openStart);
          this.atCursorPos = cursorAfter;
          this.pendingBuffer = !cursorAfter ? 0 /* Buf.No */ : from < to || openStart > active.length ? 1 /* Buf.Yes */ : 2 /* Buf.IfCursor */;
          if (this.pendingBuffer) this.bufferMarks = active.slice();
        }
      } else if (this.doc.lineAt(this.pos).from == this.pos) {
        // Line decoration
        this.getLine().addLineDeco(deco);
      }
      if (len) {
        // Advance the iterator past the replaced content
        if (this.textOff + len <= this.text.length) {
          this.textOff += len;
        } else {
          this.skip += len - (this.text.length - this.textOff);
          this.text = "";
          this.textOff = 0;
        }
        this.pos = to;
      }
      if (this.openStart < 0) this.openStart = openStart;
    }
    static build(text, from, to, decorations, dynamicDecorationMap) {
      let builder = new ContentBuilder(text, from, to, dynamicDecorationMap);
      builder.openEnd = RangeSet.spans(decorations, from, to, builder);
      if (builder.openStart < 0) builder.openStart = builder.openEnd;
      builder.finish(builder.openEnd);
      return builder;
    }
  }
  function wrapMarks(view, active) {
    for (let mark of active) view = new MarkView(mark, [view], view.length);
    return view;
  }
  class NullWidget extends WidgetType {
    constructor(tag) {
      super();
      this.tag = tag;
    }
    eq(other) {
      return other.tag == this.tag;
    }
    toDOM() {
      return document.createElement(this.tag);
    }
    updateDOM(elt) {
      return elt.nodeName.toLowerCase() == this.tag;
    }
    get isHidden() {
      return true;
    }
  }
  NullWidget.inline = /*@__PURE__*/new NullWidget("span");
  NullWidget.block = /*@__PURE__*/new NullWidget("div");

  /**
  Used to indicate [text direction](https://codemirror.net/6/docs/ref/#view.EditorView.textDirection).
  */
  var Direction = /*@__PURE__*/function (Direction) {
    // (These are chosen to match the base levels, in bidi algorithm
    // terms, of spans in that direction.)
    /**
    Left-to-right.
    */
    Direction[Direction["LTR"] = 0] = "LTR";
    /**
    Right-to-left.
    */
    Direction[Direction["RTL"] = 1] = "RTL";
    return Direction;
  }(Direction || (Direction = {}));
  const LTR = Direction.LTR,
    RTL = Direction.RTL;
  // Decode a string with each type encoded as log2(type)
  function dec(str) {
    let result = [];
    for (let i = 0; i < str.length; i++) result.push(1 << +str[i]);
    return result;
  }
  // Character types for codepoints 0 to 0xf8
  const LowTypes = /*@__PURE__*/dec("88888888888888888888888888888888888666888888787833333333337888888000000000000000000000000008888880000000000000000000000000088888888888888888888888888888888888887866668888088888663380888308888800000000000000000000000800000000000000000000000000000008");
  // Character types for codepoints 0x600 to 0x6f9
  const ArabicTypes = /*@__PURE__*/dec("4444448826627288999999999992222222222222222222222222222222222222222222222229999999999999999999994444444444644222822222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222999999949999999229989999223333333333");
  const Brackets = /*@__PURE__*/Object.create(null),
    BracketStack = [];
  // There's a lot more in
  // https://www.unicode.org/Public/UCD/latest/ucd/BidiBrackets.txt,
  // which are left out to keep code size down.
  for (let p of ["()", "[]", "{}"]) {
    let l = /*@__PURE__*/p.charCodeAt(0),
      r = /*@__PURE__*/p.charCodeAt(1);
    Brackets[l] = r;
    Brackets[r] = -l;
  }
  function charType(ch) {
    return ch <= 0xf7 ? LowTypes[ch] : 0x590 <= ch && ch <= 0x5f4 ? 2 /* T.R */ : 0x600 <= ch && ch <= 0x6f9 ? ArabicTypes[ch - 0x600] : 0x6ee <= ch && ch <= 0x8ac ? 4 /* T.AL */ : 0x2000 <= ch && ch <= 0x200c ? 256 /* T.NI */ : 0xfb50 <= ch && ch <= 0xfdff ? 4 /* T.AL */ : 1 /* T.L */;
  }
  const BidiRE = /[\u0590-\u05f4\u0600-\u06ff\u0700-\u08ac\ufb50-\ufdff]/;
  /**
  Represents a contiguous range of text that has a single direction
  (as in left-to-right or right-to-left).
  */
  class BidiSpan {
    /**
    The direction of this span.
    */
    get dir() {
      return this.level % 2 ? RTL : LTR;
    }
    /**
    @internal
    */
    constructor(
    /**
    The start of the span (relative to the start of the line).
    */
    from,
    /**
    The end of the span.
    */
    to,
    /**
    The ["bidi
    level"](https://unicode.org/reports/tr9/#Basic_Display_Algorithm)
    of the span (in this context, 0 means
    left-to-right, 1 means right-to-left, 2 means left-to-right
    number inside right-to-left text).
    */
    level) {
      this.from = from;
      this.to = to;
      this.level = level;
    }
    /**
    @internal
    */
    side(end, dir) {
      return this.dir == dir == end ? this.to : this.from;
    }
    /**
    @internal
    */
    forward(forward, dir) {
      return forward == (this.dir == dir);
    }
    /**
    @internal
    */
    static find(order, index, level, assoc) {
      let maybe = -1;
      for (let i = 0; i < order.length; i++) {
        let span = order[i];
        if (span.from <= index && span.to >= index) {
          if (span.level == level) return i;
          // When multiple spans match, if assoc != 0, take the one that
          // covers that side, otherwise take the one with the minimum
          // level.
          if (maybe < 0 || (assoc != 0 ? assoc < 0 ? span.from < index : span.to > index : order[maybe].level > span.level)) maybe = i;
        }
      }
      if (maybe < 0) throw new RangeError("Index out of range");
      return maybe;
    }
  }
  function isolatesEq(a, b) {
    if (a.length != b.length) return false;
    for (let i = 0; i < a.length; i++) {
      let iA = a[i],
        iB = b[i];
      if (iA.from != iB.from || iA.to != iB.to || iA.direction != iB.direction || !isolatesEq(iA.inner, iB.inner)) return false;
    }
    return true;
  }
  // Reused array of character types
  const types = [];
  // Fill in the character types (in `types`) from `from` to `to` and
  // apply W normalization rules.
  function computeCharTypes(line, rFrom, rTo, isolates, outerType) {
    for (let iI = 0; iI <= isolates.length; iI++) {
      let from = iI ? isolates[iI - 1].to : rFrom,
        to = iI < isolates.length ? isolates[iI].from : rTo;
      let prevType = iI ? 256 /* T.NI */ : outerType;
      // W1. Examine each non-spacing mark (NSM) in the level run, and
      // change the type of the NSM to the type of the previous
      // character. If the NSM is at the start of the level run, it will
      // get the type of sor.
      // W2. Search backwards from each instance of a European number
      // until the first strong type (R, L, AL, or sor) is found. If an
      // AL is found, change the type of the European number to Arabic
      // number.
      // W3. Change all ALs to R.
      // (Left after this: L, R, EN, AN, ET, CS, NI)
      for (let i = from, prev = prevType, prevStrong = prevType; i < to; i++) {
        let type = charType(line.charCodeAt(i));
        if (type == 512 /* T.NSM */) type = prev;else if (type == 8 /* T.EN */ && prevStrong == 4 /* T.AL */) type = 16 /* T.AN */;
        types[i] = type == 4 /* T.AL */ ? 2 /* T.R */ : type;
        if (type & 7 /* T.Strong */) prevStrong = type;
        prev = type;
      }
      // W5. A sequence of European terminators adjacent to European
      // numbers changes to all European numbers.
      // W6. Otherwise, separators and terminators change to Other
      // Neutral.
      // W7. Search backwards from each instance of a European number
      // until the first strong type (R, L, or sor) is found. If an L is
      // found, then change the type of the European number to L.
      // (Left after this: L, R, EN+AN, NI)
      for (let i = from, prev = prevType, prevStrong = prevType; i < to; i++) {
        let type = types[i];
        if (type == 128 /* T.CS */) {
          if (i < to - 1 && prev == types[i + 1] && prev & 24 /* T.Num */) type = types[i] = prev;else types[i] = 256 /* T.NI */;
        } else if (type == 64 /* T.ET */) {
          let end = i + 1;
          while (end < to && types[end] == 64 /* T.ET */) end++;
          let replace = i && prev == 8 /* T.EN */ || end < rTo && types[end] == 8 /* T.EN */ ? prevStrong == 1 /* T.L */ ? 1 /* T.L */ : 8 /* T.EN */ : 256 /* T.NI */;
          for (let j = i; j < end; j++) types[j] = replace;
          i = end - 1;
        } else if (type == 8 /* T.EN */ && prevStrong == 1 /* T.L */) {
          types[i] = 1 /* T.L */;
        }
        prev = type;
        if (type & 7 /* T.Strong */) prevStrong = type;
      }
    }
  }
  // Process brackets throughout a run sequence.
  function processBracketPairs(line, rFrom, rTo, isolates, outerType) {
    let oppositeType = outerType == 1 /* T.L */ ? 2 /* T.R */ : 1 /* T.L */;
    for (let iI = 0, sI = 0, context = 0; iI <= isolates.length; iI++) {
      let from = iI ? isolates[iI - 1].to : rFrom,
        to = iI < isolates.length ? isolates[iI].from : rTo;
      // N0. Process bracket pairs in an isolating run sequence
      // sequentially in the logical order of the text positions of the
      // opening paired brackets using the logic given below. Within this
      // scope, bidirectional types EN and AN are treated as R.
      for (let i = from, ch, br, type; i < to; i++) {
        // Keeps [startIndex, type, strongSeen] triples for each open
        // bracket on BracketStack.
        if (br = Brackets[ch = line.charCodeAt(i)]) {
          if (br < 0) {
            // Closing bracket
            for (let sJ = sI - 3; sJ >= 0; sJ -= 3) {
              if (BracketStack[sJ + 1] == -br) {
                let flags = BracketStack[sJ + 2];
                let type = flags & 2 /* Bracketed.EmbedInside */ ? outerType : !(flags & 4 /* Bracketed.OppositeInside */) ? 0 : flags & 1 /* Bracketed.OppositeBefore */ ? oppositeType : outerType;
                if (type) types[i] = types[BracketStack[sJ]] = type;
                sI = sJ;
                break;
              }
            }
          } else if (BracketStack.length == 189 /* Bracketed.MaxDepth */) {
            break;
          } else {
            BracketStack[sI++] = i;
            BracketStack[sI++] = ch;
            BracketStack[sI++] = context;
          }
        } else if ((type = types[i]) == 2 /* T.R */ || type == 1 /* T.L */) {
          let embed = type == outerType;
          context = embed ? 0 : 1 /* Bracketed.OppositeBefore */;
          for (let sJ = sI - 3; sJ >= 0; sJ -= 3) {
            let cur = BracketStack[sJ + 2];
            if (cur & 2 /* Bracketed.EmbedInside */) break;
            if (embed) {
              BracketStack[sJ + 2] |= 2 /* Bracketed.EmbedInside */;
            } else {
              if (cur & 4 /* Bracketed.OppositeInside */) break;
              BracketStack[sJ + 2] |= 4 /* Bracketed.OppositeInside */;
            }
          }
        }
      }
    }
  }
  function processNeutrals(rFrom, rTo, isolates, outerType) {
    for (let iI = 0, prev = outerType; iI <= isolates.length; iI++) {
      let from = iI ? isolates[iI - 1].to : rFrom,
        to = iI < isolates.length ? isolates[iI].from : rTo;
      // N1. A sequence of neutrals takes the direction of the
      // surrounding strong text if the text on both sides has the same
      // direction. European and Arabic numbers act as if they were R in
      // terms of their influence on neutrals. Start-of-level-run (sor)
      // and end-of-level-run (eor) are used at level run boundaries.
      // N2. Any remaining neutrals take the embedding direction.
      // (Left after this: L, R, EN+AN)
      for (let i = from; i < to;) {
        let type = types[i];
        if (type == 256 /* T.NI */) {
          let end = i + 1;
          for (;;) {
            if (end == to) {
              if (iI == isolates.length) break;
              end = isolates[iI++].to;
              to = iI < isolates.length ? isolates[iI].from : rTo;
            } else if (types[end] == 256 /* T.NI */) {
              end++;
            } else {
              break;
            }
          }
          let beforeL = prev == 1 /* T.L */;
          let afterL = (end < rTo ? types[end] : outerType) == 1 /* T.L */;
          let replace = beforeL == afterL ? beforeL ? 1 /* T.L */ : 2 /* T.R */ : outerType;
          for (let j = end, jI = iI, fromJ = jI ? isolates[jI - 1].to : rFrom; j > i;) {
            if (j == fromJ) {
              j = isolates[--jI].from;
              fromJ = jI ? isolates[jI - 1].to : rFrom;
            }
            types[--j] = replace;
          }
          i = end;
        } else {
          prev = type;
          i++;
        }
      }
    }
  }
  // Find the contiguous ranges of character types in a given range, and
  // emit spans for them. Flip the order of the spans as appropriate
  // based on the level, and call through to compute the spans for
  // isolates at the proper point.
  function emitSpans(line, from, to, level, baseLevel, isolates, order) {
    let ourType = level % 2 ? 2 /* T.R */ : 1 /* T.L */;
    if (level % 2 == baseLevel % 2) {
      // Same dir as base direction, don't flip
      for (let iCh = from, iI = 0; iCh < to;) {
        // Scan a section of characters in direction ourType, unless
        // there's another type of char right after iCh, in which case
        // we scan a section of other characters (which, if ourType ==
        // T.L, may contain both T.R and T.AN chars).
        let sameDir = true,
          isNum = false;
        if (iI == isolates.length || iCh < isolates[iI].from) {
          let next = types[iCh];
          if (next != ourType) {
            sameDir = false;
            isNum = next == 16 /* T.AN */;
          }
        }
        // Holds an array of isolates to pass to a recursive call if we
        // must recurse (to distinguish T.AN inside an RTL section in
        // LTR text), null if we can emit directly
        let recurse = !sameDir && ourType == 1 /* T.L */ ? [] : null;
        let localLevel = sameDir ? level : level + 1;
        let iScan = iCh;
        run: for (;;) {
          if (iI < isolates.length && iScan == isolates[iI].from) {
            if (isNum) break run;
            let iso = isolates[iI];
            // Scan ahead to verify that there is another char in this dir after the isolate(s)
            if (!sameDir) for (let upto = iso.to, jI = iI + 1;;) {
              if (upto == to) break run;
              if (jI < isolates.length && isolates[jI].from == upto) upto = isolates[jI++].to;else if (types[upto] == ourType) break run;else break;
            }
            iI++;
            if (recurse) {
              recurse.push(iso);
            } else {
              if (iso.from > iCh) order.push(new BidiSpan(iCh, iso.from, localLevel));
              let dirSwap = iso.direction == LTR != !(localLevel % 2);
              computeSectionOrder(line, dirSwap ? level + 1 : level, baseLevel, iso.inner, iso.from, iso.to, order);
              iCh = iso.to;
            }
            iScan = iso.to;
          } else if (iScan == to || (sameDir ? types[iScan] != ourType : types[iScan] == ourType)) {
            break;
          } else {
            iScan++;
          }
        }
        if (recurse) emitSpans(line, iCh, iScan, level + 1, baseLevel, recurse, order);else if (iCh < iScan) order.push(new BidiSpan(iCh, iScan, localLevel));
        iCh = iScan;
      }
    } else {
      // Iterate in reverse to flip the span order. Same code again, but
      // going from the back of the section to the front
      for (let iCh = to, iI = isolates.length; iCh > from;) {
        let sameDir = true,
          isNum = false;
        if (!iI || iCh > isolates[iI - 1].to) {
          let next = types[iCh - 1];
          if (next != ourType) {
            sameDir = false;
            isNum = next == 16 /* T.AN */;
          }
        }
        let recurse = !sameDir && ourType == 1 /* T.L */ ? [] : null;
        let localLevel = sameDir ? level : level + 1;
        let iScan = iCh;
        run: for (;;) {
          if (iI && iScan == isolates[iI - 1].to) {
            if (isNum) break run;
            let iso = isolates[--iI];
            // Scan ahead to verify that there is another char in this dir after the isolate(s)
            if (!sameDir) for (let upto = iso.from, jI = iI;;) {
              if (upto == from) break run;
              if (jI && isolates[jI - 1].to == upto) upto = isolates[--jI].from;else if (types[upto - 1] == ourType) break run;else break;
            }
            if (recurse) {
              recurse.push(iso);
            } else {
              if (iso.to < iCh) order.push(new BidiSpan(iso.to, iCh, localLevel));
              let dirSwap = iso.direction == LTR != !(localLevel % 2);
              computeSectionOrder(line, dirSwap ? level + 1 : level, baseLevel, iso.inner, iso.from, iso.to, order);
              iCh = iso.from;
            }
            iScan = iso.from;
          } else if (iScan == from || (sameDir ? types[iScan - 1] != ourType : types[iScan - 1] == ourType)) {
            break;
          } else {
            iScan--;
          }
        }
        if (recurse) emitSpans(line, iScan, iCh, level + 1, baseLevel, recurse, order);else if (iScan < iCh) order.push(new BidiSpan(iScan, iCh, localLevel));
        iCh = iScan;
      }
    }
  }
  function computeSectionOrder(line, level, baseLevel, isolates, from, to, order) {
    let outerType = level % 2 ? 2 /* T.R */ : 1 /* T.L */;
    computeCharTypes(line, from, to, isolates, outerType);
    processBracketPairs(line, from, to, isolates, outerType);
    processNeutrals(from, to, isolates, outerType);
    emitSpans(line, from, to, level, baseLevel, isolates, order);
  }
  function computeOrder(line, direction, isolates) {
    if (!line) return [new BidiSpan(0, 0, direction == RTL ? 1 : 0)];
    if (direction == LTR && !isolates.length && !BidiRE.test(line)) return trivialOrder(line.length);
    if (isolates.length) while (line.length > types.length) types[types.length] = 256 /* T.NI */; // Make sure types array has no gaps
    let order = [],
      level = direction == LTR ? 0 : 1;
    computeSectionOrder(line, level, level, isolates, 0, line.length, order);
    return order;
  }
  function trivialOrder(length) {
    return [new BidiSpan(0, length, 0)];
  }
  let movedOver = "";
  // This implementation moves strictly visually, without concern for a
  // traversal visiting every logical position in the string. It will
  // still do so for simple input, but situations like multiple isolates
  // with the same level next to each other, or text going against the
  // main dir at the end of the line, will make some positions
  // unreachable with this motion. Each visible cursor position will
  // correspond to the lower-level bidi span that touches it.
  //
  // The alternative would be to solve an order globally for a given
  // line, making sure that it includes every position, but that would
  // require associating non-canonical (higher bidi span level)
  // positions with a given visual position, which is likely to confuse
  // people. (And would generally be a lot more complicated.)
  function moveVisually(line, order, dir, start, forward) {
    var _a;
    let startIndex = start.head - line.from;
    let spanI = BidiSpan.find(order, startIndex, (_a = start.bidiLevel) !== null && _a !== void 0 ? _a : -1, start.assoc);
    let span = order[spanI],
      spanEnd = span.side(forward, dir);
    // End of span
    if (startIndex == spanEnd) {
      let nextI = spanI += forward ? 1 : -1;
      if (nextI < 0 || nextI >= order.length) return null;
      span = order[spanI = nextI];
      startIndex = span.side(!forward, dir);
      spanEnd = span.side(forward, dir);
    }
    let nextIndex = findClusterBreak(line.text, startIndex, span.forward(forward, dir));
    if (nextIndex < span.from || nextIndex > span.to) nextIndex = spanEnd;
    movedOver = line.text.slice(Math.min(startIndex, nextIndex), Math.max(startIndex, nextIndex));
    let nextSpan = spanI == (forward ? order.length - 1 : 0) ? null : order[spanI + (forward ? 1 : -1)];
    if (nextSpan && nextIndex == spanEnd && nextSpan.level + (forward ? 0 : 1) < span.level) return EditorSelection.cursor(nextSpan.side(!forward, dir) + line.from, nextSpan.forward(forward, dir) ? 1 : -1, nextSpan.level);
    return EditorSelection.cursor(nextIndex + line.from, span.forward(forward, dir) ? -1 : 1, span.level);
  }
  function autoDirection(text, from, to) {
    for (let i = from; i < to; i++) {
      let type = charType(text.charCodeAt(i));
      if (type == 1 /* T.L */) return LTR;
      if (type == 2 /* T.R */ || type == 4 /* T.AL */) return RTL;
    }
    return LTR;
  }
  const clickAddsSelectionRange = /*@__PURE__*/Facet.define();
  const dragMovesSelection$1 = /*@__PURE__*/Facet.define();
  const mouseSelectionStyle = /*@__PURE__*/Facet.define();
  const exceptionSink = /*@__PURE__*/Facet.define();
  const updateListener = /*@__PURE__*/Facet.define();
  const inputHandler = /*@__PURE__*/Facet.define();
  const focusChangeEffect = /*@__PURE__*/Facet.define();
  const clipboardInputFilter = /*@__PURE__*/Facet.define();
  const clipboardOutputFilter = /*@__PURE__*/Facet.define();
  const perLineTextDirection = /*@__PURE__*/Facet.define({
    combine: values => values.some(x => x)
  });
  const nativeSelectionHidden = /*@__PURE__*/Facet.define({
    combine: values => values.some(x => x)
  });
  const scrollHandler = /*@__PURE__*/Facet.define();
  class ScrollTarget {
    constructor(range, y = "nearest", x = "nearest", yMargin = 5, xMargin = 5,
    // This data structure is abused to also store precise scroll
    // snapshots, instead of a `scrollIntoView` request. When this
    // flag is `true`, `range` points at a position in the reference
    // line, `yMargin` holds the difference between the top of that
    // line and the top of the editor, and `xMargin` holds the
    // editor's `scrollLeft`.
    isSnapshot = false) {
      this.range = range;
      this.y = y;
      this.x = x;
      this.yMargin = yMargin;
      this.xMargin = xMargin;
      this.isSnapshot = isSnapshot;
    }
    map(changes) {
      return changes.empty ? this : new ScrollTarget(this.range.map(changes), this.y, this.x, this.yMargin, this.xMargin, this.isSnapshot);
    }
    clip(state) {
      return this.range.to <= state.doc.length ? this : new ScrollTarget(EditorSelection.cursor(state.doc.length), this.y, this.x, this.yMargin, this.xMargin, this.isSnapshot);
    }
  }
  const scrollIntoView = /*@__PURE__*/StateEffect.define({
    map: (t, ch) => t.map(ch)
  });
  const setEditContextFormatting = /*@__PURE__*/StateEffect.define();
  /**
  Log or report an unhandled exception in client code. Should
  probably only be used by extension code that allows client code to
  provide functions, and calls those functions in a context where an
  exception can't be propagated to calling code in a reasonable way
  (for example when in an event handler).

  Either calls a handler registered with
  [`EditorView.exceptionSink`](https://codemirror.net/6/docs/ref/#view.EditorView^exceptionSink),
  `window.onerror`, if defined, or `console.error` (in which case
  it'll pass `context`, when given, as first argument).
  */
  function logException(state, exception, context) {
    let handler = state.facet(exceptionSink);
    if (handler.length) handler[0](exception);else if (window.onerror) window.onerror(String(exception), context, undefined, undefined, exception);else if (context) console.error(context + ":", exception);else console.error(exception);
  }
  const editable = /*@__PURE__*/Facet.define({
    combine: values => values.length ? values[0] : true
  });
  let nextPluginID = 0;
  const viewPlugin = /*@__PURE__*/Facet.define();
  /**
  View plugins associate stateful values with a view. They can
  influence the way the content is drawn, and are notified of things
  that happen in the view.
  */
  class ViewPlugin {
    constructor(
    /**
    @internal
    */
    id,
    /**
    @internal
    */
    create,
    /**
    @internal
    */
    domEventHandlers,
    /**
    @internal
    */
    domEventObservers, buildExtensions) {
      this.id = id;
      this.create = create;
      this.domEventHandlers = domEventHandlers;
      this.domEventObservers = domEventObservers;
      this.extension = buildExtensions(this);
    }
    /**
    Define a plugin from a constructor function that creates the
    plugin's value, given an editor view.
    */
    static define(create, spec) {
      const {
        eventHandlers,
        eventObservers,
        provide,
        decorations: deco
      } = spec || {};
      return new ViewPlugin(nextPluginID++, create, eventHandlers, eventObservers, plugin => {
        let ext = [viewPlugin.of(plugin)];
        if (deco) ext.push(decorations.of(view => {
          let pluginInst = view.plugin(plugin);
          return pluginInst ? deco(pluginInst) : Decoration.none;
        }));
        if (provide) ext.push(provide(plugin));
        return ext;
      });
    }
    /**
    Create a plugin for a class whose constructor takes a single
    editor view as argument.
    */
    static fromClass(cls, spec) {
      return ViewPlugin.define(view => new cls(view), spec);
    }
  }
  class PluginInstance {
    constructor(spec) {
      this.spec = spec;
      // When starting an update, all plugins have this field set to the
      // update object, indicating they need to be updated. When finished
      // updating, it is set to `false`. Retrieving a plugin that needs to
      // be updated with `view.plugin` forces an eager update.
      this.mustUpdate = null;
      // This is null when the plugin is initially created, but
      // initialized on the first update.
      this.value = null;
    }
    update(view) {
      if (!this.value) {
        if (this.spec) {
          try {
            this.value = this.spec.create(view);
          } catch (e) {
            logException(view.state, e, "CodeMirror plugin crashed");
            this.deactivate();
          }
        }
      } else if (this.mustUpdate) {
        let update = this.mustUpdate;
        this.mustUpdate = null;
        if (this.value.update) {
          try {
            this.value.update(update);
          } catch (e) {
            logException(update.state, e, "CodeMirror plugin crashed");
            if (this.value.destroy) try {
              this.value.destroy();
            } catch (_) {}
            this.deactivate();
          }
        }
      }
      return this;
    }
    destroy(view) {
      var _a;
      if ((_a = this.value) === null || _a === void 0 ? void 0 : _a.destroy) {
        try {
          this.value.destroy();
        } catch (e) {
          logException(view.state, e, "CodeMirror plugin crashed");
        }
      }
    }
    deactivate() {
      this.spec = this.value = null;
    }
  }
  const editorAttributes = /*@__PURE__*/Facet.define();
  const contentAttributes = /*@__PURE__*/Facet.define();
  // Provide decorations
  const decorations = /*@__PURE__*/Facet.define();
  const outerDecorations = /*@__PURE__*/Facet.define();
  const atomicRanges = /*@__PURE__*/Facet.define();
  const bidiIsolatedRanges = /*@__PURE__*/Facet.define();
  function getIsolatedRanges(view, line) {
    let isolates = view.state.facet(bidiIsolatedRanges);
    if (!isolates.length) return isolates;
    let sets = isolates.map(i => i instanceof Function ? i(view) : i);
    let result = [];
    RangeSet.spans(sets, line.from, line.to, {
      point() {},
      span(fromDoc, toDoc, active, open) {
        let from = fromDoc - line.from,
          to = toDoc - line.from;
        let level = result;
        for (let i = active.length - 1; i >= 0; i--, open--) {
          let direction = active[i].spec.bidiIsolate,
            update;
          if (direction == null) direction = autoDirection(line.text, from, to);
          if (open > 0 && level.length && (update = level[level.length - 1]).to == from && update.direction == direction) {
            update.to = to;
            level = update.inner;
          } else {
            let add = {
              from,
              to,
              direction,
              inner: []
            };
            level.push(add);
            level = add.inner;
          }
        }
      }
    });
    return result;
  }
  const scrollMargins = /*@__PURE__*/Facet.define();
  function getScrollMargins(view) {
    let left = 0,
      right = 0,
      top = 0,
      bottom = 0;
    for (let source of view.state.facet(scrollMargins)) {
      let m = source(view);
      if (m) {
        if (m.left != null) left = Math.max(left, m.left);
        if (m.right != null) right = Math.max(right, m.right);
        if (m.top != null) top = Math.max(top, m.top);
        if (m.bottom != null) bottom = Math.max(bottom, m.bottom);
      }
    }
    return {
      left,
      right,
      top,
      bottom
    };
  }
  const styleModule = /*@__PURE__*/Facet.define();
  class ChangedRange {
    constructor(fromA, toA, fromB, toB) {
      this.fromA = fromA;
      this.toA = toA;
      this.fromB = fromB;
      this.toB = toB;
    }
    join(other) {
      return new ChangedRange(Math.min(this.fromA, other.fromA), Math.max(this.toA, other.toA), Math.min(this.fromB, other.fromB), Math.max(this.toB, other.toB));
    }
    addToSet(set) {
      let i = set.length,
        me = this;
      for (; i > 0; i--) {
        let range = set[i - 1];
        if (range.fromA > me.toA) continue;
        if (range.toA < me.fromA) break;
        me = me.join(range);
        set.splice(i - 1, 1);
      }
      set.splice(i, 0, me);
      return set;
    }
    static extendWithRanges(diff, ranges) {
      if (ranges.length == 0) return diff;
      let result = [];
      for (let dI = 0, rI = 0, posA = 0, posB = 0;; dI++) {
        let next = dI == diff.length ? null : diff[dI],
          off = posA - posB;
        let end = next ? next.fromB : 1e9;
        while (rI < ranges.length && ranges[rI] < end) {
          let from = ranges[rI],
            to = ranges[rI + 1];
          let fromB = Math.max(posB, from),
            toB = Math.min(end, to);
          if (fromB <= toB) new ChangedRange(fromB + off, toB + off, fromB, toB).addToSet(result);
          if (to > end) break;else rI += 2;
        }
        if (!next) return result;
        new ChangedRange(next.fromA, next.toA, next.fromB, next.toB).addToSet(result);
        posA = next.toA;
        posB = next.toB;
      }
    }
  }
  /**
  View [plugins](https://codemirror.net/6/docs/ref/#view.ViewPlugin) are given instances of this
  class, which describe what happened, whenever the view is updated.
  */
  class ViewUpdate {
    constructor(
    /**
    The editor view that the update is associated with.
    */
    view,
    /**
    The new editor state.
    */
    state,
    /**
    The transactions involved in the update. May be empty.
    */
    transactions) {
      this.view = view;
      this.state = state;
      this.transactions = transactions;
      /**
      @internal
      */
      this.flags = 0;
      this.startState = view.state;
      this.changes = ChangeSet.empty(this.startState.doc.length);
      for (let tr of transactions) this.changes = this.changes.compose(tr.changes);
      let changedRanges = [];
      this.changes.iterChangedRanges((fromA, toA, fromB, toB) => changedRanges.push(new ChangedRange(fromA, toA, fromB, toB)));
      this.changedRanges = changedRanges;
    }
    /**
    @internal
    */
    static create(view, state, transactions) {
      return new ViewUpdate(view, state, transactions);
    }
    /**
    Tells you whether the [viewport](https://codemirror.net/6/docs/ref/#view.EditorView.viewport) or
    [visible ranges](https://codemirror.net/6/docs/ref/#view.EditorView.visibleRanges) changed in this
    update.
    */
    get viewportChanged() {
      return (this.flags & 4 /* UpdateFlag.Viewport */) > 0;
    }
    /**
    Returns true when
    [`viewportChanged`](https://codemirror.net/6/docs/ref/#view.ViewUpdate.viewportChanged) is true
    and the viewport change is not just the result of mapping it in
    response to document changes.
    */
    get viewportMoved() {
      return (this.flags & 8 /* UpdateFlag.ViewportMoved */) > 0;
    }
    /**
    Indicates whether the height of a block element in the editor
    changed in this update.
    */
    get heightChanged() {
      return (this.flags & 2 /* UpdateFlag.Height */) > 0;
    }
    /**
    Returns true when the document was modified or the size of the
    editor, or elements within the editor, changed.
    */
    get geometryChanged() {
      return this.docChanged || (this.flags & (16 /* UpdateFlag.Geometry */ | 2 /* UpdateFlag.Height */)) > 0;
    }
    /**
    True when this update indicates a focus change.
    */
    get focusChanged() {
      return (this.flags & 1 /* UpdateFlag.Focus */) > 0;
    }
    /**
    Whether the document changed in this update.
    */
    get docChanged() {
      return !this.changes.empty;
    }
    /**
    Whether the selection was explicitly set in this update.
    */
    get selectionSet() {
      return this.transactions.some(tr => tr.selection);
    }
    /**
    @internal
    */
    get empty() {
      return this.flags == 0 && this.transactions.length == 0;
    }
  }
  class DocView extends ContentView {
    get length() {
      return this.view.state.doc.length;
    }
    constructor(view) {
      super();
      this.view = view;
      this.decorations = [];
      this.dynamicDecorationMap = [false];
      this.domChanged = null;
      this.hasComposition = null;
      this.markedForComposition = new Set();
      this.editContextFormatting = Decoration.none;
      this.lastCompositionAfterCursor = false;
      // Track a minimum width for the editor. When measuring sizes in
      // measureVisibleLineHeights, this is updated to point at the width
      // of a given element and its extent in the document. When a change
      // happens in that range, these are reset. That way, once we've seen
      // a line/element of a given length, we keep the editor wide enough
      // to fit at least that element, until it is changed, at which point
      // we forget it again.
      this.minWidth = 0;
      this.minWidthFrom = 0;
      this.minWidthTo = 0;
      // Track whether the DOM selection was set in a lossy way, so that
      // we don't mess it up when reading it back it
      this.impreciseAnchor = null;
      this.impreciseHead = null;
      this.forceSelection = false;
      // Used by the resize observer to ignore resizes that we caused
      // ourselves
      this.lastUpdate = Date.now();
      this.setDOM(view.contentDOM);
      this.children = [new LineView()];
      this.children[0].setParent(this);
      this.updateDeco();
      this.updateInner([new ChangedRange(0, 0, 0, view.state.doc.length)], 0, null);
    }
    // Update the document view to a given state.
    update(update) {
      var _a;
      let changedRanges = update.changedRanges;
      if (this.minWidth > 0 && changedRanges.length) {
        if (!changedRanges.every(({
          fromA,
          toA
        }) => toA < this.minWidthFrom || fromA > this.minWidthTo)) {
          this.minWidth = this.minWidthFrom = this.minWidthTo = 0;
        } else {
          this.minWidthFrom = update.changes.mapPos(this.minWidthFrom, 1);
          this.minWidthTo = update.changes.mapPos(this.minWidthTo, 1);
        }
      }
      this.updateEditContextFormatting(update);
      let readCompositionAt = -1;
      if (this.view.inputState.composing >= 0 && !this.view.observer.editContext) {
        if ((_a = this.domChanged) === null || _a === void 0 ? void 0 : _a.newSel) readCompositionAt = this.domChanged.newSel.head;else if (!touchesComposition(update.changes, this.hasComposition) && !update.selectionSet) readCompositionAt = update.state.selection.main.head;
      }
      let composition = readCompositionAt > -1 ? findCompositionRange(this.view, update.changes, readCompositionAt) : null;
      this.domChanged = null;
      if (this.hasComposition) {
        this.markedForComposition.clear();
        let {
          from,
          to
        } = this.hasComposition;
        changedRanges = new ChangedRange(from, to, update.changes.mapPos(from, -1), update.changes.mapPos(to, 1)).addToSet(changedRanges.slice());
      }
      this.hasComposition = composition ? {
        from: composition.range.fromB,
        to: composition.range.toB
      } : null;
      // When the DOM nodes around the selection are moved to another
      // parent, Chrome sometimes reports a different selection through
      // getSelection than the one that it actually shows to the user.
      // This forces a selection update when lines are joined to work
      // around that. Issue #54
      if ((browser.ie || browser.chrome) && !composition && update && update.state.doc.lines != update.startState.doc.lines) this.forceSelection = true;
      let prevDeco = this.decorations,
        deco = this.updateDeco();
      let decoDiff = findChangedDeco(prevDeco, deco, update.changes);
      changedRanges = ChangedRange.extendWithRanges(changedRanges, decoDiff);
      if (!(this.flags & 7 /* ViewFlag.Dirty */) && changedRanges.length == 0) {
        return false;
      } else {
        this.updateInner(changedRanges, update.startState.doc.length, composition);
        if (update.transactions.length) this.lastUpdate = Date.now();
        return true;
      }
    }
    // Used by update and the constructor do perform the actual DOM
    // update
    updateInner(changes, oldLength, composition) {
      this.view.viewState.mustMeasureContent = true;
      this.updateChildren(changes, oldLength, composition);
      let {
        observer
      } = this.view;
      observer.ignore(() => {
        // Lock the height during redrawing, since Chrome sometimes
        // messes with the scroll position during DOM mutation (though
        // no relayout is triggered and I cannot imagine how it can
        // recompute the scroll position without a layout)
        this.dom.style.height = this.view.viewState.contentHeight / this.view.scaleY + "px";
        this.dom.style.flexBasis = this.minWidth ? this.minWidth + "px" : "";
        // Chrome will sometimes, when DOM mutations occur directly
        // around the selection, get confused and report a different
        // selection from the one it displays (issue #218). This tries
        // to detect that situation.
        let track = browser.chrome || browser.ios ? {
          node: observer.selectionRange.focusNode,
          written: false
        } : undefined;
        this.sync(this.view, track);
        this.flags &= ~7 /* ViewFlag.Dirty */;
        if (track && (track.written || observer.selectionRange.focusNode != track.node)) this.forceSelection = true;
        this.dom.style.height = "";
      });
      this.markedForComposition.forEach(cView => cView.flags &= ~8 /* ViewFlag.Composition */);
      let gaps = [];
      if (this.view.viewport.from || this.view.viewport.to < this.view.state.doc.length) for (let child of this.children) if (child instanceof BlockWidgetView && child.widget instanceof BlockGapWidget) gaps.push(child.dom);
      observer.updateGaps(gaps);
    }
    updateChildren(changes, oldLength, composition) {
      let ranges = composition ? composition.range.addToSet(changes.slice()) : changes;
      let cursor = this.childCursor(oldLength);
      for (let i = ranges.length - 1;; i--) {
        let next = i >= 0 ? ranges[i] : null;
        if (!next) break;
        let {
            fromA,
            toA,
            fromB,
            toB
          } = next,
          content,
          breakAtStart,
          openStart,
          openEnd;
        if (composition && composition.range.fromB < toB && composition.range.toB > fromB) {
          let before = ContentBuilder.build(this.view.state.doc, fromB, composition.range.fromB, this.decorations, this.dynamicDecorationMap);
          let after = ContentBuilder.build(this.view.state.doc, composition.range.toB, toB, this.decorations, this.dynamicDecorationMap);
          breakAtStart = before.breakAtStart;
          openStart = before.openStart;
          openEnd = after.openEnd;
          let compLine = this.compositionView(composition);
          if (after.breakAtStart) {
            compLine.breakAfter = 1;
          } else if (after.content.length && compLine.merge(compLine.length, compLine.length, after.content[0], false, after.openStart, 0)) {
            compLine.breakAfter = after.content[0].breakAfter;
            after.content.shift();
          }
          if (before.content.length && compLine.merge(0, 0, before.content[before.content.length - 1], true, 0, before.openEnd)) {
            before.content.pop();
          }
          content = before.content.concat(compLine).concat(after.content);
        } else {
          ({
            content,
            breakAtStart,
            openStart,
            openEnd
          } = ContentBuilder.build(this.view.state.doc, fromB, toB, this.decorations, this.dynamicDecorationMap));
        }
        let {
          i: toI,
          off: toOff
        } = cursor.findPos(toA, 1);
        let {
          i: fromI,
          off: fromOff
        } = cursor.findPos(fromA, -1);
        replaceRange(this, fromI, fromOff, toI, toOff, content, breakAtStart, openStart, openEnd);
      }
      if (composition) this.fixCompositionDOM(composition);
    }
    updateEditContextFormatting(update) {
      this.editContextFormatting = this.editContextFormatting.map(update.changes);
      for (let tr of update.transactions) for (let effect of tr.effects) if (effect.is(setEditContextFormatting)) {
        this.editContextFormatting = effect.value;
      }
    }
    compositionView(composition) {
      let cur = new TextView(composition.text.nodeValue);
      cur.flags |= 8 /* ViewFlag.Composition */;
      for (let {
        deco
      } of composition.marks) cur = new MarkView(deco, [cur], cur.length);
      let line = new LineView();
      line.append(cur, 0);
      return line;
    }
    fixCompositionDOM(composition) {
      let fix = (dom, cView) => {
        cView.flags |= 8 /* ViewFlag.Composition */ | (cView.children.some(c => c.flags & 7 /* ViewFlag.Dirty */) ? 1 /* ViewFlag.ChildDirty */ : 0);
        this.markedForComposition.add(cView);
        let prev = ContentView.get(dom);
        if (prev && prev != cView) prev.dom = null;
        cView.setDOM(dom);
      };
      let pos = this.childPos(composition.range.fromB, 1);
      let cView = this.children[pos.i];
      fix(composition.line, cView);
      for (let i = composition.marks.length - 1; i >= -1; i--) {
        pos = cView.childPos(pos.off, 1);
        cView = cView.children[pos.i];
        fix(i >= 0 ? composition.marks[i].node : composition.text, cView);
      }
    }
    // Sync the DOM selection to this.state.selection
    updateSelection(mustRead = false, fromPointer = false) {
      if (mustRead || !this.view.observer.selectionRange.focusNode) this.view.observer.readSelectionRange();
      let activeElt = this.view.root.activeElement,
        focused = activeElt == this.dom;
      let selectionNotFocus = !focused && !(this.view.state.facet(editable) || this.dom.tabIndex > -1) && hasSelection(this.dom, this.view.observer.selectionRange) && !(activeElt && this.dom.contains(activeElt));
      if (!(focused || fromPointer || selectionNotFocus)) return;
      let force = this.forceSelection;
      this.forceSelection = false;
      let main = this.view.state.selection.main;
      let anchor = this.moveToLine(this.domAtPos(main.anchor));
      let head = main.empty ? anchor : this.moveToLine(this.domAtPos(main.head));
      // Always reset on Firefox when next to an uneditable node to
      // avoid invisible cursor bugs (#111)
      if (browser.gecko && main.empty && !this.hasComposition && betweenUneditable(anchor)) {
        let dummy = document.createTextNode("");
        this.view.observer.ignore(() => anchor.node.insertBefore(dummy, anchor.node.childNodes[anchor.offset] || null));
        anchor = head = new DOMPos(dummy, 0);
        force = true;
      }
      let domSel = this.view.observer.selectionRange;
      // If the selection is already here, or in an equivalent position, don't touch it
      if (force || !domSel.focusNode || (!isEquivalentPosition(anchor.node, anchor.offset, domSel.anchorNode, domSel.anchorOffset) || !isEquivalentPosition(head.node, head.offset, domSel.focusNode, domSel.focusOffset)) && !this.suppressWidgetCursorChange(domSel, main)) {
        this.view.observer.ignore(() => {
          // Chrome Android will hide the virtual keyboard when tapping
          // inside an uneditable node, and not bring it back when we
          // move the cursor to its proper position. This tries to
          // restore the keyboard by cycling focus.
          if (browser.android && browser.chrome && this.dom.contains(domSel.focusNode) && inUneditable(domSel.focusNode, this.dom)) {
            this.dom.blur();
            this.dom.focus({
              preventScroll: true
            });
          }
          let rawSel = getSelection(this.view.root);
          if (!rawSel) ;else if (main.empty) {
            // Work around https://bugzilla.mozilla.org/show_bug.cgi?id=1612076
            if (browser.gecko) {
              let nextTo = nextToUneditable(anchor.node, anchor.offset);
              if (nextTo && nextTo != (1 /* NextTo.Before */ | 2 /* NextTo.After */)) {
                let text = (nextTo == 1 /* NextTo.Before */ ? textNodeBefore : textNodeAfter)(anchor.node, anchor.offset);
                if (text) anchor = new DOMPos(text.node, text.offset);
              }
            }
            rawSel.collapse(anchor.node, anchor.offset);
            if (main.bidiLevel != null && rawSel.caretBidiLevel !== undefined) rawSel.caretBidiLevel = main.bidiLevel;
          } else if (rawSel.extend) {
            // Selection.extend can be used to create an 'inverted' selection
            // (one where the focus is before the anchor), but not all
            // browsers support it yet.
            rawSel.collapse(anchor.node, anchor.offset);
            // Safari will ignore the call above when the editor is
            // hidden, and then raise an error on the call to extend
            // (#940).
            try {
              rawSel.extend(head.node, head.offset);
            } catch (_) {}
          } else {
            // Primitive (IE) way
            let range = document.createRange();
            if (main.anchor > main.head) [anchor, head] = [head, anchor];
            range.setEnd(head.node, head.offset);
            range.setStart(anchor.node, anchor.offset);
            rawSel.removeAllRanges();
            rawSel.addRange(range);
          }
          if (selectionNotFocus && this.view.root.activeElement == this.dom) {
            this.dom.blur();
            if (activeElt) activeElt.focus();
          }
        });
        this.view.observer.setSelectionRange(anchor, head);
      }
      this.impreciseAnchor = anchor.precise ? null : new DOMPos(domSel.anchorNode, domSel.anchorOffset);
      this.impreciseHead = head.precise ? null : new DOMPos(domSel.focusNode, domSel.focusOffset);
    }
    // If a zero-length widget is inserted next to the cursor during
    // composition, avoid moving it across it and disrupting the
    // composition.
    suppressWidgetCursorChange(sel, cursor) {
      return this.hasComposition && cursor.empty && isEquivalentPosition(sel.focusNode, sel.focusOffset, sel.anchorNode, sel.anchorOffset) && this.posFromDOM(sel.focusNode, sel.focusOffset) == cursor.head;
    }
    enforceCursorAssoc() {
      if (this.hasComposition) return;
      let {
          view
        } = this,
        cursor = view.state.selection.main;
      let sel = getSelection(view.root);
      let {
        anchorNode,
        anchorOffset
      } = view.observer.selectionRange;
      if (!sel || !cursor.empty || !cursor.assoc || !sel.modify) return;
      let line = LineView.find(this, cursor.head);
      if (!line) return;
      let lineStart = line.posAtStart;
      if (cursor.head == lineStart || cursor.head == lineStart + line.length) return;
      let before = this.coordsAt(cursor.head, -1),
        after = this.coordsAt(cursor.head, 1);
      if (!before || !after || before.bottom > after.top) return;
      let dom = this.domAtPos(cursor.head + cursor.assoc);
      sel.collapse(dom.node, dom.offset);
      sel.modify("move", cursor.assoc < 0 ? "forward" : "backward", "lineboundary");
      // This can go wrong in corner cases like single-character lines,
      // so check and reset if necessary.
      view.observer.readSelectionRange();
      let newRange = view.observer.selectionRange;
      if (view.docView.posFromDOM(newRange.anchorNode, newRange.anchorOffset) != cursor.from) sel.collapse(anchorNode, anchorOffset);
    }
    // If a position is in/near a block widget, move it to a nearby text
    // line, since we don't want the cursor inside a block widget.
    moveToLine(pos) {
      // Block widgets will return positions before/after them, which
      // are thus directly in the document DOM element.
      let dom = this.dom,
        newPos;
      if (pos.node != dom) return pos;
      for (let i = pos.offset; !newPos && i < dom.childNodes.length; i++) {
        let view = ContentView.get(dom.childNodes[i]);
        if (view instanceof LineView) newPos = view.domAtPos(0);
      }
      for (let i = pos.offset - 1; !newPos && i >= 0; i--) {
        let view = ContentView.get(dom.childNodes[i]);
        if (view instanceof LineView) newPos = view.domAtPos(view.length);
      }
      return newPos ? new DOMPos(newPos.node, newPos.offset, true) : pos;
    }
    nearest(dom) {
      for (let cur = dom; cur;) {
        let domView = ContentView.get(cur);
        if (domView && domView.rootView == this) return domView;
        cur = cur.parentNode;
      }
      return null;
    }
    posFromDOM(node, offset) {
      let view = this.nearest(node);
      if (!view) throw new RangeError("Trying to find position for a DOM position outside of the document");
      return view.localPosFromDOM(node, offset) + view.posAtStart;
    }
    domAtPos(pos) {
      let {
        i,
        off
      } = this.childCursor().findPos(pos, -1);
      for (; i < this.children.length - 1;) {
        let child = this.children[i];
        if (off < child.length || child instanceof LineView) break;
        i++;
        off = 0;
      }
      return this.children[i].domAtPos(off);
    }
    coordsAt(pos, side) {
      let best = null,
        bestPos = 0;
      for (let off = this.length, i = this.children.length - 1; i >= 0; i--) {
        let child = this.children[i],
          end = off - child.breakAfter,
          start = end - child.length;
        if (end < pos) break;
        if (start <= pos && (start < pos || child.covers(-1)) && (end > pos || child.covers(1)) && (!best || child instanceof LineView && !(best instanceof LineView && side >= 0))) {
          best = child;
          bestPos = start;
        } else if (best && start == pos && end == pos && child instanceof BlockWidgetView && Math.abs(side) < 2) {
          if (child.deco.startSide < 0) break;else if (i) best = null;
        }
        off = start;
      }
      return best ? best.coordsAt(pos - bestPos, side) : null;
    }
    coordsForChar(pos) {
      let {
          i,
          off
        } = this.childPos(pos, 1),
        child = this.children[i];
      if (!(child instanceof LineView)) return null;
      while (child.children.length) {
        let {
          i,
          off: childOff
        } = child.childPos(off, 1);
        for (;; i++) {
          if (i == child.children.length) return null;
          if ((child = child.children[i]).length) break;
        }
        off = childOff;
      }
      if (!(child instanceof TextView)) return null;
      let end = findClusterBreak(child.text, off);
      if (end == off) return null;
      let rects = textRange(child.dom, off, end).getClientRects();
      for (let i = 0; i < rects.length; i++) {
        let rect = rects[i];
        if (i == rects.length - 1 || rect.top < rect.bottom && rect.left < rect.right) return rect;
      }
      return null;
    }
    measureVisibleLineHeights(viewport) {
      let result = [],
        {
          from,
          to
        } = viewport;
      let contentWidth = this.view.contentDOM.clientWidth;
      let isWider = contentWidth > Math.max(this.view.scrollDOM.clientWidth, this.minWidth) + 1;
      let widest = -1,
        ltr = this.view.textDirection == Direction.LTR;
      for (let pos = 0, i = 0; i < this.children.length; i++) {
        let child = this.children[i],
          end = pos + child.length;
        if (end > to) break;
        if (pos >= from) {
          let childRect = child.dom.getBoundingClientRect();
          result.push(childRect.height);
          if (isWider) {
            let last = child.dom.lastChild;
            let rects = last ? clientRectsFor(last) : [];
            if (rects.length) {
              let rect = rects[rects.length - 1];
              let width = ltr ? rect.right - childRect.left : childRect.right - rect.left;
              if (width > widest) {
                widest = width;
                this.minWidth = contentWidth;
                this.minWidthFrom = pos;
                this.minWidthTo = end;
              }
            }
          }
        }
        pos = end + child.breakAfter;
      }
      return result;
    }
    textDirectionAt(pos) {
      let {
        i
      } = this.childPos(pos, 1);
      return getComputedStyle(this.children[i].dom).direction == "rtl" ? Direction.RTL : Direction.LTR;
    }
    measureTextSize() {
      for (let child of this.children) {
        if (child instanceof LineView) {
          let measure = child.measureTextSize();
          if (measure) return measure;
        }
      }
      // If no workable line exists, force a layout of a measurable element
      let dummy = document.createElement("div"),
        lineHeight,
        charWidth,
        textHeight;
      dummy.className = "cm-line";
      dummy.style.width = "99999px";
      dummy.style.position = "absolute";
      dummy.textContent = "abc def ghi jkl mno pqr stu";
      this.view.observer.ignore(() => {
        this.dom.appendChild(dummy);
        let rect = clientRectsFor(dummy.firstChild)[0];
        lineHeight = dummy.getBoundingClientRect().height;
        charWidth = rect ? rect.width / 27 : 7;
        textHeight = rect ? rect.height : lineHeight;
        dummy.remove();
      });
      return {
        lineHeight,
        charWidth,
        textHeight
      };
    }
    childCursor(pos = this.length) {
      // Move back to start of last element when possible, so that
      // `ChildCursor.findPos` doesn't have to deal with the edge case
      // of being after the last element.
      let i = this.children.length;
      if (i) pos -= this.children[--i].length;
      return new ChildCursor(this.children, pos, i);
    }
    computeBlockGapDeco() {
      let deco = [],
        vs = this.view.viewState;
      for (let pos = 0, i = 0;; i++) {
        let next = i == vs.viewports.length ? null : vs.viewports[i];
        let end = next ? next.from - 1 : this.length;
        if (end > pos) {
          let height = (vs.lineBlockAt(end).bottom - vs.lineBlockAt(pos).top) / this.view.scaleY;
          deco.push(Decoration.replace({
            widget: new BlockGapWidget(height),
            block: true,
            inclusive: true,
            isBlockGap: true
          }).range(pos, end));
        }
        if (!next) break;
        pos = next.to + 1;
      }
      return Decoration.set(deco);
    }
    updateDeco() {
      let i = 1;
      let allDeco = this.view.state.facet(decorations).map(d => {
        let dynamic = this.dynamicDecorationMap[i++] = typeof d == "function";
        return dynamic ? d(this.view) : d;
      });
      let dynamicOuter = false,
        outerDeco = this.view.state.facet(outerDecorations).map((d, i) => {
          let dynamic = typeof d == "function";
          if (dynamic) dynamicOuter = true;
          return dynamic ? d(this.view) : d;
        });
      if (outerDeco.length) {
        this.dynamicDecorationMap[i++] = dynamicOuter;
        allDeco.push(RangeSet.join(outerDeco));
      }
      this.decorations = [this.editContextFormatting, ...allDeco, this.computeBlockGapDeco(), this.view.viewState.lineGapDeco];
      while (i < this.decorations.length) this.dynamicDecorationMap[i++] = false;
      return this.decorations;
    }
    scrollIntoView(target) {
      if (target.isSnapshot) {
        let ref = this.view.viewState.lineBlockAt(target.range.head);
        this.view.scrollDOM.scrollTop = ref.top - target.yMargin;
        this.view.scrollDOM.scrollLeft = target.xMargin;
        return;
      }
      for (let handler of this.view.state.facet(scrollHandler)) {
        try {
          if (handler(this.view, target.range, target)) return true;
        } catch (e) {
          logException(this.view.state, e, "scroll handler");
        }
      }
      let {
        range
      } = target;
      let rect = this.coordsAt(range.head, range.empty ? range.assoc : range.head > range.anchor ? -1 : 1),
        other;
      if (!rect) return;
      if (!range.empty && (other = this.coordsAt(range.anchor, range.anchor > range.head ? -1 : 1))) rect = {
        left: Math.min(rect.left, other.left),
        top: Math.min(rect.top, other.top),
        right: Math.max(rect.right, other.right),
        bottom: Math.max(rect.bottom, other.bottom)
      };
      let margins = getScrollMargins(this.view);
      let targetRect = {
        left: rect.left - margins.left,
        top: rect.top - margins.top,
        right: rect.right + margins.right,
        bottom: rect.bottom + margins.bottom
      };
      let {
        offsetWidth,
        offsetHeight
      } = this.view.scrollDOM;
      scrollRectIntoView(this.view.scrollDOM, targetRect, range.head < range.anchor ? -1 : 1, target.x, target.y, Math.max(Math.min(target.xMargin, offsetWidth), -offsetWidth), Math.max(Math.min(target.yMargin, offsetHeight), -offsetHeight), this.view.textDirection == Direction.LTR);
    }
  }
  function betweenUneditable(pos) {
    return pos.node.nodeType == 1 && pos.node.firstChild && (pos.offset == 0 || pos.node.childNodes[pos.offset - 1].contentEditable == "false") && (pos.offset == pos.node.childNodes.length || pos.node.childNodes[pos.offset].contentEditable == "false");
  }
  function findCompositionNode(view, headPos) {
    let sel = view.observer.selectionRange;
    if (!sel.focusNode) return null;
    let textBefore = textNodeBefore(sel.focusNode, sel.focusOffset);
    let textAfter = textNodeAfter(sel.focusNode, sel.focusOffset);
    let textNode = textBefore || textAfter;
    if (textAfter && textBefore && textAfter.node != textBefore.node) {
      let descAfter = ContentView.get(textAfter.node);
      if (!descAfter || descAfter instanceof TextView && descAfter.text != textAfter.node.nodeValue) {
        textNode = textAfter;
      } else if (view.docView.lastCompositionAfterCursor) {
        let descBefore = ContentView.get(textBefore.node);
        if (!(!descBefore || descBefore instanceof TextView && descBefore.text != textBefore.node.nodeValue)) textNode = textAfter;
      }
    }
    view.docView.lastCompositionAfterCursor = textNode != textBefore;
    if (!textNode) return null;
    let from = headPos - textNode.offset;
    return {
      from,
      to: from + textNode.node.nodeValue.length,
      node: textNode.node
    };
  }
  function findCompositionRange(view, changes, headPos) {
    let found = findCompositionNode(view, headPos);
    if (!found) return null;
    let {
        node: textNode,
        from,
        to
      } = found,
      text = textNode.nodeValue;
    // Don't try to preserve multi-line compositions
    if (/[\n\r]/.test(text)) return null;
    if (view.state.doc.sliceString(found.from, found.to) != text) return null;
    let inv = changes.invertedDesc;
    let range = new ChangedRange(inv.mapPos(from), inv.mapPos(to), from, to);
    let marks = [];
    for (let parent = textNode.parentNode;; parent = parent.parentNode) {
      let parentView = ContentView.get(parent);
      if (parentView instanceof MarkView) marks.push({
        node: parent,
        deco: parentView.mark
      });else if (parentView instanceof LineView || parent.nodeName == "DIV" && parent.parentNode == view.contentDOM) return {
        range,
        text: textNode,
        marks,
        line: parent
      };else if (parent != view.contentDOM) marks.push({
        node: parent,
        deco: new MarkDecoration({
          inclusive: true,
          attributes: getAttrs(parent),
          tagName: parent.tagName.toLowerCase()
        })
      });else return null;
    }
  }
  function nextToUneditable(node, offset) {
    if (node.nodeType != 1) return 0;
    return (offset && node.childNodes[offset - 1].contentEditable == "false" ? 1 /* NextTo.Before */ : 0) | (offset < node.childNodes.length && node.childNodes[offset].contentEditable == "false" ? 2 /* NextTo.After */ : 0);
  }
  let DecorationComparator$1 = class DecorationComparator {
    constructor() {
      this.changes = [];
    }
    compareRange(from, to) {
      addRange(from, to, this.changes);
    }
    comparePoint(from, to) {
      addRange(from, to, this.changes);
    }
    boundChange(pos) {
      addRange(pos, pos, this.changes);
    }
  };
  function findChangedDeco(a, b, diff) {
    let comp = new DecorationComparator$1();
    RangeSet.compare(a, b, diff, comp);
    return comp.changes;
  }
  function inUneditable(node, inside) {
    for (let cur = node; cur && cur != inside; cur = cur.assignedSlot || cur.parentNode) {
      if (cur.nodeType == 1 && cur.contentEditable == 'false') {
        return true;
      }
    }
    return false;
  }
  function touchesComposition(changes, composition) {
    let touched = false;
    if (composition) changes.iterChangedRanges((from, to) => {
      if (from < composition.to && to > composition.from) touched = true;
    });
    return touched;
  }
  function groupAt(state, pos, bias = 1) {
    let categorize = state.charCategorizer(pos);
    let line = state.doc.lineAt(pos),
      linePos = pos - line.from;
    if (line.length == 0) return EditorSelection.cursor(pos);
    if (linePos == 0) bias = 1;else if (linePos == line.length) bias = -1;
    let from = linePos,
      to = linePos;
    if (bias < 0) from = findClusterBreak(line.text, linePos, false);else to = findClusterBreak(line.text, linePos);
    let cat = categorize(line.text.slice(from, to));
    while (from > 0) {
      let prev = findClusterBreak(line.text, from, false);
      if (categorize(line.text.slice(prev, from)) != cat) break;
      from = prev;
    }
    while (to < line.length) {
      let next = findClusterBreak(line.text, to);
      if (categorize(line.text.slice(to, next)) != cat) break;
      to = next;
    }
    return EditorSelection.range(from + line.from, to + line.from);
  }
  // Search the DOM for the {node, offset} position closest to the given
  // coordinates. Very inefficient and crude, but can usually be avoided
  // by calling caret(Position|Range)FromPoint instead.
  function getdx(x, rect) {
    return rect.left > x ? rect.left - x : Math.max(0, x - rect.right);
  }
  function getdy(y, rect) {
    return rect.top > y ? rect.top - y : Math.max(0, y - rect.bottom);
  }
  function yOverlap(a, b) {
    return a.top < b.bottom - 1 && a.bottom > b.top + 1;
  }
  function upTop(rect, top) {
    return top < rect.top ? {
      top,
      left: rect.left,
      right: rect.right,
      bottom: rect.bottom
    } : rect;
  }
  function upBot(rect, bottom) {
    return bottom > rect.bottom ? {
      top: rect.top,
      left: rect.left,
      right: rect.right,
      bottom
    } : rect;
  }
  function domPosAtCoords(parent, x, y) {
    let closest,
      closestRect,
      closestX,
      closestY,
      closestOverlap = false;
    let above, below, aboveRect, belowRect;
    for (let child = parent.firstChild; child; child = child.nextSibling) {
      let rects = clientRectsFor(child);
      for (let i = 0; i < rects.length; i++) {
        let rect = rects[i];
        if (closestRect && yOverlap(closestRect, rect)) rect = upTop(upBot(rect, closestRect.bottom), closestRect.top);
        let dx = getdx(x, rect),
          dy = getdy(y, rect);
        if (dx == 0 && dy == 0) return child.nodeType == 3 ? domPosInText(child, x, y) : domPosAtCoords(child, x, y);
        if (!closest || closestY > dy || closestY == dy && closestX > dx) {
          closest = child;
          closestRect = rect;
          closestX = dx;
          closestY = dy;
          let side = dy ? y < rect.top ? -1 : 1 : dx ? x < rect.left ? -1 : 1 : 0;
          closestOverlap = !side || (side > 0 ? i < rects.length - 1 : i > 0);
        }
        if (dx == 0) {
          if (y > rect.bottom && (!aboveRect || aboveRect.bottom < rect.bottom)) {
            above = child;
            aboveRect = rect;
          } else if (y < rect.top && (!belowRect || belowRect.top > rect.top)) {
            below = child;
            belowRect = rect;
          }
        } else if (aboveRect && yOverlap(aboveRect, rect)) {
          aboveRect = upBot(aboveRect, rect.bottom);
        } else if (belowRect && yOverlap(belowRect, rect)) {
          belowRect = upTop(belowRect, rect.top);
        }
      }
    }
    if (aboveRect && aboveRect.bottom >= y) {
      closest = above;
      closestRect = aboveRect;
    } else if (belowRect && belowRect.top <= y) {
      closest = below;
      closestRect = belowRect;
    }
    if (!closest) return {
      node: parent,
      offset: 0
    };
    let clipX = Math.max(closestRect.left, Math.min(closestRect.right, x));
    if (closest.nodeType == 3) return domPosInText(closest, clipX, y);
    if (closestOverlap && closest.contentEditable != "false") return domPosAtCoords(closest, clipX, y);
    let offset = Array.prototype.indexOf.call(parent.childNodes, closest) + (x >= (closestRect.left + closestRect.right) / 2 ? 1 : 0);
    return {
      node: parent,
      offset
    };
  }
  function domPosInText(node, x, y) {
    let len = node.nodeValue.length;
    let closestOffset = -1,
      closestDY = 1e9,
      generalSide = 0;
    for (let i = 0; i < len; i++) {
      let rects = textRange(node, i, i + 1).getClientRects();
      for (let j = 0; j < rects.length; j++) {
        let rect = rects[j];
        if (rect.top == rect.bottom) continue;
        if (!generalSide) generalSide = x - rect.left;
        let dy = (rect.top > y ? rect.top - y : y - rect.bottom) - 1;
        if (rect.left - 1 <= x && rect.right + 1 >= x && dy < closestDY) {
          let right = x >= (rect.left + rect.right) / 2,
            after = right;
          if (browser.chrome || browser.gecko) {
            // Check for RTL on browsers that support getting client
            // rects for empty ranges.
            let rectBefore = textRange(node, i).getBoundingClientRect();
            if (rectBefore.left == rect.right) after = !right;
          }
          if (dy <= 0) return {
            node,
            offset: i + (after ? 1 : 0)
          };
          closestOffset = i + (after ? 1 : 0);
          closestDY = dy;
        }
      }
    }
    return {
      node,
      offset: closestOffset > -1 ? closestOffset : generalSide > 0 ? node.nodeValue.length : 0
    };
  }
  function posAtCoords(view, coords, precise, bias = -1) {
    var _a, _b;
    let content = view.contentDOM.getBoundingClientRect(),
      docTop = content.top + view.viewState.paddingTop;
    let block,
      {
        docHeight
      } = view.viewState;
    let {
        x,
        y
      } = coords,
      yOffset = y - docTop;
    if (yOffset < 0) return 0;
    if (yOffset > docHeight) return view.state.doc.length;
    // Scan for a text block near the queried y position
    for (let halfLine = view.viewState.heightOracle.textHeight / 2, bounced = false;;) {
      block = view.elementAtHeight(yOffset);
      if (block.type == BlockType.Text) break;
      for (;;) {
        // Move the y position out of this block
        yOffset = bias > 0 ? block.bottom + halfLine : block.top - halfLine;
        if (yOffset >= 0 && yOffset <= docHeight) break;
        // If the document consists entirely of replaced widgets, we
        // won't find a text block, so return 0
        if (bounced) return precise ? null : 0;
        bounced = true;
        bias = -bias;
      }
    }
    y = docTop + yOffset;
    let lineStart = block.from;
    // If this is outside of the rendered viewport, we can't determine a position
    if (lineStart < view.viewport.from) return view.viewport.from == 0 ? 0 : precise ? null : posAtCoordsImprecise(view, content, block, x, y);
    if (lineStart > view.viewport.to) return view.viewport.to == view.state.doc.length ? view.state.doc.length : precise ? null : posAtCoordsImprecise(view, content, block, x, y);
    // Prefer ShadowRootOrDocument.elementFromPoint if present, fall back to document if not
    let doc = view.dom.ownerDocument;
    let root = view.root.elementFromPoint ? view.root : doc;
    let element = root.elementFromPoint(x, y);
    if (element && !view.contentDOM.contains(element)) element = null;
    // If the element is unexpected, clip x at the sides of the content area and try again
    if (!element) {
      x = Math.max(content.left + 1, Math.min(content.right - 1, x));
      element = root.elementFromPoint(x, y);
      if (element && !view.contentDOM.contains(element)) element = null;
    }
    // There's visible editor content under the point, so we can try
    // using caret(Position|Range)FromPoint as a shortcut
    let node,
      offset = -1;
    if (element && ((_a = view.docView.nearest(element)) === null || _a === void 0 ? void 0 : _a.isEditable) != false) {
      if (doc.caretPositionFromPoint) {
        let pos = doc.caretPositionFromPoint(x, y);
        if (pos) ({
          offsetNode: node,
          offset
        } = pos);
      } else if (doc.caretRangeFromPoint) {
        let range = doc.caretRangeFromPoint(x, y);
        if (range) {
          ({
            startContainer: node,
            startOffset: offset
          } = range);
          if (!view.contentDOM.contains(node) || browser.safari && isSuspiciousSafariCaretResult(node, offset, x) || browser.chrome && isSuspiciousChromeCaretResult(node, offset, x)) node = undefined;
        }
      }
      // Chrome will return offsets into <input> elements without child
      // nodes, which will lead to a null deref below, so clip the
      // offset to the node size.
      if (node) offset = Math.min(maxOffset(node), offset);
    }
    // No luck, do our own (potentially expensive) search
    if (!node || !view.docView.dom.contains(node)) {
      let line = LineView.find(view.docView, lineStart);
      if (!line) return yOffset > block.top + block.height / 2 ? block.to : block.from;
      ({
        node,
        offset
      } = domPosAtCoords(line.dom, x, y));
    }
    let nearest = view.docView.nearest(node);
    if (!nearest) return null;
    if (nearest.isWidget && ((_b = nearest.dom) === null || _b === void 0 ? void 0 : _b.nodeType) == 1) {
      let rect = nearest.dom.getBoundingClientRect();
      return coords.y < rect.top || coords.y <= rect.bottom && coords.x <= (rect.left + rect.right) / 2 ? nearest.posAtStart : nearest.posAtEnd;
    } else {
      return nearest.localPosFromDOM(node, offset) + nearest.posAtStart;
    }
  }
  function posAtCoordsImprecise(view, contentRect, block, x, y) {
    let into = Math.round((x - contentRect.left) * view.defaultCharacterWidth);
    if (view.lineWrapping && block.height > view.defaultLineHeight * 1.5) {
      let textHeight = view.viewState.heightOracle.textHeight;
      let line = Math.floor((y - block.top - (view.defaultLineHeight - textHeight) * 0.5) / textHeight);
      into += line * view.viewState.heightOracle.lineLength;
    }
    let content = view.state.sliceDoc(block.from, block.to);
    return block.from + findColumn(content, into, view.state.tabSize);
  }
  // In case of a high line height, Safari's caretRangeFromPoint treats
  // the space between lines as belonging to the last character of the
  // line before. This is used to detect such a result so that it can be
  // ignored (issue #401).
  function isSuspiciousSafariCaretResult(node, offset, x) {
    let len;
    if (node.nodeType != 3 || offset != (len = node.nodeValue.length)) return false;
    for (let next = node.nextSibling; next; next = next.nextSibling) if (next.nodeType != 1 || next.nodeName != "BR") return false;
    return textRange(node, len - 1, len).getBoundingClientRect().left > x;
  }
  // Chrome will move positions between lines to the start of the next line
  function isSuspiciousChromeCaretResult(node, offset, x) {
    if (offset != 0) return false;
    for (let cur = node;;) {
      let parent = cur.parentNode;
      if (!parent || parent.nodeType != 1 || parent.firstChild != cur) return false;
      if (parent.classList.contains("cm-line")) break;
      cur = parent;
    }
    let rect = node.nodeType == 1 ? node.getBoundingClientRect() : textRange(node, 0, Math.max(node.nodeValue.length, 1)).getBoundingClientRect();
    return x - rect.left > 5;
  }
  function blockAt(view, pos) {
    let line = view.lineBlockAt(pos);
    if (Array.isArray(line.type)) for (let l of line.type) {
      if (l.to > pos || l.to == pos && (l.to == line.to || l.type == BlockType.Text)) return l;
    }
    return line;
  }
  function moveToLineBoundary(view, start, forward, includeWrap) {
    let line = blockAt(view, start.head);
    let coords = !includeWrap || line.type != BlockType.Text || !(view.lineWrapping || line.widgetLineBreaks) ? null : view.coordsAtPos(start.assoc < 0 && start.head > line.from ? start.head - 1 : start.head);
    if (coords) {
      let editorRect = view.dom.getBoundingClientRect();
      let direction = view.textDirectionAt(line.from);
      let pos = view.posAtCoords({
        x: forward == (direction == Direction.LTR) ? editorRect.right - 1 : editorRect.left + 1,
        y: (coords.top + coords.bottom) / 2
      });
      if (pos != null) return EditorSelection.cursor(pos, forward ? -1 : 1);
    }
    return EditorSelection.cursor(forward ? line.to : line.from, forward ? -1 : 1);
  }
  function moveByChar(view, start, forward, by) {
    let line = view.state.doc.lineAt(start.head),
      spans = view.bidiSpans(line);
    let direction = view.textDirectionAt(line.from);
    for (let cur = start, check = null;;) {
      let next = moveVisually(line, spans, direction, cur, forward),
        char = movedOver;
      if (!next) {
        if (line.number == (forward ? view.state.doc.lines : 1)) return cur;
        char = "\n";
        line = view.state.doc.line(line.number + (forward ? 1 : -1));
        spans = view.bidiSpans(line);
        next = view.visualLineSide(line, !forward);
      }
      if (!check) {
        if (!by) return next;
        check = by(char);
      } else if (!check(char)) {
        return cur;
      }
      cur = next;
    }
  }
  function byGroup(view, pos, start) {
    let categorize = view.state.charCategorizer(pos);
    let cat = categorize(start);
    return next => {
      let nextCat = categorize(next);
      if (cat == CharCategory.Space) cat = nextCat;
      return cat == nextCat;
    };
  }
  function moveVertically(view, start, forward, distance) {
    let startPos = start.head,
      dir = forward ? 1 : -1;
    if (startPos == (forward ? view.state.doc.length : 0)) return EditorSelection.cursor(startPos, start.assoc);
    let goal = start.goalColumn,
      startY;
    let rect = view.contentDOM.getBoundingClientRect();
    let startCoords = view.coordsAtPos(startPos, start.assoc || -1),
      docTop = view.documentTop;
    if (startCoords) {
      if (goal == null) goal = startCoords.left - rect.left;
      startY = dir < 0 ? startCoords.top : startCoords.bottom;
    } else {
      let line = view.viewState.lineBlockAt(startPos);
      if (goal == null) goal = Math.min(rect.right - rect.left, view.defaultCharacterWidth * (startPos - line.from));
      startY = (dir < 0 ? line.top : line.bottom) + docTop;
    }
    let resolvedGoal = rect.left + goal;
    let dist = distance !== null && distance !== void 0 ? distance : view.viewState.heightOracle.textHeight >> 1;
    for (let extra = 0;; extra += 10) {
      let curY = startY + (dist + extra) * dir;
      let pos = posAtCoords(view, {
        x: resolvedGoal,
        y: curY
      }, false, dir);
      if (curY < rect.top || curY > rect.bottom || (dir < 0 ? pos < startPos : pos > startPos)) {
        let charRect = view.docView.coordsForChar(pos);
        let assoc = !charRect || curY < charRect.top ? -1 : 1;
        return EditorSelection.cursor(pos, assoc, undefined, goal);
      }
    }
  }
  function skipAtomicRanges(atoms, pos, bias) {
    for (;;) {
      let moved = 0;
      for (let set of atoms) {
        set.between(pos - 1, pos + 1, (from, to, value) => {
          if (pos > from && pos < to) {
            let side = moved || bias || (pos - from < to - pos ? -1 : 1);
            pos = side < 0 ? from : to;
            moved = side;
          }
        });
      }
      if (!moved) return pos;
    }
  }
  function skipAtoms(view, oldPos, pos) {
    let newPos = skipAtomicRanges(view.state.facet(atomicRanges).map(f => f(view)), pos.from, oldPos.head > pos.from ? -1 : 1);
    return newPos == pos.from ? pos : EditorSelection.cursor(newPos, newPos < pos.from ? 1 : -1);
  }
  const LineBreakPlaceholder = "\uffff";
  class DOMReader {
    constructor(points, state) {
      this.points = points;
      this.text = "";
      this.lineSeparator = state.facet(EditorState.lineSeparator);
    }
    append(text) {
      this.text += text;
    }
    lineBreak() {
      this.text += LineBreakPlaceholder;
    }
    readRange(start, end) {
      if (!start) return this;
      let parent = start.parentNode;
      for (let cur = start;;) {
        this.findPointBefore(parent, cur);
        let oldLen = this.text.length;
        this.readNode(cur);
        let next = cur.nextSibling;
        if (next == end) break;
        let view = ContentView.get(cur),
          nextView = ContentView.get(next);
        if (view && nextView ? view.breakAfter : (view ? view.breakAfter : isBlockElement(cur)) || isBlockElement(next) && (cur.nodeName != "BR" || cur.cmIgnore) && this.text.length > oldLen) this.lineBreak();
        cur = next;
      }
      this.findPointBefore(parent, end);
      return this;
    }
    readTextNode(node) {
      let text = node.nodeValue;
      for (let point of this.points) if (point.node == node) point.pos = this.text.length + Math.min(point.offset, text.length);
      for (let off = 0, re = this.lineSeparator ? null : /\r\n?|\n/g;;) {
        let nextBreak = -1,
          breakSize = 1,
          m;
        if (this.lineSeparator) {
          nextBreak = text.indexOf(this.lineSeparator, off);
          breakSize = this.lineSeparator.length;
        } else if (m = re.exec(text)) {
          nextBreak = m.index;
          breakSize = m[0].length;
        }
        this.append(text.slice(off, nextBreak < 0 ? text.length : nextBreak));
        if (nextBreak < 0) break;
        this.lineBreak();
        if (breakSize > 1) for (let point of this.points) if (point.node == node && point.pos > this.text.length) point.pos -= breakSize - 1;
        off = nextBreak + breakSize;
      }
    }
    readNode(node) {
      if (node.cmIgnore) return;
      let view = ContentView.get(node);
      let fromView = view && view.overrideDOMText;
      if (fromView != null) {
        this.findPointInside(node, fromView.length);
        for (let i = fromView.iter(); !i.next().done;) {
          if (i.lineBreak) this.lineBreak();else this.append(i.value);
        }
      } else if (node.nodeType == 3) {
        this.readTextNode(node);
      } else if (node.nodeName == "BR") {
        if (node.nextSibling) this.lineBreak();
      } else if (node.nodeType == 1) {
        this.readRange(node.firstChild, null);
      }
    }
    findPointBefore(node, next) {
      for (let point of this.points) if (point.node == node && node.childNodes[point.offset] == next) point.pos = this.text.length;
    }
    findPointInside(node, length) {
      for (let point of this.points) if (node.nodeType == 3 ? point.node == node : node.contains(point.node)) point.pos = this.text.length + (isAtEnd(node, point.node, point.offset) ? length : 0);
    }
  }
  function isAtEnd(parent, node, offset) {
    for (;;) {
      if (!node || offset < maxOffset(node)) return false;
      if (node == parent) return true;
      offset = domIndex(node) + 1;
      node = node.parentNode;
    }
  }
  class DOMPoint {
    constructor(node, offset) {
      this.node = node;
      this.offset = offset;
      this.pos = -1;
    }
  }
  class DOMChange {
    constructor(view, start, end, typeOver) {
      this.typeOver = typeOver;
      this.bounds = null;
      this.text = "";
      this.domChanged = start > -1;
      let {
        impreciseHead: iHead,
        impreciseAnchor: iAnchor
      } = view.docView;
      if (view.state.readOnly && start > -1) {
        // Ignore changes when the editor is read-only
        this.newSel = null;
      } else if (start > -1 && (this.bounds = view.docView.domBoundsAround(start, end, 0))) {
        let selPoints = iHead || iAnchor ? [] : selectionPoints(view);
        let reader = new DOMReader(selPoints, view.state);
        reader.readRange(this.bounds.startDOM, this.bounds.endDOM);
        this.text = reader.text;
        this.newSel = selectionFromPoints(selPoints, this.bounds.from);
      } else {
        let domSel = view.observer.selectionRange;
        let head = iHead && iHead.node == domSel.focusNode && iHead.offset == domSel.focusOffset || !contains(view.contentDOM, domSel.focusNode) ? view.state.selection.main.head : view.docView.posFromDOM(domSel.focusNode, domSel.focusOffset);
        let anchor = iAnchor && iAnchor.node == domSel.anchorNode && iAnchor.offset == domSel.anchorOffset || !contains(view.contentDOM, domSel.anchorNode) ? view.state.selection.main.anchor : view.docView.posFromDOM(domSel.anchorNode, domSel.anchorOffset);
        // iOS will refuse to select the block gaps when doing
        // select-all.
        // Chrome will put the selection *inside* them, confusing
        // posFromDOM
        let vp = view.viewport;
        if ((browser.ios || browser.chrome) && view.state.selection.main.empty && head != anchor && (vp.from > 0 || vp.to < view.state.doc.length)) {
          let from = Math.min(head, anchor),
            to = Math.max(head, anchor);
          let offFrom = vp.from - from,
            offTo = vp.to - to;
          if ((offFrom == 0 || offFrom == 1 || from == 0) && (offTo == 0 || offTo == -1 || to == view.state.doc.length)) {
            head = 0;
            anchor = view.state.doc.length;
          }
        }
        this.newSel = EditorSelection.single(anchor, head);
      }
    }
  }
  function applyDOMChange(view, domChange) {
    let change;
    let {
        newSel
      } = domChange,
      sel = view.state.selection.main;
    let lastKey = view.inputState.lastKeyTime > Date.now() - 100 ? view.inputState.lastKeyCode : -1;
    if (domChange.bounds) {
      let {
        from,
        to
      } = domChange.bounds;
      let preferredPos = sel.from,
        preferredSide = null;
      // Prefer anchoring to end when Backspace is pressed (or, on
      // Android, when something was deleted)
      if (lastKey === 8 || browser.android && domChange.text.length < to - from) {
        preferredPos = sel.to;
        preferredSide = "end";
      }
      let diff = findDiff(view.state.doc.sliceString(from, to, LineBreakPlaceholder), domChange.text, preferredPos - from, preferredSide);
      if (diff) {
        // Chrome inserts two newlines when pressing shift-enter at the
        // end of a line. DomChange drops one of those.
        if (browser.chrome && lastKey == 13 && diff.toB == diff.from + 2 && domChange.text.slice(diff.from, diff.toB) == LineBreakPlaceholder + LineBreakPlaceholder) diff.toB--;
        change = {
          from: from + diff.from,
          to: from + diff.toA,
          insert: Text.of(domChange.text.slice(diff.from, diff.toB).split(LineBreakPlaceholder))
        };
      }
    } else if (newSel && (!view.hasFocus && view.state.facet(editable) || newSel.main.eq(sel))) {
      newSel = null;
    }
    if (!change && !newSel) return false;
    if (!change && domChange.typeOver && !sel.empty && newSel && newSel.main.empty) {
      // Heuristic to notice typing over a selected character
      change = {
        from: sel.from,
        to: sel.to,
        insert: view.state.doc.slice(sel.from, sel.to)
      };
    } else if ((browser.mac || browser.android) && change && change.from == change.to && change.from == sel.head - 1 && /^\. ?$/.test(change.insert.toString()) && view.contentDOM.getAttribute("autocorrect") == "off") {
      // Detect insert-period-on-double-space Mac and Android behavior,
      // and transform it into a regular space insert.
      if (newSel && change.insert.length == 2) newSel = EditorSelection.single(newSel.main.anchor - 1, newSel.main.head - 1);
      change = {
        from: change.from,
        to: change.to,
        insert: Text.of([change.insert.toString().replace(".", " ")])
      };
    } else if (change && change.from >= sel.from && change.to <= sel.to && (change.from != sel.from || change.to != sel.to) && sel.to - sel.from - (change.to - change.from) <= 4) {
      // If the change is inside the selection and covers most of it,
      // assume it is a selection replace (with identical characters at
      // the start/end not included in the diff)
      change = {
        from: sel.from,
        to: sel.to,
        insert: view.state.doc.slice(sel.from, change.from).append(change.insert).append(view.state.doc.slice(change.to, sel.to))
      };
    } else if (browser.chrome && change && change.from == change.to && change.from == sel.head && change.insert.toString() == "\n " && view.lineWrapping) {
      // In Chrome, if you insert a space at the start of a wrapped
      // line, it will actually insert a newline and a space, causing a
      // bogus new line to be created in CodeMirror (#968)
      if (newSel) newSel = EditorSelection.single(newSel.main.anchor - 1, newSel.main.head - 1);
      change = {
        from: sel.from,
        to: sel.to,
        insert: Text.of([" "])
      };
    }
    if (change) {
      return applyDOMChangeInner(view, change, newSel, lastKey);
    } else if (newSel && !newSel.main.eq(sel)) {
      let scrollIntoView = false,
        userEvent = "select";
      if (view.inputState.lastSelectionTime > Date.now() - 50) {
        if (view.inputState.lastSelectionOrigin == "select") scrollIntoView = true;
        userEvent = view.inputState.lastSelectionOrigin;
      }
      view.dispatch({
        selection: newSel,
        scrollIntoView,
        userEvent
      });
      return true;
    } else {
      return false;
    }
  }
  function applyDOMChangeInner(view, change, newSel, lastKey = -1) {
    if (browser.ios && view.inputState.flushIOSKey(change)) return true;
    let sel = view.state.selection.main;
    // Android browsers don't fire reasonable key events for enter,
    // backspace, or delete. So this detects changes that look like
    // they're caused by those keys, and reinterprets them as key
    // events. (Some of these keys are also handled by beforeinput
    // events and the pendingAndroidKey mechanism, but that's not
    // reliable in all situations.)
    if (browser.android && (change.to == sel.to && (
    // GBoard will sometimes remove a space it just inserted
    // after a completion when you press enter
    change.from == sel.from || change.from == sel.from - 1 && view.state.sliceDoc(change.from, sel.from) == " ") && change.insert.length == 1 && change.insert.lines == 2 && dispatchKey(view.contentDOM, "Enter", 13) || (change.from == sel.from - 1 && change.to == sel.to && change.insert.length == 0 || lastKey == 8 && change.insert.length < change.to - change.from && change.to > sel.head) && dispatchKey(view.contentDOM, "Backspace", 8) || change.from == sel.from && change.to == sel.to + 1 && change.insert.length == 0 && dispatchKey(view.contentDOM, "Delete", 46))) return true;
    let text = change.insert.toString();
    if (view.inputState.composing >= 0) view.inputState.composing++;
    let defaultTr;
    let defaultInsert = () => defaultTr || (defaultTr = applyDefaultInsert(view, change, newSel));
    if (!view.state.facet(inputHandler).some(h => h(view, change.from, change.to, text, defaultInsert))) view.dispatch(defaultInsert());
    return true;
  }
  function applyDefaultInsert(view, change, newSel) {
    let tr,
      startState = view.state,
      sel = startState.selection.main;
    if (change.from >= sel.from && change.to <= sel.to && change.to - change.from >= (sel.to - sel.from) / 3 && (!newSel || newSel.main.empty && newSel.main.from == change.from + change.insert.length) && view.inputState.composing < 0) {
      let before = sel.from < change.from ? startState.sliceDoc(sel.from, change.from) : "";
      let after = sel.to > change.to ? startState.sliceDoc(change.to, sel.to) : "";
      tr = startState.replaceSelection(view.state.toText(before + change.insert.sliceString(0, undefined, view.state.lineBreak) + after));
    } else {
      let changes = startState.changes(change);
      let mainSel = newSel && newSel.main.to <= changes.newLength ? newSel.main : undefined;
      // Try to apply a composition change to all cursors
      if (startState.selection.ranges.length > 1 && view.inputState.composing >= 0 && change.to <= sel.to && change.to >= sel.to - 10) {
        let replaced = view.state.sliceDoc(change.from, change.to);
        let compositionRange,
          composition = newSel && findCompositionNode(view, newSel.main.head);
        if (composition) {
          let dLen = change.insert.length - (change.to - change.from);
          compositionRange = {
            from: composition.from,
            to: composition.to - dLen
          };
        } else {
          compositionRange = view.state.doc.lineAt(sel.head);
        }
        let offset = sel.to - change.to,
          size = sel.to - sel.from;
        tr = startState.changeByRange(range => {
          if (range.from == sel.from && range.to == sel.to) return {
            changes,
            range: mainSel || range.map(changes)
          };
          let to = range.to - offset,
            from = to - replaced.length;
          if (range.to - range.from != size || view.state.sliceDoc(from, to) != replaced ||
          // Unfortunately, there's no way to make multiple
          // changes in the same node work without aborting
          // composition, so cursors in the composition range are
          // ignored.
          range.to >= compositionRange.from && range.from <= compositionRange.to) return {
            range
          };
          let rangeChanges = startState.changes({
              from,
              to,
              insert: change.insert
            }),
            selOff = range.to - sel.to;
          return {
            changes: rangeChanges,
            range: !mainSel ? range.map(rangeChanges) : EditorSelection.range(Math.max(0, mainSel.anchor + selOff), Math.max(0, mainSel.head + selOff))
          };
        });
      } else {
        tr = {
          changes,
          selection: mainSel && startState.selection.replaceRange(mainSel)
        };
      }
    }
    let userEvent = "input.type";
    if (view.composing || view.inputState.compositionPendingChange && view.inputState.compositionEndedAt > Date.now() - 50) {
      view.inputState.compositionPendingChange = false;
      userEvent += ".compose";
      if (view.inputState.compositionFirstChange) {
        userEvent += ".start";
        view.inputState.compositionFirstChange = false;
      }
    }
    return startState.update(tr, {
      userEvent,
      scrollIntoView: true
    });
  }
  function findDiff(a, b, preferredPos, preferredSide) {
    let minLen = Math.min(a.length, b.length);
    let from = 0;
    while (from < minLen && a.charCodeAt(from) == b.charCodeAt(from)) from++;
    if (from == minLen && a.length == b.length) return null;
    let toA = a.length,
      toB = b.length;
    while (toA > 0 && toB > 0 && a.charCodeAt(toA - 1) == b.charCodeAt(toB - 1)) {
      toA--;
      toB--;
    }
    if (preferredSide == "end") {
      let adjust = Math.max(0, from - Math.min(toA, toB));
      preferredPos -= toA + adjust - from;
    }
    if (toA < from && a.length < b.length) {
      let move = preferredPos <= from && preferredPos >= toA ? from - preferredPos : 0;
      from -= move;
      toB = from + (toB - toA);
      toA = from;
    } else if (toB < from) {
      let move = preferredPos <= from && preferredPos >= toB ? from - preferredPos : 0;
      from -= move;
      toA = from + (toA - toB);
      toB = from;
    }
    return {
      from,
      toA,
      toB
    };
  }
  function selectionPoints(view) {
    let result = [];
    if (view.root.activeElement != view.contentDOM) return result;
    let {
      anchorNode,
      anchorOffset,
      focusNode,
      focusOffset
    } = view.observer.selectionRange;
    if (anchorNode) {
      result.push(new DOMPoint(anchorNode, anchorOffset));
      if (focusNode != anchorNode || focusOffset != anchorOffset) result.push(new DOMPoint(focusNode, focusOffset));
    }
    return result;
  }
  function selectionFromPoints(points, base) {
    if (points.length == 0) return null;
    let anchor = points[0].pos,
      head = points.length == 2 ? points[1].pos : anchor;
    return anchor > -1 && head > -1 ? EditorSelection.single(anchor + base, head + base) : null;
  }
  class InputState {
    setSelectionOrigin(origin) {
      this.lastSelectionOrigin = origin;
      this.lastSelectionTime = Date.now();
    }
    constructor(view) {
      this.view = view;
      this.lastKeyCode = 0;
      this.lastKeyTime = 0;
      this.lastTouchTime = 0;
      this.lastFocusTime = 0;
      this.lastScrollTop = 0;
      this.lastScrollLeft = 0;
      // On iOS, some keys need to have their default behavior happen
      // (after which we retroactively handle them and reset the DOM) to
      // avoid messing up the virtual keyboard state.
      this.pendingIOSKey = undefined;
      /**
      When enabled (>-1), tab presses are not given to key handlers,
      leaving the browser's default behavior. If >0, the mode expires
      at that timestamp, and any other keypress clears it.
      Esc enables temporary tab focus mode for two seconds when not
      otherwise handled.
      */
      this.tabFocusMode = -1;
      this.lastSelectionOrigin = null;
      this.lastSelectionTime = 0;
      this.lastContextMenu = 0;
      this.scrollHandlers = [];
      this.handlers = Object.create(null);
      // -1 means not in a composition. Otherwise, this counts the number
      // of changes made during the composition. The count is used to
      // avoid treating the start state of the composition, before any
      // changes have been made, as part of the composition.
      this.composing = -1;
      // Tracks whether the next change should be marked as starting the
      // composition (null means no composition, true means next is the
      // first, false means first has already been marked for this
      // composition)
      this.compositionFirstChange = null;
      // End time of the previous composition
      this.compositionEndedAt = 0;
      // Used in a kludge to detect when an Enter keypress should be
      // considered part of the composition on Safari, which fires events
      // in the wrong order
      this.compositionPendingKey = false;
      // Used to categorize changes as part of a composition, even when
      // the mutation events fire shortly after the compositionend event
      this.compositionPendingChange = false;
      this.mouseSelection = null;
      // When a drag from the editor is active, this points at the range
      // being dragged.
      this.draggedContent = null;
      this.handleEvent = this.handleEvent.bind(this);
      this.notifiedFocused = view.hasFocus;
      // On Safari adding an input event handler somehow prevents an
      // issue where the composition vanishes when you press enter.
      if (browser.safari) view.contentDOM.addEventListener("input", () => null);
      if (browser.gecko) firefoxCopyCutHack(view.contentDOM.ownerDocument);
    }
    handleEvent(event) {
      if (!eventBelongsToEditor(this.view, event) || this.ignoreDuringComposition(event)) return;
      if (event.type == "keydown" && this.keydown(event)) return;
      this.runHandlers(event.type, event);
    }
    runHandlers(type, event) {
      let handlers = this.handlers[type];
      if (handlers) {
        for (let observer of handlers.observers) observer(this.view, event);
        for (let handler of handlers.handlers) {
          if (event.defaultPrevented) break;
          if (handler(this.view, event)) {
            event.preventDefault();
            break;
          }
        }
      }
    }
    ensureHandlers(plugins) {
      let handlers = computeHandlers(plugins),
        prev = this.handlers,
        dom = this.view.contentDOM;
      for (let type in handlers) if (type != "scroll") {
        let passive = !handlers[type].handlers.length;
        let exists = prev[type];
        if (exists && passive != !exists.handlers.length) {
          dom.removeEventListener(type, this.handleEvent);
          exists = null;
        }
        if (!exists) dom.addEventListener(type, this.handleEvent, {
          passive
        });
      }
      for (let type in prev) if (type != "scroll" && !handlers[type]) dom.removeEventListener(type, this.handleEvent);
      this.handlers = handlers;
    }
    keydown(event) {
      // Must always run, even if a custom handler handled the event
      this.lastKeyCode = event.keyCode;
      this.lastKeyTime = Date.now();
      if (event.keyCode == 9 && this.tabFocusMode > -1 && (!this.tabFocusMode || Date.now() <= this.tabFocusMode)) return true;
      if (this.tabFocusMode > 0 && event.keyCode != 27 && modifierCodes.indexOf(event.keyCode) < 0) this.tabFocusMode = -1;
      // Chrome for Android usually doesn't fire proper key events, but
      // occasionally does, usually surrounded by a bunch of complicated
      // composition changes. When an enter or backspace key event is
      // seen, hold off on handling DOM events for a bit, and then
      // dispatch it.
      if (browser.android && browser.chrome && !event.synthetic && (event.keyCode == 13 || event.keyCode == 8)) {
        this.view.observer.delayAndroidKey(event.key, event.keyCode);
        return true;
      }
      // Preventing the default behavior of Enter on iOS makes the
      // virtual keyboard get stuck in the wrong (lowercase)
      // state. So we let it go through, and then, in
      // applyDOMChange, notify key handlers of it and reset to
      // the state they produce.
      let pending;
      if (browser.ios && !event.synthetic && !event.altKey && !event.metaKey && ((pending = PendingKeys.find(key => key.keyCode == event.keyCode)) && !event.ctrlKey || EmacsyPendingKeys.indexOf(event.key) > -1 && event.ctrlKey && !event.shiftKey)) {
        this.pendingIOSKey = pending || event;
        setTimeout(() => this.flushIOSKey(), 250);
        return true;
      }
      if (event.keyCode != 229) this.view.observer.forceFlush();
      return false;
    }
    flushIOSKey(change) {
      let key = this.pendingIOSKey;
      if (!key) return false;
      // This looks like an autocorrection before Enter
      if (key.key == "Enter" && change && change.from < change.to && /^\S+$/.test(change.insert.toString())) return false;
      this.pendingIOSKey = undefined;
      return dispatchKey(this.view.contentDOM, key.key, key.keyCode, key instanceof KeyboardEvent ? key : undefined);
    }
    ignoreDuringComposition(event) {
      if (!/^key/.test(event.type)) return false;
      if (this.composing > 0) return true;
      // See https://www.stum.de/2016/06/24/handling-ime-events-in-javascript/.
      // On some input method editors (IMEs), the Enter key is used to
      // confirm character selection. On Safari, when Enter is pressed,
      // compositionend and keydown events are sometimes emitted in the
      // wrong order. The key event should still be ignored, even when
      // it happens after the compositionend event.
      if (browser.safari && !browser.ios && this.compositionPendingKey && Date.now() - this.compositionEndedAt < 100) {
        this.compositionPendingKey = false;
        return true;
      }
      return false;
    }
    startMouseSelection(mouseSelection) {
      if (this.mouseSelection) this.mouseSelection.destroy();
      this.mouseSelection = mouseSelection;
    }
    update(update) {
      this.view.observer.update(update);
      if (this.mouseSelection) this.mouseSelection.update(update);
      if (this.draggedContent && update.docChanged) this.draggedContent = this.draggedContent.map(update.changes);
      if (update.transactions.length) this.lastKeyCode = this.lastSelectionTime = 0;
    }
    destroy() {
      if (this.mouseSelection) this.mouseSelection.destroy();
    }
  }
  function bindHandler(plugin, handler) {
    return (view, event) => {
      try {
        return handler.call(plugin, event, view);
      } catch (e) {
        logException(view.state, e);
      }
    };
  }
  function computeHandlers(plugins) {
    let result = Object.create(null);
    function record(type) {
      return result[type] || (result[type] = {
        observers: [],
        handlers: []
      });
    }
    for (let plugin of plugins) {
      let spec = plugin.spec;
      if (spec && spec.domEventHandlers) for (let type in spec.domEventHandlers) {
        let f = spec.domEventHandlers[type];
        if (f) record(type).handlers.push(bindHandler(plugin.value, f));
      }
      if (spec && spec.domEventObservers) for (let type in spec.domEventObservers) {
        let f = spec.domEventObservers[type];
        if (f) record(type).observers.push(bindHandler(plugin.value, f));
      }
    }
    for (let type in handlers) record(type).handlers.push(handlers[type]);
    for (let type in observers) record(type).observers.push(observers[type]);
    return result;
  }
  const PendingKeys = [{
    key: "Backspace",
    keyCode: 8,
    inputType: "deleteContentBackward"
  }, {
    key: "Enter",
    keyCode: 13,
    inputType: "insertParagraph"
  }, {
    key: "Enter",
    keyCode: 13,
    inputType: "insertLineBreak"
  }, {
    key: "Delete",
    keyCode: 46,
    inputType: "deleteContentForward"
  }];
  const EmacsyPendingKeys = "dthko";
  // Key codes for modifier keys
  const modifierCodes = [16, 17, 18, 20, 91, 92, 224, 225];
  const dragScrollMargin = 6;
  function dragScrollSpeed(dist) {
    return Math.max(0, dist) * 0.7 + 8;
  }
  function dist(a, b) {
    return Math.max(Math.abs(a.clientX - b.clientX), Math.abs(a.clientY - b.clientY));
  }
  class MouseSelection {
    constructor(view, startEvent, style, mustSelect) {
      this.view = view;
      this.startEvent = startEvent;
      this.style = style;
      this.mustSelect = mustSelect;
      this.scrollSpeed = {
        x: 0,
        y: 0
      };
      this.scrolling = -1;
      this.lastEvent = startEvent;
      this.scrollParents = scrollableParents(view.contentDOM);
      this.atoms = view.state.facet(atomicRanges).map(f => f(view));
      let doc = view.contentDOM.ownerDocument;
      doc.addEventListener("mousemove", this.move = this.move.bind(this));
      doc.addEventListener("mouseup", this.up = this.up.bind(this));
      this.extend = startEvent.shiftKey;
      this.multiple = view.state.facet(EditorState.allowMultipleSelections) && addsSelectionRange(view, startEvent);
      this.dragging = isInPrimarySelection(view, startEvent) && getClickType(startEvent) == 1 ? null : false;
    }
    start(event) {
      // When clicking outside of the selection, immediately apply the
      // effect of starting the selection
      if (this.dragging === false) this.select(event);
    }
    move(event) {
      if (event.buttons == 0) return this.destroy();
      if (this.dragging || this.dragging == null && dist(this.startEvent, event) < 10) return;
      this.select(this.lastEvent = event);
      let sx = 0,
        sy = 0;
      let left = 0,
        top = 0,
        right = this.view.win.innerWidth,
        bottom = this.view.win.innerHeight;
      if (this.scrollParents.x) ({
        left,
        right
      } = this.scrollParents.x.getBoundingClientRect());
      if (this.scrollParents.y) ({
        top,
        bottom
      } = this.scrollParents.y.getBoundingClientRect());
      let margins = getScrollMargins(this.view);
      if (event.clientX - margins.left <= left + dragScrollMargin) sx = -dragScrollSpeed(left - event.clientX);else if (event.clientX + margins.right >= right - dragScrollMargin) sx = dragScrollSpeed(event.clientX - right);
      if (event.clientY - margins.top <= top + dragScrollMargin) sy = -dragScrollSpeed(top - event.clientY);else if (event.clientY + margins.bottom >= bottom - dragScrollMargin) sy = dragScrollSpeed(event.clientY - bottom);
      this.setScrollSpeed(sx, sy);
    }
    up(event) {
      if (this.dragging == null) this.select(this.lastEvent);
      if (!this.dragging) event.preventDefault();
      this.destroy();
    }
    destroy() {
      this.setScrollSpeed(0, 0);
      let doc = this.view.contentDOM.ownerDocument;
      doc.removeEventListener("mousemove", this.move);
      doc.removeEventListener("mouseup", this.up);
      this.view.inputState.mouseSelection = this.view.inputState.draggedContent = null;
    }
    setScrollSpeed(sx, sy) {
      this.scrollSpeed = {
        x: sx,
        y: sy
      };
      if (sx || sy) {
        if (this.scrolling < 0) this.scrolling = setInterval(() => this.scroll(), 50);
      } else if (this.scrolling > -1) {
        clearInterval(this.scrolling);
        this.scrolling = -1;
      }
    }
    scroll() {
      let {
        x,
        y
      } = this.scrollSpeed;
      if (x && this.scrollParents.x) {
        this.scrollParents.x.scrollLeft += x;
        x = 0;
      }
      if (y && this.scrollParents.y) {
        this.scrollParents.y.scrollTop += y;
        y = 0;
      }
      if (x || y) this.view.win.scrollBy(x, y);
      if (this.dragging === false) this.select(this.lastEvent);
    }
    skipAtoms(sel) {
      let ranges = null;
      for (let i = 0; i < sel.ranges.length; i++) {
        let range = sel.ranges[i],
          updated = null;
        if (range.empty) {
          let pos = skipAtomicRanges(this.atoms, range.from, 0);
          if (pos != range.from) updated = EditorSelection.cursor(pos, -1);
        } else {
          let from = skipAtomicRanges(this.atoms, range.from, -1);
          let to = skipAtomicRanges(this.atoms, range.to, 1);
          if (from != range.from || to != range.to) updated = EditorSelection.range(range.from == range.anchor ? from : to, range.from == range.head ? from : to);
        }
        if (updated) {
          if (!ranges) ranges = sel.ranges.slice();
          ranges[i] = updated;
        }
      }
      return ranges ? EditorSelection.create(ranges, sel.mainIndex) : sel;
    }
    select(event) {
      let {
          view
        } = this,
        selection = this.skipAtoms(this.style.get(event, this.extend, this.multiple));
      if (this.mustSelect || !selection.eq(view.state.selection, this.dragging === false)) this.view.dispatch({
        selection,
        userEvent: "select.pointer"
      });
      this.mustSelect = false;
    }
    update(update) {
      if (update.transactions.some(tr => tr.isUserEvent("input.type"))) this.destroy();else if (this.style.update(update)) setTimeout(() => this.select(this.lastEvent), 20);
    }
  }
  function addsSelectionRange(view, event) {
    let facet = view.state.facet(clickAddsSelectionRange);
    return facet.length ? facet[0](event) : browser.mac ? event.metaKey : event.ctrlKey;
  }
  function dragMovesSelection(view, event) {
    let facet = view.state.facet(dragMovesSelection$1);
    return facet.length ? facet[0](event) : browser.mac ? !event.altKey : !event.ctrlKey;
  }
  function isInPrimarySelection(view, event) {
    let {
      main
    } = view.state.selection;
    if (main.empty) return false;
    // On boundary clicks, check whether the coordinates are inside the
    // selection's client rectangles
    let sel = getSelection(view.root);
    if (!sel || sel.rangeCount == 0) return true;
    let rects = sel.getRangeAt(0).getClientRects();
    for (let i = 0; i < rects.length; i++) {
      let rect = rects[i];
      if (rect.left <= event.clientX && rect.right >= event.clientX && rect.top <= event.clientY && rect.bottom >= event.clientY) return true;
    }
    return false;
  }
  function eventBelongsToEditor(view, event) {
    if (!event.bubbles) return true;
    if (event.defaultPrevented) return false;
    for (let node = event.target, cView; node != view.contentDOM; node = node.parentNode) if (!node || node.nodeType == 11 || (cView = ContentView.get(node)) && cView.ignoreEvent(event)) return false;
    return true;
  }
  const handlers = /*@__PURE__*/Object.create(null);
  const observers = /*@__PURE__*/Object.create(null);
  // This is very crude, but unfortunately both these browsers _pretend_
  // that they have a clipboard API—all the objects and methods are
  // there, they just don't work, and they are hard to test.
  const brokenClipboardAPI = browser.ie && browser.ie_version < 15 || browser.ios && browser.webkit_version < 604;
  function capturePaste(view) {
    let parent = view.dom.parentNode;
    if (!parent) return;
    let target = parent.appendChild(document.createElement("textarea"));
    target.style.cssText = "position: fixed; left: -10000px; top: 10px";
    target.focus();
    setTimeout(() => {
      view.focus();
      target.remove();
      doPaste(view, target.value);
    }, 50);
  }
  function textFilter(state, facet, text) {
    for (let filter of state.facet(facet)) text = filter(text, state);
    return text;
  }
  function doPaste(view, input) {
    input = textFilter(view.state, clipboardInputFilter, input);
    let {
        state
      } = view,
      changes,
      i = 1,
      text = state.toText(input);
    let byLine = text.lines == state.selection.ranges.length;
    let linewise = lastLinewiseCopy != null && state.selection.ranges.every(r => r.empty) && lastLinewiseCopy == text.toString();
    if (linewise) {
      let lastLine = -1;
      changes = state.changeByRange(range => {
        let line = state.doc.lineAt(range.from);
        if (line.from == lastLine) return {
          range
        };
        lastLine = line.from;
        let insert = state.toText((byLine ? text.line(i++).text : input) + state.lineBreak);
        return {
          changes: {
            from: line.from,
            insert
          },
          range: EditorSelection.cursor(range.from + insert.length)
        };
      });
    } else if (byLine) {
      changes = state.changeByRange(range => {
        let line = text.line(i++);
        return {
          changes: {
            from: range.from,
            to: range.to,
            insert: line.text
          },
          range: EditorSelection.cursor(range.from + line.length)
        };
      });
    } else {
      changes = state.replaceSelection(text);
    }
    view.dispatch(changes, {
      userEvent: "input.paste",
      scrollIntoView: true
    });
  }
  observers.scroll = view => {
    view.inputState.lastScrollTop = view.scrollDOM.scrollTop;
    view.inputState.lastScrollLeft = view.scrollDOM.scrollLeft;
  };
  handlers.keydown = (view, event) => {
    view.inputState.setSelectionOrigin("select");
    if (event.keyCode == 27 && view.inputState.tabFocusMode != 0) view.inputState.tabFocusMode = Date.now() + 2000;
    return false;
  };
  observers.touchstart = (view, e) => {
    view.inputState.lastTouchTime = Date.now();
    view.inputState.setSelectionOrigin("select.pointer");
  };
  observers.touchmove = view => {
    view.inputState.setSelectionOrigin("select.pointer");
  };
  handlers.mousedown = (view, event) => {
    view.observer.flush();
    if (view.inputState.lastTouchTime > Date.now() - 2000) return false; // Ignore touch interaction
    let style = null;
    for (let makeStyle of view.state.facet(mouseSelectionStyle)) {
      style = makeStyle(view, event);
      if (style) break;
    }
    if (!style && event.button == 0) style = basicMouseSelection(view, event);
    if (style) {
      let mustFocus = !view.hasFocus;
      view.inputState.startMouseSelection(new MouseSelection(view, event, style, mustFocus));
      if (mustFocus) view.observer.ignore(() => {
        focusPreventScroll(view.contentDOM);
        let active = view.root.activeElement;
        if (active && !active.contains(view.contentDOM)) active.blur();
      });
      let mouseSel = view.inputState.mouseSelection;
      if (mouseSel) {
        mouseSel.start(event);
        return mouseSel.dragging === false;
      }
    }
    return false;
  };
  function rangeForClick(view, pos, bias, type) {
    if (type == 1) {
      // Single click
      return EditorSelection.cursor(pos, bias);
    } else if (type == 2) {
      // Double click
      return groupAt(view.state, pos, bias);
    } else {
      // Triple click
      let visual = LineView.find(view.docView, pos),
        line = view.state.doc.lineAt(visual ? visual.posAtEnd : pos);
      let from = visual ? visual.posAtStart : line.from,
        to = visual ? visual.posAtEnd : line.to;
      if (to < view.state.doc.length && to == line.to) to++;
      return EditorSelection.range(from, to);
    }
  }
  let inside = (x, y, rect) => y >= rect.top && y <= rect.bottom && x >= rect.left && x <= rect.right;
  // Try to determine, for the given coordinates, associated with the
  // given position, whether they are related to the element before or
  // the element after the position.
  function findPositionSide(view, pos, x, y) {
    let line = LineView.find(view.docView, pos);
    if (!line) return 1;
    let off = pos - line.posAtStart;
    // Line boundaries point into the line
    if (off == 0) return 1;
    if (off == line.length) return -1;
    // Positions on top of an element point at that element
    let before = line.coordsAt(off, -1);
    if (before && inside(x, y, before)) return -1;
    let after = line.coordsAt(off, 1);
    if (after && inside(x, y, after)) return 1;
    // This is probably a line wrap point. Pick before if the point is
    // above its bottom.
    return before && before.bottom >= y ? -1 : 1;
  }
  function queryPos(view, event) {
    let pos = view.posAtCoords({
      x: event.clientX,
      y: event.clientY
    }, false);
    return {
      pos,
      bias: findPositionSide(view, pos, event.clientX, event.clientY)
    };
  }
  const BadMouseDetail = browser.ie && browser.ie_version <= 11;
  let lastMouseDown = null,
    lastMouseDownCount = 0,
    lastMouseDownTime = 0;
  function getClickType(event) {
    if (!BadMouseDetail) return event.detail;
    let last = lastMouseDown,
      lastTime = lastMouseDownTime;
    lastMouseDown = event;
    lastMouseDownTime = Date.now();
    return lastMouseDownCount = !last || lastTime > Date.now() - 400 && Math.abs(last.clientX - event.clientX) < 2 && Math.abs(last.clientY - event.clientY) < 2 ? (lastMouseDownCount + 1) % 3 : 1;
  }
  function basicMouseSelection(view, event) {
    let start = queryPos(view, event),
      type = getClickType(event);
    let startSel = view.state.selection;
    return {
      update(update) {
        if (update.docChanged) {
          start.pos = update.changes.mapPos(start.pos);
          startSel = startSel.map(update.changes);
        }
      },
      get(event, extend, multiple) {
        let cur = queryPos(view, event),
          removed;
        let range = rangeForClick(view, cur.pos, cur.bias, type);
        if (start.pos != cur.pos && !extend) {
          let startRange = rangeForClick(view, start.pos, start.bias, type);
          let from = Math.min(startRange.from, range.from),
            to = Math.max(startRange.to, range.to);
          range = from < range.from ? EditorSelection.range(from, to) : EditorSelection.range(to, from);
        }
        if (extend) return startSel.replaceRange(startSel.main.extend(range.from, range.to));else if (multiple && type == 1 && startSel.ranges.length > 1 && (removed = removeRangeAround(startSel, cur.pos))) return removed;else if (multiple) return startSel.addRange(range);else return EditorSelection.create([range]);
      }
    };
  }
  function removeRangeAround(sel, pos) {
    for (let i = 0; i < sel.ranges.length; i++) {
      let {
        from,
        to
      } = sel.ranges[i];
      if (from <= pos && to >= pos) return EditorSelection.create(sel.ranges.slice(0, i).concat(sel.ranges.slice(i + 1)), sel.mainIndex == i ? 0 : sel.mainIndex - (sel.mainIndex > i ? 1 : 0));
    }
    return null;
  }
  handlers.dragstart = (view, event) => {
    let {
      selection: {
        main: range
      }
    } = view.state;
    if (event.target.draggable) {
      let cView = view.docView.nearest(event.target);
      if (cView && cView.isWidget) {
        let from = cView.posAtStart,
          to = from + cView.length;
        if (from >= range.to || to <= range.from) range = EditorSelection.range(from, to);
      }
    }
    let {
      inputState
    } = view;
    if (inputState.mouseSelection) inputState.mouseSelection.dragging = true;
    inputState.draggedContent = range;
    if (event.dataTransfer) {
      event.dataTransfer.setData("Text", textFilter(view.state, clipboardOutputFilter, view.state.sliceDoc(range.from, range.to)));
      event.dataTransfer.effectAllowed = "copyMove";
    }
    return false;
  };
  handlers.dragend = view => {
    view.inputState.draggedContent = null;
    return false;
  };
  function dropText(view, event, text, direct) {
    text = textFilter(view.state, clipboardInputFilter, text);
    if (!text) return;
    let dropPos = view.posAtCoords({
      x: event.clientX,
      y: event.clientY
    }, false);
    let {
      draggedContent
    } = view.inputState;
    let del = direct && draggedContent && dragMovesSelection(view, event) ? {
      from: draggedContent.from,
      to: draggedContent.to
    } : null;
    let ins = {
      from: dropPos,
      insert: text
    };
    let changes = view.state.changes(del ? [del, ins] : ins);
    view.focus();
    view.dispatch({
      changes,
      selection: {
        anchor: changes.mapPos(dropPos, -1),
        head: changes.mapPos(dropPos, 1)
      },
      userEvent: del ? "move.drop" : "input.drop"
    });
    view.inputState.draggedContent = null;
  }
  handlers.drop = (view, event) => {
    if (!event.dataTransfer) return false;
    if (view.state.readOnly) return true;
    let files = event.dataTransfer.files;
    if (files && files.length) {
      // For a file drop, read the file's text.
      let text = Array(files.length),
        read = 0;
      let finishFile = () => {
        if (++read == files.length) dropText(view, event, text.filter(s => s != null).join(view.state.lineBreak), false);
      };
      for (let i = 0; i < files.length; i++) {
        let reader = new FileReader();
        reader.onerror = finishFile;
        reader.onload = () => {
          if (!/[\x00-\x08\x0e-\x1f]{2}/.test(reader.result)) text[i] = reader.result;
          finishFile();
        };
        reader.readAsText(files[i]);
      }
      return true;
    } else {
      let text = event.dataTransfer.getData("Text");
      if (text) {
        dropText(view, event, text, true);
        return true;
      }
    }
    return false;
  };
  handlers.paste = (view, event) => {
    if (view.state.readOnly) return true;
    view.observer.flush();
    let data = brokenClipboardAPI ? null : event.clipboardData;
    if (data) {
      doPaste(view, data.getData("text/plain") || data.getData("text/uri-list"));
      return true;
    } else {
      capturePaste(view);
      return false;
    }
  };
  function captureCopy(view, text) {
    // The extra wrapper is somehow necessary on IE/Edge to prevent the
    // content from being mangled when it is put onto the clipboard
    let parent = view.dom.parentNode;
    if (!parent) return;
    let target = parent.appendChild(document.createElement("textarea"));
    target.style.cssText = "position: fixed; left: -10000px; top: 10px";
    target.value = text;
    target.focus();
    target.selectionEnd = text.length;
    target.selectionStart = 0;
    setTimeout(() => {
      target.remove();
      view.focus();
    }, 50);
  }
  function copiedRange(state) {
    let content = [],
      ranges = [],
      linewise = false;
    for (let range of state.selection.ranges) if (!range.empty) {
      content.push(state.sliceDoc(range.from, range.to));
      ranges.push(range);
    }
    if (!content.length) {
      // Nothing selected, do a line-wise copy
      let upto = -1;
      for (let {
        from
      } of state.selection.ranges) {
        let line = state.doc.lineAt(from);
        if (line.number > upto) {
          content.push(line.text);
          ranges.push({
            from: line.from,
            to: Math.min(state.doc.length, line.to + 1)
          });
        }
        upto = line.number;
      }
      linewise = true;
    }
    return {
      text: textFilter(state, clipboardOutputFilter, content.join(state.lineBreak)),
      ranges,
      linewise
    };
  }
  let lastLinewiseCopy = null;
  handlers.copy = handlers.cut = (view, event) => {
    let {
      text,
      ranges,
      linewise
    } = copiedRange(view.state);
    if (!text && !linewise) return false;
    lastLinewiseCopy = linewise ? text : null;
    if (event.type == "cut" && !view.state.readOnly) view.dispatch({
      changes: ranges,
      scrollIntoView: true,
      userEvent: "delete.cut"
    });
    let data = brokenClipboardAPI ? null : event.clipboardData;
    if (data) {
      data.clearData();
      data.setData("text/plain", text);
      return true;
    } else {
      captureCopy(view, text);
      return false;
    }
  };
  const isFocusChange = /*@__PURE__*/Annotation.define();
  function focusChangeTransaction(state, focus) {
    let effects = [];
    for (let getEffect of state.facet(focusChangeEffect)) {
      let effect = getEffect(state, focus);
      if (effect) effects.push(effect);
    }
    return effects ? state.update({
      effects,
      annotations: isFocusChange.of(true)
    }) : null;
  }
  function updateForFocusChange(view) {
    setTimeout(() => {
      let focus = view.hasFocus;
      if (focus != view.inputState.notifiedFocused) {
        let tr = focusChangeTransaction(view.state, focus);
        if (tr) view.dispatch(tr);else view.update([]);
      }
    }, 10);
  }
  observers.focus = view => {
    view.inputState.lastFocusTime = Date.now();
    // When focusing reset the scroll position, move it back to where it was
    if (!view.scrollDOM.scrollTop && (view.inputState.lastScrollTop || view.inputState.lastScrollLeft)) {
      view.scrollDOM.scrollTop = view.inputState.lastScrollTop;
      view.scrollDOM.scrollLeft = view.inputState.lastScrollLeft;
    }
    updateForFocusChange(view);
  };
  observers.blur = view => {
    view.observer.clearSelectionRange();
    updateForFocusChange(view);
  };
  observers.compositionstart = observers.compositionupdate = view => {
    if (view.observer.editContext) return; // Composition handled by edit context
    if (view.inputState.compositionFirstChange == null) view.inputState.compositionFirstChange = true;
    if (view.inputState.composing < 0) {
      // FIXME possibly set a timeout to clear it again on Android
      view.inputState.composing = 0;
    }
  };
  observers.compositionend = view => {
    if (view.observer.editContext) return; // Composition handled by edit context
    view.inputState.composing = -1;
    view.inputState.compositionEndedAt = Date.now();
    view.inputState.compositionPendingKey = true;
    view.inputState.compositionPendingChange = view.observer.pendingRecords().length > 0;
    view.inputState.compositionFirstChange = null;
    if (browser.chrome && browser.android) {
      // Delay flushing for a bit on Android because it'll often fire a
      // bunch of contradictory changes in a row at end of compositon
      view.observer.flushSoon();
    } else if (view.inputState.compositionPendingChange) {
      // If we found pending records, schedule a flush.
      Promise.resolve().then(() => view.observer.flush());
    } else {
      // Otherwise, make sure that, if no changes come in soon, the
      // composition view is cleared.
      setTimeout(() => {
        if (view.inputState.composing < 0 && view.docView.hasComposition) view.update([]);
      }, 50);
    }
  };
  observers.contextmenu = view => {
    view.inputState.lastContextMenu = Date.now();
  };
  handlers.beforeinput = (view, event) => {
    var _a, _b;
    // In EditContext mode, we must handle insertReplacementText events
    // directly, to make spell checking corrections work
    if (event.inputType == "insertReplacementText" && view.observer.editContext) {
      let text = (_a = event.dataTransfer) === null || _a === void 0 ? void 0 : _a.getData("text/plain"),
        ranges = event.getTargetRanges();
      if (text && ranges.length) {
        let r = ranges[0];
        let from = view.posAtDOM(r.startContainer, r.startOffset),
          to = view.posAtDOM(r.endContainer, r.endOffset);
        applyDOMChangeInner(view, {
          from,
          to,
          insert: view.state.toText(text)
        }, null);
        return true;
      }
    }
    // Because Chrome Android doesn't fire useful key events, use
    // beforeinput to detect backspace (and possibly enter and delete,
    // but those usually don't even seem to fire beforeinput events at
    // the moment) and fake a key event for it.
    //
    // (preventDefault on beforeinput, though supported in the spec,
    // seems to do nothing at all on Chrome).
    let pending;
    if (browser.chrome && browser.android && (pending = PendingKeys.find(key => key.inputType == event.inputType))) {
      view.observer.delayAndroidKey(pending.key, pending.keyCode);
      if (pending.key == "Backspace" || pending.key == "Delete") {
        let startViewHeight = ((_b = window.visualViewport) === null || _b === void 0 ? void 0 : _b.height) || 0;
        setTimeout(() => {
          var _a;
          // Backspacing near uneditable nodes on Chrome Android sometimes
          // closes the virtual keyboard. This tries to crudely detect
          // that and refocus to get it back.
          if ((((_a = window.visualViewport) === null || _a === void 0 ? void 0 : _a.height) || 0) > startViewHeight + 10 && view.hasFocus) {
            view.contentDOM.blur();
            view.focus();
          }
        }, 100);
      }
    }
    if (browser.ios && event.inputType == "deleteContentForward") {
      // For some reason, DOM changes (and beforeinput) happen _before_
      // the key event for ctrl-d on iOS when using an external
      // keyboard.
      view.observer.flushSoon();
    }
    // Safari will occasionally forget to fire compositionend at the end of a dead-key composition
    if (browser.safari && event.inputType == "insertText" && view.inputState.composing >= 0) {
      setTimeout(() => observers.compositionend(view, event), 20);
    }
    return false;
  };
  const appliedFirefoxHack = /*@__PURE__*/new Set();
  // In Firefox, when cut/copy handlers are added to the document, that
  // somehow avoids a bug where those events aren't fired when the
  // selection is empty. See https://github.com/codemirror/dev/issues/1082
  // and https://bugzilla.mozilla.org/show_bug.cgi?id=995961
  function firefoxCopyCutHack(doc) {
    if (!appliedFirefoxHack.has(doc)) {
      appliedFirefoxHack.add(doc);
      doc.addEventListener("copy", () => {});
      doc.addEventListener("cut", () => {});
    }
  }
  const wrappingWhiteSpace = ["pre-wrap", "normal", "pre-line", "break-spaces"];
  // Used to track, during updateHeight, if any actual heights changed
  let heightChangeFlag = false;
  function clearHeightChangeFlag() {
    heightChangeFlag = false;
  }
  class HeightOracle {
    constructor(lineWrapping) {
      this.lineWrapping = lineWrapping;
      this.doc = Text.empty;
      this.heightSamples = {};
      this.lineHeight = 14; // The height of an entire line (line-height)
      this.charWidth = 7;
      this.textHeight = 14; // The height of the actual font (font-size)
      this.lineLength = 30;
    }
    heightForGap(from, to) {
      let lines = this.doc.lineAt(to).number - this.doc.lineAt(from).number + 1;
      if (this.lineWrapping) lines += Math.max(0, Math.ceil((to - from - lines * this.lineLength * 0.5) / this.lineLength));
      return this.lineHeight * lines;
    }
    heightForLine(length) {
      if (!this.lineWrapping) return this.lineHeight;
      let lines = 1 + Math.max(0, Math.ceil((length - this.lineLength) / (this.lineLength - 5)));
      return lines * this.lineHeight;
    }
    setDoc(doc) {
      this.doc = doc;
      return this;
    }
    mustRefreshForWrapping(whiteSpace) {
      return wrappingWhiteSpace.indexOf(whiteSpace) > -1 != this.lineWrapping;
    }
    mustRefreshForHeights(lineHeights) {
      let newHeight = false;
      for (let i = 0; i < lineHeights.length; i++) {
        let h = lineHeights[i];
        if (h < 0) {
          i++;
        } else if (!this.heightSamples[Math.floor(h * 10)]) {
          // Round to .1 pixels
          newHeight = true;
          this.heightSamples[Math.floor(h * 10)] = true;
        }
      }
      return newHeight;
    }
    refresh(whiteSpace, lineHeight, charWidth, textHeight, lineLength, knownHeights) {
      let lineWrapping = wrappingWhiteSpace.indexOf(whiteSpace) > -1;
      let changed = Math.round(lineHeight) != Math.round(this.lineHeight) || this.lineWrapping != lineWrapping;
      this.lineWrapping = lineWrapping;
      this.lineHeight = lineHeight;
      this.charWidth = charWidth;
      this.textHeight = textHeight;
      this.lineLength = lineLength;
      if (changed) {
        this.heightSamples = {};
        for (let i = 0; i < knownHeights.length; i++) {
          let h = knownHeights[i];
          if (h < 0) i++;else this.heightSamples[Math.floor(h * 10)] = true;
        }
      }
      return changed;
    }
  }
  // This object is used by `updateHeight` to make DOM measurements
  // arrive at the right nides. The `heights` array is a sequence of
  // block heights, starting from position `from`.
  class MeasuredHeights {
    constructor(from, heights) {
      this.from = from;
      this.heights = heights;
      this.index = 0;
    }
    get more() {
      return this.index < this.heights.length;
    }
  }
  /**
  Record used to represent information about a block-level element
  in the editor view.
  */
  class BlockInfo {
    /**
    @internal
    */
    constructor(
    /**
    The start of the element in the document.
    */
    from,
    /**
    The length of the element.
    */
    length,
    /**
    The top position of the element (relative to the top of the
    document).
    */
    top,
    /**
    Its height.
    */
    height,
    /**
    @internal Weird packed field that holds an array of children
    for composite blocks, a decoration for block widgets, and a
    number indicating the amount of widget-create line breaks for
    text blocks.
    */
    _content) {
      this.from = from;
      this.length = length;
      this.top = top;
      this.height = height;
      this._content = _content;
    }
    /**
    The type of element this is. When querying lines, this may be
    an array of all the blocks that make up the line.
    */
    get type() {
      return typeof this._content == "number" ? BlockType.Text : Array.isArray(this._content) ? this._content : this._content.type;
    }
    /**
    The end of the element as a document position.
    */
    get to() {
      return this.from + this.length;
    }
    /**
    The bottom position of the element.
    */
    get bottom() {
      return this.top + this.height;
    }
    /**
    If this is a widget block, this will return the widget
    associated with it.
    */
    get widget() {
      return this._content instanceof PointDecoration ? this._content.widget : null;
    }
    /**
    If this is a textblock, this holds the number of line breaks
    that appear in widgets inside the block.
    */
    get widgetLineBreaks() {
      return typeof this._content == "number" ? this._content : 0;
    }
    /**
    @internal
    */
    join(other) {
      let content = (Array.isArray(this._content) ? this._content : [this]).concat(Array.isArray(other._content) ? other._content : [other]);
      return new BlockInfo(this.from, this.length + other.length, this.top, this.height + other.height, content);
    }
  }
  var QueryType = /*@__PURE__*/function (QueryType) {
    QueryType[QueryType["ByPos"] = 0] = "ByPos";
    QueryType[QueryType["ByHeight"] = 1] = "ByHeight";
    QueryType[QueryType["ByPosNoHeight"] = 2] = "ByPosNoHeight";
    return QueryType;
  }(QueryType || (QueryType = {}));
  const Epsilon = 1e-3;
  class HeightMap {
    constructor(length,
    // The number of characters covered
    height,
    // Height of this part of the document
    flags = 2 /* Flag.Outdated */) {
      this.length = length;
      this.height = height;
      this.flags = flags;
    }
    get outdated() {
      return (this.flags & 2 /* Flag.Outdated */) > 0;
    }
    set outdated(value) {
      this.flags = (value ? 2 /* Flag.Outdated */ : 0) | this.flags & ~2 /* Flag.Outdated */;
    }
    setHeight(height) {
      if (this.height != height) {
        if (Math.abs(this.height - height) > Epsilon) heightChangeFlag = true;
        this.height = height;
      }
    }
    // Base case is to replace a leaf node, which simply builds a tree
    // from the new nodes and returns that (HeightMapBranch and
    // HeightMapGap override this to actually use from/to)
    replace(_from, _to, nodes) {
      return HeightMap.of(nodes);
    }
    // Again, these are base cases, and are overridden for branch and gap nodes.
    decomposeLeft(_to, result) {
      result.push(this);
    }
    decomposeRight(_from, result) {
      result.push(this);
    }
    applyChanges(decorations, oldDoc, oracle, changes) {
      let me = this,
        doc = oracle.doc;
      for (let i = changes.length - 1; i >= 0; i--) {
        let {
          fromA,
          toA,
          fromB,
          toB
        } = changes[i];
        let start = me.lineAt(fromA, QueryType.ByPosNoHeight, oracle.setDoc(oldDoc), 0, 0);
        let end = start.to >= toA ? start : me.lineAt(toA, QueryType.ByPosNoHeight, oracle, 0, 0);
        toB += end.to - toA;
        toA = end.to;
        while (i > 0 && start.from <= changes[i - 1].toA) {
          fromA = changes[i - 1].fromA;
          fromB = changes[i - 1].fromB;
          i--;
          if (fromA < start.from) start = me.lineAt(fromA, QueryType.ByPosNoHeight, oracle, 0, 0);
        }
        fromB += start.from - fromA;
        fromA = start.from;
        let nodes = NodeBuilder.build(oracle.setDoc(doc), decorations, fromB, toB);
        me = replace(me, me.replace(fromA, toA, nodes));
      }
      return me.updateHeight(oracle, 0);
    }
    static empty() {
      return new HeightMapText(0, 0);
    }
    // nodes uses null values to indicate the position of line breaks.
    // There are never line breaks at the start or end of the array, or
    // two line breaks next to each other, and the array isn't allowed
    // to be empty (same restrictions as return value from the builder).
    static of(nodes) {
      if (nodes.length == 1) return nodes[0];
      let i = 0,
        j = nodes.length,
        before = 0,
        after = 0;
      for (;;) {
        if (i == j) {
          if (before > after * 2) {
            let split = nodes[i - 1];
            if (split.break) nodes.splice(--i, 1, split.left, null, split.right);else nodes.splice(--i, 1, split.left, split.right);
            j += 1 + split.break;
            before -= split.size;
          } else if (after > before * 2) {
            let split = nodes[j];
            if (split.break) nodes.splice(j, 1, split.left, null, split.right);else nodes.splice(j, 1, split.left, split.right);
            j += 2 + split.break;
            after -= split.size;
          } else {
            break;
          }
        } else if (before < after) {
          let next = nodes[i++];
          if (next) before += next.size;
        } else {
          let next = nodes[--j];
          if (next) after += next.size;
        }
      }
      let brk = 0;
      if (nodes[i - 1] == null) {
        brk = 1;
        i--;
      } else if (nodes[i] == null) {
        brk = 1;
        j++;
      }
      return new HeightMapBranch(HeightMap.of(nodes.slice(0, i)), brk, HeightMap.of(nodes.slice(j)));
    }
  }
  function replace(old, val) {
    if (old == val) return old;
    if (old.constructor != val.constructor) heightChangeFlag = true;
    return val;
  }
  HeightMap.prototype.size = 1;
  class HeightMapBlock extends HeightMap {
    constructor(length, height, deco) {
      super(length, height);
      this.deco = deco;
    }
    blockAt(_height, _oracle, top, offset) {
      return new BlockInfo(offset, this.length, top, this.height, this.deco || 0);
    }
    lineAt(_value, _type, oracle, top, offset) {
      return this.blockAt(0, oracle, top, offset);
    }
    forEachLine(from, to, oracle, top, offset, f) {
      if (from <= offset + this.length && to >= offset) f(this.blockAt(0, oracle, top, offset));
    }
    updateHeight(oracle, offset = 0, _force = false, measured) {
      if (measured && measured.from <= offset && measured.more) this.setHeight(measured.heights[measured.index++]);
      this.outdated = false;
      return this;
    }
    toString() {
      return `block(${this.length})`;
    }
  }
  class HeightMapText extends HeightMapBlock {
    constructor(length, height) {
      super(length, height, null);
      this.collapsed = 0; // Amount of collapsed content in the line
      this.widgetHeight = 0; // Maximum inline widget height
      this.breaks = 0; // Number of widget-introduced line breaks on the line
    }
    blockAt(_height, _oracle, top, offset) {
      return new BlockInfo(offset, this.length, top, this.height, this.breaks);
    }
    replace(_from, _to, nodes) {
      let node = nodes[0];
      if (nodes.length == 1 && (node instanceof HeightMapText || node instanceof HeightMapGap && node.flags & 4 /* Flag.SingleLine */) && Math.abs(this.length - node.length) < 10) {
        if (node instanceof HeightMapGap) node = new HeightMapText(node.length, this.height);else node.height = this.height;
        if (!this.outdated) node.outdated = false;
        return node;
      } else {
        return HeightMap.of(nodes);
      }
    }
    updateHeight(oracle, offset = 0, force = false, measured) {
      if (measured && measured.from <= offset && measured.more) this.setHeight(measured.heights[measured.index++]);else if (force || this.outdated) this.setHeight(Math.max(this.widgetHeight, oracle.heightForLine(this.length - this.collapsed)) + this.breaks * oracle.lineHeight);
      this.outdated = false;
      return this;
    }
    toString() {
      return `line(${this.length}${this.collapsed ? -this.collapsed : ""}${this.widgetHeight ? ":" + this.widgetHeight : ""})`;
    }
  }
  class HeightMapGap extends HeightMap {
    constructor(length) {
      super(length, 0);
    }
    heightMetrics(oracle, offset) {
      let firstLine = oracle.doc.lineAt(offset).number,
        lastLine = oracle.doc.lineAt(offset + this.length).number;
      let lines = lastLine - firstLine + 1;
      let perLine,
        perChar = 0;
      if (oracle.lineWrapping) {
        let totalPerLine = Math.min(this.height, oracle.lineHeight * lines);
        perLine = totalPerLine / lines;
        if (this.length > lines + 1) perChar = (this.height - totalPerLine) / (this.length - lines - 1);
      } else {
        perLine = this.height / lines;
      }
      return {
        firstLine,
        lastLine,
        perLine,
        perChar
      };
    }
    blockAt(height, oracle, top, offset) {
      let {
        firstLine,
        lastLine,
        perLine,
        perChar
      } = this.heightMetrics(oracle, offset);
      if (oracle.lineWrapping) {
        let guess = offset + (height < oracle.lineHeight ? 0 : Math.round(Math.max(0, Math.min(1, (height - top) / this.height)) * this.length));
        let line = oracle.doc.lineAt(guess),
          lineHeight = perLine + line.length * perChar;
        let lineTop = Math.max(top, height - lineHeight / 2);
        return new BlockInfo(line.from, line.length, lineTop, lineHeight, 0);
      } else {
        let line = Math.max(0, Math.min(lastLine - firstLine, Math.floor((height - top) / perLine)));
        let {
          from,
          length
        } = oracle.doc.line(firstLine + line);
        return new BlockInfo(from, length, top + perLine * line, perLine, 0);
      }
    }
    lineAt(value, type, oracle, top, offset) {
      if (type == QueryType.ByHeight) return this.blockAt(value, oracle, top, offset);
      if (type == QueryType.ByPosNoHeight) {
        let {
          from,
          to
        } = oracle.doc.lineAt(value);
        return new BlockInfo(from, to - from, 0, 0, 0);
      }
      let {
        firstLine,
        perLine,
        perChar
      } = this.heightMetrics(oracle, offset);
      let line = oracle.doc.lineAt(value),
        lineHeight = perLine + line.length * perChar;
      let linesAbove = line.number - firstLine;
      let lineTop = top + perLine * linesAbove + perChar * (line.from - offset - linesAbove);
      return new BlockInfo(line.from, line.length, Math.max(top, Math.min(lineTop, top + this.height - lineHeight)), lineHeight, 0);
    }
    forEachLine(from, to, oracle, top, offset, f) {
      from = Math.max(from, offset);
      to = Math.min(to, offset + this.length);
      let {
        firstLine,
        perLine,
        perChar
      } = this.heightMetrics(oracle, offset);
      for (let pos = from, lineTop = top; pos <= to;) {
        let line = oracle.doc.lineAt(pos);
        if (pos == from) {
          let linesAbove = line.number - firstLine;
          lineTop += perLine * linesAbove + perChar * (from - offset - linesAbove);
        }
        let lineHeight = perLine + perChar * line.length;
        f(new BlockInfo(line.from, line.length, lineTop, lineHeight, 0));
        lineTop += lineHeight;
        pos = line.to + 1;
      }
    }
    replace(from, to, nodes) {
      let after = this.length - to;
      if (after > 0) {
        let last = nodes[nodes.length - 1];
        if (last instanceof HeightMapGap) nodes[nodes.length - 1] = new HeightMapGap(last.length + after);else nodes.push(null, new HeightMapGap(after - 1));
      }
      if (from > 0) {
        let first = nodes[0];
        if (first instanceof HeightMapGap) nodes[0] = new HeightMapGap(from + first.length);else nodes.unshift(new HeightMapGap(from - 1), null);
      }
      return HeightMap.of(nodes);
    }
    decomposeLeft(to, result) {
      result.push(new HeightMapGap(to - 1), null);
    }
    decomposeRight(from, result) {
      result.push(null, new HeightMapGap(this.length - from - 1));
    }
    updateHeight(oracle, offset = 0, force = false, measured) {
      let end = offset + this.length;
      if (measured && measured.from <= offset + this.length && measured.more) {
        // Fill in part of this gap with measured lines. We know there
        // can't be widgets or collapsed ranges in those lines, because
        // they would already have been added to the heightmap (gaps
        // only contain plain text).
        let nodes = [],
          pos = Math.max(offset, measured.from),
          singleHeight = -1;
        if (measured.from > offset) nodes.push(new HeightMapGap(measured.from - offset - 1).updateHeight(oracle, offset));
        while (pos <= end && measured.more) {
          let len = oracle.doc.lineAt(pos).length;
          if (nodes.length) nodes.push(null);
          let height = measured.heights[measured.index++];
          if (singleHeight == -1) singleHeight = height;else if (Math.abs(height - singleHeight) >= Epsilon) singleHeight = -2;
          let line = new HeightMapText(len, height);
          line.outdated = false;
          nodes.push(line);
          pos += len + 1;
        }
        if (pos <= end) nodes.push(null, new HeightMapGap(end - pos).updateHeight(oracle, pos));
        let result = HeightMap.of(nodes);
        if (singleHeight < 0 || Math.abs(result.height - this.height) >= Epsilon || Math.abs(singleHeight - this.heightMetrics(oracle, offset).perLine) >= Epsilon) heightChangeFlag = true;
        return replace(this, result);
      } else if (force || this.outdated) {
        this.setHeight(oracle.heightForGap(offset, offset + this.length));
        this.outdated = false;
      }
      return this;
    }
    toString() {
      return `gap(${this.length})`;
    }
  }
  class HeightMapBranch extends HeightMap {
    constructor(left, brk, right) {
      super(left.length + brk + right.length, left.height + right.height, brk | (left.outdated || right.outdated ? 2 /* Flag.Outdated */ : 0));
      this.left = left;
      this.right = right;
      this.size = left.size + right.size;
    }
    get break() {
      return this.flags & 1 /* Flag.Break */;
    }
    blockAt(height, oracle, top, offset) {
      let mid = top + this.left.height;
      return height < mid ? this.left.blockAt(height, oracle, top, offset) : this.right.blockAt(height, oracle, mid, offset + this.left.length + this.break);
    }
    lineAt(value, type, oracle, top, offset) {
      let rightTop = top + this.left.height,
        rightOffset = offset + this.left.length + this.break;
      let left = type == QueryType.ByHeight ? value < rightTop : value < rightOffset;
      let base = left ? this.left.lineAt(value, type, oracle, top, offset) : this.right.lineAt(value, type, oracle, rightTop, rightOffset);
      if (this.break || (left ? base.to < rightOffset : base.from > rightOffset)) return base;
      let subQuery = type == QueryType.ByPosNoHeight ? QueryType.ByPosNoHeight : QueryType.ByPos;
      if (left) return base.join(this.right.lineAt(rightOffset, subQuery, oracle, rightTop, rightOffset));else return this.left.lineAt(rightOffset, subQuery, oracle, top, offset).join(base);
    }
    forEachLine(from, to, oracle, top, offset, f) {
      let rightTop = top + this.left.height,
        rightOffset = offset + this.left.length + this.break;
      if (this.break) {
        if (from < rightOffset) this.left.forEachLine(from, to, oracle, top, offset, f);
        if (to >= rightOffset) this.right.forEachLine(from, to, oracle, rightTop, rightOffset, f);
      } else {
        let mid = this.lineAt(rightOffset, QueryType.ByPos, oracle, top, offset);
        if (from < mid.from) this.left.forEachLine(from, mid.from - 1, oracle, top, offset, f);
        if (mid.to >= from && mid.from <= to) f(mid);
        if (to > mid.to) this.right.forEachLine(mid.to + 1, to, oracle, rightTop, rightOffset, f);
      }
    }
    replace(from, to, nodes) {
      let rightStart = this.left.length + this.break;
      if (to < rightStart) return this.balanced(this.left.replace(from, to, nodes), this.right);
      if (from > this.left.length) return this.balanced(this.left, this.right.replace(from - rightStart, to - rightStart, nodes));
      let result = [];
      if (from > 0) this.decomposeLeft(from, result);
      let left = result.length;
      for (let node of nodes) result.push(node);
      if (from > 0) mergeGaps(result, left - 1);
      if (to < this.length) {
        let right = result.length;
        this.decomposeRight(to, result);
        mergeGaps(result, right);
      }
      return HeightMap.of(result);
    }
    decomposeLeft(to, result) {
      let left = this.left.length;
      if (to <= left) return this.left.decomposeLeft(to, result);
      result.push(this.left);
      if (this.break) {
        left++;
        if (to >= left) result.push(null);
      }
      if (to > left) this.right.decomposeLeft(to - left, result);
    }
    decomposeRight(from, result) {
      let left = this.left.length,
        right = left + this.break;
      if (from >= right) return this.right.decomposeRight(from - right, result);
      if (from < left) this.left.decomposeRight(from, result);
      if (this.break && from < right) result.push(null);
      result.push(this.right);
    }
    balanced(left, right) {
      if (left.size > 2 * right.size || right.size > 2 * left.size) return HeightMap.of(this.break ? [left, null, right] : [left, right]);
      this.left = replace(this.left, left);
      this.right = replace(this.right, right);
      this.setHeight(left.height + right.height);
      this.outdated = left.outdated || right.outdated;
      this.size = left.size + right.size;
      this.length = left.length + this.break + right.length;
      return this;
    }
    updateHeight(oracle, offset = 0, force = false, measured) {
      let {
          left,
          right
        } = this,
        rightStart = offset + left.length + this.break,
        rebalance = null;
      if (measured && measured.from <= offset + left.length && measured.more) rebalance = left = left.updateHeight(oracle, offset, force, measured);else left.updateHeight(oracle, offset, force);
      if (measured && measured.from <= rightStart + right.length && measured.more) rebalance = right = right.updateHeight(oracle, rightStart, force, measured);else right.updateHeight(oracle, rightStart, force);
      if (rebalance) return this.balanced(left, right);
      this.height = this.left.height + this.right.height;
      this.outdated = false;
      return this;
    }
    toString() {
      return this.left + (this.break ? " " : "-") + this.right;
    }
  }
  function mergeGaps(nodes, around) {
    let before, after;
    if (nodes[around] == null && (before = nodes[around - 1]) instanceof HeightMapGap && (after = nodes[around + 1]) instanceof HeightMapGap) nodes.splice(around - 1, 3, new HeightMapGap(before.length + 1 + after.length));
  }
  const relevantWidgetHeight = 5;
  class NodeBuilder {
    constructor(pos, oracle) {
      this.pos = pos;
      this.oracle = oracle;
      this.nodes = [];
      this.lineStart = -1;
      this.lineEnd = -1;
      this.covering = null;
      this.writtenTo = pos;
    }
    get isCovered() {
      return this.covering && this.nodes[this.nodes.length - 1] == this.covering;
    }
    span(_from, to) {
      if (this.lineStart > -1) {
        let end = Math.min(to, this.lineEnd),
          last = this.nodes[this.nodes.length - 1];
        if (last instanceof HeightMapText) last.length += end - this.pos;else if (end > this.pos || !this.isCovered) this.nodes.push(new HeightMapText(end - this.pos, -1));
        this.writtenTo = end;
        if (to > end) {
          this.nodes.push(null);
          this.writtenTo++;
          this.lineStart = -1;
        }
      }
      this.pos = to;
    }
    point(from, to, deco) {
      if (from < to || deco.heightRelevant) {
        let height = deco.widget ? deco.widget.estimatedHeight : 0;
        let breaks = deco.widget ? deco.widget.lineBreaks : 0;
        if (height < 0) height = this.oracle.lineHeight;
        let len = to - from;
        if (deco.block) {
          this.addBlock(new HeightMapBlock(len, height, deco));
        } else if (len || breaks || height >= relevantWidgetHeight) {
          this.addLineDeco(height, breaks, len);
        }
      } else if (to > from) {
        this.span(from, to);
      }
      if (this.lineEnd > -1 && this.lineEnd < this.pos) this.lineEnd = this.oracle.doc.lineAt(this.pos).to;
    }
    enterLine() {
      if (this.lineStart > -1) return;
      let {
        from,
        to
      } = this.oracle.doc.lineAt(this.pos);
      this.lineStart = from;
      this.lineEnd = to;
      if (this.writtenTo < from) {
        if (this.writtenTo < from - 1 || this.nodes[this.nodes.length - 1] == null) this.nodes.push(this.blankContent(this.writtenTo, from - 1));
        this.nodes.push(null);
      }
      if (this.pos > from) this.nodes.push(new HeightMapText(this.pos - from, -1));
      this.writtenTo = this.pos;
    }
    blankContent(from, to) {
      let gap = new HeightMapGap(to - from);
      if (this.oracle.doc.lineAt(from).to == to) gap.flags |= 4 /* Flag.SingleLine */;
      return gap;
    }
    ensureLine() {
      this.enterLine();
      let last = this.nodes.length ? this.nodes[this.nodes.length - 1] : null;
      if (last instanceof HeightMapText) return last;
      let line = new HeightMapText(0, -1);
      this.nodes.push(line);
      return line;
    }
    addBlock(block) {
      this.enterLine();
      let deco = block.deco;
      if (deco && deco.startSide > 0 && !this.isCovered) this.ensureLine();
      this.nodes.push(block);
      this.writtenTo = this.pos = this.pos + block.length;
      if (deco && deco.endSide > 0) this.covering = block;
    }
    addLineDeco(height, breaks, length) {
      let line = this.ensureLine();
      line.length += length;
      line.collapsed += length;
      line.widgetHeight = Math.max(line.widgetHeight, height);
      line.breaks += breaks;
      this.writtenTo = this.pos = this.pos + length;
    }
    finish(from) {
      let last = this.nodes.length == 0 ? null : this.nodes[this.nodes.length - 1];
      if (this.lineStart > -1 && !(last instanceof HeightMapText) && !this.isCovered) this.nodes.push(new HeightMapText(0, -1));else if (this.writtenTo < this.pos || last == null) this.nodes.push(this.blankContent(this.writtenTo, this.pos));
      let pos = from;
      for (let node of this.nodes) {
        if (node instanceof HeightMapText) node.updateHeight(this.oracle, pos);
        pos += node ? node.length : 1;
      }
      return this.nodes;
    }
    // Always called with a region that on both sides either stretches
    // to a line break or the end of the document.
    // The returned array uses null to indicate line breaks, but never
    // starts or ends in a line break, or has multiple line breaks next
    // to each other.
    static build(oracle, decorations, from, to) {
      let builder = new NodeBuilder(from, oracle);
      RangeSet.spans(decorations, from, to, builder, 0);
      return builder.finish(from);
    }
  }
  function heightRelevantDecoChanges(a, b, diff) {
    let comp = new DecorationComparator();
    RangeSet.compare(a, b, diff, comp, 0);
    return comp.changes;
  }
  class DecorationComparator {
    constructor() {
      this.changes = [];
    }
    compareRange() {}
    comparePoint(from, to, a, b) {
      if (from < to || a && a.heightRelevant || b && b.heightRelevant) addRange(from, to, this.changes, 5);
    }
  }
  function visiblePixelRange(dom, paddingTop) {
    let rect = dom.getBoundingClientRect();
    let doc = dom.ownerDocument,
      win = doc.defaultView || window;
    let left = Math.max(0, rect.left),
      right = Math.min(win.innerWidth, rect.right);
    let top = Math.max(0, rect.top),
      bottom = Math.min(win.innerHeight, rect.bottom);
    for (let parent = dom.parentNode; parent && parent != doc.body;) {
      if (parent.nodeType == 1) {
        let elt = parent;
        let style = window.getComputedStyle(elt);
        if ((elt.scrollHeight > elt.clientHeight || elt.scrollWidth > elt.clientWidth) && style.overflow != "visible") {
          let parentRect = elt.getBoundingClientRect();
          left = Math.max(left, parentRect.left);
          right = Math.min(right, parentRect.right);
          top = Math.max(top, parentRect.top);
          bottom = Math.min(parent == dom.parentNode ? win.innerHeight : bottom, parentRect.bottom);
        }
        parent = style.position == "absolute" || style.position == "fixed" ? elt.offsetParent : elt.parentNode;
      } else if (parent.nodeType == 11) {
        // Shadow root
        parent = parent.host;
      } else {
        break;
      }
    }
    return {
      left: left - rect.left,
      right: Math.max(left, right) - rect.left,
      top: top - (rect.top + paddingTop),
      bottom: Math.max(top, bottom) - (rect.top + paddingTop)
    };
  }
  function inWindow(elt) {
    let rect = elt.getBoundingClientRect(),
      win = elt.ownerDocument.defaultView || window;
    return rect.left < win.innerWidth && rect.right > 0 && rect.top < win.innerHeight && rect.bottom > 0;
  }
  function fullPixelRange(dom, paddingTop) {
    let rect = dom.getBoundingClientRect();
    return {
      left: 0,
      right: rect.right - rect.left,
      top: paddingTop,
      bottom: rect.bottom - (rect.top + paddingTop)
    };
  }
  // Line gaps are placeholder widgets used to hide pieces of overlong
  // lines within the viewport, as a kludge to keep the editor
  // responsive when a ridiculously long line is loaded into it.
  class LineGap {
    constructor(from, to, size, displaySize) {
      this.from = from;
      this.to = to;
      this.size = size;
      this.displaySize = displaySize;
    }
    static same(a, b) {
      if (a.length != b.length) return false;
      for (let i = 0; i < a.length; i++) {
        let gA = a[i],
          gB = b[i];
        if (gA.from != gB.from || gA.to != gB.to || gA.size != gB.size) return false;
      }
      return true;
    }
    draw(viewState, wrapping) {
      return Decoration.replace({
        widget: new LineGapWidget(this.displaySize * (wrapping ? viewState.scaleY : viewState.scaleX), wrapping)
      }).range(this.from, this.to);
    }
  }
  class LineGapWidget extends WidgetType {
    constructor(size, vertical) {
      super();
      this.size = size;
      this.vertical = vertical;
    }
    eq(other) {
      return other.size == this.size && other.vertical == this.vertical;
    }
    toDOM() {
      let elt = document.createElement("div");
      if (this.vertical) {
        elt.style.height = this.size + "px";
      } else {
        elt.style.width = this.size + "px";
        elt.style.height = "2px";
        elt.style.display = "inline-block";
      }
      return elt;
    }
    get estimatedHeight() {
      return this.vertical ? this.size : -1;
    }
  }
  class ViewState {
    constructor(state) {
      this.state = state;
      // These are contentDOM-local coordinates
      this.pixelViewport = {
        left: 0,
        right: window.innerWidth,
        top: 0,
        bottom: 0
      };
      this.inView = true;
      this.paddingTop = 0; // Padding above the document, scaled
      this.paddingBottom = 0; // Padding below the document, scaled
      this.contentDOMWidth = 0; // contentDOM.getBoundingClientRect().width
      this.contentDOMHeight = 0; // contentDOM.getBoundingClientRect().height
      this.editorHeight = 0; // scrollDOM.clientHeight, unscaled
      this.editorWidth = 0; // scrollDOM.clientWidth, unscaled
      this.scrollTop = 0; // Last seen scrollDOM.scrollTop, scaled
      this.scrolledToBottom = false;
      // The CSS-transformation scale of the editor (transformed size /
      // concrete size)
      this.scaleX = 1;
      this.scaleY = 1;
      // The vertical position (document-relative) to which to anchor the
      // scroll position. -1 means anchor to the end of the document.
      this.scrollAnchorPos = 0;
      // The height at the anchor position. Set by the DOM update phase.
      // -1 means no height available.
      this.scrollAnchorHeight = -1;
      // See VP.MaxDOMHeight
      this.scaler = IdScaler;
      this.scrollTarget = null;
      // Briefly set to true when printing, to disable viewport limiting
      this.printing = false;
      // Flag set when editor content was redrawn, so that the next
      // measure stage knows it must read DOM layout
      this.mustMeasureContent = true;
      this.defaultTextDirection = Direction.LTR;
      this.visibleRanges = [];
      // Cursor 'assoc' is only significant when the cursor is on a line
      // wrap point, where it must stick to the character that it is
      // associated with. Since browsers don't provide a reasonable
      // interface to set or query this, when a selection is set that
      // might cause this to be significant, this flag is set. The next
      // measure phase will check whether the cursor is on a line-wrapping
      // boundary and, if so, reset it to make sure it is positioned in
      // the right place.
      this.mustEnforceCursorAssoc = false;
      let guessWrapping = state.facet(contentAttributes).some(v => typeof v != "function" && v.class == "cm-lineWrapping");
      this.heightOracle = new HeightOracle(guessWrapping);
      this.stateDeco = state.facet(decorations).filter(d => typeof d != "function");
      this.heightMap = HeightMap.empty().applyChanges(this.stateDeco, Text.empty, this.heightOracle.setDoc(state.doc), [new ChangedRange(0, 0, 0, state.doc.length)]);
      for (let i = 0; i < 2; i++) {
        this.viewport = this.getViewport(0, null);
        if (!this.updateForViewport()) break;
      }
      this.updateViewportLines();
      this.lineGaps = this.ensureLineGaps([]);
      this.lineGapDeco = Decoration.set(this.lineGaps.map(gap => gap.draw(this, false)));
      this.computeVisibleRanges();
    }
    updateForViewport() {
      let viewports = [this.viewport],
        {
          main
        } = this.state.selection;
      for (let i = 0; i <= 1; i++) {
        let pos = i ? main.head : main.anchor;
        if (!viewports.some(({
          from,
          to
        }) => pos >= from && pos <= to)) {
          let {
            from,
            to
          } = this.lineBlockAt(pos);
          viewports.push(new Viewport(from, to));
        }
      }
      this.viewports = viewports.sort((a, b) => a.from - b.from);
      return this.updateScaler();
    }
    updateScaler() {
      let scaler = this.scaler;
      this.scaler = this.heightMap.height <= 7000000 /* VP.MaxDOMHeight */ ? IdScaler : new BigScaler(this.heightOracle, this.heightMap, this.viewports);
      return scaler.eq(this.scaler) ? 0 : 2 /* UpdateFlag.Height */;
    }
    updateViewportLines() {
      this.viewportLines = [];
      this.heightMap.forEachLine(this.viewport.from, this.viewport.to, this.heightOracle.setDoc(this.state.doc), 0, 0, block => {
        this.viewportLines.push(scaleBlock(block, this.scaler));
      });
    }
    update(update, scrollTarget = null) {
      this.state = update.state;
      let prevDeco = this.stateDeco;
      this.stateDeco = this.state.facet(decorations).filter(d => typeof d != "function");
      let contentChanges = update.changedRanges;
      let heightChanges = ChangedRange.extendWithRanges(contentChanges, heightRelevantDecoChanges(prevDeco, this.stateDeco, update ? update.changes : ChangeSet.empty(this.state.doc.length)));
      let prevHeight = this.heightMap.height;
      let scrollAnchor = this.scrolledToBottom ? null : this.scrollAnchorAt(this.scrollTop);
      clearHeightChangeFlag();
      this.heightMap = this.heightMap.applyChanges(this.stateDeco, update.startState.doc, this.heightOracle.setDoc(this.state.doc), heightChanges);
      if (this.heightMap.height != prevHeight || heightChangeFlag) update.flags |= 2 /* UpdateFlag.Height */;
      if (scrollAnchor) {
        this.scrollAnchorPos = update.changes.mapPos(scrollAnchor.from, -1);
        this.scrollAnchorHeight = scrollAnchor.top;
      } else {
        this.scrollAnchorPos = -1;
        this.scrollAnchorHeight = this.heightMap.height;
      }
      let viewport = heightChanges.length ? this.mapViewport(this.viewport, update.changes) : this.viewport;
      if (scrollTarget && (scrollTarget.range.head < viewport.from || scrollTarget.range.head > viewport.to) || !this.viewportIsAppropriate(viewport)) viewport = this.getViewport(0, scrollTarget);
      let viewportChange = viewport.from != this.viewport.from || viewport.to != this.viewport.to;
      this.viewport = viewport;
      update.flags |= this.updateForViewport();
      if (viewportChange || !update.changes.empty || update.flags & 2 /* UpdateFlag.Height */) this.updateViewportLines();
      if (this.lineGaps.length || this.viewport.to - this.viewport.from > 2000 /* LG.Margin */ << 1) this.updateLineGaps(this.ensureLineGaps(this.mapLineGaps(this.lineGaps, update.changes)));
      update.flags |= this.computeVisibleRanges(update.changes);
      if (scrollTarget) this.scrollTarget = scrollTarget;
      if (!this.mustEnforceCursorAssoc && update.selectionSet && update.view.lineWrapping && update.state.selection.main.empty && update.state.selection.main.assoc && !update.state.facet(nativeSelectionHidden)) this.mustEnforceCursorAssoc = true;
    }
    measure(view) {
      let dom = view.contentDOM,
        style = window.getComputedStyle(dom);
      let oracle = this.heightOracle;
      let whiteSpace = style.whiteSpace;
      this.defaultTextDirection = style.direction == "rtl" ? Direction.RTL : Direction.LTR;
      let refresh = this.heightOracle.mustRefreshForWrapping(whiteSpace);
      let domRect = dom.getBoundingClientRect();
      let measureContent = refresh || this.mustMeasureContent || this.contentDOMHeight != domRect.height;
      this.contentDOMHeight = domRect.height;
      this.mustMeasureContent = false;
      let result = 0,
        bias = 0;
      if (domRect.width && domRect.height) {
        let {
          scaleX,
          scaleY
        } = getScale(dom, domRect);
        if (scaleX > .005 && Math.abs(this.scaleX - scaleX) > .005 || scaleY > .005 && Math.abs(this.scaleY - scaleY) > .005) {
          this.scaleX = scaleX;
          this.scaleY = scaleY;
          result |= 16 /* UpdateFlag.Geometry */;
          refresh = measureContent = true;
        }
      }
      // Vertical padding
      let paddingTop = (parseInt(style.paddingTop) || 0) * this.scaleY;
      let paddingBottom = (parseInt(style.paddingBottom) || 0) * this.scaleY;
      if (this.paddingTop != paddingTop || this.paddingBottom != paddingBottom) {
        this.paddingTop = paddingTop;
        this.paddingBottom = paddingBottom;
        result |= 16 /* UpdateFlag.Geometry */ | 2 /* UpdateFlag.Height */;
      }
      if (this.editorWidth != view.scrollDOM.clientWidth) {
        if (oracle.lineWrapping) measureContent = true;
        this.editorWidth = view.scrollDOM.clientWidth;
        result |= 16 /* UpdateFlag.Geometry */;
      }
      let scrollTop = view.scrollDOM.scrollTop * this.scaleY;
      if (this.scrollTop != scrollTop) {
        this.scrollAnchorHeight = -1;
        this.scrollTop = scrollTop;
      }
      this.scrolledToBottom = isScrolledToBottom(view.scrollDOM);
      // Pixel viewport
      let pixelViewport = (this.printing ? fullPixelRange : visiblePixelRange)(dom, this.paddingTop);
      let dTop = pixelViewport.top - this.pixelViewport.top,
        dBottom = pixelViewport.bottom - this.pixelViewport.bottom;
      this.pixelViewport = pixelViewport;
      let inView = this.pixelViewport.bottom > this.pixelViewport.top && this.pixelViewport.right > this.pixelViewport.left;
      if (inView != this.inView) {
        this.inView = inView;
        if (inView) measureContent = true;
      }
      if (!this.inView && !this.scrollTarget && !inWindow(view.dom)) return 0;
      let contentWidth = domRect.width;
      if (this.contentDOMWidth != contentWidth || this.editorHeight != view.scrollDOM.clientHeight) {
        this.contentDOMWidth = domRect.width;
        this.editorHeight = view.scrollDOM.clientHeight;
        result |= 16 /* UpdateFlag.Geometry */;
      }
      if (measureContent) {
        let lineHeights = view.docView.measureVisibleLineHeights(this.viewport);
        if (oracle.mustRefreshForHeights(lineHeights)) refresh = true;
        if (refresh || oracle.lineWrapping && Math.abs(contentWidth - this.contentDOMWidth) > oracle.charWidth) {
          let {
            lineHeight,
            charWidth,
            textHeight
          } = view.docView.measureTextSize();
          refresh = lineHeight > 0 && oracle.refresh(whiteSpace, lineHeight, charWidth, textHeight, contentWidth / charWidth, lineHeights);
          if (refresh) {
            view.docView.minWidth = 0;
            result |= 16 /* UpdateFlag.Geometry */;
          }
        }
        if (dTop > 0 && dBottom > 0) bias = Math.max(dTop, dBottom);else if (dTop < 0 && dBottom < 0) bias = Math.min(dTop, dBottom);
        clearHeightChangeFlag();
        for (let vp of this.viewports) {
          let heights = vp.from == this.viewport.from ? lineHeights : view.docView.measureVisibleLineHeights(vp);
          this.heightMap = (refresh ? HeightMap.empty().applyChanges(this.stateDeco, Text.empty, this.heightOracle, [new ChangedRange(0, 0, 0, view.state.doc.length)]) : this.heightMap).updateHeight(oracle, 0, refresh, new MeasuredHeights(vp.from, heights));
        }
        if (heightChangeFlag) result |= 2 /* UpdateFlag.Height */;
      }
      let viewportChange = !this.viewportIsAppropriate(this.viewport, bias) || this.scrollTarget && (this.scrollTarget.range.head < this.viewport.from || this.scrollTarget.range.head > this.viewport.to);
      if (viewportChange) {
        if (result & 2 /* UpdateFlag.Height */) result |= this.updateScaler();
        this.viewport = this.getViewport(bias, this.scrollTarget);
        result |= this.updateForViewport();
      }
      if (result & 2 /* UpdateFlag.Height */ || viewportChange) this.updateViewportLines();
      if (this.lineGaps.length || this.viewport.to - this.viewport.from > 2000 /* LG.Margin */ << 1) this.updateLineGaps(this.ensureLineGaps(refresh ? [] : this.lineGaps, view));
      result |= this.computeVisibleRanges();
      if (this.mustEnforceCursorAssoc) {
        this.mustEnforceCursorAssoc = false;
        // This is done in the read stage, because moving the selection
        // to a line end is going to trigger a layout anyway, so it
        // can't be a pure write. It should be rare that it does any
        // writing.
        view.docView.enforceCursorAssoc();
      }
      return result;
    }
    get visibleTop() {
      return this.scaler.fromDOM(this.pixelViewport.top);
    }
    get visibleBottom() {
      return this.scaler.fromDOM(this.pixelViewport.bottom);
    }
    getViewport(bias, scrollTarget) {
      // This will divide VP.Margin between the top and the
      // bottom, depending on the bias (the change in viewport position
      // since the last update). It'll hold a number between 0 and 1
      let marginTop = 0.5 - Math.max(-0.5, Math.min(0.5, bias / 1000 /* VP.Margin */ / 2));
      let map = this.heightMap,
        oracle = this.heightOracle;
      let {
        visibleTop,
        visibleBottom
      } = this;
      let viewport = new Viewport(map.lineAt(visibleTop - marginTop * 1000 /* VP.Margin */, QueryType.ByHeight, oracle, 0, 0).from, map.lineAt(visibleBottom + (1 - marginTop) * 1000 /* VP.Margin */, QueryType.ByHeight, oracle, 0, 0).to);
      // If scrollTarget is given, make sure the viewport includes that position
      if (scrollTarget) {
        let {
          head
        } = scrollTarget.range;
        if (head < viewport.from || head > viewport.to) {
          let viewHeight = Math.min(this.editorHeight, this.pixelViewport.bottom - this.pixelViewport.top);
          let block = map.lineAt(head, QueryType.ByPos, oracle, 0, 0),
            topPos;
          if (scrollTarget.y == "center") topPos = (block.top + block.bottom) / 2 - viewHeight / 2;else if (scrollTarget.y == "start" || scrollTarget.y == "nearest" && head < viewport.from) topPos = block.top;else topPos = block.bottom - viewHeight;
          viewport = new Viewport(map.lineAt(topPos - 1000 /* VP.Margin */ / 2, QueryType.ByHeight, oracle, 0, 0).from, map.lineAt(topPos + viewHeight + 1000 /* VP.Margin */ / 2, QueryType.ByHeight, oracle, 0, 0).to);
        }
      }
      return viewport;
    }
    mapViewport(viewport, changes) {
      let from = changes.mapPos(viewport.from, -1),
        to = changes.mapPos(viewport.to, 1);
      return new Viewport(this.heightMap.lineAt(from, QueryType.ByPos, this.heightOracle, 0, 0).from, this.heightMap.lineAt(to, QueryType.ByPos, this.heightOracle, 0, 0).to);
    }
    // Checks if a given viewport covers the visible part of the
    // document and not too much beyond that.
    viewportIsAppropriate({
      from,
      to
    }, bias = 0) {
      if (!this.inView) return true;
      let {
        top
      } = this.heightMap.lineAt(from, QueryType.ByPos, this.heightOracle, 0, 0);
      let {
        bottom
      } = this.heightMap.lineAt(to, QueryType.ByPos, this.heightOracle, 0, 0);
      let {
        visibleTop,
        visibleBottom
      } = this;
      return (from == 0 || top <= visibleTop - Math.max(10 /* VP.MinCoverMargin */, Math.min(-bias, 250 /* VP.MaxCoverMargin */))) && (to == this.state.doc.length || bottom >= visibleBottom + Math.max(10 /* VP.MinCoverMargin */, Math.min(bias, 250 /* VP.MaxCoverMargin */))) && top > visibleTop - 2 * 1000 /* VP.Margin */ && bottom < visibleBottom + 2 * 1000 /* VP.Margin */;
    }
    mapLineGaps(gaps, changes) {
      if (!gaps.length || changes.empty) return gaps;
      let mapped = [];
      for (let gap of gaps) if (!changes.touchesRange(gap.from, gap.to)) mapped.push(new LineGap(changes.mapPos(gap.from), changes.mapPos(gap.to), gap.size, gap.displaySize));
      return mapped;
    }
    // Computes positions in the viewport where the start or end of a
    // line should be hidden, trying to reuse existing line gaps when
    // appropriate to avoid unneccesary redraws.
    // Uses crude character-counting for the positioning and sizing,
    // since actual DOM coordinates aren't always available and
    // predictable. Relies on generous margins (see LG.Margin) to hide
    // the artifacts this might produce from the user.
    ensureLineGaps(current, mayMeasure) {
      let wrapping = this.heightOracle.lineWrapping;
      let margin = wrapping ? 10000 /* LG.MarginWrap */ : 2000 /* LG.Margin */,
        halfMargin = margin >> 1,
        doubleMargin = margin << 1;
      // The non-wrapping logic won't work at all in predominantly right-to-left text.
      if (this.defaultTextDirection != Direction.LTR && !wrapping) return [];
      let gaps = [];
      let addGap = (from, to, line, structure) => {
        if (to - from < halfMargin) return;
        let sel = this.state.selection.main,
          avoid = [sel.from];
        if (!sel.empty) avoid.push(sel.to);
        for (let pos of avoid) {
          if (pos > from && pos < to) {
            addGap(from, pos - 10 /* LG.SelectionMargin */, line, structure);
            addGap(pos + 10 /* LG.SelectionMargin */, to, line, structure);
            return;
          }
        }
        let gap = find(current, gap => gap.from >= line.from && gap.to <= line.to && Math.abs(gap.from - from) < halfMargin && Math.abs(gap.to - to) < halfMargin && !avoid.some(pos => gap.from < pos && gap.to > pos));
        if (!gap) {
          // When scrolling down, snap gap ends to line starts to avoid shifts in wrapping
          if (to < line.to && mayMeasure && wrapping && mayMeasure.visibleRanges.some(r => r.from <= to && r.to >= to)) {
            let lineStart = mayMeasure.moveToLineBoundary(EditorSelection.cursor(to), false, true).head;
            if (lineStart > from) to = lineStart;
          }
          let size = this.gapSize(line, from, to, structure);
          let displaySize = wrapping || size < 2000000 /* VP.MaxHorizGap */ ? size : 2000000 /* VP.MaxHorizGap */;
          gap = new LineGap(from, to, size, displaySize);
        }
        gaps.push(gap);
      };
      let checkLine = line => {
        if (line.length < doubleMargin || line.type != BlockType.Text) return;
        let structure = lineStructure(line.from, line.to, this.stateDeco);
        if (structure.total < doubleMargin) return;
        let target = this.scrollTarget ? this.scrollTarget.range.head : null;
        let viewFrom, viewTo;
        if (wrapping) {
          let marginHeight = margin / this.heightOracle.lineLength * this.heightOracle.lineHeight;
          let top, bot;
          if (target != null) {
            let targetFrac = findFraction(structure, target);
            let spaceFrac = ((this.visibleBottom - this.visibleTop) / 2 + marginHeight) / line.height;
            top = targetFrac - spaceFrac;
            bot = targetFrac + spaceFrac;
          } else {
            top = (this.visibleTop - line.top - marginHeight) / line.height;
            bot = (this.visibleBottom - line.top + marginHeight) / line.height;
          }
          viewFrom = findPosition(structure, top);
          viewTo = findPosition(structure, bot);
        } else {
          let totalWidth = structure.total * this.heightOracle.charWidth;
          let marginWidth = margin * this.heightOracle.charWidth;
          let horizOffset = 0;
          if (totalWidth > 2000000 /* VP.MaxHorizGap */) for (let old of current) {
            if (old.from >= line.from && old.from < line.to && old.size != old.displaySize && old.from * this.heightOracle.charWidth + horizOffset < this.pixelViewport.left) horizOffset = old.size - old.displaySize;
          }
          let pxLeft = this.pixelViewport.left + horizOffset,
            pxRight = this.pixelViewport.right + horizOffset;
          let left, right;
          if (target != null) {
            let targetFrac = findFraction(structure, target);
            let spaceFrac = ((pxRight - pxLeft) / 2 + marginWidth) / totalWidth;
            left = targetFrac - spaceFrac;
            right = targetFrac + spaceFrac;
          } else {
            left = (pxLeft - marginWidth) / totalWidth;
            right = (pxRight + marginWidth) / totalWidth;
          }
          viewFrom = findPosition(structure, left);
          viewTo = findPosition(structure, right);
        }
        if (viewFrom > line.from) addGap(line.from, viewFrom, line, structure);
        if (viewTo < line.to) addGap(viewTo, line.to, line, structure);
      };
      for (let line of this.viewportLines) {
        if (Array.isArray(line.type)) line.type.forEach(checkLine);else checkLine(line);
      }
      return gaps;
    }
    gapSize(line, from, to, structure) {
      let fraction = findFraction(structure, to) - findFraction(structure, from);
      if (this.heightOracle.lineWrapping) {
        return line.height * fraction;
      } else {
        return structure.total * this.heightOracle.charWidth * fraction;
      }
    }
    updateLineGaps(gaps) {
      if (!LineGap.same(gaps, this.lineGaps)) {
        this.lineGaps = gaps;
        this.lineGapDeco = Decoration.set(gaps.map(gap => gap.draw(this, this.heightOracle.lineWrapping)));
      }
    }
    computeVisibleRanges(changes) {
      let deco = this.stateDeco;
      if (this.lineGaps.length) deco = deco.concat(this.lineGapDeco);
      let ranges = [];
      RangeSet.spans(deco, this.viewport.from, this.viewport.to, {
        span(from, to) {
          ranges.push({
            from,
            to
          });
        },
        point() {}
      }, 20);
      let changed = 0;
      if (ranges.length != this.visibleRanges.length) {
        changed = 8 /* UpdateFlag.ViewportMoved */ | 4 /* UpdateFlag.Viewport */;
      } else {
        for (let i = 0; i < ranges.length && !(changed & 8 /* UpdateFlag.ViewportMoved */); i++) {
          let old = this.visibleRanges[i],
            nw = ranges[i];
          if (old.from != nw.from || old.to != nw.to) {
            changed |= 4 /* UpdateFlag.Viewport */;
            if (!(changes && changes.mapPos(old.from, -1) == nw.from && changes.mapPos(old.to, 1) == nw.to)) changed |= 8 /* UpdateFlag.ViewportMoved */;
          }
        }
      }
      this.visibleRanges = ranges;
      return changed;
    }
    lineBlockAt(pos) {
      return pos >= this.viewport.from && pos <= this.viewport.to && this.viewportLines.find(b => b.from <= pos && b.to >= pos) || scaleBlock(this.heightMap.lineAt(pos, QueryType.ByPos, this.heightOracle, 0, 0), this.scaler);
    }
    lineBlockAtHeight(height) {
      return height >= this.viewportLines[0].top && height <= this.viewportLines[this.viewportLines.length - 1].bottom && this.viewportLines.find(l => l.top <= height && l.bottom >= height) || scaleBlock(this.heightMap.lineAt(this.scaler.fromDOM(height), QueryType.ByHeight, this.heightOracle, 0, 0), this.scaler);
    }
    scrollAnchorAt(scrollTop) {
      let block = this.lineBlockAtHeight(scrollTop + 8);
      return block.from >= this.viewport.from || this.viewportLines[0].top - scrollTop > 200 ? block : this.viewportLines[0];
    }
    elementAtHeight(height) {
      return scaleBlock(this.heightMap.blockAt(this.scaler.fromDOM(height), this.heightOracle, 0, 0), this.scaler);
    }
    get docHeight() {
      return this.scaler.toDOM(this.heightMap.height);
    }
    get contentHeight() {
      return this.docHeight + this.paddingTop + this.paddingBottom;
    }
  }
  class Viewport {
    constructor(from, to) {
      this.from = from;
      this.to = to;
    }
  }
  function lineStructure(from, to, stateDeco) {
    let ranges = [],
      pos = from,
      total = 0;
    RangeSet.spans(stateDeco, from, to, {
      span() {},
      point(from, to) {
        if (from > pos) {
          ranges.push({
            from: pos,
            to: from
          });
          total += from - pos;
        }
        pos = to;
      }
    }, 20); // We're only interested in collapsed ranges of a significant size
    if (pos < to) {
      ranges.push({
        from: pos,
        to
      });
      total += to - pos;
    }
    return {
      total,
      ranges
    };
  }
  function findPosition({
    total,
    ranges
  }, ratio) {
    if (ratio <= 0) return ranges[0].from;
    if (ratio >= 1) return ranges[ranges.length - 1].to;
    let dist = Math.floor(total * ratio);
    for (let i = 0;; i++) {
      let {
          from,
          to
        } = ranges[i],
        size = to - from;
      if (dist <= size) return from + dist;
      dist -= size;
    }
  }
  function findFraction(structure, pos) {
    let counted = 0;
    for (let {
      from,
      to
    } of structure.ranges) {
      if (pos <= to) {
        counted += pos - from;
        break;
      }
      counted += to - from;
    }
    return counted / structure.total;
  }
  function find(array, f) {
    for (let val of array) if (f(val)) return val;
    return undefined;
  }
  // Don't scale when the document height is within the range of what
  // the DOM can handle.
  const IdScaler = {
    toDOM(n) {
      return n;
    },
    fromDOM(n) {
      return n;
    },
    scale: 1,
    eq(other) {
      return other == this;
    }
  };
  // When the height is too big (> VP.MaxDOMHeight), scale down the
  // regions outside the viewports so that the total height is
  // VP.MaxDOMHeight.
  class BigScaler {
    constructor(oracle, heightMap, viewports) {
      let vpHeight = 0,
        base = 0,
        domBase = 0;
      this.viewports = viewports.map(({
        from,
        to
      }) => {
        let top = heightMap.lineAt(from, QueryType.ByPos, oracle, 0, 0).top;
        let bottom = heightMap.lineAt(to, QueryType.ByPos, oracle, 0, 0).bottom;
        vpHeight += bottom - top;
        return {
          from,
          to,
          top,
          bottom,
          domTop: 0,
          domBottom: 0
        };
      });
      this.scale = (7000000 /* VP.MaxDOMHeight */ - vpHeight) / (heightMap.height - vpHeight);
      for (let obj of this.viewports) {
        obj.domTop = domBase + (obj.top - base) * this.scale;
        domBase = obj.domBottom = obj.domTop + (obj.bottom - obj.top);
        base = obj.bottom;
      }
    }
    toDOM(n) {
      for (let i = 0, base = 0, domBase = 0;; i++) {
        let vp = i < this.viewports.length ? this.viewports[i] : null;
        if (!vp || n < vp.top) return domBase + (n - base) * this.scale;
        if (n <= vp.bottom) return vp.domTop + (n - vp.top);
        base = vp.bottom;
        domBase = vp.domBottom;
      }
    }
    fromDOM(n) {
      for (let i = 0, base = 0, domBase = 0;; i++) {
        let vp = i < this.viewports.length ? this.viewports[i] : null;
        if (!vp || n < vp.domTop) return base + (n - domBase) / this.scale;
        if (n <= vp.domBottom) return vp.top + (n - vp.domTop);
        base = vp.bottom;
        domBase = vp.domBottom;
      }
    }
    eq(other) {
      if (!(other instanceof BigScaler)) return false;
      return this.scale == other.scale && this.viewports.length == other.viewports.length && this.viewports.every((vp, i) => vp.from == other.viewports[i].from && vp.to == other.viewports[i].to);
    }
  }
  function scaleBlock(block, scaler) {
    if (scaler.scale == 1) return block;
    let bTop = scaler.toDOM(block.top),
      bBottom = scaler.toDOM(block.bottom);
    return new BlockInfo(block.from, block.length, bTop, bBottom - bTop, Array.isArray(block._content) ? block._content.map(b => scaleBlock(b, scaler)) : block._content);
  }
  const theme = /*@__PURE__*/Facet.define({
    combine: strs => strs.join(" ")
  });
  const darkTheme = /*@__PURE__*/Facet.define({
    combine: values => values.indexOf(true) > -1
  });
  const baseThemeID = /*@__PURE__*/StyleModule.newName(),
    baseLightID = /*@__PURE__*/StyleModule.newName(),
    baseDarkID = /*@__PURE__*/StyleModule.newName();
  const lightDarkIDs = {
    "&light": "." + baseLightID,
    "&dark": "." + baseDarkID
  };
  function buildTheme(main, spec, scopes) {
    return new StyleModule(spec, {
      finish(sel) {
        return /&/.test(sel) ? sel.replace(/&\w*/, m => {
          if (m == "&") return main;
          if (!scopes || !scopes[m]) throw new RangeError(`Unsupported selector: ${m}`);
          return scopes[m];
        }) : main + " " + sel;
      }
    });
  }
  const baseTheme$1 = /*@__PURE__*/buildTheme("." + baseThemeID, {
    "&": {
      position: "relative !important",
      boxSizing: "border-box",
      "&.cm-focused": {
        // Provide a simple default outline to make sure a focused
        // editor is visuall