'use strict';

var minDash = require('min-dash');
var moddle = require('moddle');
var moddleXml = require('moddle-xml');

/**
 * A sub class of {@link Moddle} with support for import and export of DMN xml files.
 *
 * @class DmnModdle
 * @extends Moddle
 *
 * @param {Object|Array} packages to use for instantiating the model
 * @param {Object} [options] additional options to pass over
 */
function DmnModdle(packages, options) {
  moddle.Moddle.call(this, packages, options);
}

DmnModdle.prototype = Object.create(moddle.Moddle.prototype);


/**
 * Instantiates a DMN model tree from a given xml string.
 *
 * @param {String}   xmlStr
 * @param {String}   [typeName='dmn:Definitions'] name of the root element
 * @param {Object}   [options]  options to pass to the underlying reader
 * @param {Function} done       callback that is invoked with (err, result, parseContext)
 *                              once the import completes
 */
DmnModdle.prototype.fromXML = function(xmlStr, typeName, options, done) {

  if (!minDash.isString(typeName)) {
    done = options;
    options = typeName;
    typeName = 'dmn:Definitions';
  }

  if (minDash.isFunction(options)) {
    done = options;
    options = {};
  }

  var reader = new moddleXml.Reader(minDash.assign({ model: this, lax: true }, options));
  var rootHandler = reader.handler(typeName);

  reader.fromXML(xmlStr, rootHandler, done);
};


/**
 * Serializes a DMN object tree to XML.
 *
 * @param {String}   element    the root element, typically an instance of `Definitions`
 * @param {Object}   [options]  to pass to the underlying writer
 * @param {Function} done       callback invoked with (err, xmlStr) once the import completes
 */
DmnModdle.prototype.toXML = function(element, options, done) {

  if (minDash.isFunction(options)) {
    done = options;
    options = {};
  }

  var writer = new moddleXml.Writer(options);

  var result;
  var err;

  try {
    result = writer.toXML(element);
  } catch (e) {
    err = e;
  }

  return done(err, result);
};

var name = "DC";
var prefix = "dc";
var uri = "http://www.omg.org/spec/DMN/20180521/DC/";
var types = [
	{
		name: "Dimension",
		properties: [
			{
				name: "width",
				isAttr: true,
				type: "Real"
			},
			{
				name: "height",
				isAttr: true,
				type: "Real"
			}
		]
	},
	{
		name: "Bounds",
		properties: [
			{
				name: "height",
				isAttr: true,
				type: "Real"
			},
			{
				name: "width",
				isAttr: true,
				type: "Real"
			},
			{
				name: "x",
				isAttr: true,
				type: "Real"
			},
			{
				name: "y",
				isAttr: true,
				type: "Real"
			}
		]
	},
	{
		name: "Point",
		properties: [
			{
				name: "x",
				isAttr: true,
				type: "Real"
			},
			{
				name: "y",
				isAttr: true,
				type: "Real"
			}
		]
	},
	{
		name: "Color",
		properties: [
			{
				name: "red",
				type: "UML_Standard_Profile.mdzip:eee_1045467100323_917313_65"
			},
			{
				name: "green",
				type: "UML_Standard_Profile.mdzip:eee_1045467100323_917313_65"
			},
			{
				name: "blue",
				type: "UML_Standard_Profile.mdzip:eee_1045467100323_917313_65"
			}
		]
	}
];
var associations = [
];
var enumerations = [
	{
		name: "AlignmentKind",
		literalValues: [
			{
				name: "start"
			},
			{
				name: "center"
			},
			{
				name: "end"
			}
		]
	}
];
var DcPackage = {
	name: name,
	prefix: prefix,
	uri: uri,
	types: types,
	associations: associations,
	enumerations: enumerations
};

var name$1 = "DI";
var prefix$1 = "di";
var uri$1 = "http://www.omg.org/spec/DMN/20180521/DI/";
var types$1 = [
	{
		name: "DiagramElement",
		isAbstract: true,
		properties: [
			{
				name: "id",
				isAttr: true,
				isId: true,
				type: "String"
			},
			{
				name: "style",
				isReference: true,
				type: "Style",
				xml: {
					serialize: "property"
				}
			},
			{
				name: "sharedStyle",
				isReference: true,
				isVirtual: true,
				type: "Style"
			}
		]
	},
	{
		name: "Diagram",
		superClass: [
			"DiagramElement"
		],
		properties: [
			{
				name: "name",
				isAttr: true,
				type: "String"
			},
			{
				name: "documentation",
				isAttr: true,
				type: "String"
			},
			{
				name: "resolution",
				isAttr: true,
				type: "Real"
			}
		]
	},
	{
		name: "Shape",
		isAbstract: true,
		properties: [
			{
				name: "bounds",
				type: "dc:Bounds"
			}
		],
		superClass: [
			"DiagramElement"
		]
	},
	{
		name: "Edge",
		isAbstract: true,
		properties: [
			{
				name: "waypoint",
				type: "dc:Point",
				isMany: true,
				xml: {
					serialize: "property"
				}
			}
		],
		superClass: [
			"DiagramElement"
		]
	},
	{
		name: "Style",
		isAbstract: true,
		properties: [
			{
				name: "id",
				isAttr: true,
				isId: true,
				type: "String"
			}
		]
	}
];
var associations$1 = [
];
var enumerations$1 = [
];
var xml = {
	tagAlias: "lowerCase"
};
var DiPackage = {
	name: name$1,
	prefix: prefix$1,
	uri: uri$1,
	types: types$1,
	associations: associations$1,
	enumerations: enumerations$1,
	xml: xml
};

var name$2 = "DMN";
var prefix$2 = "dmn";
var uri$2 = "https://www.omg.org/spec/DMN/20191111/MODEL/";
var types$2 = [
	{
		name: "AuthorityRequirement",
		superClass: [
			"DMNElement"
		],
		properties: [
			{
				name: "requiredAuthority",
				type: "DMNElementReference",
				xml: {
					serialize: "property"
				}
			},
			{
				name: "requiredDecision",
				type: "DMNElementReference",
				xml: {
					serialize: "property"
				}
			},
			{
				name: "requiredInput",
				type: "DMNElementReference",
				xml: {
					serialize: "property"
				}
			}
		]
	},
	{
		name: "ItemDefinition",
		superClass: [
			"NamedElement"
		],
		properties: [
			{
				name: "typeRef",
				type: "String"
			},
			{
				name: "allowedValues",
				type: "UnaryTests",
				xml: {
					serialize: "property"
				}
			},
			{
				name: "typeLanguage",
				type: "String",
				isAttr: true
			},
			{
				name: "itemComponent",
				type: "ItemDefinition",
				isMany: true,
				xml: {
					serialize: "property"
				}
			},
			{
				name: "functionItem",
				type: "FunctionItem"
			},
			{
				name: "isCollection",
				isAttr: true,
				type: "Boolean"
			}
		]
	},
	{
		name: "Definitions",
		superClass: [
			"NamedElement"
		],
		properties: [
			{
				name: "import",
				type: "Import",
				isMany: true
			},
			{
				name: "itemDefinition",
				type: "ItemDefinition",
				isMany: true
			},
			{
				name: "drgElement",
				type: "DRGElement",
				isMany: true
			},
			{
				name: "artifact",
				type: "Artifact",
				isMany: true
			},
			{
				name: "elementCollection",
				type: "ElementCollection",
				isMany: true
			},
			{
				name: "businessContextElement",
				type: "BusinessContextElement",
				isMany: true
			},
			{
				name: "namespace",
				type: "String",
				isAttr: true
			},
			{
				name: "expressionLanguage",
				type: "String",
				isAttr: true
			},
			{
				name: "typeLanguage",
				type: "String",
				isAttr: true
			},
			{
				name: "exporter",
				isAttr: true,
				type: "String"
			},
			{
				name: "exporterVersion",
				isAttr: true,
				type: "String"
			},
			{
				name: "dmnDI",
				type: "dmndi:DMNDI"
			}
		]
	},
	{
		name: "KnowledgeSource",
		superClass: [
			"DRGElement"
		],
		properties: [
			{
				name: "authorityRequirement",
				type: "AuthorityRequirement",
				isMany: true
			},
			{
				name: "type",
				type: "String"
			},
			{
				name: "owner",
				type: "DMNElementReference",
				xml: {
					serialize: "property"
				}
			},
			{
				name: "locationURI",
				type: "String",
				isAttr: true
			}
		]
	},
	{
		name: "DecisionRule",
		superClass: [
			"DMNElement"
		],
		properties: [
			{
				name: "inputEntry",
				type: "UnaryTests",
				isMany: true,
				xml: {
					serialize: "property"
				}
			},
			{
				name: "outputEntry",
				type: "LiteralExpression",
				isMany: true,
				xml: {
					serialize: "property"
				}
			},
			{
				name: "annotationEntry",
				type: "RuleAnnotation",
				isMany: true,
				xml: {
					serialize: "property"
				}
			}
		]
	},
	{
		name: "Expression",
		isAbstract: true,
		superClass: [
			"DMNElement"
		],
		properties: [
			{
				name: "typeRef",
				isAttr: true,
				type: "String"
			}
		]
	},
	{
		name: "InformationItem",
		superClass: [
			"NamedElement"
		],
		properties: [
			{
				name: "typeRef",
				isAttr: true,
				type: "String"
			}
		]
	},
	{
		name: "Decision",
		superClass: [
			"DRGElement"
		],
		properties: [
			{
				name: "question",
				type: "String",
				xml: {
					serialize: "property"
				}
			},
			{
				name: "allowedAnswers",
				type: "String",
				xml: {
					serialize: "property"
				}
			},
			{
				name: "variable",
				type: "InformationItem",
				xml: {
					serialize: "property"
				}
			},
			{
				name: "informationRequirement",
				type: "InformationRequirement",
				isMany: true
			},
			{
				name: "knowledgeRequirement",
				type: "KnowledgeRequirement",
				isMany: true
			},
			{
				name: "authorityRequirement",
				type: "AuthorityRequirement",
				isMany: true
			},
			{
				name: "supportedObjective",
				isMany: true,
				type: "DMNElementReference",
				xml: {
					serialize: "property"
				}
			},
			{
				name: "impactedPerformanceIndicator",
				type: "DMNElementReference",
				isMany: true,
				xml: {
					serialize: "property"
				}
			},
			{
				name: "decisionMaker",
				type: "DMNElementReference",
				isMany: true,
				xml: {
					serialize: "property"
				}
			},
			{
				name: "decisionOwner",
				type: "DMNElementReference",
				isMany: true,
				xml: {
					serialize: "property"
				}
			},
			{
				name: "usingProcess",
				isMany: true,
				type: "DMNElementReference",
				xml: {
					serialize: "property"
				}
			},
			{
				name: "usingTask",
				isMany: true,
				type: "DMNElementReference",
				xml: {
					serialize: "property"
				}
			},
			{
				name: "decisionLogic",
				type: "Expression"
			}
		]
	},
	{
		name: "Invocation",
		superClass: [
			"Expression"
		],
		properties: [
			{
				name: "calledFunction",
				type: "Expression"
			},
			{
				name: "binding",
				type: "Binding",
				isMany: true
			}
		]
	},
	{
		name: "OrganisationalUnit",
		superClass: [
			"BusinessContextElement"
		],
		properties: [
			{
				name: "decisionMade",
				type: "Decision",
				isReference: true,
				isMany: true
			},
			{
				name: "decisionOwned",
				type: "Decision",
				isReference: true,
				isMany: true
			}
		]
	},
	{
		name: "Import",
		superClass: [
			"NamedElement"
		],
		properties: [
			{
				name: "importType",
				type: "String",
				isAttr: true
			},
			{
				name: "locationURI",
				type: "String",
				isAttr: true
			},
			{
				name: "namespace",
				type: "String",
				isAttr: true
			}
		]
	},
	{
		name: "InformationRequirement",
		superClass: [
			"DMNElement"
		],
		properties: [
			{
				name: "requiredDecision",
				type: "DMNElementReference",
				xml: {
					serialize: "property"
				}
			},
			{
				name: "requiredInput",
				type: "DMNElementReference",
				xml: {
					serialize: "property"
				}
			}
		]
	},
	{
		name: "ElementCollection",
		superClass: [
			"NamedElement"
		],
		properties: [
			{
				name: "drgElement",
				type: "DMNElementReference",
				isMany: true,
				xml: {
					serialize: "property"
				}
			}
		]
	},
	{
		name: "DRGElement",
		isAbstract: true,
		superClass: [
			"NamedElement"
		],
		properties: [
		]
	},
	{
		name: "InputData",
		superClass: [
			"DRGElement"
		],
		properties: [
			{
				name: "variable",
				type: "InformationItem",
				xml: {
					serialize: "property"
				}
			}
		]
	},
	{
		name: "DMNElement",
		isAbstract: true,
		properties: [
			{
				name: "description",
				type: "String"
			},
			{
				name: "extensionElements",
				type: "ExtensionElements"
			},
			{
				name: "id",
				type: "String",
				isAttr: true,
				isId: true
			},
			{
				name: "extensionAttribute",
				type: "ExtensionAttribute",
				isMany: true
			},
			{
				name: "label",
				isAttr: true,
				type: "String"
			}
		]
	},
	{
		name: "InputClause",
		superClass: [
			"DMNElement"
		],
		properties: [
			{
				name: "inputExpression",
				type: "LiteralExpression",
				xml: {
					serialize: "property"
				}
			},
			{
				name: "inputValues",
				type: "UnaryTests",
				xml: {
					serialize: "property"
				}
			}
		]
	},
	{
		name: "DecisionTable",
		superClass: [
			"Expression"
		],
		properties: [
			{
				name: "input",
				type: "InputClause",
				isMany: true,
				xml: {
					serialize: "property"
				}
			},
			{
				name: "output",
				type: "OutputClause",
				isMany: true,
				xml: {
					serialize: "property"
				}
			},
			{
				name: "annotation",
				type: "RuleAnnotationClause",
				isMany: true,
				xml: {
					serialize: "property"
				}
			},
			{
				name: "rule",
				type: "DecisionRule",
				isMany: true,
				xml: {
					serialize: "property"
				}
			},
			{
				name: "hitPolicy",
				type: "HitPolicy",
				isAttr: true,
				"default": "UNIQUE"
			},
			{
				name: "aggregation",
				type: "BuiltinAggregator",
				isAttr: true
			},
			{
				name: "preferredOrientation",
				type: "DecisionTableOrientation",
				isAttr: true
			},
			{
				name: "outputLabel",
				isAttr: true,
				type: "String"
			}
		]
	},
	{
		name: "LiteralExpression",
		superClass: [
			"Expression"
		],
		properties: [
			{
				name: "expressionLanguage",
				type: "String",
				isAttr: true
			},
			{
				name: "text",
				type: "String"
			},
			{
				name: "importedValues",
				type: "ImportedValues"
			}
		]
	},
	{
		name: "Binding",
		properties: [
			{
				name: "parameter",
				type: "InformationItem",
				xml: {
					serialize: "property"
				}
			},
			{
				name: "bindingFormula",
				type: "Expression"
			}
		]
	},
	{
		name: "KnowledgeRequirement",
		superClass: [
			"DMNElement"
		],
		properties: [
			{
				name: "requiredKnowledge",
				type: "DMNElementReference",
				xml: {
					serialize: "property"
				}
			}
		]
	},
	{
		name: "BusinessKnowledgeModel",
		superClass: [
			"Invocable"
		],
		properties: [
			{
				name: "encapsulatedLogic",
				type: "FunctionDefinition",
				xml: {
					serialize: "property"
				}
			},
			{
				name: "knowledgeRequirement",
				type: "KnowledgeRequirement",
				isMany: true
			},
			{
				name: "authorityRequirement",
				type: "AuthorityRequirement",
				isMany: true
			}
		]
	},
	{
		name: "BusinessContextElement",
		isAbstract: true,
		superClass: [
			"NamedElement"
		],
		properties: [
			{
				name: "URI",
				type: "String",
				isAttr: true
			}
		]
	},
	{
		name: "PerformanceIndicator",
		superClass: [
			"BusinessContextElement"
		],
		properties: [
			{
				name: "impactingDecision",
				type: "DMNElementReference",
				isMany: true,
				xml: {
					serialize: "property"
				}
			}
		]
	},
	{
		name: "FunctionDefinition",
		superClass: [
			"Expression"
		],
		properties: [
			{
				name: "formalParameter",
				type: "InformationItem",
				isMany: true,
				xml: {
					serialize: "property"
				}
			},
			{
				name: "body",
				type: "Expression"
			},
			{
				name: "kind",
				type: "FunctionKind",
				isAttr: true
			}
		]
	},
	{
		name: "Context",
		superClass: [
			"Expression"
		],
		properties: [
			{
				name: "contextEntry",
				type: "ContextEntry",
				isMany: true
			}
		]
	},
	{
		name: "ContextEntry",
		superClass: [
			"DMNElement"
		],
		properties: [
			{
				name: "variable",
				type: "InformationItem",
				xml: {
					serialize: "property"
				}
			},
			{
				name: "value",
				type: "Expression"
			}
		]
	},
	{
		name: "List",
		superClass: [
			"Expression"
		],
		properties: [
			{
				name: "elements",
				isMany: true,
				type: "Expression"
			}
		]
	},
	{
		name: "Relation",
		superClass: [
			"Expression"
		],
		properties: [
			{
				name: "column",
				type: "InformationItem",
				isMany: true,
				xml: {
					serialize: "property"
				}
			},
			{
				name: "row",
				type: "List",
				isMany: true,
				xml: {
					serialize: "property"
				}
			}
		]
	},
	{
		name: "OutputClause",
		superClass: [
			"DMNElement"
		],
		properties: [
			{
				name: "outputValues",
				type: "UnaryTests",
				xml: {
					serialize: "property"
				}
			},
			{
				name: "defaultOutputEntry",
				type: "LiteralExpression",
				xml: {
					serialize: "property"
				}
			},
			{
				name: "name",
				isAttr: true,
				type: "String"
			},
			{
				name: "typeRef",
				isAttr: true,
				type: "String"
			}
		]
	},
	{
		name: "UnaryTests",
		superClass: [
			"Expression"
		],
		properties: [
			{
				name: "text",
				type: "String"
			},
			{
				name: "expressionLanguage",
				type: "String",
				isAttr: true
			}
		]
	},
	{
		name: "NamedElement",
		isAbstract: true,
		superClass: [
			"DMNElement"
		],
		properties: [
			{
				name: "name",
				isAttr: true,
				type: "String"
			}
		]
	},
	{
		name: "ImportedValues",
		superClass: [
			"Import"
		],
		properties: [
			{
				name: "importedElement",
				type: "String"
			},
			{
				name: "expressionLanguage",
				type: "String",
				isAttr: true
			}
		]
	},
	{
		name: "DecisionService",
		superClass: [
			"Invocable"
		],
		properties: [
			{
				name: "outputDecision",
				type: "DMNElementReference",
				isMany: true,
				xml: {
					serialize: "property"
				}
			},
			{
				name: "encapsulatedDecision",
				type: "DMNElementReference",
				isMany: true,
				xml: {
					serialize: "property"
				}
			},
			{
				name: "inputDecision",
				type: "DMNElementReference",
				isMany: true,
				xml: {
					serialize: "property"
				}
			},
			{
				name: "inputData",
				type: "DMNElementReference",
				isMany: true,
				xml: {
					serialize: "property"
				}
			}
		]
	},
	{
		name: "ExtensionElements",
		properties: [
			{
				name: "values",
				type: "Element",
				isMany: true
			}
		]
	},
	{
		name: "ExtensionAttribute",
		properties: [
			{
				name: "value",
				type: "Element"
			},
			{
				name: "valueRef",
				type: "Element",
				isAttr: true,
				isReference: true
			},
			{
				name: "name",
				isAttr: true,
				type: "String"
			}
		]
	},
	{
		name: "Element",
		isAbstract: true,
		properties: [
			{
				name: "extensionAttribute",
				type: "ExtensionAttribute",
				isAttr: true,
				isReference: true
			},
			{
				name: "elements",
				type: "ExtensionElements",
				isAttr: true,
				isReference: true
			}
		]
	},
	{
		name: "Artifact",
		isAbstract: true,
		superClass: [
			"DMNElement"
		],
		properties: [
		]
	},
	{
		name: "Association",
		superClass: [
			"Artifact"
		],
		properties: [
			{
				name: "sourceRef",
				type: "DMNElementReference",
				xml: {
					serialize: "property"
				}
			},
			{
				name: "targetRef",
				type: "DMNElementReference",
				xml: {
					serialize: "property"
				}
			},
			{
				name: "associationDirection",
				type: "AssociationDirection",
				isAttr: true
			}
		]
	},
	{
		name: "TextAnnotation",
		superClass: [
			"Artifact"
		],
		properties: [
			{
				name: "text",
				type: "String"
			},
			{
				name: "textFormat",
				isAttr: true,
				type: "String",
				"default": "text/plain"
			}
		]
	},
	{
		name: "RuleAnnotationClause",
		properties: [
			{
				name: "name",
				isAttr: true,
				type: "String"
			}
		]
	},
	{
		name: "RuleAnnotation",
		properties: [
			{
				name: "text",
				type: "String"
			}
		]
	},
	{
		name: "Invocable",
		isAbstract: true,
		superClass: [
			"DRGElement"
		],
		properties: [
			{
				name: "variable",
				type: "InformationItem",
				xml: {
					serialize: "property"
				}
			}
		]
	},
	{
		name: "Group",
		superClass: [
			"Artifact"
		],
		properties: [
			{
				name: "name",
				isAttr: true,
				type: "String"
			}
		]
	},
	{
		name: "FunctionItem",
		superClass: [
			"DMNElement"
		],
		properties: [
			{
				name: "parameters",
				isMany: true,
				type: "InformationItem",
				xml: {
					serialize: "property"
				}
			},
			{
				name: "outputTypeRef",
				isAttr: true,
				type: "String"
			}
		]
	},
	{
		name: "DMNElementReference",
		properties: [
			{
				isAttr: true,
				name: "href",
				type: "String"
			}
		]
	}
];
var enumerations$2 = [
	{
		name: "HitPolicy",
		literalValues: [
			{
				name: "UNIQUE"
			},
			{
				name: "FIRST"
			},
			{
				name: "PRIORITY"
			},
			{
				name: "ANY"
			},
			{
				name: "COLLECT"
			},
			{
				name: "RULE ORDER"
			},
			{
				name: "OUTPUT ORDER"
			}
		]
	},
	{
		name: "BuiltinAggregator",
		literalValues: [
			{
				name: "SUM"
			},
			{
				name: "COUNT"
			},
			{
				name: "MIN"
			},
			{
				name: "MAX"
			}
		]
	},
	{
		name: "DecisionTableOrientation",
		literalValues: [
			{
				name: "Rule-as-Row"
			},
			{
				name: "Rule-as-Column"
			},
			{
				name: "CrossTable"
			}
		]
	},
	{
		name: "AssociationDirection",
		literalValues: [
			{
				name: "None"
			},
			{
				name: "One"
			},
			{
				name: "Both"
			}
		]
	},
	{
		name: "FunctionKind",
		literalValues: [
			{
				name: "FEEL"
			},
			{
				name: "Java"
			},
			{
				name: "PMML"
			}
		]
	}
];
var associations$2 = [
];
var xml$1 = {
	tagAlias: "lowerCase"
};
var DmnPackage = {
	name: name$2,
	prefix: prefix$2,
	uri: uri$2,
	types: types$2,
	enumerations: enumerations$2,
	associations: associations$2,
	xml: xml$1
};

var name$3 = "DMNDI";
var prefix$3 = "dmndi";
var uri$3 = "https://www.omg.org/spec/DMN/20191111/DMNDI/";
var types$3 = [
	{
		name: "DMNDI",
		properties: [
			{
				name: "diagrams",
				type: "DMNDiagram",
				isMany: true
			},
			{
				name: "styles",
				type: "DMNStyle",
				isMany: true
			}
		]
	},
	{
		name: "DMNStyle",
		superClass: [
			"di:Style"
		],
		properties: [
			{
				name: "fillColor",
				type: "dc:Color",
				isAttr: true
			},
			{
				name: "strokeColor",
				type: "dc:Color",
				isAttr: true
			},
			{
				name: "fontColor",
				type: "dc:Color",
				isAttr: true
			},
			{
				name: "fontSize",
				isAttr: true,
				type: "Real"
			},
			{
				name: "fontFamily",
				isAttr: true,
				type: "String"
			},
			{
				name: "fontItalic",
				isAttr: true,
				type: "Boolean"
			},
			{
				name: "fontBold",
				isAttr: true,
				type: "Boolean"
			},
			{
				name: "fontUnderline",
				isAttr: true,
				type: "Boolean"
			},
			{
				name: "fontStrikeThrough",
				isAttr: true,
				type: "Boolean"
			},
			{
				name: "labelHorizontalAlignment",
				type: "dc:AlignmentKind",
				isAttr: true
			},
			{
				name: "labelVerticalAlignment",
				type: "dc:AlignmentKind",
				isAttr: true
			}
		]
	},
	{
		name: "DMNDiagram",
		superClass: [
			"di:Diagram"
		],
		properties: [
			{
				name: "dmnElementRef",
				type: "dmn:DMNElement",
				isAttr: true,
				isReference: true
			},
			{
				name: "size",
				type: "Size"
			},
			{
				name: "localStyle",
				type: "DMNStyle",
				isVirtual: true
			},
			{
				name: "sharedStyle",
				type: "DMNStyle",
				isVirtual: true,
				isReference: true,
				redefines: "di:DiagramElement#sharedStyle"
			},
			{
				name: "diagramElements",
				type: "DMNDiagramElement",
				isMany: true
			}
		]
	},
	{
		name: "DMNDiagramElement",
		isAbstract: true,
		superClass: [
			"di:DiagramElement"
		],
		properties: [
			{
				name: "dmnElementRef",
				type: "dmn:DMNElement",
				isAttr: true,
				isReference: true
			},
			{
				name: "sharedStyle",
				type: "DMNStyle",
				isVirtual: true,
				isReference: true,
				redefines: "di:DiagramElement#sharedStyle"
			},
			{
				name: "localStyle",
				type: "DMNStyle",
				isVirtual: true
			},
			{
				name: "label",
				type: "DMNLabel"
			}
		]
	},
	{
		name: "DMNLabel",
		superClass: [
			"di:Shape"
		],
		properties: [
			{
				name: "text",
				type: "Text"
			}
		]
	},
	{
		name: "DMNShape",
		superClass: [
			"di:Shape",
			"DMNDiagramElement"
		],
		properties: [
			{
				name: "isListedInputData",
				isAttr: true,
				type: "Boolean"
			},
			{
				name: "decisionServiceDividerLine",
				type: "DMNDecisionServiceDividerLine"
			},
			{
				name: "isCollapsed",
				isAttr: true,
				type: "Boolean"
			}
		]
	},
	{
		name: "DMNEdge",
		superClass: [
			"di:Edge",
			"DMNDiagramElement"
		],
		properties: [
			{
				name: "sourceElement",
				type: "DMNDiagramElement",
				isAttr: true,
				isReference: true
			},
			{
				name: "targetElement",
				type: "DMNDiagramElement",
				isAttr: true,
				isReference: true
			}
		]
	},
	{
		name: "DMNDecisionServiceDividerLine",
		superClass: [
			"di:Edge"
		]
	},
	{
		name: "Text",
		properties: [
			{
				name: "text",
				isBody: true,
				type: "String"
			}
		]
	},
	{
		name: "Size",
		superClass: [
			"dc:Dimension"
		]
	}
];
var associations$3 = [
];
var enumerations$3 = [
];
var DmnDiPackage = {
	name: name$3,
	prefix: prefix$3,
	uri: uri$3,
	types: types$3,
	associations: associations$3,
	enumerations: enumerations$3
};

var name$4 = "bpmn.io DI for DMN";
var uri$4 = "http://bpmn.io/schema/dmn/biodi/2.0";
var prefix$4 = "biodi";
var xml$2 = {
	tagAlias: "lowerCase"
};
var types$4 = [
	{
		name: "DecisionTable",
		isAbstract: true,
		"extends": [
			"dmn:DecisionTable"
		],
		properties: [
			{
				name: "annotationsWidth",
				isAttr: true,
				type: "Integer"
			}
		]
	},
	{
		name: "OutputClause",
		isAbstract: true,
		"extends": [
			"dmn:OutputClause"
		],
		properties: [
			{
				name: "width",
				isAttr: true,
				type: "Integer"
			}
		]
	},
	{
		name: "InputClause",
		isAbstract: true,
		"extends": [
			"dmn:InputClause"
		],
		properties: [
			{
				name: "width",
				isAttr: true,
				type: "Integer"
			}
		]
	}
];
var BioDiPackage = {
	name: name$4,
	uri: uri$4,
	prefix: prefix$4,
	xml: xml$2,
	types: types$4
};

var packages = {
  dc: DcPackage,
  di: DiPackage,
  dmn: DmnPackage,
  dmndi: DmnDiPackage,
  biodi: BioDiPackage
};

function simple(additionalPackages, options) {
  var pks = minDash.assign({}, packages, additionalPackages);

  return new DmnModdle(pks, options);
}

module.exports = simple;
