'use strict';

var _interopRequireWildcard = function (obj) { return obj && obj.__esModule ? obj : { 'default': obj }; };

var _classCallCheck = function (instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError('Cannot call a class as a function'); } };

exports.__esModule = true;

var _invariant = require('invariant');

var _invariant2 = _interopRequireWildcard(_invariant);

var _keyMirror = require('keymirror');

var _keyMirror2 = _interopRequireWildcard(_keyMirror);

var _isArray = require('lodash/lang/isArray');

var _isArray2 = _interopRequireWildcard(_isArray);

var _getNextUniqueId = require('./getNextUniqueId');

var _getNextUniqueId2 = _interopRequireWildcard(_getNextUniqueId);

var HandlerRoles = _keyMirror2['default']({
  SOURCE: null,
  TARGET: null
});

function validateSourceContract(source) {
  _invariant2['default'](typeof source.canDrag === 'function', 'Expected canDrag to be a function.');
  _invariant2['default'](typeof source.beginDrag === 'function', 'Expected beginDrag to be a function.');
  _invariant2['default'](typeof source.endDrag === 'function', 'Expected endDrag to be a function.');
}

function validateTargetContract(target) {
  _invariant2['default'](typeof target.canDrop === 'function', 'Expected canDrop to be a function.');
  _invariant2['default'](typeof target.hover === 'function', 'Expected hover to be a function.');
  _invariant2['default'](typeof target.drop === 'function', 'Expected beginDrag to be a function.');
}

function validateType(type, allowArray) {
  if (allowArray && _isArray2['default'](type)) {
    type.forEach(function (t) {
      return validateType(t, false);
    });
    return;
  }

  _invariant2['default'](typeof type === 'string' || typeof type === 'symbol', allowArray ? 'Type can only be a string, a symbol, or an array of either.' : 'Type can only be a string or a symbol.');
}

function getNextHandlerId(role) {
  var id = _getNextUniqueId2['default']().toString();
  switch (role) {
    case HandlerRoles.SOURCE:
      return 'S' + id;
    case HandlerRoles.TARGET:
      return 'T' + id;
    default:
      _invariant2['default'](false, 'Unknown role: ' + role);
  }
}

function parseRoleFromHandlerId(handlerId) {
  switch (handlerId[0]) {
    case 'S':
      return HandlerRoles.SOURCE;
    case 'T':
      return HandlerRoles.TARGET;
    default:
      _invariant2['default'](false, 'Cannot parse handler ID: ' + handlerId);
  }
}

var HandlerRegistry = (function () {
  function HandlerRegistry(actions) {
    _classCallCheck(this, HandlerRegistry);

    this.actions = actions;

    this.types = {};
    this.handlers = {};

    this.pinnedSourceId = null;
    this.pinnedSource = null;
  }

  HandlerRegistry.prototype.addSource = function addSource(type, source) {
    validateType(type);
    validateSourceContract(source);

    var sourceId = this.addHandler(HandlerRoles.SOURCE, type, source);
    this.actions.addSource(sourceId);
    return sourceId;
  };

  HandlerRegistry.prototype.addTarget = function addTarget(type, target) {
    validateType(type, true);
    validateTargetContract(target);

    var targetId = this.addHandler(HandlerRoles.TARGET, type, target);
    this.actions.addTarget(targetId);
    return targetId;
  };

  HandlerRegistry.prototype.addHandler = function addHandler(role, type, handler) {
    var id = getNextHandlerId(role);
    this.types[id] = type;
    this.handlers[id] = handler;

    return id;
  };

  HandlerRegistry.prototype.containsHandler = function containsHandler(handler) {
    var _this = this;

    return Object.keys(this.handlers).some(function (key) {
      return _this.handlers[key] === handler;
    });
  };

  HandlerRegistry.prototype.getSource = function getSource(sourceId, includePinned) {
    _invariant2['default'](this.isSourceId(sourceId), 'Expected a valid source ID.');

    var isPinned = includePinned && sourceId === this.pinnedSourceId;
    var source = isPinned ? this.pinnedSource : this.handlers[sourceId];

    return source;
  };

  HandlerRegistry.prototype.getTarget = function getTarget(targetId) {
    _invariant2['default'](this.isTargetId(targetId), 'Expected a valid target ID.');
    return this.handlers[targetId];
  };

  HandlerRegistry.prototype.getSourceType = function getSourceType(sourceId) {
    _invariant2['default'](this.isSourceId(sourceId), 'Expected a valid source ID.');
    return this.types[sourceId];
  };

  HandlerRegistry.prototype.getTargetType = function getTargetType(targetId) {
    _invariant2['default'](this.isTargetId(targetId), 'Expected a valid target ID.');
    return this.types[targetId];
  };

  HandlerRegistry.prototype.isSourceId = function isSourceId(handlerId) {
    var role = parseRoleFromHandlerId(handlerId);
    return role === HandlerRoles.SOURCE;
  };

  HandlerRegistry.prototype.isTargetId = function isTargetId(handlerId) {
    var role = parseRoleFromHandlerId(handlerId);
    return role === HandlerRoles.TARGET;
  };

  HandlerRegistry.prototype.removeSource = function removeSource(sourceId) {
    _invariant2['default'](this.getSource(sourceId), 'Expected an existing source.');

    delete this.handlers[sourceId];
    delete this.types[sourceId];

    this.actions.removeSource(sourceId);
  };

  HandlerRegistry.prototype.removeTarget = function removeTarget(targetId) {
    _invariant2['default'](this.getTarget(targetId), 'Expected an existing target.');

    delete this.handlers[targetId];
    delete this.types[targetId];

    this.actions.removeTarget(targetId);
  };

  HandlerRegistry.prototype.pinSource = function pinSource(sourceId) {
    var source = this.getSource(sourceId);
    _invariant2['default'](source, 'Expected an existing source.');

    this.pinnedSourceId = sourceId;
    this.pinnedSource = source;
  };

  HandlerRegistry.prototype.unpinSource = function unpinSource() {
    _invariant2['default'](this.pinnedSource, 'No source is pinned at the time.');

    this.pinnedSourceId = null;
    this.pinnedSource = null;
  };

  return HandlerRegistry;
})();

exports['default'] = HandlerRegistry;
module.exports = exports['default'];