'use strict';

exports.__esModule = true;

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { 'default': obj }; }

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError('Cannot call a class as a function'); } }

function _typeof(obj) { return obj && obj.constructor === Symbol ? 'symbol' : typeof obj; }

var _invariant = require('invariant');

var _invariant2 = _interopRequireDefault(_invariant);

var _lodashLangIsArray = require('lodash/lang/isArray');

var _lodashLangIsArray2 = _interopRequireDefault(_lodashLangIsArray);

var _utilsGetNextUniqueId = require('./utils/getNextUniqueId');

var _utilsGetNextUniqueId2 = _interopRequireDefault(_utilsGetNextUniqueId);

var _actionsRegistry = require('./actions/registry');

var HandlerRoles = {
  SOURCE: 'SOURCE',
  TARGET: 'TARGET'
};

function validateSourceContract(source) {
  _invariant2['default'](typeof source.canDrag === 'function', 'Expected canDrag to be a function.');
  _invariant2['default'](typeof source.beginDrag === 'function', 'Expected beginDrag to be a function.');
  _invariant2['default'](typeof source.endDrag === 'function', 'Expected endDrag to be a function.');
}

function validateTargetContract(target) {
  _invariant2['default'](typeof target.canDrop === 'function', 'Expected canDrop to be a function.');
  _invariant2['default'](typeof target.hover === 'function', 'Expected hover to be a function.');
  _invariant2['default'](typeof target.drop === 'function', 'Expected beginDrag to be a function.');
}

function validateType(type, allowArray) {
  if (allowArray && _lodashLangIsArray2['default'](type)) {
    type.forEach(function (t) {
      return validateType(t, false);
    });
    return;
  }

  _invariant2['default'](typeof type === 'string' || (typeof type === 'undefined' ? 'undefined' : _typeof(type)) === 'symbol', allowArray ? 'Type can only be a string, a symbol, or an array of either.' : 'Type can only be a string or a symbol.');
}

function getNextHandlerId(role) {
  var id = _utilsGetNextUniqueId2['default']().toString();
  switch (role) {
    case HandlerRoles.SOURCE:
      return 'S' + id;
    case HandlerRoles.TARGET:
      return 'T' + id;
    default:
      _invariant2['default'](false, 'Unknown role: ' + role);
  }
}

function parseRoleFromHandlerId(handlerId) {
  switch (handlerId[0]) {
    case 'S':
      return HandlerRoles.SOURCE;
    case 'T':
      return HandlerRoles.TARGET;
    default:
      _invariant2['default'](false, 'Cannot parse handler ID: ' + handlerId);
  }
}

var HandlerRegistry = (function () {
  function HandlerRegistry(store) {
    _classCallCheck(this, HandlerRegistry);

    this.store = store;

    this.types = {};
    this.handlers = {};

    this.pinnedSourceId = null;
    this.pinnedSource = null;
  }

  HandlerRegistry.prototype.addSource = function addSource(type, source) {
    validateType(type);
    validateSourceContract(source);

    var sourceId = this.addHandler(HandlerRoles.SOURCE, type, source);
    this.store.dispatch(_actionsRegistry.addSource(sourceId));
    return sourceId;
  };

  HandlerRegistry.prototype.addTarget = function addTarget(type, target) {
    validateType(type, true);
    validateTargetContract(target);

    var targetId = this.addHandler(HandlerRoles.TARGET, type, target);
    this.store.dispatch(_actionsRegistry.addTarget(targetId));
    return targetId;
  };

  HandlerRegistry.prototype.addHandler = function addHandler(role, type, handler) {
    var id = getNextHandlerId(role);
    this.types[id] = type;
    this.handlers[id] = handler;

    return id;
  };

  HandlerRegistry.prototype.containsHandler = function containsHandler(handler) {
    var _this = this;

    return Object.keys(this.handlers).some(function (key) {
      return _this.handlers[key] === handler;
    });
  };

  HandlerRegistry.prototype.getSource = function getSource(sourceId, includePinned) {
    _invariant2['default'](this.isSourceId(sourceId), 'Expected a valid source ID.');

    var isPinned = includePinned && sourceId === this.pinnedSourceId;
    var source = isPinned ? this.pinnedSource : this.handlers[sourceId];

    return source;
  };

  HandlerRegistry.prototype.getTarget = function getTarget(targetId) {
    _invariant2['default'](this.isTargetId(targetId), 'Expected a valid target ID.');
    return this.handlers[targetId];
  };

  HandlerRegistry.prototype.getSourceType = function getSourceType(sourceId) {
    _invariant2['default'](this.isSourceId(sourceId), 'Expected a valid source ID.');
    return this.types[sourceId];
  };

  HandlerRegistry.prototype.getTargetType = function getTargetType(targetId) {
    _invariant2['default'](this.isTargetId(targetId), 'Expected a valid target ID.');
    return this.types[targetId];
  };

  HandlerRegistry.prototype.isSourceId = function isSourceId(handlerId) {
    var role = parseRoleFromHandlerId(handlerId);
    return role === HandlerRoles.SOURCE;
  };

  HandlerRegistry.prototype.isTargetId = function isTargetId(handlerId) {
    var role = parseRoleFromHandlerId(handlerId);
    return role === HandlerRoles.TARGET;
  };

  HandlerRegistry.prototype.removeSource = function removeSource(sourceId) {
    _invariant2['default'](this.getSource(sourceId), 'Expected an existing source.');
    this.store.dispatch(_actionsRegistry.removeSource(sourceId));
    delete this.handlers[sourceId];
    delete this.types[sourceId];
  };

  HandlerRegistry.prototype.removeTarget = function removeTarget(targetId) {
    _invariant2['default'](this.getTarget(targetId), 'Expected an existing target.');
    this.store.dispatch(_actionsRegistry.removeTarget(targetId));
    delete this.handlers[targetId];
    delete this.types[targetId];
  };

  HandlerRegistry.prototype.pinSource = function pinSource(sourceId) {
    var source = this.getSource(sourceId);
    _invariant2['default'](source, 'Expected an existing source.');

    this.pinnedSourceId = sourceId;
    this.pinnedSource = source;
  };

  HandlerRegistry.prototype.unpinSource = function unpinSource() {
    _invariant2['default'](this.pinnedSource, 'No source is pinned at the time.');

    this.pinnedSourceId = null;
    this.pinnedSource = null;
  };

  return HandlerRegistry;
})();

exports['default'] = HandlerRegistry;
module.exports = exports['default'];