'use strict';

exports.__esModule = true;
exports.beginDrag = beginDrag;
exports.publishDragSource = publishDragSource;
exports.hover = hover;
exports.drop = drop;
exports.endDrag = endDrag;

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { 'default': obj }; }

var _utilsMatchesType = require('../utils/matchesType');

var _utilsMatchesType2 = _interopRequireDefault(_utilsMatchesType);

var _invariant = require('invariant');

var _invariant2 = _interopRequireDefault(_invariant);

var _lodashLangIsArray = require('lodash/lang/isArray');

var _lodashLangIsArray2 = _interopRequireDefault(_lodashLangIsArray);

var _lodashLangIsObject = require('lodash/lang/isObject');

var _lodashLangIsObject2 = _interopRequireDefault(_lodashLangIsObject);

var BEGIN_DRAG = 'dnd-core/BEGIN_DRAG';
exports.BEGIN_DRAG = BEGIN_DRAG;
var PUBLISH_DRAG_SOURCE = 'dnd-core/PUBLISH_DRAG_SOURCE';
exports.PUBLISH_DRAG_SOURCE = PUBLISH_DRAG_SOURCE;
var HOVER = 'dnd-core/HOVER';
exports.HOVER = HOVER;
var DROP = 'dnd-core/DROP';
exports.DROP = DROP;
var END_DRAG = 'dnd-core/END_DRAG';

exports.END_DRAG = END_DRAG;

function beginDrag(sourceIds) {
  var _ref = arguments.length <= 1 || arguments[1] === undefined ? {} : arguments[1];

  var _ref$publishSource = _ref.publishSource;
  var publishSource = _ref$publishSource === undefined ? true : _ref$publishSource;
  var _ref$clientOffset = _ref.clientOffset;
  var clientOffset = _ref$clientOffset === undefined ? null : _ref$clientOffset;
  var getSourceClientOffset = _ref.getSourceClientOffset;

  _invariant2['default'](_lodashLangIsArray2['default'](sourceIds), 'Expected sourceIds to be an array.');

  var monitor = this.getMonitor();
  var registry = this.getRegistry();
  _invariant2['default'](!monitor.isDragging(), 'Cannot call beginDrag while dragging.');

  for (var i = 0; i < sourceIds.length; i++) {
    _invariant2['default'](registry.getSource(sourceIds[i]), 'Expected sourceIds to be registered.');
  }

  var sourceId = null;
  for (var i = sourceIds.length - 1; i >= 0; i--) {
    if (monitor.canDragSource(sourceIds[i])) {
      sourceId = sourceIds[i];
      break;
    }
  }
  if (sourceId === null) {
    return;
  }

  var sourceClientOffset = null;
  if (clientOffset) {
    _invariant2['default'](typeof getSourceClientOffset === 'function', 'When clientOffset is provided, getSourceClientOffset must be a function.');
    sourceClientOffset = getSourceClientOffset(sourceId);
  }

  var source = registry.getSource(sourceId);
  var item = source.beginDrag(monitor, sourceId);
  _invariant2['default'](_lodashLangIsObject2['default'](item), 'Item must be an object.');

  registry.pinSource(sourceId);

  var itemType = registry.getSourceType(sourceId);
  return {
    type: BEGIN_DRAG,
    itemType: itemType,
    item: item,
    sourceId: sourceId,
    clientOffset: clientOffset,
    sourceClientOffset: sourceClientOffset,
    isSourcePublic: publishSource
  };
}

function publishDragSource(manager) {
  var monitor = this.getMonitor();
  if (!monitor.isDragging()) {
    return;
  }

  return {
    type: PUBLISH_DRAG_SOURCE
  };
}

function hover(targetIds) {
  var _ref2 = arguments.length <= 1 || arguments[1] === undefined ? {} : arguments[1];

  var _ref2$clientOffset = _ref2.clientOffset;
  var clientOffset = _ref2$clientOffset === undefined ? null : _ref2$clientOffset;

  _invariant2['default'](_lodashLangIsArray2['default'](targetIds), 'Expected targetIds to be an array.');
  targetIds = targetIds.slice(0);

  var monitor = this.getMonitor();
  var registry = this.getRegistry();
  _invariant2['default'](monitor.isDragging(), 'Cannot call hover while not dragging.');
  _invariant2['default'](!monitor.didDrop(), 'Cannot call hover after drop.');

  var draggedItemType = monitor.getItemType();
  for (var i = 0; i < targetIds.length; i++) {
    var targetId = targetIds[i];
    _invariant2['default'](targetIds.lastIndexOf(targetId) === i, 'Expected targetIds to be unique in the passed array.');

    var target = registry.getTarget(targetId);
    _invariant2['default'](target, 'Expected targetIds to be registered.');

    var targetType = registry.getTargetType(targetId);
    if (_utilsMatchesType2['default'](targetType, draggedItemType)) {
      target.hover(monitor, targetId);
    }
  }

  return {
    type: HOVER,
    targetIds: targetIds,
    clientOffset: clientOffset
  };
}

function drop() {
  var _this = this;

  var monitor = this.getMonitor();
  var registry = this.getRegistry();
  _invariant2['default'](monitor.isDragging(), 'Cannot call drop while not dragging.');
  _invariant2['default'](!monitor.didDrop(), 'Cannot call drop twice during one drag operation.');

  var targetIds = monitor.getTargetIds().filter(monitor.canDropOnTarget, monitor);

  targetIds.reverse();
  targetIds.forEach(function (targetId, index) {
    var target = registry.getTarget(targetId);

    var dropResult = target.drop(monitor, targetId);
    _invariant2['default'](typeof dropResult === 'undefined' || _lodashLangIsObject2['default'](dropResult), 'Drop result must either be an object or undefined.');
    if (typeof dropResult === 'undefined') {
      dropResult = index === 0 ? {} : monitor.getDropResult();
    }

    _this.store.dispatch({
      type: DROP,
      dropResult: dropResult
    });
  });
}

function endDrag() {
  var monitor = this.getMonitor();
  var registry = this.getRegistry();
  _invariant2['default'](monitor.isDragging(), 'Cannot call endDrag while not dragging.');

  var sourceId = monitor.getSourceId();
  var source = registry.getSource(sourceId, true);
  source.endDrag(monitor, sourceId);

  registry.unpinSource();

  return {
    type: END_DRAG
  };
}