var domassist = (function (exports) {
  'use strict';

  function isWindow(obj) {
    return obj != null && obj === obj.window;
  }

  function find(selector, context) {
    if (context === void 0) {
      context = null;
    }

    if (selector instanceof HTMLElement || selector instanceof Node || isWindow(selector)) {
      return [selector];
    } else if (selector instanceof NodeList) {
      return [].slice.call(selector);
    } else if (typeof selector === 'string') {
      var startElement = context ? find(context)[0] : document;
      return [].slice.call(startElement.querySelectorAll(selector));
    }

    return [];
  }

  function addClass(selector, cls) {
    if (Array.isArray(selector)) {
      return selector.forEach(function (item) {
        return addClass(item, cls);
      });
    }

    var els = find(selector);

    if (els.length) {
      var clsArray = [].concat(cls);
      els.forEach(function (el) {
        clsArray.forEach(function (item) {
          el.classList.add(item);
        });
      });
      return els;
    }
  }

  function on(selector, event, cb, capture) {
    if (capture === void 0) {
      capture = false;
    }

    if (Array.isArray(selector)) {
      selector.forEach(function (item) {
        return on(item, event, cb, capture);
      });
      return;
    }

    var data = {
      cb: cb,
      capture: capture
    };

    if (!window._domassistevents) {
      window._domassistevents = {};
    }

    window._domassistevents["_" + event] = data;
    var el = find(selector);

    if (el.length) {
      el.forEach(function (item) {
        item.addEventListener(event, cb, capture);
      });
    }
  }

  function matches(el, selector) {
    var proto = Element.prototype;
    var match = false;
    var prefixes = ['matches', 'matchesSelector', 'webkitMatchesSelector', 'mozMatchesSelector', 'msMatchesSelector', 'oMatchesSelector'];
    prefixes.forEach(function (prefix) {
      if (proto.hasOwnProperty(prefix)) {
        match = proto[prefix];
      }
    });

    if (match) {
      return el ? match.call(el, selector) : null;
    }
  }

  function closest(el, selector) {
    do {
      if (matches(el, selector)) {
        return el;
      }

      el = el.parentElement || el.parentNode;
    } while (el !== null && el.nodeType === 1);

    return null;
  }

  function delegate(el, event, selector, cb, capture) {
    if (capture === void 0) {
      capture = false;
    }

    on(el, event, function (e) {
      if (e.target && closest(e.target, selector)) {
        return cb(e);
      }
    }, capture);
  }

  function findOne(selector, el) {
    var found = find(selector, el);

    if (found.length) {
      return found[0];
    }

    return null;
  }

  // Check for the usage of native support for CustomEvents which is lacking
  // completely on IE.
  //

  function canIuseNativeCustom() {
    try {
      var p = new CustomEvent('t', {
        detail: {
          a: 'b'
        }
      });
      return p.type === 't' && p.detail.a === 'b';
    } catch (e) {
      return false;
    }
  } // Lousy polyfill for the Custom Event constructor for IE.


  var IECustomEvent = function CustomEvent(type, params) {
    var e = document.createEvent('CustomEvent');

    if (params) {
      e.initCustomEvent(type, params.bubbles, params.cancelable, params.detail);
    } else {
      e.initCustomEvent(type, false, false, undefined);
    }

    return e;
  };

  var DomassistCustomEvent = false;

  function fire(selector, type, params) {
    if (params === void 0) {
      params = {};
    }

    if (Array.isArray(selector)) {
      return selector.forEach(function (item) {
        return fire(item, type, params);
      });
    }

    if (!DomassistCustomEvent) {
      DomassistCustomEvent = canIuseNativeCustom() ? CustomEvent : IECustomEvent;
    }

    var els = find(selector);

    if (els.length) {
      if (params.bubbles !== false) {
        params.bubbles = true;
      }

      els.forEach(function (el) {
        var event = new DomassistCustomEvent(type, params);
        el.dispatchEvent(event);
      });
      return els;
    }
  }

  function hasClass(selector, cls) {
    var el = findOne(selector);

    if (!el) {
      return false;
    }

    return el.classList.contains(cls);
  }

  function hide(selector) {
    if (Array.isArray(selector)) {
      selector.forEach(function (item) {
        return hide(item);
      });
    }

    var els = find(selector);

    if (els.length) {
      els.forEach(function (el) {
        var currentDisplay = window.getComputedStyle(el).getPropertyValue('display');

        if (currentDisplay !== 'none') {
          el.dataset._currentDisplay = currentDisplay;
          el.style.display = 'none';
        }
      });
    }
  }

  function hover(el, enter, exit) {
    on(el, 'mouseenter', enter);
    on(el, 'mouseleave', exit);
  }

  function off(selector, event) {
    if (Array.isArray(selector)) {
      selector.forEach(function (item) {
        return off(item, event);
      });
    }

    if (!window._domassistevents) {
      window._domassistevents = {};
    }

    var data = window._domassistevents["_" + event];

    if (!data) {
      return;
    }

    var el = find(selector);

    if (el.length) {
      el.forEach(function (item) {
        item.removeEventListener(event, data.cb, data.capture);
      });
    }
  }

  function once(el, event, run, capture) {
    if (capture === void 0) {
      capture = false;
    }

    on(el, event, function (e) {
      off(el, event);
      run(e);
    }, capture);
  }

  var transform = null;
  function prefixedTransform() {
    if (transform) {
      return transform;
    }

    var testEl = document.createElement('div');

    if (testEl.style.transform === null) {
      var vendors = ['Webkit', 'webkit', 'Moz', 'ms'];
      var property = null;

      for (var i = 0, len = vendors.length; i < len && !property; i++) {
        var tProperty = vendors[i] + "Transform";

        if (typeof testEl.style[tProperty] !== 'undefined') {
          property = tProperty;
        }
      }

      transform = property;
    } else {
      transform = 'transform';
    }

    return transform;
  }

  function removeClass(selector, cls) {
    if (Array.isArray(selector)) {
      return selector.forEach(function (item) {
        return removeClass(item, cls);
      });
    }

    var els = find(selector);

    if (els.length) {
      var clsArray = [].concat(cls);
      els.forEach(function (el) {
        clsArray.forEach(function (item) {
          el.classList.remove(item);
        });
      });
      return els;
    }
  }

  var SCROLLABLE_CONTAINER;

  function getScrollableContainer() {
    if (SCROLLABLE_CONTAINER) {
      return SCROLLABLE_CONTAINER;
    }

    var documentElement = window.document.documentElement;
    var scrollableContainer;
    documentElement.scrollTop = 1;

    if (documentElement.scrollTop === 1) {
      documentElement.scrollTop = 0;
      scrollableContainer = documentElement;
    } else {
      scrollableContainer = document.body;
    }

    SCROLLABLE_CONTAINER = scrollableContainer;
    return scrollableContainer;
  }

  function show(selector) {
    if (Array.isArray(selector)) {
      selector.forEach(function (item) {
        return show(item);
      });
    }

    var els = find(selector);

    if (els.length) {
      els.forEach(function (el) {
        el.style.display = el.dataset._currentDisplay || 'block';
      });
    }
  }

  function toggleClass(selector, cls) {
    if (Array.isArray(selector)) {
      return selector.forEach(function (item) {
        return toggleClass(item, cls);
      });
    }

    var els = find(selector);

    if (els.length) {
      var clsArray = [].concat(cls);
      els.forEach(function (el) {
        clsArray.forEach(function (item) {
          el.classList.toggle(item);
        });
      });
      return els;
    }
  }

  var setupReady = function setupReady(callbacks) {
    return function (callback) {
      callbacks.push(callback);

      function execute() {
        while (callbacks.length) {
          var fn = callbacks.shift();

          if (typeof fn === 'function') {
            fn();
          }
        }
      }

      function loaded() {
        document.removeEventListener('DOMContentLoaded', loaded);
        execute();
      }

      setTimeout(function () {
        if (document.readyState !== 'loading') {
          return execute();
        }
      }, 0);
      document.addEventListener('DOMContentLoaded', loaded);
    };
  };

  var ready = setupReady([]);

  function styles(selector, css) {
    if (Array.isArray(selector)) {
      selector.forEach(function (item) {
        return styles(item, css);
      });
    }

    var els = find(selector);

    if (els.length) {
      els.forEach(function (el) {
        Object.keys(css).forEach(function (key) {
          el.style[key] = css[key];
        });
      });
    }
  }

  function addAttrs(selector, attrs) {
    if (Array.isArray(selector)) {
      return selector.forEach(function (item) {
        return addAttrs(item, attrs);
      });
    }

    var els = find(selector);

    if (els.length) {
      els.forEach(function (item) {
        Object.keys(attrs).forEach(function (attr) {
          if (attr in item) {
            item[attr] = attrs[attr];
          } else {
            item.dataset[attr] = attrs[attr];
          }
        });
      });
    }

    return els;
  }

  function html(selector, value) {
    if (Array.isArray(selector)) {
      selector.forEach(function (item) {
        return html(item, value);
      });
    }

    var el = find(selector);

    if (el.length) {
      var length = el.length;

      for (var i = 0; i < length; i += 1) {
        el[i].innerHTML = value;
      }
    }
  }

  /* global DocumentTouch */
  function isTouch() {
    return 'ontouchstart' in window || window.DocumentTouch && document instanceof DocumentTouch;
  }

  function bindEvents(el, events) {
    Object.keys(events).forEach(function (event) {
      on(el, event, events[event]);
    });
  }

  function modify(selector, params) {
    if (Array.isArray(selector)) {
      selector.forEach(function (item) {
        return modify(item, params);
      });
    }

    var modules = {
      addClass: addClass,
      removeClass: removeClass,
      html: html,
      events: on,
      styles: styles
    };
    var els = find(selector);

    if (els.length) {
      els.forEach(function (el) {
        Object.keys(params).forEach(function (param, index) {
          if (param in modules) {
            if (param === 'events') {
              bindEvents(el, params[param]);
            }

            modules[param](el, params[param]);
          }
        });
      });
    }
  }

  function append(selector, value) {
    if (Array.isArray(selector)) {
      return selector.forEach(function (item) {
        return append(item, value);
      });
    }

    var els = find(selector);

    if (els.length) {
      els.forEach(function (el) {
        if (typeof value === 'string') {
          el.insertAdjacentHTML('beforeend', value);
        } else {
          el.appendChild(value);
        }
      });
    }
  }

  function toArray(value) {
    if (!value) {
      return [];
    }

    if (Array.isArray(value)) {
      return value;
    }

    if (value instanceof Node) {
      return [value];
    }

    return [].slice.call(value);
  }

  function remove(selector, context) {
    if (Array.isArray(selector)) {
      return selector.forEach(function (item) {
        return remove(item, context);
      });
    }

    var els = find(selector, context);

    if (els.length) {
      els.forEach(function (el) {
        if (el.prototype && el.prototype.remove) {
          el.remove();
        } else if (el.parentNode) {
          el.parentNode.removeChild(el);
        }
      });
    }
  }

  var domassist_default = {
    addClass: addClass,
    delegate: delegate,
    find: find,
    findOne: findOne,
    fire: fire,
    hasClass: hasClass,
    hide: hide,
    hover: hover,
    isWindow: isWindow,
    off: off,
    on: on,
    once: once,
    prefixedTransform: prefixedTransform,
    removeClass: removeClass,
    scrollableContainer: getScrollableContainer,
    show: show,
    matches: matches,
    toggleClass: toggleClass,
    closest: closest,
    ready: ready,
    styles: styles,
    addAttrs: addAttrs,
    html: html,
    isTouch: isTouch,
    modify: modify,
    append: append,
    toArray: toArray,
    remove: remove
  };

  exports.addAttrs = addAttrs;
  exports.addClass = addClass;
  exports.append = append;
  exports.closest = closest;
  exports.default = domassist_default;
  exports.delegate = delegate;
  exports.find = find;
  exports.findOne = findOne;
  exports.fire = fire;
  exports.hasClass = hasClass;
  exports.hide = hide;
  exports.hover = hover;
  exports.html = html;
  exports.isTouch = isTouch;
  exports.isWindow = isWindow;
  exports.matches = matches;
  exports.modify = modify;
  exports.off = off;
  exports.on = on;
  exports.once = once;
  exports.prefixedTransform = prefixedTransform;
  exports.ready = ready;
  exports.remove = remove;
  exports.removeClass = removeClass;
  exports.scrollableContainer = getScrollableContainer;
  exports.show = show;
  exports.styles = styles;
  exports.toArray = toArray;
  exports.toggleClass = toggleClass;

  return exports;

}({}));
//# sourceMappingURL=domassist.bundle.js.map
