"use strict";
/*
 * Doppioh is DoppioJVM's answer to javah, although we realize the 'h' no longer
 * has a meaning.
 *
 * Given a class or package name, Doppioh will generate JavaScript or TypeScript
 * templates for the native methods of that class or package.
 *
 * Options:
 * -classpath Where to search for classes/packages.
 * -d [dir]   Output directory
 * -js        JavaScript template [default]
 * -ts [dir]  TypeScript template, where 'dir' is a path to DoppioJVM's
 *            TypeScript definition files.
 */
var option_parser_1 = require('../src/option_parser');
var path = require('path');
var fs = require('fs');
var util_1 = require('../src/util');
var classpath_1 = require('../src/classpath');
var ClassData_1 = require('../src/ClassData');
var JDKInfo = require('../vendor/java_home/jdk.json');
var enums_1 = require('../src/enums');
var async = require('async');
// Makes our stack traces point to the TypeScript source code lines.
require('source-map-support').install({
    handleUncaughtExceptions: true
});
var classpath = null, parser = new option_parser_1.OptionParser({
    default: {
        classpath: {
            type: 3 /* NORMAL_VALUE_SYNTAX */,
            alias: 'cp',
            optDesc: ' <class search path of directories and zip/jar files>',
            desc: 'A : separated list of directories, JAR archives, and ZIP archives to search for class files.'
        },
        help: { alias: '?', desc: 'print this help message' },
        directory: {
            type: 3 /* NORMAL_VALUE_SYNTAX */,
            alias: 'd',
            optDesc: ' <directory>',
            desc: 'Output directory'
        },
        javascript: {
            alias: 'js',
            desc: 'Generate JavaScript templates (Default is true)'
        },
        typescript: {
            alias: 'ts',
            desc: 'Generate TypeScript templates'
        },
        "doppiojvm-path": {
            type: 3 /* NORMAL_VALUE_SYNTAX */,
            optDesc: ' <path to doppiojvm module>',
            alias: 'dpath',
            desc: "Path to the doppiojvm module. Defaults to 'doppiojvm', referring to the NPM module."
        },
        "force_headers": {
            type: 3 /* NORMAL_VALUE_SYNTAX */,
            optDesc: ':[<classname>:]',
            alias: 'f',
            desc: '[TypeScript only] Forces doppioh to generate TypeScript headers for specified JVM classes'
        },
        "headers_only": {
            alias: 'ho',
            desc: '[TypeScript only] Only generate header file.'
        }
    }
});
function printEraseableLine(line) {
    // Undocumented functions.
    if (process.stdout['clearLine']) {
        process.stdout.clearLine();
        process.stdout.cursorTo(0);
        process.stdout.write(line);
    }
}
function printHelp() {
    process.stdout.write("Usage: doppioh [flags] class_or_package_names\n" + parser.help('default') + "\n");
}
// Remove "node" and "path/to/doppioh.js".
var parseResults = parser.parse(process.argv.slice(2)), args = parseResults['default'];
if (args.flag('help', false) || process.argv.length === 2) {
    printHelp();
    process.exit(1);
}
var outputDirectory = args.stringOption('directory', '.');
/**
 * java/lang/String.class => Ljava/lang/String;
 */
function file2desc(fname) {
    return "L" + fname.slice(0, fname.length - 6).replace(/\\/g, '/') + ";";
}
var cache = {};
/**
 * Returns the classes in the given directory in descriptor format.
 */
function getClasses(item) {
    var rv = [];
    // Find classpath items that contains this item as a directory.
    var cpItems = [];
    for (var i = 0; i < classpath.length; i++) {
        var searchedItem = item;
        var stat = classpath[i].tryStatSync(searchedItem);
        if (!stat) {
            searchedItem = item + ".class";
            stat = classpath[i].tryStatSync(searchedItem);
        }
        if (!stat) {
            continue;
        }
        else {
            if (!stat.isDirectory()) {
                // Files only counts if it is a class file.
                // Prevents an issue with the `doppio` shortcut counting as the `doppio` executable.
                if (path.extname(searchedItem) === '.class') {
                    rv.push(file2desc(searchedItem));
                }
            }
            else {
                cpItems.push(classpath[i]);
            }
        }
    }
    if (rv.length === 0 && cpItems.length === 0) {
        throw new Error("Unable to find resource " + item + ".");
    }
    if (cpItems.length > 0) {
        // Recursively process.
        var dirStack = [item];
        while (dirStack.length > 0) {
            var dir = dirStack.pop();
            for (var i = 0; i < cpItems.length; i++) {
                var dirListing = cpItems[i].tryReaddirSync(dir);
                if (dirListing === null) {
                    continue;
                }
                else {
                    for (var i_1 = 0; i_1 < dirListing.length; i_1++) {
                        var item_1 = dirListing[i_1];
                        var itemPath = path.join(dir, item_1);
                        if (path.extname(itemPath) === ".class") {
                            rv.push(file2desc(itemPath));
                        }
                        else {
                            dirStack.push(itemPath);
                        }
                    }
                }
            }
        }
    }
    return rv;
}
function loadClass(type) {
    for (var i = 0; i < classpath.length; i++) {
        var item = classpath[i];
        switch (item.hasClass(type)) {
            case enums_1.TriState.INDETERMINATE:
            case enums_1.TriState.TRUE:
                var buff = item.tryLoadClassSync(type);
                if (buff !== null) {
                    return buff;
                }
                break;
        }
    }
    throw new Error("Unable to find class " + type);
}
function findClass(descriptor) {
    if (cache[descriptor] !== undefined) {
        return cache[descriptor];
    }
    var rv;
    try {
        switch (descriptor[0]) {
            case 'L':
                rv = new ClassData_1.ReferenceClassData(loadClass(util_1.descriptor2typestr(descriptor)));
                // Resolve the class.
                var superClassRef = rv.getSuperClassReference(), interfaceClassRefs = rv.getInterfaceClassReferences(), superClass = null, interfaceClasses = [];
                if (superClassRef !== null) {
                    superClass = findClass(superClassRef.name);
                }
                if (interfaceClassRefs.length > 0) {
                    interfaceClasses = interfaceClassRefs.map(function (iface) { return findClass(iface.name); });
                }
                rv.setResolved(superClass, interfaceClasses);
                break;
            case '[':
                rv = new ClassData_1.ArrayClassData(descriptor.slice(1), null);
                break;
            default:
                rv = new ClassData_1.PrimitiveClassData(descriptor, null);
                break;
        }
        cache[descriptor] = rv;
        return rv;
    }
    catch (e) {
        throw new Error("Unable to read class file for " + descriptor + ": " + e + "\n" + e.stack);
    }
}
function processClassData(stream, template, classData) {
    var fixedClassName = classData.getInternalName().replace(/\//g, '_'), nativeFound = false;
    // Shave off L and ;
    fixedClassName = fixedClassName.substring(1, fixedClassName.length - 1);
    var methods = classData.getMethods();
    methods.forEach(function (method) {
        if (method.accessFlags.isNative()) {
            if (!nativeFound) {
                template.classStart(stream, fixedClassName);
                nativeFound = true;
            }
            template.method(stream, classData.getInternalName(), method.signature, method.accessFlags.isStatic(), method.parameterTypes, method.returnType);
        }
    });
    if (nativeFound) {
        template.classEnd(stream, fixedClassName);
    }
}
/**
 * TypeScript declaration file (JVMTypes.d.ts).
 */
var TSDeclarationFile = (function () {
    function TSDeclarationFile(doppiojvmPath, outputPath) {
        var _this = this;
        this.headerCount = 0;
        this.headerSet = {};
        this.generateQueue = [];
        this.headerPath = path.resolve(outputPath, "JVMTypes.d.ts");
        this.doppiojvmPath = path.relative(outputPath, doppiojvmPath);
        // Parse existing types file for existing definitions. We'll remake them.
        try {
            var existingHeaders = fs.readFileSync(this.headerPath).toString(), searchIdx = 0, clsName;
            // Pass 1: Classes.
            while ((searchIdx = existingHeaders.indexOf("export class ", searchIdx)) > -1) {
                clsName = existingHeaders.slice(searchIdx + 13, existingHeaders.indexOf(" ", searchIdx + 13));
                if (clsName.indexOf("JVMArray") !== 0) {
                    this.generateClassDefinition(this.tstype2jvmtype(clsName));
                }
                searchIdx++;
            }
            searchIdx = 0;
            // Pass 2: Interfaces.
            while ((searchIdx = existingHeaders.indexOf("export interface ", searchIdx)) > -1) {
                clsName = existingHeaders.slice(searchIdx + 17, existingHeaders.indexOf(" ", searchIdx + 17));
                this.generateClassDefinition(this.tstype2jvmtype(clsName));
                searchIdx++;
            }
        }
        catch (e) {
        }
        this.headerStream = fs.createWriteStream(this.headerPath);
        this.headersStart();
        // Generate required types.
        this.generateArrayDefinition();
        this.generateMiscDefinitions();
        this.generateClassDefinition('Ljava/lang/Throwable;');
        if (args.stringOption('force_headers', null)) {
            var clses = args.stringOption('force_headers', null).split(':');
            clses.forEach(function (clsName) {
                _this.generateClassDefinition(util_1.int_classname(clsName));
            });
        }
    }
    TSDeclarationFile.prototype.headersStart = function () {
        this.headerStream.write("// TypeScript declaration file for JVM types. Automatically generated by doppioh.\n// http://github.com/plasma-umass/doppio\nimport * as DoppioJVM from '" + this.doppiojvmPath.replace(/\\/g, '/') + "';\nimport JVMThread = DoppioJVM.VM.Threading.JVMThread;\nimport Long = DoppioJVM.VM.Long;\nimport ClassData = DoppioJVM.VM.ClassFile.ClassData;\nimport ArrayClassData = DoppioJVM.VM.ClassFile.ArrayClassData;\nimport ReferenceClassData = DoppioJVM.VM.ClassFile.ReferenceClassData;\nimport Monitor = DoppioJVM.VM.Monitor;\nimport ClassLoader = DoppioJVM.VM.ClassFile.ClassLoader;\nimport Interfaces = DoppioJVM.VM.Interfaces;\n\ndeclare module JVMTypes {\n");
    };
    /**
     * Emits TypeScript type declarations. Separated from fileEnd, since one can
     * use doppioh to emit headers only.
     */
    TSDeclarationFile.prototype.headersEnd = function () {
        this._processGenerateQueue();
        // Print newline to clear eraseable line.
        printEraseableLine("Processed " + this.headerCount + " classes.\n");
        this.headerStream.end("}\nexport = JVMTypes;\n", function () { });
    };
    TSDeclarationFile.prototype._processHeader = function (cls) {
        var _this = this;
        var desc = cls.getInternalName(), interfaces = cls.getInterfaceClassReferences().map(function (iface) { return iface.name; }), superClass = cls.getSuperClassReference(), methods = cls.getMethods().concat(cls.getMirandaAndDefaultMethods()), fields = cls.getFields(), methodsSeen = {}, injectedFields = cls.getInjectedFields(), injectedMethods = cls.getInjectedMethods(), injectedStaticMethods = cls.getInjectedStaticMethods();
        printEraseableLine("[" + this.headerCount++ + "] Processing header for " + util_1.descriptor2typestr(desc) + "...");
        if (cls.accessFlags.isInterface()) {
            // Interfaces map to TypeScript interfaces.
            this.headerStream.write("  export interface " + this.jvmtype2tstype(desc, false));
        }
        else {
            this.headerStream.write("  export class " + this.jvmtype2tstype(desc, false));
        }
        // Note: Interface classes have java.lang.Object as a superclass.
        // While java_lang_Object is a class, TypeScript will extract an interface
        // for the class under-the-covers and extract it, correctly providing us
        // with injected JVM methods on interface types (e.g. getClass()).
        if (superClass !== null) {
            this.headerStream.write(" extends " + this.jvmtype2tstype(superClass.name, false));
        }
        if (interfaces.length > 0) {
            if (cls.accessFlags.isInterface()) {
                // Interfaces can extend multiple interfaces, and can extend classes!
                // Add a comma after the guaranteed "java_lang_Object".
                this.headerStream.write(", ");
            }
            else {
                // Classes can implement multiple interfaces.
                this.headerStream.write(" implements ");
            }
            this.headerStream.write("" + interfaces.map(function (ifaceName) { return _this.jvmtype2tstype(ifaceName, false); }).join(", "));
        }
        this.headerStream.write(" {\n");
        Object.keys(injectedFields).forEach(function (name) { return _this._outputInjectedField(name, injectedFields[name], _this.headerStream); });
        Object.keys(injectedMethods).forEach(function (name) { return _this._outputInjectedMethod(name, injectedMethods[name], _this.headerStream); });
        Object.keys(injectedStaticMethods).forEach(function (name) { return _this._outputInjectedStaticMethod(name, injectedStaticMethods[name], _this.headerStream); });
        fields.forEach(function (f) { return _this._outputField(f, _this.headerStream); });
        methods.forEach(function (m) { return _this._outputMethod(m, _this.headerStream); });
        cls.getUninheritedDefaultMethods().forEach(function (m) { return _this._outputMethod(m, _this.headerStream); });
        this.headerStream.write("  }\n");
    };
    /**
     * Converts a typestring to its equivalent TypeScript type.
     */
    TSDeclarationFile.prototype.jvmtype2tstype = function (desc, prefix) {
        if (prefix === void 0) { prefix = true; }
        switch (desc[0]) {
            case '[':
                return (prefix ? 'JVMTypes.' : '') + ("JVMArray<" + this.jvmtype2tstype(desc.slice(1), prefix) + ">");
            case 'L':
                // Ensure all converted reference types get generated headers.
                this.generateClassDefinition(desc);
                return (prefix ? 'JVMTypes.' : '') + util_1.descriptor2typestr(desc).replace(/_/g, '__').replace(/\//g, '_');
            case 'J':
                return 'Long';
            case 'V':
                return 'void';
            default:
                // Primitives.
                return 'number';
        }
    };
    /**
     * Converts a TypeScript type into its equivalent JVM type.
     */
    TSDeclarationFile.prototype.tstype2jvmtype = function (tsType) {
        if (tsType.indexOf('JVMArray') === 0) {
            return "[" + this.tstype2jvmtype(tsType.slice(9, tsType.length - 1));
        }
        else if (tsType === 'number') {
            throw new Error("Ambiguous.");
        }
        else if (tsType === 'void') {
            return 'V';
        }
        else {
            // _ => /, and // => _ since we encode underscores as double underscores.
            return "L" + tsType.replace(/_/g, '/').replace(/\/\//g, '_') + ";";
        }
    };
    /**
     * Generates a TypeScript class definition for the given class object.
     */
    TSDeclarationFile.prototype.generateClassDefinition = function (desc) {
        if (this.headerSet[desc] !== undefined || util_1.is_primitive_type(desc)) {
            // Already generated, or is a primitive.
            return;
        }
        else if (desc[0] === '[') {
            // Ensure component type is created.
            return this.generateClassDefinition(desc.slice(1));
        }
        else {
            // Mark this class as queued for headerification. We use a queue instead
            // of a recursive scheme to avoid stack overflows.
            this.headerSet[desc] = true;
            this.generateQueue.push(findClass(desc));
        }
    };
    /**
     * Outputs a method signature for the given method on the given stream.
     * NOTE: We require a class argument because default interface methods are
     * defined on classes, not on the interfaces they belong to.
     */
    TSDeclarationFile.prototype._outputMethod = function (m, stream, nonVirtualOnly) {
        var _this = this;
        if (nonVirtualOnly === void 0) { nonVirtualOnly = false; }
        var argTypes = m.parameterTypes, rType = m.returnType, args = "", cbSig = "e?: java_lang_Throwable" + (rType === 'V' ? "" : ", rv?: " + this.jvmtype2tstype(rType, false)), methodSig, methodFlags = "public" + (m.accessFlags.isStatic() ? ' static' : '');
        if (argTypes.length > 0) {
            // Arguments are a giant tuple type.
            // NOTE: Long / doubles take up two argument slots. The second argument is always NULL.
            args = "args: [" + argTypes.map(function (type, i) { return ("" + _this.jvmtype2tstype(type, false) + ((type === "J" || type === "D") ? ', any' : '')); }).join(", ") + "]";
        }
        else {
            args = "args: {}[]";
        }
        methodSig = "(thread: JVMThread, " + args + ", cb?: (" + cbSig + ") => void): void";
        // A quick note about methods: It's illegal to have two methods with the
        // same signature in the same class, even if one is static and the other
        // isn't.
        if (m.cls.accessFlags.isInterface()) {
            if (m.accessFlags.isStatic()) {
            }
            else {
                // Virtual only, TypeScript interface syntax.
                stream.write("    \"" + m.signature + "\"" + methodSig + ";\n");
            }
        }
        else {
            if (!nonVirtualOnly) {
                stream.write("    " + methodFlags + " \"" + m.signature + "\"" + methodSig + ";\n");
            }
            stream.write("    " + methodFlags + " \"" + m.fullSignature + "\"" + methodSig + ";\n");
        }
    };
    /**
     * Outputs the field's type for the given field on the given stream.
     */
    TSDeclarationFile.prototype._outputField = function (f, stream) {
        var fieldType = f.rawDescriptor, cls = f.cls;
        if (cls.accessFlags.isInterface()) {
            // XXX: Ignore static interface fields for now, as reconciling them with TypeScript's
            // type system would be messy.
            return;
        }
        if (f.accessFlags.isStatic()) {
            stream.write("    public static \"" + util_1.descriptor2typestr(cls.getInternalName()) + "/" + f.name + "\": " + this.jvmtype2tstype(fieldType, false) + ";\n");
        }
        else {
            stream.write("    public \"" + util_1.descriptor2typestr(cls.getInternalName()) + "/" + f.name + "\": " + this.jvmtype2tstype(fieldType, false) + ";\n");
        }
    };
    /**
     * Outputs information on a field injected by the JVM.
     */
    TSDeclarationFile.prototype._outputInjectedField = function (name, type, stream) {
        stream.write("    public " + name + ": " + type + ";\n");
    };
    /**
     * Output information on a method injected by the JVM.
     */
    TSDeclarationFile.prototype._outputInjectedMethod = function (name, type, stream) {
        stream.write("    public " + name + type + ";\n");
    };
    /**
     * Output information on a static method injected by the JVM.
     */
    TSDeclarationFile.prototype._outputInjectedStaticMethod = function (name, type, stream) {
        stream.write("    public static " + name + type + ";\n");
    };
    TSDeclarationFile.prototype._processGenerateQueue = function () {
        while (this.generateQueue.length > 0) {
            this._processHeader(this.generateQueue.pop());
        }
    };
    /**
     * Generates the generic JVM array type definition.
     */
    TSDeclarationFile.prototype.generateArrayDefinition = function () {
        this.headerStream.write("  export class JVMArray<T> extends java_lang_Object {\n    /**\n     * NOTE: Our arrays are either JS arrays, or TypedArrays for primitive\n     * types.\n     */\n    public array: T[];\n    public getClass(): ArrayClassData<T>;\n    /**\n     * Create a new JVM array of this type that starts at start, and ends at\n     * end. End defaults to the end of the array.\n     */\n    public slice(start: number, end?: number): JVMArray<T>;\n  }\n");
    };
    TSDeclarationFile.prototype.generateMiscDefinitions = function () {
        this.headerStream.write("  // Basic, valid JVM types.\n  export type BasicType = number | java_lang_Object | Long;\n  export type JVMFunction = (thread: JVMThread, args: BasicType[], cb: (e?: JVMTypes.java_lang_Object, rv?: BasicType) => void) => void;\n");
    };
    return TSDeclarationFile;
}());
/**
 * TypeScript output template.
 */
var TSTemplate = (function () {
    function TSTemplate(doppiojvmPath, outputPath) {
        this.classesSeen = [];
        this.doppiojvmPath = path.relative(outputPath, doppiojvmPath);
        if (TSTemplate.declFile === null) {
            TSTemplate.declFile = new TSDeclarationFile(doppiojvmPath, outputPath);
        }
    }
    TSTemplate.prototype.getExtension = function () { return 'ts'; };
    TSTemplate.prototype.fileStart = function (stream) {
        // Reference all of the doppio interfaces.
        stream.write("import JVMTypes = require(\"./JVMTypes\");\nimport DoppioJVM = require('" + this.doppiojvmPath.replace(/\\/g, '/') + "');\nimport JVMThread = DoppioJVM.VM.Threading.JVMThread;\nimport Long = DoppioJVM.VM.Long;\ndeclare var registerNatives: (natives: any) => void;\n");
    };
    TSTemplate.prototype.fileEnd = function (stream) {
        var i;
        // Export everything!
        stream.write("\n// Export line. This is what DoppioJVM sees.\nregisterNatives({");
        for (i = 0; i < this.classesSeen.length; i++) {
            var kls = this.classesSeen[i];
            if (i > 0)
                stream.write(',');
            stream.write("\n  '" + kls.replace(/_/g, '/') + "': " + kls);
        }
        stream.write("\n});\n");
    };
    TSTemplate.prototype.classStart = function (stream, className) {
        stream.write("\nclass " + className + " {\n");
        this.classesSeen.push(className);
        TSTemplate.declFile.generateClassDefinition("L" + className.replace(/_/g, "/") + ";");
    };
    TSTemplate.prototype.classEnd = function (stream, className) {
        stream.write("\n}\n");
    };
    TSTemplate.prototype.method = function (stream, classDesc, methodName, isStatic, argTypes, rType) {
        var trueRtype = TSTemplate.declFile.jvmtype2tstype(rType), rval = "";
        if (trueRtype === 'number') {
            rval = "0";
        }
        else if (trueRtype !== 'void') {
            rval = "null";
        }
        argTypes.concat([rType]).forEach(function (type) {
            TSTemplate.declFile.generateClassDefinition(type);
        });
        stream.write("\n  public static '" + methodName + "'(thread: JVMThread" + (isStatic ? '' : ", javaThis: " + TSTemplate.declFile.jvmtype2tstype(classDesc)) + (argTypes.length === 0 ? '' : ', ' + argTypes.map(function (type, i) { return ("arg" + i + ": " + TSTemplate.declFile.jvmtype2tstype(type)); }).join(", ")) + "): " + TSTemplate.declFile.jvmtype2tstype(rType) + " {\n    thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');" + (rval !== '' ? "\n    return " + rval + ";" : '') + "\n  }\n");
    };
    TSTemplate.declFile = null;
    return TSTemplate;
}());
/**
 * JavaScript output template.
 */
var JSTemplate = (function () {
    function JSTemplate() {
        this.firstMethod = true;
        this.firstClass = true;
    }
    JSTemplate.prototype.getExtension = function () { return 'js'; };
    JSTemplate.prototype.fileStart = function (stream) {
        stream.write("// This entire object is exported. Feel free to define private helper functions above it.\nregisterNatives({");
    };
    JSTemplate.prototype.fileEnd = function (stream) {
        stream.write("\n});\n");
    };
    JSTemplate.prototype.classStart = function (stream, className) {
        this.firstMethod = true;
        if (this.firstClass) {
            this.firstClass = false;
        }
        else {
            stream.write(",\n");
        }
        stream.write("\n  '" + className.replace(/_/g, '/') + "': {\n");
    };
    JSTemplate.prototype.classEnd = function (stream, className) {
        stream.write("\n\n  }");
    };
    JSTemplate.prototype.method = function (stream, classDesc, methodName, isStatic, argTypes, rType) {
        // Construct the argument signature, figured out from the methodName.
        var argSig = 'thread', i;
        if (!isStatic) {
            argSig += ', javaThis';
        }
        for (i = 0; i < argTypes.length; i++) {
            argSig += ', arg' + i;
        }
        if (this.firstMethod) {
            this.firstMethod = false;
        }
        else {
            // End the previous method.
            stream.write(',\n');
        }
        stream.write("\n    '" + methodName + "': function(" + argSig + ") {");
        stream.write("\n      thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');");
        stream.write("\n    }");
    };
    return JSTemplate;
}());
var JAVA_HOME = path.resolve(__dirname, "../vendor/java_home");
var classpathPaths = JDKInfo.classpath.map(function (item) { return path.resolve(JAVA_HOME, item); }).concat(args.stringOption('classpath', '.').split(':'));
var classNames = args.unparsedArgs();
if (classNames.length === 0) {
    throw new Error("Must specify a class name.");
}
if (!fs.existsSync(outputDirectory)) {
    fs.mkdirSync(outputDirectory);
}
// Initialize classpath.
classpath_1.ClasspathFactory(JAVA_HOME, classpathPaths, function (items) {
    // Normally, JARs are loaded asynchronously. Force them to be loaded, which allows us
    // to load classes synchronously.
    async.each(items, function (item, cb) {
        if (item instanceof classpath_1.UnindexedClasspathJar || item instanceof classpath_1.IndexedClasspathJar) {
            item.loadJar(cb);
        }
        else {
            cb();
        }
    }, function (e) {
        if (e) {
            throw e;
        }
        classpath = items;
        try {
            classNames.forEach(function (className) {
                var targetName = className.replace(/\//g, '_').replace(/\./g, '_'), targetPath = className.replace(/\./g, '/');
                var template = args.flag('typescript', false) ? new TSTemplate(args.stringOption('doppiojvm-path', 'doppiojvm'), outputDirectory) : new JSTemplate();
                var stream = fs.createWriteStream(path.join(outputDirectory, targetName + '.' + template.getExtension()));
                template.fileStart(stream);
                var classes = getClasses(targetPath);
                for (var i = 0; i < classes.length; i++) {
                    var desc = classes[i];
                    processClassData(stream, template, findClass(desc));
                }
                template.fileEnd(stream);
                stream.end(new Buffer(''), function () { });
                if (args.flag('typescript', false) && args.flag('headers_only', false)) {
                    fs.unlinkSync(path.join(outputDirectory, targetName + '.' + template.getExtension()));
                }
            });
            if (args.flag('typescript', false)) {
                TSTemplate.declFile.headersEnd();
            }
        }
        catch (e) {
            console.error("Encountered error: " + e);
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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