"use strict";
/**
 * Downloads DoppioJVM's JDK into vendor/java_home.
 */
var fs = require('fs');
var path = require('path');
var https = require('https');
var rimraf = require('rimraf');
var glob = require('glob');
var gunzip = require('gunzip-maybe');
var tarFs = require('tar-fs');
var JDK_URL = "https://github.com/plasma-umass/doppio_jcl/releases/download/v3.2/java_home.tar.gz";
var JDK_PATH = path.resolve(__dirname, "..", "..", "..", "vendor");
var JDK_FOLDER = "java_home";
/**
 * Checks if the JDK is already downloaded and installed.
 */
function doesJDKExist() {
    var jdkInfoPath = path.resolve(JDK_PATH, JDK_FOLDER, "jdk.json");
    if (fs.existsSync(jdkInfoPath)) {
        try {
            var jdkInfo = JSON.parse(fs.readFileSync(jdkInfoPath).toString());
            return jdkInfo['url'] === JDK_URL;
        }
        catch (e) {
            return false;
        }
    }
    return false;
}
function mkdirp(p) {
    p = path.resolve(p);
    if (!fs.existsSync(p)) {
        // Make parent first if it doesn't exist.
        mkdirp(path.dirname(p));
        fs.mkdirSync(p);
    }
}
/**
 * Downloads the JDK at JDK_URL into the destination path.
 * Creates the path if it doesn't exist.
 */
function downloadJDK(url, destPath, cb) {
    var ended = false;
    mkdirp(destPath);
    https.get(url, function (res) {
        if (res.statusCode === 302) {
            // Redirected.
            ended = true;
            downloadJDK(res.headers['location'], destPath, cb);
            return;
        }
        // Will be NaN if not given by server.
        var contentLength = parseInt(res.headers['content-length']);
        var progressBytes = 0;
        var startTime = new Date().getTime();
        function printStatus() {
            var percent = "??";
            if (!isNaN(contentLength)) {
                percent = ((progressBytes / contentLength) * 100).toFixed(0);
            }
            var lastPrint = new Date().getTime();
            // bytes => kbytes
            var dlSoFar = progressBytes >> 10;
            var rate = dlSoFar / ((lastPrint - startTime) / 1000);
            console.log("[" + percent + "%] Received " + dlSoFar + " KB [" + rate.toFixed(2) + " KB/s]");
        }
        var interval = setInterval(function () {
            printStatus();
        }, 5000);
        function end(err) {
            if (!ended) {
                ended = true;
                clearInterval(interval);
                cb(err);
            }
        }
        res.pipe(gunzip()).pipe(tarFs.extract(destPath)).on('error', end).on('finish', end);
        res.on('data', function (d) {
            progressBytes += d.length;
        });
    }).on('error', function (err) {
        if (!ended) {
            ended = true;
            cb(err);
        }
    });
}
function writeJdkJson() {
    var globSearch = path.resolve(JDK_PATH, JDK_FOLDER).replace(/\\/g, '/') + "/lib/*.jar";
    glob(globSearch, function (e, classpath) {
        if (e) {
            console.error("Failed to locate JDK JAR items: " + e);
            process.exit(1);
        }
        var rtIndex = -1;
        classpath = classpath.map(function (item, i) {
            switch (path.basename(item)) {
                case "rt.jar":
                    rtIndex = i;
                    break;
            }
            return path.relative(path.resolve(JDK_PATH, JDK_FOLDER), item);
        });
        var rt = classpath[rtIndex];
        classpath.splice(rtIndex, 1);
        classpath.unshift(rt);
        var jdkJson = {
            url: JDK_URL,
            // Convert Windows-style paths to Unix-style paths
            classpath: classpath.map(function (cpItem) { return cpItem.replace(/\\/g, '/'); })
        };
        // Ensure our generated doppio.jar is added.
        if (jdkJson.classpath.filter(function (cpItem) { return path.basename(cpItem) === 'doppio.jar'; }).length === 0) {
            jdkJson.classpath.push('lib/doppio.jar');
        }
        fs.writeFileSync(path.resolve(JDK_PATH, JDK_FOLDER, "jdk.json"), new Buffer(JSON.stringify(jdkJson), "utf8"));
        // TypeScript typings, so it can be used as a module. :)
        fs.writeFileSync(path.resolve(JDK_PATH, JDK_FOLDER, "jdk.json.d.ts"), new Buffer("declare let JDKInfo: {\n  url: string;\n  classpath: string[];\n};\nexport = JDKInfo;\n", "utf8"));
    });
}
if (!doesJDKExist()) {
    console.log("JDK is out of date! Removing old JDK...");
    rimraf(path.resolve(JDK_PATH, JDK_FOLDER), function (err) {
        if (err) {
            console.error("Error removing old JDK: " + err);
            process.exit(1);
        }
        console.log("Downloading new JDK...");
        downloadJDK(JDK_URL, JDK_PATH, function (err) {
            if (err) {
                console.error("Failed to download JDK: " + err + ".");
                process.exit(1);
            }
            else {
                console.log("Successfully downloaded JDK.");
                writeJdkJson();
            }
        });
    });
}
else {
    console.log("JDK is up-to-date.");
}
//# sourceMappingURL=data:application/json;base64,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